;;; -*- Mode: Lisp; Syntax: Common-lisp; Base: 10; Package: PROTOS -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)




;;;=============================================================================
;;;
;;;         E X P L A N A T I O N    U T I L I T Y    F U N C T I O N S
;;;   ------------------------------------------------------------------------
;;;
;;;  Overview:   This file contains various utility functions for entering and
;;;              processing explanations.  The explanation parser, however, is
;;;              in a separate file (parse.lisp).
;;;
;;;  Functions:	 install-relation
;;;              delete-relation
;;;              get-leaves
;;;              examine-explanations
;;;              enter-explanations
;;;              enter-explantion
;;;              invert-explanation
;;;              truncate-explanation
;;;              compute-explanation-strength
;;;              get-log-text
;;;              get-specific-explanation
;;;=============================================================================



(defparameter *explanation-comment* nil)
(defparameter *relation-direction* nil)

;;;-----------------------------------------------------------------------------
;;;  Function:  (install-relation from-nodes to-nodes quantifiers verb condition)
;;;
;;;  Given:     all the components of a 1-step explanation (i.e., the verb,
;;;             optional quantifiers, list of antecedents, and list of
;;;             consequents;
;;;
;;;  Returns:   the created forward relation.
;;;
;;;  Does:      This function installs the new relation in two steps.  It first 
;;;             creates the forward relation structure and adds it to the list 
;;;             of relations associated with the antecedent node(s).  It then
;;;             installs the inverse relation in the consequent node(s).
;;;
;;;             The forward and inverse relations are installed in the category
;;;             network unless:
;;;             (1)  the new relation is identical to an existing relation, or
;;;             (2)  the new relation conflicts with an existing relation and
;;;                  the teacher says to discard the new relation.
;;;
;;;-----------------------------------------------------------------------------

(defun install-relation (from-nodes to-nodes quantifiers verb condition)
  (let ((iquantifier  (verb-iquantifier verb))   ;inverse quantifiers
        (iverb        (verb-inverse verb))       ;inverse verb
         frel                                    ;forward relation
         irel                                    ;inverse relation
         iquantifiers)                           ;list of inverse quantifiers

      ;; Build inverse quantifier list.
      (cond ((eql 'same iquantifier)    (setq iquantifiers quantifiers))
	    ((null iquantifier)         (setq iquantifiers nil))
	    ((quantifier-p iquantifier) (setq iquantifiers (list iquantifier)))
	    (t  (format t "~%Error: install-relation: unknown iquantifier")))

      ;; Attempt to install "forward" relation
      (setq *relation-direction* 'forward)
      (setq frel
        (install-relation1 from-nodes to-nodes quantifiers verb condition))

      (if (eq verb *verb-spurious*)    ;; spurious relations have no inverse
          (setf (relation-inverse frel) nil)

          (progn
            ;; otherwise attempt to install "inverse" relation
            (setq *relation-direction* 'inverse)
            (setq irel
              (install-relation1 to-nodes from-nodes iquantifiers iverb condition))

            ;; Set the inverse relation slot of the forward and inverse relations.
            (setf (relation-inverse irel) frel)
            (setf (relation-inverse frel) irel)))

      (return-from install-relation frel)))


;;;-----------------------------------------------------------------------------
;;;  Function:  (install-relation1 from-nodes to-nodes quantifiers verb condition)
;;;
;;;  Given:     all the components of a 1-step explanation (i.e., the verb,
;;;             optional quantifiers, list of antecedents, and list of
;;;             consequents;
;;;
;;;  Returns:   the created relation in the (from -> to) direction.
;;;
;;;  Does:      This function installs the relation in the category network
;;;             unless:
;;;             (1)  the new relation is identical to an existing relation, or
;;;             (2)  the new relation conflicts with an existing relation and
;;;                  the teacher says to discard the new relation.
;;;
;;;             Note:  This function is different from install-relation in
;;;                  that it only attempts to install a relation in one
;;;                  direction (forward or inverse), but not both.
;;;-----------------------------------------------------------------------------

(defun install-relation1 (from-nodes to-nodes quantifiers verb condition)

  (let ((conflicts nil))
    ;; The purpose of the following code is to check to see if this new
    ;; relation conflicts with an existing relation in the category network.
    ;; For example, if there is a new relation "A sometimes causes B" and
    ;; there is an existing relation "A suggests B", Protos will ask the user
    ;; which relation to retain.

    ;; Check every relation emanating from the first antecedent node.
    (dolist (rel (node-relations (first from-nodes)))
      ;; If it has the exact same set of antecedents and consequents ...
      (if (and (null (set-exclusive-or0 from-nodes (relation-from-nodes rel)))
	       (null (set-exclusive-or0 to-nodes   (relation-to-nodes   rel))))
	  ;; then if it also has the same verb, quantifiers, and condition ...
	  (if (and (eq verb (relation-verb rel))
		   (null (set-exclusive-or0 quantifiers (relation-quantifiers rel)))
		   (let ((rcondition (relation-condition rel)))
 		     (or (eql condition rcondition)
			 (and condition 
                              rcondition
                              (eql (condition-type condition)
				   (condition-type rcondition))
			      (eq  (condition-category condition)
				   (condition-category rcondition))
			      (null (set-exclusive-or0 (condition-features condition)
						      (condition-features rcondition)))))))
	      ;; then just return since nothing needs to be changed.
	      (return-from install-relation1 rel)
	      ;; else we have a conflict to report to the teacher.
	      ;; where rel is the old relation, and nrel is the new relation
	      (push rel conflicts))))

    ;; To get here, we know that the proposed new relation is not identical
    ;; to some existing relation.  So, we now construct the new relation
    ;; structure.

    (let* ((strength    (verb-strength verb))	; strength of "forward" verb
	    nrel)

      ;; Adjust strength of forward relation per the quantifiers, if any.
      (dolist (quantifier quantifiers)
	(setq strength (* strength (quantifier-strength quantifier))))

      ;; Make the "new" relation.
      (setq nrel (make-relation :verb        verb
	   			:from-nodes  from-nodes
				:to-nodes    to-nodes
				:strength    strength
				:condition   condition
				:quantifiers quantifiers
				:comment     *explanation-comment*))

      ;; If no conflict found ...
      (if (null conflicts)
	  ;; then install the newly created relation and return.
	  (return-from install-relation1 (add-new-rel nrel))
	  ;; else ask the user about each conflicting relation
	  (let ((last-rel nil))
            (if *loading-kb* 
              (let ((conflict-list nil))
	         (dolist (rel conflicts)
                    (if (eq verb (relation-verb rel))
                        (push rel conflict-list)))
                 (if conflict-list
                     (dolist (rel conflict-list)
                       ;;(format t "~%   nrel = ")
                       ;;(print-relation nrel t 1)
                       ;;(format t ", dir (~A)~% conflict = " *relation-direction*)
                       ;;(print-relation rel t 1)
                       (if (eq *relation-direction* 'forward)
                           (setq last-rel (replace-relation rel nrel))
                           (setq last-rel rel)))
                     (setq last-rel (add-new-rel nrel))))
	      (dolist (rel conflicts)
	         (setq last-rel (install-relation2 rel nrel))))
	    (return-from install-relation1 last-rel))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (install-relation2  new-relation  existing-relation)
;;;
;;;  Given:     a new relation that shares the same set of antecedent nodes
;;;             and the same set of consequent nodes as the existing relation;
;;;
;;;  Returns:   the new relation if it was installed, otherwise the existing
;;;             relation.
;;;
;;;  Does:      this function may either install or discard the new relation
;;;             and may either retain or delete the existing relation,
;;;             depending on the heuristics given below.
;;;
;;;  Returns:   nothing of significance.
;;;
;;;  Heuristics:  Six cases are considered.  The new link is added when:
;;; 
;;;               1.  The existing link and the new link are both unconditional
;;;                   and the new link is stronger.  In this case the old link
;;;                   is removed.
;;;
;;;               2.  The existing link and the new link are both unconditional,
;;;                   the new link is weaker, but the teacher tells Protos to
;;;                   replace the existing link when Protos asks what to do.
;;;
;;;               3.  The existing link and the new link have different 
;;;                   conditions associated with them.  The new link is added
;;;                   without removing the existing one.
;;;
;;;               4.  The existing link has an associated condition and the new
;;;                   link is unconditional.  If the existing conditional link
;;;                   is stronger, the unconditional link is added without
;;;                   removing it.  Otherwise, the conditional link is replaced.
;;;
;;;               5.  The existing link is unconditional and a stronger
;;;                   conditional link is given.  The existing link is not
;;;                   removed.
;;;
;;;               6.  The existing link is unconditional and a weaker condi-
;;;                   tional link is given.  In this case the teacher is queried
;;;                   and if he/she concurrs, the existing link is replaced.
;;;
;;;  Note:     This function checks each of the above items (1 - 6) pretty much
;;;            in that order.  To comment what kind of test and/or action a
;;;            piece of code performs, a number corresponding to the cases
;;;            above appears in the comment.
;;;-----------------------------------------------------------------------------

(defun install-relation2  (old-rel new-rel)
  (let ((old-condition (relation-condition old-rel)) 
	(new-condition (relation-condition new-rel))
	;; A conditional relation is defined to be a relation whose condition field
	;; is non-NIL.  In this function, the condition fields of old-rel and new-rel
	;; will be used not only to determine if the relations are conditional, but 
	;; also to compare if the two conditions are the same.
	(old-strength (relation-strength old-rel))
	(new-strength (relation-strength new-rel)))
    
    (cond
      ;; If both old and new relations are unconditional or
      ;;    both relations have the same condition then ...
      ((or (and (null old-condition) (null new-condition))
           (and old-condition new-condition
                (equal-conditions old-condition new-condition)))
       ;; if the new relation is stronger than the old one ...
       (if (> new-strength old-strength)
	   ;; then replace the old relation with the new one [case 1]
	   (replace-relation old-rel new-rel)
	   ;; else ask user what to do [case 2].
	   (install-rel2-query old-rel new-rel)))

      ;; If the old link and new link have different conditions ...
      ((and old-condition
	    new-condition
	    (not (equal-conditions old-condition new-condition)))
       ;; then add the new relation [case 3].
       (add-new-rel new-rel))

      ;; If the old link has a condition but the new link does not ...
      ((and old-condition (not new-condition))
       ;; and if the old link is stronger than the new link ...
       (if (> old-strength new-strength)
	   ;; then add the new unconditional link [case 4a]
	   (add-new-rel new-rel)
	   ;; else replace the old conditional link [case 4b].
	   (replace-relation old-rel new-rel)))

      ;; If the old link is unconditional but the new link is conditional
      ((and (not old-condition) new-condition)
       ;; and is stronger ...
       (if (>= new-strength old-strength)
	   ;; then add the new relation [case 5]
	   (add-new-rel new-rel)
	   ;; else ask the user what to do [case 6].
	   (install-rel2-query old-rel new-rel))))))


;-----------------------------------------------------------------------------------

(defun equal-conditions  (old-condition  new-condition)
  (and (eql (condition-type old-condition)
	    (condition-type new-condition))
       (eq  (condition-category old-condition)
	    (condition-category new-condition))
       (null (set-exclusive-or0 (condition-features old-condition)
			       (condition-features new-condition)))))


(defun add-new-rel (nrel)
  "this function was adapted from your previous code, and its purpose is
   to add the new relation in the (from -> to) direction to the network."
  
  (let ((from-nodes (relation-from-nodes nrel)))

	  ;; Add the forward relation to each of its from-nodes.
	  (dolist (node from-nodes)
	    (push nrel (node-relations node)))
    
        nrel))


(defun add-new-rel2 (frel)
  "this function was adapted from your previous code, and its purpose is
   to add the new relation and its inverse relation to the network.  This
   code is currently unused."
  
  (let ((from-nodes (relation-from-nodes frel))
	(to-nodes   (relation-to-nodes frel))
	(irel       (relation-inverse frel)))

	  ;; Add the forward relation to each of its from-nodes.
	  (dolist (node from-nodes)
	    (push frel (node-relations node)))
    
	  ;; Add the inverse relation to each of the to-nodes.
	  (dolist (node to-nodes)
	    (push irel (node-relations node)))
  
  frel))

;---------------------------------------------------------------------------------

(defun delete-relation (nrel)
  
  (let ((from-nodes (relation-from-nodes nrel)))

    (dolist (node from-nodes)
      (setf (node-relations node) (delete nrel (node-relations node))))))


;-------------------------------------------------------------------------------------		
(defun replace-relation (old-rel new-rel)
  (delete-relation old-rel)
  (add-new-rel new-rel))


(defparameter *relation-menu* (make-menu
   :label "~%~%Which do you want to keep?~
             ~%=========================="
   :items '((#\F . ("Former"   return  former))
	    (#\L . ("Latter"   return  latter))
	    (#\B . ("Both"     return  both)))))

;; This parameter is set to T while loading a knowledge base from a file
;; so that conflicting relations will just be displayed but will not stop
;; and ask the user what to do about them.
(defparameter *only-warn-on-conflict* nil)

(defun install-rel2-query (old-rel new-rel)
  (format t "~%WARNING! Your relation:   ")
  (print-relation new-rel t 1)
  (format t "~%         conflicts with:  ")
  (print-relation old-rel t 1)

  (if (not (eq (relation-verb old-rel) (relation-verb new-rel)))
      (format t "~%   verb mismatch."))
  (if (set-exclusive-or0 (relation-quantifiers old-rel) (relation-quantifiers new-rel))
      (format t "~%   quantifier mismatch."))
  (if (set-exclusive-or0 (relation-from-nodes old-rel) (relation-from-nodes new-rel))
      (format t "~%   from-nodes mismatch."))
  (if (set-exclusive-or0 (relation-to-nodes old-rel) (relation-to-nodes new-rel))
      (format t "~%   to-nodes mismatch."))

  (if *only-warn-on-conflict*
      (add-new-rel new-rel)
      ;; Let the teacher choose to keep the former, the latter, or both.
      (case (menu-select *relation-menu*)
	(former     (replace-relation old-rel new-rel))
	(latter     old-rel)
	(both       (add-new-rel new-rel))
	(otherwise  (format t "Error: install-relation: bad menu return")))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (get-leaves  explanation  to  leaves)
;;;
;;;  Given:     explanation, an instance of an explanation structure; and
;;;             to, which is non-nil if we want the leaves in the "to" direction
;;;                 (otherwise we get the leaves in the "from" direction); and
;;;             leaves, a list of leaves already found (should be nil in initial
;;;                     call).
;;;
;;;  Returns:   The list of leaf nodes of the explanation, when searched in
;;;             the "to" or "from" direction.  For example, given the explanation
;;;             "A implies B causes C" and told to search in the "to" direction,
;;;             it would return (C).  Given the explanation
;;;             "A is caused by (B and [C: C equivalent to D])",
;;;             it would return (B D).
;;;-----------------------------------------------------------------------------

(defun get-leaves (explanation to leaves)
  ;; If direction of search is "to" (forward) ...
  (if to
      ;; then examine the to-terms
      (dolist (term (explanation-to-terms explanation))
	(typecase term
	  (node         (pushnew term leaves))
	  (explanation  (setq leaves (get-leaves term to leaves)))
	  (otherwise    (cerror "Ignore this term."
				"get-leaves: wasn't a term or an explanation."))))
      ;; else examine the from-terms.
      (dolist (term (explanation-from-terms explanation))
	(typecase term
	  (node         (pushnew term leaves))
	  (explanation  (setq leaves (get-leaves term to leaves)))
	  (otherwise    (cerror "Ignore this term."
				"get-leaves: wasn't a term or an explanation.")))))
  leaves)


;;;-----------------------------------------------------------------------------
;;;  Function:  (examine-explanations  match)
;;;
;;;  Purpose:   Given a match, this function allows the user to examine any
;;;             of the match's explanations by specifying them by number (i.e.,
;;;             by the line numbers printed by 'print-match').
;;;-----------------------------------------------------------------------------

(defun examine-explanations (match)
  (let (input)
    (if (prompt "~%Do you wish to examine any of the explanations? " nil
		'y-or-n nil nil)
	(loop
          (setq input (prompt "~%Type a number (or q to quit): " nil 'integer nil nil))
	  (if (null input) (return (values)))
	  (if (and (numberp input) (plusp input))
	      (let ((result (nth (1- input) (match-results match))))
		(if result
		    (let ((expl (result-explanation result)))
		      (format t "~&~2D   " input)
		      (if (null expl)
			  (if (eql 'identical (result-type result))
			      (format t "=== identical terms ===")
			      (format t "(no explanation)"))
			  (progn
			    (print-explanation expl)
			    (if (and *switch-critiques*
				     *logstream*
				     (y-or-n-p "~%Do you wish to critique this explanation? "))
				(progn
				  (if *logstream*
				      (progn
					(write-line "FROM KBPM:  " *logstream*)
					(print-explanation2 expl *logstream*)))
				  (format *query-io* "~%Terminate with a blank line:")
				  (get-log-text))))))
		    (format *query-io* "~%No, you must enter a number between 1 and ~D"
			    (length (match-results match)))))
	      (format *query-io* "~%No, you must enter a number between 1 and ~D"
		      (length (match-results match))))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  get-log-text
;;;
;;;  Does:      This function prompts the user for a multi-line comment and
;;;             returns a string containing the whole comment (or nil if no
;;;             comment was entered).  Also, it writes the comment to the log
;;;             file, if it is currently open.
;;;-----------------------------------------------------------------------------

(defun get-log-text ()
  (let ((strings nil))
    (if *logstream*
	(write-line "ENGLISH:" *logstream*))
    (loop
      (format *query-io* "~&(text): ")
      (let ((input (read-line *query-io* nil nil)))
	;; If end of input, decide what to return.
	(if (or (null input) (string= input ""))
	    (return (if (null strings)
			nil
			(progn
			  (setq strings (nreverse strings))
			  (if (cdr strings)
			      (format nil "~A~{~%~A~}" (car strings) (cdr strings))
			      (car strings))))))
	(push input strings)
	(if *logstream*
	    (write-line input *logstream*))))))



;;;-----------------------------------------------------------------------------
;;;-----------------------------------------------------------------------------
(defun enter-explanations ()
  (let (expl)
    (loop
      (setq expl (enter-explanation))
      (if (null expl) (return (values)))
      (if (not (prompt "~%More explanations? " nil 'y-or-n nil nil))
	  (return (values))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (enter-explanation)
;;;
;;;  Returns:   an explanation structure corresponding to one of the (possibly
;;;             several) explanation forms entered by the teacher.
;;;
;;;  Purpose:   This function first asks for the explanation in natural-language
;;;             text, then asks for it in the Protos explanation language.
;;;             The natural-language text (which is saved for protocol analysis)
;;;             is primarily important because it allows the teacher to organize
;;;             his/her thoughts and express them in a form that is completely
;;;             natural.  Then, and only then, should the teacher perform the
;;;             distinct task of translating from natural language to Protos'
;;;             explanation language.  An earlier knowledge-engineering
;;;             experiment with Protos showed that this 2-step approach resulted
;;;             in much better explanations to Protos than when the teacher
;;;             tried to express himself directly in Protos' language.
;;;-----------------------------------------------------------------------------

(defun enter-explanation ()
  (prog ((expl-forms nil)
	 (expl       nil)
	 (comment    nil))

        (consume-white-space)
	(format *query-io*
		"~%Please enter your thoughts first in English,~
                 ~%and terminate with a blank line.")
	(setq comment (get-log-text))
      
	(format *query-io*
		"~%Now, translate this into one or more Protos-style explanations,~
                 ~%enclosed in parentheses (terminate with blank line):")
	(if *logstream*
	    (write-line "PROTOS:" *logstream*))
	
	;; The following code reads in all the teacher's explanations BEFORE
	;; processing them.  This allows the teacher to focus on translating
	;; the English, without being "interrupted" to discuss new terms that
	;; appear in these explanations.
	
     get-expl
	(setq expl-forms (prompt "" "~&(Protos explanation): "
				 'explanation nil nil))
	(if *logstream*
	    (dolist (form expl-forms)
	      (princ form *logstream*)
	      (terpri *logstream*)))

;; This is the old code, replaced by the above code.
;	(loop
;	  (fresh-line *query-io*)
;	  (write-string "(Protos explanation): " *query-io*)
;	  (if (char= #\Newline (peek-char nil *query-io* nil nil))
;	      (progn
;		(read-char *query-io* nil nil)	   ; consume the newline
;		(go process-expls)))
;	  (setq input (read *query-io* nil nil))
;	  (if (null input) (return (values)))
;	  (consume-white-space)
;	  (push input expl-forms)
	
     process-expls
	(if expl-forms
	    (let ((error-forms nil)
		  expl2)

	      ;; The following global, *explanation-comment*, is read by
	      ;; install-relation to attach any comment to each relation
	      ;; of this explanation.  I know this use of a global is not
	      ;; pretty, but it would be more work to pass this argument
	      ;; through the parser.
	      (setq *explanation-comment* comment)

	      (dolist (expl-form expl-forms)
		(setq expl2 (get-explanation expl-form))
		;; IF an explanation is obtained from the teacher's input ...
		(if expl2
		    ;; THEN compute its strength and remember it for return
		    (progn
		      (setq expl expl2)
		      (compute-explanation-strength expl2))
		    ;; ELSE there must have been an error in the input.
		    (push expl-form error-forms)))

	      (setq *explanation-comment* nil)   ; just a precaution.
	      (if error-forms
		  (progn
		    (dolist (expl-form error-forms)
		      (format *query-io* "~%Please correct ~A" expl-form))
		    (go get-expl)))))

;	    (loop
;	      (let ((expl-form (pop expl-forms))
;		    expl2)
;		(if (null expl-form) (return (values)))
;		(setq expl2 (get-explanation expl-form))
;		;; IF an explanation is obtained from the teacher's input ...
;		(if expl2
;		    ;; THEN compute its strength and remember it for return
;		    (progn
;		      (setq expl expl2)
;		      (compute-explanation-strength expl2))
;		    ;; ELSE there must have been an error in the input.
;		    (progn
;		      (format *query-io* "~%Please correct ~A" expl-form)
;		      (go get-expl)))))

	(return-from enter-explanation expl)))
     



;;;-----------------------------------------------------------------------------
;;;  Feature:  (invert-explanation  explanation)
;;;
;;;  Returns:  the inverse of the given explanation.
;;;
;;;  Called by:  discuss-explanation, and ???
;;;-----------------------------------------------------------------------------

(defun invert-explanation (explanation)
  (let (inverse)
    (setq inverse (invert-explanation-2 explanation nil nil nil))
    (compute-explanation-strength inverse)
    inverse))


;;;-----------------------------------------------------------------------------
;;;  Feature:  (invert-explanation-2  expl  root-expl  to-expl  backward)
;;;
;;;  Given:    -- expl, the explanation to be inverted.
;;;            -- root-expl, if non-nil, is the inverted explanation
;;;                  structure that will be the root of this entire inverted
;;;                  explanation.  It is nil until the root is found.
;;;            -- to-expl, if non-nil, is the explanation of the start-term of
;;;                  expl, and is therefore to be put into the to-terms of the
;;;                  inverse of expl in place of start-term (got that?).
;;;            -- backward, if non-nil, indicates that expl is an explanation
;;;                  of a from-term and that we therefore should keep the
;;;                  start-term the same in the inverse as in the original.
;;;
;;;  Returns:  the inverse of the given explanation.  For example, the
;;;            explanation [A causes B] has inverse [B sometimes caused by A].
;;;
;;;  Called by:  invert-explanation
;;;
;;;  Note:     Although this function is not long, it is one of the harder ones
;;;            to understand.  Good luck.
;;;-----------------------------------------------------------------------------

(defun invert-explanation-2  (expl  root-expl  to-expl   backward)

  (let* ((relation   (explanation-relation expl))
	 (start-term nil)
	 e-start-term
	 inverse-expl)

    ;; A spurious relation cannot be inverted, so it is returned unmodified.
    ;; Also, if root has already been established or if this is a backward
    ;; explanation, then just return it, don't invert it.
    (if (or (eq *verb-spurious* (relation-verb relation))
	    backward
	    root-expl)
	(return-from invert-explanation-2 expl))

    ;; Create the inverse explanation structure.  The remainder of this
    ;; function is dedicated to filling it in.
    (setq inverse-expl (make-explanation :from-terms nil
					 :to-terms nil))

    ;; For each antecedent of this explanation, make it a consequent of
    ;; the inverted explanation.

    (dolist (term (explanation-from-terms expl))
      (typecase term
	(node        ;; If this is the start-term of the forward explanation, and
	             ;; if it is an explained term (i.e., if to-expl is non-nil),
		     ;; then push the explanation onto to-terms.  Otherwise, just
		     ;; push the term itself onto to-terms.
	             (if (and to-expl (eq term (explanation-start-term expl)))
			 (push to-expl (explanation-to-terms inverse-expl))
			 (push term    (explanation-to-terms inverse-expl))))

	(explanation ;; This is an explained antecedent term (i.e., a backward
	             ;; explanation), so invert it and push it on to to-terms.
	             (push (invert-explanation-2 term nil nil t)
			   (explanation-to-terms inverse-expl)))

	(otherwise   (cerror "Ignore this term"
			     "invert-explanation: bad type."))))

    ;; For each consequent of this explanation, make it an antecedent of
    ;; the inverted explanation.

    (dolist (term (explanation-to-terms expl))
      (typecase term
	(node        (push term (explanation-from-terms inverse-expl))
	   	     (if (null start-term)
			 (setq start-term term))
		     (if (null root-expl)
			 (setq root-expl inverse-expl)))

	(explanation (setq e-start-term (explanation-start-term term))
		     (if (null start-term)
			 (setq start-term e-start-term))
		     (if (and (null root-expl)
			      (not (eq *verb-spurious* (relation-verb (explanation-relation term)))))
			 (progn
			   (push e-start-term (explanation-from-terms inverse-expl))
			   (setq root-expl (invert-explanation-2 term nil inverse-expl nil)))
			 (push (invert-explanation-2 term root-expl nil nil)
			       (explanation-from-terms inverse-expl))))

	(otherwise   (cerror "Ignore this term"
			     "invert-explanation: bad type."))))

    (setf (explanation-start-term inverse-expl) start-term)

    (let ((inv-rel (relation-inverse relation)))
      (setf (explanation-relation inverse-expl)
	    (if inv-rel
	        inv-rel
		*relation-noInverse*)))

    root-expl))


;;;-----------------------------------------------------------------------------
;;;  Function:  (compute-explanation-strength  expl)
;;;
;;;  Purpose:   Given an explanation, this function computes the overall
;;;             strength of the explanation.  Specifically, it runs through the
;;;             entire hierarchical structure of explanations and sets the
;;;             strength of each explanation component.
;;;-----------------------------------------------------------------------------

(defun compute-explanation-strength (expl)
  (if (null expl)
      (progn
	(format t "~%ERROR: COMPUTE-EXPLANATION-STRENGTH HAD NIL EXPL")
	(return-from compute-explanation-strength 1.0)))

  (let* ((relation (explanation-relation expl))
	 (strength (verb-strength (relation-verb relation))))

    ;; Adjust strength based on quantifiers.
    (dolist (quantifier (relation-quantifiers relation))
      (setq strength (* strength (quantifier-strength quantifier))))

    ;; Adjust strength based on minimum strength among the from-terms.
    (setq strength (* strength (apply #'min (mapcar #'get-strength (explanation-from-terms expl)))))

    ;; If no "to-terms", such as in a `spurious' relation, then all done.
    (if (explanation-to-terms expl)

	;; Adjust strength based on minimum strength among the to-terms.
	(setq strength (* strength (apply #'min (mapcar #'get-strength (explanation-to-terms expl))))))

    ;; Set the strength in the explanation (and return its value).
    (setf (explanation-strength expl) strength)))


;;;-----------------------------------------------------------------------------
;;;  Function:  (get-strength  thing)
;;;
;;;  Returns:   If "thing" is a term, it returns 1.0;
;;;             if "thing" is an explanation, it returns the computed strength
;;;             of that explanation.
;;;
;;;  FUTURE:    If we ever decide to attach a certainty or degree-of-belief
;;;             to terms, then if "thing" is a term it should return the
;;;             certainty rather than 1.0.
;;;-----------------------------------------------------------------------------

(defun get-strength (thing)
  (if (explanation-p thing)
      (compute-explanation-strength thing)
      1.0))


;;;-----------------------------------------------------------------------------
;;;  Function:  (truncate-explanation explanation forward-direction)
;;;
;;;  Purpose:   Given an explanation, this function returns a possibly truncated
;;;             version of the explanation.  A relation within the explanation
;;;             will be deleted if:
;;;             (a)  its strength is weak,
;;;             (b)  its verb is "has typical specialization", or
;;;             (c)  its verb is "has part".
;;;
;;;             These truncations ensure that remindings will be based on
;;;             at least moderate-strength relations and that remindings will
;;;             be to the more general categories.
;;;
;;;  Called by:  set-importance
;;;-----------------------------------------------------------------------------

(defun truncate-explanation (expl)
  (let ((new-expl  (truncate-explanation2 expl)))

    (cond ((null new-expl)     (return-from truncate-explanation nil))
	  ((eq new-expl expl)  (return-from truncate-explanation expl))
	  (t                   (compute-explanation-strength new-expl)
			       (return-from truncate-explanation new-expl)))))


(defun truncate-explanation2 (explanation)
  (let* ((to-terms    (explanation-to-terms explanation))
	 (relation    (explanation-relation explanation))
	 (verb        (relation-verb relation))
	 (new-terms   nil)
	 (modified    nil))

    ;; If weak relation, then truncate here.
    (if (< (relation-strength relation) *weak-relation*)
	(return-from truncate-explanation2 nil))

    ;; If specialization or has-part, then truncate here.
    (if (member verb (list *verb-hasTypicalSpec* *verb-hasPart*))
	(return-from truncate-explanation2 nil))

    ;; Possibly truncate any explained "to" terms.
    (dolist (object to-terms)
      (etypecase object
	;; This is a primitive term, so just save it in new-terms.
	(node         (push object new-terms))
	;; This is an explained term, so see if it can be truncated.
	(explanation  (let ((expl (truncate-explanation2 object)))
			;; Notice if anything has actually been modified.
			(if (not (eq expl object))
			    (setq modified t))
			;; If a non-nil explanation was returned ...
			(if expl
			    ;; then save the explanation
			    (push expl new-terms)
			    ;; else save the start-term of the (truncated) explanation.
			    (push (explanation-start-term object) new-terms))))))

    ;; If everything was truncated, then return nil.
    (if (null new-terms)
	(return-from truncate-explanation2 nil))

    ;; If no modifications, then return the original explanation,
    ;; thus avoiding the unnecessary creation of another explanation object.
    (if (not modified)
	(return-from truncate-explanation2 explanation))

    ;; Create a copy of the old explanation, but modify its list of to-terms.
    ;; The reason why a copy is made rather than just modifying in place the
    ;; existing explanation is that the original explanation might be used
    ;; for some additional purpose.
    (setq explanation (copy-explanation explanation))
    (setf (explanation-to-terms explanation) (nreverse new-terms))
    explanation))



;;;-----------------------------------------------------------------------------
;;;  Function:  (get-specific-explanation  feature  target  direction)
;;;
;;;  Given:     -- a feature and a target (category or exemplar) that are to be
;;;                related in an explanation; and
;;;             -- direction, which must be either 'FtoT, 'TtoF, or 'either,
;;;                specifying whether the returned explanation must be in the
;;;                form feature-to-target, target-to-feature, or either.
;;;
;;;  Returns:   an explanation linking the the feature and target, or nil if
;;;             the teacher would not give such an explanation.
;;;-----------------------------------------------------------------------------

(defun get-specific-explanation (feature target direction e-features)
  
  (let (expl inverse)
    (format t "~%You may revise any or all parts of the explanation.")
    (setq expl (enter-explanation))
    (if (null expl)
	(return-from get-specific-explanation nil))
    
    ;; Try to verify the explanation by the simple syntactic strategy
    ;; of noticing if the explanation begins with 'feature' and ends
    ;; with 'target' (or vice versa).  If so, then we don't have to
    ;; do the more time-consuming search through the category network.
    
    (let ((from-nodes  (get-leaves expl nil nil))
	  (to-nodes    (get-leaves expl t nil)))
      
      (cond  ;; Is this a feature-to-[category or exemplar] explanation?
	((and (member feature from-nodes)  (member target to-nodes))
	 (ecase direction
	   (either  (return-from get-specific-explanation expl))
	   (FtoT    (return-from get-specific-explanation expl))
	   (TtoF    (setq inverse (invert-explanation expl))
		    (return-from get-specific-explanation inverse))))
	
	;; Is this a [category or exemplar]-to-feature explanation?
	((and (member target from-nodes)  (member feature to-nodes))
	 (ecase direction
	   (either  (return-from get-specific-explanation expl))
	   (TtoF    (return-from get-specific-explanation expl))
	   (FtoT    (setq inverse (invert-explanation expl))
		    (return-from get-specific-explanation inverse))))))
    
    ;; Apparently this is a fragment of an explanation linking
    ;; feature with target, so we must search the category network.
    
    (let ((result (kbpm 'FtoT feature *importance-big* target e-features)))
      
      (setq expl (result-explanation result))
      (if (null expl)
	  (return-from get-specific-explanation nil))
      (ecase direction
	(either  (return-from get-specific-explanation expl))
	(FtoT    (return-from get-specific-explanation expl))
	(TtoF    (setq inverse (invert-explanation expl))
		 (return-from get-specific-explanation inverse))))))


(defun remove-relation ()
  (format t "~%remove-relation: Sorry, not implemented yet.~%"))

