;;; -*- Mode:Common-Lisp; Package:PROTOS; Syntax:COMMON-LISP; Base:10 -*-
;;;     Copyright (c) 1988, Daniel L. Dvorak.

(in-package 'protos)



;;;=============================================================================
;;;
;;;        M A I N    D A T A    S T R U C T U R E S    O F    P R O T O S
;;;   -------------------------------------------------------------------------
;;;
;;;  Contents:	  This file contains the definitions for the main structures of
;;;               Protos' "category network", plus their corresponding print
;;;		  functions.
;;;
;;;  Comment:	  Protos accumulates all of its learning in the "category
;;;		  network", a network of domain knowledge that includes
;;;		  features, categories, exemplars and the various connections
;;;		  among them (membership, relations, and differences).
;;;		  All of the Bareiss & Porter publications on Protos consis-
;;;		  tently refer to this network as the "category structure", but
;;;		  this implementation refers to it as the "category network" to
;;;		  avoid confusion with Common Lisp "structures" as defined by
;;;		  defstruct.
;;;
;;;  Overview:	  The following key points may be helpful in getting the 
;;;		  "big picture":
;;;
;;;  [Network:]	  -- A category network consists of terms connected by links.
;;;
;;;    [Terms:]	  -- A term can be a feature, a category, or an exemplar.
;;;		  -- Syntactically, a term can be a proposition (e.g. "female")
;;;		     or a predicate with arguments (e.g. "(fever mild)").
;;;
;;;    [Links:]	  -- Links are of several types: feature-of-exemplar links,
;;;		     exemplar-of-category links, remindings/censors, difference
;;;		     links, and relational links.
;;;		  -- A feature is "linked" to an exemplar by simply appearing
;;;		     in the exemplar's list of features.
;;;		  -- An exemplar is "linked" to a category by simply appearing
;;;		     in the category's list of exemplars.
;;;		  -- Remindings and censors emanating from a feature to a
;;;                  category or exemplar are stored in the feature's alist of
;;;                  remindings.
;;;		  -- A difference link (which points to a target exemplar) is
;;;		     stored in the "from" exemplar's list of differences.
;;;		  -- A relational link connects together two or more nodes of
;;;		     any type in a given relation, with an optional condition.
;;;                  
;;;
;;;    [Other:]	  -- An "explanation" is a path through the category network
;;;                  from one node to another.  Usually it is a linear path,
;;;                  but it can be an arbitrary subgraph of the network.
;;;                  Explanations are represented as a hierarchy of explanation
;;;                  structures.
;;;		  -- A "match" is the collection of explanations of how each
;;;		     exemplar feature matches some feature of the new case.
;;;		  -- Featural importances are stored in the 'importances' alist
;;;		     of the associated category or exemplar.
;;;
;;;=============================================================================
;;;
;;;  Inheritance:  The following inheritance diagram shows which structures
;;;		   inherit slots from other structures.  In this diagram,
;;;                "X <--- Y" means that structure X inherits slots from
;;;                structure Y.
;;;
;;;                    term <--- exemplar <--- feature <--- category <--- node
;;;
;;;                This diagram shows that the 'node' structure supplies slots
;;;                shared by category, feature, exemplar, and term.  Similarly,
;;;                'term' contains the union of all slots of node, category,
;;;                feature, and exemplar.
;;;
;;;  Pointers:     The following "pointer diagram" shows which structures
;;;		   refer to (point to) other structures.  In this diagram,
;;;		   "X ---> Y" or "X ---> list of Ys" means that structure X
;;;                has a slot whose value is a pointer to structure Y.
;;;
;;;			case ---> category
;;;			     ---> list of features
;;;
;;;			category ---> list of exemplars
;;;
;;;			exemplar ---> category
;;;				 ---> list of features
;;;				 ---> list of differences
;;;
;;;			difference ---> exemplar
;;;				   ---> list of features
;;;
;;;			match ---> exemplar
;;;			      ---> case
;;;			      ---> list of results
;;;
;;;			result ---> feature (of exemplar)
;;;			       ---> explanation (of equivalence to a case feature)
;;;
;;;			explanation ---> explanation (this is hierarchical)
;;;				    ---> list of conjoined antecedent terms
;;;				    ---> list of conjoined consequent terms
;;;				    ---> relation (of this step)
;;;
;;;                     node ---> list of relations
;;;
;;;			relation ---> list of conjoined antecedent terms
;;;				 ---> list of conjoined consequent terms
;;;				 ---> verb
;;;				 ---> list of quantifiers (if any)
;;;				 ---> condition (if any)
;;;
;;;			condition ---> category         (if specified)
;;;				  ---> list of features (if specified)
;;;
;;;			term ---> predicate (for terms that are predicates)
;;;
;;;			predicate ---> list of terms
;;;
;;;=============================================================================
;;;
;;;  Structures:  case
;;;		  category
;;;		  condition
;;;		  difference
;;;		  exemplar
;;;               explanation
;;;		  feature
;;;		  match
;;;               menu
;;;		  node
;;;               predicate
;;;		  relation
;;;               result
;;;		  term
;;;               transformation
;;;               verb
;;;
;;;  Functions:	  item-label, item-function, item-args (menu access functions)
;;;               print-alist
;;;               print-category
;;;		  print-condition
;;;		  print-difference
;;;               print-eterm
;;;               print-eterms
;;;		  print-exemplar
;;;               print-explanation
;;;		  print-feature
;;;		  print-match
;;;		  print-node
;;;		  print-node-names
;;;		  print-relation
;;;               print-result
;;;               print-term
;;;               print-transformation
;;;               set-exclusive-or
;;;=============================================================================


;;;-----------------------------------------------------------------------------
;;;  Structure:  node
;;;
;;;  Purpose:	 This structure defines the slots shared by all types of terms,
;;;              whether features, categories, and exemplars.
;;;-----------------------------------------------------------------------------

(defstruct node
  "structure for a node in the category network"
  name                             ; Name of this node.
 (abbrev     nil)                  ; Abbreviation, if any, of the name.
 (synonyms   nil)                  ; List of synonyms, if any.
 (relations  nil)                  ; List of relational links to other nodes.
 (variations nil)                  ; Alist of variations on this node, such as
                                   ;   "history-of" and "no".
 (creator    *username*)           ; ID of person who created this node.
 (comment    nil)                  ; Text comments.
)


(defun print-node (node stream depth)
  (do ((rels  (node-relations node) (cdr rels))
       (title "Relations:" "          "))
      ((endp rels))
    (format stream "~&   ~A    "  title)
    (print-relation (car rels) stream depth))
  (if (node-comment node)
      (format stream "~&   Comment:      ~A" (node-comment node)))
  (format stream "~&   Created by:   ~(~A~)" (node-creator node)))


;  (multiple-value-bind (second minute hour date month year)
;	  (decode-universal-time (node-timestamp node))
;    (declare (ignore second))
;    (format stream "~&   Created by:  ~A on ~D/~D/~D  ~D:~2,'0D"
;	(node-creator node) month date (mod year 100) hour minute)))




;;;-----------------------------------------------------------------------------
;;;  Structure:	 category
;;;
;;;  Purpose:	 A category represents a concept.  Some categories, such as
;;;		 "chair", may be represented through a collection of exemplars.
;;;		 Other categories, such as "vehicle", might be a common general-
;;;		 ization of other categories such as "car", "bus", "airplane",
;;;		 etc.  This 'category' structure serves both types.
;;;
;;;  Note:       The importance of a feature to a category is stored here in
;;;		 the importances alist.  If feature F has importance I to 
;;;		 category C, then the pair (F . I) appears in C's importances
;;;		 alist.
;;;-----------------------------------------------------------------------------

(defstruct (category  (:include node))
  "structure for a category"
 (exemplars nil)    ; List of exemplars of this category, ordered by
                    ;   prototypicality (i.e., by exemplar-typicality).  This
                    ;   is nil if this is a non-exemplar containing category.
 (importances nil)  ; Alist of (feature importance) entries for this category,
                    ;   sorted in decreasing order of importance.
 (faultvars nil)    ; List of fault variables for this category, i.e.,
                    ;   mathematical variables that are not responding to
                    ;   their inputs.
)


(defun print-category (cnode stream depth)
  (format stream "~&Category: ~@(~A~)~
		  ~%   Exemplars:   "
			(getname cnode))
  (print-node-names (category-exemplars cnode) stream ", ")
  (format stream "~%   Importances: ")
  (print-alist (category-importances cnode) t 1)
  (print-node cnode stream depth)
  (format stream "~%   Fault Vars:  ")
  (print-node-names (category-faultvars cnode) stream ", "))



;;;-----------------------------------------------------------------------------
;;;  Structure:	 feature
;;;
;;;  Purpose:	 A feature is an attribute of a case (a case consists of a set
;;;		 of features).  Features (terms) are created when they are first
;;;		 mentioned, either as a feature of a case or as a term in an
;;;		 explanation.
;;;
;;;  Note:       1. A feature structure is normally only instantiated for a feature 
;;;                 with symbolic values. Features with numeric values are
;;;
;;;                   EITHER          
;;;                    directly transformed into a symbolic feature, by the procedure
;;;                    given in the xform-function slot of the feature's predicate
;;;                    structure,
;;;                   OR
;;;                    (xform-function = nil) given as input to a computation function
;;;                   OR
;;;                    occasionally, instantiated as a 'numerical' feature with a symbolic
;;;                    interpretation of the number, i.e. as if the number was written 
;;;                    with letters. Example: (number-of-wheels 4).
;;;
;;;                 (See further description in the comments for the predicate structure)
;;;                
;;;                 If a symbolic feature value has been transformed from a
;;;                 numeric one, the latter is stored in the xformed-from
;;;                 slot of this structure. Examples:
;;;                          Feature-instance: (legs many)
;;;                              xformed-from: (legs 8)
;;;                          Feature-instance: (weather sunny mild rain strong) 
;;;                              xformed-from: (weather sunny 72 rain 30)
;;;
;;;                  (?: Necessary to save whether the feature is an input from
;;;                  the terminal, or a computed feature? )
;;;   
;;;              2. The reminding that a feature evokes to a category or exemplar
;;;		    is stored here in the remindings alist.  If feature F evokes
;;;		    a reminding of strength R to target T, then the pair (T . R)
;;;		    appears in F's remindings alist.  A feature may have remindings
;;;		    to several targets.  Censors, which are remindings having
;;;		    negative strength, are stored here in the same alist.
;;;
;;;-----------------------------------------------------------------------------

(defstruct (feature  (:include category))
  "structure for feature (whether predicate or proposition)"
 (remindings nil)	; Alist of remindings to categories and exemplars, if
			;  any.  Each reminding is of the form (node . strength)
 (of-exemplars nil)     ; List of exemplars that this is a feature of.
 (xformed-from nil)     ; The numerical value(s) from which this feature was transformed
)


(defun print-feature (fnode stream depth)
  (format stream "~&Feature: ~@(~A~)~
		  ~%   Remindings:  "
			(getname fnode))
  (print-alist (feature-remindings fnode) stream depth)
  (format stream "~%   Transformed from:  ")
  (print-alist (feature-xformed-from fnode) stream depth)
  (print-node fnode stream depth))


(defun print-alist (rems stream depth)
  (declare (ignore depth))
  ;; If any pairs to print ...
  (if rems
      (progn
	;; then print the set
	(format stream "~(~A~) ~4,2F"  (getname (caar rems)) (cdar rems))
	(dolist (rem (cdr rems))
	  (format stream ", ~(~A~) ~4,2F"  (getname (car rem)) (cdr rem))))
      ;; else just say [none].
      (format stream "none")))



;;;-----------------------------------------------------------------------------
;;;  Structure:  exemplar
;;;
;;;  Purpose:	 An exemplar is a case which Protos has processed and retained
;;;		 as an example of a category.  
;;;
;;;  Note:       Notice the 'importances' slot below that has been commented
;;;              out.  The original Protos allowed features to have idiosyn-
;;;              cratic importance to an exemplar, which would override any
;;;              importance of the feature to the category.  Ray subsequently
;;;              discarded this idea, so importances are now stored only in
;;;              the category.  However, all the code for handling idiosyn-
;;;              cratic importances remains here in CL-Protos but has been
;;;              commented out.  It can be located by scanning the system for
;;;              the string "exemplar-importances".
;;;-----------------------------------------------------------------------------

(defstruct (exemplar  (:include feature))
  "structure for exemplar"
  category		; Category of this exemplar.
 (features nil)  	; List of features of this case.
;; (importances nil)	; Alist of (feature . importance) entries for this
			;  exemplar, sorted in decreasing order of importance.
			;  This alist contains entries ONLY for features that
			;  are idiosyncratic to the exemplar, and therefore
			;  override the feature's importance to the category.
 (typicality nil)	; Prototypicality of this exemplar for its category.
 (differences nil)	; List of difference links to other exemplars.
)


(defun my-exemplar-p (node)    ; Returns non-nil if node is an exemplar.
  (exemplar-features node))

(defun my-category-p (node)    ; Returns non-nil if node is a category.
  (not (my-exemplar-p node)))

(defun print-exemplar (enode stream depth)
  (format stream "~%~A  (an exemplar)~
		  ~%   Category:     ~(~A~), typicality = ~4,2F~
		  ~%   Features:     "
			(getname enode)
			(getname (exemplar-category enode))
			(exemplar-typicality enode))
  (print-node-names (exemplar-features enode) stream ", ")
  ;;(format stream "~%   Importances: ")
  ;;(print-alist (exemplar-importances enode) t 1)
  (do ((diffs (exemplar-differences enode) (cdr diffs))
       (title "Diff Links:" "           "))
      ((endp diffs))
    (format stream "~%   ~A   " title)
    (print-difference (car diffs) stream depth))
  (print-node enode stream depth))


;;;-----------------------------------------------------------------------------
;;;  Structure:	 term
;;;
;;;  Purpose:	 A "term" is simply the combination of all slots of node,
;;;		 feature, category, and exemplar.  A given term may, at 
;;;		 different times, be treated as a feature, a category, or
;;;		 an exemplar.  For example, the term "engine" might be a
;;;		 feature of the case "my_car" and at the same time might be
;;;		 a category with specializations such as "4-cylinder_engine"
;;;		 and "Wankel_engine".  The category "4-cylinder_engine" or
;;;              its exemplar "20R_engine" might be used as a feature of the
;;;		 case "Porsche_car".
;;;
;;;		 Regardless of whether a term begins life as a feature or a
;;;		 category or an exemplar, it may later be used as any of the
;;;		 other two.  Thus, CL-Protos *NEVER* does a 'make-feature' or
;;;		 a 'make-category' or a 'make-exemplar'; it does a 'make-term'
;;;		 instead.
;;;
;;;  Design:	 The "term" structure inherits all of the slots of exemplar,
;;;		 feature, category, and node in this way:
;;;			term ------> includes ---> exemplar,
;;;			exemplar --> includes ---> feature,
;;;			feature ---> includes ---> category,
;;;			category --> includes ---> node.
;;;
;;;  QUESTION:   A category may be used as a feature, and an exemplar may be
;;;              used as a feature, but does it ever make sense for an
;;;              exemplar to be used as a category?  In theory, there may be
;;;              categories which can be defined with a single exemplar, in
;;;              which case it would be reasonable for the category and the
;;;              exemplar to be the same term.  (The current data structures
;;;              permit this, but the algorithm does not currently permit
;;;              exemplar and category to be the same term).
;;;-----------------------------------------------------------------------------

(defstruct (term  (:include exemplar))
  "structure for terms [feature + category]"
  (type nil)            ; type of term (only used in *diagnostic-model*)
  (predicate nil)       ; nil if the term is a proposition, otherwise this
                        ; points to the associated predicate structure.  For
                        ; example, the term "(fever mild)" points to the
                        ; predicate structure for "fever".
)


(defun print-term (term stream depth)
  ;; Make sure this is a term ...
  (if (not (term-p term))
      ;; then give error message.
      (progn
	(format stream "~&Sorry, this is a ~A, not a term!"
		(type-of term))
	(return-from print-term (values))))

  ;; GENERAL INFO ABOUT THIS TERM.
  (format stream "~%~A  (a term"  (node-name term))
  (if (my-exemplar-p term)
      (format stream "/exemplar"))
  (format stream ")")
  (if (node-abbrev term)
      (format stream "~%   Abbreviation: ~A" (node-abbrev term)))
  (if (node-synonyms term)
      (progn
	(format stream "~%   Synonyms:    ")
	(dolist (syn (node-synonyms term))
	  (write-char #\space stream)
	  (princ syn stream))))
  (if (and *diagnostic-model* (term-type term))
      (format stream "~%   Type:         ~@(~A~)"  (term-type term)))

  ;; EXEMPLAR SLOTS 
  (if (my-exemplar-p term)
      (progn
	(format stream "~%   Category:     ~(~A~), typicality = ~4,2F~
		        ~%   Features:     "
		(getname (exemplar-category term))
		(exemplar-typicality term))
	(print-node-names (exemplar-features term) stream ", ")
	(do ((diffs (exemplar-differences term) (cdr diffs))
	     (title "Diff Links:" "           "))
	    ((endp diffs))
	  (format stream "~%   ~A   " title)
	  (print-difference (car diffs) stream depth))))
	
  ;; CATEGORY SLOTS
  (if (category-exemplars term)
      (progn
	(format stream "~%   Exemplars:    " (getname term))
	(print-node-names (category-exemplars term) stream ", ")))
  (if (category-importances term)
      (progn
	(format stream "~%   Importances:  ")
	(print-alist (category-importances term) t 1)))
  (if (category-faultvars term)
      (progn
	(format stream "~%   Fault Vars:   ")
	(print-node-names (category-faultvars term) stream ", ")))

  ;; FEATURE SLOTS
  (if (feature-remindings term)
      (progn
	(format stream "~%   Remindings:   ")
	(print-alist (feature-remindings term) stream depth)))
  (if (feature-of-exemplars term)
      (progn
	(format stream "~%   Feature of:   ")
	(print-node-names (feature-of-exemplars term) t " ")))
  (if (feature-xformed-from term)
      (let ((outlist nil))
	(dolist (trm (feature-xformed-from term) outlist)
	  (push (string-downcase (format nil "~A" trm)) outlist))
	(format stream "~%   Trans.from:   ")
	(dolist (prt-feature outlist)
	  (format stream "~A~%" prt-feature))))
  (print-node term stream depth))



;;;-----------------------------------------------------------------------------
;;;  Structure:	 predicate
;;;
;;;  Background: Syntactically, a term can be expressed as either a proposition,
;;;              such as "pregnant", or a predicate with argument(s), such as
;;;              "(fever mild)" or "(weather sunny warm)".  In all cases, the
;;;              the term is represented in the knowledge base as an instance
;;;              of the term structure with its 'name' slot containing a
;;;              printable form of the term, such as "(fever mild)".
;;;
;;;              Every time that a term is entered by the teacher, whether as
;;;              part of a new case or part of an explanation, the function
;;;              'check-term-name' looks to see if the term already exists.
;;;              If it does, it simply returns the associated term structure.
;;;              Otherwise, it must create a new term structure, which is
;;;              returned.
;;;
;;;  Purpose:	 This structure serves four distinct purposes:
;;;
;;;              1.  For each predicate name (e.g., "fever", "weather", etc.),
;;;                  an instance of this structure is created to provide a
;;;                  directory that maps from each distinct set of argument
;;;                  values to the associated term structure.  For example,
;;;                  "(weather sunny warm)" maps to one term structure while
;;;                  "(weather sunny cool)" maps to another.  This directory
;;;                  is used by and expanded by 'check-term-name' as it
;;;                  encounters terms that are expressed as predicates.
;;;
;;;              2.  Each argument may optionally have an ordered quantity
;;;                  space that lists all its legal values showing their
;;;                  relative "nearness".  For example, the predicate "fever"
;;;                  has one argument whose quantity space is "(normal mild
;;;                  moderate severe profound)".  Protos may use this information
;;;                  to transformationally match "(fever moderate)" to
;;;                  "(fever mild)", with some strength lower than a perfect
;;;                  match, of course.  [Not currently implemented].
;;;
;;;              3.  Some predicates are qualitative variables that have
;;;                  relations to other qualitative variables.  For example:
;;;                  "level M+ pressure", where M+ is a positive monotonic
;;;                  constraint between level and pressure.  Thus, if we have
;;;                  "(level moderately-elevated increasing)", then we must
;;;                  match "(pressure moderately-elevated increasing)".
;;;
;;;              4.  Pointers to transformation structures involving a particular
;;;                  predicate name are stored in the input-xforms and output-
;;;                  xform slots. These slots serve as indices to the transformation
;;;                  functions, which are retrieved and applied  whenever 
;;;                  numerical features are encountered. A typical situation will
;;;                  be a numerical-to-symbolic  transformation, having
;;;                  the same value in the input-xforms slot as in the output-xform
;;;                  slot. If the value of the input-xform slot is a single 
;;;                  transformation structure, the value of the xform-output slot
;;;                  has to be identical - and the transformation has to be a
;;;                  numeric-to-symbolic one. The reason is that it does not make
;;;                  sense to 'compute' a numerical value from another single 
;;;                  numerical value.
;;;
;;;
;;;  Design:	 The "args" slot is an alist with a key for each possible value
;;;		 of the first argument of this predicate.  If this predicate has
;;;		 only one argument, then the associated value for each key is
;;;		 the pointer to the associated term node.  If this predicate has
;;;		 more than one argument, then the associated value for each key
;;;		 is another alist with a key for each value that the second
;;;		 argument has taken given the value of the first argument.
;;;		 These alists simply nest up to the depth of the number of
;;;		 arguments.
;;;
;;;		 As an example, consider a "weather" predicate where we have
;;;		 seen:
;;;			(weather sunny mild),
;;;			(weather sunny cool),
;;;			(weather cloudy mild).
;;;
;;;		 The "args" alist of the weather predicate will look like this:
;;;
;;;			((sunny  . ((mild . node1) (cool . node2)))
;;;			 (cloudy . ((mild . node3))))
;;;
;;;              where node1 is the term "(weather sunny mild)", for instance.
;;;
;;;  FUTURE:     Possible additions include:
;;;              -- specifying  default values for missing arguments, e.g.,
;;;                 "fever" defaults to "(fever moderate)".
;;;              -- an argument value could be another predicate-with-arguments
;;;                 (but we think this is unnecessary for most applications).
;;;-----------------------------------------------------------------------------

(defstruct (predicate  (:include node))
  (args nil)		; Alist keyed on argument-1 values.
  (input-xforms nil)    ; Transformation structures taking this predicate as input
  (output-xform nil)    ; Transformation structure giving this predicate as output
)


(defun print-predicate (pred stream depth)
  ;; Make sure this is a predicate ...
  (if (not (predicate-p pred))
      ;; then give error message.
      (progn
	(format stream "~&Sorry, this is a ~A, not a predicate!"
		(type-of pred))
	(return-from print-predicate (values))))
  
  (format stream "~%~A  (a predicate)"  (node-name pred))
  (if (node-abbrev pred)
      (format stream "~%   Abbreviation: ~A" (node-abbrev pred)))
  (if (node-synonyms pred)
      (progn
	(format stream "~%   Synonyms:    ")
	(dolist (syn (node-synonyms pred))
	  (write-char #\space stream)
	  (princ syn stream))))

  (if (predicate-args pred)
      (progn
	(format stream "~%   Terms:     ")
	(print-pred-args (list (getname pred)) (predicate-args pred))))

  (do ((rels  (predicate-relations pred) (cdr rels))
       (title "Relations:" "          "))
      ((endp rels))
    (format stream "~&   ~A    "  title)
    (print-relation (car rels) stream depth))
)

(defun print-pred-args (preface alist)
  (if (term-p (cdar alist))
      (format t "   ~A" (reverse (cons (caar alist) preface)))
      (print-pred-args (cons (caar alist) preface) (cdar alist))))



;;;-----------------------------------------------------------------------------
;;;  Structure:  transformation
;;;
;;;  Purpose:    Protos can only reason with symbolic values like big, warm,
;;;              normal, strong, etc. The transformation mechanism in CL-Protos
;;;              makes it possible to have the system assign a symbolic value
;;;              to a numeric feature-input. A numeric feature input is given to
;;;              the transformation procedure, which attempts to return a symbolic 
;;;              feature value to be used by the system.
;;;
;;;              An instance of this structure is constructed for each transformation 
;;;              procedure defined for some predicate.
;;;
;;;              The transformation procedure (a lisp function), the predicates
;;;              for which the transformation applies, as well as the actual
;;;              features involved are stored in this structure.
;;;
;;;              NOTE: PROTOS actually allows numerical features
;;;              (e.g. (number-of-wheels 4)), but the interpretation of the number
;;;              is as a symbol (e.g. (number-of-wheels four)). No numeric operations
;;;              are performed on the number.
;;;
;;;           THE TRANSFORMATION MECHANISM IN CL-PROTOS:
;;;
;;;              Only numerical values of features are transformed, i.e all
;;;              features applicable to transformations are of the form
;;;              (<feature-name> <value1> <value2> ...), where at least one of
;;;              <valueX> is numerical.
;;;
;;;              Protos handles two different kinds of transformation:
;;;
;;;              1. Numeric-to-symbolic (quantitative to qualitative) transformation
;;;
;;;              The transformation may be a pure 'value transformation' where
;;;              the feature-name (i.e. the predicate) is not changed,
;;;
;;;              Examples:
;;;              (body-temperature 98.6) transforms to (body-temperarture normal)
;;;              (blood-pressure 150 100) transforms to (blood-pressure high high)
;;;              (weather sunny 70 rain) transforms to (weather sunny mild rain)
;;;
;;;              Or, the predicate may change as well:
;;;
;;;              Example:
;;;              (pta-value 13.3) transforms to (air normal)
;;;              Or more complicated:
;;;              ((systolic-blood-pressure > 150) (diastolic-blood-pressure > 95) 
;;;              (heart-size normal)) transforms to Essential-Hypertension.              
;;;
;;;              2. Numeric-to-numeric computation
;;; 
;;;              A transformation may involve one or more computations, as
;;;              intermediate steps that compute a numerical feature from other
;;;              numerical features. 
;;;
;;;              Example (from audiology):
;;;              ((air-500hz 5) (air-1khz 15) (air-2khz 20)) computes to (pta-value 13.3)
;;;              
;;;             (4. Symbolic-to-symbolic transformations are handled by the explanation
;;;              links, and should not be regarded as part of the 'transformation
;;;              mechanism' as described here)
;;;
;;;           THE TRANSFORMATION STRUCTURE AND SLOTS
;;;
;;;              The transformation lisp-function has to be defined by the teacher for
;;;              each predicate that may take numeric values.
;;;              
;;;              The inputs to and the results from the transformation procedure
;;;              applied to a particular feature is held in the feature structure
;;;              (feature-xformed-from and feature-name, respectively). 
;;;                
;;;              The numerical feature transformed may be given as input from the
;;;              keyboard, or it may be computed on the basis of other numerical
;;;              features.
;;;
;;;              For numeric-numeric transformations we always have more than one
;;;              input feature, and the transformation procedure is usually a
;;;              mathematical calculation.
;;;
;;;              The transformation structure includes the node structure, in order
;;;              to inherit the 'creator' and 'timestamp' slot names. The 'relations'
;;;              slot on node is not used at present, but may be used to represent
;;;              relations between transformations - such as dependencies, shared
;;;              predicates or features, etc.
;;;
;;;
;;;-----------------------------------------------------------------------------

(defstruct (transformation (:include node))
  "structure of a numeric-to-symbolic transformation"

  type                 ; The type of transformation, one of (computation 
                       ; qq-transformation)
  in-predicates        ; Alist of (predicate . value-type) pairs necessary to perform
                       ; the transformation, e.g: ((fever . numeric) (pulse . symbolic)).
  out-predicate        ; The feature name resulting from the transformation.
  in-features          ; Current list of input features, no transformation is done before
                       ; the length of this list equals the length of the in-predicates
                       ; list.
  out-feature          ; The result after the transformation is done (symbolic feature).
  description          ; English text describing the transformation procedure.
  lisp-function        ; The actual lisp code performing the transformation.
)

(defun print-transformation (xform stream depth)
  (format stream "~%~A  (a transformation)~%" (transformation-name xform))
  (print-node xform stream depth)
  (format stream "~%   Type:             ~A~
		  ~%   In-predicates:    ~A~
                  ~%   Out-predicate:    ~A~
                  ~%   Description:      ~A~
                  ~%   Lisp Function: ~
                  ~%  ~A"
			(transformation-type xform)
			(transformation-in-predicates xform)
			(transformation-out-predicate xform)
			(transformation-description xform)
			(transformation-lisp-function xform)))




;;;-----------------------------------------------------------------------------
;;;  Structure:	 case
;;;
;;;  Purpose:	 A case is a featural description presented to Protos for
;;;		 classification.  A case may be preclassified by the teacher,
;;;		 but that information is not used by Protos in its normal
;;;		 attempts to classify the case itself.  Every case ends up
;;;		 with one of three dispositions:
;;;		    1.  It is discarded after being correctly and strongly
;;;			matched with an	existing exemplar.
;;;		    2.	It is retained as a new exemplar of its category.
;;;		    3.	It is merged into an existing exemplar of its category.
;;;
;;;-----------------------------------------------------------------------------

(defstruct case
  "structure for a case"
  name			; Name of this case (e.g., patient36).
 (category nil)		; Classification of this case .  This is NIL until
                        ;   either the teacher has preclassified the case or
                        ;   has approved Protos' classification of the case.
 (preclassify nil)      ; T if teacher preclassified the case.  This detail is
                        ;   retained in case of a "replay" of the KB.
  features		; List of features of this case.
  ofeatures             ; Original features, before being augmented with
                        ;  matched exemplars and categories as new "features".
 (disposition nil)      ; Final disposition of this case, which is either:
                        ; -- `(became ,exemplar) if made into an exemplar, or
                        ; -- `(merged ,exemplar) if merged into the exemplar, or
                        ; -- NIL if the case was abandoned.
 (comment nil)		; Text comments about this case.
 (creator *username*)	; ID of person who entered this case.
)


(defun print-case (case stream depth)
  (declare (ignore depth))
  (format stream "~%~A  (a case)~
		  ~%   Category:    ~(~A~)~
		  ~%   Features:    "
			(case-name case)
			(getname (case-category case)))
  (print-node-names (case-features case) stream ", ")
  (if (case-comment case)
      (format stream "~%   Comment:  ~A" (case-comment case))))




;;;---------------------------------------------------------------------------
;;;  Structure:	 relation
;;;
;;;  Purpose:	 A relation, such as "fire causes heat", is represented with
;;;		 a relational link between the two nodes ("fire" and "heat").
;;;		 A relation typically has a single antecedent and single
;;;		 consequent, but it may have multiple antecedents and multiple
;;;		 consequents (e.g., "heat and humidity cause muggy_weather").
;;;		 A relation always has a strength.  A relation may be
;;;		 universally true or it may be conditional (see the defstruct
;;;		 for 'condition').
;;;
;;;  Note:	 Whenever a relation is installed linking node X to node Y,
;;;		 the inverse relation is usually installed linking node Y
;;;		 to node X.
;;;---------------------------------------------------------------------------

(defstruct relation
  "structure of a relational link"
  verb                        ; Pointer to appropriate verb structure.
  from-nodes                  ; Names of the antecedent(s).
  to-nodes                    ; Names of the consequent(s).
  strength		      ; Strength of relation based on type and quantifier.
 (quantifiers nil)	      ; List of quantifier structures (if any).
 (condition nil)              ; Condition under which this relation is valid.
 (inverse nil)		      ; Pointer to inverse relational link.
 (creator *username*)         ; ID of person who entered this relation.
 (comment nil)                ; Text comments.
)


(defun print-relation (rel stream depth)
  (let ((from-nodes  (relation-from-nodes rel))
	(to-nodes    (relation-to-nodes rel))
	(condition   (relation-condition rel))
	(quantifiers (relation-quantifiers rel)))

    (if condition
	(print-condition condition stream depth))

    (print-node-names from-nodes stream " and ")

    (if quantifiers
	(dolist (quant quantifiers)
	  (format stream " ~(~A~)" (getname quant))))

    ;; Print the verb.
    (format stream " ~(~A~) " (getname (relation-verb rel)))

    ;; to-nodes may be nil if verb = "spurious"
    (if to-nodes
	(print-node-names to-nodes stream " and "))

    ;; Print strength of relation.
    (format stream ", strength = ~4,2F" (relation-strength rel))

    ;; If there is an associated comment, print it too on separate line.
    (if (and (relation-comment rel) (string/= "" (relation-comment rel)))
	(print (relation-comment rel)))
  ))





;;;-----------------------------------------------------------------------------
;;;  Structure:  verb
;;;
;;;  Purpose:    An instance of this structure is created for each verb in the
;;;              explanation language.  These instances, which are created at
;;;              compile-time and never modified during execution, contain
;;;              "everything you could possibly want to know" about each verb.
;;;-----------------------------------------------------------------------------

(defstruct verb
  name                   ; pretty verb name, such as "is caused by".
 (abbrev nil)            ; Abbreviation, if any, of the name.
  strength               ; strength (0 to 1) of this verb (without quantifiers).
  from-types             ; a list of legal types of antecedent terms.
  to-types               ; a list of legal types of consequent terms.
  from-proof             ; if non-nil, prove 1 antecedent, else prove all.
  to-proof               ; if non-nil, prove 1 consequent, else prove all.
  inverse                ; pointer to inverse verb's structure.
  iquantifier            ; if nil, then no quantifier applied to inverse verb.
                         ; if 'same, apply this verb's quantifier to inverse.
                         ; otherwise, apply this quantifier name to inverse.
)


;;;-----------------------------------------------------------------------------
;;;  Structure:  quantifier
;;;
;;;  Purpose:    An instance of this structure is created for each quantifier
;;;              in the explanation language.  These instances, which are
;;;		 created at compile-time and never modified during execution,
;;;		 contain "everything you could possibly want to know" about
;;;		 each quantifier.
;;;
;;;  Note:	 With only two slots, this information could have been stored
;;;		 in a simple alist.  The reason that a structure was created
;;;		 was in the expectation that further research into explanations
;;;		 might require additional knowledge about each quantifier.
;;;-----------------------------------------------------------------------------

(defstruct quantifier
  name                   ; pretty quantifier name, such as "usually".
 (abbrev nil)            ; Abbreviation, if any, of the name.
  strength               ; multiplicative strength in range 0 to 1.
)


;;;-----------------------------------------------------------------------------
;;;  Function:  print-node-names
;;;
;;;  Purpose:	This is a utility function used by some of the other print
;;;		functions to print the names of a list of one or more nodes.
;;;		If the list contains just one node, then its name is printed
;;;		unadorned.  If the list contains multiple nodes, then their
;;;		names are printed in the form "(Name1 and Name2 and ...)",
;;;		if separator = " and ".  Another likely separator is ", ".
;;;-----------------------------------------------------------------------------

(defun print-node-names (nodes stream separator)

  ;; Check for null nodes since 'spurious has no to-nodes.
  (if (null nodes)
      ;; then print "none"
      (format stream "none")
      ;; else if more than one node ...
      (if (cdr nodes)
	  ;; then print the names of all the nodes as a list
	  (progn
	    (format stream "(~(~A~)" (getname (car nodes)))
	    (dolist (node (cdr nodes))
	      (format stream "~A~(~A~)" separator (getname node)))
	    (format stream ")"))
	  
	  ;; else print the single node name unadorned.
	  (format stream "~(~A~)" (getname (car nodes))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  print-node-names2
;;;
;;;  Purpose:	This is a utility function used by print-condition
;;;		to print the names of a list of one or more nodes.
;;;             This function is different from print-node-names in that
;;;             it prints both a list with one node or a list with 
;;;             multiple nodes surrounded by parenthesis.  The node
;;;		names are printed in the form "(Name1 and Name2 and ...)",
;;;		if separator = " and ".  Another likely separator is ", ".
;;;-----------------------------------------------------------------------------

(defun print-node-names2 (nodes stream separator)

  ;; Check for null nodes since 'spurious has no to-nodes.
  (if (null nodes)
      ;; then print "none"
      (format stream "none")
      ;; else print the names of all the nodes as a list
      (progn
        (format stream "(~(~A~)" (getname (car nodes)))
        (dolist (node (cdr nodes))
          (format stream "~A~(~A~)" separator (getname node)))
        (format stream ")"))))

;;;-----------------------------------------------------------------------------
;;;  Structure:	 condition
;;;
;;;  Purpose:	 A condition is an optional component of a relational link.
;;;		 If a condition is present, it specifies under what situations
;;;		 the relation applies.  A condition may have one of four forms:
;;;
;;;		 1.  The relation applies only when Protos is trying to classify
;;;		     the new case into the given category.
;;;		 2.  The relation applies only when the new case has the given
;;;		     list of features.
;;;		 3.  The relation applies only when the exemplar being matched
;;;		     has the given list of features.
;;;		 4.  The relation applies only to arguments of the specified
;;;		     predicate (e.g., "if predicate is fever then mild is
;;;                  equivalent to moderate.
;;;
;;;  Note:	 The constructor function for this structure is explicitly
;;;		 named "build-condition".  This was necessary because Symbolics
;;;		 Common Lisp already has a function named "make-condition" for
;;;		 handling signals in Genera.
;;;-----------------------------------------------------------------------------

(defstruct (condition (:constructor build-condition))
  "structure of a condition"
  type              ; Type of condition.  Must be 'category, 'newcase, 'exemplar,
                    ;   or 'predicate.
 (category nil)     ; If type 'category, this is the classification category.
 (predicate nil)    ; If type 'predicate, this is the predicate.
 (features nil)     ; If type 'newcase or 'exemplar, this is the list of
                    ;   required features.
)
 

(defun print-condition (con stream depth)
  (declare (ignore depth))
  (if con
      (case (condition-type con)
	(category  (format stream "if category is ~A then "
			   (getname (condition-category con))))
	(newcase   (format stream "if case has features ")
		   (print-node-names2 (condition-features con) stream " and ")
		   (format stream " then "))
	(exemplar  (format stream "if exemplar has features ")
		   (print-node-names2 (condition-features con) stream " and ")
		   (format stream " then "))
	(predicate (format stream "if predicate is ~A then "
			   (getname (condition-predicate con)))))))


;;;-----------------------------------------------------------------------------
;;;  Structure:	 difference
;;;
;;;  Purpose:	 A difference link connects two exemplars (in the same or
;;;		 different categories) and records important features which
;;;		 discriminate them.  A difference link is stored in the "from"
;;;		 exemplar, so only the target exemplar need be specified in this
;;;		 structure.  A difference link is a one-way connection, so when
;;;		 Protos decides to annotate the differences between exemplars
;;;		 X and Y, it actually installs two links: one from X to Y and
;;;		 its inverse from Y to X.
;;;-----------------------------------------------------------------------------

(defstruct difference
  "structure of difference link"
  node             ; Target exemplar node of this difference link.
  features         ; List of features of the target exemplar not present in
                   ;   the current exemplar.
  sum		   ; Temporary storage for the sum of the importances of
		   ;   unmatched features.
)


(defun print-difference (diff stream depth)
  (declare (ignore depth))
  (let ((features (difference-features diff)))
    (format stream "(~@(~A~)"
	    (getname (car features)))
    (dolist (feature (cdr features))
      (format stream " ~@(~A~)" (getname feature)))
    (format stream ") ---> ~A" (getname (difference-node diff)))))



;;;-----------------------------------------------------------------------------
;;;  Structure:	 explanation
;;;
;;;  Purpose:	 This structure is used to represent an explanation, whether it
;;;		 be a feature-to-feature or feature-to-category or feature-to-
;;;		 exemplar explanation.  Explanations can be quite complex; some
;;;		 examples here show how even the complex explanations are
;;;		 represented with this simple structure.  We'll use the
;;;		 following shorthand notation:
;;;
;;;			[start-term:  (from-terms)  relation  (to-terms)]
;;;
;;;		 Thus, if we're looking for a feature-to-feature explanation
;;;		 starting from feature "A" and we find "A causes B", we'll show
;;;		 it here as [A:  (A) causes (B)].  Now, some others:
;;;
;;;		 "A causes B and C" where "A" is known to be true:
;;;		 ------>  [A:  (A) causes (B C)]
;;;
;;;		 "A causes B implies C" where only "A" is known to be true:
;;;		 ------>  [A:  (A) causes ([B: (B) implies (C)])]
;;;
;;;		 "B and C causes D, C results-from A" where "B" and "A" are
;;;		 known to be true:
;;;		 ------>  [B:  (B [C: (C) results-from (A)]) causes (D)]
;;;
;;;		 As these examples begin to illustrate, instances of this
;;;		 explanation structure can nest to any depth to explain why a
;;;		 "from" term or a "to" term is believed to be true (when it is
;;;		 not known to be trivially true).
;;;-----------------------------------------------------------------------------

(defstruct explanation
  start-term	; Pointer to the term which led us to try this relation in the
		;   first place.  For example, the relation "A and B causes C"
		;   could have a start-term of A or B.  Start-term must be a
		;   member of from-terms.
		;
  from-terms	; List of antecedent term(s) of the relation that were actually
		;   used in constructing this explanation (this must be a
		;   subset of relation-from-nodes).  Each element of this list
		;   must be either a pointer to a term or, if the term is not
		;   trivially true, a pointer to an explanation involving that
		;   term (whose identity will be stored in the start-term slot
		;   of the pointed-to explanation).
		;
  to-terms	; List of consequent term(s) of the relation that were actually
		;   used in constructing this explanation (this must be a
		;   subset of relation-to-nodes).  Each element of this list
		;   must be either a pointer to a term or, if the term is not
		;   trivially true, a pointer to an explanation involving that
		;   term (whose identity will be stored in the start-term slot
		;   of the pointed-to explanation).
		;
  relation	; Pointer to the relation connecting from-terms to to-terms.
		;
  strength	; Strength of the explanation.  If all of the from-terms and
		;   to-terms are "primitive" (i.e., not "explained"), then this
		;   is the strength of the relation composed with the certainty
		;   of the from-terms.  If any of the from-terms or to-terms are
		;   "explained", then the strengths of those explanations are
		;   included in the calculation of this strength.
)



;;;-----------------------------------------------------------------------------
;;;  Function:  (print-explanation  explanation)
;;;
;;;  Purpose:   This function prints an explanation in a reasonably readable
;;;             format.  For example, the explanations shown above in the
;;;             prologue to defstruct explanation would be printed as:
;;;                 "[A causes B]"
;;;                 "[A causes (B and C)]"
;;;                 "[A causes B implies C]"
;;;                 "[B, (B and [C results from A]) causes D]"
;;;-----------------------------------------------------------------------------

(defun print-explanation (expl)
  (if expl
      (print-explanation2 expl t)
      (format t " No explanation.")))

(defun print-explanation2 (expl stream)
  (let* ((from-terms  (explanation-from-terms expl))
	 (to-terms    (explanation-to-terms expl))
	 (relation    (explanation-relation expl))
	 (condition   (relation-condition relation))
	 (quantifiers (relation-quantifiers relation))
	 (from-expls  nil)
	 (to-expls    nil))

    ;; If conditional explanation, print the condition.
    (if condition
	(print-condition condition t 1))

    (setq from-expls (print-eterms from-terms stream))

    ;; If a quantifier was given, print it (or them, if more than one).
    (if quantifiers
	(dolist (quant quantifiers)
	  (format stream " ~(~A~)"  (getname quant))))

    ;; Print the verb.
    (format stream " ~(~A~) "  (getname (relation-verb relation)))

    (setq to-expls (print-eterms to-terms stream))

    ;; If there are any explained terms, print those explanations.
    (dolist (expl (nconc from-expls to-expls))
      (format stream ",~%   ")
      (print-explanation2 expl stream))

    (values)))


(defun print-eterms (eterms stream)
  (let ((expls nil)
	expl)
    ;; eterms may be null (if spurious).
    (if eterms
	(progn
	  (if (setq expl (print-eterm (car eterms) stream))
	      (push expl expls))
	  (dolist (eterm (cdr eterms))
	    (format stream " and ")
	    (if (setq expl (print-eterm eterm stream))
		(push expl expls)))))
    expls))


(defun print-eterm (eterm stream)
  (etypecase eterm
    (explanation
      (format stream "~A"  (getname (explanation-start-term eterm)))
      eterm)
    (node
     (format stream "~A"  (getname eterm))
     nil)))


;;;---------------------------------------------------------------------------
;;;  Structure:	 match
;;;
;;;  Purpose:	 An instance of this structure is created each time an
;;;		 exemplar-to-newcase match is computed.  The details of
;;;		 the match are stored here.  Some functions communicate
;;;		 through this structure (compare, explore-differences).
;;;---------------------------------------------------------------------------

(defstruct match
   exemplar		   ; Pointer to the exemplar node.
   newcase		   ; Pointer to the new case.
   similarity		   ; Overall similarity of the match.
   nth-root-of-similarity  ; nth-root of similarity value.
  (results nil)		   ; List of results for each exemplar feature
			   ;    (see the "result" structure below).
  (unmatched nil)	   ; Unmatched features of newcase.
)



;;;-----------------------------------------------------------------------------
;;;  Function:	print-match
;;;
;;;  Purpose:	This function prints a summary of the results of attempting to
;;;		match features of an exemplar to features of the new case.
;;;		The output begins with the most important exemplar feature and
;;;		proceeds to the least important -- this is done so that the
;;;		teacher can quickly see any unmatched or poorly matched import-
;;;		ant features.  Below is an example printout:
;;;
;;;                                                          CATEGORY X
;;;                    NEW CASE ABC                          EXEMPLAR X1
;;;             Line   Feature Name                   Sim.   Feature Name   Imp.
;;;             ----   ------------                   ----   ------------   ----
;;;               1     Feature_A =================== 1.00    Feature_A     0.87
;;;               2     Feature_B =================== 1.00    Feature_B     0.66
;;;               3     Feature_C ------------------- 0.70    Feature_Y     0.52
;;;               4                    unmatched ---> 0.60    Feature_Z     0.40
;;;               5     Feature_C
;;;                                Match Similarity = 0.42
;;;
;;;		Lines 1 and 2 are examples of syntactically identical matches.
;;;		Line 3 is an explained match, where the explanation has a
;;;		similarity of 0.70.  Line 4 shows an unmatched exemplar feature;
;;;		line 5 shows an unmatched newcase feature.
;;;
;;;		All exemplar features are printed first, in order of importance,
;;;		no matter how good or bad their match is.  Then, if there are
;;;		any unmatched new case features, they are printed.  The line
;;;		numbers are actually printed to permit easy selection of any
;;;		line for examination of its explanation.
;;;-----------------------------------------------------------------------------

(defun print-match (match)
  (let* ((n 0)
	 (exemplar (match-exemplar match))
	 (newcase  (match-newcase match))
	 (category (exemplar-category exemplar)))

    (format t                   "~%~51TCATEGORY: ~(~A~)"
	    (getname category))
    (format t "~%~5TCASE: ~(~A~) ~51TEXEMPLAR: ~(~A~)"
	    (case-name newcase)  (getname exemplar))
    (format t "~%~%~5TFEATURE~45TSim.  FEATURE~75TImp.")
    (format t "~%~79,1,0,'\~A" '\~)

    ;; Print each feature of exemplar, in order of importance.
    (dolist (result (match-results match))
      (print-result (incf n) result))

    ;; Print any features of the new case that remain unmatched.
    (dolist (feature (match-unmatched match))
      (format t "~%~3D  ~(~A~) <--- unmatched"
	      (incf n) (getname feature)))

    ;; Print overall match similarity.
    (format t "~%~79,1,0,'\~A" '\~)
    (format t "~%~26TMatch Similarity = ~4,2F~%"
	    (if *switch-nth-root*
		(match-nth-root-of-similarity match)
		(match-similarity match)))))



;;;-----------------------------------------------------------------------------
;;;  Structure:	 result
;;;
;;;  Purpose:    The knowledge-based pattern-matching function 'kbpm' returns
;;;		 in this structure the result of attempting to match a feature
;;;		 to some target.  (Instances of this structure are stored into
;;;		 the 'results' slot of the 'match' structure.)
;;;-----------------------------------------------------------------------------

(defstruct result

   feature		; Feature of the exemplar for which match was attempted.
   importance		; Importance of the feature to the exemplar or, if not
			;   present, importance of the feature to the category.
   type			; Summary of what was found (or not found) by the
			;   knowledge-based pattern matching algorithm.  See
			;   table below for legal values.
   quality		; Quality of the explanation linking the exemplar
			;   feature to the case feature.  This value depends on
			;   the 'type' of result, as shown in the table below.
			;   Any value less than 1.0 represents a weakening of
			;   the overall similarity of the exemplar-to-newcase
			;   match.
			;
			;         type                quality           expl?
			;     -----------        -------------------    -----
			;      identical  ----->  1.0                    no
			;      explained  ----->  (value from kbpm)      yes
			;      spurious   ----->  1.0                    yes
			;      excluded   ----->  0.0                    yes
			;      unmatched  ----->  1.0 - importance       no
			;
  (explanation nil)	; Explanation found (if any).  This will be an instance
                        ;   of the explanation structure.  The above table shows
)			;   for which 'type's an explanation will be returned.



;;;-----------------------------------------------------------------------------
;;;  Function:	(print-result  n  result)
;;;
;;;  Given:	-- n, a line number to be printed at the left margin;
;;;		-- result, an instance of the 'result' structure.
;;;
;;;  Purpose:	This is a subfunction of the print-match function.  It prints
;;;		a single line showing the result of knowledge-based pattern
;;;		matching from an exemplar feature to any feature of the newcase.
;;;-----------------------------------------------------------------------------

(defun print-result (n  result)

  (let* ((feature     (result-feature result))
	 (fname       (getname feature))
	 (importance  (result-importance result))
	 (icon        (icon-importance importance))
	 (quality     (result-quality result))
	 (explanation (result-explanation result))
	 sname)

    (case (result-type result)

      (identical
	  (format t "~%~3D  ~(~38,1,1,'=A~)  ~4,2F  ~(~A~)~75T~4A"
			 n (format nil "~(~A~) " fname) quality fname icon))

      (explained
	  (setq sname (getname (explanation-start-term explanation)))
	  (format t "~%~3D  ~(~38,1,1,'-A~)  ~4,2F  ~(~A~)~75T~4A"
			 n (format nil "~(~A~) " sname) quality fname icon))

      (excluded
	  (setq sname (getname (explanation-start-term explanation)))
	  (format t "~%~3D  ~(~38,1,1,'!A~)  ~4,2F  ~(~A~)~75T~4A"
			 n (format nil "~(~A~) " sname) quality fname icon))

      (spurious
	  (format t "~%~3D  ~29T  spurious ---> ~4,2F  ~(~A~)~75T~4A"
			 n quality fname icon))

      (unmatched
	  (format t "~%~3D  ~29T unmatched ---> ~4,2F  ~(~A~)~75T~4A"
			 n quality fname icon)))))


(defun icon-importance (importance)
  (cond ((> importance *importance-necessary*) '++++)     ; necessary
	((> importance *importance-high*)      '+++)      ; high
	((> importance *importance-moderate*)  '++)       ; moderate
	((> importance *importance-low*)       '+)        ; low
	(t                                     '-)))      ; spurious


;;;-----------------------------------------------------------------------------
;;;  Structure:  menu
;;;
;;;  Purpose:    An instance of this structure is created for each menu of
;;;              choices that may be presented to the user.  This structure is
;;;              used only by the function 'menu-select'.  The access functions
;;;              shown below following the structure are also used only by the
;;;              function 'menu-select'.
;;;-----------------------------------------------------------------------------

(defstruct menu
  label            ; Label/title of this menu.
 (repeat nil)      ; t if multiple selections are allowed before exiting menu.
 (redisplay nil)   ; If repeat=t then this controls whether the entire menu
		   ;   should be redisplayed for each selection.
 (displayvar nil)  ; t if value of associated variable is to be displayed.
 (twocol nil)      ; t if 2-column display for this menu.
  items            ; Alist of menu items of the form '(key . (label function args))
                   ;   where key is nil if the label is to be a subheading.
)

;;; Access functions for the elements of a menu item.
(defun item-label    (item)  (cadr  item))
(defun item-function (item)  (caddr item))
(defun item-args     (item)  (cdddr item))


;;;-----------------------------------------------------------------------------
;;;  Function:  (getname object)
;;;
;;;  Purpose:   This functions returns either the full name of the given object
;;;             or its abbreviation, depending on the value of the switch
;;;             *abbreviation-mode*.  If an object does not have an abbrevia-
;;;             tion, then its full name is returned.  Thus, this function
;;;             helps to customize the user interface -- a domain expert may
;;;             prefer to see abbreviations whereas a casual user may want to
;;;             see the full names of objects.
;;;-----------------------------------------------------------------------------

(defun getname (object)
  (typecase object

    (node        (if *abbreviation-mode*
		     (or (and (node-abbrev object)
			      (getname2 (node-abbrev object)))
			 (getname2 (node-name object)))
		     (getname2 (node-name object))))

    (verb        (if *abbreviation-mode*
		     (or (verb-abbrev object) (verb-name object))
		     (verb-name object)))

    (quantifier  (if *abbreviation-mode*
		     (or (quantifier-abbrev object) (quantifier-name object))
		     (quantifier-name object)))

    (otherwise   (format t "~%getname: Unexpected type ~A" (type-of object))
		 object)))


(defun getname2 (name)
  (typecase name
    (number      name)
    (symbol      name)
    (list        (mapcar #'getname3 name))
    (otherwise   (format t "~%getname2: Unexpected type ~A for ~A"
			 (type-of name) name)
		 name)))

(defun getname3 (name)
  (typecase name
    (number      name)
    (symbol      (if (boundp name)
		     (getname (eval name))
		     name))
    (list        (mapcar #'getname3 name))
    (otherwise   (format t "~%getname3: Unexpected type ~A for ~A"
			 (type-of name) name)
		 name)))

;;;-----------------------------------------------------------------------------
;;;  Parameters:  not, no, h-o, h/o, history-of
;;;
;;;  Note:        These parameters are used as predicate names, as in
;;;               "(history-of diabetes)" and "(no infection)" but are treated
;;;               specially since they imply a certain relationship to their
;;;               arguments, namely, "(no infection) mutually-exlusive-with
;;;               infection" and "(history-of diabetes) sometimes exhibits
;;;               diabetes".
;;;-----------------------------------------------------------------------------

(defparameter not (make-term :name 'not  :synonyms '(no)))
(defparameter no not)
(defparameter h-o (make-term :name 'h-o :synonyms '(history-of h/o)))
(defparameter h/o h-o)
(defparameter history-of h-o)
(defparameter xxx (make-term :name 'xxx))


;;;-----------------------------------------------------------------------------
;;;  SET-EXCLUSIVE-OR
;;;
;;;    Creates and returns a list consisting of elements in LIST1 and not in LIST2 
;;;  plus those in LIST2 and not in LIST1.
;;;    Normally, this function is provided, but on the TI-Explorers, the code for
;;;  this function performs a destructive set-exclusive-or using nconc.  Since 
;;;  Protos needs the function to be non-destructive, the function is 
;;;  redefined here for all types of machines so the TI-Bug will hopefully
;;;  not cause any more problems.
;;;-----------------------------------------------------------------------------

#-TI
(Defun SET-EXCLUSIVE-OR0 (&rest args)
  (apply #'set-exclusive-or args))

#+TI
(Defun SET-EXCLUSIVE-OR0 (list1 list2 &KEY key (test #'EQL) test-not)
  (SET-EXCLUSIVE-OR1 list1 list2 test key test-not))

#+TI
(Defun SET-EXCLUSIVE-OR1 (list1 list2 &OPTIONAL (test #'EQL) key test-not)
  (append                                                                            ;; replaced for nconc
    (REMOVE-IF #'(LAMBDA (x) (MEMBER* x list2 test key test-not)) (The List list1) :key key)
    (REMOVE-IF #'(LAMBDA (x) (MEMBER* x list1
				      ;; must reverse arguments
				      #'(lambda (x y) (funcall test y x))
				      key test-not)) (The List list2) :key key)))


