" (c) 1990, 1991 Copyright (c) University of Washington 
  Written by Stephen Soderland, Tony Barrett, Steve Hanks and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Top-level Data Structures and Global Variables for SPA

(in-package 'spa) 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Global Variables
;;; see also debug.lisp

(defvar *THE-QUEUE* '())                ; the actual search queue

(defvar       *TEMPLATES* nil)          ; list of pairs of dummy step, bindings
(defparameter *CBR-IPLAN-LIMIT* 5000)	; max # plans to generate
(defparameter *AGGRESSIVE* t)           ; DSW 1/5/92 only backchain on add
                                        ; list not effects
(defvar       *EXTEND-CHOOSE-FUN*)      ; chooses the next open/unsafe to extend 
(defvar       *RETRACT-CHOOSE-FUN*)     ; chooses the next open/unsafe to retract
(defparameter *AUTO-CHOOSE* t)          ; choose retraction manually or automatically
                                        ;   (see CHOOSE-FUNS.LISP for more on these 3).

(defvar *id-generator* 0)               ; provides unique ID numbers for plans etc.

;;;***********************************************

(defun init-id-generator (num)
  (when (> num *id-generator*)
    (setf *id-generator* (+ num 1)))
  ;; up to nearest 100
  (incf *id-generator* 100)
  (decf *id-generator* (mod *id-generator* 100)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SNLP-PLAN

(defstruct (SNLP-PLAN  (:print-function print-plan))
  steps                   ; list of STEP structures
  links                   ; list of LINK structures 
  unsafe                  ; list of UNSAFE structures 
  open                    ; list of OPEN structures
  ordering                ; list of ORDERING structures
  bindings                ; a CS strucure
  decisions               ; list of DECISION structures
  high-step               ; integer number of highest step in plan
  (high-id 0)             ; integer number of highest id (link, etc.) in plan
  rank                    ; used to cache rank value for best first search
  id                      ; used for printing
  )

(defun PRINT-PLAN (plan stream depth)
  (declare (ignore depth))
  (format stream "#<~a: S=~a; O=~a; U=~a>" 
          (snlp-plan-id plan)
          (- (length (snlp-plan-steps plan)) 2)
          (length (snlp-plan-open plan))
          (length (snlp-plan-unsafe plan))))

;;;************

(defun make-empty-plan (initial goal)
  (let ((init-conds (mapcar #'as-condx initial))
	(goal-conds (mapcar #'as-condx goal)))
    (make-snlp-plan 
     :steps (list (make-step :ID      0
			     :action  ':Initial-State
			     :postcond init-conds
			     :cost    0)
		  (make-step :ID      ':Goal
			     :action  ':Goal-State
			     :precond goal-conds
			     :cost    0))
     :open (mapcar #'(lambda (goal-cond) (make-open :condition goal-cond
						    :step-id ':Goal))
		   goal-conds)
     :bindings (make-cs)
     :high-step 0)))

;;;************

(defun get-step (step_id plan)
  (find step_id (snlp-plan-steps plan) :key #'step-id))

(defun get-link (link_id plan)
  (find link_id (snlp-plan-links plan) :key #'link-id))

(defun get-ordering (ordering_id plan)
  (find ordering_id (snlp-plan-ordering plan) :key #'ordering-id))

(defun get-decision (decision_id plan)
  (find decision_id (snlp-plan-decisions plan) :key #'decision-id))

(defun get-decision-establishing-link (link_id plan)
  (find-if #'(lambda (x) (and (eq (decision-type x) ':new-link)
			      (eq (decision-link x) link_id)))
	   (snlp-plan-decisions plan)))

(defun get-decision-establishing-step (step_id plan)
  (find-if #'(lambda (x) (and (eq (decision-type x) ':new-step)
			      (eq (decision-step x) step_id)))
	   (snlp-plan-decisions plan)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (STEP (:print-function print-step))
  id                      ; integer step number or :Goal
  action                  ; formula such as (puton ?X1 ?Y1)
  precond                 ; list of conditions such as (clear ?X1)
  postcond                ; list of post-conditions, which may be "add"s,
                          ; "delete"s or "unknown"s.
  (effect-count 0)        ; number of postconds that should not be
                          ; backtracked on
  producing-decisions     ; dec's which establish links from us
  consuming-decisions     ; dec's which establish links to us
  avoiding-decisions      ; decisions made to protect other links from us
                          ; (all are lists of decision ids)
  (cost 0)                ; cost of executing the step
  )

(defun PRINT-STEP (step stream depth)
  (declare (ignore depth))
  (format stream "#<STEP~a ~a>" (step-id step) (step-action step)))

;;;***********

(defun init-step? (step)
  (eq (step-id step) 0))

(defun init-step-id? (step_id)
  (eq step_id 0))

(defun goal-step? (step)
  (eq (step-id step) ':Goal))
       
(defun goal-step-id? (step_id)
  (eq step_id ':Goal))

;;;***********

(defun add-new-produce (plan step-id decision-id)
  ;; add decision to the protect list for link in plan.
  ;; we cannot modify the link structure in place, so copy
  ;; (see comments near top of snlp.lisp for explanation)
  (let ((part-of-step-list
	 (member step-id (snlp-plan-steps plan) :key #'step-id)))
    (unless part-of-step-list
      (error "attempt to modify non-existent step?"))
    (setf (car part-of-step-list)
          (copy-step (car part-of-step-list)))
    (push decision-id (step-producing-decisions (car part-of-step-list)))))

(defun add-new-consume (plan step-id decision-id)
  ;; add decision to the protect list for link in plan.
  ;; we cannot modify the link structure in place, so copy
  ;; (see comments near top of snlp.lisp for explanation)
  (let ((part-of-step-list
	 (member step-id (snlp-plan-steps plan) :key #'step-id)))
    (unless part-of-step-list
      (error "attempt to modify non-existent step?"))
    (setf (car part-of-step-list)
          (copy-step (car part-of-step-list)))
    (push decision-id (step-consuming-decisions (car part-of-step-list)))))

(defun add-new-avoid (plan step-id decision-id)
  (let ((part-of-step-list
	 (member step-id (snlp-plan-steps plan) :key #'step-id)))
    (unless part-of-step-list
      (error "attempt to modify non-existent step?"))
    (setf (car part-of-step-list)
          (copy-step (car part-of-step-list)))
    (push decision-id (step-avoiding-decisions (car part-of-step-list)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Accessors/setters for initial and goal conditions

(defun snlp-plan-initial-conditions (p)
  (let ((the-step (get-step 0 p)))
	(if (null the-step)
		(error "No initial step in this plan!")
		(step-postcond the-step))))

(defun set-plan-initial-conditions! (p conds)
  (let ((the-step (get-step 0 p)))
	(if (null the-step)
		(error "No initial step in this plan!")
		(setf (step-postcond the-step) conds))))

(defsetf snlp-plan-initial-conditions set-plan-initial-conditions!)


(defun snlp-plan-goal-conditions (p)
  (let ((the-step (get-step ':Goal p)))
	(if (null the-step)
		(error "No goal step in this plan!")
		(step-precond the-step))))

(defun set-plan-goal-conditions! (p conds)
  (let ((the-step (get-step ':Goal p)))
	(if (null the-step)
		(error "No goal step in this plan!")
		(setf (step-precond the-step) conds))))

(defsetf snlp-plan-goal-conditions set-plan-goal-conditions!)

;;;********************************************************************
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Subsidiary Data Structures for SPA plans

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The one remaining data type which has not been turned into a
;;; structure is the FORM: A FORM is an s-expression representing a
;;; proposition.  It is a list of atoms; the CAR of the atom is the
;;; proposition and the rest are the arguments.  The routines in
;;; OPERATE-ON-PLAN make use of this definition for a FORM.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; CONDXs are conditions, which are propositions (forms), plus a
;;; truth value which is one of :true :false or :unknown, and a flag
;;; 'handsoff', which indicates how the condition is to be used.
;;; {Unfortunately, CLOS claims the name "condition", so we have
;;;  to make do with an inferior one...}

(defstruct (condx (:conc-name cond-)
		  (:print-function print-condx)
		  (:constructor make-condx))
  form
  value
  (handsoff nil))

(defun print-condx (self stream depth)
  (declare (ignore depth))
  (format stream "[~a]" (condx-to-list self)))

(defun condx-to-list (cond)
  (let ((inner-form
         (cond ((eq (cond-value cond) :true)
                (cond-form cond))
               ((eq (cond-value cond) :false)
                (list 'not (cond-form cond)))
               (t
                (list 'unknown (cond-form cond))))))
    (if (cond-handsoff cond)
        (list 'handsoff inner-form)
        inner-form)))

;;; convert a list or a symbol to a condx.  lists of the form '(not x)
;;; or '(unknown x) are converted to condxs with false or unknown
;;; truth values respectively.  may also have '(handsoff x), which
;;; does the obvious thing.

(defun as-condx (thing)
  (cond ((condx-p thing)
	 thing)
	((and (consp thing) (eq (car thing) 'handsoff))
	 (as-handsoff (cadr thing)))
	((and (consp thing) (eq (car thing) 'not))
	 (make-condx :form (cadr thing) :value :false))
	((and (consp thing) (eq (car thing) 'unknown))
	 (make-condx :form (cadr thing) :value :unknown))
	(t
	 (make-condx :form thing :value :true))))

(defun as-handsoff (thing)
  (let ((x (as-condx thing)))
    (setf (cond-handsoff x) t)
    (values x)))

(defun cond-vars (condx)
  (let ((x (as-condx condx)))
    (remove-if-not #'plan-time-variable? (cdr (cond-form x)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; CONSTRAINTs establish equality or disequality between variables
;;; and constants.  Similarly to conditions, they have a printed
;;; list form (x y) for equality, and (not (x y)) for disequality

(defstruct (CONSTRAINT (:print-function print-cf)
		       (:conc-name cf-)
		       (:constructor make-constraint)
		       (:constructor eq-cf (var1 var2
					    &aux (equals t)))
		       (:constructor neq-cf (var1 var2
					     &aux (equals nil))))
  (equals t)  ; t for equality, nil for disequality
  var1 var2)

(defun print-cf (self stream depth)
  (declare (ignore depth))
  (write (cf-to-list self) :stream stream))

(defun cf-to-list (self)
  (if (cf-equals self)
      (list (cf-var1 self) (cf-var2 self))
      (list 'not (list (cf-var1 self) (cf-var2 self)))))

(defun as-cf (thing)
  (cond ((constraint-p thing) thing)
	((and (consp thing)
	      (eq (car thing) 'not)
	      (eq (length (cadr thing)) 2))
	 (neq-cf (caadr thing) (cadadr thing))) ; blech
	((and (consp thing)
	      (eq (length thing) 2))
	 (eq-cf (car thing) (cadr thing)))
	(t
	 (error "cannot interpret ~a as a constraint form" thing))))

(defun negate-cf (cf)
  (let ((real-cf (as-cf cf)))
    (if (cf-equals real-cf)
	(neq-cf (cf-var1 real-cf) (cf-var2 real-cf))
	(eq-cf (cf-var1 real-cf) (cf-var2 real-cf)))))

;;; cf-equiv? says if two constraints are equivalent to each other,
;;; that is they concern the same variables and have the same sense.
;;; the implementation uses #'eq, which relies on the fact that vars
;;; are symbols

(defun cf-equiv? (cf-a cf-b)
  (and (eq (cf-equals cf-a) (cf-equals cf-b))
       (or (and (eq (cf-var1 cf-a) (cf-var1 cf-b))
		(eq (cf-var2 cf-a) (cf-var2 cf-b)))
	   (and (eq (cf-var2 cf-a) (cf-var1 cf-b))
		(eq (cf-var1 cf-a) (cf-var2 cf-b))))))


;;; compare lists of bindings, regardless of order

(defun cf-list-equiv? (cf-list1 cf-list2)
  (let ((comp-list (copy-list cf-list2)))
    (and
     (dolist (cf1 cf-list1 t)
       (let ((cf2 (find cf1 comp-list :test #'cf-equiv?)))
	 (if cf2 (setf comp-list (delete cf2 comp-list))
	         (return nil))))
     (null comp-list))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (LINK (:print-function print-link))
  (id (incf *id-generator*))
  producer              ;; step-id of producer
  condition             ;; pre-condition {condx} satisfied by link
  consumer              ;; step-id of consumer of link
  bindings              ;; bindings unifying post-condition with pre-condition
                        ;; {list of constraint structures}
  ordering              ;; id of ordering imposed by this link, else nil
  protecting-decisions) ;; list of decisions which protect this link

(defun print-link (self stream depth)
  (declare (ignore depth))
  (format stream "#<LINK~a ~a ~a ~a>"
	  (link-id self)
	  (link-producer self)
	  (link-condition self)
	  (link-consumer self)))

(defun add-new-protect (plan link-id decision-id)
  ;; add decision to the protect list for link in plan.
  ;; we cannot modify the link structure in place, so copy
  ;; (see comments near top of snlp.lisp for explanation)
  (let ((part-of-link-list
	 (member link-id (snlp-plan-links plan) :key #'link-id)))
    (unless part-of-link-list
      (error "attempt to modify non-existent link?"))
    (setf (car part-of-link-list)
          (copy-link (car part-of-link-list)))
    (push decision-id (link-protecting-decisions (car part-of-link-list)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (UNSAFE (:print-function print-unsafe))
  link                 ;; id of threatened link
  clobber-step         ;; id of step which threatens it
  clobber-bind)        ;; list of bindings which cause the threat

(defun print-unsafe (self stream depth)
  (declare (ignore depth))
  (format stream "#<UNSAFE link~a step~a ~a>"
	  (unsafe-link self)
	  (unsafe-clobber-step self)
	  (unsafe-clobber-bind self)))

(defun display-unsafe (self plan &optional (stream *debug-io*) (indent 0))
  (indent-stream stream indent)
  (format stream "Unsafe: ~a threatens ~a under ~a"
	  (get-step (unsafe-clobber-step self) plan)
	  (get-link (unsafe-link plan) plan)
	  (unsafe-clobber-bind self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (OPEN (:print-function print-open))
  condition            ;; open precondition {condx}
  step-id)             ;; id of step which requires it

(defun print-open (self stream depth)
  (declare (ignore depth))
  (format stream "#<OPEN ~a step~a>" 
     (open-condition self) (open-step-id self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct (ORDERING (:print-function print-ordering))
  (id (incf *id-generator*))
  pred                 ;; step-id of predecessor step
  succ)                ;; step-id of successor step

(defun print-ordering (self stream depth)
  (declare (ignore depth))
  (format stream "#<ORDERING~a ~a ~a>" 
	  (ordering-id self)
	  (ordering-pred self)
	  (ordering-succ self)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; DECISION structures use different fields depending on the type:
;;;
;;; 1. (:new-step step cf-list)
;;;    Add new step.  cf-list is the set of constraints imposed by the
;;;    template for the step.
;;;
;;; 2. (:new-link link)
;;;    Establish the new link. Refer to the link structure to find
;;;    the bindings and ordering which go along with this link.
;;;
;;; 3. (:promote ordering link step unsafe), or
;;;    (:demote  ordering link step unsafe)
;;;    Establish the indicated ordering to protect link, which would
;;;    otherwise be clobbered by the step indicated.  unsafe holds
;;;    the unsafe treated by this decision, so we know what to
;;;    re-assert when we retract
;;;
;;; 4. (:separate link step cf-list unsafe)
;;;    Protect link from step by establishing constraints in
;;;    cf-list (all of which will be dis-equalities).

(defstruct (DECISION (:print-function print-decision))
  (id (incf *id-generator*))
  type                 ;; one of the key-words listed above
  step                 ;; id of step
  link                 ;; id of link
  ordering             ;; id of ordering
  cf-list              ;; list of bindings (constraint structures)\
  unsafe)              ;; an unsafe structure


(defun print-decision (self stream depth)
  (declare (ignore depth))
  (format stream "#<DECISION ~a " (decision-type self))
  (ecase (decision-type self)
    ((:new-step) (format stream "step~a" (decision-step self)))
    ((:new-link) (format stream "link~a" (decision-link self)))
    ((:promote :demote) (format stream "ordering~a" (decision-ordering self)))
    ((:separate) (format stream "step~a from link~a"
			 (decision-step self) (decision-link self))))
  (format stream ">"))

(defun display-decision (self plan &optional (stream *debug-io*) (indent 0))
  (indent-stream stream indent)
  (format stream "~a: " (decision-type self))
  (let ((step (get-step (decision-step self) plan))
	(link (get-link (decision-link self) plan)))
    (ecase (decision-type self)
      ((:new-step) (format stream "~a"
			   (if step
			       (instantiate-form (step-action step)
						 (snlp-plan-bindings plan))
			       (decision-step self))))
      ((:new-link) (format stream "~a"
			   (if link
			       (instantiate link (snlp-plan-bindings plan))
			       (decision-link self))))
      ((:promote :demote) 
       (format stream "~a to protect ~a"
	       (get-ordering (decision-ordering self) plan)
	       (or link (decision-link self))))
      ((:separate)
       (format stream "separate ~a from ~a by ~a"
	       (or link (decision-link self))
	       (or step (decision-step self))
	       (decision-cf-list self))))))

;;;**************************************************************************
;;;  Definitions for queue-related structures
;;;**************************************************************************

;;; Queue holds statistics info as well
(defstruct (Q (:print-function print-q))
  entries                               ; a linked list
  rank-fun                              ; low numbers mean good iplan
  cpu-time-created
  real-time-created
  iplans-enqueued
  iplans-dequeued
  max-length
  (iterations 0)
  (length 0)
  )

;;; Entries on the adapt-plan queue
(defstruct (QENTRY (:constructor make-qentry*)
	           (:print-function print-qentry))
  iplan                                 ; incomplete partially ordered plan
  dir                                   ; is entry being retracted or extended
  (stops nil)                           ; choices not to be retracted
  (rank nil)
  ;; the following fields are used for debugging only -- see *debug-save-tree*
  (parent nil)                          ; q-entry of retracted plan
  (retracted nil)                       ; decision retracted to yield parent
  (children nil)                        ; q-entries of extended plans
  (retracted-from nil)                  ; q-entry of plan we retracted from
  (resolved nil)                        ; open or unsafe considered for kids
  (next nil)                            ; q-entry of next-considered plan
  (prev nil)                            ; q-entry of prev-considered plan
  (order nil))                          ; order in which plan was considered


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun PRINT-Q (q &optional (s *standard-output*) ignore)
  (declare (ignore ignore))
  (format s "#<Q w/ ~a entries>" (q-length q)))

(defun PRINT-QENTRY (e &optional (stream *standard-output*) depth)
  (declare (ignore depth))
  (format stream "#<QEntry ~a (~a) Rank ~a>"
          (snlp-plan-id (qentry-iplan e))
          (if (eql (qentry-dir e) :retract) "Retract" "Extend")
          (qentry-rank e)))

(defun MAKE-QENTRY (&rest args)
  (let ((the-qe (apply #'make-qentry* args)))
    (set-names the-qe)
    (values the-qe)))

