(in-package 'spa)

;;;****************************************************************
;;;  The most basic entry point is 
;;;      (defun ADAPT-PLAN (init-plan 
;;;                          &optional (rank-fun #'crank) 
;;;                                    (option-chooser #'open-first))
;;;             ...)
;;;  defined in ADAPT.LISP.
;;;  There are two more reasonable entry points, for generative 
;;;  and adaptive planning, where the input and goal conditions 
;;;  come either directly as arguments or through the plan library, 
;;;  and for refit planning.
;;;

;;;*********************************************************8
;;;  Generative planning
;;;
;;;  Either supply the name of a problem that's in the 
;;;  library or a list of initials and goals (but not both).
;;;  If you supply explicit initial and goal, you are responsible 
;;;  for loading a domain (set of operator schemas) ahead of time.

;;;  Function returns two values:  the plan itself and the elapsed 
;;;  time to build it.

(defun plan-from-scratch (&key (problem-name nil)
                               (initial nil)
                               (goal nil)
                               (rank-fun #'crank))
  (init-spa)
  (when (and problem-name (or initial goal))
    (error "Can't give both a problem name and explicit initial/goal."))
  (when problem-name
    (let ((the-entry (find-lib-entry problem-name)))
      (if (null the-entry)
          (error "Can't find entry for ~a" problem-name))
      (setf initial (lib-entry-initial the-entry))
      (setf goal (lib-entry-goal the-entry))
      (load-domain (lib-entry-domain-name the-entry))))
  (domain-load-check)
  (time-stat "Start scratch planning for ~a" problem-name)
  (let* ((the-plan (adapt-plan (make-empty-plan initial goal) :rank-fun rank-fun))
         (elapsed-time (time-stat "... finished scratch planning")))
    (values the-plan elapsed-time)))

;;;;*************************************************************

;;;  Here provide either a problem name or initial/goal.  Also the lib-hint
;;;  is the name of a problem you think should be used as the library plan.
;;;  Function returns three values:  the plan, time to retrieve/fit, and 
;;;  time to adapt.

(defun plan-from-library (&key (problem-name nil)
                               (initial nil)
                               (goal nil)
                               (lib-hint nil) 
                               (rank-fun #'refine-first-then-reduce-open))
  (init-spa)
  (when (and problem-name (or initial goal))
    (error "Can't give both a problem name and explicit initial/goal."))
  (when problem-name
    (let ((the-entry (find-lib-entry problem-name)))
      (if (null the-entry)
          (error "Can't find entry for ~a" problem-name))
      (setf initial (lib-entry-initial the-entry))
      (setf goal (lib-entry-goal the-entry))
      (load-domain (lib-entry-domain-name the-entry))))
  (domain-load-check)
  (time-stat "Start library planning for ~a using library ~a " 
             problem-name 
             lib-hint)
  (let* ((retrieved-plan (retrieve-plan initial goal lib-hint))
         (retrieve-time (time-stat ".. finished retrieval, start adaptation"))
         (adapted-plan (adapt-plan retrieved-plan :rank-fun rank-fun))
         (adapt-time (time-stat ".. finished adaptation")))
    (values adapted-plan retrieve-time adapt-time)))
      

;;;************************************************************
;;; Initialization

(defun INIT-SPA ()
  (clear-names)
  (setf *dbst-last-plan* nil)
  (setf *dbst-most-retracted* nil)
  (setq *the-queue* (make-q))
  (values))

