(in-package 'spa)

;;;**************************************************************************
;;;   A FORM is an action description, condition proposition, etc. 
;;;   Precisely, it's either 
;;;       1.  a keyword atom like :INITIAL-STATE, or
;;;       2.  a constant like TABLE
;;;       3.  a list of atoms whose CAR is special (will never be 
;;;           transformed (variabilized or instantiated).


(defun operate-on-form (form form-fun destructive?)
  (funcall form-fun form destructive?))

(defun operate-on-forms (sexp-list form-fun destructive?)
  (cond
    (destructive?
     (let ((len (length sexp-list)))
       (dotimes (i len sexp-list)
         (setf (nth i sexp-list) 
               (operate-on-form (nth i sexp-list) form-fun t))))
     sexp-list)
    (t (mapcar #'(lambda (form) 
                   (operate-on-form form form-fun nil)) 
               sexp-list))))

;;; this procedure will be called internally by most 'form-fun's.
;;; 'item's, then are the atoms of a form: variables or constants.

(defun iterate-on-form (form item-fun destructive?)
  (cond
    ((keywordp form) form)
    ((atom form) (funcall item-fun form))
    (destructive?
     (let ((len (length form)))
       (dotimes (i len form)
         (if (not (= i 0))
             (setf (nth i form) (funcall item-fun (nth i form)))))))
    (t (cons (car form)
             (mapcar item-fun (cdr form))))))

;;;*************************************************************************
;;;  Operate-on can be used on all snlp data structures _except_ forms.
;;;
;;;  Apply the FORM-FUN to every form (s-expression) in the object.
;;;  FORM-FUN must take two arguments, a FORM and a DESTRUCTIVE? flag.
;;;  The OP-KEY tells what sort of operation we're doing---present
;;;  possibilities are :INSTANTIATE :COPY and :VARIABILIZE.  This
;;;  (redundant) parameter is sent to OPERATE-ON-CS in the hope that it
;;;  can perform the operation more efficiently if it knows what it's
;;;  doing.  If DESTRUCTIVE? is non-nil, the object will be copied down
;;;  to the s-expressions.  If DESTRUCTIVE? is nil, then the returned
;;;  object will be eq to the argument (and in most cases, so will
;;;  its slots).

(defun operate-on (thing form-fun destructive? &key op-key)
  (etypecase thing
    (snlp-plan  (operate-on-plan thing form-fun op-key destructive?))
    (step       (operate-on-step thing form-fun destructive?))
    (link       (operate-on-link thing form-fun destructive?))
    (open       (operate-on-open thing form-fun destructive?))
    (ordering   (operate-on-ordering thing form-fun destructive?))
    (unsafe     (operate-on-unsafe thing form-fun destructive?))
    (decision   (operate-on-decision thing form-fun destructive?))
    (condx      (operate-on-condx thing form-fun destructive?))
    (constraint (operate-on-cf thing form-fun destructive?))
    (cs         (operate-on-cs thing form-fun op-key destructive?))
    (list  ; list of one of the above types
     (if destructive?
	 (let ((len (length thing)))
	   (dotimes (i len thing)
	     (setf (nth i thing) 
		   (operate-on (nth i thing) form-fun t :op-key op-key)))
	   (values thing))
	 (mapcar #'(lambda (x) 
		     (operate-on x form-fun nil :op-key op-key)) 
		 thing)))))

;;;**************************************************************************
;;; Operate on snlp types:

(defun operate-on-plan (plan form-fun &optional
			              (op-keyword nil) (destructive? nil))
  (let ((the-plan (if destructive? plan (copy-plan-quick plan))))
    (setf (snlp-plan-steps the-plan)
	  (operate-on (snlp-plan-steps the-plan) form-fun destructive?))
    (setf (snlp-plan-links the-plan)
	  (operate-on (snlp-plan-links the-plan) form-fun destructive?))
    (setf (snlp-plan-unsafe plan)
	  (operate-on (snlp-plan-unsafe the-plan) form-fun destructive?))
    (setf (snlp-plan-open plan)
	  (operate-on (snlp-plan-open the-plan) form-fun destructive?))
    (setf (snlp-plan-ordering the-plan)
	  (operate-on (snlp-plan-ordering the-plan) form-fun destructive?))
    (setf (snlp-plan-bindings the-plan)
	  (operate-on-cs (snlp-plan-bindings the-plan)
			 form-fun op-keyword destructive?))
    (setf (snlp-plan-decisions the-plan)
	  (operate-on (snlp-plan-decisions the-plan) form-fun destructive?))
    (values the-plan)))

;;;*********

(defun operate-on-step (step form-fun destructive?)
  (let ((the-step (if destructive? step (copy-step step))))
    (setf (step-action the-step)
          (operate-on-form (step-action the-step) form-fun destructive?))
    (setf (step-precond the-step)
          (operate-on (step-precond the-step) form-fun destructive?))
    (setf (step-postcond the-step)
          (operate-on (step-postcond the-step) form-fun destructive?))
    ;; the following fields don't contain forms, but they must be
    ;; copied if destructive? == nil
    (unless destructive?
      (setf (step-producing-decisions the-step)
	    (copy-list (step-producing-decisions the-step)))
      (setf (step-consuming-decisions the-step)
	    (copy-list (step-consuming-decisions the-step)))
      (setf (step-avoiding-decisions the-step)
	    (copy-list (step-avoiding-decisions the-step))))
    (values the-step)))

;;;*********

(defun operate-on-link (link form-fun destructive?)
  (let ((the-link (if destructive? link (copy-link link))))
    (setf (link-condition the-link) 
          (operate-on-condx (link-condition the-link) form-fun destructive?))
    (setf (link-bindings the-link)
          (operate-on (link-bindings the-link) form-fun destructive?))
    ;; the following field doesn't contain forms, but must be
    ;; copied if destructive? == nil
    (unless destructive?
      (setf (link-protecting-decisions the-link)
	    (copy-list (link-protecting-decisions the-link))))
    (values the-link)))

;;;*********

(defun operate-on-unsafe (unsafe form-fun destructive?)
  (let ((the-unsafe (if destructive? unsafe (copy-unsafe unsafe))))
    (setf (unsafe-clobber-bind the-unsafe)
	  (operate-on (unsafe-clobber-bind the-unsafe) form-fun destructive?))
    (values the-unsafe)))

;;;*********

(defun operate-on-open (open form-fun destructive?)
  (let ((the-open (if destructive? open (copy-open open))))
    (setf (open-condition the-open) 
	  (operate-on-condx (open-condition the-open) form-fun destructive?))
    (values the-open)))

;;;*********

(defun operate-on-ordering (ordering form-fun destructive?)
  (declare (ignore form-fun))
  (if destructive? ordering (copy-ordering ordering)))

;;;*********

(defun operate-on-decision (decision form-fun destructive?)
  (let ((the-decision (if destructive? decision (copy-decision decision))))
    (ecase (decision-type the-decision)
      ((:new-step)
       (setf (decision-cf-list the-decision)
	     (operate-on (decision-cf-list the-decision)
			 form-fun destructive?)))
      ((:new-link))
      ((:separate)
       (setf (decision-cf-list the-decision)
	     (operate-on (decision-cf-list the-decision)
			 form-fun destructive?))
       (setf (decision-unsafe the-decision)
	     (operate-on-unsafe (decision-unsafe the-decision)
				form-fun destructive?)))
      ((:promote :demote)
       (setf (decision-unsafe the-decision)
	     (operate-on-unsafe (decision-unsafe the-decision)
				form-fun destructive?))))
    (values the-decision)))


;;;*********

(defun operate-on-condx (condx form-fun destructive?)
  (let ((the-condx (if destructive? condx (copy-condx condx))))
    (setf (cond-form the-condx)
	  (operate-on-form (cond-form the-condx) form-fun destructive?))
    (values the-condx)))

;;;*********

(defun operate-on-cf (cf form-fun destructive?)
  (let ((the-cf (if destructive? cf (copy-constraint cf))))
    (setf (cf-var1 the-cf)
          (operate-on-form (cf-var1 the-cf) form-fun destructive?))
    (setf (cf-var2 the-cf)
          (operate-on-form (cf-var2 the-cf) form-fun destructive?))
    (values the-cf)))
