;;; -*- Lisp -*-
(unless (find-package 'utils) (make-package 'utils))
(in-package 'utils)

;;; "Pure CL"   Character-oriented menu system

;;; The idea is to duplicate some reasonable subset of
;;; the old Symbolics menu system, w/o all the machine-specific
;;; aspects.

;;; Written by Ken Forbus, hacked 2/89 by Dan Weld

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(export '(cmenu-choose cmenu-choose-values))



;;;; Basic choice menu

;; When there is a discrete choice to make, this is the menu to use.
;; Items are specified as (<printed form> <value returned>). 

(defun cmenu-choose (item-list &optional (header "Choose one of")
			       (help "~% Sorry, no help available."))
  (cond ((null item-list) nil)
	((null (cdr item-list)) (cadar item-list)) ;; no choice, really!
	(t 
  (do ((len (length item-list))
       (answer nil)
       (result nil)
       (done? nil))
      (done? result)
    (show-basic-menu-choices header item-list)
    (setq answer (read-stuff))
    (cond ((integerp answer)
	   (cond ((or (< answer 1) (> answer len))
		  (format *standard-output* "~%  Must be between 1 and ~D." len))
		 (t (setq result (cadr (nth (1- answer) item-list))
			  done? t))))
	  ((or (eq answer :PUNT) (eq answer :QUIT) (eq answer :Q) (eq answer :A))
	   (setq done? t))
	  ((or (eq answer :HELP) (eq answer :H) (eq answer :?))
	   (format *standard-output* "~A" help))
	  (t (format *standard-output* "~%  Must be an integer between 1 and ~D."
		     len)))))))

(defun show-basic-menu-choices (header items)
  (format *standard-output* "~%~A:" header)
  (do ((i 1 (1+ i))
       (choice (car items) (car rest))
       (rest (cdr items) (cdr rest)))
      ((null choice))
    (format *standard-output* "~%  ~D: ~A" i (car choice))))

(defun print-cmenu-prompt (&optional (header ""))
  (format *standard-output* "~%~A>" header))

(defun read-stuff (&optional (header ""))
  (print-cmenu-prompt header)
  (let ((thing (read *standard-input*)))
    (case thing
      (? :HELP)
      (q :PUNT)
      (t thing))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Choosing variable values
;;;
;;; Once a variable has been chosen, select a value for it.
;;; Item lists here are (<printed version> <variable> <value-type>).
;;; It is assumed that all variables are bound.

;;; Supported datatypes: (data-driven, so new types may be added easilly)
;;;    :anything
;;;    :integer
;;;    :float
;;;    :string
;;;    :list-of-strings (for lists of files)
;;;    :one-of
;;;    :some-of
;;; Example:
;;; (cmenu-choose-values '(("Radix"   *r* :one-of (("Decimal" 10) ("Octal" 8)))
;;;                        ("Name"    *s* :string)
;;;                        ("Friends" *f* :some-of (("Jim" jim) ("Ann" ann)))))

(defun cmenu-choose-values (item-list &optional (header "Choose values for")
				      (help "Sorry, no help available."))
  (do ((len (length item-list))
       (answer nil)
       (done? nil))
      (done? nil)
	(format *standard-output* "~%~A:" header)
	(format *standard-output* "~%  ~D: ~A" 0 "QUIT!")
    (show-cvalues-menu-choices item-list)
    (setq answer (read-stuff))
    (cond
	 ((integerp answer)
	  (cond
	   ((= answer 0)
		(setq done? t))
	   ((or (< answer 1) (> answer len))
		(format *standard-output* "~%  Must be between 0 and ~D." len))
	   (t (twiddle-variable-value (nth (1- answer) item-list)))))
	 ((eq answer :PUNT) (setq done? t))
	 ((eq answer :HELP) (format *standard-output* "~A" help))
	 (t (format *standard-output* "~%  Must be an integer between 1 and ~D."
				len)))))

(defun show-cvalues-menu-choices (items)
  (do ((i 1 (1+ i))
       (choice (car items) (car rest))
       (rest (cdr items) (cdr rest)))
      ((null choice))
    (format *standard-output* "~%  ~D: ~A [~A]" i (car choice) 
			(symbol-value (cadr choice)))))


(defun twiddle-variable-value (var-spec)
  (let ((val (funcall (lookup-cvalues-type-reader (caddr var-spec)) var-spec)))
    (unless (eq val ':NO-VALUE-PROVIDED)
      (set (cadr var-spec) val))))

(defvar *cvalues-types* nil)

(defun lookup-cvalues-type-reader (type)
  (let ((entry (assoc type *cvalues-types*)))
    (if entry (cdr entry) (function read))))

(defmacro Define-cvalues-type (type function)
  `(let ((entry (assoc ',type *cvalues-types*)))
     (unless entry
       (push (setq entry (cons ',type nil)) *cvalues-types*))
     (setf (cdr entry) (function ,function))))

(defun read-list-of-unempty-strings ()
  (do ((answer nil)
       (current nil))
      (nil)
    (setq current (read-line))
    (when (string= current "")
      (return (nreverse answer)))
    (push current answer)))

(defun read-integer ()
  (do ((answer nil))
      (nil)
    (setq answer (read))
    (cond ((integerp answer) (return answer))
	  (t (format t "~% Must be an integer...try again.")
	     (print-cmenu-prompt)))))

(defun read-float ()
  (do ((answer nil))
      (nil)
    (setq answer (read))
    (cond ((floatp  answer) (return answer))
	  (t (format t "~% Must be a flonum...try again.")
	     (print-cmenu-prompt)))))

(Define-cvalues-type :anything
		     (lambda (var-spec)
		       (format t "~% New value for ~A:" (car var-spec))
		       (print-cmenu-prompt)
		       (read)))
(Define-cvalues-type :string
		     (lambda (var-spec)
		       (format t "~% New value for ~A:" (car var-spec))
		       (print-cmenu-prompt)
		       (read-line)))
(Define-cvalues-type :list-of-strings
		     (lambda (var-spec)
		       (format t "~% New value for ~A:" (car var-spec))
		       (print-cmenu-prompt)
		       (read-list-of-unempty-strings)))
(Define-cvalues-type :integer
		     (lambda (var-spec)
		       (format t "~% New value for ~A:" (car var-spec))
		       (print-cmenu-prompt)
		       (read-integer)))
(Define-cvalues-type :float
		     (lambda (var-spec)
		       (format t "~% New value for ~A:" (car var-spec))
		       (print-cmenu-prompt)
		       (read-float)))
(Define-cvalues-type :one-of
             (lambda (var-spec)
			   (let ((choices (cadr (member :one-of var-spec))))
				 (cmenu-choose
				  choices 
				  (format nil "~% Select new value for ~A:"
						  (car var-spec))))))

;;; var can have zero or more values from pos-val list
;;; Allow user to
(Define-cvalues-type :some-of
  (lambda (var-spec)
	(format *standard-output* "~%Possible Values:")
	(let ((choices (cadr (member :some-of var-spec))))
	  (show-cvalues-possible-choices choices)
	  (format *standard-output*
		    "~% Enter changes for ~A (index to add, negative num to delete):"
			(car var-spec))
	  (new-some-of-val (read-some-of-list (length choices))
					   (symbol-value (cadr var-spec)) choices))))

(defun SHOW-CVALUES-POSSIBLE-CHOICES (items)
  (do ((i 1 (1+ i))
       (choice (car items) (car rest))
       (rest (cdr items) (cdr rest)))
      ((null choice))
    (format *standard-output* "~%  ~D: ~A " i (car choice))))

;;; Read a list of integers and make sure that their abs > 0 and < n+1
(defun READ-SOME-OF-LIST (n)
  (do ()
      (nil)
    (multiple-value-bind (l ok?)
						 (convert-string-to-list-of-integers (read-line))
     (cond
	  ((not ok?)
	   (format *standard-output* "~% Must be a line of integers...try again.")
	   (print-cmenu-prompt))
	  ((member nil (mapcar #'(lambda (i) (and (<= (abs i) n) (/= 0 i))) l))
	   (format t "~%  Must be an integer with abs. val. between 1 and ~D." n)
	   (print-cmenu-prompt))
	  (t (return l))))))

;;; Given a correctly input list, the previous val and choices (which decodes
;;; the input list), return the new value
(defun NEW-SOME-OF-VAL (l val choices)
  (let ((pvals (mapcar #'cadr choices))
		(ret-val val))
	(dolist (i l)
	   (cond
		((and (> i 0)					                  ; add
			  (not (member (nth (- i 1) pvals) ret-val))) ; not already there
		 (push  (nth (- i 1) pvals) ret-val)) 
		((and (< i 0)					                  ; delete
			  (member (nth (- (abs i) 1) pvals) ret-val))       ; there now
		 (setq ret-val (delete (nth (- (abs i) 1) pvals) ret-val)))))
	ret-val))


;;; Returns nil on error
;;; Handles negative numbers
(defun SOFT-STRING-TO-INTEGER (string &optional (radix 10))
  (let ((str (string-left-trim '(#\-) string))
		(s (if (find #\- string) -1 1)))
  (do ((j 0 (+ j 1))
	   (n 0 (+ (* n radix)
			   (or (digit-char-p (char str j) radix)
				   (return nil)))))
	  ((= j (length str)) (* s n)))))

;;; "3 6 1" => (3 6 1)
;;; returns two things: list and error-flag (nil means bad char encountered)
(defun CONVERT-STRING-TO-LIST-OF-INTEGERS (str)
  (do* ((l nil l)						; return this
		(s (string-left-trim '(#\Space #\Tab) (concatenate 'string str " "))
		   (string-left-trim '(#\Space #\Tab) (subseq s i)))
		(i (position #\Space s) (position #\Space s)))
	   ((null i) (values (nreverse l) t))                ; happy ending
	(let ((n (soft-string-to-integer (subseq s 0 i))))
	  (if n
		  (push n l)
		(return (values nil nil))))))

