#|
*******************************************************************************
PRODIGY/EBL Module Version 2.0  
Copyright 1989 by Steven Minton.

The PRODIGY/EBL module was designed and built by Steven Minton. Thanks
to Jaime Carbonell and Craig Knoblock for their helpful advice. Andy
Philips contributed to the version 2.0 modifications.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#

(proclaim '(special *HIST-NUM* *STATIC-PREDS* *CLOSED-PREDS* *HISTS*
	     *ONLY-TOP-GIS* *TARGET-CONCEPTS*))
	     	

(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types")
	(load-path *EBL-PATH* "ebl-data-types"))

; Code for augmenting history (trace) information at each node.

; This code includes the target concept recognizer code, which has
; been proceduralized and spread out into a couple of different places.
; This is now recognized to have been a mistake, and if I ever do
; another version of this code, it'll be one of the first things to change.


; ADD-HISTORY-INFO-TO-TREE is called by the observer (see obs.lisp) to 
; to add history information to the nodes for learning. First, the
; system resets the candidate-goals to the default candidate goals. 
; (TAKEN OUT FOR NOW, makes debugging easier)- SNM
; (This property is modified by the planner as it runs, so we have to reset 
; it.) Secondly, at  every node, the time spent at that 
; subtree is added up in accumulate-node-times.
; Finally, each node in the tree is marked as to whether a goal-interaction
; can be learned there. (The Target-concept recognizer for goal-interaction
; is currently built into the code). Note: Formerly success/failure was
; also marked in tree at this point, but now this is taken care of earlier.

; Note that accumulation-node-times is not run
; on nodes that have been added subsequent to the original problem solving
; episode ("restart nodes") to verify interaction.

(defun add-history-info-to-tree ()
      (reset-candidate-goals (eval 'N1))
      (accumulate-node-times (eval 'N1))
      ; tree has already been marked for success/failure, now mark interactions
      (mark-gi-instances-in-tree (eval 'N1)))
	  
; Resets the candidate-goals to the default candidate goals.
; (This property is modified by the planner as it runs, so we have to reset ;
; it for the learning to be able to get this info.) 

(defun reset-candidate-goals (n)
   (setf (node-candidate-goals n)
 	 (alt-unmatched-conds (node-generating-alt n)))
   (dolist (c (node-children n))
         (reset-candidate-goals c)))



; Note: node-time should never be nil if the node has children

(defun accumulate-node-times (n)
  (or (node-cum-time n)
      (let ((cum-time (or (node-time n) 0)))
	(dolist (c (node-children n))
		(setq cum-time (+ cum-time (accumulate-node-times c))))
	(setf (node-cum-time n) cum-time)
	cum-time)))
      

; similar to accumulate node time, but does not keep totals.
; useful for added nodes.
(defun get-total-node-time (n)
    (or (node-cum-time n)
	(g-loop (init cs (node-children n) c nil 
		    ret-val (or (node-time n) 0))
	      (while (setq c (pop cs)))
	      (do  (setq ret-val (+ ret-val (get-total-node-time c))))
	      (result ret-val))))


; Target concept recognizer function for SUCCESS.
; called from functions, interaction-check, utility-validation, expand-all.

; check children in order, that way if multiple success children, first
; success becomes success-child

(defun mark-success-in-tree (n)
  (g-loop (init children (node-children n) c nil)
	  (while (setq c (pop children)))
	  (do (cond ((mark-success-in-tree c)
		     (setf (node-success-child n) c)
		     (setf (node-succeeded n) t))))
	  (result (node-succeeded n)))
  (node-succeeded n))

; Target concept recognizer function for Failure.
; called from functions, interaction-check, utility-validation, expand-all.

(defun mark-failure-reasons-in-tree (n)
  (g-map (c in (node-children n))
	 (do (mark-failure-reasons-in-tree c)))
  (cond ((node-succeeded n) nil)  ; do nothing
	((node-failure-reason n) nil) ; do nothing
	((node-reject-node-hst n)
	 (setf (node-failure-reason n) '(node-reject-rule-fired)))         
	; result is indeterminate, never explored node.
        ((not (node-expanded n)) nil)				
	; result is indeterminate, never finished exploring node.
	((node-alternatives n) nil)
	; node result is indeterminate because interaction found before
        ; we explored node. 
	((and (node-added-after-restart n) 
	      (or (node-protection-violation n)
		  (node-prerequisite-violation n)))
	 nil) 
        ; all children failed, set failure reason
	((g-loop (init c nil cs (node-children n))
		 (while (setq c (pop cs)))
		 (do (or (node-failure-reason c)
			 (return nil)))
		 (result t))
	 (setf (node-failure-reason n)
	       '(out-of-alts)))))

; MARK-GI-INSTANCES-IN-TREE assumes the tree has already been marked for 
; success and failure.
; Code sets gi-label to 'no, 'yes or 'above or 'maybe. 'No means
;  that the node does not have a goal-interaction below,
;  'yes means it does, above means there's a gi-above.
;  'maybe means that all expanded nodes exhibit a gi.
;  if a node is a failure, it isnt necessarily a gi.
;  failure means there was a failure, supporting-failure
;  means there was a failure, and you are using that failure
;  to learn a goal-interaction.

; To be a gi, either it has to be exhibited directly
; at the node, or exhibited by at least 1 child node (or below)
; and all other children are failures.

; yes's should propagate up, so should maybes (priority)

(defun mark-gi-instances-in-tree (node)
  (cond ((node-gi-label node)) ; already labeled
	; hack to aviod case where you have a goal repeat and goal-interaction
       	((eq (car (node-failure-reason node)) 'goal-repeat) 
	 (setf (node-gi-label node) 'failure))
	((or (hist-protection-violation node t)
	     (and (not-top-level-node node)
		  (hist-prerequisite-violation 
	 	; wrong, doesn't get goal for applied nodes, but ok.
                    node (car (alt-unmatched-conds 
			         (node-generating-alt node))) t))
	     (and (is-top-level-node node)
                  (some #'(lambda (g) (hist-prerequisite-violation node g t))
                      (alt-unmatched-conds (node-generating-alt node)))))
	 (above-label-children node)
	 (setf (node-gi-label node) 'yes))
	((and (not (node-children node))
	      (node-failure-reason node))
	 (setf (node-gi-label node) 'failure))
	((and (node-succeeded node)
	      (or (null (node-children node))
		  (eq (node-gi-label (node-success-child node)) 'no)))
	 (setf (node-gi-label node) 'no))
	(t (g-map (c in (node-children node))
		  (do (mark-gi-instances-in-tree c)))
           (label-gi-depending-on-children node))))

(defun above-label-children (node)
    (g-map (c in (node-children node))
	   (do (progn (setf (node-gi-label c) 'above)
		      (above-label-children c)))))

; failures can be labeled as failures or supporting-failures --
; those used to immediately prove a gi

(defun label-gi-depending-on-children (node)
  (prog (no-found yes-found maybe-found)
	(g-loop (init cs (node-children node) c nil child-gi-label nil)
		(while (setq c (pop cs)))
		(do (setq child-gi-label (node-gi-label c))
		    (cond ((eq child-gi-label 'no)
			   (setq no-found t))
			  ((eq child-gi-label 'yes)
			   (setq yes-found t))
			  ((eq child-gi-label 'maybe)
			   (setq maybe-found t)))))
	(cond ((and (not no-found)
		    (or maybe-found
			(not (node-expanded node))
			(node-alternatives node)))
	       (setf (node-gi-label node) 'maybe))
	      (no-found 
	       (setf (node-gi-label node) 'no)
	       (if yes-found ; maybe setting extras, but thats oK
		   (dolist (c (node-children node))
			   (cond ((eq 'failure (node-gi-label c))
				  (setf (node-gi-label c)
					'supporting-failure))))))
	      (yes-found
	       (setf (node-gi-label node) 'yes)
	       (dolist (c (node-children node))
		       (cond ((eq 'failure (node-gi-label c))
			      (setf (node-gi-label c)
				    'supporting-failure)))))
	      ((node-failure-reason node)
	       (setf (node-gi-label node) 'failure)))))
				     




; Adds information to node that is not directly recorded by planner.
; Puts current-goal at failed nodes. 
; Calls target concept recognizers for failure, success, goal-interactions

(defun augment-node-trace (node)
    (prog ()
	  (let ((freason (car (node-failure-reason node))))
	       (cond ((null freason))
		     ((eq freason 'goal-repeat)
		      (find-gr-condition node)
		      (setf (node-current-goal node)
			    (cadr (node-failure-reason node))))
		     ((eq freason 'goal-already-achieved) 
		      (format t "~%Warning: goal already achieved, code shaky"))))
	  (label-for-failure-target-concepts node)
	  (transform-bindings-rej-hst node)
	  (label-for-brother-failure node)
	  (if (interaction-tcs)
	      (label-for-gi-target-concepts node))
          (create-match-failure-history node)))


; ----------------------   Labeling for target concepts --------------

; if there is a node-level failure, record that, otherwise look
; for goal-level failures, operator-level failures, and bindings-level
; failures.

; Silly names for properties, e.g. node-node-level-failures, came
; from franz, where they originally were called 

(defun label-for-failure-target-concepts (n)
  (cond ((node-failure-reason n)
	 (setf (node-node-level-failure n) t))
	(t (setf (node-goal-level-failures n)
		 (get-goal-level-failures n))
           (setf (node-op-level-failures n)
		 (get-op-level-failures n))
	   (setf (node-bindings-level-failures n)
		 (get-bindings-level-failures n)))))

; goal-level failures are stored as a list of failed goals

(defun get-goal-level-failures (n)
  (if (and (is-top-level-node n)
	   (not (node-added-after-restart n)))
      (g-map (goal in (alt-unmatched-conds (node-generating-alt n)))
	     (when (and (all-children-with-goal-failed n goal)
         		; no unexplored alternatives with this goal
			(notany #'(lambda (alt) (equal goal (alt-goal alt)))
				(node-alternatives n))))
	     (save goal))))

; Only get op-level-failures for a goal that succeeds. (Otherwise they are
; classified as goal-failures...

; op-level failures are stored as a list of ops. 

; for each op, checks to see that there are no children that didnt fail,
; and no unexpanded alts.



(defun get-op-level-failures (n)
  (if (and (not (node-reset-alt n)) ; otherwise we get untried ops below
	   (not (node-added-after-restart n)))
      (let ((succ-goal (get-successful-goal n)))
	(g-map (op in (calc-poss-ops succ-goal))
	       (when (and (all-children-with-goal-and-op-failed n succ-goal op)
			; no unexplored alternatives with this goal and op
			(notany #'(lambda (alt)
				    (and (equal (alt-goal alt) succ-goal)
					 (eq op (alt-op alt))))
				(node-alternatives n))))
	       (save op)))))


; bindings-level failures are stored as a list of children, where where
; each child node corresponds to a (different) failed set of bindings.

; again, we only care about bindings failures for the goal and op that
; succeeded. All other bindings failures are uninteresting, since
; they are (at least) op-level failures as well.

(defun get-bindings-level-failures (n)
  (if (not (node-added-after-restart n))
      (let ((op-failures (node-op-level-failures n))
	    (succ-goal (get-successful-goal n)))
	(g-map (c in (node-children n))
	       (when (and (node-failure-reason c)
			  (not (node-added-after-restart c))
			  (equal (alt-goal (node-generating-alt c))
				 succ-goal)
			  (not (member (alt-op (node-generating-alt c))
				       op-failures))))
	       (save c)))))

; returns nil if a node did not fail, or if there are no children with 
; the goal, or if children were added when looking for an interaction 
; (so result must have been indeterminate).

(defun all-children-with-goal-failed (n goal)
  (g-loop (init children (get-children-with-goal n goal) c nil)
	  (before-starting (cond ((null children) (return nil))))
	  (while (setq c (pop children)))
	  (do (cond ((not (node-failure-reason c))
		     (return nil))
		    ((node-added-after-restart c)
		     (return nil))))					  
	  (result t)))

; similar to above

(defun all-children-with-goal-and-op-failed (n goal op)
  (g-loop (init children (get-children-with-goal-and-op n goal op) c nil)
	  (before-starting (cond ((null children) (return nil))))
	  (while (setq c (pop children)))
	  (do (cond ((not (node-failure-reason c))
		     (return nil))
		    ((node-added-after-restart c)
		     (return nil))))
	  (result t)))


(defun get-successful-goal (n)
   (if (node-success-child n)
       (alt-goal (node-generating-alt (node-success-child n)))))


(defun get-children-with-goal (n g)
  (g-map (c in (node-children n))
	 (when (equal g (alt-goal (node-generating-alt c))))
	 (save c)))

(defun get-children-with-goal-and-op (n g op)
  (g-map (c in (node-children n))
	 (when (and (equal g (alt-goal (node-generating-alt c)))
		    (eq op (alt-op (node-generating-alt c)))))
	 (save c)))


; Calls target concept recognizers for goal-interactions.
; works a lot like label-gi-for-failure-target-concepts, except the
; data for op gi's are stored a bit differently.

(defun label-for-gi-target-concepts (n)
  (cond ((member (node-gi-label n) '(failure above))) ; do nothing
	((null (node-gi-label n))
	 (error "weird, null gi-label"))
	((eq (node-gi-label n) 'yes)
	 (setf (node-node-level-gi n) t))
	(t (setf (node-goal-level-gis n) 
                 (get-goal-level-gis n))
	   (setf (node-op-level-gis n)
		 (get-op-level-gis n))
	   (setf (node-bindings-level-gis n)
		 (get-bindings-level-gis n)))))
	
; The goal-level gis are stored as a list of goals

(defun get-goal-level-gis (n)
  (if (is-top-level-node n)
      (g-loop (init goals (alt-unmatched-conds (node-generating-alt n))
		    ret-val nil)
	      (while goals)
	      (do (if  (and (interaction-exhibited-by-children 
			        (get-children-with-goal n (car goals)))
			    (notany #'(lambda (alt) (equal (car goals)
							   (alt-goal alt)))
				    (node-alternatives n)))
		      (push (car goals) ret-val)))
	      (next goals (cdr goals))
	      (result ret-val))))
    
; The op-level-gis are stored as a list of (goals op) pairs.
; (indicates that the operator resulted in a gi when attempting
; to  achieve the goal).

; (Unlike the case for failure, we have to look at all goals and ops, not
; just the ops which failed to achieve the successful goal.)

; assumes one goal!
(defun get-op-level-gis (n)
  (g-loop (init ret-val nil
		goals (cond ((not-top-level-node n)
			     (list (car (alt-unmatched-conds 
					 (node-generating-alt n)))))
			    (t (set-difference (alt-unmatched-conds
						(node-generating-alt n))
					       (node-goal-level-gis n)
					       :test #'equal))))
	  (while goals)
	  (do (g-loop (init ops (calc-poss-ops
				 (car (alt-unmatched-conds
					     (node-generating-alt n)))))
		      (while ops)
		      (do (if (and (interaction-exhibited-by-children 
				    (get-children-with-goal-and-op n
					     (car goals)(car ops)))
				   (notany #'(lambda (alt) 
					       (and (equal (car goals)
							   (alt-goal alt)))
					       (eq (car ops) (alt-op alt)))
					   (node-alternatives n)))
			      (push (list (car goals) (car ops)) ret-val)))
		      (next ops (cdr ops))))
	  (next goals (cdr goals))
	  (result ret-val)))

; bindings-level gi's are stored as a list of children. (The generating
; alt of the child indicates the bindings, op and goal which resulted in a 
; gi when attempting to solve GOAL with OP).

(defun get-bindings-level-gis (n)
    (let ((op-gis (node-op-level-gis n))
	  (goal-gis (node-goal-level-gis n)))
	 (g-map (c in (node-children n))
	       (when (and (eq 'yes (node-gi-label c))
			  (not (member (alt-goal (node-generating-alt c))
				     goal-gis :test #'equal))
			  (not (member (list (alt-goal (node-generating-alt c))
					     (alt-op (node-generating-alt c)))
				       op-gis :test #'equal))))
	       (save c))))

; uses the gi-labels stored earlier.

(defun interaction-exhibited-by-children (children)
  (g-loop (init c nil found nil)
	  (while (setq c (pop children)))
	  (do (cond ((and (not (equal (node-gi-label c) 'yes))
                          (not (equal (node-gi-label c) 'supporting-failure)))
		     (return nil))
		    ((eq (node-gi-label c) 'yes)
		     (setq found t))))
	  (result found))) ; at least one 'yes


;------------ Code for labeling brother failures -----------------

; this labeling info is used by the discrimination functions to
; figure out whether the axioms for "brother deletes brother failures"
; apply.

; LABEL-FOR-BROTHER-FAILURE finds nodes where achieving goal1
; has deleted goal2, and vice versa

;  assumes one goal. So if goal fails, node-fails

(defun label-for-brother-failure (node)
  (if (and (node-reset-alt node)
	   (left-clone node)) ; filters out last node, where DONE is achieved
                              ; it should be the only reset-node with no lclone
     (let ((unachieved-subgoals
	    (g-map (g in (get-all-attempted-subgoals node))
		   (when (not (atomic-formula-match g (node-state node))))
		   (save g))))
      (find-del-bro-goals node unachieved-subgoals))))

(defun get-all-attempted-subgoals (node)
  (if (node-reset-alt node)
      (let ((goals (get-all-attempted-subgoals
		    (left-clone  node)))
	    (goal (alt-goal (node-generating-alt node))))
	(cond ((member goal goals :test #'equal) goals)
	      (t (cons goal goals))))))

         
(defun find-del-bro-goals (n subgoals)
  (g-loop (init ret-val nil other-goal nil)
	  (while subgoals)
	  (do (g-loop (init clone (left-clone n) lnode n)
		      (while clone)
		      (do (cond ((atomic-formula-match (car subgoals)
						       (node-state clone))
				 (setq other-goal
				       (alt-goal (node-generating-alt lnode)))
				 (if (same-to-you-bro other-goal
						      (car subgoals) n)
				     (setq ret-val (list other-goal 
							 (car subgoals)))))))
		      (next lnode clone
			    clone (left-clone lnode))))
	  (until ret-val)
	  (next subgoals (cdr subgoals))
	  (result (setf (node-bro-deletion-node n) ret-val))))
		    
; deleter goal also clobbered deleted goal.

(defun same-to-you-bro (deleter deleted n)
  (g-loop (while n)
	  (do (if (and (node-reset-alt n)
		       (not (atomic-formula-match deleter (node-state n)))
		       (equal deleted (alt-goal (node-generating-alt n))))
		  (return t)))
	  (next n (left-clone n))))



;;;; -------------- Grimy Code -------------------------------

; ....for augmenting, or changing the format of, various 
; types of trace information at a node.

; sets match-failure-history to be (((op add goal) failed-exp failed-exp))...)
; [note, should really be (((goal op add) failed-exp....) but this is a late
;  correction]

(defun create-match-failure-history (node)
  (g-loop (init hist nil hists (node-history-for-ebl node) failures nil
		op nil bindings nil goal nil post nil match-explanation nil)
	  (while (setq hist (pop hists)))
	  (do (setq op (car hist))
	      (setq bindings (cadr hist))
	      (setq goal (caddr hist))
	      (setq post (cadddr hist))
	      (setq match-explanation (caddr (cddr hist)))
	      (setq failures
		    (cons (cons (list op post goal) 
				; 1st guy is definer (or exp is definer)
				(append (get-rhs-definers node op post)
					(break-out-failures 
					 (get (car hist) 'lpreconds)
					 match-explanation node goal op post)))
			  failures)))
	  (result (setf (node-match-failure-history node) failures))))

; first guy on ret-val should be exp

; fs are the failures, won't have all of them cause matcher doesn't
; record all failures

(defun break-out-failures (exp fs node goal op add)
    (cond ((eq (car exp) 'exists)
	   (let ((gen-failure (break-out-failures (get-gen-exp exp)
				  fs node goal op add)))
		(cond (gen-failure
			  (cons exp (cons (get-gen-exp exp)
					  (break-out-failures (get-exp exp) fs
					      node goal op add))))
		      (t (let ((subs (break-out-failures (get-exp exp) fs
					 node goal op add)))
			      (and subs (cons exp subs)))))))
	  ((eq (car exp) 'forall)
	   (let ((subs (break-out-failures (get-exp exp) fs
			   node goal op add)))
		(and subs (cons exp subs))))
	  ((eq (car exp) 'and)
	   (let ((subs (g-map (e in (cdr exp))
			     (splice (break-out-failures e fs
					 node goal op add)))))
		(and subs (cons exp subs))))
	  ((eq (car exp) 'or) 	; hacked up, not really correct
	   (let ((subs (g-map (e in (cdr exp))
			     (filter (break-out-failures e fs
					 node goal op add)))))
		(cond ((eq (length (cdr exp))(length subs))
		       (cons exp (apply 'append subs))))))
	  ;  doesn't work for negated exists
	  ((eq (car exp) '~) 
	   (and (assoc exp fs :test #'equal) (list exp)))
	  ((atomic-formula-p exp) 
	   (or (and (assoc exp fs :test #'equal) 
		    (or (static-p exp)
			(exists-a-non-gr exp node goal op add))
		    (list exp))
	       ; goal-repeat hack
	       (and (exists-a-secondary-gr exp node goal op add)
		    (list exp))))))


; if bindings rejection rules fired, save that info in a easier to use
; format at the node.

(defun transform-bindings-rej-hst (node)
    (let ((new-entrys
	      (g-map (entry in (node-reject-bindings-hst node))
		     (save (get-meta-state-info-for-rej-hst entry)))))
	 (and new-entrys 
	      (setf (node-EBL-reject-bindings-hst node)
		    (nreverse new-entrys)))))

; scr-entry stored by planner is (rule bindings obj)
; makes a new entry with format (dvars op real-goal rule-name)
; dvars is the variables defined (i.e. bound) in the rule.

(defun get-meta-state-info-for-rej-hst (entry)
  (list (get-important-dvars-for-b-reject (car entry)) ; dvars
	(get-nth-from-sig-for-ebs 4 (car entry)) ; op
	(subst-bindings 
	 (get-nth-from-sig-for-ebs 3 (car entry)) ; goal
	 (cadr entry))
	(subst-bindings 
	 (get-nth-from-sig-for-ebs 5 (car entry)) ; bindings
	 (cadr entry))
	(car entry))) ; rule-nm

	       
; returns params that are not vars and were mentioned in rest of reject rule. 

(defun get-important-dvars-for-b-reject (rule)
  (let* ((cand-bindings (get-form-from-exp 'candidate-bindings 
					   (get rule 'lhs)))
         ; must keep all conditions in rest of rule, since important dvars
	 ; might be in goal
	 (rest-of-rule (del-eq cand-bindings (get rule 'lhs)))
	 (op (get-nth-from-sig-for-ebs 4 rule)))
    (cond ((null rest-of-rule) ; can happen
	   (format t "WARNING: get-important-dvars-for-b-reject, short rule ~a" rule))
	  ((null op)	; should never happen
	   (format t "WARNING: get-important-dvars-for-b-reject, no op,")
	   (warning-stop)))
     (g-map (rule-param in (caddr cand-bindings))
	    (op-param in (get op 'params))
	    (when (r-memq rule-param rest-of-rule))
	    (save op-param))))



;  this is used because when there is an op application, or 
;  some rejected bindings,
;  you need to treat the generators of the rhs of the rule (or op)
;  as failure points, and store them away for the EBS process.

;  all-vars-in-effects shouldnt include wildcard vars, but does
; so I came up with this lhs-rhs thing. 

(defun get-rhs-definers (n op post)
  (unionq (if (node-EBL-reject-bindings-hst n)
	      (r-find-definers (get op 'lpreconds)
			       (let ((all-important-vars 
				      (mapcar #'car (node-EBL-reject-bindings-hst n))))
				 (g-map (v in (get op 'params))
					(when (r-memq v all-important-vars))
					(save v)))
			       (g-map (v in (cdr post))
				      (when (is-variable v))
				      (save v)) 
			       nil))
	  (if (g-loop (init cs (node-children n) c nil)
		      (while (setq c (pop cs)))
		      (do (if (and (node-applied-node c)
				   (eq (alt-op (node-generating-alt c)) op)
				   (equal (alt-post-cond
					   (node-generating-alt c)) post))
			      (return t))))
	      (r-find-definers (get op 'lpreconds)
			       (get op 'all-rhs-lhs-vars)
			       (g-map (v in (cdr post))
				      (when (is-variable v))
				      (save v)) 
			       nil))))
    
; rvars are rule variables, dvars are defined (bound) vars.

(defun r-find-definers (exp rvars dvars outsides)
    (cond ((all-defined rvars dvars) outsides)
	  ((eq exp t) nil)
	  ((eq (car exp) 'exists)
	   (r-find-definers (get-exp exp) rvars
	       (append (get-vars-lst exp) dvars)
	       (cons exp outsides)))
	  ((eq (car exp) 'and)
	   (g-loop (init orig-exp exp ret-val nil)
		   (next exp (cdr exp))
		   (while exp)
		   (do (setq ret-val (r-find-definers (car exp) rvars dvars 
					 (cons orig-exp outsides))))
		   (until ret-val)
		   (result ret-val)))))
    
(defun all-defined (rvars dvars)
    (g-loop (while rvars)
	    (do (or (member (pop rvars) dvars) (return)))
	    (result t)))
    


; There was a goal-cycle failure at a node, but the goal in question
; was not the primary candidate goal.

(defun has-secondary-gr-reason (n)
    (and (eq 'goal-repeat (car (node-failure-reason n)))
	 (not (equal (cadr (node-failure-reason n))
		     (car (alt-unmatched-conds (node-generating-alt n)))))))

(defun is-parentage (c goal op add)
  (and (eq goal (alt-goal (node-generating-alt c)))
       (eq op (alt-op (node-generating-alt c)))
       (equal add (alt-post-cond (node-generating-alt c)))))

; There exists a child node generated by an alt consistent with 
; exp, goal, op & add, where a secondary goal-cycle DID NOT happen (see above
; for definition).

(defun exists-a-non-gr (exp n goal op add)
  (g-loop (init children (node-children n) c nil)
	  (while (setq c (pop children)))
	  (do (and (is-parentage c goal op add)
		   (equal exp (alt-failed-cond (node-generating-alt c)))
		   (not (has-secondary-gr-reason c))
		   (return t)))))
	      

; There exists a child node generated by an alt consistent with 
; exp, goal, op & add, where a secondary goal-cycle DID happen (see above
; for definition).

(defun exists-a-secondary-gr (exp n goal op add)
  (g-loop (init children (node-children n) c nil)
	  (while (setq c (pop children)))
	  (do (and (equal exp (node-gr-cond c))
		   (has-secondary-gr-reason c)
		   (is-parentage c goal op add)
		   (return t)))))


; Called when the failure reason is GOAL-REPEAT. (ie. goal-stack cycle)
; This finds the literal in the operator's preconditions that 
; caused the goal cycle. (The instantiated literal is the cadr of the 
; node-failure-reason).

(defun find-gr-condition (node)
    (g-loop (init alt (node-generating-alt node)
		  op (alt-op alt) gr-cond nil
		  lits (get-lits (get op 'preconds))
		  bindings (g-map (v in (get op 'vars))
				  (val in (alt-vars alt))
				  (save (list v val)))
		  gr (cadr (node-failure-reason node)))
	    (while lits)
	    (do (if (equal gr (subst-bindings (car lits) bindings))
 	            (setq gr-cond (car lits))))
	    (until gr-cond)
	    (next lits (cdr lits))
	    (result (cond ((null gr-cond) (error "Find-gr-cond"))
			  (t (setf (node-gr-cond node) gr-cond))))))


; finds first protected goal violation found at node, returns the violated 
; goal (or sets node property to be violated goal if putflag is T).
				
; called during cntrl, so must use structures
(defun hist-protection-violation (node putflag)
  (if (node-applied-node node)
      (g-loop (init parent-state (node-state (node-parent node))
		    op (alt-op (node-generating-alt node))
		    ivars (alt-vars (node-generating-alt node))
		    idels (expand-conditional-dels parent-state
				   (subpair (get op 'vars) ivars 
					    (get op 'del-list)))
		    iadds (expand-conditional-adds parent-state
				   (subpair (get op 'vars) ivars 
					    (get op 'add-list)))
		    protected-goals (cond (*ONLY-TOP-GIS*
				; should really only be true goals
  				            (mapcar #'cadar
						    (in-goal-exp node '<x>)))
					  (t (mapcar #'cadar
						     (protected-goal node '<x>))))
		    pgoal nil found nil)
	      (while (setq pgoal (pop protected-goals)))
	      (do (cond ((has-vars pgoal)) ;  do nothing
			((negated-p pgoal)
			 (if (member (cadr pgoal) iadds :test #'equal)
			     (setq found pgoal)))
			((not (closed-predicate pgoal))
			 (cond ((and (member pgoal idels :test #'equal)
				     (not (member pgoal iadds
						  :test #'equal))) ;not readded
				(setq found pgoal))
			       ((member (negate pgoal) iadds
					:test #'equal)
				(setq found pgoal))))
			((and (member pgoal idels :test #'equal)
			      (not (member pgoal iadds
					   :test #'equal))) ;not readded
			 (setq found pgoal))))
	      (until found)
	      (result (and found 
			   (cond (putflag
				  (setf (node-protection-violation node)
					found))
				 (t found)))))))

; Returns goal if there was a hist-prereq violation (or sets node property)

; only do it for closed world current-goals (not other goals)

(defun hist-prerequisite-violation (orig-node goal putflag)
  (if (and (closed-literal goal)
           (not *ONLY-TOP-GIS*)  ; maybe should change so that we do chck for
				; prereq-violationss, but only top-level
	   (not (negated-p goal))
	   (not (has-vars goal))) ;  not necessary !
      (g-loop (init child nil node orig-node found nil)
	      (until (null (node-parent node)))
	      (next child node
		    node (node-parent node))
	      (do (if (and (node-applied-node child)
			   (get (alt-op (node-generating-alt child))
				'operator) ;  is operator
			   (atomic-formula-match goal (node-state node)))
		      (setq found goal)))
	      (until found)
	      (result (and found
			   (cond (putflag
				  (setf (node-prerequisite-violation orig-node)
					found))
				 (found)))))))
	 

; ----------------------------------------------------------

; Functions for making and resetting histories.

; There are histories (abbrieviated "h") associated with each node to
; store extra information not stored at the node. The information
; stored in a history is relevent to a single target-concept/training example
; pair, and is not specific to the node itself.

(defun make-hist (n)
  (let ((h (intern (concatenate 'string "HST"
				(prin1-to-string (get-next-hst))))))
    (setf (symbol-plist h) nil)
    (setf (get h 'node) n)
    (setf (node-recent-ebl-hist n) h) ;  used for debugging
    (setf (get h 'reset-alt) (node-reset-alt n))
    (setf (get h 'protection-violation) (node-protection-violation n))
    (setf (get h 'prerequisite-violation) (node-prerequisite-violation n))
    (setf (get h 'EBL-reject-bindings-hst) (node-EBL-reject-bindings-hst n))
    (setf (get h 'gi-label) (node-gi-label n))
    (setf (get h 'failure-reason) (node-failure-reason n))
    (setf (get h 'children) (node-children n))
    (setf (get h 'children) (g-map (c in (get h 'children))
				   (save c)))
    (push h *HISTS*)
    h))


(defun make-inter-hist (n hst)
    (let ((h (intern (concatenate 'string "INT" (subseq (symbol-name hst) 3))
		     'USER)))
	 (setf (get hst 'inter-hst) h)
	 (setf (symbol-plist h) nil)
	 (setf (get h 'orig-hst) hst)
	 (setf (get h 'node) n)
         (push h *HISTS*)
	 h))

(defun get-next-hst ()
  (setq *HIST-NUM* (+ 1 *HIST-NUM*)))
    
(defun reset-hists ()
  (setq *HIST-NUM* 0)
  (if (boundp '*HISTS*)
      (dolist (h *HISTS*)
	  (setf (symbol-plist h) nil)))
  (setq *HISTS* nil))
