
;;; This file contains code to determine the constraints on
;;; generators.  If a particular literal generates a binding that
;;; is then used by another literal, it is important that the
;;; generating literal is not removed from the domain before the other
;;; one.  If it is, then unpredictable things will happen.  
;;; Calling determine-generator-constraints with the operators in the
;;; domain will return a set of constraints to avoid this problem.
;;; Note that static generators (literals in the state or functions) 
;;; should be able to handle unbound variables, so only constraints on
;;; nonstatic literals are generated.
;;;
;;; The constraints are of the form ((x1 y1)(x2 y2)), where the x's
;;; must be higher in the abstraction hierarchy than the y's.
;;;


;;; This function adds the generator constraints to the given
;;; hierarchy.
;;;
(defun add-generator-constraints (hierarchy)
  (let ((constraints (determine-generator-constraints *ops*)))
    (add-each-generator-constraint hierarchy constraints)
    (setq *GENERATORS* (remove-duplicates (mapcar #'car constraints) 
					  :test #'equal))
    (setq *NONSUBGOALABLE-GENERATORS*
	  (mapcan #'(lambda (x)(if (not-subgoalable x)(list x)))
		  *GENERATORS*))))
				 
(defun add-each-generator-constraint (hierarchy constraints)
  (cond ((null constraints))
	(t (add-constraint (first (car constraints))
			   (second (car constraints))
			   hierarchy)
	   (add-each-generator-constraint hierarchy (cdr constraints)))))


;;; Remove any duplicate constraints on the generators.
(defun determine-generator-constraints (ops)
  (remove-duplicates (generator-constraints ops) :test #'equal))

;;; Recur through all the operators.  
(defun generator-constraints (ops)
  (cond ((null ops) nil)
	(t (append (determine-gen-constraints 
		    (car ops)
		    (initially-bound (get (car ops) 'unrenamed-typed-effects)
				     (get (car ops) 'all-vars)))
		   (generator-constraints (cdr ops))))))

;;; Using information about primary effects, determine which variables
;;; are guaranteed to be initially bound when the operator is used.
;;;
(defun initially-bound (effects bound)
  (cond ((null effects) bound)
	((eq 'secondary (formula-status (car effects)))
	 (initially-bound (cdr effects) bound))
	(t (initially-bound (cdr effects)
			    (intersect 
			     (find-vars (formula-instance (car effects)))
			     bound)))))

;;; Find all the variable in an instance.
(defun find-vars (instance)
  (cond ((null instance) nil)
	((eq '~ (car instance))
	 (find-vars (cadr instance)))
	((not (is-variable (car instance)))
	 (find-vars (cdr instance)))
	(t (cons (car instance)
		 (find-vars (cdr instance))))))

;;; Determine the set of constraints and then remove any that involve
;;; static literals.
(defun determine-gen-constraints (op bound)
  (let ((constraints 
	 (gen-constraints 
	  op
	  (get op 'unrenamed-typed-preconds)
	  bound
	  (mapcar #'(lambda (x)(cons x 'primary)) bound)
	  nil)))
    (remove-static-constraints constraints constraints)))

;;; If a static literal is constrained to be lower in the hierarchy than a
;;; another literal, the constraint can simply be discarded.  However,
;;; if a literal is constrained to be lower than a static literal,
;;; that static literal may in turn be constrained below some other
;;; literal, so the constraints must be dereferenced.  That's what
;;; this does.
(defun remove-static-constraints (constraints all-constraints)
  (cond ((null constraints) nil)
	(t (let ((x (caar constraints))
		 (y (cadar constraints)))
	     (cond ((member x *static* :test #'equal)
		    (remove-static-constraints (cdr constraints) 
					       all-constraints))
		   ((member y *static* :test #'equal)
		    (append (find-nonstatic-constraint x y all-constraints
						       all-constraints)
			    (remove-static-constraints (cdr constraints)
						       all-constraints)))
		   (t (cons (car constraints)
			    (remove-static-constraints (cdr constraints)
						       all-constraints))))))))
;;;
(defun find-nonstatic-constraint (x y constraints all-constraints)
  (cond ((null constraints) nil)
	((equal y (caar constraints))
	 (let ((new-y (cadar constraints)))
	   (cond ((member new-y *static* :test #'equal)
		  (append (find-nonstatic-constraint x new-y all-constraints
						     all-constraints)
			  (find-nonstatic-constraint x y (cdr constraints)
						     all-constraints)))
		 ((equal x new-y)
		  (find-nonstatic-constraint x y (cdr constraints)
					     all-constraints))
		 (t (cons (list x new-y)
			  (find-nonstatic-constraint x y (cdr constraints)
						     all-constraints))))))
	(t (find-nonstatic-constraint x y (cdr constraints) all-constraints))))
	   
;;; Given an operator, a list of literals which are either the
;;; preconditions or effect of the operator, list of bound vars, data
;;; structure containing the generators of all the bound vars, and the
;;; constraints collected so far, this function finds all the
;;; necessary generator constraints.  The idea is to first work scan
;;; through the preconditions and for each literal determine if it
;;; uses any previously bound vars.  For each of the variable used,
;;; unless they are bound in the primary effects, a constraint is
;;; formed.  Then it notes any variables generated by the literal and
;;; records it in the gen-struct.  After the preconditions, it scans
;;; through the effects.
;;;
(defun gen-constraints (op lits bound gen-struct constraints)
  (cond ((null lits)
	 (cond ((null op) 
		constraints)
	       (t (gen-constraints nil
				   (get op 'unrenamed-typed-effects)
				   bound
				   gen-struct
				   constraints))))
	(t (let* ((vars (find-vars (formula-instance (car lits))))
		  (gen (set-difference vars bound))
		  (used (set-difference vars gen)))
	     (cond ((null gen)
		    (gen-constraints op (cdr lits) 
				     bound gen-struct
				     (append constraints 
					     (collect-constraints 
					      (extract-instance
					       (strip-negation 
						(formula-instance (car lits))))
					      used gen-struct))))
		   (t (gen-constraints 
		       op
		       (cdr lits)
		       (union bound gen)
		       (append (mapcar 
				#'(lambda (x)
				    (cons x (extract-instance
					     (strip-negation
					      (formula-type (car lits))))))
				       gen)
			       gen-struct)
		       (append constraints 
			       (collect-constraints
				(extract-instance
				 (strip-negation 
				  (formula-type (car lits))))
				used gen-struct)))))))))

;;; Forms the constraints.  Given a particular literal and the
;;; variables used (not generated) by the literal, determines what
;;; literals it will depend on.  It depends on every variable that
;;; generates one of the variables it uses.  This information can
;;; simply be looked up in the gen-struct data structure.  If it
;;; depends on a variable that was bound by a primary effect, then no
;;; constraint is needed since when the primary effect is removed the
;;; operator can't be used.
;;;
(defun collect-constraints (type used gen-struct)
  (cond ((null used) nil)
	(t (let ((constraint (cdr (assoc (car used) gen-struct))))
	     (cond ((or (eq 'primary constraint)
			(equal type constraint))
		    (collect-constraints type (cdr used) gen-struct))
		   (t (cons (list constraint type)
			    (collect-constraints type (cdr used) gen-struct))))))))
