;; File:  rob-bindings.lisp
;;
;; Author: Rob Spiger
;; Sponsoring Professor: Oren Etzioni
;; Date:  May 1992
;; Docs: In this file.
;;
;;
;; The purpose of this code is to:
;;   1.  Figure out typing information for every variable in the PSG.  
;;   2.  Apply that typing information in simplifying expressions by
;;       interfacing directly with the simplifier.
;;
;; Note:  This code does not rule out bindings when choosing operators when
;;        creating the PSG.  For information about that kind of typing see\
;;        rob-simplify/var-typing.lisp.
;;--------------------------------------------------------------------
;; Explanation of purpose #1.
;; Figuring out the typing information:
;;--------------------------------------------------------------------
;; The typing information for each variable in the PSG is put in to
;; the global variable *var-bind-info*.  Each variable is paired with
;; the static generators in the preconditions of the first operator
;; the variable occurs in.
;;
;; An example would be Var <V55> has generators: (CARRIABLE) and
;; (IS-KEY)
;;
;; These would have been found in the following way:
;;
;; 1.  The PSG would have been created.
;; 2.  Every variable would have been created by binding to an
;;   operator.  
;; 3.  Several cases would arise.
;;    A.  For a normal variable starting with V all the static
;;        predicates logically ANDed in the preconditions would be
;;        looked at.  Those which contained the variable would be
;;        considered to be the generators.
;;        (Some variables will get more generators than one if the
;;         preconditions of the operator they are from hold multiple
;;         static predicates for the variable in the preconditions.)
;;
;;       For example in the following operator:
;;(LOCK
;; (params (<door2> <k1> <rm-b>))
;; (preconds
;;     (and (is-door <door2>)
;;	  (is-key <door2> <k1>)
;;	  (holding <k1>)
;;	  (dr-to-rm <door2> <rm-b>)
;;	  (inroom <k1> <rm-b>)
;;	  (next-to robot <door2>)
;;	  (dr-closed <door2>)
;;	  (unlocked <door2>)))
;; (effects 
;;    ((del (unlocked <door2>))
;;     (add (locked <door2>)))))
;;
;;  variable <k1> would have the generators is-key.
;;  variable <door2> would have the generators is-door is-key dr-to-rm.
;;
;;    B.  For a universally or explicitly existentially quantified
;;        variable the generator predicate would be taken as the
;;        generator for the variable.  (This must be static.)
;;  
;;  For example:
;;
;;  (INFER-ARM-EMPTY
;;   (params nil)
;;   (preconds
;;       (~ (exists (<ob>) (object <ob>) (holding <ob>))))
;;   (effects ((add (arm-empty)))))
;;
;;  variable <ob> would have the generator object.
;;
;;    C.  The variable is one which appears in an shared literal.
;;        It's generators are the intersection of all its sets of
;;        generators from each operator it is in.  
;;
;; NOTES:
;;
;;No relevances is placed on the variable's location in the generator,
;; infact, only the predicate of the generator is stored.
;; This removes a lot of complications.
;; 
;; Static does not look to see if the generator is negated or not.
;; Of course static suffers a slight loss in information when the 
;; static generators for stuff are both negated and not-negated.
;; Currently, it will not differentiate between the two generators.
;;
;; This code cold be much improved by using the get and putprop commands for
;;  the data structure.

;;-----------------------------------------------------------------
;;Explanation of purpose#2:
;;Using the typing information with the simplifier:
;;-----------------------------------------------------------------
;; Once having figured out the type information for each variable,
;;   the information is used in the following way by the simpilifier.
;;
;;Var1 can bind to var2 if the generator for var1 is a subset of 
;; the generator for var2.  NOTE:  Just because var1 can bind to
;; var2 does not mean var2 can bind to var1.
;;
;; For example:  <x> might have generators CARRIABLE and IS-KEY
;;               <y> might have generator  CARRIABLE
;;  In this case, <y> can bind to <x>, but <x> can't bind to <y>.
;;
;;Wildcard variables bind to anything, so their generators are
;;  set to nil.  And nothing can bind to them, so even though
;;  something else might not have a generator, it still shouldn't
;;  be allowed to bind to a wildcard.  (This is a special case.)
;;
;;Constants are considered to have no generators, so a variable
;; can only bind to a constant if it has no generators.  (This will
;; include wildcards.)
;;
;;----------------------------------------------------------------

;; Variable Binding Information
;;------------------------------
(defstruct (var-bind-info
 (:print-function
  (lambda (p s k)
   (format s "(Var ~s has generators: ~s)" 
       (var-bind-info-var p)
       (var-bind-info-generator p)))))
 (var nil)       ;;name of the variable
 (generator nil));;a list of the static generators for the variable

;;These are stored in global variable *var-bind-info*
;; If the var has already had it generators stored, then 
;;  this is the case where it is a shared literal.
;;  For shared literals, take the intersection of it's 
;;  current list of generators, and it's previous list of generators

;;--------------------------------
;; Function:  Store-var-generator
;; Purpose: Stores the generators of a variable in *var-bind-info*.  
;; Inputs:  The variable ie. <V12>
;;          A list of the variable generators ie. (Is-key Carriable)
;; Outputs: None
;; Side-effects:  Modifies *var-bind-info*
;;
(defun store-var-generator (var generator)
 (setq generator (mapcar #'predicate-name generator))
  (let (
    (node (already-has-generators-stored var)))
   (if node
    (setf (var-bind-info-generator node)
               (intersection generator 
               (var-bind-info-generator node)))
    (setq *var-bind-info*
      (cons
       (make-var-bind-info
        :var var
        :generator generator) *var-bind-info*))))
    nil)  ;;return value

;;Function: Already-has-generators-stored
;;Purpose:  Sees if the variable has already been stored in
;;*var-bind-info* and if so, returns the information on it.
;;Inputs:  A variable ie. <v12>
;;Outputs: A pointer to the node in *var-bind-info* pointing to the
;;input variable or else nil.
;;
;;Notes:  This is used because for a shared literal, it's generators
;;will be stored more than once it is always stored as the intersect
;;of the current generators and the previously stored generators.
;;
(defun already-has-generators-stored (var)
 (find-if #'(lambda (var-gen)
               (eq var (var-bind-info-var var-gen)))
              *var-bind-info*))
 


;;Function: find-and-store-generators
;;Inputs:  (1) A list of static predicates.  This should be usually
;;obtained by finding the static predicates which are ANDed together
;;in a an operators preconditions.  
;; ie.  ((is-key <V23>) (is-key-to-door <V32> <V45>))
;; 
;;(2) A list of variable which should not be stored.  These are usually
;;the variables that bound with the goal when the operator was being
;;applied, these variable have already had their types stored.
;;
;; ie (<V45> <v23>)
;;
;;Outputs:  None
;;Side-effects:  For every varaible present in the list of static
;;predicates which is not a variable in dont-store-vars it will have
;;it's generators stored as those predicates which contain it.
;;
(defun find-and-store-generators (list-of-static-preds
					&optional   (dont-store-vars nil))
 (let* (
   (all-vars   (find-variables list-of-static-preds))
   (store-vars (remove-if #'(lambda (x) (member x dont-store-vars))
						  all-vars))
   (var-preds  (mapcar #'(lambda (var) 
                          (remove-if-not
                           #'(lambda (pred)
                              (member var (find-variables pred)))
                           list-of-static-preds)) store-vars)))
  (n-mapcar #'store-var-generator store-vars var-preds)))

;;Function: find-and-store-new-var-generators
;;Input:  Operator node of the PSG.
;;Outputs: None
;:Side-effects:  Stores in *var-bind-info* the static generators
;; for every varaible not bound with the goal in the operator.
;; (Does the whole things given a operator node in the PSG.)
;;

(defun find-and-store-new-var-generators (node)
 (unless (childless-node? node)
  (let* (
    (parent-name     (literal-name (operator-parent node)))
    (dont-store-vars (find-variables parent-name))
    (preconds-node   (operator-preconditions node))
    (static-preds    (internalnode-anded-static-preds preconds-node)))
   (find-and-store-generators static-preds dont-store-vars))))

;;Function: Find-var-types
;;Inputs: *roots*
;;Outputs: None
;;Side-effects:  Finds typing information for every variable in every
;;PSG, and stores it in *Var-bind-info*

(defun find-var-types ()
 (setq *var-bind-info* nil)
 (iter:iterate
 (iter:for root iter:in *roots*)
  (find-and-store-var-generators-for-psg root)))


(defun find-and-store-var-generators-for-psg (root)
 (find-and-store-var-generators-for-node root))

(defun find-and-store-var-generators-for-literal (node)
 (unless (childless-node? node)
  (mapcar #'find-and-store-var-generators-for-node
              (literal-operators node)))
 nil) ;;return value


(defun find-and-store-var-generators-for-operator (node)
 (find-and-store-new-var-generators node)
 (unless (childless-node? node)
  (find-and-store-var-generators-for-node
              (operator-preconditions node)))
 nil) ;;return value

(defun find-and-store-var-generators-for-internal (node)
 (when (OR (eq (internalnode-name node) 'exists) 
           (eq (internalnode-name node) 'forall))
  (find-and-store-generators
    (remove-if-not #'is-static-pred-and-not-meta-func   ;;removen non-static gen
     (list 
      (literal-name
       (first
        (internalnode-operands node)))))))  ;;do generator
 (unless (childless-node? node)
  (mapcar #'find-and-store-var-generators-for-node 
              (internalnode-operands node)))
 nil) ;return value


(defun find-and-store-var-generators-for-node (node)
 (cond
  ((null node) nil)
  ((literal-p node) (find-and-store-var-generators-for-literal node))
  ((operator-p node) (find-and-store-var-generators-for-operator node))
  ((internalnode-p node) (find-and-store-var-generators-for-internal node))
  (t (format t "~%***Unknown node given to find-and-store-var-gens.***~%"))))




;;Returns a list containing the generators for a variable.
;;(looks the information up in *var-bind-info*
;;
(defun fetch-var-generator (var)
 (when (rob-is-var? var)
  (let (
    (node    (find-if #'(lambda (var-gen) 
               (eq var (var-bind-info-var var-gen)))
              *var-bind-info*)))
   (when node  ;;returns nil if var not in list
    (var-bind-info-generator node)))))


;;returns true if it is possible to bind
;; var1 to var2.  
;;

(defun possible-to-bind (var1 var2)
 (when (rob-is-var? var1)  ;;constants can't bind
  (let* (
    (gen1 (fetch-var-generator var1))
    (gen2 (fetch-var-generator var2)))
    (when (is-subset gen1 gen2)
     (when (OR (not (wild-var? var2)) (wild-var? var1))
                   ;;both must be wild vars if var2 is wild
                   ;;so something doesn't get instantiated to wild
      T)))))

;;returns T if it is completely okay to bind var1 to var2
;;
(defun okay-to-bind (var1 var2)
 (unless *rob-simplify-or-passed*
  (unless (member var1 *rob-simplify-not-vars*)
   (possible-to-bind var1 var2))))
  

;;returns (exp, blist)
;;
;;If exp is (is-equal a b) it will see if a can bind to b.
;; 


(defun simplify-if-is-equal (exp)
 (if (atom exp)
  `(,exp nil)
  (progn 
   (when (and (eq (first exp) '~)    ;;transfrom (~ (not-equal x y)) to (is-equal x y)
              (eq (first (second exp)) 'not-equal))
       (setq exp `(is-equal ,(second (second exp)) ,(third (second exp)))))

 (if (not (eq (first exp) 'is-equal))
  `(,exp nil)  ;;wrong predicate
  (let (
    (var1 (second exp))
    (var2 (third  exp)))
   (cond
    ((OR (not (atom var1)) ;;ie (is-equal (on <x> <y>) (on a b))
         (not (atom var2)))
     `(,exp nil))
    ((okay-to-bind var1 var2)
     `(T ((,var1 ,var2))))
    ((okay-to-bind var2 var1)
     `(T ((,var2 ,var1))))
    ((possible-to-bind var1 var2)
     `(,exp nil))
    ((possible-to-bind var2 var1)
     `(,exp nil))
    (t    ;;no binding possible
     `(nil nil))))))))

;;returns just the expression
(defun simplify-if-not-equal (exp)
 (if (OR (AND (not (atom exp))
            (not (eq (first exp) 'not-equal)))
         (atom exp))
  exp          ;;wrong predicate
  (let* (
    (var1 (second exp))
    (var2 (third exp)))
  (unless (equal var1 var2)   ;;returns nil if they're equal
          ;;returns nil if two things are equal
          ;;good idea to use equal here
   (if (AND (rob-is-var? var1) ;;are they both
            (rob-is-var? var2) ;;variables?
            (not (possible-to-bind var1 var2))
            (not (possible-to-bind var2 var1)))
                 ;;can they never bind?
    T  ;;they can never bind so return T
    (if (AND (atom var1)          ;if both vars are
                                  ;constants then evaulate
                                  ;if they are equal or not.
             (not (rob-is-var? var1))
             (atom var2)
             (not (rob-is-var? var2)))
     (not (eq var1 var2))

    exp))))))



;;Should probably write something to handle negated
;; is-equal's and not-equal's
  
;;If the predicate is on the false list returns nil
;;If the negation of the predicate is on the false
;;     list returns T
;;Otherwise, returns the predicate unchanged
(defun simplify-if-on-false-list (pred)
 (cond
  ((member pred *false-list* :test #'equal)
   nil)
  ((member (negate-exp pred) *false-list* :test #'equal)
   T)
  (t pred)))

;;( (unless (member pred *false-list* :test #'equal)
;;  pred))


;;returns (exp, blist)
;;
(defun simplify-predicate (pred true-list) 
 (let* (
   (exp (simplify-if-on-false-list pred))
   (exp (simplify-if-not-equal exp))
   (exp (simplify-if-is-equal exp)))
  (simplify-if-known exp true-list)))

;;inputs (exp, blist)
;;outputs (exp, blist)
(defun simplify-if-known (pred-blist true-list)
 (if (atom (first pred-blist))
  pred-blist
  (if (not (eq (first (first pred-blist)) 'known))
   pred-blist
  (let* (
    (pred  (first pred-blist))
    (name  (first pred))
    (node  (second pred))
    (exp   (third pred))
    (smaller-true-list (remove-useless-things-in-true-list
                                    true-list))
    (resultant (sub-rs exp smaller-true-list))
    (blist  (second resultant))
    (exp    (first resultant)))
   (cond
    ((eq exp T)
     `(T ,blist))
    ((not (null exp))
     `((,name ,node ,exp) ,blist))
    (T
     `(nil ,blist)))))))

   
(defun remove-useless-things-in-true-list (true-list)
 (remove-if #'(lambda (x) 
               (member (predicate-name x)
                      '(CANDIDATE-GOAL
                        CURRENT-NODE)))
   true-list))



