(defvar psg-depth nil)
(defvar shared-literals nil)
(defvar next-level-of-ops nil)
(defvar siblings nil)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Static Analyzer for Full Prodigy language
;;
;;  create-PSG.lisp contains the lisp code to create the PSG
;;  the top level function is called create-PSG
;;
;;
;;  Author: Julie Roomy
;;  Mods: Rob Spiger
;;  Sponsoring Professor: Oren Etzioni
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;Notes on modifications:  This file has been modified to create
;; the PSG breadth first instead of depth first.


;;------------
;; create-PSG
;; creates the literal node for the top level goal then
;; calls create-operator-level for the operators of the goal
(defun create-PSG (goal)
  (let*
      ( (root            ;; Create a literal node for the goal
	(make-Literal 
	 :name goal 
	 :parent nil
         :bad-bindings nil
	 :bindings nil
	 :label nil
	 :failure-condition nil
	 :goal-stack nil
     :completely-shared nil
     :necessary-effects nil))
      (next-level-of-ops-bad-bindings 
       (relevant-ops-with-shared-lits goal root))
      (next-level-of-ops-value (list (first next-level-of-ops-bad-bindings)))
      (bad-bindings            (first (second next-level-of-ops-bad-bindings))))
    (setf next-level-of-ops next-level-of-ops-value)
    (setf (literal-bad-bindings root) bad-bindings)
    (setf psg-depth 0)
    (create-next-level-of-literals)
    ;; return value
    (list root)))
   
       
 
;; KILL BELOW
;;	 )))
;;    (setf next-level-of-ops 
;;          (list (relevant-ops-with-shared-lits goal root)))
;;    (setf psg-depth 0)
;;    (create-next-level-of-literals)
;;    ;; return value
;;    (list root)))
;;KILL ABOVE

;;This function's inputs are a list of literal nodes to be achieved in
;; next-level-of-ops  

;; next-level-of-ops consists of what's returned from
;;    relevant-ops-with-shared-lits
;;
;;The literals nodes will have their
;; children set to nil.  If children are created for the node, then
;; they will specify themselves as children of the node.
;;
;;
(defun create-next-level-of-literals ()
(unless (null next-level-of-ops)
;; (format t "~%Current depth is: ~s" psg-depth)
;; (format t "~%Current width is: ~s" (length next-level-of-ops))
  (if (eq *limit* psg-depth)
   (format t "~%PSG cut-off at level ~s,
      ~%If you wish to expand past this level re-run create-psgs."
 		  *limit*)
   (progn
    (let* (
       (this-level-of-ops        next-level-of-ops))
     (setq next-level-of-ops nil)
     (iter:iterate
     (iter:for literal-parent iter:in this-level-of-ops)
      (let (
        (each-ops-parent-node    (first  literal-parent))
        (each-ops-name           (second literal-parent))
        (each-ops-binding-list   (third  literal-parent))
        (each-ops-preconditions  (fourth literal-parent))
        (each-ops-effects        (fifth  literal-parent))
        (shared-lits             (sixth  literal-parent)))
       (setq siblings nil)
       (setq shared-literals shared-lits)
       (find-typing-information-for-shared-lits shared-lits)
       (n-mapcar #'create-operator 
                    each-ops-parent-node
                    each-ops-name
                    each-ops-binding-list
                    each-ops-preconditions
                    each-ops-effects))))
    (setf psg-depth (+ psg-depth 1))
    (create-next-level-of-literals)))))



;;
(defun create-operator (parent-node name blist preconds effects
                  &aux (goal-stack (cons (literal-name parent-node)
                                   (literal-goal-stack parent-node))))
 (let* (
   (operator        (make-operator
					 :name name
					 :parent parent-node
					 :bindings blist
					 :label nil
					 :failure-condition nil
					 :effects effects
					 :necessary-effects nil)))
  (assign-parent-extra-child parent-node operator)
  (find-typing-information-for-op-node operator preconds)
  (create-preconditions-levels preconds operator goal-stack));;create children

 nil)   ;;return value


;;----------------------------
;; create-preconditions-levels
;; Returns precond-node, where precond-node is the top level precondition.
;; This function may be called by an operator, or by an InternalNode.
;; The function starts with the top level precondition and decides
;; whether to build an Internal or Literal node (based upon whether
;; or not top-level precondition is a keyword).
;; notes:
;;  The function checks for series of clauses, (ex. (and (P1) (P2) (P3)) ).
;;  The preconditions are already bound

(defun create-preconditions-levels (preconditions parent goal-stack)
;;(format t "~%preconditions: ~s" preconditions)
 (cond
  ;; if first precondition is NOT 
  ((eq (first preconditions) '~)        ;; not
   ;; check for negated exists
   (if (eq (first (second preconditions)) 'EXISTS)
      ;;This is an interesting case.  If there is a negated exists, then
      ;; substitute the logical equivalent since
      ;;  NOT EXISTS P G F = FORALL P G ~F
      ;; Furthermore, Prodigy also accepts exists statements without
      ;;   a function.  These get transformed as follows:
      ;;  NOT EXISTS P G   = FORALL P G ~G
      ;;
      ;;No NOT node is created, we just go back to start of this
  	  ;;function and pretend we started all along with the logical 
      ;;equivalent.
    (create-preconditions-levels (negate-sentence (second preconditions))
                                             parent goal-stack)

    ;; else must be negated literal
    (Analyze-literal preconditions parent goal-stack)))

   ;; if first precondition is AND
   ((eq (first preconditions) 'AND)
    (create-internal-node 
     'AND (rest preconditions) parent goal-stack nil nil nil))

   ;; if first precondition is OR
   ((eq (first preconditions) 'OR)
    (create-internal-node 
     'OR (rest preconditions) parent goal-stack nil nil nil))

   ;; if first precondition is FORALL
   ;;  ignore parameter list (second preconditions)
   ((eq (first preconditions) 'FORALL)
    (create-internal-node 'FORALL 
                          `(,(negate-sentence (third preconditions)) ,(fourth preconditions))
			  parent goal-stack (second preconditions)
              (negate-sentence (third preconditions)) (fourth preconditions)));;negated the generator

   ;; if first precondition is EXISTS
   ;;  ignore parameter list (second preconditions)
   ((eq (first preconditions) 'EXISTS)
    (create-internal-node 'EXISTS (nthcdr 2 preconditions)
			  parent goal-stack (second preconditions)
              (third preconditions) (fourth preconditions)))

   ;; If first term of preconditions is not a keyword 
   ;; (ie it isn't NOT, AND, OR, EXISTS, or FORALL) 
   ;; then it must be a predicate
   (T (Analyze-literal preconditions parent goal-stack))))





;;---------------------
;; create-internal-node
;; create an internal node and label it with the appropriate keyword
;; call Analyze_Preconditions with each of its operands
(defun create-internal-node (keyword operands parent goal-stack params
                             generator f-expression)

 (let (
   (internal   (make-InternalNode
				:name keyword
				:parent parent
				:label nil
				:failure-condition nil
                :necessary-effects nil
                :anded-static-preds 
                   (find-anded-static-preds-for-node keyword operands)
                :params params
                :generator generator
                :f-expression f-expression
				:necessary-effects nil
				:goal-stack goal-stack)))
  (assign-parent-extra-child parent internal)
  (iter:iterate
  (iter:for operand iter:in operands)
    (iter:collect (create-preconditions-levels operand internal
										  goal-stack))))

  ;; return value
 nil)
    
;;----------------
;; analyze-literal (also creates the literal)
;; First check to see if the literal makes any sense (literals on
;; the false list will never occur in prodigy).
;; If the literal already exists at this level point to it.
;;   (this is when its in shared-literals)
;;
;; Otherwise create a literal node, lit.

(defun Analyze-literal (possibly-negated-literal parent goal-stack)
 (let (
   (equal-sibling     (first (member possibly-negated-literal siblings
                             :test #'(lambda (x y) (equal x 
                                      (literal-name y)))))))

  (if equal-sibling  ;;If it's already been created...
   (progn
   (setf (literal-shared equal-sibling)  ;;mark the shared field
      (cons parent (Literal-shared equal-sibling)))  ;;by adding
                                                     ;;another parent
   (assign-parent-extra-child parent equal-sibling))

    ;; if no equal-sibling exists for it...
    ;; check for nonsensical literal, if so don't create it.
   (unless (illegitimate-precondition? possibly-negated-literal)
    (progn
    (setq literal (create-and-label-literal possibly-negated-literal
									  parent goal-stack))
    (assign-parent-extra-child parent literal)
    (setq siblings (cons literal siblings))))))  ;;add this guy to list of
											 ;;siblings

  ;;return value
 nil)


;;------------------
;; illegitimate-precondition?
;; False list can't have var
(defun illegitimate-precondition? (precond)
  (member precond *false-list* :test #'equal))
  
;;-------------------------
;; create-and-label-literal
;; Returns newly created labeled literal
;; Labeling: if the literal holds in the current state, label the 
;; literal s-holds.  If the predicate of the literal always holds
;; (non-gratis-list) label the literal p-holds.  If the literal
;; appears on the goal-stack, label the literal gs-cycle.  If the
;; predicate of the literal appears on the goal-stack, label the
;; literal unknown.  If the predicate of the literal is too expensive
;; to expand (dont-expand-list) label the literal unknown.  If
;; there are no operators which achieve the literal, label the
;; literal unachievable. Otherwise call create operator-levels
;; with literal as subgoal.
(defun create-and-label-literal (literal parent goal-stack)
;;(format t "~%literal: ~s" literal)
  (let* (
    (is-shared-lit (member literal shared-literals :test #'equal))
    (literal-node  (make-Literal 
  			        :name literal
                                :bad-bindings nil
	 		        :parent parent
			        :necessary-effects nil
		            :bindings nil
			        :failure-condition nil
				    :completely-shared is-shared-lit
		            :goal-stack goal-stack)))

    (cond

     ;; check for non gratis literal, if so label lit p-holds.
     ((non-gratis? literal)
      (setf (Literal-label literal-node) 'P-HOLDS))

     ;; otherwise, if the literal holds on goal-stack then label it s-holds. 
     ((holds? literal goal-stack)
      (setf (Literal-label literal-node) 'S-HOLDS))

     ;; otherwise, if lit appears on goal stack label lit gs-cycle or unknown,
     ((setf (Literal-label literal-node)
	  (check-goal-stack literal goal-stack)) nil)

 	 ;; if literal is too expensive to expand then label it unknown
	 ((dont-expand? literal)
	  (setf (Literal-label literal-node) 'UNKNOWN))

	  ;;otherwise
	  ;; if some ops achieve literal put it in the next
      ;; level of operators, leaving the literal's label as nil.

     (t 
      (let* (
        (operators-bad-bindings (relevant-ops-with-shared-lits literal literal-node))
        (operators              (first operators-bad-bindings))
        (bad-bindings           (first (second operators-bad-bindings))))
       (setf (literal-bad-bindings literal-node) bad-bindings)
       (cond 
        (operators 
          (if (OR (not is-shared-lit)
                  (AND is-shared-lit *EXPAND-BELOW-SHARED-LITERALS*))
           (setf next-level-of-ops   
            (cons operators next-level-of-ops))  ;;expand below node
           (setf (literal-label literal-node) 'UNKNOWN)));;don't expand

      ;;if the operators returned to achieve the literal were nil,
	  ;;then it must be unachievable

        (t (setf (literal-label literal-node) 'UNACHIEVABLE))))))


    ;; return value
    literal-node))



;;-------------
;; dont-expand?
;;KILL BELOW
;;(defun dont-expand? (p)
;;  (member p *dont-expand-list*
;;	  :test #'(lambda (p1 p2)
;;		    (or (eq (first p1) p2)
;;			(and (eq (first p1) '~)
;;			     (eq (first (second p1)) p2))))))
;;KILL ABOVE
(defun dont-expand? (p)
  (not (not (member p *dont-expand-list*
	  :test #'(lambda (pred1 pred2) 
                   (match-nonvar-or-var-to-var-only pred1 pred2))))))


;;------------
;; non-gratis?
(defun non-gratis? (p)
  (member p *non-gratis-list*
	  :test #'(lambda (p1 p2)
		    (or (eq (first p1) p2)
			(and (eq (first p1) '~)
			     (eq (first (second p1)) p2))))))




;;-------------------------
;; check-goal-stack
;; if lit appears on goal stack label the lit gs-cycle or unknown,
(defun check-goal-stack (literal goal-stack)

  ;; exact match => gs-cycle
  (let ((label
	 (iter:iterate
	  (iter:for goal iter:in goal-stack)
	  (if (equal goal literal)    
	      (iter:collect
	       'GS-CYCLE)))))

    ;; if there wasn't a gs-cycle check for recursion
    (if (null label)
	(setq 
	 label
	 (check-for-recursion literal goal-stack))
      (first label))))  ;; return just the atom GS-CYCLE not (GS-CYCLE)
	   

;;--------------------
;; check for recursion
(defun check-for-recursion (literal goal-stack)
  (if goal-stack
      (let* ((goal (first goal-stack))
	     (label
	      ;; if the literal is negated check against negated goals
	      (cond ((eq (first literal) '~)  
		     (if (and 
			  (eq (first goal) '~)
			  (eq (first (second literal))    ;; check predicates
			      (first (second goal))))
			 'UNKNOWN))       ;; if they match => unknown
	       
		    ;; if the literal is not negated check goal predicate
		    ((eq (first literal)             
			 (first goal))
		     'UNKNOWN))))
	(if  (null label)         ;; still no label
	     (check-for-recursion literal (rest goal-stack))
	  label))))
	     
	    
(defun assign-parent-extra-child (parent-node child-node)
 (cond
  ((internalnode-p parent-node)
   (setf (internalnode-operands parent-node) 
          (append (internalnode-operands parent-node) (list child-node))))
                              ;;childnode must go last not to
                              ;;mess up generators etc.
  ((operator-p parent-node)
   (setf (operator-preconditions parent-node) child-node))

  ((literal-p parent-node)
   (setf (literal-operators parent-node)
          (cons child-node (literal-operators parent-node))))))

