;
;
;	BUILD II  --  a robotic assembly planning system
;
;						John Nagle
;						Version 1.28 of 5/19/87
;
;       Vector and matrix operations, geometric transformations.
;
;	Unless otherwise specified, only apply to vectors with three elements.
;
(require 'builddefs "builddefs")
(use-package 'builddefs)
(provide 'vector)
(in-package 'vector)
(export '(
	  aeq
	  clean
	  vector3
	  v3zerop
	  v3equalp
	  v3dot
	  v3mag
	  v3norm
	  v3plus
	  v3diff
	  v3negate
	  v3scale
	  v3cross
	  setf-vector3
	  location
	  rotation-type
	  plane
	  convert-vertices
	  convert-planes
	  convert-vertex
	  m3x3mult
	  relate-location
	  unrelate-location
	  make-vector3
	  copy-vector3
	  vector3-p
	  vector3-x
	  vector3-y
	  vector3-z
	  make-location
	  copy-location
	  location-p
	  location-rotation
	  make-rotation
	  rotation-from-angles
	  angles-from-rotation
	  make-plane
	  copy-plane
	  plane-p
	  make-plane-from-points
	  reverse-plane
	  plane-distance
	  point-plane-distance
	  point-plane-test
	  point-point-distance-squared
	  vec
	  loc
	  print-vector3
	  print-location
	  ))
(use-package '(builddefs))
;
;      aeq  --  approximately equal
;
;	Used in geometric tests.
;
(defun aeq (a b)
  (> tol1 (abs (- a b))))
;
;      Functions to remove roundoff crud.
;
;	clean  --  if near an integer value, return that value.
;
(defun clean (n &aux x)
  (cond ((< (abs (- n (setq x (floor n)))) tol2) x)
	((< (abs (- n (setq x (ceiling n)))) tol2) x)
	(t n)))
;
;	vector3  --  vector of three elements
;
(defstruct (vector3 (:print-function print-vector3)
		    (:constructor make-vector3 (x y z)))
  (x 0.0 :type long-float)
  (y 0.0 :type long-float)
  (z 0.0 :type long-float))
;
;	v3zerop  --  is vector all zero?
;
;	Tolerance values apply for this test.
;
(defun v3zerop (v)
  (declare (type vector3 v))
  (not (or
	(> (vector3-x v) tol1) (< (vector3-x v) mtol1)
	(> (vector3-y v) tol1) (< (vector3-y v) mtol1)
	(> (vector3-z v) tol1) (< (vector3-z v) mtol1))))
;
;	v3equalp  -- are two vectors equal?
;
;	Tolerance values apply for this test.
;
(defun v3equalp (v1 v2)
  (declare (type vector3 v1 v2))
  (v3zerop (v3diff v1 v2)))		; could be more efficient.
;
;	v3dot  --  dot product
;
(defun v3dot (v1 v2)
  (declare (type vector3 v1 v2))
  (+
   (* (vector3-x v1) (vector3-x v2))
   (* (vector3-y v1) (vector3-y v2))
   (* (vector3-z v1) (vector3-z v2))))
;
;	v3mag  --  scalar magnitude of vector
;
(defun v3mag (v) 
  (declare (type vector3 v))
  (sqrt (v3dot v v)))
;
;	v3norm  --  normalized vector
;
(defun v3norm (v) 
  (declare (type vector3 v))
  (v3scale v (/ (v3mag v))))
;
;	v3plus  --  vector sum
;
(defun v3plus (v1 v2)
  (declare (type vector3 v1 v2))
  (make-vector3
   (+ (vector3-x v1) (vector3-x v2))
   (+ (vector3-y v1) (vector3-y v2))
   (+ (vector3-z v1) (vector3-z v2))))
;
;	v3diff  --  vector difference
;
(defun v3diff (v1 v2)
  (declare (type vector3 v1 v2))
  (make-vector3
   (- (vector3-x v1) (vector3-x v2))
   (- (vector3-y v1) (vector3-y v2))
   (- (vector3-z v1) (vector3-z v2))))
;
;	v3negate  --  vector negation
;
(defun v3negate (v)
  (declare (type vector3 v))
  (make-vector3
   (- (vector3-x v))
   (- (vector3-y v))
   (- (vector3-z v))))
;
;	v3scale  --  vector scaled by scale factor
;
;	The scale factor may be any numeric type.
;
(defun v3scale (v s)
  (declare (type vector3 v))
  (make-vector3
   (* (vector3-x v) s)
   (* (vector3-y v) s)
   (* (vector3-z v) s)))
;
;	v3cross  --  vector cross product of two vectors
;
(defun v3cross (v1 v2)
  (declare (type vector3 v1 v2))
  (make-vector3
   (- (* (vector3-y v1) (vector3-z v2)) (* (vector3-z v1) (vector3-y v2)))
   (- (* (vector3-z v1) (vector3-x v2)) (* (vector3-x v1) (vector3-z v2)))
   (- (* (vector3-x v1) (vector3-y v2)) (* (vector3-y v1) (vector3-x v2)))))
;
;	setf-vector3  --  set vector3 into object containing a vector3.
;
;	Modifies the TARGET argument.
;
(defun setf-vector3 (target v)
  (setf (vector3-x target) (vector3-x v))	; copy X value
  (setf (vector3-y target) (vector3-y v))	; copy Y value
  (setf (vector3-z target) (vector3-z v))	; copy Z value
  target)					; return modified value
;
;	Higher-level constructs
;
;
;	plane  --  normal vector plus a distance from the origin
;
(defstruct (plane (:include vector3))
	(distance 0.0 :type long-float))
;
;	rotation  --  a 3x3 rotation matrix
;
;	Can specify a rotation, but not a scaling.
;
(deftype rotation-type () `(array long-float (3 3)))
;
;	identity-rotation  --  the no-rotation identity matrix
;
(unless (boundp 'identity-rotation)	; avoid reload trouble
  (defconstant identity-rotation
	       (make-array '(3 3)
			   :element-type 'long-float
			   :initial-contents '((1.0 0.0 0.0)
					       (0.0 1.0 0.0)
					       (0.0 0.0 1.0)))))
;
;	Latitude/longitude calculations for rotations.
;
;	Latitudes range from -0.25 to 0.25.  Positive values represent +Z.
;	Longitudes range from 0 to 1.
;	The XY plane is the equator.
;	The +X direction is the prime meridian (lng=0).
;	CCW viewed from +Z is increasing longitude.
;
;
;	make-rotation  --  make rotation matrix given rotations about
;			      X, Y, and Z axes.
;
;	Rotation values are of the form 1=full circle.
;
(defun make-rotation (rx ry rz)
  (let ((rot 	(make-array '(3 3) :element-type 'long-float))
	(cx	(cos (* rx twopi)))	; sines and cosines of rotation
	(cy	(cos (* ry twopi)))
	(cz	(cos (* rz twopi)))
	(sx	(sin (* rx twopi)))
	(sy	(sin (* ry twopi)))
	(sz	(sin (* rz twopi))))
       (store (rot 0 0)	(* cy cz)) ; build new rotation matrix
       (store (rot 0 1)	(* cy sz))
       (store (rot 0 2)	(- sy))
       (store (rot 1 0)	(- (* sx sy cz) (* cx sz)))
       (store (rot 1 1)	(+ (* sx sy sz) (* cx cz)))
       (store (rot 1 2)	(* sx cy))
       (store (rot 2 0)	(+ (* cx sy cz) (* sx sz)))
       (store (rot 2 1)	(- (* cx sy sz) (* sx cz)))
       (store (rot 2 2)	(* cx cy))
rot))
;
;	Latitude/longitude calculations for rotations.
;
;	Latitudes range from -0.25 to 0.25.  Positive values represent +Z.
;	Longitudes range from 0 to 1.
;	The XY plane is the equator.
;	The +X direction is the prime meridian (lng=0).
;	CCW viewed from +Z is increasing longitude.
;
;
;	angles-from-rotation  -- returns latitude and longitude
;				    given rotation matrix.
;
;	Returns (lat . lng)
;
;	Returned angles are in form 1.0=full circle.
;
(defun angles-from-rotation (orient)
  (let* ((x (aref orient 0 0))		; X axis unit vector rotated.
	 (y (aref orient 0 1))
	 (z (aref orient 0 2))
	 (latitude (asin z))		; latitude
	 (xymag (sqrt (+ (sq x) (sq y)))) ; magnitude in equatorial plane
	 (longitude (if (aeq xymag 0.0) ; if at N or S pole, irrelev.
			0.0
			(if (< x 0.0); angle from prime meridian
			    (- pi (fix-complex (asin (/ y xymag))))
			    (fix-complex (asin (/ y xymag)))))))
	;	Constrain longitude to range 0..1, and return results.
	(cons (/ latitude twopi)
	      (mod (1+ (/ longitude twopi)) 1.0)))) ; mod in KCL is bad for <0
;
;	rotation-from-angles  --  inverse of angles-from-rotation
;
;
(defun rotation-from-angles (lat lng)
  (let ((latrot (make-rotation 0.0 (- lat) 0.0)); rotation about Y
	(lngrot (make-rotation 0.0 0.0 lng)))	; rotation about Z
	(m3x3mult latrot lngrot)))		; first Y, then Z.
;
;	fix-complex  --  fix unwanted complex number
;
;	KCL's asin and acos sometimes produce complex numbers with small
;	imaginary parts for values in the range -1.0..1.0.
;
(defun fix-complex (val)
  (assert (< (abs (imagpart val)) 1.0E-10)) ; trap non-trivial imaginary part
  (realpart val))			; return real part
;
;	location  --  a rotation and position in space
;
(defstruct (location
	    (:include vector3)
	    (:print-function print-location))
  (rotation identity-rotation :type rotation-type))
;  
;	Transforms using locations.
;
;	convert-vertices  --  convert array of vertices in block space to 
;			    real space.
;
;	Converts block defined by A to be at LOC, returns new points.
;
(defun convert-vertices (a loc)
  (check-type a vector)				; must be array of points
  (map '(vector vector3)	
       (function
	(lambda (pnt)
		(convert-vertex pnt loc)))
       a))				 ; apply convert-vertex to all
;
;     	convert-planes  --  convert array full of planes.
;
;	Converts block at A to be at LOC.  Stores result in B.
;
(defun convert-planes (a loc 
		   &aux (rot (location-rotation loc)))
  (check-type a vector)			; must be array
  (map '(vector plane)			; construct new array
       (function
	(lambda (pln)
		(check-type pln plane)	; must be a plane
		(let ((x (vector3-x pln))
		      (y (vector3-y pln))
		      (z (vector3-z pln)))
		     (make-plane
		      ;	Rotated normal vector of plane.
		      :x
		      (+ (* (aref rot 0 0) x)
			 (* (aref rot 1 0) y)
			 (* (aref rot 2 0) z))
		      :y
		      (+ (* (aref rot 0 1) x)
			 (* (aref rot 1 1) y)
			 (* (aref rot 2 1) z))
		      :z
		      (+ (* (aref rot 0 2) x)
			 (* (aref rot 1 2) y)
			 (* (aref rot 2 2) z))
		      ;	Distance from origin after move.
		      :distance
		      (+ (plane-distance pln) (v3dot loc pln))))))
       a))					; apply to old array
;
;	convert-vertex  --  convert a single point or vector per LOC
;
(defun convert-vertex (pnt loc)
  (check-type pnt vector3)	; must be vector3
  (let ((x (vector3-x pnt))	; extract coords of point
	(y (vector3-y pnt))
	(z (vector3-z pnt))
	(rot (location-rotation loc)))	; rotation matrix
       ;	Construct new point
       (make-vector3
	(+ (vector3-x loc) 	; X component
	   (* (aref rot 0 0) x)
	   (* (aref rot 1 0) y)
	   (* (aref rot 2 0) z))
	(+ (vector3-y loc) 	; Y component
	   (* (aref rot 0 1) x)
	   (* (aref rot 1 1) y)
	   (* (aref rot 2 1) z))
	(+ (vector3-z loc) 	; Z component
	   (* (aref rot 0 2) x)
	   (* (aref rot 1 2) y)
	   (* (aref rot 2 2) z)))))
;
;	m3x3mult  --  3x3 matrix multiply
;
;	Used on location matrices.
;
(defun m3x3mult (a b)
  (declare (type (array long-float (3 3)) a))
  (declare (type (array long-float (3 3)) a))
  (check-type a (array long-float (3 3)))
  (check-type b (array long-float (3 3)))
  (let ((c (make-array '(3 3) :element-type 'long-float))) ; result matrix
       (do ((row 0 (1+ row)))		; row loop
	   ((= row 3) nil)
	   (do ((col 0 (1+ col)))	; column loop
	       ((= col 3))
	       (setf (aref c row col)	; inner loop written out
		     (+
		      (* (aref a row 0) (aref b 0 col))
		      (* (aref a row 1) (aref b 1 col))
		      (* (aref a row 2) (aref b 2 col))))))
       c))
;
;      relate-location  --  Create location equivalent to A, but expressed
;			    relative to B.
;
(defun relate-location (a b)
  (declare (type location a))
  (declare (type location a))
  (check-type a location)
  (check-type b location)
  (let ((arot (location-rotation a))	; rotatation component
	(brot (location-rotation b))	; rotatation component
	(trans (v3diff a b))		; translation component
	(c (make-array '(3 3) :element-type 'long-float))); new working array
       ;	Generate new rotation matrix.
       ;	This is a matrix multiply with a transpose.
       (do ((row 0 (1+ row)))
	   ((= row 3) nil)
	   (do ((col 0 (1+ col)))
	       ((= col 3))
	       (let ((sum 0.0))
		    (do ((i 0 (1+ i)))
			((= i 3) nil)
			(setq sum (+ sum
				     (* (aref brot col i)
					(aref arot row i)))))
		    (store (c row col) sum))))
       ;	Generate translation vector and return result.
       (make-location
	:x (+ (* (aref brot 0 0) (vector3-x trans))
	      (* (aref brot 0 1) (vector3-y trans))
	      (* (aref brot 0 2) (vector3-z trans)))
	:y (+ (* (aref brot 1 0) (vector3-x trans))
	      (* (aref brot 1 1) (vector3-y trans))
	      (* (aref brot 1 2) (vector3-z trans)))
	:z (+ (* (aref brot 2 0) (vector3-x trans))
	      (* (aref brot 2 1) (vector3-y trans))
	      (* (aref brot 2 2) (vector3-z trans)))
	:rotation c)))
;
;	unrelate-location  --  inverse of relate-location
;
;	Effectively combines two locations into one.
;
(defun unrelate-location (a b)
  (declare (type location a))
  (declare (type location a))
  (check-type a location)
  (check-type b location)
  (location-from-rotation-and-translation
   (m3x3mult (location-rotation a)
	     (location-rotation b))		; new rotation
   (convert-vertex a b)))	; new translation
;
;	location-from-rotation-and-translation
;
;	Generate location object given rotation and translation
;
(defun location-from-rotation-and-translation (rot trans)
  (make-location
   :rotation rot
   :x (vector3-x trans)
   :y (vector3-y trans)
   :z (vector3-z trans)))
;
;	point-plane-distance  --  distance from point to plane
;
;	>0 indicates outside, <0 indicates inside.
;
(defun point-plane-distance (point pln)
  (- (v3dot point pln) (plane-distance pln)))
;
;      point-plane-test --  find out if point is inside, outside, or on plane.
;
(defun point-plane-test (pnt pln) 
  (let ((dist (point-plane-distance pnt pln)))	; distance from point to plane
       (cond ((> dist tol1) 'outside)		; outside if > 0
	     ((< dist mtol1) 'inside)		; inside if < 0
	     (t 'on))))				; on if near 0.
;
;      point-point-distance-squared  --  square of distance between two points.
;
(defun point-point-distance-squared (a b)
  (+ (sq (- (vector3-x a) (vector3-x b)))
     (sq (- (vector3-y a) (vector3-y b)))
     (sq (- (vector3-z a) (vector3-z b)))))
;
;      make-plane-from-points  --  make a plane through p1, p2, p3
;
;	Returns NIL if collinear points.
;
(defun make-plane-from-points (p1 p2 p3)
  (let* ((perpen (v3cross (v3diff p2 p1)	; perpendicular to plane
			  (v3diff p3 p1))))
	(when (v3zerop perpen) 			; if points collinear
	      (return-from make-plane-from-points nil))	; no good
	(let ((normal (v3norm perpen)))		; normalize normal vector
	     (make-plane
	      :x (vector3-x normal)
	      :y (vector3-y normal)
	      :z (vector3-z normal)
	      :distance (v3dot normal p1)))))	; return new plane
;
;	reverse-plane  --  invert direction of a plane
;
(defun reverse-plane (pln)
  (check-type pln plane)			; must be a plane
  (make-plane
   :x (- (vector3-x pln))
   :y (- (vector3-y pln))
   :z (- (vector3-z pln))
   :distance (- (plane-distance pln))))		; return new plane
;
;	I/O functions
;
;	vec  --  input a vector3
;
(defun vec (x y z) (make-vector3 x y z))
;
;	loc  --  input a location
;
(defun loc (x y z &optional (latitude 0.0) (longitude 0.0))
  (make-location
   :x x
   :y y
   :z z
   :rotation (rotation-from-angles latitude longitude)))
;
;	print-vector3  --  print a vector3
;
;	Generates a VEC form.
;
(defun print-vector3 (v stream depth)
  (declare (ignore depth))
  (format stream "(vec ~a ~a ~a)" 
	  (vector3-x v) (vector3-y v) (vector3-z v)))
;
;	print-location  --  print a location
;
;	Generates a LOC form.
;
(defun print-location (loc stream depth)
  (declare (ignore depth))
  (labels
   ;	Format rotation as latitude and longitude, or void if zero.
   ((format-rotation
     (rot)
     (if (compare-rotation rot identity-rotation)	; if null rotation
	 ""
	 (let ((latlng (angles-from-rotation rot)))
	      (format nil " ~,4f ~,4f" (car latlng) (cdr latlng))))))
   ;	Main.
   (format stream "(loc ~,4f ~,4f ~,4f~a)"
	   (vector3-x loc) (vector3-y loc) (vector3-z loc)
	   (format-rotation (location-rotation loc)))))
;
;	compare-rotation  --  compare two rotation matrices for equality.
;
;	KCL's EQUALP takes a core dump if used for this.
;
(defun compare-rotation (rot1 rot2)
  (dotimes (i 3)
	   (dotimes (j 3)
		    (unless (= (aref rot1 i j) (aref rot2 i j))
			    (return-from compare-rotation nil))))
  t)				; success
