;
;	BUILD II  --  a blocks world
;
;						John Nagle
;						Version 1.12 of 5/19/87
;
(require 'robotio "robotio")
(require 'amlparse "amlparse")
(provide 'robottest)
;
;	Simple interaction with robot
;
(defconstant aml-true -1)			; TRUE in AML
(defconstant aml-false 0)			; FALSE in AML
;
;	Startup
;
;	robot-startup  --  start up robot if necessary
;
(defun robot-startup nil
  (simple-command-robot "bstart"))
;
;	robot-shutdown  --  turns off hydraulics
;
(defun robot-shutdown nil
  (simple-command-robot "shutdown" nil))
;
;	robot-freeze  --  enters freeze mode
;
(defun robot-freeze nil
  (simple-command-robot "freeze" nil))
;
;	simple-command-robot  --  issue command, expect specified value.
;
(defun simple-command-robot (cmd &optional (expected aml-true))
  (let ((stat (parse-aml-expression (command-robot cmd))))
       (unless (equal stat expected)
	       (error "Robot did not properly execute '~a'~%  Expected ~a, received ~a" cmd expected stat))))
;
;	Calibration info
;
;
;	"BCALIB" actually gets the location of the post.
;
;
;	post-offset  --  location of the top of the alignment post.
;
(defparameter post-offset (make-vector3 0.0 100.0 87.0))
(defparameter grip-force 400.0)		; gripping force (grams)
(defparameter hiz 200.0)		; Z value that clears everything
(defvar calibration-offset nil)
;
;	getpost  --  get coords of post
;
(defun getpost nil
  (let ((post1 (parse-aml-expression (command-robot "post1"))))
       (unless (and (listp post1) (= (length post1) 3))
	       (error "Robot returned: ~a" post1))
       (setq calibration-offset post1)))
;
;	move-robot  --  move to indicated loc, biased by calibration
;
(defun move-robot (pos)
  (let ((newpos (transpos pos)))
       (simple-command-robot (format nil "MOVE (<jx,jy,jz>, <~a, ~a, ~a>)"
			      (first newpos)
			      (second newpos)
			      (third newpos)))))
;
;	transpos  --  translate position to robot coordinates
;
(defun transpos (pos)
  (unless calibration-offset (getpost))	; get calibration if needed
  (setq pos (v3diff pos post-offset))  ; our pos relative to post
  (list
   (+ (- (vector3-y pos)) (first calibration-offset))
   (+ (vector3-x pos) (second calibration-offset))
   (+ (vector3-z pos) (third calibration-offset))))
;
;	robot-move-block  --  move block with robot
;
(defun robot-move-block (pos1 roll1 pos2 roll2)
  (when (< (vector3-z pos1) 0.0) (error "Pos1 too close to table"))
  (when (< (vector3-z pos2) 0.0) (error "Pos2 too close to table"))
  (let* ((jpos1 (transpos (grip-offset pos1)))
	 (jpos2 (transpos (grip-offset pos2)))
	 (stat (parse-aml-expression
		(command-robot (format nil
				       "MV1(<~,1f,~,1f,~,1f,~,1f>,<~,1f,~,1f,~,1f,~,1f>,~,1f,~,1f)"
				       (first jpos1)
				       (second jpos1)
				       (third jpos1)
				       roll1
				       (first jpos2)
				       (second jpos2)
				       (third jpos2)
				       roll2
				       hiz
				       grip-force)))))
	(unless (equal stat -1)
		(error "Move failed: ~a" stat)))
t)
;
;	grip-offset  --  add offset for gripping
;
;	Very dumb approach.
;	Only works for blocks about 50mm in size.
;
(defun grip-offset (jpos)
  (when (< (vector3-z jpos) 10.0) (error "Collision with table at ~a" jpos))
  (v3plus jpos (make-vector3 0.0 0.0 0.0)))
;
;	doplan  --  carry out a plan
;
;	***VERY PRIMITIVE  --  MANY CHECKS ABSENT***
;
(defun doplan (plan)
  (robot-startup)
  (map nil 
       (function
	(lambda (step)
		(unless (eq (first (third step)) 'nop)
			(format t
				"Action: ~a ~%        [because ~a]~%"
				(third step)
				(second (fourth step)))
			;	Move indicated blocks
			(cond ((eq (first (third step)) 'move)
			       (domove (second (third step)) ; which
				       (third (third step)) ; from
				       (fourth (third step)) ; to
				       ))
			      (t (format t "BAD ACTION ~a in plan.~%" 
					 (third step)))))))
       plan))
;
;	domove  --  move indicated block
;
(defun domove (blk from to)
  (let ((stat
	 (robot-move-block from (roll-component from)	; do the move
			   to (roll-component to))))
       (when (eq stat t)			; update world model
	     (in-context real-world-context
			 (verify-at blk from)
			 (remblock blk)
			 (addat blk to)))))
;
;	roll-component  --  extract roll component for grasp from rotation.
;
;	Built into this is the assumption that all blocks can and should
;	be grasped across their Y dimension.
;
(defun roll-component (loc)
  (let ((latlong (angles-from-rotation (location-rotation loc))))
       (unless (aeq (car latlong) 0.0)
	       (error "Can't handle non-vertical positions yet: ~a" latlong))
       (let* ((roll (+ 45.0 (* -360.0 (cdr latlong)))) ; get roll angle
	      (bounded-roll
	       (cond ((< roll -135.0) (+ roll 360.0))
		     ((> roll 135.0) (- roll 360.0))
		     (t roll))))
	     (when (or (> bounded-roll 135.0) (< bounded-roll -135.0))
		   (error
		    "Orientation of ~a results in roll of ~a which is not possible for robot"
		    latlong bounded-roll))
	     bounded-roll)))
;
;	buildit  --  build to spec, starting from real world
;
(defun buildit (goalstate)
   (format t "Beginning planning.~%")
   (setq buildplan (dobuild real-world-context goalstate))
   (when buildplan
        (format t "Beginning execution of plan.~%")
	(doplan buildplan)))
;
;	set-real-world  --  set real world context to given state
;
(defun set-real-world (cntxt)
  (setq real-world-context (in-context cntxt (push-context))))
