;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.10 of 3/21/87
;
(require 'builddefs "builddefs")
(require 'database "database")
(use-package 'builddefs)
(use-package 'database)
;
;	Planning utility functions.
;
;	verify-at  --  make sure B is at LOC in current context.
;
;	Failure is fatal.
;
(defun verify-at (b loc)
	(unless (equalp loc (getat b))
		(error "~a is not at ~a" b loc)))
;
;	verify-unhit  --  make sure B is not hitting anything.
;
;	Touching is OK.
;	If B is nil, we do a full collision test.
;
(defun verify-unhit (&optional (b nil) &aux hits)
  (cond (b (setq hits (gettouch b))
	   (when (eq (first hits) 'hit)		; single-block test
		 (error "~a: unexpected collision - ~a"
			b hits))) 
	(t (setq hits (gettouch))		; total test
	   (when (eq (first hits) 'hit)
		 (error "Unexpected collision - ~a"
			hits)))) t) 
;
;	verify-stable  --  make sure B is stable.
;
;	If B is nil, we do a full stability test.
;
(defun verify-stable (&optional (b nil) &aux losers)
  (cond (b (setq losers (checkstab (list b)))	; single-block test
	   (unless (eq losers t)
		 (error "~a: unexpected instability - ~a"
			b losers))) 
	(t (setq losers (checkstate))		; total test
	   (unless (eq losers t)
		 (error "Unexpected instability - ~a"
			losers)))) t) 
;
;	tracks  --  print logging information about plan
;
(defun tracks (&rest args &aux msg)
	(setq msg (apply #'format nil args))
	(format t "~a> ~a~%" (ndashes goaldepth) msg))
;
;	ndashes  -- returns a string of N dashes
;
(defun ndashes (n)
  (cond ((< n 1) "")
	((= n 1) "-")
	((= n 2) "--")
	((= n 3) "---")
	((= n 4) "----")
	(t (concatenate 'string "-----" (ndashes (- n 5)))))) ; general case
;
;	pname-of-fn  --  get print name of function
;
;	There is probably some better way to do this.
;
(defun pname-of-fn (fn)
	(cond ((not (functionp fn)) fn)	; not function, just return self
		((compiled-function-p fn)
		(format nil "~a" fn)) ; convert name to string
		((not (listp fn)) fn)	; not list, not compiled?
		((eq (first fn) 'lambda-block) (second fn))
		((eq (first fn) 'lambda-block-closure) (fifth fn))
		(t fn)))		; unknown, return self.
;
;	dumpstate  --  dump where all blocks are.
;
(defun dumpstate (&optional (context context))
  (format t "~a:~%" context)
  (map nil 
       (function
	(lambda (item)
		(format t "  ~a~%" item)))
       (context-lookup '(at * *))))
;
;	showplan  --  show each step in a plan
;
;	Produces pictorial output, pausing after each step.
;
(defun showplan (plan &aux (n 1) (msg "Initial state") step)
  (unless (eq (first (first plan)) (second (first plan)))
	  (in-context (first (first plan)) (show msg)))
  (loop
   (setq step (pop plan))			; next step of plan
   (unless step (return))			; end of list
   (setq msg (format nil "After ~astep ~a - ~a" 
		     (if plan "" "final ") n (third step)))
   (in-context (second step) (show msg))	; display picture of state
   (pause msg)					; wait for user
   (incf n)))
;
;	showcontext  --  show a specific context
;
;	Used as a tracing aid
;	Rebinds context.
;
(defun showcontext(context msg)
	(show msg)
	(pause msg))
;
;	pause  --  wait for user input
;
;	Used when doing graphics.
;
(defun pause (&optional (msg ""))
	(format t "~a~%   Press RETURN to continue.~%" msg)
	(clear-input)			; flush input
	(when (listen) (read-line))	; clear-input doesn't flush properly
	(read-line))			; read a line and wait.
