;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.4 of 3/2/87
;
;
;	Utility routines for scaffolding and counterweights.
;
(require 'builddefs "builddefs")		; prerequisite
;
;	getspares  --  get list of all spare blocks
;
;	Sorted so least buried is first.
;	Blocks not in place yet have priority.
;	Ignore block SPB or any member of SUPL.
;	SGB is unused in the original.
;
(defun getspares (sgb spb sset supl sg
		      &aux goodspares badspares item z)
  ;	Examine all blocks; construct lists of spares.
  (mapc
   (function
    (lambda (x)
	    (cond ((eq x spb))
		  ((setq item (present (list 'in-place x '*)))
		   (unless (member (third item) supl)
			   (push x badspares)))
		  ((setq z (matchl x sset))
		   (setq sset (remmem z sset)))
		  ((push x goodspares)))))
   (getobs))
  (list (supsort goodspares) (supsort badspares)))
;
;	matchl  --  return first member of L that matches block B.
;
(defun matchl (b l)
  (cond ((null l) nil)
	((match b (first l)) (first l))
	(t (matchl b (rest l)))))
;
;	getsgblock  --  get SG block that B matches.
;
(defun getsgblock (b)
  ;	Expect to find an 'in-place database item.
  (let ((item (present (list 'in-place b '*))))
       (unless item
	       (error "~a not in place - getsgblock" b))
       (third item)))			; return matching block of SG state.
;
;	remmems  --  remove members of L1 from L2
;
;	This is SET-DIFFERENCE in Common LISP.
;
(defun remmems (l1 l2)
	(set-difference l1 l2))
;
;	stabsubs  --  find minimal subsets of L which, if added, 
;		      make current state stable.
;
;	This appears to be an exhaustive combinatoric search with an
;	expensive CHECKSTATE cycle for every try.
;
(defun stabsubs (l loss sg
		   &aux (context context) (oldcontext context) ll y newloss)
  (mapc
   (function 
    (lambda (x)
	    ;	Obtain a new scratch context, a copy of the situation at entry.
	    (setq context (in-context oldcontext (push-context)))
	    (cond ((unreadysup x sg))
		  ((and (genblock1 (setq y (gensym))
				   (get x 'shape)
				   (get x 'size))
			(addat y (in-context sg (getat x)))
			(context-add (list 'in-place y x))
			nil))
		  ((eq t (setq newloss (checkstate)))
		   (setq ll (addset (list x) ll)))
		  ((eq 'hit (first newloss))
		   (error "Hit in stabsubs"))
		  ((worse (rest newloss) loss))
		  ((mapc
		    (function
		     (lambda (z) (setq ll (addset z ll))))
		    (stabsubs (remmem x l)
			      (rest newloss)
			      sg))))))
   l)
  ll)
;
;	addset  --  add X to list L. 
;
;	Flush any L members that contain other L members.
;
(defun addset (x l
		 &aux (ll l) outl)
  (loop
   (cond ((endp ll) (return-from addset (cons x outl)))
	 ((subsetp (first ll) x) (return-from addset l))
	 ((subsetp x (first ll)))
	 ((push (first ll) outl)))
   (pop ll)))

;
;	worse  --  see if new loss basically like old loss, only worse.
;
(defun worse (newloss oldloss)
  (cond ((null oldloss) t)
	((worse1 newloss (first oldloss))
	 (worse newloss (rest oldloss)))
	(t nil)))

(defun worse1 (newloss oldloss)
  (cond ((null newloss) nil)
	((eq (caar newloss) (first oldloss))
	 (worse2 (first newloss) oldloss))
	((worse1 (rest newloss) oldloss))))

(defun worse2 (newloss oldloss)
  (cond ((not (vzerop (third oldloss)))
	 (> (- (vdot (third newloss) (vnorm (third oldloss)))
	       (vmag (fourth oldloss)))
	    mtol1))
	(t nil)))
;
;	pmotion  --  find which way point is moving due to loss.
;
(defun pmotion (p loss
		  &aux x)
  (cond ((not (vzerop (third loss))) 
	 (return-from pmotion (third loss)))
	((vzerop (setq x (vcross (fourth loss)
				 (vdiff p (second loss)))))
	 (return-from pmotion '(0.0 0.0 0.0)))
	((return-from pmotion (vnorm x)))))
;
;	piercepoint  --  find point, if any, where plane of (facep L) is
;			 pierced by extending POINT in direction VEC.
;
;	***ORIGINAL MARKED UP HERE  --  NEEDS WORK***
;
(defun piercepoint (facep i point vec)
  (prog (pnorm x y)
	(setq pnorm (pvec facep i))
	(cond ((aeq 0.0 (setq x (- (facep i 3)
				   (vdot point pnorm))))
	       (return point))
	      ((aeq 0.0 (setq y (vdot vec pnorm))) (return nil))
	      ((minusp (setq x (/  x y))) (return nil))
	      ((return (vplus point (vscale vec (/  x y))))))))

;
;	interior  --  true if POINT is within boundaries of face I.
;
(defun interior (facep facel i point)
  ;	Examine all points on face I, fail if any fails test.
  (map nil
   (function
    (lambda (face)
	    (when (> (ptplz point facep face) tol1) 
		  (return-from interior nil))))
   (aref facel i 1))
  t)					; return T if all pass.
;
;	ptplz  --  like PTPL, but point is vector, not array and index.
;
(defun ptplz (pt pl n)
  (+ (* (first pt) (aref pl n 0))
     (* (second pt) (aref pl n 1))
     (* (third pt) (aref pl n 2))
     (- (aref pl n 3))))
;
;	pierce  --  see if ray from POINT actually pierces face.
;
;	Return POINT if so.
;
(defun pierce (facep facel i point vec)
  (let ((x (piercepoint facep i point vec)))
       (if (and x (interior facep facel i x))	; if pierces and inside
	   x					; return X
	   nil)))				; fails
;	
;	outmoving  --  see if vertex is moving outward.
;
(defun outmoving (vertl i facep vec)
  (map nil 
       (function 
	(lambda (vertex)
		(when (> (vdot (pvec facep vertex) vec) tol1) ; any outward?
		      (return-from outmoving t))))	; yes, T.
       (aref vertl i 1)))		; over vertex I
;
;	covered  --  see if point P on block B is covered by other blocks.
;
;	P must be a vertex of B.
;
(defun covered (b p
		  &aux touchblocks touchpoints)
  ;	Get list of touching blocks and iterate over them.
  (setq touchblocks (gettouch b))
  (when (eq 'hit (first touchblocks))	; collision not expected
	(error "Hit in covered"))
  (loop 
   (when (endp touchblocks) (return-from covered nil))
   (cond ((not (aeq 0.0 (ptplx p (cadar touchblocks))))
	  (pop touchblocks))
	 (t (setq touchpoints (cddar touchblocks))
	    ;	Iterate over touch points.
	    (loop
	     (when (endp touchpoints)	; end of touchpoints, do next block.
		   (pop touchblocks)
		   (return))		; escape inner loop
	     (when (vzerop (vdiff p (first touchpoints))) ; find?
		   (return-from covered (caar touchblocks))) ; yes, done
	     (pop touchpoints))))))	; no, keep looking at touchpoints.
;
;	loweradj  --  see if vertex I is adjacent to another lower vertex.
;
(defun loweradj (verts vertl i
		       &aux (h (aref verts i 2)) l)
  (mapc (function
	 (lambda (x)
		 (when (< (- (verts x 2) h) mtol1)
		       (push x l))))
	(aref vertl i 0))
  l)
;
;	obbelow  --  see if there is an obejct on line from P down to table.
;
(defun obbelow (p
		&aux pl (h 0.0) ph facep facel nf x)
  (setq ph (third p))
  ;	Examine all objects.
  (do ((l (getobs) (rest l)))
      ((endp l) nil)
      (setq facep (getfacep (first l))
	    facel (get (get (first l) 'shape) 'facel)
	    nf (array-dimension facep 0))
      ;		Examine every face of every object.
      (do ((i 0 (1+ i)))
	  ((= 1 nf))
	  (and (setq x (pierce facep
			       facel
			       i   
			       p
			       '(0.0 0.0 -1.0)))
	       (< (- (third x) ph) mtol1)
	       (> (third x) h)
	       (setq pl x 
		     h (third pl))
	       (not (vzerop (vdiff (pvec facep i) '(0.0 0.0 1.0))))
	       (return-from obbelow 'lose))))
  pl)
;
;	suplist  --  list all supports of L members, even indirect ones.
;
;	***NOT SURE ABOUT ASSOC BELOW, LIST NOT OF PAIRS***
;
(defun suplist (l
		&aux sl)
  ;	For all blocks in list L, look up supporting blocks.
  (mapc
   (function 
    (lambda (x)
	    ;	Save new supporters of the block.
	    (mapc 
	     (function 
	      (lambda (y)
		      (or (member y sl)		; ignore if duplicate
			  (assoc y l)		; ignore if member of L
			  (push y sl))))
	     (supporting x))))
   l)
  sl)
;
;	getforce  --  get force at P in direction V needed to cancel LOSS.
;
(defun getforce (p loss v
		   &aux d x)
  (cond ((not (vzerop (third loss)))
	 (return-from getforce (- (vdot (third loss) v))))
	((setq d (vdot (vnorm (setq x (vcross (fourth loss)
					      v)))
		       (vdiff p (second loss))))
	 (return-from getforce (/ (vmag x) d)))))
