;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.30 of 5/19/87
;
(require 'builddefs "builddefs")			; prerequisite
(require 'database "database")
(require 'plandefs "plandefs")
(use-package 'builddefs)
(use-package 'database)
;
;	place  --  place a block at indicated place in goal state.
;
; 	Place is given goal block to put in place
;	must find best matching starting block.
;
;	This is a choice function.
;
(defun place (b sg
		&aux candidates y loc (startcontext context) 
		(startplan plan) (ourplan plan))
  ;	Gripe handlers
  (gripehandlers
   ;
   ;	move-by-place  --  recover from MOVE trouble.
   ;
   ((move-by-place (message returnable)
		   (cond ((eq (car message) 'unstab-rem) ; after moving B?
			  ;	Use DIGUP to extract B cleanly.
			  (setq plan startplan)		; undo anything so far
			  (setq context startcontext)	
			  (setq ourplan 
				(prepgoal message
					  'digup		; what
					  (list (caar candidates)
						(caddr message)
						nil
						(list sg))
					  digup-by-place))
			  (go retry-place))		; try MOVE again.
			 ((pass message returnable)))) ; anything else, pass upward.
    ;	
    ;	digup-by-place  --  recover from DIGUP trouble.
    ;
    (digup-by-place (message returnable)
		    (cond ((eq (car message) 'undigable)
			   (setq y (cons message y))
			   (unless (setq candidates (cdr candidates))
				   (gripe (list no-digable-candidates b y)))
			   (setq ourplan startplan) ; undo any damage
			   (go next-place))	; Try using a different block.
			  ((pass message returnable)))))
   ;
   ;	Body of PLACE
   ;
   ;	Preliminary checks
   (assert (eq context (second (first ourplan))))	; sync check
   ;	Did we do this one already?
   (when (present (list 'in-place '* b))
	 (gripe (list 'already-placed b)))
   ;	Get list of candidate blocks to put where B is in goal state.
   (unless (setq candidates (getmatches b sg))
	   (gripe (list 'unmatchable (list b))))
   (setq candidates (supsort candidates))	; Try best ones first.
   (setq y nil)
   next-place
   (setq plan ourplan)		; reset plan
   (setq context (second (first ourplan))) ; accept context
   (setq loc
	 (unrelate-location (cdar candidates) (in-context sg (getat b))))
   ;	Try the move; B may not be free to move.
   retry-place
   (setq plan ourplan)		; reset plan
   (setq context (second (first ourplan)))	; accept context
   ;	Unless at desired location, move block.
   (setq ourplan
	 (if (equalp (getat (caar candidates)) loc)	; if at dest.
	     ourplan		; just use old plan
	     ;	Attempt move.
	     (goal 'move 
		   (list (caar candidates) (getat (caar candidates)) loc)
		   move-by-place)))
   ;	Success; note B in proper place.
   (setq context (second (first ourplan)))	; accept new context
   (context-add (list 'in-place (caar candidates) b)))
  (verify-at (caar candidates) loc)	; had better be there
  ourplan)				; success, return plan
;
;	getmatches  --  list all blocks in goal state that match
;			B, are movable, and are not in place already.
;
;	Return block and orientation at which match occurs, rel to B.
;
(defun getmatches (b sg
		     &aux l obl gobl x)
  ;	If B is in current state, return immediately.
  (when (member b (setq obl (getobs)))
	(return-from getmatches (list (cons b orgloc))))
  (setq l nil gobl (in-context sg (getobs)))
  ;	Search objects in current state for matches.
  (map nil
       (function
	(lambda (bg)
		(unless
		 (or (not (setq x (match bg b)))	; reject if no match
		     (present (list 'in-place bg '*))	; reject if in place
		     (isimmovable bg)			; reject if immovable
		     (member bg gobl))			; reject if in goal
		 (push (cons bg x) l))))		; keep
       obl)
  l)
;
;	supsort  --  sort list of blocks so that least buried are
;		     considered first.
;
(defun supsort (l)
  (mapcar #'car
	  (lsort (mapcar
		  (function (lambda (y) (cons y (dependon (car y)))))
		  l))))
;
;	lsort  --  sort list of lists by length, shortest first.
;
(defun lsort (l)
	(sort l #'< :key #'length))
;
;	get-rid-of  --  finds a place to put B, then moves it there.
;
;	STATES has list of other states to avoid blocks of.
;	UNTRUDE, if set, forces block to be put where it doesn't intrude.
;
(defun get-rid-of (b &optional (states nil) (untrude nil) (loc nil)
		     &aux hcon (startcontext context) 
		     (startplan plan))
  (gripehandlers
   ;	
   ;	move-by-get-rid-of  --  gripe handler for MOVE trouble.
   ;
   ((move-by-get-rid-of 
     (message returnable)
     (setq context startcontext)	; back out failed move
     (setq plan startplan)
     (cond ((eq (car message) 'unstab-rem) ; unstable after lift?
	    (setq plan (prepgoal message
				 'digup
				 (list b
				       (caddr message)
				       loc
				       states)
				 digup-by-get-rid-of))
	    (setq context (cadar plan))	; accept result
	    ;	Retry, then unwind.
	    (return-from get-rid-of
			 (get-rid-of b states untrude nil)))
	   ;	Other errors, try another place.
	   ((member (car message) '(hit unstab-add))
	    (setq plan startplan)	; back out failed move
	    (setq context startcontext)
	    ;	Retry, then unwind.
	    (return-from get-rid-of
			 (get-rid-of b states untrude loc)))
	   ((pass message returnable)))) ; other problem; pass upwards.
    ;	
    ;	digup-by-get-rid-of  --  gripe handler for DIGUP trouble.
    ;
    (digup-by-get-rid-of (message returnable)
			 (pass message returnable)))	; no recovery here
   ;
   ;	Body of GET-RID-OF.
   ;
   ;	If UNTRUDE testing is requested, create an extra context
   ;	for intrusion test against original state.
   (verify-stable)			; must be stable at entry
   (when untrude
	 (setq hcon (push-context))
	 (in-context hcon (remblock b)))
   ;	Look for space to put block.
   ;	If LOC is set, try space after LOC in FINDSPACE sequence.
   nextloc
   (unless (setq loc (findspace b loc states)) 
	   (gripe (list 'no-space b)))
   ;	Check for intrusion against goal state if requested.
   (when untrude
	 (in-context hcon
		     (addat b loc)
		     (unless (untruding b) 
			     (go nextloc))))
   ;	Try to move the block to the space found.
   (unless (equalp (getat b) loc)		; unless at destination
	   (setq plan (goal 'move 
			    (list b (getat b) loc)	; construct new goal
			    move-by-get-rid-of))
	   (setq context (cadar plan))))	; accept new context.
  plan)						; success, return plan.
;
;	digup  --  prepares block B for moving.
;
;	Is given or finds initial lossage.
;	LOC is intended destination of B. It will be left clear.
;
;	This is a choice function.
;
(defun digup (b &optional (lossage nil) (loc nil) (states nil)
		&aux loser (ghost nil) (d (dependon b)))
  (gripehandlers
   ;
   ;	Gripe handlers
   ;
   ((get-rid-of-by-digup (message returnable)
			 (pass message returnable))	; no recovery here
    (unbuild-by-digup (message returnable)
		      (pass message returnable)))	; no recovery here
   ;
   ;	Body of DIGUP
   ;
   verify
   (verify-stable)			; must be stable at entry
   ;	If initial lossage not given, find it with stability check.
   ;	The question is, what will fall if we remove B?
   (unless lossage
	   (in-context (push-context)
		       (remblock b)		; try without B.
		       (setq lossage (checkstate))	; do stability check
		       ;	If new state is stable, no DIGUP needed.
		       (when (eq 't lossage) (return-from digup plan))
		       (when (eq 'hit (car lossage))
			     (error "Hit ~a while in digup of ~b" 
				    (cdr lossage) b))
		       (assert (eq (car lossage) 'lose))
		       (setq lossage (cdr lossage)))) ; remove 'lose
   ;	If destination given, make sure it doesnt cause a collision.
   (when loc 
	 (setq context (push-context))		; new context
	 (setq ghost (genghost b loc))		; make ghost of B.
	 (when (eq 'hit
		   (car (setq x (gettouch ghost))))
	       (gripe (list 'bad-loc b (cdr x)))))
   ;	Lossage is now of the form ((B1 ...) (B2 ...))
   ;	Convert lossage into form ((B1 . B1LOC) (B2 . B2LOC) ...)
   ;	***Note that if we go through this twice, we will have junk.***
   (setq lossage (mapcar (function (lambda (x) (cons x (getat (car x)))))
			 lossage))
   ;	Examine losing blocks.
   next
   (loop
    (setq loser (pop lossage))		; get first loser
    (format t "Digup ~a: loser ~a~%  Lossage: ~a~%" b loser lossage) ; ***TEMP***
    (cond ((null loser) 			; no lossage, try again
	   (when ghost (remghost ghost))
	   (go verify))
	  ((not (equalp (getat (caar loser)) (cdr loser))) t) 
	  ((member (caar loser) d)	; depends on b, just get rid of.
	   (setq plan (goal 
		       'get-rid-of 
		       (list (caar loser)) 
		       get-rid-of-by-digup))
	   (setq context (cadar plan)))	; accept new context.
	  (t 
	   (setq plan (goal
		       'unbuild
		       (list (getlosers (list (caar loser))) sg)
		       unbuild-by-digup))
	   (setq context (cadar plan)))))))	; on to next block
;
;	genghost  --  create place-holding ghost of B at location LOC.
;
(defun genghost (b loc)
  (let ((newb (duplicate-block
	       b
	       (format nil "ghost-of-~a" (block-name b))
	       t)))	; generate a ghost block just like B
       (addat newb loc)		; place at indicated location
       newb))
;
;	remghost  --  remove ghost block created by GENGHOST.
;
(defun remghost (g &aux item) 
  (assert (block-isghost g))		; must be a ghost.
  (setq item (present (list 'at g '*)))	; look up
  (assert item)				; must find
  (context-delete item))		; delete item
;
;	getlosers  --  find all blocks which fall if members of L
;		       removed from state.
;
(defun getlosers (l)
  (mapcar #'car
	  (cdr (getlosses (mapcar #'list l)))))
