;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.47 of 5/19/87
;
;      Functions for manipulating objects of type block.
;
(require 'builddefs "builddefs")		; prerequisite
(require 'vector "vector")
(require 'database "database")
(provide 'buildutil)
(in-package 'buildutil)
(export '(
	  genblock
	  gentable
	  block-name
	  block-shape
	  block-size
	  block-cg
	  block-edges
	  block-vertices
	  block-findpos
	  block-match
	  block-maxdimension
	  block-isghost 
	  block-isimmovable
	  name-block
	  duplicate-block
	  getobs
	  getat
	  getverts
	  getfacep
	  getfacel
	  getweight
	  getcgrav
	  getedgefaces
	  getvertexedges
	  isimmovable
	  translation-volume
	  ))
(use-package 'builddefs)
(use-package 'vector)
(use-package 'database)
;
;	local variables
;
(defvar tvserial 0)				; generated block serial number
;
;	block  --  the type block
;
;	The internal structure of this type is hidden outside this
;	package.
;	
;	All components of the block are independent of its location
;	and orientation.  Position and orientation are stored in the
;	database.
;
(defstruct (block
	    (:print-function block-print))
  (name1 :read-only)		; printable name
  (name2 :read-only)		; serial number if generated block
  (vertices :read-only)		; vertices relative to CG
  (edges :read-only)		; edges (vertex indices)
  (shape :read-only)		; shape name if any
  (maxdimension :read-only)	; max extension from CG
  (weight :read-only)		; weight (scalar)
  (cg :read-only)		; center of gravity 
  (size :read-only)		; size parameters
  (findpos :read-only)		; findpos function, if any
  (match :read-only)  		; match function, if any
  (isghost)			; T if ghost block
  (isimmovable)			; T if immovable object
				; derived fields, computed when needed
  (edgefaces)			; planes touching an edge
				; An array (edges,2) of face numbers.
  (vertexedges)			; Edges touching a vertex.
				; An array of edge indices, indexed by vertex.
  (faceplanes)			; face planes relative to CG
				; An array of planes, indexed by face number.
  (faceinfo)			; Misc. face edge information
				; A 2D array indexed by face.
				; (face,0): list of vertices of face
				; (face,1): list of adjacent faces
				; (face,2): list of edges of face
  )
;
;	blocknames  --  name to block lookup
;
;	Used by user interface.
;
(defvar blocknames (make-hash-table :test #'equal :size 100))
;
;	block-name  --  get name of block
;
;	For serially numbered blocks, this has to be generated.
;
(defun block-name (blk)
  (if (block-name2 blk)			; print serial if any
      (format nil "~a-~a" (block-name1 blk) (block-name2 blk))
      (format nil "~a" (block-name1 blk)))); name only
;
;	block-print  --  print routine for blocks
;
(defun block-print (blk stream depth)
  (declare (ignore depth))			; name always short
  (format stream "<~a>" (block-name blk)))
;
;	getobs  --  get list of all blocks 
; 
;	Return list of all objects tagged with 'at.
;	Ignores blocks marked with 'ghost-of property.
;
(defun getobs nil
  (mapcan 
   (function
    (lambda (item)
	    (if (and (eq (first item) 'at) ; if AT
		     (not (block-isghost (second item)))) ; and not ghost 
		(list (second item))  ; block ident
		nil)))	; otherwise skip
   (context-lookup '(at * *))))		; over all AT items 
;
;
;	genblock  --  create block given name, shape, and size.
;
;	The number of size parameters may vary with the shape.
;
(defun genblock (name shape &rest sizelist)
  (unless (and (symbolp shape)
	       (get shape 'shape-name))	; is a shape
	  (error "There is no shape named ~A." shape))
  (genblock1 name			; get name
	     shape			; get shape
	     (fillarray (list (length sizelist))	; make size array
			(mapcar (function (lambda (x) (clean (float x))))
				sizelist))))
;
;	genblock1  --  actually generate the object for the block.
;
;	The argument "size" is a vector of parameters which are interpreted
;	by the shape definitions to parameterize the shape as required for
;	this instance.  The shape definitions contain references to "size".
;
(defun genblock1 (name shape size)
  (let* ((verts (generateverts shape size))	; vertices
	 (cg (eval (get shape 'shape-cgrav)))	; center of gravity
	 (maxd (generatemaxd cg verts))		; max dimension
	 (blk
	  (make-block
	   :name1 name
	   :name2 nil
	   :shape shape
	   :size size
	   :cg cg			
	   :weight (clean (* density (eval (get shape 'shape-weight))))
	   :maxdimension maxd
	   :vertices verts			; unrotated vertices
	   :edges (get shape 'shape-edges)	; edges of block
	   :findpos (get shape 'shape-findpos-function) ; find parking pos
	   :match (get shape 'shape-match-function)))) ; matches blocks
	(setf (gethash name blocknames) blk)	; attach block to name
	;	Avoid undersized-block pathological case.
	(when (equal (get name 'shape-weight) 0)	; if weight is zero
	      (error "Block ~A is too small; it has zero weight." name))
	blk))
;
;	gentable  --  generate the table object
;
;	The table is a single face, not a true polyhedron, and we can't apply
;	the general primitives to it.  Some canned information is used
;	to supply the information that is generated for polyhedrons.
;
(defun gentable (name shape)
  (let* ((size nil)
	 (blk (genblock name shape)))
	(setf (block-faceplanes blk)
	      (generatefacep shape size)) ; face planes (scaled)
	(setf (block-faceinfo blk)
	      (get shape 'shape-faceinfo)) ; face lines
	(setf (block-edgefaces blk)
	      (get shape 'shape-edgefaces)) ; planes neighboring edges
	(setf (block-vertexedges blk)
	      (get shape 'shape-vertexedges))
	blk)) ; vertex to edge map
;
;	duplicate-block  --  make a new block just like an old one
;
;	The new block is not accessable by name if GHOST is set.
;
(defun duplicate-block (oldblock name &optional (ghost nil))
  (let ((blk (copy-block oldblock)))
       (setf (block-name1 blk) name)	; new name
       (setf (block-isghost blk) ghost)	; set ghost status
       (unless ghost
	       (setf (gethash name blocknames) blk))	; attach block to name
       blk))
;
;	name-block  --  given name, return block
;
;	Used by user interface routines.
;
(defun name-block (name)
  (if (block-p name) 			; if already in block form
      name
      (gethash name blocknames)))	; look up, NIL if fail.
;
;      isimmovable  --  true if block is immovable.
;
(defun isimmovable (b)
  (or (block-isimmovable b)			; if inherently immovable
      (present (list 'immovable b))))		; or marked immovable, true.
;
;	getat  --  get location of block
;
(defun getat (b)
  (let ((item (present (list 'at b '*))))	; look up B.
       (unless item (error "~A not present - getat" b))
       (check-type (third item) location)	; must be a location
       (third item)))				; return location
;
;      getverts  --  get or create block's verts array
;
;	This is called whenever we need vertex information about
;	the block.  If the information is on file for the current
;	position, we just return it; otherwise we calculate it.
;	
(defun getverts	(b
		 &aux (item (present (list 'at b '*))))
  (unless item (error "~A not present - getverts" b))
  (let ((cacheitem (dget+ item 'shape-vertices)))	; vertices in database?
       (when cacheitem
	     (return-from getverts (second cacheitem)))); yes, just return
  ;	Vertices not in cache, must calculate them.
  (let ((a (convert-vertices (block-vertices b) (third item))))
       (dput+ item a 'shape-vertices)			; save in cache
       a))						; return new vertices
;
;	generateverts  --  generate vertices of block
;
;	The shape definition is used in conjunction with the size information
;	to generate the vertex list for the block.
;	This result is independent of the orientation and position of the
;	block.  Used only at block-generation time.
;
;	The shape-vertices are an array of expressions which generate the
;	coordinates of the vertices.
;
(defun generateverts (shape size)
	(map '(vector vector3) #'eval (get shape 'shape-vertices)))
;
;      getfacep  --  get or create face planes array
;
;	The logic is very similar to that used above for vertices.
;
(defun getfacep	(b)
  (let ((item (present (list 'at b '*))))		; get location of B.
       (unless item
	       (error "~A not present - getfacep" b))
       (let ((a (dget+ item 'shape-faceplanes)))	; look in cache
	    (when a (return-from getfacep (second a))))	; found in cache
       ;	Not in cache, must calculate face planes at current location.
       (let ((a (convert-planes (get-block-faceplanes b) (third item))))
	    (dput+ item a 'shape-faceplanes)			; save in cache
	    a)))					; return new face planes
;
;	get-block-faceplanes  --  get or create face planes array
;				  of underlying block.
;
;	This information can be created when needed from the vertex
;	information.
;
(defun get-block-faceplanes (b)
  (unless (block-faceplanes b)			; generate if necessary
	  (generatefaceinfo b))
  (assert (block-faceplanes b))		; must generate
  (block-faceplanes b))				; return block faceplanes
;
;	getfacel  --  get face edges
;
;	This just looks up the face list.
;
(defun getfacel (b)
  (unless (block-faceinfo b)		; generate if necessary
	  (generatefaceinfo b))
  (assert (block-faceinfo b))		; must generate
  (block-faceinfo b))			; return face line vertex list
;
;      generatefacep  --  create face planes array
;
;	The logic is very similar to that used above for vertices.
;
(defun generatefacep (shape size)
	(map '(vector vector3) #'eval (get shape 'shape-faceplanes)))
;
;      getweight --  Get block's gravitational force vector
;
(defun getweight (b) (v3scale grav (block-weight b)))
;
;      Get block's center of gravity in current context.
;
(defun getcgrav (b) (convert-vertex (block-cg b) (getat b)))
;
;	generatemaxd  --  compute distance of vertex furthest from CG
;
;	Maxd is the radius of the smallest sphere centered at CG which
;	contains all the vertices of the block; this is used for the cheap
;	first test in collision detection.
;
(defun generatemaxd (cg vertices)
  (let
   ((maxdsq 0.0))		; furthest distance, squared
   (map nil 
	(function 
	 (lambda (vertex)
		 ;	Distance squared from CG to vertex.
		 (let* ((distvec (v3diff vertex cg))
			(d (+
			    (sq (vector3-x distvec))
			    (sq (vector3-y distvec))
			    (sq (vector3-z distvec)))))
		       (when (> d maxdsq)	 ; if bigger,
			     (setq maxdsq d)))))	; keep
	vertices)
   (sqrt maxdsq)))	; return sqrt.
;
;
;	getedgefaces  --  get edge to neighboring plane array
;
;	This is used primarily in the graphics system, and is only
;	computed when needed for objects that aren't of built in
;	shape types.
;	Once computed, the value is stored in the block.
;
(defun getedgefaces (b)
  (unless (block-edgefaces b) 		; if not available
	  (generatefaceinfo b))		; compute it
  (assert (block-edgefaces b))
  (block-edgefaces b))			; return result
;
;	getvertexedges  --  get vertex to neighboring edge mappings
;
(defun getvertexedges (b)
  (unless (block-vertexedges b)		; if not available
	  (generatevertexedges b))	; generate it
  (assert (block-vertexedges b))
  (block-vertexedges b))		; return result
;
;	generatevertexedges  --  compute neighbors of each vertex given edges
;
;	The edge list is a sequence of pairs of vertex numbers.
;
;	Returns an array indexed by vertex number of lists of edge numbers
;	of edges containing that vertex.
;
(defun generatevertexedges (b)
  (let* ((nv (array-dimension (block-vertices b) 0))	; number of verts
	 (edges (block-edges b))			; edge array
	 (nl (array-dimension edges 0))			; number of edges
	 (neighbors (make-array (list nv))))		; new neighbors
	(do ((n 0 (1+ n)))				; over all edges
	    ((= n nl))					; until limit
	    (let ((v1 (aref edges n 0))			; get verts of edge
		  (v2 (aref edges n 1)))
		 ;	Sanity checks on edge.
		 (unless (and (< v1 nv) (< v2 nv) 
			      (not (< v1 0)) (not (< v2 0))
			      (not (eql v1 v2)))
			 (error "Edge ~a of block ~a invalid"
				n b))
		 (push n (aref neighbors v2))	; v1 is neighbor of v2
		 (push n (aref neighbors v1))))	; v2 is neighbor of v1
	;	Now validate that the neighbor list has some basic properties.
	;	Every vertex must have at least three neighbors.
	(let ((n 0))				; vertex number
	     (map nil
		  (function
		   (lambda (neighborlist)
			   (when (< (length neighborlist) 3)
				 (error
				  "Too few neighbors for vertex ~a of block ~a" 
				  n b))
			   (incf n)))		; count vertices for msg
		  neighbors))			; mapping over neighbor lists
	(setf (block-vertexedges b) neighbors)	; save as property of the block
	neighbors))				; return neighbor lists
;
;	generatefaceinfo  --  generate the face planes for a block.
;
;	We start with an arbitrary edge.
;	Each edge has two vertices, each of which has N neighboring edges.
;	For each vertex and each pair of neighboring
;	edges, a face plane is calculated.  The plane is oriented so that
;	the CG is on the inside of the plane.
;
;	This new face is associated with the starting vertex and the two
;	edges.  We then extend the face by calling extendface, which
;	tracks out the additional vertices and edges which make up the
;	face.
;
;	We continue processing the vertices, ignoring any edges for which
;	two neighboring faces have already been recorded.
;
;	This process generates not only the face planes, but the face edge
;	lists and the edge face association array.
;
;	We are able to calculate the number of faces from the number of
;	vertices and edges using Euler's formula, 
;
;		VERTICES - EDGES + FACES = 2
;
;	This algorithm is protected against most pathological situations
;	but does not contain a general check for invalid topology.
;	In particular, it does not check that the polyhedron is convex.
;
(defun generatefaceinfo (b)
  (let* ((edges (block-edges b))		; edges
	 (verts (block-vertices b))		; vertices	
	 (vertedges (getvertexedges b))		; vertex-edge association lists
	 (nl (array-dimension edges 0))		; number of edges
	 (nv (array-dimension verts 0))		; number of vertices
	 (nf (- (+ nl 2) nv))			; number of faces, per Euler.
	 (edgefaces (make-array (list nl 2)))	; edge->face association.
	 (faceplanes (make-array (list nf)))	; new face planes
	 (faceinfo (make-array (list nf 3)))	; lines on face
	 (nextf 0)				; no faces yet
	 )
	(labels
	 (
	  ;
	  ;	putedgeface  --  add face to list of faces neighboring edge.
	  ;
	  ;	When we are done, exactly two faces must neighbor each edge.
	  ;
	  (putedgeface (edge face)
		       ;	Associate face with edge.
		       (cond ((null (aref edgefaces edge 0))	; first face
			      (setf (aref edgefaces edge 0) face))
			     ((null (aref edgefaces edge 1))	; second face
			      (setf (aref edgefaces edge 1) face))
			     (t (error
				 "Too many faces for edge ~a of block ~b."
				 edge b))))
	  ;
	  ;	badplane  --  is this plane invalid as a face?
	  ;
	  ;	All the vertices must be on the inside or on the plane.
	  ;
	  (badplane (pln)
		    (unless pln (return-from badplane t)); bad if no plane
		    (map nil
			 (function
			  (lambda (vert)
				  (when (eq (point-plane-test vert pln)
					    'outside)	; if outside, bad.
					(return-from badplane t))))
			 verts)		; try all vertices
		    nil)		; if all OK, return no problem.
	  ;
	  ;	startface  --  start a new face
	  ;	
	  ;	If both edges still haven't been associated with two faces each,
	  ;	and both edges aren't associated with the same face, we start
	  ;	a new face.  
	  ;
	  (startface (v0 edge1 edge2 neighborcnt)
		     ;	Do these two edges define a new face?
		     ;	Or have they been handled already?
		     (when (or (eql edge1 edge2)	; vacuous
			       (aref edgefaces edge1 1) ; used up
			       (aref edgefaces edge2 1)) ; used up
			   (return-from startface))	; just return
		     ;	Check for edges already common to a face.
		     (when (and (aref edgefaces edge1 0)
				(eql (aref edgefaces edge1 0)
				     (aref edgefaces edge2 0)))
			   (return-from startface))
		     
		     ;	We have two edges which are not yet assigned to a face.
		     (let* ((v1 (if (eql v0 (aref edges edge1 0)); get other end
				    (aref edges edge1 1)
				    (aref edges edge1 0)))
			    (v2 (if (eql v0 (aref edges edge2 0)); get other end
				    (aref edges edge2 1)
				    (aref edges edge2 0)))
			    ;	Generate new plane from three points, and
			    ;	orient it so that the CG is on the inside.
			    (newplane (orient-inside-plane
				       (make-plane-from-points
					(aref verts v0)
					(aref verts v1)
					(aref verts v2))
				       (block-cg b))))	 ; new plane
			   ;	If this is a complex vertex, with more
			   ;	than 3 edges meeting, we have to check
			   ;	the plane to insure that it is valid.
			   ;	For a 3-edge vertex, all possible planes
			   ;	generated by combining the edges are valid.
			   (when (> neighborcnt 3)
				(when (badplane newplane)
					(return-from startface)))
			   (unless newplane
				   (error
				    "Block ~a has collinear or coincident points ~a ~a ~a"
				    b
				    (aref verts v0)
				    (aref verts v1)
				    (aref verts v2)))
			   (unless (< nextf nf)
				   (error
				    "Block ~b has bad topology - too many faces"
				    b))
			   (setf (aref faceplanes nextf) newplane); add new face
			   ;	Trace out the new face
			   (extendface nextf newplane v0 edge1 v0 v1)
			   (incf nextf)))	; increment count of faces
	  ;
	  ;	extendface  --  trace out a face given a starting point.
	  ;
	  (extendface
	   (face pln startvert edge lastvert vert)
	   ;	Put edge between lastvert and vert into the face.
	   (putedgeface edge face)		; add edge to face
	   (push vert (aref faceinfo face 0))	; add point to face
	   (when (eql startvert vert) (return-from extendface)) ; closed, done.
	   ;	Examine neighboring edges, checking for one in
	   ;	the current plane.
	   (map nil
		(function 
		 (lambda (nextedge)
			;	Skip if edge already associated with two faces.
			 (unless
			  (aref edgefaces nextedge 1) ; unless used up
			  ;	Get next vertex along this edge.
			  (let ((newvert 
				 (if (eql (aref edges nextedge 0) vert)
				     (aref edges nextedge 1)
				     (aref edges nextedge 0))))
			       ;	Test the new vertex.
			       ; Must be new, and on plane.
			       (when
				(and
				 (not (eql newvert lastvert))
				 (eq
				  (point-plane-test (aref verts newvert) pln)
				  'on))
				;	Vertex passed, extend along this path.
				(return-from extendface
					     (extendface
					      face
					      pln
					      startvert
					      nextedge
					      vert newvert)))))))
		(aref vertedges vert))	; over edges at vert.
	   (error "Unable to close face ~a of block ~a" face b))
	  )
	 ;
	 ;	Body of generatefaceplanes.
	 ;
	 (do ((edge1 0 (1+ edge1)))			; iterate over edges
	     ((= edge1 nl))				; done when complete
	     ;	Examine all edge pairs at each vertex of edge1.
	     (do ((n 0 (1+ n)))				; 2 vertices of edge
		 ((or (= n 2)				; tried both ends
		      (aref edgefaces edge1 1)))	; found both faces
		 ;	Get neighboring edges of vertex, and count them
		 (let* ((neighboredges (aref vertedges (aref edges edge1 n)))
			(neighborcnt (length neighboredges)))
			;	Map over edges that touch this point
		       (map nil
			    (function
			     (lambda (edge2)
				     (startface
				      (aref edges edge1 n)	; vertex
				      edge1 edge2	; edges
				      neighborcnt)))
			    neighboredges))))
	 ;	Insure that proper number of faces was generated.
	 (unless (eql nf nextf)			; must have proper face count
		 (error "Block ~a should have ~a faces but only has ~a"
			b nf nextf))
	 ;	Final check  --  make sure each edge neighbors two faces.
	 ;	Also update face edge lists.
	 (do ((edge 0 (1+ edge)))			; check edges.
	     ((= edge nl))				; all edges
	     (let ((face0 (aref edgefaces edge 0))	; one face
		   (face1 (aref edgefaces edge 1)))	; other face
		  (unless (and face0 face1)		; check completeness
			  (error
			   "Did not find two faces for edge ~a of block ~a"
			   edge b))
		  (push edge (aref faceinfo face0 2))	; update edge list
		  (push face1 (aref faceinfo face0 1))	; update neighbor faces
		  (push edge (aref faceinfo face1 2))	; update edge list
		  (push face0 (aref faceinfo face1 1)))); update neighbor faces
	 ;	Update block B with new derived information.
	 (setf (block-faceplanes b) faceplanes)		; set face planes.
	 (setf (block-faceinfo b) faceinfo)		; set face edges
	 (setf (block-edgefaces b) edgefaces)		; set edge faces
	 )))
;
;	orient-inside-plane  --  invert plane if necessary to make point 
;				 be inside it.
;
(defun orient-inside-plane (pln pnt)
  (unless pln (return-from orient-inside-plane nil)) ; nil if nil plane.
  (case (point-plane-test pnt pln)	; is point inside?
	(inside pln)			; inside, OK
	(outside (reverse-plane pln))	; outside, invert plane
	(on	nil)))			; otherwise no plane
		
;
;	translation-volume  --  compute volume occupied by path of moving block
;
;	Each block is a convex polyhedron.  Thus, the volume occupied by
;	a translated block is also a convex polyhedron.  The polyhedron of
;	translation is constructed as follows:
;
;	1.  Examine all faces of the polyhedron being moved.  If the face
;	    faces away from the direction of the move, that face will be
;	    a face at the "from" position.  If it faces toward the direction
;	    of the move, it will be a face at the "to" position.  If it is
;	    parallel to the move, it will be elongated.  Create an array
;	    indexed by face number which contains FROM, TO, or 'elongated
;	    for each face.  
;
;	    For each vertex for each face, decide whether that vertex is
;	    to be at the FROM or TO position, based on the rules:
;
;	    - All vertices of FROM faces are FROM vertices
;	    - All vertices of TO faces are TO vertices.
;
;	    This should result in all vertices being tagged as TO or FROM.
;	    Some vertices will be tagged as both; this indicates that a
;	    new edge between such vertices will have to be inserted.
;	    For any vertex tagged as both FROM and TO, a new vertex entry
;	    must be generated for the new vertices array.
;
;	    The output of this process consists of two arrays; the new
;	    array of vertex coordinates, and the mapping from old vertex
;	    number and from/to status to the new vertex number.
;
;	    At this point, the generation of the new figure is essentially
;	    complete; it remains only to generate all the derived data
;	    structure which is the array of edges of the figure.
;
;	2.  New edges are constructed as follows:
;
;	    - If, for an old edge, both vertices are tagged as FROM,
;	      generate a new edge.
;	    - Similarly for TO.  
;
;	    - For each vertex tagged as both FROM and TO, generate a
;	      new edge.
;
;	    It is thus possible for two connected vertices to generate one,
;	    two, or four new edges.
;
;	3.  Generate a block structure and set all the relevant properties.
;
;	4.  Add the new block at the midpoint of the vector FROM..TO.
;
;	5.  Return the new block structure.
;	    
;
;	Translation-volume constructs a block to represent the translation
;	volume; the new block is returned, but not inserted.
;
;	B is the block to use in constructing the translation volume.  It
;	need not be present in any context.
;	TRANSVEC is the move of the block in the block's own coordinate system.
;
;
(defun translation-volume (b transvec)
  (let* 
   ((edges (block-edges b))		; Vertex indices for edges.
    (nl (array-dimension edges 0))	; edge list
    (edgefaces (getedgefaces b))	; faces neighboring edges
    (facep (get-block-faceplanes b))	; get face planes of block
    (nf (array-dimension facep 0))	; get number of faces
    (pdirs (make-array (list nf)))	; assignment of faces to from/to
    (verts (block-vertices b))		; vertices of B
    (nv (array-dimension verts 0))	; get number of vertices
    (halftransvec (v3scale transvec 0.5)) ; move from FROM to new CG.
    (isfromvert (make-array (list (* 2 nv)))) ; FROM vertex map
    (istovert (make-array (list (* 2 nv)))) ; TO vertex map
    (newverts (make-array (list (* 2 nv)))) ; new vertices, oversize
    (newedges (make-array (list (* 3 nl) 2))) ; new edges, oversize
    (newvertn 0)			; count of vertices in new block
    (newedgen 0)			; count of edges in new block
    )
   ;
   ;	Local functions for translation-volume function.
   ;
   (flet 
    (
     ;
     ;	newedge  --  add new edge to array if both ends non-null
     ;
     (newedge (va vb)
	      (when (and va vb)			; if valid edge
		    (when (< vb va)		; order canonically
			  (let ((temp va))	; so dup check will work
			       (setq va vb)
			       (setq vb temp)))
		    (do ((i 0 (1+ i)))		; check all vertices
			((= i newedgen))	; up to number used
			(when (and (eql (aref newedges i 0) va)
				   (eql (aref newedges i 1) vb))
			      (return-from newedge)))	; ignore duplicates
		    (assert (< newedgen (array-dimension newedges 0))) ; fit?
		    (setf (aref newedges newedgen 0) va) ; first endpoint
		    (setf (aref newedges newedgen 1) vb) ; second endpoint
		    (incf newedgen)))	; another edge
	)
    ;	Divide faces into "from" and "to" faces.
    (do ((n 0 (1+ n)))			; iterate over faces
	((= n nf))
	(let ((pdir (planedir facep n transvec))) ; which direction
	     (setf (aref pdirs n)		; set direction
		   (cond ((> pdir tol1)	; toward destination
			  1)		; record as "to" face
			 ((< pdir mtol1); away from destination
			  -1)		; record as "from" face
			 (t 0)))))	; record as parallel face
    ;	Examine edges and tag vertices as FROM or TO
    (do ((n 0 (1+ n)))			; iterate over edges
	((= n nl))
	(let ((d1 (aref pdirs (aref edgefaces n 0))) ; dir for 1st face 
	      (d2 (aref pdirs (aref edgefaces n 1)))) ; dir for 2nd face
	     (when (or (eql d1 1) (eql d2 1)) ; TO face
		   (setf (aref istovert (aref edges n 0)) t)
		   (setf (aref istovert (aref edges n 1)) t))
	     (when (or (eql d1 -1) (eql d2 -1)) ; FROM face
		   (setf (aref isfromvert (aref edges n 0)) t)
		   (setf (aref isfromvert (aref edges n 1)) t))))
    ;	Adjust vertices per assignment as FROM, TO, or both.
    (do ((n 0 (1+ n)))			; iterate over old vertices
	((= n nv))
	(assert (or (aref isfromvert n) (aref istovert n))) ; must be used
	(when (aref isfromvert n)		; if FROM vertex.
	      (setf (aref newverts newvertn)
		    (v3diff (aref verts n) halftransvec))
	      (setf (aref isfromvert n) newvertn) ; map old to new
	      (incf newvertn))		; advance fill
	(when (aref istovert n)		; if TO vertex
	      (setf (aref newverts newvertn)
		    (v3plus (aref verts n) halftransvec))
	      (setf (aref istovert n) newvertn) ; map old to new
	      (incf newvertn)))		; advance fill
    ;	Examine old edges, generate new ones.
    (do ((n 0 (1+ n)))			; iterate over edges
	((= n nl))
	(let ((v1 (aref edges n 0))		; first vertex
	      (v2 (aref edges n 1))		; second vertex
	      (oldnewedgen newedgen))		; for edge count crosscheck
	     ;	There are six possible edges, but only ones
	     ;	where the vertex mappings are both non-nil 
	     ;	are actually included.  Further, the stretched
	     ;	forms only apply if the translated form isn't included.
	     ;	Thus, no more than four new edges can be generated per 
	     ;	old edge.
	     ;
	     ;  Old edges that are still present.
	     (newedge (aref isfromvert v1) (aref isfromvert v2))
	     (newedge (aref istovert v1) (aref istovert v2))
	     ;	New edges generated by translation of vertices.
	     (newedge (aref isfromvert v1) (aref istovert v1))
	     (newedge (aref isfromvert v2) (aref istovert v2))
	     ;	New edges generated by stretching of old edges.
	     (unless (or (aref istovert v1) (aref isfromvert v2))
		     (newedge (aref isfromvert v1) (aref istovert v2)))
	     (unless (or (aref isfromvert v1) (aref istovert v2))
		     (newedge (aref istovert v1) (aref isfromvert v2)))
	     (assert (not (> newedgen (+ oldnewedgen 4)))) ; no more than 4 new
))
    ;	Edges and vertices have been computed.
    ;	Adjust arrays to hold just the number actually present.
    (setq newverts (shrink-array newverts (list newvertn)))
    (setq newedges (shrink-array newedges (list newedgen 2)))
    ;	Construct the new block.
    (let* ((newmaxd (generatemaxd (block-cg b) newverts)) ; max dist from CG
	   (newblk
	    (make-block
	     :name1 "translation-volume"	; dummy name
	     :name2 (incf tvserial)		; serial number for dummy
	     :cg (block-cg b)			; same as old CG
	     :vertices newverts			; new vertices.
	     :maxdimension newmaxd		; max dist from CG
	     :edges newedges			; edges
	     :faceplanes nil			; generate later if needed
	     :faceinfo nil			; generate later if needed
	     :edgefaces nil			; generate later if needed
	     :shape nil				; block not derived from shape
	     :size nil				; block not derived from shape
	     :weight nil			; not meaningful
	     :findpos nil			; not placeable
	     :match nil				; not matchable
	     )))
	  newblk))))				; finally return newblk
;
;	planedir  --  which direction does plane face, relative to vector?
;
;	<0 indicates vector points toward inside of plane.
;	>0 indicates vector points toward outside of plane.
;	=0 indicates vector is parallel to plane.
;
;	This is a dot product of the vector and the normal to the plane.
;
(defun planedir (planes n vec)
  (v3dot (aref planes n) vec))		; just a dot product
;
;	shrink-array  --  change size of array by recopying
;
;
(defun shrink-array (contents dimensions)
  (case (length dimensions)
	(1 (make-array dimensions :initial-contents contents))	; 1D array.
	(2 (let ((a (make-array dimensions)))			; 2D array.
		(do ((i 0 (1+ i)))		; must copy explicitly.
		    ((= i (first dimensions)))
		    (do ((j 0 (1+ j)))
			((= j (second dimensions)))
			(setf (aref a i j) (aref contents i j))))
		a))
	(otherwise (error "shrink-array is unimplemented for ~a-D arrays."
			  (length dimensions)))))
