;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 1.15 of 5/20/87
;
(require 'builddefs "builddefs")
(use-package 'builddefs)
(require 'database "database")
(use-package 'database)
(require 'vector "vector")
(use-package 'vector)
;
;
;	Findspace  --  find place to put block when unwanted.
;
;	TEMPORARY - supports only brick and wedge.
;
;
;	findpos   --  finds a stable orientation for block
;		      at correct height from table.
;
;	Uses pre-packaged formula for each shape.  
;	Will not support new shapes.
;
(defun findpos (b &aux fn)
  (setq fn (block-findpos b))		; get FINDPOS function for shape
  (unless (functionp fn)
	  (error "No FINDPOS function for shape"))
  (funcall fn (block-size b)))		; try FINDPOS function
;
;	fpos   --  find-stable-position function for bricks and wedges.
;
;	Stand block on minimal base, consistent with acceptable stability.
;
(defun fpos (size)
  (cond ((< (aref size 0) (* woblim (aref size 2)))
	 (atchange orgloc
		   0.0 
		   0.0 
		   (+ zmin (* 0.5 (aref size 0)))
		   0.0
		   0.25
		   0.0))
	((< (aref size 1) (* woblim (aref size 2)))
	 (atchange orgloc
		   0.0
		   0.0
		   (* 0.5 (aref size 1))
		   0.25
		   0.0
		   0.0))
	(t 
	 (atchange orgloc
		   0.0
		   0.0
		   (* 0.5 (aref size 2))
		   0.0
		   0.0
		   0.0))))
;	***TEMP*** dumb version of FPOS, never rotates block.
(defun fpos (size)
	 (atchange orgloc
		   0.0
		   0.0
		   (* 0.5 (aref size 2))
		   0.0
		   0.0
		   0.0))
;
;	findspace  --  find place to put block.
;
;	Findspace gets orientation from findpos, coarsely scans table for
;	good X,Y position.  
;	To get another value, pass back last output as 2nd arg.
;	3rd arg is list of other states whose blocks must be avoided.
;
(defun findspace (b &optional (oldloc nil) (states nil)
		    &aux pos verts gx gy lx ly obs new
		    (x 0.0) (y 0.0) (prevx 0.0) (prevy 0.0))
  (setq obs (fsobs (cons context states)))
  (cond (oldloc (setq pos oldloc)	; if old location to start search
		(setq x (vector3-x pos)); get old X and Y.
		(setq y (vector3-y pos))
		(setf (vector3-x pos) 0.0)	; reset POS to origin.
		(setf (vector3-y pos) 0.0))
	(t (setq pos (findpos b))))
  ;	Make a working copy of the vertex list of B, and locate it at POS.
  (setq verts (convert-vertices (block-vertices b) pos))
  ;	Find X,Y bounds of object.
  (setq gx (setq lx (vector3-x (aref verts 0)))); initialize with 1st vertex.
  (setq gy (setq ly (vector3-y (aref verts 0))))
  (map nil 
       (function
	(lambda (vert)				; examine each vertex
		(setq lx (min lx (vector3-x vert))) ; get min X of vertices
		(setq ly (min ly (vector3-y vert))) ; get min Y of vertices
		(setq gx (max gx (vector3-x vert))) ; get max X of vertices
		(setq gy (max gy (vector3-y vert))))) ; get max Y of vertices
       verts)						
  ;	Did caller specify a starting position for the search?
  (unless (and oldloc (not (and (= x 0.0) (= y 0.0))))
	  (setq x (- xmax gx)) 		; no, use right rear of table.
	  (setq y (- ymax gy))
	  (setq new t))			; skip first time.
  ;	Look for space, trying goodloc at regular intervals.
  (loop
   (update verts (- x prevx) (- y prevy))	; relative update of vertices
   (setq prevx x prevy y)			; for next update.
   ;	Is current position at X,Y clear?
   (when new
	 (when
	  (goodloc b 
		   verts
		   (make-vector3 x y (vector3-z pos))
		   obs
		   (v3plus (make-vector3 x y 0.0)
			   (convert-vertex (block-cg b) pos)))
	  (setf (vector3-x pos) x)	; found space
	  (setf (vector3-y pos) y)	; found space
	  (return-from findspace pos)))	; success, found space
   (setq new t)				; no longer first time.
   ;	Advance search position.
   (setq x (- x findstep))		; advance X
   (cond ((> (- x lx) xmin))		; within X limit, OK
	 ((and (setq x (- xmax gx))	; X at lim, reset, advance Y.
	       (> (+ ly (setq y (+ y (- findstep))))
		  ymin)))
	 (t (return-from findspace nil)))	; or give up.
   ))
;
;	fsobs  --  get list of objects in all listed states.
;
;	*** SHOULD ELIMINATE DUPLICATE ITEMS***
;
;	Items returned are of the form (B AT VERTS CG).
;
(defun fsobs (states &aux l b)
  ;	Examine all listed contexts.
  (mapc
   (function 
    (lambda (s)
	    ;	Examine all blocks in context.
	    (in-context s
			(mapc 
			 (function 
			  (lambda (item)
				  (setq b (second item))	; which block
				  (unless (eq b table)		; special case
					  (push
					   (list b		; which block
						 (third item)	; where
						 (getverts b)	; vertices
						 (getcgrav b))	; C of G
					   l))))
			 (context-lookup '(at * *))))))
   states)
  l)
;
;	update  --  update X and Y of VERTS array.
;
;	This moves our "test block" through space efficiently during 
;	FINDSPACE searches.
;
(defun update (v dx dy)
  (map nil
       (function
	(lambda (pnt)
		(incf (vector3-x pnt) dx) ; adjust vertices by dx,dy.
		(incf (vector3-y pnt) dy)))
       v))		; adjust all vertices
;
;	goodloc  --  is this a good location to put an object?
;
;	Checks location for other colliding objects, using only the
;	SEPARATE function of the touch test.
;
(defun goodloc (b verts loc obs cg 
		  &aux (maxdim (block-maxdimension b)))
  (do ((ob (pop obs) (pop obs)))		; iterate over objects
      ((null ob) t)				; no collision if end reached.
      (cond 
       ;	Anything within collision range?
       ((> (point-point-distance-squared loc (second ob))
	   (sq (+ maxdim
		  (block-maxdimension  (first ob))	; max dim of block
		  tol1))))
       ;	Must try expensive collision test.
       ((separate verts				; vertices of B.
		  (third ob)			; vertices of ob.
		  cg				; CG of B.
		  (fourth ob)))			; CG of ob.
       ((return-from goodloc nil)))))		; fails, collision
