;
;	BUILD  --  a blocks world
;
;	Originally by Scott Fahlman, as part of a Master's thesis
;	at MIT in 1973.	Described in AI	Journal, Vol 5,	pp 1ff,
;	in 1974.
;
;	Brought back to life in 1987 by John Nagle.
;	Keyboarded from Fahlman's Conniver source, converted to Common
;	Lisp, and extended.
;
;						Version 3.26 of 3/20/87
;
;      Stability testing package.
;
(require 'builddefs "builddefs")
(require 'buildutil "buildutil")
(require 'vector "vector")
(require 'collision "build5")
(require 'database  "database")
(provide 'stability)   			; provides this resource
(in-package 'stability)			; this is package STABILITY
;
;	Exported functions
;
(export
  '(checkstate
    checkstab
    deplist
    getsupsof
    dependon
    supporting
    delete-all-stability-relationships
    delete-stability-relationships))
;
;	Imported functions
;
(use-package '(builddefs buildutil vector collision database))
;
;	Local special variables
;
(defvar axis)				; Moment axis (?)
(defvar forsw)				; T if zeroing forces, NIL if moments
(defvar netf)				; Current net force on block
(defvar netm)				; Current net moment about POINT
(defvar point)				; Point for moment computation
(defvar touchers)			; Touching blocks
(defparameter internal-checking t "Internal checking enabled if T") 
;
;	Updates to global special variables
;	(Types used here were not available where global specials were defined.)
;
(setq grav (make-vector3 0.0 0.0 -1.0))	; gravity is down
;
;      checksup	 --  test status of single block		 
;
;      Test status of single block, in terms of	touching objects and
;      known forces acting on it.  Return type of lossage or forces this
;      block exerts on others.
;
;	The result is similar but not identical in form to that from
;	GETTOUCHERS.  Result is
;
;		(win  (<touchers> <pushers>))
;	or	(lose (<touchers> <pushers>) (<point> <netf> <netm>))
;	
;	where <touchers> and <pushers> are as defined in GETTOUCHERS,
;	except that now the <force output> entries may be nonzero.
;
;	A "win" is defined as a situation where the net force and moment
;	acting on the current block have been reduced to zero by the
;	application of forces to other blocks at their touchpoints.
;	This information is used in CHECKSTAB.  
;
(defun checksup	(b
		 &aux point netf netm touchers forsw axis tries didfix)
  (setq point (getcgrav b))		; start at center of gravity of block
  (setq touchers (gettouchers b point))	; calc forces/moments relative to that
  (when (> debug 2) (dumptouchers b touchers)) ; dump toucher info
  ;       B should not be colliding, just touching.	   
  (when (eq (first touchers) 'hit)
	(error "~A colliding ~A - checksup" b touchers))
  ;	Compute initial net force and moment.
  (setq netf (v3plus (getweight b) (first touchers)))
  (setq netm (second touchers))
  (when (> debug 2)			; debug output
	(format t "  Checksup ~a: applied ~a gravity ~a~%"  
		b (first touchers) (getweight b)))
  (setq touchers (cddr touchers))
  (when (or (first touchers) (second touchers))	; if anything to work on
	(setq tries 0)			; anti-infinite-loop counter
	;	Iterative solution loop; try to get both force and moment
	;	to go to zero.  This may take some time to converge in
	;	pathological cases.  Fahlman claims it will always converge.
	;	I (Nagle) have installed a backup counter.
	(loop  
	 (setq didfix nil)		; no success this round
	 (when (> debug 3)		; debug output
	       (format t "   Checksup ~a: force ~a moment ~a at ~a~%" 
		       b netf netm point))
	 (chk-netf-netm b touchers)	; ***TEMP*** debug trap
	 (when (and (v3zerop netf) (v3zerop netm)) (return)) ; success?
	 ;	Force loop; try to make force go away.
	 (loop
	  (when (v3zerop netf) (return))	; done with force loop
	  (setq forsw t)		; force mode
	  (unless (fixforce) (return))	; if fail, done
	  (setq didfix t))		; fixed something, note.
	 ;	Moment loop; try to make moment go away.
	 (loop
	  (when (v3zerop netm) (return))	; done with moment loop
	  (setq forsw nil)		; moment mode
	  (unless (fixmoment) (return))	; if fail, done
	  (setq didfix t))		; fixed something, note.
	 (unless didfix (return))	; quit if did nothing
	 (setq tries (1+ tries))	; tally 
	 (when (> tries 10)		; if stuck
	       (when (> debug 0)	; this should not happen
		     (format t "Checkstab: gave up on block ~a~%" b))
	       (return))))		; so give up
  ;	Done.  Construct and return result, win or lose.
  (if (and (v3zerop netf) (v3zerop netm))	; if force & moment are 0
      (list 'win touchers)		; we won
      (list 'lose touchers (list point netf netm))))	; we lost
;	
;      fixforce
;
;      Try to find touchpoint that will	accept some of the unbalanced
;      force.  Prefer to remove	applied	force, then apply new normal
;      force.  Finally apply friction.
;
(defun fixforce (&aux l ll pnorm accepted newmag best)
  ;	Try all 8 kinds of searches.
  (do ((desp 0 (1+ desp)))
      ((or best (> desp 7)) nil)
      ;	Try next kind of search.
      (setq l (if (member desp '(0 1 6 7)) 
		  (second touchers)
		  (first touchers)))
      (when l
	    ;	Try next plane.
	    (map nil 
		 (function 
		  (lambda 
		   (planeitem)
		   (setq ll (cddr planeitem))
		   (setq pnorm (second planeitem))
		   ;	Try points on next plane.
		   (map nil 
			(function 
			 (lambda 
			  (forceitem)
			  ;	Try touchpoint.	
			  (setq accepted 
				(accept desp
					pnorm
					netf
					(third forceitem)
					(second forceitem)))
			  ;	If force accepted, is it best so far?
			  (when accepted
				(setq newmag (v3mag (v3diff netf accepted)))
				(when (> debug 4)
				      (format t "    Fixforce: mag ~a at ~a~%"
					      newmag (first forceitem)))
				;	If new smallest force, save.
				(when (or (null best) 
					  (> (first best) newmag))
				      (setq best 
					    (list newmag accepted forceitem))))))
			ll)))
		 l)))
  (unless best (return-from fixforce nil))		; fails if no find
  ;	Search complete; apply the force.
  (putforce (second best) (third best))
  t)				; success
;
;	putforce  --  put force somewhere else
;
(defun putforce (accepted forcetofix)
  (movemomentpoint (first forcetofix))	; move to new point
  (moveforce accepted forcetofix)	; apply force there
  (setq axis nil)		; unclear what AXIS does.
  forcetofix)			; return forcetofix so trace will show it.
;
;      fixmoment -- try	to find	touchpoint to accept some of the
;		    unbalanced moment.
;
;      Shift point if moment all gone at current point.
;
(defun fixmoment (&aux l ll pnorm accepted newmag moment best d)
  ;	Try all 8 kinds of searches.
  (do ((desp 0 (1+ desp)))
      ((or best (> desp 7)) nil)
      ;	Try next kind of search.
      (setq l (if (member desp '(0 1 6 7)) 
		  (second touchers)
		  (first touchers)))
      (when l
	    ;	Try next plane.
	    (map nil 
		 (function 
		  (lambda 
		   (planeitem)
		   (setq ll (cddr planeitem))
		   (setq pnorm (second planeitem))
		   ;	Try points on next plane.
		   (map nil 
			(function 
			 (lambda 
			  (forceitem)
			  ;	Try touchpoint.	
			  (setq accepted 
				(accept desp
					pnorm
					(list netm	
					      (setq d 
						    (v3diff (first forceitem)
							   point))
					      axis)
					(third forceitem)
					(second forceitem)))
			  ;	If force accepted, is it best so far?
			  (when accepted
				(setq newmag 
				      (v3mag (setq moment 
						   (v3diff netm 
							  (v3cross d 
								  accepted)))))
				(when (> debug 4)
				      (format t "    Fixmoment: mag ~a at ~a~%"
					      newmag (first forceitem)))
				;	If new smallest moment, save.
				(when (or (null best) 
					  (> (first best) newmag))
				      (setq best 
					    (list newmag moment accepted forceitem))))))
			ll)))
		 l)))
  (unless best (return-from fixmoment nil))		; fails if no find
  ;	Search complete; apply the moment.
  (putmoment (first best) (second best) (third best) (fourth best))
  t)				; success
;
;	putmoment  --  apply change to moment
;
(defun putmoment (newmag moment accepted forceitem &aux oldpoint)
  (setq oldpoint point)			; save old point
  (putforce accepted forceitem)		; apply the force
  ;	If magnitude of moment is zero, use new working point.
  (unless (aeq 0.0 newmag)
	  (movemomentpoint oldpoint)	; move back to old point.
	  (assert (v3zerop (v3diff netm moment))) ; answer must match old code
	  (setq axis (v3norm (v3diff point (first forceitem))))
	  (and axis (aeq 0.0 (v3dot axis netm)) (setq axis nil)))
  forceitem)			; return updated item for trace
;
;	moveforce  --  move force to another block
;
;	Conservation of forces applies; the NETF of the current block and
;	the force at the selected touchpoint of another block will be
;	updated such that their sum is unchanged.
;
;	This operation is only permissible when POINT is the point at
;	which NETM is meaningful; otherwise changing NETF would change the
;	moment NETM also.
;
(defun moveforce (force forceitem)
  (assert (v3zerop (v3diff point (first forceitem))))
  (setf (third forceitem)			; update force at touchpoint
	(v3plus force (third forceitem)))
  (setq netf (v3diff netf force))) ; update local net force.
;
;	movemomentpoint  --  move the point about which moments are computed.
;
;	This simply changes the representation of the moment
;	acting on the block; it does not represent any real change in state.
;
(defun movemomentpoint (newpoint &aux movevector)
  (setq movevector (v3diff point newpoint))	; amount of move
  (when (not (v3zerop movevector))		; if not null operation
	(setq netm (v3plus netm (v3cross movevector netf))) ; new mom.
	(setq point newpoint)))			; finally move point
  
;
;      accept  --  find	out how	much of	force F	can be accepted	by a
;		   point with normal N.	 Old applied force OF and
;		   external force P are	considered.  DESP tells	what
;		   to try.
;
(defun accept (desp n f	of p)
  (cond ((or (= desp 0) (= desp 2)) (frem n f of p))
	((or (= desp 1) (= desp 3)) (nrem n f of p))
	((or (= desp 4) (= desp 6)) (nadd n f of p))
	((or (= desp 5) (= desp 7)) (fadd n f of p))))
;
;      frem  --	 reduce	or eliminate old frictional force, if this   
;		 reduces new force.
;
(defun frem (n f of p)
  (cond ((v3zerop of) nil)
	((v3zerop (setq of (v3diff of (v3scale n (v3dot of n)))))
	 nil)
	((xrem n f of p))))
;
;      nrem  --	 reduce	or eliminate old normal	force.
;
(defun nrem (n f of p)
  (prog (mnof nof ans fof mfof afof)
	(and (v3zerop of) (return nil))
	(and (< (setq mnof (v3dot of n)) tol1) (return nil))
	(or (setq ans (xrem n f (setq nof (v3scale n mnof)) p))
	    (return nil))
	(and (> (setq mfof (v3mag (setq fof (v3diff of nof))))
		(setq afof (* mu (v3mag (v3plus nof ans)))))
	     (return (v3plus ans (v3scale fof (/ (- afof mfof)
						       mfof)))))
	(return ans)))
;
;      xrem  --	 does work of frem and nrem
;
(defun xrem (n f of p)
  (declare (ignore n) (ignore p))		; present only for uniformity
  (prog	(normof	magof ofpart x)
	(setq magof (v3mag of) 
	      normof (v3scale of (/ 1.0 magof)))
	(cond (forsw (setq ofpart (v3dot f normof)))
	      (t (when (v3zerop 
			(setq x 
			      (v3cross (cond ((third f)
					     (v3cross (v3cross (third f)
							     (second f))
						     (third f)))
					    ((second f)))
				      normof)))
		       (return nil))
		 (setq ofpart (/ (v3dot (first f) x) (v3dot x x)))))
	(when (> ofpart mtol1) (return nil))
	(when (> (- ofpart) magof) (return (v3negate of)))
	(return (v3scale normof ofpart))))
;
;      nadd  --	 accept	positive (pushing) normal component of new force.
;
;	OF and P are unused in the original.
;
(defun nadd (n f of p)
  (declare (ignore of) (ignore p))		; present for uniformity only
  (prog (x npart)
	(cond ((null forsw)
	       (and (v3zerop (setq x (v3cross (cond ((third f)
						   (v3cross (v3cross (third f)
								   (second f))
							   (third f)))
						  ((second f)))
					    n)))
		    (return nil))
	       (setq npart (/ (v3dot (first f) x) (v3dot x x))))
	      ((setq npart (v3dot f n))))
	(and (> npart tol1) (return (v3scale n npart)))))
;
;      fadd  --	 accept	frictional part	of new force
;
;      Must not	be greater than	MU times normal	forces.
;      Must not	cancel frictional force	from other block.
;
(defun fadd (n f of p)
  (prog (mnf ff mnof fof mnp fp x y z)
	(cond ((null forsw)
	       (and (v3zerop (setq x 
				  (v3cross (first f)
					  (cond ((third f)
						 (v3cross (v3cross (third f)
								 (second f))
							 (third f)))
						((second f))))))
		    (return nil))
	       (setq f (v3scale x (/ (v3dot (first f) (car f)) (v3dot x x))))))
	
	;      Separate	frictional and normal components.
	(setq mnf (v3dot f n) ff (v3diff f (v3scale n mnf)))
	
	;      If frictional part insignificant, punt.
	(or (> (/ (v3mag ff) (v3mag f)) tol1) (return nil))
	
	;      More separating.
	(setq mnof (v3dot of n))
	(setq mnp (v3dot p n))
	(setq fof (v3diff of (v3scale n mnof)))
	(setq fp (v3diff p (v3scale n mnp)))
	
	;      Do not use normal force from other block.
	(or (> mnof tol1) (return nil))
	
	;      Flush part that cancels frictional force	from other block.
	(and (> (setq x (v3dot ff fp)) 0.0)
	     (setq ff (v3diff ff (v3scale fp (/  x (v3dot fp fp))))))
	
	;      Flush part that is too big for the normal force.
	(and (> (setq x (v3mag (setq y (v3plus ff (v3diff fof fp)))))
		(setq z (* mu (cond (forsw (- mnof mnp)) (mnof)))))
	     (setq ff (v3diff (v3scale y (/ z x)) (v3diff fof fp))))
	
	;      If remainder insignificant, punt.
	(or (> (/ (v3mag ff) (v3mag f)) tol1) (return nil))
	
	;      Else, accept it.
	(return ff)))
;
;      gettouchers  --	get list structure with	all info on touching and
;			pushing	objects.
;
;      Also compute the	moment and force at the	center of gravity.
;
;	For each block touching B, list items are created for each
;	touchpoint.  Each list item is of the form
;
;		(<force vector> <force input> <force output>)
;
;	where <force input> is derived from relevant SUP-BY relations.
;	The <force output> is always (0 0 0), but will be updated later
;	in CHECKSUP via calls to FIXFORCE and FIXMOMENT.
;
;	The force list items for each block become components of a list
;	of the form (<blk> (<force item> <force item> ...)).  Finally,
;	these per-block items are listed in two lists; "touching" and
;	"pushing".  Blocks in the "pushing" list have at least one nonzero
;	force input.
;
(defun gettouchers (b point 
		      &aux netf netm l lp lt sw t1 p)
  (setq netf (make-vector3 0.0 0.0 0.0)); net force acting on CG
  (setq netm (make-vector3 0.0 0.0 0.0)); net moment acting on CG
  (setq lp nil)				; pushing blocks
  (setq lt nil)				; touching blocks
  (setq l (gettouch b))			; touching info
  (and (eq (first l) 'hit) (return-from gettouchers l))      ; collision, punt.
  ;	Examine all touching blocks for SUP-BY relations.
  ;	Note that we want both directions of the SUP-BY relationship.
  ;	But only one direction affects toucher/pusher status.
  (mapc (function
	 (lambda (x)
		 (setq sw nil)		; assume not supporting
		 (setq p (reverse-forces (get-support (first x) b)))
		 ;	We only consider this a pusher if B is 
		 ;	pushed by the block under consideration,
		 ;	and not the other way round.
		 (cond (p
			(setq sw t))	; supported by b.
		       (t
			(setq p (get-support b (first x))))) ; get forces
		 ;	For one touching block, construct list of
		 ;	touchpoints and forces.
		 (setq t1
		       (cons (first x) 		; block
			     (cons (second x)
				   (mapcar 
				    (function
				     ;	Iterate over touchpoints.
				     (lambda (y)
					     (list y
						   ; Check for incoming force.
						   (touchpoint-force y p)
						   (make-vector3 0.0 0.0 0.0))))
				    (cddr x)))))
		 (if sw
		     (push t1 lp)		; add to pushers
		     (push t1 lt))))		; add to touchers
	l)
  (list netf netm lt lp))
;
;	touchpoint-force  --  attach old forces to proper touchpoints.
;
(defun touchpoint-force (tp inforces
			    &aux inforce)
  ;	Search list of applied forces.  Elements are of form (tp force).
  (loop
   (unless inforces (return-from touchpoint-force 
				 (make-vector3 0.0 0.0 0.0)))
   (setq inforce (pop inforces))		; get next one
   (when (equalp (first inforce) tp)   
	 ;	Update applied force and moment.
	 (setq netf (v3plus netf (second inforce)))
	 (setq netm (v3plus netm (v3cross (v3diff tp point) (second inforce))))
	 (return-from touchpoint-force (second inforce))) ; find
   ))
;
;	get-support  --  get support item for pair of blocks
;
;	One of (sup-by b1 b2 forces) or (sup-by b2 b1 forces) should
;	be present, but not both.
;
(defun get-support (b1 b2)
  (let ((item (present (list 'sup-by b1 b2 '*))))
       (if item
	   (fourth item)		; support list
	   nil)))			; none, nil
;
;	reverse-forces  --  reverse forces on object
;
;	We have the forces A is exerting on B, and we want the forces B
;	is exerting on A, which is just a sign change.
;
;	Input and output are lists of elements of the form
;
;	(touchpoint force)
;
(defun reverse-forces (forces)
  (mapcar 
   (function
    (lambda (forceitem)
	    (list (first forceitem)		; touchpoint
		  (v3negate (second forceitem)))))	; force
   forces))
;
;      checkstab  --  check the	stability of the unknowns and any other
;		      blocks made unknown in the process.
;
;	Returns:
;		(lose (B1 . ???) (B2 . ???) (B3 . ???) ...)
;
;	This is the main function of the stability test.  The basic
;	concept is that a block is "winning" or "losing", which depends
;	upon whether CHECKSUP was able to find a way to get the net force
;	and moment acting upon the block to go to zero, and that a block
;	is either "known" or "unknown", which depends upon whether the
;	forces applied to the block have been changed since the last time
;	the block was examined.
;
;	The goal of CHECKSUP is to reach a state where all blocks are
;	"known", and preferably "winning".  The process iterates until
;	all unknowns are eliminated.  But blocks may become unknown during
;	the examination process.  This is normal; it is how forces are
;	propagated through the system.
;
;	The process should eventually terminate in all cases, according
;	to Fahlman's paper, but I (Nagle) have been having problems with
;	infinite looping in cases of mutual support.
;
(defun checkstab (unknowns
		  &aux b losers x)
  (when (> debug 1) (format t "Checkstab start.~%"))
  (loop
   ;	Termination test
   (unless unknowns (return))		; end loop
   ;       Try next block.
   (setq b (pop unknowns))
   ;	Debug output
   (when (> debug 1)
	 (format t " Checkstab ~a:  Unknowns: ~a~%  Losers:~%    ~a~%"
		 b unknowns losers))
   
   ;	If immovable, mark as unsupported and try another.
   (cond ((isimmovable b) 
	  (add-support b nil nil)) ; add to database
	 (t
	  ;       Remove it from loser list
	  (when (setq x (assoc b losers)) 
		(setq losers (delete x losers)))
	  
	  ;       Test it
	  (setq x (checksup b))		; support test
	  (when (eq (first x) 'lose)	; if B lost, add to losers
		(push (cons b (third x)) losers))
	  
	  ;	Update interblock forces.
	  (setq x (second x))		; (touchers pushers)
	  (setq unknowns (update-force-relations b (first x) unknowns))
	  (setq unknowns (update-force-relations b (second x) unknowns)))))
  ;	End of iteration, analyze and deal with results.
  (when (> debug 1) (format t "Checkstab losers ~a~%" losers))
  (unless losers (return-from checkstab t))	; no losers, good
  ;	There are losing blocks.  Database items for losing blocks
  ;	must be removed.
  ;	This has the annoying property that repeated CHECKSTATE
  ;	calls on losing situations are very expensive.  But not doing
  ;	this is unsound.
  (mapc #'delete-stability-relationships (mapcar #'car losers))
  (return-from checkstab (cons 'lose losers)))	; return losing state.
;
;	update-force-relations
;
;	Takes the output of checksup and determines what changed, updating
;	the database appropriately.
;
(defun update-force-relations (b rel unknowns)
  (mapc
   (function
    (lambda (blockitem)
	    (let ((bk (first blockitem)))	; block with relation to B.
		 ;	Update SUP-BY relations, note if changed.
		 (when (and (update-force-relation b bk (cddr blockitem))
			    (not (member bk unknowns)))
		       ; add new unknown last
		       (setq unknowns (append unknowns (list bk))))))) 
   rel)
  unknowns)				; return new unknowns
;
;	update-force-relation  --  update force relationship for one block pair
;
;	Updates the SUP-BY relationships with another block.
;
(defun update-force-relation (b1 b2 forcelist
				 &aux newforces changed totforce)
  ;	Construct list of nonzero force items.
  (setq newforces
	(mapcan
	 (function 
	  (lambda (forceitem)
		  (when (not (v3zerop (third forceitem)))
			(setq changed t))	; will output SUP-BY item.
		  ;	Compute total force, old - new
		  (if (v3zerop (setq totforce 
				    (v3diff (second forceitem) 
					   (third forceitem))))
		      nil
		      (list (list (first forceitem) totforce)))))
	 forcelist))			; iterate over CHECKSUP data
  (when (> debug 2)
	(format t "  Blocks ~a:~a  ~a forces: ~a~%" 
		b1 b2 (if changed "NEW" "same") newforces))
  ;	If any change, delete relevant SUP-BY items and replace with new
  (when changed
	(remall (list 'sup-by b1 b2 '*))
	(remall (list 'sup-by b2 b1 '*))
	(when newforces
		(add-support b1 b2 newforces)))
  changed)
;
;      deplist	--  list blocks	supported, even	indirectly, by L members.
;
(defun deplist (l &aux dl)
  (mapc (function 
	 (lambda (x)
		 (mapc (function 
			(lambda (y)
				(unless (member y dl)
					(setq dl (cons y dl)))))
		       (cons x (dependon x)))))
	l)
  dl)
;
;      getsupitems  --	get support item patterns for B	in context CON.
;
; 	Note that this returns a list of new-format support items.
;
(defun getsupitems (b con)
     (context-lookup (list 'sup-by b '* '*) con))
;
;      getsupsof  --  get list of blocks supporting B.
;
(defun getsupsof (b &aux blk items blocks)
  ;	Get relevant database items.
  (setq items		
	(cond ((context-lookup (list 'sup-by b '* '*))) ; lookup
	      ((eq t (checkstate))	; if not computed, do
	       (context-lookup (list 'sup-by b '* '*))) ; look again
	      (t (error "Bad state ~A - getsupsof" b)))) ; nothing, error.
  ;	Extract blocks from database items.
  (mapc (function 
	 (lambda (item) 
		 (setq blk (third item)) ; get supporting block
		 (when blk 
		       (setq blocks (cons blk blocks))))) ; add if non-null
	items)
  blocks)
;
;      getsupby	 --  get list of blocks	supported directly by B.
;
;	The approach is inefficient.
;
(defun getsupby	(b)
  (prog (l)
	(setq l nil)
	(mapc (function (lambda (x)
				(and (member b (getsupsof x))
				     (setq l (cons x l)))))
	      (getobs))
	(return l)))
;
;      dependon	 --  get list of blocks	supported, even	indirectly, by B.
;
;	This is a transitive closure and a brutally inefficient one.
;	Every iteration of the outer loop results in a call to GETSUPBY,
;	which in turn applies GETSUPSOF to every block.
;
(defun dependon	(b)
  (prog (l ll)
	(setq l (getsupby b) 
	      ll nil)
	loop
	(when (null l) (return ll))
	(setq ll (cons (first l) ll) l (rest l))
	(do ((x (getsupby (first ll)) (rest x)))
	    ((null x))
	    (or (eq (first x) b)
		(member (first x) l)
		(member (first x) ll)
		(setq l (cons (first x) l))))
	(go loop)))
;		
;      supporting  --  get list	of all blocks supporting B, even indirectly.
;
;	Another brute-force transitive closure.
;
(defun supporting (b)
  (prog (l ll)
	(setq l (getsupsof b) 
	      ll nil)
	loop 
	(and (null l) (return ll))
	(setq ll (cons (first l) ll)
	      l  (rest l))
	(do ((x (getsupsof (first ll)) (rest x)))		; for all supports
	    ((null x))
	    (or (eq (first x) b)
		(member (first x) l)
		(member (first x) ll)
		(setq l (cons (first x) l))))
	(go loop)))
;
;      checkstate  --  check entire state for touch and	stability lossage.
;
;	This looks very expensive.
;
(defun checkstate (&aux x)
  (cond ((eq (first (setq x (gettouch))) 'hit) ; if hit, return hit list
	 x)
	(t (checkstab (getunknowns))))) ; otherwise return stability info
;
;	getunknowns  --  find all blocks of unknown support.
;
;	A block of unknown support is one that lacks any "sup-by"
;	relations.
;	The returned list is sorted by altitude.
;
(defun getunknowns (&aux items b h u)
	;	Look at all blocks, take unknown ones.
	(setq items (context-lookup '(at * *))) ; get all "at" items.
	(map nil				; examine them.
	 (function (lambda (item)
			   (setq b (second item) 	; block
				 h (vector3-z (third item))) ; Z of AT point.
			   (or (isimmovable b)	; ignore if immovable
			       (present (list 'sup-by b '* '*)) ; or has supp.
			       (setq u (hsort b h u))))) ; add block U in place.
	 items)
	(mapcar #'cdr u))	; Just return block list.
;
;	hsort  --  sort list by height
;
;	This is a merge, not a sort.
;	Merges block B (at height H) into list L, which is of the form
;	((H1 . B1) (H2 . B2) ...)
;
(defun hsort (b	h l &aux ll)
  (unless l (return-from hsort (list (cons h b))))
  (when (> h (caar l)) (return-from hsort (cons (cons h b) l)))
  (setq ll l)
  (loop
   (cond ((or (null (rest ll)) (> h (caadr ll)))
	  (rplacd ll (cons (cons h b) (rest ll)))
	  (return l)))
   (setq ll (rest ll))))
;
;	Support item database maintenance
;
;	Relevant database items are of the form
;
;		(sup-by SUPPORTED-BLOCK SUPPORTING-BLOCK FORCES)
;	or	(sup-by SUPPORTED-BLOCK NIL NIL)
;
;	(Previously, the forms were
;		(SUPPORTED-BLOCK sup-by SUPPORTING-BLOCK FORCES)
;		(SUPPORTED-BLOCK sup-by nil)
;	but this has been changed as part of the removal of CONNIVER code.)
;
;	add-support  --  add a SUP-BY relationship.
;
;	There may be only one such relationship in effect for any pair of
;	blocks at any time.
;
(defun add-support (b1 b2 forces &aux item)
  (setq item (present (list 'sup-by b1 b2 '*))) ; look up existing
  ;	Add if nonexistent, exact duplicate is OK, partial dup is error.
  (cond (item
	 (unless (equal (fourth item) forces)
		 (error "Duplicate SUP-BY entry: ~a" item))) ; trouble
	(t
	 (context-add (list 'sup-by b1 b2 forces)))))
;
;	delete-support-dependencies  --  delete all sup-by relations for B.
;
;	Something is being done to B, so all support relationships
;	dependent on B must go.
;
(defun delete-support-dependencies (b)
  (map nil #'delete-support-dependency 
       (context-lookup (list 'sup-by '* b '*))))
;
;	delete-support-dependency  --  delete a single SUP-BY relation.
;
;	This also deletes all dependent SUP-BY relations.
;
(defun delete-support-dependency (relation)
  (assert (eq (first relation) 'sup-by))	; must be SUP-BY
  (context-delete relation)		; delete this one
  (when (third relation)		; when not SUP-BY B NIL
	(delete-support-dependencies (third relation)))) ; del indirect supports
;
;	delete-stability-relationships  --  delete for dependents and self.
;
;	Used when B is moved.  
;	
(defun delete-stability-relationships (b)
  (unless (or (null b) (isimmovable b))
	  ;	Delete all depending on B
	  (map nil
	       (function 
		(lambda (item)
			(context-delete item)
			(delete-stability-relationships (third item))))
	       (context-lookup (list 'sup-by b '* '*)))
	  ;	Both directions of dependency
	  (mapc
	   (function 
	    (lambda (item)
		    (context-delete item)
		    (delete-stability-relationships (second item))))
	   (context-lookup (list 'sup-by '* b '*)))))
;
;	delete-all-stability-relationships
;
;	Deletes all stability-related database items, forcing a complete
;	recomputation of stability.  Legitimately used when gravity is
;	adjusted during shake simulation.  Non-legitimately used by some
;	code that modifies the database.
;
(defun delete-all-stability-relationships nil
  (remall '(sup-by * * *)))
;
;	Internal checking -- validates that NETF and NETM cumulative 
;	computations are right.
;
;
;	calcnetf  --  recalculate net force
;
;	This is a debug aid; we keep NETF as a running value, and this
;	routine is used to insure that it was correctly kept.
;
(defun calcnetf (blocklist 
		 &aux (netf (make-vector3 0.0 0.0 0.0)))
  ;	For all blocks in block list
  (mapc (function 
	 (lambda (blk)
		 ;	For all touchpoints of block
		 (mapc (function
			(lambda (forceitem)
				;	Add applied force.
				(setq netf (v3plus netf (second forceitem)))
				;	Subtract force applied this cycle
				(setq netf (v3diff netf (third forceitem)))))
		       (cddr blk)))) ; over all forces, old and new
	blocklist)			; over all blocks on list
  netf)
;
;	calcnetm  --  recalculate net moment
;
(defun calcnetm (blocklist point 
			   &aux dist (netm (make-vector3 0.0 0.0 0.0)))
  (mapc (function 
	 (lambda (blk)
		 (mapc (function
			(lambda (forceitem)
				(setq dist (v3diff (first forceitem) point))
				(setq netm (v3plus netm 
						  (v3cross dist 
							  (second forceitem))))
				(setq netm (v3diff netm 
						  (v3cross dist 
							  (third forceitem))))))
		       (cddr blk)))) ; over all forces, old and new
	blocklist)			; over all blocks on list
  netm)
;
;
;	chk-netf-netm  --  check that NETF and NETM have the correct values.
;
(defun chk-netf-netm (blk touchers 
			    &aux wnetf wnetm cgrav weight)
  (unless internal-checking (return-from chk-netf-netm t)) ; checking on?
  (setq cgrav (getcgrav blk))
  (setq weight (getweight blk))
  ;	Calculate net force.
  (setq wnetf (v3plus (calcnetf (first touchers))  
		     (calcnetf (second touchers))))
  (setq wnetf (v3plus wnetf weight))
  ;	Calculate net moment at POINT.
  (setq wnetm (v3plus (calcnetm (first touchers) point)  
		     (calcnetm (second touchers) point)))
  (setq wnetm (v3plus wnetm
		     (v3cross (v3diff cgrav point) weight)))
  (unless (v3zerop (v3diff netf wnetf))
	  (cerror "Use old value."
		  "NETF incorrect, is ~a, should be ~a" netf wnetf))
  (unless (v3zerop (v3diff netm wnetm))
	  (cerror "Use new value."
		  "NETM incorrect, is ~a, should be ~a" netm wnetm)
	  (setq netm wnetm)))
;
;	dumptouchers  --  output toucher information
;
(defun dumptouchers (b touchers
		     &aux (lt (third touchers)) (lp (fourth touchers)))
	(format t "  Touchers of ~a: ~a  Pushers: ~a~%"
		b
		(mapcar #'car lt)		; just block names
		(mapcar #'car lp)))
