;
;	BUILD, by Scott Fahlman, 1972
;
;						Version 1.36 of 5/19/87
;
;	Add/Move/Delete Block operations
;
(require 'database "database")
(use-package 'database)
(require 'builddefs "builddefs")
(use-package 'builddefs)
(require 'buildutil "buildutil")
(use-package 'buildutil)
(require 'vector "vector")
(use-package 'vector)
(require 'stability "build6")
(use-package 'stability)
(provide 'addblock)
;
;	User interface, low level.
;
; 	addblock  --  add a block by name.
;
;	No collision detection.
;
(defun addblock (bname x y z 
		   &optional (rx 0.0) (ry 0.0) (rz 0.0) (context context)
		   &aux newat dat b)
  (unless (setq b (name-block bname))	; look up block
	(error "There is no block ~a." bname))
  (when (setq dat (present (list 'at b '*))) ; look up presence of block
    (block-deletion-update b)	; delete old use of B
    (format t "~A already in context - old instance removed." b) (terpri)
    (context-delete dat))		; delete instance of B
  (setq newat (atchange orgloc x y z rx ry rz)) ; AT matrix relative to origin
  (addat b newat)			; add the block
  (picture-changed b context))		; update graphics
;
; 	addatloc  --  add a block by name.
;
;	Uses LOC form for position.
;	No collision detection.
;
(defun addatloc (bname loc
		   &aux dat b)
  (unless (setq b (name-block bname))	; look up block
	(error "There is no block ~a." bname))
  (when (setq dat (present (list 'at b '*))) ; look up presence of block
    (block-deletion-update b)		; delete old use of B
    (format t "~A already in context - old instance removed." b) (terpri)
    (context-delete dat))		; delete instance of B
  (addat b loc))			; add the block
;
;	removeblock  --  remove a block by name
;
(defun removeblock (bname)
(let ((b (name-block bname)))		; look up block by name
(unless b (error "There is no block ~a." bname)) ; must exist
  (remblock b)				; remove the block
  (picture-changed b context)		; update the display
  t))
;	
;      moveblock  --  move a block, no checking
;
;	The move is relative to the present position, and angle is
;	defined in such a way that 1.0 is a complete circle.
;
(defun moveblock (bname  x y z 
			 &optional (rx 0.0) (ry 0.0) (rz 0.0) (context context)
			 &aux item loc)
  (let ((b (name-block bname)))		; name to block conversion
       (unless b (error "There is no block ~a." bname)) ; fails
       (unless				
	(setq item (present (list 'at b '*)))	; look up
	(error "~A not present - moveblock" b))
       (setq loc (third item))		; get old location
       (context-delete item)		; delete at old location
       (block-deletion-update b)	; delete any interblock relationships
       ;	Add at new location
       (context-add (list 'at b 
			  (atchange loc x y z rx ry rz)))
       (picture-changed b context)))		; update graphics
;
;	Internal routines for manipulating the solid model
;
;	addat  --  add block at given position, internal
;
;	No checking.  The block had better not be placed in this context.
;
(defun addat (b newat)
  (check-type newat location)		; must be a location
  (check-type (location-rotation newat) rotation-type) ; matrix form OK
  (context-add (list 'at b newat))	; B is now at NEWAT.
  (block-addition-update b))		; update implications
;
;      remblock  --  Remove a block, no checking
;
(defun remblock (b &optional (context context) &aux item)
  (unless
   (setq item (present (list 'at b '*)))	; look up
   (error "~A not present - remblock" b)) ; no find is fatal
  (context-delete item)			; release relation
  (block-deletion-update b)		; delete all relations involving B
  t)
;
;	atchange  --  calculate new location for new position of block.
;
;	This relocates the block described by locp to the location
;	and rotation described by tx,ty,tz,rx,ry,rz.
;
;	The operation is relative to the present position, and angles
;	are defined such that 1.0 is a full circle, or 360 degrees.
;
(defun atchange	(locp tx ty tz rx ry rz)
  (let ((rot 	(make-array '(3 3) :element-type 'long-float))
	(pos	(v3plus locp (make-vector3 tx ty tz))) ; new position
	(cx	(cos (* rx twopi)))	; sines and cosines of rotation
	(cy	(cos (* ry twopi)))
	(cz	(cos (* rz twopi)))
	(sx	(sin (* rx twopi)))
	(sy	(sin (* ry twopi)))
	(sz	(sin (* rz twopi))))
       (store (rot 0 0)	(* cy cz)) ; build new rotation matrix
       (store (rot 0 1)	(* cy sz))
       (store (rot 0 2)	(- sy))
       (store (rot 1 0)	(- (* sx sy cz) (* cx sz)))
       (store (rot 1 1)	(+ (* sx sy sz) (* cx cz)))
       (store (rot 1 2)	(* sx cy))
       (store (rot 2 0)	(+ (* cx sy cz) (* sx sz)))
       (store (rot 2 1)	(- (* cx sy sz) (* sx cz)))
       (store (rot 2 2)	(* cx cy))
       (make-location
	:rotation (m3x3mult (location-rotation locp) rot); new matrix
       :x (vector3-x pos) 		; new position
       :y (vector3-y pos) 
       :z (vector3-z pos))))
;
;      rotblock  --  rotate B around arbitrary X Y Z point.
;
(defun rotblock (b x y z rx ry rz &optional (context context)
		   &aux (item (present (list 'at b '*)))) ; look up LOC of B.
  (unless item (error "~A not present - rotblock" b))
  (let ((loc (third item)))		; get loc from relation
       (context-delete item)		; release relation
       (block-deletion-update b)	; delete any relations with other
       ;	Add at new location.
       (addat b
	      (unrelate-location (atchange loc x y z 0.0 0.0 0.0)
				 (atchange orgloc x y z rx ry rz)))
       (picture-changed b context)))		; update graphics
;
;
;	Delete stability and collision relationships.
;
;	These really ought to be implemented as daemons of the database,
;	but we only need to do this from one place, so it wasn't worth it
;	implementing a general daemon mechanism.
;
;	block-deletion-update
;
(defun block-deletion-update (b)
  (delete-collision-relationships b)		; update COLLISION package
  (delete-stability-relationships b))		; update STABILITY package
;
;	block-addition-update  --  block has been added, update implications
;
(defun block-addition-update (b)
  (let ((touchers (gettouch b)))		; get touching blocks
       (when (eq (first touchers) 'hit)		; if collision
	     (return-from block-addition-update touchers)) ; bad
       ;	Force stability recomputation for all relevant blocks.
       (mapc #'delete-stability-relationships 
	     (mapcar #'car touchers)))	; over touchers
  t)
;
;      getlosses  --  get all losing blocks with losses, not just final	ones.
;
;	Getlosses removes losing blocks until a stable state is achieved.
;
;	Returns (stable-context . losing-blocks)
;
(defun getlosses (l &aux y)
  (in-context 
   (push-context)		; begin thought experiment
   ;	Remove initial losing blocks.
   (mapc (function
	  (lambda (x) 
		  (remblock (first x))))	; remove block
	 l)				; remove all losing blocks.
   ;	Remove losing blocks until checkstate is happy.
   (loop
    (cond ((eq t (setq y (checkstate)))	; If checkstate is happy
	   (return-from getlosses (cons context l))) ; done
	  ((eq 'hit (first y))		; A hit is not expected.
	   (error "hit in getlosses ~A" y)))
    ;	Remove new losing blocks returned by checkstate.
    (mapc (function 
	   (lambda (x) 
		   (remblock (first x))	; remove block
		   (push x l)))		; add to losing blocks
	  (rest y)))))			; apply to checkstate result.
;
;	add-translation-volume  --  construct and add a translation volume
;
;	The volume is added in the current context, so BCONTEXT should be
;	other than the current context, or there will be a collision.
;
(defun add-translation-volume (b to bcontext)
  (let* ((from (in-context bcontext (getat b)))		; where is B now?
	 (newloc (setf-vector3 (copy-location from)	; new loc is midpoint
			       (v3scale (v3plus from to) 0.5)))
	 (transvec (relate-location to from))	; move in block coord system
	 (vol (translation-volume b transvec))) 	; new volume
	(addat vol newloc)		; add the new block
	vol))				; return the new block
