" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to
  bug-snlp@cs.washington.edu; the same address should be used for problems."
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Program   : "TOPI"
;;;  Author    : Stephen Soderland  
;;;              (modified and enhanced by Dan Weld and Tony Barrett)
;;;  Date      : Summer/Autumn 1990/Winter 1991
;;;
;;;  Description:
;;;
;;;  Logic:
;;;
;;;    This program is a domain independent, conjuntive, linear planner
;;;    designed to compare its performance with the non-linear version,
;;;    called McTweak.  It uses the same representation as McTweak, and the
;;;    same routines for unification and adding variable bindings.
;;;
;;;    Each iteration of selects a plan from a priority queue of plans.  
;;;    For each goal of the current plan, it finds all "new plans" that achieve
;;;    that goal.  Constraints are added if necessary to prevent the new step
;;;    from "clobbering" any other goals of the current plan.  All such plans 
;;;    are added to the queue.  This continues until a plan is reached in 
;;;    which all goals can be unified with the initial conditions.
;;;
;;;    The branching factor is high, since completeness requires that many
;;;    possible combinations of bindings be considered.  If a new step might
;;;    delete some goals, there may be many possible constraints that prevent
;;;    the clobbering.  For each of these, there may be bindings that 
;;;    resolve only one goal, but other bindings that resolve additional
;;;    goals.  Each such possible next plan must be created and added to
;;;    the queue. 

(in-package 'topi)

(use-package 'variable)
(use-package 'plan-utils)

(export '(plan display-plan))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Structures

(defstruct (MCS-PLAN (:constructor make-mcs-plan*)
                     (:print-function print-plan))

  initial				; list of initial conditions
  steps                   ; list of steps
  goals                   ; list of sub-goals of current plan
  bindings                ; hashtable of bind
  high-step               ; integer number of highest step in plan
  open                    ; number of goals that don't unify with initial
  lazy                    ; used for lazy evaluation by mcstrips2
  rank                    ; used to cache evaluation of plan
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Variables

(defvar *lazy* nil)
(defvar *bindings* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Handy interface for using the program

;;; This returns two values: a plan and a stat record
;;; If this is the top-level call, then both are printed.
(defun PLAN (initial 
	    goals 
	    &key 
	    (rank-fun #'mcs-rank2)
	    (search-fun #'bestf-search))
  (multiple-value-bind (plan done? time q-len av-branch)
      (mcs* initial goals rank-fun search-fun)
    (values plan (make-stat :algo         "TOPI"              
                            :date         (today)
                            :prob-num     1
                            :num-init     (length initial)       
                            :num-goal     (length goals)
                            :plan-len     (if plan (mcs-plan-high-step plan) 0)
                            :reached-max? (>= *nodes-visited* *search-limit*)
                            :complete?    done?
                            :time         time
                            :visited      *nodes-visited*     
                            :created      *plans-created*
                            :q-len        q-len
                            :ave-branch   (float av-branch)
                            :unify-count  *unify-count*
                            :rank-unifies *compute-rank-unifies*
                            :add-bindings *add-bind-count*)
            )))

;;; Quiet version
(defun MCS* (init goal rank-fn search-fn)
  (if *lazy*
      (mcstrips2 (make-McS-plan :initial init
                                :goals goal
                                :bindings (new-bindings)
                                :high-step 0
                                :open (length goal)
                                :lazy 0)
                 rank-fn search-fn) 
      (mcstrips (make-McS-plan :initial init
                               :goals goal
                               :bindings (new-bindings)
                               :high-step 0
                               :open (length goal)
                               :lazy 0)
                rank-fn search-fn)))

;;; used mcs-rank2 because it can handle the sussman anomally

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Main control level of TOPI

;;; Returns 5 values: final plan, t or nil denoting success, run time
;;; length of q at termination, average branching factor
(defun MCSTRIPS (init-plan rank-fun search-fn)
  (reset-stat-vars)			; clear globals for metering
  (let* ((init-time (get-internal-run-time)))
    (multiple-value-bind (plan bfactor qlength)
        (funcall search-fn init-plan
                 #'plan-refinements #'is-complete? rank-fun *search-limit*)
      (values plan                      ; the plan itself
              (and plan
                   (null (mcs-plan-goals plan))) ; plan successfull?
              (- (get-internal-run-time) init-time) ; time
              qlength                   ; final length of the queue
              bfactor))))		; average branching factor

;;;;;;;;;;;;;;;;;;;;;;;;
;;; This program is similar to MCSTRIPS except that it uses a lazy
;;; plan-refinement function
(defun MCSTRIPS2 (init-plan rank-fun search-fn)
  (reset-stat-vars)			; clear globals for metering
  (let* ((init-time (get-internal-run-time)))
    (multiple-value-bind (plan bfactor qlength)
        (funcall search-fn init-plan
                 #'lazy-plan-refinements #'is-complete? rank-fun 
		 *search-limit*)
      (values plan                      ; the plan itself
              (and plan
                   (null (mcs-plan-goals plan))) ; plan successfull?
              (- (get-internal-run-time) init-time) ; time
              qlength                   ; final length of the queue
              bfactor))))               ; average branching factor

(defun PLAN-REFINEMENTS (plan &aux (ret nil))
  (dolist (goal-cond (McS-plan-goals plan))
    (setf ret (nconc (next-plans goal-cond plan) ret)))
  ret)
  
(defun LAZY-PLAN-REFINEMENTS (plan &aux (ret nil))
  (let ((goal-cond (nth (McS-plan-lazy plan) (McS-plan-goals plan))))
    (when goal-cond
      (setf ret (next-plans goal-cond plan))
      (when (nth (incf (mcs-plan-lazy plan)) (McS-plan-goals plan))
	(push plan ret))))
  ret)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Testing for completion

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  
;;;  Test if all goals can be unified with initial conditions
;;;  
(defun IS-COMPLETE? (plan)
  (if (> *trace* 1) (format t "~%Is Plan complete?"))
  (if plan
      (let ((final-bind nil))
        (cond ((= 0 (McS-plan-open plan))
               (if (> *trace* 1) (format t "~%  Possibly complete plan"))
               (setf *computing-rank* t)
               (setf final-bind (complete-test2 plan 
                                                (McS-plan-goals plan)
                                                (McS-plan-bindings plan)))
	       (when final-bind
		 (setf (mcs-plan-bindings plan) final-bind)
		 (setf (mcs-plan-goals plan) nil))
	       (setf *computing-rank* nil)))
        final-bind)
    nil))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  If each goal separately unifies with some initial condition, try to build
;;;  a binding that will allow all goals to unify with initial conditions.
;;;  Return this binding or nil if none exists.
(defun COMPLETE-TEST2 (plan goals binding)
  (setf *bindings* binding)
  (cond
    ((null goals)
     binding)
    (t
     (let ((final-bind  nil)
	   (bind-list nil)
	   (goal (car goals)))
       (if (> *trace* 1) (format t "~%    Test goal for complete: ~a" goal))
       (setf goals (cdr goals))
       (dolist (init (McS-plan-initial plan))
	 (let ((new-bind (unify goal init binding)))
	   (if (and (> *trace* 1) new-bind)
	       (format t "~%      Goal ~a matches init ~a with bindings ~a"
		       (mapcar #'bind-var goal)
		       (mapcar #'bind-var init)
		       new-bind))
	   (if new-bind
	       (setf bind-list
		     (cons (add-bind (car new-bind) (copy-bindings binding))
			   bind-list)))))
       (dolist (bind bind-list final-bind)
	 (setf final-bind (complete-test2 plan goals bind))
	 (if final-bind (return final-bind)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Ranking partial plans

;;;;;;;;;;;;;;;;;;;;;;;
;;;  To find a plan with the least searching.  The ranking is based
;;;  on the number of open-goals plus number of steps.  Open goals are those 
;;;  that don't unify with initial conditions.  If two plans have the same 
;;;  number of open-goals, give a lower rank to the one with fewer goals.
(defun McS-RANK (plan)
  (let ((w .01) ; tie breaking weighting factor
	(num-steps (length (McS-plan-steps plan)))
	(open-goals (McS-plan-open plan))
	(num-goals (length (McS-plan-goals plan))))
    (if (null open-goals) (setf open-goals 0))
    (+ open-goals num-steps
       (* w num-goals))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This causes a depth first search
(defun McS-DF-RANK (plan)
  (- 100 (length (McS-plan-steps plan))))

(defun DF-TIRE-RANK (plan)
  (if (< 22 (length (McS-plan-steps plan)))
      most-positive-fixnum
    (- (McS-plan-open plan)
       (* 100 (length (McS-plan-steps plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A slightly more agressive search that ignores the number of steps.
(defun McS-RANK1 (plan)
  (let ((w .01) ; tie breaking weighting factor
	(open-goals (McS-plan-open plan))
	(num-goals (length (McS-plan-goals plan))))
    (if (null open-goals) (setf open-goals 0))
    (+ open-goals
       (* w num-goals))))


;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To find an optimal plan with fewest steps.  The ranking is based 
;;;  primarily on the number of steps.  If two plans have the same 
;;;  number of steps, give a lower rank to the one with fewer  open-goals.
;;;  Open goals are those that don't unify with initial conditions.
;;;  If there are two such plans, give lower rank to the one with fewer
;;;  goals.  If a plan with no open-goals is reached, give it lowest 
;;;  rank.
(defun McS-RANK2 (plan)
  (let ((w 1000) ; weighting factor
	(num-steps (length (McS-plan-steps plan)))
	(open-goals (McS-plan-open plan))
	(num-goals (length (McS-plan-goals plan))))
    (if (null open-goals) (setf open-goals 0))
    (cond 
      ((= open-goals 0)
       num-steps)
      (t
       (+ (* num-steps w w)
	  (+ (* open-goals w)
	     num-goals))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To find a plan with the least searching.  The ranking is based
;;;  on the number of open-goals plus number of steps.  Open goals are those 
;;;  that don't unify with initial conditions.  If two plans have the same 
;;;  number of open-goals, give a lower rank to the one with fewer goals.
(defun McS-RANK-STD (plan)
  (let ((w .01) ; tie breaking weighting factor
	(num-steps (length (McS-plan-steps plan)))
	(open-goals (McS-plan-open plan))
	(num-goals (length (McS-plan-goals plan))))
    (if (null open-goals) (setf open-goals 0))
    (+ open-goals num-steps
       (* w num-goals))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Compute the number of goals which cannot be unified with any initial
;;;  conditions.  This is used in ranking plans for the priority queue, as
;;;  well as deciding if a plan might be a complete plan.
(defun COUNT-OPEN (plan)
  (setf *computing-rank* t)
  (let ((open-goals (length (McS-plan-goals plan))))
    (dolist (goal (McS-plan-goals plan))
      (dolist (init (McS-plan-initial plan))
	(if (unify goal init (McS-plan-bindings plan))
	   (decf open-goals))))
    (setf *computing-rank* nil)
    open-goals))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Compute a plan's refinements

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Next-Plans (goal plan)
;;; Returns a list of all valid new plans that acheive that goal
;;;
;;; Logic:
;;;   New-plan-list = nil
;;;   For each Step in templates
;;;     For each Add of Step
;;;       If Add unifies with Goal under *McS-plan* bindings
;;;         Append new plans from CREATE-PLANS to New-plan-list
;;;   Return New-plan-list
(defun NEXT-PLANS (goal plan)
  (let ((new-plan-list nil))
    (dolist (templ *templates* new-plan-list)
      (if (> *trace* 1) 
          (format t "~%New step? ~a" (plan-step-action (car templ))))
      (let ((step (instantiate-step (car templ) 
                                    (+ (mcs-plan-high-step plan) 1)))
	    (templ-bind (instantiate-bind (cadr templ)
                                          (+ (mcs-plan-high-step plan) 1))))
	(dolist (add-cond (plan-step-add step))
	  (let ((new-bind (unify goal add-cond (McS-plan-bindings plan))))
	    (when new-bind
              (if (> *trace* 1) 
                  (format t "~% Possible new step? ~a bind ~a"
                          (plan-step-action step) (car new-bind)))
              (let ((new-plans 
                     (create-plans plan goal step (car new-bind) templ-bind)))
                (if new-plans 
                    (setf new-plan-list 
                          (nconc new-plans new-plan-list)))))))))))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Create-Plans (plan goal step new-bind template-bind)
;;; Returns a list of plans that achieve the goal by adding the step and adding
;;; new-bind and template-bind to the plan bindings.  Additional bindings may
;;; be needed to avoid having a step delete clobber other goals.  More goals
;;; may also be achieved by adding further bindings.  For completeness, a new
;;; plan must be created for every possible combination of additional bindings.
;;; 
;;; Logic:
;;;   New-plan-list = nil
;;;   Bind2 = add new-bind and template-bind to plan bindings 
;;;   Bind-List = bindings that resolve any clobbers by step deletes
;;;   For each Bind in Bind-list
;;;     New-plan = plan with step added, bindings = Bind, Goal removed,
;;;                and step preconditions added to goals
;;;     Add New-plan to New-plan-list
;;;     Find any other plans that remove other goals and add to New-plan-list
;;;  Return New-plan-list
(defun CREATE-PLANS (plan goal step new-bind template-bind)
  (let ((bind2 (add-bind template-bind 
			 (add-bind new-bind 
				   (copy-bindings (McS-plan-bindings plan)))))
	(goals2 (remove goal (mcs-plan-goals plan) :test #'equal))
	(new-plan-list nil)
	(bind-list nil))
    ;;  bug - goal might unify with add-cond before template bindings are
    ;;  added but not unify with template bindings, e.g. (puton ?X1 ?Y1)
    ;;  adds (on A Table) but violates template bindings (not (?Y1 Table))
    ;;  Does not happen in our artificial domains.
    
    (setf bind-list (resolve-clobber plan step bind2))
    (dolist (bind-pair bind-list)
      (if (> *trace* 2) 
	  (format t "~%   Computing open-goals of new plan"))
      (let ((bind (car bind-pair))
	    (constraint (cadr bind-pair))
	    (new-plan                                 
	     (make-McS-plan 
	      :initial (McS-plan-initial plan)
	      :steps (cons step (McS-plan-steps plan))
	      :goals (add-precond step goals2 (car bind-pair))
	      :bindings (car bind-pair)
              :lazy 0
	      :high-step (plan-step-id step))))
	(setf (McS-plan-open new-plan) (count-open new-plan))
	(if (> *trace* 0) 
	    (new-plan-msg goal step bind constraint 
			  (McS-plan-open new-plan) (McS-plan-goals new-plan))
	    (format t "."))
	(if (> *trace* 4) (display-plan new-plan))
	(push new-plan new-plan-list)
	(setf new-plan-list 
	      (append (remove-goals plan step goals2 bind constraint)
		      new-plan-list))))
    new-plan-list))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Resolve-Clobber (plan step bind2)
;;; Returns a list of plan bindings that resolve any step deletes that clobber
;;; plan goals.  If there is more than one way to separate variables so that
;;; the delete does not unify with a goal, then there will be more than one
;;; possible binding returned.  For clarity in diagnostic output, the
;;; constraints used for separation are also returned.
;;;
;;; Logic:
;;;   New-plan-list = nil
;;;   For each Dele of Step
;;;     For each Goal of plan
;;;       For each Binding found so far that resolves clobbers
;;;         Case: Dele unifies with Goal with no new bindings
;;;           Retain Binding
;;;         Case: Dele does not unifies with Goal
;;;           Discard Binding
;;;         Case: Dele unifies with Goal with new bindings
;;;           Make new bindings by adding constraints to Binding
;;;   Return New-plan-list
(defun RESOLVE-CLOBBER (plan step bind2)
  (let ((bind-list (list (list bind2 nil)))
	(new-bind-list nil))
    (if (> *trace* 1) 
	(format t "~% Resolving clobbers for ~a" (plan-step-action step)))
    (dolist (dele (plan-step-dele step))
      (dolist (goal (McS-plan-goals plan))
	(dolist (bind-pair bind-list)
	  (if (> *trace* 1) 
	      (format t "~%   Clobber? goal ~a and dele ~a" goal dele))
	  (let ((clobber (unify dele goal (car bind-pair))))
	    (cond
	      ((null clobber)
	       (setf new-bind-list (cons bind-pair new-bind-list)))
	      ((redundant-bind clobber (car bind-pair))
	       (if (> *trace* 1)
		   (format t "~%   Discard step ~a" (plan-step-action step)))
	       nil)	; discard bind-pair
	      (t
	       (setf new-bind-list 
		     (append (constrain bind-pair clobber) new-bind-list))))))
	(setf bind-list new-bind-list)
	(setf new-bind-list nil)))
    bind-list))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; 
;;; returns list of bind-pairs
;;; 
(defun CONSTRAIN (bind-pair clobber)
  (let ((bind-list nil)
	(binding (car bind-pair))
	(constraint (cadr bind-pair))
	(bind2 nil)
	(neg-bind nil))
    ;; bug- Systematicity bug.  Should add both codesig. and noncodesig 
    ;; constraints.  Problem does not arise in our artificial domains (with 
    ;; single variable predicates).
    (dolist (clobber-bind (car clobber))
      (if (eql (car clobber-bind) 'not)
	  (setf neg-bind (cdr clobber-bind))
	  (setf neg-bind (list (list 'not clobber-bind))))
      (if (> *trace* 1) (format t "~%   Separate with ~a" neg-bind))
      (setf bind2
	    (add-bind neg-bind (copy-bindings binding)))
      (setf bind-list
	    (cons (list bind2 (cons neg-bind constraint))
		  bind-list)))
    bind-list))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Remove-Goals (plan step goals2 bind2 constraint)
;;; Returns a list of new plans that remove further goals by unifying step adds
;;; with remaining goals.  If such a new plan is found, Remove-Goals is called
;;; recursively on the remaining goals.
;;;
;;; Logic:
;;;   New-plan-list = nil
;;;   For each Goal of Goals2
;;;     For each Add of Step
;;;       If Goal unifies with Add
;;;         Goals3 = remove Goal from Goals2 
;;;         Bind3 = add unifying bindings to Bind2
;;;         New-plan = plan with Step added, bindings = Bind3,
;;;                    and goals = add Step preconditions to Goals3
;;;         Add New-plan to New-plan-list
;;;         Call Remove-Goals recursively to add more plans to New-plan-list
;;;   Return New-plan-list
;;;
(defun REMOVE-GOALS (plan step goals2 bind2 constraint)
  (if (> *trace* 2) (format t "~% Remove more goals? goals ~a" goals2))
  (let ((new-plan nil)
	(new-plan-list nil))
    (dolist (goal goals2)
      (dolist (add-cond (plan-step-add step))
	(let ((new-bind (unify add-cond goal bind2)))
	  ;; bug- systematicity bug.  Problems arise when a step can add
	  ;; more than one goal.
	  (cond
	    (new-bind
	     (let ((bind3 (add-bind (car new-bind) (copy-bindings bind2)))
		   (goals3 (remove goal goals2 :test #'equal)))
	       (setf new-plan
		     (make-McS-plan 
		      :initial (McS-plan-initial plan)
		      :steps (cons step (McS-plan-steps plan))
		      :goals (add-precond step goals3 bind3)
		      :bindings bind3
                      :lazy 0
		      :high-step (plan-step-id step)))
	       (setf (McS-plan-open new-plan) (count-open new-plan))
	       (if (> *trace* 0)
		   (new-plan-msg goal step bind3 constraint
				 (McS-plan-open new-plan) 
				 (McS-plan-goals new-plan))
		   (format t "."))
	       (if (> *trace* 4) (display-plan new-plan))
	       (setf new-plan-list 
		     (append (remove-goals plan step goals3 bind3 constraint)
			     (cons new-plan new-plan-list)))
	       ))))))
    new-plan-list))


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Adds step preconditions to goals, but avoids adding
;;; duplicated goals
;;; 
(defun ADD-PRECOND (step goals bindings)
  (let ((new-goals goals)
        (ok? t))
    (dolist (precond (plan-step-precond step))
      (setf ok? t)
      (dolist (goal goals)
	(if (redundant-bind (unify precond goal bindings)
			    bindings)
	    (setf ok? nil)))
      (if ok?
	  (push precond new-goals)))
    new-goals))

(defun MAKE-MCS-PLAN (&rest args)
  (setq *plans-created* (+ 1 *plans-created*))
  (apply #'make-mcs-plan* args))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 10. Print functions 

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print function for Plan
(defun DISPLAY-PLAN (plan &optional (st t))
  (format st "~%Initial    : ~a~%Steps      :" (McS-plan-initial plan))
  (setf *bindings* (McS-plan-bindings plan))
  (dolist (step-n (McS-plan-steps plan))
    (if (< (plan-step-id step-n) (McS-plan-high-step plan))
	(format st "~%            "))
    (format st " ~15a" 
	    (mapcar #'bind-var (plan-step-action step-n))))
  (format st "~%Goals      : ~a ~%Open goals : ~a"
	  (mapcar #'(lambda (x) (mapcar #'bind-var x))
		  (McS-plan-goals plan))
	  (McS-plan-open plan))
  (format st "~&")
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Diagnostic message when new plan is created
(defun NEW-PLAN-MSG (goal step bind2 constraint open-goals goals2)
  (setf *bindings* bind2)
  (format t "~% * New Step resolving ~a with action ~a"
	  (mapcar #'bind-var goal)
	  (mapcar #'bind-var (plan-step-action step)))
  (if constraint
      (format t "~%    Constrain clobber by ~a"
	  (mapcar #'bind-var constraint)))
  (format t "~%    new goals (~a are open) : ~a ~%    unify-count ~a"
	 open-goals
	 (mapcar #'bind-var goals2)
	 *unify-count*))

(defun bind-var (variable)
  (bind-variable variable *bindings*))

(defun PRINT-PLAN (plan &optional (stream t) depth)
  (declare (ignore depth))
  (if *verbose* (display-plan plan stream)
    (format stream "#TOPI-Plan<S=~a; O=~a>"
	    (length (mcs-plan-steps plan))
	    (length (mcs-plan-goals plan)))))
