" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-snlp@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Program   : "SLP"
;;;  Authors   : Stephen Soderland
;;;              (modified & extended by Dan Weld and Tony Barrett)
;;;              (turned into a total-order planner by Tony Barrett)
;;;  Date      : Summer/Autumn 1990
;;;
;;;  Description:
;;;  
;;;    This program is a domain independent, conjuctive, total-order planner
;;;    which claims to be both complete and sound.  The representation of
;;;    conditions and actions is similar to STRIPS, with each step having
;;;    a list of added conditions and a list of deleted conditions.  
;;;
;;;    A plan includes a list of steps and a list of links between steps. 
;;;    Links are in the form  (id1 condition1 id2) where id1 is a step that
;;;    establishes condition1, which is in turn a precondition of step id2.
;;;    The plan also has a list of open conditions (not yet acheived) and a 
;;;    list of unsafe links (possibly clobbered by another step).  The other 
;;;    components of a plan are ordering constraints and bindings of variables.
;;;
;;;    Each iteration of the main control level creates possible next plans 
;;;    which are added to a priority queue.  New plans are created by adding
;;;    constraints to resolve unsafe links.  If there are no unsafe links,
;;;    new plans are created by adding a new step or new link to achieve
;;;    an open condition.  The plan is completed when there are no more unsafe
;;;    links or open conditions.  
;;;  
;;;    SNLPlan is adapted from a non-linear planner by David McAllester.    
;;;    As the first phase of this project, I implemented Tweak, a non-linear
;;;    planner described by David Chapman in Artificial Intelligence 1987.
;;;    Code from my Tweak program was incorporated into SNLPlan.  The data
;;;    structures for variable bindings were later revised to enhance
;;;    performance.  
;;;
;;;    SLPlan was addapted from SNLPlan.

(in-package 'tocl)

(use-package 'variable)
(use-package 'plan-utils)

(export '(plan display-plan))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Outline of this file
;;;
;;;  1.   Data Structures
;;;  2.   Global Variables
;;;  3.   Handy interface for using the program
;;;  4.   Main control level of SNLP
;;;  5.   Ranking partial plans
;;;  6.   Handling open conditions
;;;  6.1.   Adding new steps
;;;  6.2.   Adding links to old steps
;;;  7.   Protecting causal links
;;;  7.1.   Resolving an unsafe link
;;;  7.2.   Detecting unsafety conditions
;;;  8.   handle partial orderings.
;;;  9.   Handy utility functions
;;; 10.   Print functions 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Data Structures

(defstruct (SNLP-PLAN (:constructor make-SNLP-plan*)
	    (:print-function print-plan))
  steps                   ; list of steps
  links                   ; list of (ID1 condition ID2)
  unsafe                  ; list of (link clobber-ID clobber-bind)
  open                    ; list of (condition ID)   
  ordering                ; list of (ID1 ID2)
  bindings                ; hashtable of bind 
  high-step               ; integer number of highest step in plan
  )

(defstruct ORDER
  ordering        ;; Array for the ordering of the steps
  (range nil))    ;; (Last-added-step-after-here . Last-added-step-before-here)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Handy interface for using the program

;;; This returns two values: a plan and a stat record
;;; If this is the top-level call, then both are printed.
(defun PLAN (initial
	     goals
	     &key
	     (rank-fun #'rank3)
	     (search-fun #'bestf-search))
  (multiple-value-bind (plan done? time q-len av-branch)
      (snlplan initial goals rank-fun search-fun)
    (values plan (make-stat :algo        "TOCL"             
                            :date        (today)
                            :prob-num    1
                            :num-init    (length initial)       
                            :num-goal    (length goals)
                            :plan-len    (if plan (snlp-plan-high-step plan) 0)
                            :reached-max? (>= *nodes-visited* *search-limit*)
                            :complete?    done?
                            :time         time
                            :visited      *nodes-visited*     
                            :created      *plans-created*
                            :q-len        q-len
                            :ave-branch   (float av-branch)
                            :unify-count  *unify-count*
                            :rank-unifies *compute-rank-unifies*
                            :add-bindings *add-bind-count*)
            )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Main control level of SNLP

;;; Returns 5 values: final plan, t or nil denoting success, run time
;;; length of q at termination, average branching factor
(defun SNLPLAN (init goals rank-fun search-fun)
  (reset-stat-vars)			; clear globals for metering
  (let* ((init-time (get-internal-run-time))
	 (goal-step (make-plan-step :ID :Goal :precond goals))
	 (init-steps (list (make-plan-step :ID '0 :add init)
			   goal-step))
	 (init-plan (make-SNLP-plan
		     :steps init-steps
		     :open (mapcar #'(lambda (x) (list x goal-step)) goals)
		     :bindings (new-bindings)
		     :ordering (make-order :ordering init-steps)
		     :high-step 0)))
    (multiple-value-bind (plan bfactor qlength)
	(funcall search-fun init-plan
		 #'plan-refinements #'plan-test rank-fun *search-limit*)
      (values plan			; the plan itself
	      (and plan
		   (null (snlp-plan-unsafe plan))
		   (null (snlp-plan-open plan))) ; plan successfull?
	      (- (get-internal-run-time) init-time) ; time
	      qlength			; final length of the queue
	      bfactor))))		; average branching factor

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Goal test function for a search.
(defun PLAN-TEST (plan)
  (not (or (SNLP-plan-unsafe plan)
	   (SNLP-plan-open plan)
	   (order-range (SNLP-plan-ordering plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a list of one step refinements to the current plan.
;;;
;;;   Select one unsafe link and find all new plans that
;;; resolve it by separation, demotion, or promotion.  Separation is 
;;; adding bindings that prevent unifying of the condition with the
;;; added conditions of a possible clobbering step.  Demotion and
;;; promotion add ordering constraints so that a possibly clobbering
;;; step comes before the first step of the link or after the second
;;; step.
;;;   If there are no unsafe links, select an open condition to resolve
;;; by adding a new step or making a link to an existing step.  Adding 
;;; a new step or new link may cause a link to become unsafe (possibly 
;;; clobbered).
;;;
(defun PLAN-REFINEMENTS (plan)
  (cond ((SNLP-plan-unsafe plan)                 (handle-unsafe plan))
	((order-range (SNLP-plan-ordering plan)) (linearize plan))
	(t (handle-open plan))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. ranking partial plans

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Perform an A* search on the number of causal links in a plan.
(defun RANK (plan)
  (let ((num-links (length (SNLP-plan-links plan)))
	(open (length (SNLP-plan-open plan))))
    (+ num-links open)))

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;;  A slightly more agressive search, ignoring the number of steps
(defun RANK1 (plan)
  (let ((unsafe (length (SNLP-plan-unsafe plan)))
	(open (length (SNLP-plan-open plan))))
    (+ unsafe open)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; To find the plan with the fewest steps.  If a complete plan is found
;;; give it a lower rank than an incomplete plan with one fewer steps.
(defun RANK2 (plan)
  (let ((num-steps (length (SNLP-plan-steps plan)))
	(unsafe (length (SNLP-plan-unsafe plan)))
	(open (length (SNLP-plan-open plan))))
    (cond
      ((= 0 (+ unsafe open))
       (- num-steps 2))
      (t
       num-steps))))

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;;  To find a plan with the least searching.  Ranking is based on the
;;;  number of unsafe plus the number of goals to be resolved plus the
;;;  number of steps.  Including the number of steps prevents creating
;;;  long plans that add cycles of moves getting no closer to the goal.
(defun RANK3 (plan)
  (let ((num-steps (length (SNLP-plan-steps plan)))
	(unsafe (length (SNLP-plan-unsafe plan)))
	(open (length (SNLP-plan-open plan))))
    (+ unsafe open num-steps)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Rank function for performing a depth first search.
;;; Prefers plans with more links.
(defun DF-RANK (plan)
  (- 1000 (length (SNLP-plan-links plan))))

(defun DF-TIRE-RANK (plan &aux (count 0))
  (dolist (o (SNLP-plan-open plan))
    (when (eql :goal (plan-step-id (cadr o)))
      (incf count)))
  (cond ((< 22 (length (SNLP-plan-steps plan)))
	 most-positive-fixnum)
	(t
	 (+ (length (SNLP-plan-steps plan))
	    (length (SNLP-plan-open plan))
	    (* 1000 count)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. Handling open conditions

(defun HANDLE-OPEN (plan)
  (multiple-value-bind (open-cond plan2)
      (remove-open plan)
    (append (add-step open-cond plan2)
	    (new-link open-cond plan2))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a new current plan with the first open condition removed, and
;;;  returns the removed open condition and the new plan
(defun REMOVE-OPEN (plan)
  (let ((open-cond (car (SNLP-plan-open plan))))
    (values open-cond
	    (tweak-plan plan
			:open (cdr (SNLP-plan-open plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6.1. Adding new steps

;;;;;;;;;;;;;;;;;;;;;;;;                                                 
;;; Returns a list of pairs of new plans and step#.  The new plan
;;; establishes the condition by adding a new step, instantiated 
;;; from a step in *templates* with post-conditions that unify with
;;; the condition. 
;;; Modified by DSW 1/91 to reduce consing
(defun ADD-STEP (open-cond plan)
  (let ((new-plans nil)
	(new-plan nil)
	(bindings nil)
	(condition (car open-cond))
	(id (cadr open-cond)))
    (dolist (templ *templates*) 
      (if (> *trace* 1) (format t "~%New step? ~a" 
				(plan-step-action (car templ))))
      (let* ((new-step-num (+ (snlp-plan-high-step plan) 1))
             (step (instantiate-step (car templ) new-step-num))
             (templ-bind (instantiate-bind (cadr templ) new-step-num)))
	(dolist (add-cond (plan-step-add step))
	  (let ((new-bind (unify condition add-cond 
				 (SNLP-plan-bindings plan))))
	    (cond
	      (new-bind
	       (setf new-bind (car new-bind))
	       (if (> *trace* 0.5) 
		   (format t "~%* New Step ~a  New bindings ~a  unify-count ~a"
			   (plan-step-action step) new-bind *unify-count*))
	       (if (> *trace* 0) (format t "S"))
	       (setf bindings 
		     (add-bind new-bind
			       (add-bind templ-bind 
					 (copy-bindings 
					  (SNLP-plan-bindings plan)))))
	       (when bindings
		 (setf new-plan
		       (tweak-plan plan
			:steps (cons step (SNLP-plan-steps plan))
			:links (cons (list step condition id)
				     (SNLP-plan-links plan))
			:ordering 
			(make-order
			 :range (cons (find 0 (order-ordering (SNLP-plan-ordering plan)) 
					    :key #'plan-step-id)
				      id)
			 :ordering (order-ordering (SNLP-plan-ordering plan)))
			:open (nconc (new-step-open step)
				     (SNLP-plan-open plan))
			:bindings bindings
			:high-step (plan-step-id step)))
		 (test-step new-plan step) 
		 (test-link new-plan (car (SNLP-plan-links new-plan)))
		 (push new-plan new-plans))
	       ))))))
    new-plans))

;;;;;;;;;;;;;;;;;;;
;;;  Adds the preconditions of the new step as open conditions of the new plan.
(defun NEW-STEP-OPEN (step)
  (let ((new-open nil))
    (dolist (precond (plan-step-precond step))
      (setf new-open (cons 
		      (list precond step)
		      new-open)))
    new-open))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6.2. Adding links to old steps

;;;;;;;;;;;;;;;;;;;;;;;;                                                 PAGE 5
;;;  Creates new plans to achieve an open condition.  Each possibly prior step
;;;  is tested to see if it has an add condition matching the open condition.
;;;  If so, a new plan is created with that link added.
(defun NEW-LINK (open-cond plan)
  (let ((new-plans nil)
        (new-link nil)
        (condition (car open-cond))
        (id (cadr open-cond)))
    (do ((steps (order-ordering (SNLP-plan-ordering plan)) (cdr steps)))
	((eq (car steps) id) new-plans)
      (let ((step (car steps)))
	(if (> *trace* 1) (format t "~%New link?  step ~a" (plan-step-id step)))
	(dolist (add-cond (plan-step-add step))
	  (let ((new-bind (unify condition add-cond
				 (SNLP-plan-bindings plan))))
	    (when new-bind
	      (setf new-bind (car new-bind))
	      (setf new-link (list step condition id))
	      (if (> *trace* 0.5)
		  (format t "~% * New link ~a , bindings ~a   unify-count ~a"
			  new-link new-bind *unify-count*))
	      (if (> *trace* 0) (format t "L"))
	      (push (tweak-plan plan
				:links (cons new-link (SNLP-plan-links plan))
				:bindings (add-bind new-bind
						    (copy-bindings
						     (SNLP-plan-bindings plan))))
		    new-plans)
	      (test-link (car new-plans) new-link)
	      )))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. Handling unsafe links

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Code used to handle a detected unsafety condition.  Returns a list
;;;  of plan refinements that resolve the threat.
(defun HANDLE-UNSAFE (plan)
  (multiple-value-bind (unsafe-link plan2)
      (remove-unsafe plan)
    (if (and (unify (third unsafe-link)          ; if threat still exists...
		    (second (car unsafe-link))
		    (SNLP-plan-bindings plan2))
	     (possibly-between (second unsafe-link) 
			       (car (car unsafe-link)) (third (car unsafe-link))
			       plan2))
	(append (separate unsafe-link plan2)
		(demote unsafe-link plan2)
		(promote unsafe-link plan2))
	(list plan2))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Creates a new current plan with the first unsafe link removed, and
;;;  returns the removed link and the new plan.
(defun REMOVE-UNSAFE (plan)
  (values (car (SNLP-plan-unsafe plan))
          (tweak-plan plan
           :unsafe (cdr (SNLP-plan-unsafe plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7.1. Resolving an unsafe link

;;;  To resolve an unsafe link, add constraints that prevent the bindings that
;;;  would clobber it.  The input is a link, the ID of the step that clobbers
;;;  it, and the bindings that clobber it.  The output is a list of new plans.
(defun SEPARATE (unsafe-link plan)
  (let* ((clobber-id (cadr unsafe-link))
         (new-step-id (car (SNLP-plan-steps plan)))
         (source-id (caar unsafe-link))
         (dest-id (third (car unsafe-link)))
         (range (order-range (SNLP-plan-ordering plan)))
         (new-range (cond ((eql new-step-id clobber-id)
                           (cons (if (before (car range) source-id plan)
                                     source-id (car range))
                                 (if (before dest-id (cdr range) plan)
                                     dest-id (cdr range))))
                          ((eql new-step-id source-id)
                           (cons (car range)
                                 (if (before clobber-id (cdr range) plan)
                                     clobber-id (cdr range))))
                          ((null range) range)
                          (t (error "Unexpected condition arose"))))
         (new-order (make-order
                     :range new-range
                     :ordering
                     (order-ordering (SNLP-plan-ordering plan)))))
    (mapcar #'(lambda (x)
                (tweak-plan plan
                 :ordering new-order
                 :bindings x))
            (not-unify (third unsafe-link)
                       (second (car unsafe-link))
                       (SNLP-plan-bindings plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; check if id1 is before id2 in a plan.  This works if the orders of both
;;; id1 and id2 have been finalized.
(defun BEFORE (id1 id2 plan)
  (and (not (eq id1 id2))
       (eq id1 (find-if 
		#'(lambda (x) 
		    (or (eq x id1) (eq x id2)))
		(order-ordering (SNLP-plan-ordering plan))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes before the establishing step.  The output is a list of new
;;;  plans.
;;;
;;;    Case 1:  *--------------| =>          *--------|
;;;                    |                    |
;;;
;;;    Case 2:  |--------------| =>    |--------------|
;;;                    *              *
(defun DEMOTE (unsafe-link plan)
  (when (order-range (SNLP-plan-ordering plan))
    (let* ((clobber-id (cadr unsafe-link))
           (id (caar unsafe-link))
           (new-step-id (car (SNLP-plan-steps plan)))
           (range (order-range (SNLP-plan-ordering plan)))
           (new-range  (cond ((eql id new-step-id)           ; Case 1
                              (if (before (car range) clobber-id plan)
                                  (cons clobber-id (cdr range))
                                  range))
                             ((eql clobber-id new-step-id)   ; Case 2
                              (if (before id (cdr range) plan)
                                  (cons (car range) id)
                                  range))
                             (t (cons (car range) (car range))))))
      (if (> *trace* 1) (format t "~%   Demote? step ~a" clobber-id))
      (if (before (car new-range) (cdr new-range) plan)
          (list (tweak-plan plan
			    :ordering 
			    (make-order
			     :range new-range
			     :ordering
			     (order-ordering (SNLP-plan-ordering plan)))))
          nil))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  To resolve an unsafe link, add an ordering constraint so that the clobber
;;;  step comes after the second step of the link.  The output is a list of new
;;;  plans.
;;;
;;;    Case 1:  |--------------* =>   |----*
;;;                    |                    |
;;;
;;;    Case 2:  |--------------| =>   |--------------|
;;;                    *                              *
(defun PROMOTE (unsafe-link plan)
  (when (order-range (SNLP-plan-ordering plan))
    (let* ((clobber-id (cadr unsafe-link))
           (id (third (car unsafe-link)))
           (new-step-id (car (SNLP-plan-steps plan)))
           (range (order-range (SNLP-plan-ordering plan)))
           (new-range  (cond ((eql id new-step-id)           ; Case 1
                              (if (before clobber-id (cdr range) plan)
                                  (cons (car range) clobber-id)
                                  range))
                             ((eql clobber-id new-step-id)   ; Case 2
                              (if (before (car range) id plan)
                                  (cons id (cdr range))
                                  range))
                             (t (cons (car range) (car range))))))
      (if (> *trace* 1) (format t "~%   Promote? step ~a" clobber-id))
      (if (before (car new-range) (cdr new-range) plan)
          (list (tweak-plan plan 
			    :ordering
			    (make-order
			     :range new-range
			     :ordering
			     (order-ordering (SNLP-plan-ordering plan)))))
	nil))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7.2. Detecting unsafety conditions

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether a link is possibly clobbered by the delete conditions of any
;;;  steps possibly between the ID1 and ID2 of the link.  Returns nil if the
;;;  link is safe, otherwise returns a list of (link clobber-id clobber-bind)

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Tests whether a link is possibly clobbered by the delete conditions of any
;;;  steps possibly between the ID1 and ID2 of the link.  Returns nil if the
;;;  link is safe, otherwise returns a list of (link clobber-id clobber-bind)
(defun TEST-LINK (plan link)
  (let ((first (position (car link) (order-ordering (SNLP-plan-ordering plan))))
	(last (position (caddr link) (order-ordering (SNLP-plan-ordering plan)))))
    (when (null first)
      (setf first (position (car (order-range (SNLP-plan-ordering plan)))
			    (order-ordering (SNLP-plan-ordering plan)))))
    (when (null last)
      (setf last (position (cadr (order-range (SNLP-plan-ordering plan)))
			   (order-ordering (SNLP-plan-ordering plan)))))
    (dolist (step (subseq (order-ordering (SNLP-plan-ordering plan))
			  (1+ first) last))
      (test-l plan link step))))

(defun TEST-STEP (plan step)
  (dolist (link (SNLP-plan-links plan))
    (when (possibly-between step (car link) (third link) plan)
      (test-l plan link step))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Test the interaction between a single link and a single step
;;;  Return nil (if no interaction) otherwise an unsafety condition
(defun TEST-L (plan link step)
  (let ((bind2 (SNLP-plan-bindings plan)))
    (dolist (dele-cond (plan-step-dele step))
      (when (unify dele-cond (second link) bind2)
	(push (list link step dele-cond) (SNLP-plan-unsafe plan))
	
	(if (> *trace* 0.5)
	    (format t "~%   * New unsafe ~a   unify-count ~a"
		    (list link step dele-cond)
		    *unify-count*))
	))
    (dolist (add-cond (plan-step-add step))
      (when (unify add-cond (second link) bind2)
	(push (list link step add-cond) (SNLP-plan-unsafe plan))
	
	(if (> *trace* 0.5)
	    (format t "~%   * New unsafe ~a   unify-count ~a"
		    (list link step add-cond)
		    *unify-count*))
	))
    plan))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 8. Linearization

;;;;;;;;;;;;;;;;;;;;;;;
;;;  Take a plan with one step only partially ordered and create
;;;  all linear plans that match the partial ordering.
(defun LINEARIZE (plan)
  (mapcar #'(lambda (o)
	      (tweak-plan plan :ordering (make-order :ordering o)))
          (orders (order-ordering (SNLP-plan-ordering plan))
		  (car (order-range (SNLP-plan-ordering plan)))
		  (cdr (order-range (SNLP-plan-ordering plan)))
		  (car (SNLP-plan-steps plan)))))

;;;;;;;;;;;;;;;;;;;;;;;
;;;  Take a list like (1 2 3 4 5 6 7), a range like (3 . 5), and a step-id
;;;  like 33.  Creates a list of lists.  Above would give:
;;;    ((1 2 3 33 4 5 6 7)(1 2 3 4 33 5 6 7))
(defun ORDERS (ord first last step)
  (cond (first (let ((ords (if (eq first (car ord))
                               (orders (cdr ord) nil last step)
                               (orders (cdr ord) first last step))))
                 (do ((o ords (cdr o)))
                     ((null o) ords)
                   (push (car ord) (car o)))))
        ((eq (car ord) last) (list (cons step ord)))
        (t (cons (cons step ord)
                 (let ((ords (orders (cdr ord) nil last step)))
                   (do ((o ords (cdr o)))
                       ((null o) ords)
                     (push (car ord) (car o))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 8. handle total orderings.

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Tests if id can become between first-id and last-id within plan.
(defun POSSIBLY-BETWEEN (id first-id last-id plan)
  (and (not (eql first-id last-id))
       (possibly-prior first-id id         plan)
       (possibly-prior          id last-id plan)))

;;;;;;;;;;;;;;;;;;;;;;;;
;;; test if id might come before step-id
(defun POSSIBLY-PRIOR (id step-id plan)
  (let* ((new-step-id (car (SNLP-plan-steps plan))))
    (cond ((eql id step-id) nil)
          ((null (order-range (SNLP-plan-ordering plan)))
           (before id step-id plan))
          ((eql id new-step-id)
           (before (car (order-range (SNLP-plan-ordering plan))) step-id plan))
          ((eql step-id new-step-id)
           (before id (cdr (order-range (SNLP-plan-ordering plan))) plan))
          (t (before id step-id plan)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 9. Handy utility functions

(defun tweak-plan (plan &key           ; initially written by jsp
                        (steps :same)
                        (links :same)
                        (unsafe :same)
                        (open :same)
                        (ordering :same)
                        (bindings :same)
                        (high-step :same))
  "Return a new plan that is a copy of PLAN with the supplied
   instance variable tweaked."
  (flet ((tweak-it (keyword plan-accessor)
           (if (eq keyword :same)
               (funcall plan-accessor plan)
               keyword)))
    (make-snlp-plan
     :steps (tweak-it steps #'snlp-plan-steps)
     :links (tweak-it links #'snlp-plan-links)
     :unsafe (tweak-it unsafe #'snlp-plan-unsafe)
     :open (tweak-it open #'snlp-plan-open)
     :ordering (tweak-it ordering  #'snlp-plan-ordering)
     :bindings (tweak-it bindings #'snlp-plan-bindings)
     :high-step (tweak-it high-step #'snlp-plan-high-step))))


(defun MAKE-SNLP-PLAN (&rest args)
  (setq *plans-created* (+ 1 *plans-created*))
  (apply #'make-SNLP-plan* args))

(defun prop-output (prop plan)
  (cons (car prop)
	(mapcar #'(lambda (x) 
		    (bind-variable x (snlp-plan-bindings plan)))
		(cdr prop))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 10. Print functions 

;;;;;;;;;;;;;;;;
;;; This prints out a plan produced by SLP.  Notice that no topological
;;; ordering is needed.  snlp-plan-ordering already contains the linear order.
(defun DISPLAY-PLAN (pl &optional (stream t) ignore)
  (declare (ignore ignore))
  (format stream "~%")
  (let* ((plan (if (order-range (snlp-plan-ordering pl))
                   (car (linearize pl))
                   pl))
         (order (order-ordering (snlp-plan-ordering plan)))
         (i 0))
    (dolist (step-n order)
      (cond
        ((eql (plan-step-id step-n) '0)
         (format stream "~%Initial  : ~a~%" (plan-step-add step-n)))
        ((eql (plan-step-id step-n) :Goal)
	 (format stream "~%~%Goal     : ~a" (plan-step-precond step-n)))
	(t (format stream "~%Step ~3a : ~15a  Created ~2a" 
		   (incf i)
		   (prop-output (plan-step-action step-n) pl)
		   (plan-step-id step-n)))))
    (if (or (SNLP-plan-unsafe plan)
            (SNLP-plan-open plan))
        (format stream "~%Unsafe   : ~a ~%Open     : ~a"
                (mapcar #'(lambda (x)
                            `((,(caar x)
                               ,(prop-output (cadar x) plan) ,(caddar x))
                              ,(cadr x)))
                        (SNLP-plan-unsafe plan))
                (mapcar #'(lambda (x)
                            `(,(prop-output (car x) plan) ,(cadr x)))
                        (SNLP-plan-open plan)))
      (format stream "~%Complete!"))))

(defun PRINT-PLAN (plan &optional (stream t) depth)
  (declare (ignore depth))
  (if *verbose* (display-plan plan stream)
    (format stream "#TOCL-Plan<S=~a; O=~a; U=~a>"
	    (- (length (snlp-plan-steps plan)) 2)
	    (length (snlp-plan-open plan))
	    (length (snlp-plan-unsafe plan)))))
