/*      STRLIST.C - functions for playing with strlist's
 ***************************************************************************
 *
 *	struct strlist *add_to_strlist(list,str)
 *	struct strlist *list;
 *	char *str;
 *
 *	struct strlist *append_strlist(list,new)
 *	struct strlist *list;
 *	struct strlist *new;
 *
 *	int member_strlist(list,str)
 *	struct strlist *list;
 *	char *str;
 *
 *	void free_strlist(list)
 *	struct strlist *list;
 *
 *	int size_of_strlist(list)
 *	struct strlist *list;
 *
 *      struct strlist *remove_from_strlist(list,str)
 *      struct strlist *list;
 *      char *str;
 *
 *      struct strlist *squeeze_strlist(list)
 *      struct strlist *list;
 *
 *      void write_strlist(list,sep,outfp)
 *      struct strlist *list;
 *      char sep;
 *      FILE *outfp;
 *
 ***************************************************************************
 *      EDIT HISTORY
 *      12-May-89       SRMc - write from scratch, pulling code in from
 *                              throughout AMPLE for part of it
 *      13-Jul-89       hab  - de-"lint" the source
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 ***************************************************************************
 * Copyright 1988, 1989 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <string.h>
#include "strlist.h"

#ifdef __STDC__
# define	P(s) s
#else
# define P(s) ()
#endif

/* strlist.c */
struct strlist *add_to_strlist P((struct strlist *list , char *str ));
struct strlist *append_strlist P((struct strlist *list , struct strlist *new ));
int member_strlist P((struct strlist *list , char *str ));
void free_strlist P((struct strlist *list ));
int size_of_strlist P((struct strlist *list ));
struct strlist *remove_from_strlist P((struct strlist *list , char *str ));
void write_strlist P((struct strlist *list , int sep , FILE *outfp ));
struct strlist *squeeze_strlist P((struct strlist *list ));

/* myalloc.c */
void myalloc_fail P((void ));
char *myalloc P((unsigned size ));
char *mystrdup P((char *str ));
char *myrealloc P((char *s , unsigned size ));
char *myshrink P((char *s ));
void myfree P((char *s ));

#undef P

/***************************************************************************
 * NAME
 *    add_to_strlist
 * ARGUMENTS:
 *    list - pointer to head of strlist
 *    str  - pointer to string to be added
 * DESCRIPTION
 *    Add a string to a list of strings only if it is not already
 *    present in the list.
 * RETURN VALUE
 *    pointer to new list
 */
struct strlist *add_to_strlist(list,str)
struct strlist *list;
char *str;
{
register struct strlist *sp;

if (str == (char *)NULL)
    return(list);		/* just to be safe... */
/*
 *  search for the string in the list
 */
for ( sp = list ; sp ; sp = sp->slink )
    {
    if ( !strcmp( str, sp->stri) )
	return( list );		/* found it, so we don't need a new node */
    }
/*
 *  allocate a new node, and link it in
 */
sp = (struct strlist *)myalloc( sizeof(struct strlist) );
sp->stri = strcpy( myalloc((unsigned)strlen(str)+1), str);
sp->slink = list;
return( sp );
}

/***************************************************************************
 * NAME
 *    append_strlist
 * ARGUMENTS:
 *    list - pointer to head of strlist
 *    new  - pointer to strlist to add to list
 * DESCRIPTION
 *    Append a strlist to another strlist.
 *    Redundant strings in new are deallocated along the way.
 * RETURN VALUE
 *    pointer to new list
 */
struct strlist *append_strlist(list,new)
struct strlist *list;
struct strlist *new;
{
register struct strlist *next;

while ( new )
    {
    next = new->slink;
    if (member_strlist(list,new->stri))
	{
	myfree( new->stri );	/* string already there, so free */
	myfree( (char *)new );	/*   the space used by this node */
	}
    else
	{
	new->slink = list;		/* move this node over to list */
	list = new;
	}
    new = next;
    }
return( list );
}

/***************************************************************************
 * NAME
 *    member_strlist
 * ARGUMENTS:
 *    list - pointer to head of strlist
 *    str  - pointer to string to be checked
 * DESCRIPTION
 *    Check whether a string is stored in a strlist list.
 * RETURN VALUE
 *    nonzero if the string is found in the list, zero otherwise
 */
int member_strlist(list,str)
char *str;
struct strlist *list;
{
register struct strlist *sp;

if (str == (char *)NULL)
    return( 0 );		/* just to be safe... */
/*
 *  search for the string in the list
 */
for ( sp = list ; sp ; sp = sp->slink )
    {
    if ( !strcmp( str, sp->stri) )
	return( 1 );		/* found it! */
    }
return( 0 );
}

/***************************************************************************
 * NAME
 *    free_strlist
 * ARGUMENTS:
 *    list - pointer to head of strlist
 * DESCRIPTION
 *    Delete a strlist, freeing all the space it used.
 * RETURN VALUE
 *    none
 */
void free_strlist( list )
struct strlist *list;
{
register struct strlist *sp, *nextp;

for ( sp = list ; sp ; sp = nextp )
    {
    nextp = sp->slink;		/* save the address of the next node */
    if (sp->stri != (char *)NULL)
	myfree( sp->stri );	/* release the string space */
    myfree( (char *)sp );	/* release the node space */
    }
}

/***************************************************************************
 * NAME
 *    size_of_strlist
 * ARGUMENTS:
 *    list - pointer to head of strlist
 * DESCRIPTION
 *    Return the number of elements stored in the list.
 * RETURN VALUE
 *    number of nodes in the list
 */
int size_of_strlist(list)
struct strlist *list;
{
register struct strlist *sp;
register int size;
/*
 *  step through the list incrementing a counter
 */
for ( size = 0, sp = list ; sp ; ++size, sp = sp->slink )
    ;
return( size );
}

/***************************************************************************
 * NAME
 *    remove_from_strlist
 * ARGUMENTS:
 *    str  - pointer to string to be added
 *    list - pointer to head of strlist
 *    ambig - pointer to ambiguity count for a word
 * DESCRIPTION
 *    Remove a string from a list of strings (only the first occurrence).
 * RETURN VALUE
 *    pointer to the (possibly shorter) list.
 *    NULL if the only item in the list was removed
 */
struct strlist *remove_from_strlist(list,str)
struct strlist *list;
char *str;
{
register struct strlist *sp, *lastp;

if (str == (char *)NULL)
    return(list);		/* just to be safe... */
/*
 *  search for the string in the list
 */
for ( lastp = (struct strlist *)NULL, sp = list ; sp ; sp = sp->slink )
    {
    if ( !strcmp( str, sp->stri) )
	{
	/*
	 *  unlink the node from the list
	 */
	if (lastp == (struct strlist *)NULL)
	    list = sp->slink;		/* (sp == list) at this point */
	else
	    lastp->slink = sp->slink;
	/*
	 *  release the space used by the node
	 */
	if (sp->stri != (char *)NULL)
	    myfree( sp->stri );
	myfree( (char *)sp );
	break;			/* leave the for loop */
	}
    lastp = sp;
    }
return(list);
}

/***************************************************************************
 * NAME
 *    write_strlist
 * ARGUMENTS:
 *    list  - pointer to head of strlist
 *    sep   - pointer to string used to separate the elements
 *    outfp - output FILE pointer
 * DESCRIPTION
 *    Write the members of the list to an output file, separating them by the
 *    indicated character.
 * RETURN VALUE
 *    pointer to new list
 */
void write_strlist(list,sep,outfp)
struct strlist *list;
char sep;
FILE *outfp;
{
register struct strlist *sp;
/*
 *  move along the list, writing the strings to the output file
 */
for ( sp = list ; sp ; sp = sp->slink )
    {
    if (sp->stri != (char *)NULL)	/* except for NULL strings, */
	fputs(sp->stri, outfp);		/*     print the string */
    if (sp->slink)			/* except for the last element */
	putc(sep,outfp);		/*     print separating character */
    }
}

/***************************************************************************
 * NAME
 *    squeeze_strlist
 * ARGUMENTS:
 *    list - pointer to head of strlist
 * DESCRIPTION
 *    Remove redundant strings from a string list.
 * RETURN VALUE
 *    pointer to possibly smaller list
 */
struct strlist *squeeze_strlist(list)
struct strlist *list;
{
register struct strlist *slp, *nlp, *trailp;

for ( slp = list ; slp ; slp = slp->slink )
    {
    trailp = slp;
    for ( nlp = slp->slink ; nlp ; trailp = nlp, nlp = nlp->slink )
	{
	if ( !strcmp( slp->stri, nlp->stri ) )	/* If equal */
	    {
	    trailp->slink = nlp->slink;		/* Delete the node */
	    myfree( nlp->stri );
	    myfree( (char *)nlp );
	    nlp = trailp;
	    }
	}
    }
return( list );
}
