/*      ENVIR.H - define environment constraint structures for AMPLE program
 ***************************************************************************
 *
 *	struct env_item
 *	struct env_cond
 *	struct allo_env
 *	struct morphlist
 *	struct morph_rule
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	 7-May-88	Steve McConnel
 *	19-May-88	SRMc - add ec_flags field to env_cond structure
 *			     - add ENV_NOTMARK lexical token
 *	 7-Sep-88	SRMc - add definitions of STRING_ENVIR and MORPH_ENVIR
 *	 9-Sep-88	SRMc - define E_OPTIONAL, ENV_LPAREN, and ENV_RPAREN
 *	15-Oct-88	SRMc - define struct morphlist, MORPH_NAME, and
 *				MORPH_CLASS
 *	21-Oct-88	SRMc - reorganize the file header comments
 ***************************************************************************
 * Copyright 1988 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */

/**************************************************************************
 * structure and definitions for a single item of an environment constraint
 *
 *  This is used for either string (allomorph surface) environment
 *  constraints or morpheme environment constraints, and for constraints
 *  to either the right or the left.   Each side of a constraint consists
 *  of a linked list of these items.
 *  Below are constants for use in ei_flags.
 */
#define E_NOT             01    /* do NOT want to match this item */
#define E_CLASS           02    /* use class member, not literal string */
#define E_ELLIPSIS        04    /* does not have to be contiguous */
#define E_OPTIONAL       010    /* optional item */
#define E_CAT            020    /* category, not literal string */
#define E_CCL            040    /* category class, not literal string */
#define E_MORPHEME      0100    /* morpheme environ, not string */

#define STRING_ENVIR 0          /* for function parameter passing */
#define MORPH_ENVIR  1          /*  to env_parse() and env_side() */

struct env_item
    {
    char ei_flags;      /* NOT (E_NOT) */
			/* morpheme class, not literal (E_CLASS) */
			/* contiguous or ellipsis (E_ELLIPSIS) */
			/* optional (E_OPTIONAL) */
			/* string or morpheme (E_MORPHEME) */
			/* category, not literal (E_CAT) */
			/* category class, not literal (E_CCL) */
    union
	{
	char *ei_string;                /* literal string or morphname */
	struct string_class *ei_scl;    /* string class */
	struct morph_class *ei_mcl;     /* morpheme class */
	} ei_val;
    struct env_item *ei_link;           /* pointer to next item */
    };

/*************************************************************************
 * structure for environment constraint conditions
 *
 *  This is used for either string (allomorph surface) environment
 *  constraints or morpheme environment constraints.   All such constraint
 *  conditions of a given type are stored in a single linked list.
 *  Each node of this list has a pointer to the left hand environment and
 *  a pointer to the right hand environment, both working outward.  Either
 *  environment may be NULL.  Each node also has a pointer to the next node
 *  in the list.
 */
struct env_cond
    {
    char ec_flags;		/* NOT this environment (E_NOT) */
				/* string or morpheme (E_MORPHEME) */
    struct env_item *ec_left;	/* environment to the left */
    struct env_item *ec_right;	/* environment to the right */
    struct env_cond *ec_link;	/* link to alternative constraint */
    };

/*************************************************************************
 * structure for allomorph environment constraints
 *
 *  This structure contains a pointer to the allomorph surface string
 *  environment constraints and a pointer to the morpheme environment
 *  constraints.  Either may be NULL.
 */
struct allo_env
    {
    struct env_cond *string_cond;	/* environment constraints */
    struct env_cond *morph_cond;	/* environment constraints */
    };

/*************************************************************************
 * structure for morpheme co-occurrence rules
 *
 *  Each node of this list points to either a morphname or a morph_class.
 */
struct morphlist
    {
    unsigned char ml_type;	/* either MORPH_NAME or MORPH_CLASS */
    union { char *mname; struct morph_class *mclass; } ml;
    struct morphlist *ml_link;
    };
#define MORPH_NAME  0
#define MORPH_CLASS 1

/*************************************************************************
 * structure for morpheme co-occurrence rules (constraints)
 *
 *  All of the rules are stored in a single linked list.
 *  Each node of this list has a pointer to a linked list of morphnames (or
 *  morpheme classes), a pointer to an environment list, and a pointer to
 *  the next node in the list.
 */
struct morph_rule
    {
    struct morphlist *mr_morph;	/* morph names/classes */
    struct env_cond *mr_env;	/* environment constraints */
    struct morph_rule *mr_link;	/* link to next node */
    };

/***************************************************************************
 *  values returned by elex_get()
 */
#define ENV_END		0	/* end of this environment */
#define ENV_STRENV	1	/* '/' */
#define ENV_MORPH	2	/* '+/' */
#define ENV_BOUND	3	/* '#' */
#define ENV_ELLIPSIS	4	/* '...' */
#define ENV_LBRACK	5	/* '[' */
#define ENV_RBRACK	6	/* ']' */
#define ENV_MARK	7	/* '_' */
#define ENV_NOT		8	/* '~' */
#define ENV_LITERAL	9	/* literal string (stored in elex_string) */
#define ENV_NOTMARK	10	/* '~_' */
#define ENV_LPAREN	11	/* '(' */
#define ENV_RPAREN	12	/* ')' */
