/*      CHANGE.C - functions for working with string changes
 ***************************************************************************
 *
 *	struct change_list *parse_change(cp)
 *	char *cp;
 *
 *	struct env_cond *ccenv_parse(line)
 *	char *line;
 *
 *	char *apply_cc(buf,cc)
 *	char *buf;
 *	struct change_list *cc;
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	 9-May-88	Steve McConnel - created ENVPAR.C
 *	20-May-88	Steve McConnel - create LOADCC.C and APPLCC.C
 *	 7-Sep-88	SRMc - split ENVPAR.C into ENVPAR.C, AENVPA.C,
 *				CENVPA.C, IENVPA.C, and MENVPA.C
 *			     - split LOADCC.C into LOADCC.C, NEWCHG.C, and
 *                              LOADTB.C
 *      21-Oct-88       SRMc - reorganize the file header comments
 *      10-Nov-88       SRMc - replace free() with myfree()
 *      15-May-89       SRMc - changed the function call apply_cc() to
 *                              make using it easier
 *      17-May-89       SRMc - fixed longstanding bug in environment check
 *                              using new interface to strenv_check()
 *                           - rewrite new_change() as parse_change()
 *      18-May-89       SRMc - pull NEWCHG.C, CENVPA.C, and APPLCC.C into
 *                              CHANGE.C
 *      13-Jul-89       hab  - de-"lint" the source
 * 1.1b 29-Jun-90 BK/ALB Fix for portability to MAC, add string.h
 * 1.1c 29-Jun-90 ALB Fix bug of 8-bit fail in envir, add myisspace
 *	17-Jan-91	SRMc - add ANSI-fied function prototypes
 *			     - add #ifdef BSD for <strings.h>
 *	28-Feb-91	SRMc - add code for handling insertion changes to
 *				apply_cc()
 *	 3-Jan-92	SRMc - bug fix rewrite of apply_cc(), removing size
				argument and safely handling arbitrary inputs
 ***************************************************************************
 * Copyright 1988, 1992 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <ctype.h>
#ifndef isascii
#define isascii(x) (!(x & ~0177))
#endif
#ifdef BSD
#include <strings.h>
#else
#include <string.h>
#endif
#include "opaclib.h"
#include "class.h"
#include "envir.h"
#include "change.h"

#ifdef __STDC__
#define P(s) s
#else
#define P(s) ()
#endif

/* change.c */
struct change_list *parse_change P((char *cp ));
struct env_cond *ccenv_parse P((char *line ));
char *apply_cc P((char *buf , struct change_list *cc ));

/* envlex.c */
void elex_init P((char *line ));
int elex_get P((void ));

/* envpar.c */
void show_badenv P((char *envir ));
struct env_cond *env_parse P((int ismorph ,
			      struct string_class *(*find_cl)() ));
extern char *enverrhead, *enverrtail;

/* envchk.c */
extern int strenv_check P((char *leftstring , int leftsize ,
			   char *rightstring , struct env_cond *ec ));

/* myallo.c */
/* extern char *myalloc P((unsigned size )); */
extern char *mystrdup P((char *str ));
extern char *myrealloc P((char *s, unsigned size));
extern void myfree P((char *s ));

/* sindex.c */
extern char *sindex P((char *s , char *t ));

/* splitl.c */
extern char *split_line P((char *line , char **p_rep , char **p_marks ,
		    char **p_env ));

/* strcla.c */
extern struct string_class *find_scl P((char *name ));

#undef P

/***************************************************************************
 * NAME
 *    parse_change
 * ARGUMENTS
 *    cp  - pointer to a change definition string
 * DESCRIPTION
 *    Parse a single record entry to build a change_list structure.
 * RETURN VALUE
 *    pointer to newly allocated change_list node, or NULL if error
 */
struct change_list *parse_change(cp)
char *cp;
{
static char errhead[] = "\nCHANGE: ";
char buffer[BUFSIZE+1];
register struct change_list *cc;
struct env_cond *pe;
char *match, *replace, *env;
/*
 *  split the change into its pieces, and make sure that everything is okay
 */
strcpy(buffer,cp);
match = split_line(buffer, &replace, (char **)NULL, &env);
if (!match)
    {
    printf("%sInvalid change definition: %s", errhead, cp);
    return( (struct change_list *)NULL );
    }
if (env)
    {
    pe = ccenv_parse( env );
    if (!pe)
	{
	printf("%sInvalid environment in change: %s", errhead, cp);
	return( (struct change_list *)NULL );
	}
    }
else
    pe = (struct env_cond *)NULL;
/*
 *  allocate and fill in a change_list struct, then return its address
 */
cc = structalloc( change_list );
cc->change.ch_old = mystrdup( match );
cc->change.ch_new = mystrdup( replace );
cc->change.ch_env = pe;
return( cc );
}

/*************************************************************************
 * NAME
 *    ccenv_parse
 * ARGUMENTS
 *    line - pointer to NUL-terminated string containing the environment
 *		constraints on a consistent change
 * DESCRIPTION
 *    Parse the environment constraint string for a consistent change,
 *    which may contain only string environment constraints.
 * RETURN VALUE
 *    pointer to env_cond structure containing the constraints, or NULL
 *    if none or error
 */
struct env_cond *ccenv_parse(line)
char *line;
{
register int token;
register struct env_cond *e_cond, *head;
int parserror;

enverrhead = "\nCHANGE ENVIRONMENT: ";
enverrtail = (char *)NULL;
/*
 *  first, check for an empty environment
 */
if (line == (char *)NULL)
    return( (struct env_cond *)NULL);   /* no environment */
while (myisspace(*line))
    ++line;				/* skip leading whitespace */
if (*line == NUL)
    return( (struct env_cond *)NULL);	/* no environment */

elex_init(line);			/* initialize the lexical scan */
parserror = FALSE;			/* no errors yet */

head = (struct env_cond *)NULL;

while ( (token = elex_get()) != EOF )
    {
    if (token == ENV_MORPH)
	token = ENV_STRENV;	/* only string environments, so why not? */
    if ( (token == ENV_STRENV) &&
	 ((e_cond=env_parse(STRING_ENVIR,find_scl))!=(struct env_cond *)NULL))
	{
	e_cond->ec_link = head;	/* link into the list */
	head = e_cond;
	}
    else
	{
	parserror = TRUE;	/* nothing else if valid here */
	break;
	}
    }
if (parserror)
    show_badenv(line);			/* show the bad environment */

return( head );
}

#define MEMINCR 64	/* change to 0 for testing, 64 for production */
#define CHKMEM(ptr, idx, need, size) \
	if ((idx + need) > size) \
	    { \
	    size = idx + need + MEMINCR; \
	    ptr = myrealloc(ptr, size); \
	    }

/***************************************************************************
 * NAME
 *    apply_cc
 * ARGUMENTS
 *    str  - pointer to string to be changed
 *    cc   - pointer to list of changes to apply
 * DESCRIPTION
 *    Apply a list of changes to a string, iterating over the change list
 *    and substituting for each match found.
 * RETURN VALUE
 *    Pointer to dynamically allocated (possibly) changed string.
 */
char *apply_cc(str,cc)
char *str;
struct change_list *cc;
{
char *oldbuf;		/* pointer to working copy of string */
int old;		/* index of current location in oldbuf */
unsigned oldsize;	/* length of string in oldbuf */
char *newbuf;		/* pointer to buffer for changed string */
int new;		/* index of current location in newbuf */
unsigned newsize;	/* allocated size of newbuf */
int pos;		/* index of match position in oldbuf */
int len;		/* length of current match string */
int lensubst;		/* length of current substitution string */
char *p;		/* pointer for searching oldbuf */

if (str==(char *)NULL)
    return(str);
if (cc==(struct change_list *)NULL)
    return( mystrdup(str) );
/*
 * allocate working buffers
 */
oldbuf = mystrdup(str);
newsize = strlen(str) + MEMINCR;
newbuf = myalloc(newsize);
/*
 * apply each change in turn (order is significant)
 */
for ( ; cc ; cc = cc->cl_link)
    {
    old = new = 0;
    lensubst =  strlen(cc->change.ch_new);
    oldsize = strlen(oldbuf);
    if (*cc->change.ch_old == NUL)
	{
	/*
	 * insertion change--try each location in the string
	 */
	++oldsize;	/* include the terminating NUL */
	while (old <= oldsize)
	    {
	    if (strenv_check(newbuf,new,oldbuf+old,cc->change.ch_env))
	        {
		/*
		 * environment matches OK, so insert
		 */
		CHKMEM(newbuf,new,lensubst+1,newsize);
		strcpy(newbuf+new,cc->change.ch_new);
		new += lensubst;
		}
	    /*
	     * copy single character, moving to next position
	     */
	    CHKMEM(newbuf,new,1,newsize);
	    newbuf[new++] = oldbuf[old++];
	    }
	}
    else if ((p = strstr(oldbuf,cc->change.ch_old)) != (char *)NULL)
	{
	len = strlen(cc->change.ch_old);
	do  {
	    pos = p - oldbuf;
	    /*
	     * copy the leading, unmatched stuff
	     */
	    CHKMEM(newbuf,new,pos-old,newsize);
	    while (old < pos)
	        newbuf[new++] = oldbuf[old++];
	    /*
	     * check environment, then copy appropriately
	     */
	    if (strenv_check(newbuf,new,oldbuf+old+len,cc->change.ch_env))
	        {
		/*
		 * environment matches OK, make the change
		 */
		old += len;	/* skip over match in input string */
		CHKMEM(newbuf,new,lensubst+1,newsize);
		strcpy(newbuf+new,cc->change.ch_new);
		new += lensubst;
		}
	    else if (oldbuf[old] != NUL)
	        {
	        /*
		 * move past unwanted match
		 */
		CHKMEM(newbuf,new,1,newsize);
		newbuf[new++] = oldbuf[old++];
		}
	    if (old >= oldsize)
	    	break;
	    } while ( p = strstr(oldbuf+old,cc->change.ch_old));
	/*
	 * copy the trailing unmatched stuff
	 */
	if (old <= oldsize)
	    {
	    CHKMEM(newbuf,new,oldsize-old+1,newsize);
	    strcpy(newbuf+new,oldbuf+old);
	    }
	}
    else
	continue;	/* no changes made this pass */
    /*
     * prepare for next change in the list
     */
   if (strcmp(oldbuf,newbuf) != 0)
	{
	myfree(oldbuf);
	oldbuf = mystrdup(newbuf);
	}
    }
myfree(newbuf);
return(oldbuf);
}
