/* KTEXT.C - KIMMO parsing in an AMPLE framework
 ***************************************************************************
 *
 *	int main(argc, argv)
 *	int argc;
 *	char **argv;
 *
 ***************************************************************************
 *	EDIT HISTORY
 *	27-Feb-91	SRMc - start editing PCKIMMO.C and WF.C (from the
 *				not-yet-released OPAC primer package) to
 *				produce this program
 *	28-Feb-91	SRMc - eliminate needless duplication of analyses in
 *				the output file
 *			     - define and implement control file for this
 *				program, including "changes" for modifying
 *				the format of the analysis and decomposition
 *				output
 *			     - pass ambigchar as an argument to dtbout()
 *	 2-Mar-91	SRMc - fix for using OPAC function library
 *	 6-Mar-91	SRMc - add myshrink() for use by some of the functions
 *				from AMPLE/STAMP
 *			     - write Analysis Failures (AF:) to the log file,
 *				if a log file has been requested
 *	 7-Mar-91	SRMc - rename from PCKAMP to KTEXT
 *	11-Mar-91	SRMc - edit for THINK C
 *	19-Nov-91	SRMc - add fflush(stderr) for THINK C
 *			     - free change buffer in process() if apply_cc() 
 *				allocates a new output buffer
 *			     - add myfree(resp) for head of result list
 *	 3-Jan-92	SRMc - change argument list of apply_cc()
 ***************************************************************************
 * Copyright 1991, 1992 by the Summer Institute of Linguistics, Inc.
 * All rights reserved.
 */
#include <stdio.h>
#include <ctype.h>
#include <setjmp.h>
#ifdef BSD
#include <strings.h>
#define strchr index
#define strrchr rindex
#else
#include <string.h>
#endif

#define NO_STRUCTALLOC
#ifdef THINK_C
#include ":opaclib:opaclib.h"
#include ":opaclib:template.h"
#include ":opaclib:strlist.h"
#include ":opaclib:change.h"
#include ":opaclib:codetab.h"
#else
#include "opaclib/opaclib.h"
#include "opaclib/template.h"
#include "opaclib/strlist.h"
#include "opaclib/change.h"
#include "opaclib/codetab.h"
#endif
#include "pckimmo.h"
#include "patchlevel.h"

#ifdef THINK_C
#include <console.h>
#include <unix.h>
#endif

#ifdef UNIX
#define DIRSEPCHAR '/'
#endif
#ifdef MSDOS
#define DIRSEPCHAR '\\'
#endif
#ifdef MACINTOSH
#define DIRSEPCHAR ':'
#endif

#ifdef __STDC__
char *set_filename(char *filename, char *default_name, char *extension);
int load_ctl_file(char *fname);
int main(int argc, char **argv);
char *myshrink(char *s);

#define P(s) s
#else
#define P(s) ()
#endif

/* standard library functions */
extern void exit P((int status));
extern int isatty P((int fd));

/* lib/ufopen.c */
FILE *ufopen P((char *fname , char *mode ));

/* lib/textin.c */
extern struct word_template *textin P((FILE *infp ));
extern void load_intx_ctl_file P((char *fname ));

/* lib/dtbout.c */
extern void dtbout P((struct word_template *wtp , FILE *outfp ,
		      char *outfilename , int do_orig , int ambigchar ));

/* lib/change.c */
extern struct change_list *parse_change P((char *cp ));
extern char *apply_cc P((char *buf , struct change_list *cc ));
extern void free_change_list P((struct change_list *clp ));

/* lib/record.c */
extern void init_record P((FILE *infp , char *rec_mark , int comment_char ));
extern char *get_record P((FILE *infp , char *rec_mark , int *rec_read ,
			   struct code_table *code_tab ));
extern int free_record P((char *rp ));

/* lib/getwd.c */
extern char *skipwhite P((char *cp ));

/* lib/strcla.c */
extern void init_scl P((void ));
extern void add_scl P((char *line ));

/* lib/strlist.c */
extern void free_strlist P((struct strlist *list ));

/* pckfuncs.c */
					/* conflicts with opaclib/opaclib.h */
/*
extern VOIDP myalloc P((unsigned size ));
*/
extern VOIDP myrealloc P((VOIDP s , unsigned size ));
extern VOIDP mystrdup P((char *str ));
extern void myfree P((VOIDP s ));

/* recogniz.c */
extern RESULT *recognizer P((unsigned char *surf_form , LANGUAGE *lang ,
			     int limit , int trace , FILE *logfp ));

/* rules.c */
int load_rules P((unsigned char *rulefile , LANGUAGE *lang ,
		  unsigned comment ));

/* lexicon.c */
int load_lexicons P((unsigned char *lexiconfile , LANGUAGE *lang ,
		     unsigned comment ));

#undef P

/*****************************************************************************
 *  VARIABLES SET BY COMMAND LINE OPTIONS
 */
int comment_char = ';';			/* -c <comment character> */
char *ctl_file  = (char *)NULL;		/* -x <control file> */
char *in_file   = (char *)NULL;		/* -i <input data file> */
char *out_file  = (char *)NULL;		/* -o <output file> */
char *log_file = (char *)NULL;		/* -l <log file> */
FILE *Log_fp = (FILE *)NULL;		/*      (log FILE pointer) */
int Trace_flag = 0;			/* -t   (flag for tracing on/off) */
int do_orig = 0;			/* -w   (flag for writing \w field) */

/*****************************************************************************
 *  VARIABLES SET BY THE CONTROL FILE
 */
char *txtin_file = (char *)NULL;	/* AMPLE text input control file */
char *rule_file = (char *)NULL;		/* PC-KIMMO rule file */
char *lex_file  = (char *)NULL;		/* PC-KIMMO lexicon file */
struct change_list *an_chg = (struct change_list *)NULL;
struct change_list *dc_chg = (struct change_list *)NULL;

/*****************************************************************************
 *  VARIABLES SET BY THE TEXT INPUT CONTROL FILE (xxINTX.CTL)
 */
/*
 *  number of words read from the input file(s)
 */
int num_words = 0;
/*
 *  list of format markers to include
 */
struct strlist *incl_stdfmt;
/*
 *  list of format markers to exclude
 */
struct strlist *excl_stdfmt;
/*
 *  alphabetic characters
 */
char *alpha;
char *loweralpha;
char *upperalpha;
/*
 *  output internal capitalization info
 */
int incap = 1;
/*
 *  input orthography change table
 */
struct change_list *ortho_chg;
/*
 *  "format markers" begin with a special character (normally '\'), and
 *  continue until a whitespace character
 */
char forminit = '\\';
/*
 *  "bar codes" begin with a special character (normally '|'), and have
 *  one additional character immediately following
 */
char barchar = '|';
char *barcodes = "bcdef";
/*
 *  do de-capitalization processing flag
 */
int do_decap = 1;
/*
 *  ambiguity marker character -- must not be alphabetic
 */
int ambigchar = '%';
/*
 *  decomposition marker character -- must not be alphabetic
 */
int decomp_char = '-';

/***************************************************************************
 *  OTHER GLOBAL VARIABLES
 */
LANGUAGE Lang;			/* data for the current language */
int Limit_flag = 0;		/* flag for limit on/off */
				/* same chars as isspace() */
unsigned char whiteSpc[] = { ' ', '\t', '\n', '\v', '\f', '\r', '\0' };

#ifdef NOGETOPT
/*************************************************************************
 * Here's something you've all been waiting for:  the AT&T public domain
 * source for getopt(3).  It is the code which was given out at the 1985
 * UNIFORUM conference in Dallas.  I obtained it by electronic mail
 * directly from AT&T.  The people there assure me that it is indeed
 * in the public domain.
 *
 * There is no manual page.  That is because the one they gave out at
 * UNIFORUM was slightly different from the current System V Release 2
 * manual page.  The difference apparently involved a note about the
 * famous rules 5 and 6, recommending using white space between an option
 * and its first argument, and not grouping options that have arguments.
 * Getopt itself is currently lenient about both of these things White
 * space is allowed, but not mandatory, and the last option in a group can
 * have an argument.  That particular version of the man page evidently
 * has no official existence, and my source at AT&T did not send a copy.
 * The current SVR2 man page reflects the actual behavor of this getopt.
 * However, I am not about to post a copy of anything licensed by AT&T.
 */
static int	opterr = 1;
static int	optind = 1;
static int	optopt;
static char	*optarg;

static ERR(argv0, s, c)
char *argv0;
char *s;
char c;
{
static char errbuf[2];
(void) write(2, argv0, (unsigned)strlen(argv0));
(void) write(2, s, (unsigned)strlen(s));
errbuf[0] = c;
errbuf[1] = '\n';
(void) write(2, errbuf, 2);
}

static int getopt(argc, argv, opts)
int	argc;
char	**argv, *opts;
{
static int sp = 1;
register int c;
register char *cp;

if (sp == 1)
    {
    if (optind >= argc || argv[optind][0] != '-' || argv[optind][1] == '\0')
	return(EOF);
    else if(strcmp(argv[optind], "--") == 0)
	{
	optind++;
	return(EOF);
	}
    }
optopt = c = argv[optind][sp];
if (c == ':' || (cp=strchr(opts, c)) == NULL)
    {
    if (opterr)
	ERR(argv[0], ": illegal option -- ", c);
    if (argv[optind][++sp] == '\0')
	{
	optind++;
	sp = 1;
	}
    return('?');
    }
if (*++cp == ':')
    {
    if (argv[optind][sp+1] != '\0')
			optarg = &argv[optind++][sp+1];
    else if(++optind >= argc)
	{
	if (opterr)
	    ERR(argv[0], ": option requires an argument -- ", c);
	sp = 1;
	return('?');
	}
    else
	optarg = argv[optind++];
    sp = 1;
    }
else
    {
    if (argv[optind][++sp] == '\0')
	{
	sp = 1;
	optind++;
	}
    optarg = NULL;
    }
return(c);
}

#else /*NOGETOPT*/

#ifdef __STDC__
extern int getopt(int argc, char **argv, const char *optstring);
#else
extern int getopt();
#endif

extern int optind;
extern char *optarg;

#endif /*NOGETOPT*/

/****************************************************************************
 * NAME
 *    usage
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Print a short description of the command line usage of this program.
 * RETURN VALUE
 *    none
 */
static void usage()
{
fputs("\
Usage:  ktext [-c char] file_specifications\n\
    -c char      make char the comment character (default is ;)\n\
    -t           set tracing on (default is off)\n\
    -w           output words in \\w field (default is no \\w field)\n\
", stderr );
fputs("\
    -x ctlfile   specify the control file\n\
    -i infile    specify the input data file\n\
    -o outfile   specify the output file\n\
    -l logfile   specify the PC-KIMMO log file (default is none)\n\
", stderr );
}

/****************************************************************************
 * NAME
 *    parse_cmd_line
 * ARGUMENTS
 *    argc - number of command line arguments
 *    argv - pointer to array of command line arguments
 * DESCRIPTION
 *    Handle the command line arguments for PC-KIMMO.
 * RETURN VALUE
 *    none
 */
static void parse_cmd_line( argc, argv )
int argc;
char **argv;
{
int k;
int errflag = 0;

while ((k = getopt(argc, argv, "c:i:x:o:l:tw")) != EOF)
    {
    switch (k)
	{
	case 'c':		/* comment character */
	    comment_char = *optarg;
	    break;

	case 'x':		/* control filename */
	    ctl_file = optarg;
	    break;

	case 'i':		/* input data filename */
	    in_file = optarg;
	    break;

	case 'o':		/* output filename */
	    out_file = optarg;
	    break;

	case 'l':		/* log filename */
	    log_file = optarg;
	    break;

	case 't':		/* tracing flag */
	    Trace_flag = 1;
	    break;

	case 'w':		/* write \\w (original word) fields */
	    do_orig = 1;
	    break;

	default:
	    ++errflag;
	    break;
	}
    }
if (errflag)
    {
    usage();
    exit(2);
    }
}
#if 99
#ifdef __STDC__
extern VOIDP myalloc (unsigned size);
#endif
#endif

/*****************************************************************************
 * NAME
 *    set_filename
 * ARGUMENTS
 *    filename     - filename given by user (may be NULL)
 *    default_name - default full filename (may be NULL)
 *    extension    - default extension, including leading separation char (.)
 * DESCRIPTION
 *    Check a filename provided by the user, appending a default extension
 *    if the file lacks one, or even providing a default filename.
 * RETURN VALUE
 *    pointer to a dynamically allocated filename string
 */
char *set_filename(filename, default_name, extension)
char *filename;
char *default_name;
char *extension;
{
register char *p;

if (filename == (char *)NULL)
    {
    if (default_name == (char *)NULL)
	return( (char *)NULL );
    else
	return( (char *)mystrdup(default_name) );
    }
#ifdef DIRSEPCHAR
/*
 *  find the beginning of the filename proper, following an optional
 *    string of directory names separated by DIRSEPCHAR
 */
p = strrchr(filename, DIRSEPCHAR);
if (p != (char *)NULL)
    ++p;
else
#endif
    p = filename;

if (strchr(p, *extension) == (char *)NULL)
    {
    return(strcat(strcpy((char *)myalloc((unsigned)
				      (strlen(filename)+strlen(extension)+1)),
			 filename),
		  extension) );
    }
return( (char *)mystrdup(filename) );
}

/****************************************************************************
 * NAME
 *    load_ctl_file
 * ARGUMENTS
 *    fname - name of the control
 * DESCRIPTION
 *    Load the control file.
 * RETURN VALUE
 *    zero if successful, -1 if error occurs
 */
int load_ctl_file(fname)
char *fname;
{
FILE *fp;
char *recp;
char *rp;
int rec_read;

char *end;
int code;
struct change_list *an_tail, *dc_tail, *cc;
static char errhead[] = "\nLOAD CONTROL: ";
short seen_textin, seen_rules, seen_lexicon;
/*
 *  control code table
 */
static struct code_table ctl_codes =
    {
    "\
\\textin\0T\0\
\\rules\0R\0\
\\lexicon\0L\0\
\\scl\0S\0\
\\ach\0A\0\
\\dch\0D\0\
",
    6
    };
fp = fopen(fname, "r");
if (fp == (FILE *)NULL)
    return( -1 );
init_record(fp, (char *)NULL, comment_char);
recp = get_record(fp, (char *)EOF, &rec_read, &ctl_codes);
fclose(fp);
if (recp == (char *)NULL)
    return( 0 );
/*
 *  get the information from the record
 */
init_scl();		/* initialize the string classes */
dc_tail = an_tail = (struct change_list *)NULL;
seen_textin = seen_rules = seen_lexicon = 0;
for ( rp = recp ; (rp != (char *)NULL) && (*rp != EOR) ; )
    {
    code = *rp++;		/* grab the table code */
    rp = skipwhite(rp);         /* skip following whitespace */
    switch (code)
	{
	case 'T':		/* \textin - text input control file */
	    end = rp + strlen(rp);
	    if (seen_textin)
		{
		printf(
	"%sText input control file already set - ignoring this \\textin field",
		       errhead );
		}
	    else if (*rp == NUL)
		printf("%sEmpty text input control file (\\textin) field",
		       errhead );
	    else
		{
		rp = strtok(rp, (char *)whiteSpc);
		if (rp != (char *)NULL)
		    {
		    txtin_file = mystrdup(rp);
		    ++seen_textin;
		    }
		}
	    rp = end;
	    break;

	case 'R':		/* \rules - PC-KIMMO rules file */
	    end = rp + strlen(rp);
	    if (seen_rules)
		{
		printf(
		   "%sRules file already set - ignoring this \\rules field",
		       errhead );
		}
	    else if (*rp == NUL)
		printf("%sEmpty rules file (\\rules) field",
		       errhead );
	    else
		{
		rp = strtok(rp, (char *)whiteSpc);
		if (rp != (char *)NULL)
		    {
		    rule_file = mystrdup(rp);
		    ++seen_rules;
		    }
		}
	    rp = end;
	    break;

	case 'L':		/* \lexicon - PC-KIMMO lexicon file */
	    end = rp + strlen(rp);
	    if (seen_lexicon)
		{
		printf(
		   "%sRules file already set - ignoring this \\lexicon field",
		       errhead );
		}
	    else if (*rp == NUL)
		printf("%sEmpty lexicon file (\\lexicon) field",
		       errhead );
	    else
		{
		rp = strtok(rp, (char *)whiteSpc);
		if (rp != (char *)NULL)
		    {
		    lex_file = mystrdup(rp);
		    ++seen_lexicon;
		    }
		}
	    rp = end;
	    break;

	case 'S':               /* \scl - string class definition */
	    end = rp + strlen(rp);
	    add_scl(rp);	/* load the string class definition */
	    rp = end;
	    break;

	case 'A':		/* \ach - analysis field change */
	    end = rp + strlen(rp);
	    cc = parse_change(rp);
	    if (cc != (struct change_list *)NULL)
		{               /* link change to end of list */
		if (an_chg == (struct change_list *)NULL)
		    an_chg = cc;
		else
		    an_tail->cl_link = cc;
		an_tail = cc;
		}
	    rp = end;
	    break;

	case 'D':		/* \dch - decomposition field change */
	    end = rp + strlen(rp);
	    cc = parse_change(rp);
	    if (cc != (struct change_list *)NULL)
		{               /* link change to end of list */
		if (dc_chg == (struct change_list *)NULL)
		    dc_chg = cc;
		else
		    dc_tail->cl_link = cc;
		dc_tail = cc;
		}
	    rp = end;
	    break;

	} /* end switch */
    while (*rp++ != NUL)	/* skip rest of this entry in the record */
	;
    } /* end for */

free_record(recp);
return( 0 );
}

/****************************************************************************
 * NAME
 *    initialize
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Load the control files.
 * RETURN VALUE
 *    none
 */
static void initialize()
{
/*
 *  load the control file
 */
ctl_file = set_filename( ctl_file, "ktext.ctl", ".ctl");
if (load_ctl_file(ctl_file) < 0)
    {
    fprintf(stderr, "Cannot load control file %s\n", ctl_file);
    exit(2);
    }
/*
 *  load the (optional) text input control file
 */
if (txtin_file != (char *)NULL)
    {
    load_intx_ctl_file( txtin_file );
    fputc('\n', stderr);
    }
/*
 *  load the rules
 */
if (rule_file == (char *)NULL)
    {
    fprintf(stderr, "KTEXT: Rules file not specified in control file\n");
    exit(2);
    }
else if (load_rules((unsigned char *)rule_file,&Lang,comment_char) < 0)
    {
    fprintf(stderr, "KTEXT: Cannot load rules file %s\n", rule_file);
    exit(2);
    }
/*
 *  load the lexicon
 */
if (lex_file == (char *)NULL)
    {
    fprintf(stderr, "KTEXT: Lexicon file not specified in control file\n");
    exit(2);
    }
else if (load_lexicons((unsigned char *)lex_file,&Lang,comment_char) < 0)
    {
    fprintf(stderr, "KTEXT: Cannot load lexicon file %s\n", lex_file);
    exit(2);
    }
/*
 *  if requested, open the log file
 */
if (log_file != (char *)NULL)
    {
    Log_fp = fopen(log_file, "w");
    if (Log_fp == (FILE *)NULL)
	fprintf(stderr,
		"KTEXT: Warning -- cannot open log file %s\n", log_file );
    }
}

/****************************************************************************
 * NAME
 *    duplicate_analysis
 * ARGUMENTS
 *    resp  - pointer to PC-KIMMO RESULT structure
 *    wordp - pointer to AMPLE/STAMP word template structure
 * DESCRIPTION
 *    Check whether or not this PC-KIMMO result has been stored yet.
 * RETURN VALUE
 *    zero if result isn't yet on the list, nonzero if it is
 */
static int duplicate_analysis(resp, wordp)
RESULT *resp;
struct word_template *wordp;
{
register struct strlist *anp;
register struct strlist *dcp;

for (	anp = wordp->anlist, dcp = wordp->dclist ;
	anp && dcp ;
	anp = anp->slink, dcp = dcp->slink )
    {
    if (    (strcmp((char *)resp->feat, anp->stri) == 0) &&
	    (strcmp((char *)resp->str,  dcp->stri) == 0) )
	return( 1 );		/* aha! found an identical analysis! */
    }
return( 0 );			/* this analysis is uniquely original */
}

/****************************************************************************
 * NAME
 *    process
 * ARGUMENTS
 *    none
 * DESCRIPTION
 *    Process the input file to produce the output file.
 * RETURN VALUE
 *    none
 */
static void process()
{
FILE *infp;
FILE *outfp;
register struct word_template *wtp;
register RESULT *resp;
RESULT *rp;
struct strlist *anp;
struct strlist *dcp;
int k;
int char_count;
char *p;
/*
 *  open the files
 */
infp  = ufopen(in_file, "r");
outfp = ufopen(out_file, "w");
char_count = 0;
while ((wtp = textin(infp)) != (struct word_template *)NULL)
    {
    if (wtp->word != (char *)NULL)
	{
	resp = recognizer((unsigned char *)wtp->word, &Lang,
			  Limit_flag, Trace_flag, Log_fp );
	/*
	 *  convert the RESULT list for output via dtbout()
	 */
	if (resp != (RESULT *)NULL)
	    {
	    anp = (struct strlist *)myalloc(sizeof(struct strlist));
	    anp->stri = (char *)resp->feat;
	    wtp->anlist = anp;
	    dcp = (struct strlist *)myalloc(sizeof(struct strlist));
	    dcp->stri = (char *)resp->str;
	    wtp->dclist = dcp;
	    for ( rp = resp->link, myfree(resp) ; rp ; rp = resp )
		{
		if (duplicate_analysis(rp, wtp))
		    {
		    myfree(rp->feat);
		    myfree(rp->str);
		    }
		else
		    {
		    anp->slink = (struct strlist *)
					myalloc(sizeof(struct strlist));
		    anp->slink->stri = (char *)rp->feat;
		    anp = anp->slink;
		    dcp->slink = (struct strlist *)
					myalloc(sizeof(struct strlist));
		    dcp->slink->stri = (char *)rp->str;
		    dcp = dcp->slink;
		    }
		resp = rp->link;
		myfree( (char *)rp );
		}
	    if (char_count == 79)
	        {
		fputc('\n', stderr);
		char_count = 0;
		}
	    fputc('.', stderr);
#ifdef THINK_C
	    fflush(stderr);    
#endif
	    ++char_count;
	    }
	else
	    {
	    if (char_count)
		fputc('\n', stderr);
	    fprintf(stderr, "AF: %s [%s]\n", wtp->orig_word, wtp->word );
	    if (!isatty(fileno(stdout)))
		printf("AF: %s [%s]\n", wtp->orig_word, wtp->word );
	    if (Log_fp != (FILE *)NULL)
		fprintf(Log_fp, "AF: %s [%s]\n", wtp->orig_word, wtp->word );
	    char_count = 0;
	    }
	}
    /*
     *  apply any desired changes to the analysis and decomposition strings
     */
    if (an_chg)
	{
	for ( anp = wtp->anlist ; anp ; anp = anp->slink )
	    {
	    p = apply_cc(anp->stri, an_chg);
	    myfree(anp->stri);
	    anp->stri = p;
	    }
	}
    if (dc_chg)
	{
	for ( dcp = wtp->dclist ; dcp ; dcp = dcp->slink )
	    {
	    p = apply_cc(dcp->stri, dc_chg);
	    myfree(dcp->stri);
	    dcp->stri = p;
	    }
	}
    dtbout( wtp, outfp, out_file, do_orig, ambigchar );
    /*
     *  free the memory allocated for this word
     */
    if (wtp->orig_word != (char *)NULL)
	myfree( wtp->orig_word );
    if (wtp->word != (char *)NULL)
	myfree( wtp->word );
    if (wtp->format != (char *)NULL)
	myfree( wtp->format );
    if (wtp->non_alpha != (char *)NULL)
	myfree( wtp->non_alpha );
    if (wtp->anlist != (struct strlist *)NULL)
	free_strlist(wtp->anlist);
    if (wtp->dclist != (struct strlist *)NULL)
	free_strlist(wtp->dclist);
    myfree( (char *)wtp );
    }
#ifndef MSDOS
if (char_count)
    fputc('\n', stderr);
#endif
}

/****************************************************************************
 * NAME
 *    main
 * ARGUMENTS
 *    argc - number of command line arguments
 *    argv - pointer to array of command line arguments
 * DESCRIPTION
 *    main() for PC-KIMMO -- where the program starts executing
 * RETURN VALUE
 *    status for return: 0 if okay, nonzero if error
 */
int main( argc, argv )
int argc;
char **argv;
{
fprintf(stderr,"\
KTEXT TWO-LEVEL PROCESSOR\n\
Version %d.%d.%d (%s %s), Copyright %s SIL\n",
			VERSION, REVISION, PATCHLEVEL, DATE, YEAR, YEAR );

#ifdef DJGPP
fprintf(stderr,"Beta version for 386 CPU under MS-DOS (VCPI compliant)\n");
#endif

#ifdef THINK_C
SetWTitle( FrontWindow(), "\pKTEXT");
argc = ccommand(&argv);
#endif

parse_cmd_line(argc,argv);	/* parse the command line */
initialize();			/* load the control files */
process();
return(0);
}

/***************************************************************************
 * NAME
 *    myshrink
 * ARGUMENTS
 *    s - pointer to string in overlarge allocated buffer
 * DESCRIPTION
 *    Shrink the allocated buffer to exactly fit the string.  Abort the
 *    program with an error message if we somehow run out of memory.
 * RETURN VALUE
 *    pointer to reallocated block
 */
char *myshrink(s)
char *s;
{
register char *p;

if (s == (char *)NULL)
    return((char *)NULL);
p = myrealloc( s, (unsigned)strlen(s)+1 );
return( p );
}
