;;
;;
;;      Title : DebugParse
;;
;;      Function : Debugger for the chart parser. 
;;
;;      Author : Alan W Black   February 1987
;;               Dept of A.I. University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Description :
;;        This is an interactive debugger that allows the user
;;        to examine the results of a parse. Basically it allows
;;        users to display edges and vertices selectively so they
;;        may anlyse the results of a parse (failed or otherwise).
;;
;;        It is assumed that the string can be correctly segmented
;;        that is this only describes the complete segmented strings
;;        rather than those that can be partially segemented (i.e.
;;        the edge for pre- in preach will not be shown.  If the user
;;        wants lower level infomation they can use the spelling rule
;;        debugger.
;;
;;        This takes the chart result from a parse and allows the user
;;        to ask about it.  This function is intended only to be
;;        used from the command interpreter.
;;
;;        Note a special class of edge in the chart can be also be displayed
;;        that is failed edges, which are complete edges that have failed
;;        the feature passing conventions.  They are probably of interested
;;        to people checking failed parses.
;;
;;
;;      Parameters :
;;      Returns :
;;      Side Effect :
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;
;;


(declare
   (special
      D-ALLEDGES
      D-ALLVERTICES
      D-LOADEDPARTS
      D-LOOKUPFORMAT
      D-GENNUMSUFFIX   ;; for holding a global number when naming edges etc
   )
   (localf
      D-NameEdgesAndVertices          
      D-DisplayResultOfParse          
      D-DisplayEdge                   
      D-DisplayVertex                 
      D-ShowVertex                    
      D-ShowVertexII                  
      D-ShowVertexCO                  
      D-IsAnEdge                      
      D-SearchForEdge                 
      D-IsAVertex                     
      D-SearchForVertex               
      D-ShowChart                     
      D-ShowEdge                      
   )
)

(include "macros")
(include "keywords")
(include "subrout")
(include "readatom")

(defun D-DebugParse (word)
;;
;;  allows the user to query the resulting chart from parsing this
;;  word
;;
   (cond
      ((not (equal (length D-LOADEDPARTS) 3))
	 (error "Lexicon, grammar, or spelling rules not loaded")
      )
      (t
	 (let ( (oldformat D-LOOKUPFORMAT) 
		parseresult )
	    (setq D-LOOKUPFORMAT 'D-DEBUG)
	    (setq parseresult (D-LookUp word))
	    (setq D-LOOKUPFORMAT oldformat)
	    (D-NameEdgesAndVertices)
            (D-DisplayResultOfParse
	       word
	       (car parseresult)    ;; initial vertex
	       (cddr parseresult)   ;; edges list
            )
         )
      )
   )
)

(defun D-NameEdgesAndVertices ()
;;
;;  Name the vertices and the edges with numbers
;;
   (setq D-GENNUMSUFFIX (length D-ALLEDGES))
   (mapcar
      #'(lambda (edge)
	 (D-putedgeNAME edge  D-GENNUMSUFFIX)
	 (setq D-GENNUMSUFFIX (- D-GENNUMSUFFIX 1)))
      D-ALLEDGES)
   
   (setq D-GENNUMSUFFIX (length D-ALLVERTICES))
   (mapcar
      #'(lambda (vertex)
	 (D-putvertexNAME vertex D-GENNUMSUFFIX)
	 (setq D-GENNUMSUFFIX (- D-GENNUMSUFFIX 1)))
      D-ALLVERTICES)
)

(defun D-DisplayResultOfParse (word initvertex edges)
;;
;;  Goes into a sub command loop displaying edges etc as the
;;  user requests
;;
   (terpri)
   (princ "Parse debugger for the word: ") (princ word)
   (terpri)
   (D-DisplayEdgesStructure edges)
   (let ( (command '(INIT)) )
      (D-while (not (equal command 'EXIT))
	 (princ "debug> ")   (drain) ;; prompt
	 (setq command (D-Command)) ;; get command 
	 (cond
	    ((null command)       ;; nothing typed in
	       nil     ;; do nothing
            )
	    ((or (eq command 'EOF) 
		 (memq (car command) (DK-COM-EXITS))) ;; exit from analyser
               (terpri)
	       (princ "exit from debugger")
	       (terpri)
	       (setq command 'EXIT)
            )
	    ((memq (car command) (DK-COM-HELPS))  ;; display help
	       (D-ShowDebugHelp)
            )
	    ((equal (car command) (DK-DEBUG-DISPLAY-EDGE)) ;; display edge
	               (D-DisplayEdge (cdr command))
            )
	    ((equal (car command) (DK-DEBUG-DISPLAY-VERTEX))
	               (D-DisplayVertex (cdr command))
            )
	    ((equal (car command) (DK-DEBUG-DISPLAY-CHART))
	       (D-DisplayEdgesStructure edges))
	    (t     
	       (princ "ERROR >>>> input unknown")
	       (terpri)
	       (princ "enter h for help")
	       (terpri)
            )
         )
      )
   )
)

(defun D-DisplayEdgesStructure (edges)
;;
;;  Displays the edges and vertex structure of the chart
;;
   (princ "Number of edges: ") (princ (length D-ALLEDGES))
   (terpri)
   (princ "Number of vertices ") (princ (length D-ALLVERTICES))
   (terpri)
)


(defun D-DisplayEdge (params)
;;
;;  Displays the given edge name
;;
   (cond
      (params 
	 (let ((edge (D-IsAnEdge (car params))))
	    (cond
	       ((null edge)
		  (princ (car params)) (princ " not a valid edge")
		  (terpri))
               (t (D-ShowEdge edge))))
      )
      (t
	 (princ "No edge name given") (terpri)
      )
   )
)

(defun D-DisplayVertex (params)
;;
;;  Displays the given vertex name
;;
   (cond
      ((and params (memq (car params) (DK-DEBUG-VTYPE)))
	 (cond
	    ((eq (length params) 1)
	       (princ "No vertex names given") (terpri))
            ((eq (car params) (DK-DEBUG-VII))
	       (mapcar
		  #'D-ShowVertexII
		  (cdr params)))  ;; list of vertices
            (t    ;; complete outgoing
	       (mapcar
		  #'D-ShowVertexCO
		  (cdr params)))))
      (params
	 (mapcar
	    #'D-ShowVertex
	    params)    ;; list of vertex names
      )
      (t
	 (princ "No vertex name given") (terpri)
      )
   )
)

(defun D-ShowVertex (vertexname)
;;
;;  prints out a vertex 
;;
   (let ( (vertex (D-IsAVertex vertexname)) )
      (cond
	 ((null vertex)
	    (princ vertexname) (princ " is not a valid vertex name")
	    (terpri))
         (t
	    (princ "Vertex ") (princ vertexname)
	    (cond
	       ((eq 'END (D-getvertexSTATUS vertex))
		  (princ ": no remaining string"))
               (t
		  (princ ": remaining surface string ")
	          (D-PList (cdr (D-getvertexSTATUS vertex)))))
            (terpri)
	    (D-ShowVertexII vertexname)
	    (D-ShowVertexCO vertexname)
         )
      )
   )
)

(defun D-ShowVertexII (vertexname)
;;
;;  Displays the list of edges that are incomplete and end at this
;;  vertex
;;
   (let ( (vertex (D-IsAVertex vertexname)) )
      (cond
	 ((null vertex)
	    (princ vertexname) (princ " is not a valid vertex name")
	    (terpri))
         ((null (D-getvertexEDGEINI vertex))
	    (princ "No incoming incomplete edges to vertex ")
	    (princ vertexname) (terpri))
         (t
	    (princ "Incoming incomplete edges to vertex ")
	    (princ vertexname) (princ ":") (terpri)
	    (mapcar #'D-ShowEdge (D-getvertexEDGEINI vertex))
         )
      )
   )
)

(defun D-ShowVertexCO (vertexname)
;;
;;  Displays the list of edges that are complete and start at this 
;;  vertex
;;
   (let ( (vertex (D-IsAVertex vertexname)) )
      (cond
	 ((null vertex)
	    (princ vertexname) (princ " is not a valid vertex name")
	    (terpri))
         ((null (D-getvertexEDGEOUTC vertex))
	    (princ "No outgoing complete edges from vertex ")
	    (princ vertexname) (princ ":") (terpri))
         (t
	    (princ "Outgoing complete edges from vertex ")
	    (princ vertexname) (princ ":") (terpri)
	    (mapcar #'D-ShowEdge (D-getvertexEDGEOUTC vertex))
         )
      )
   )
)

(defun D-IsAnEdge (edge)
;;
;;  checks to see if the given name is an edge returns the edge
;;  if it exists, null otherwise
;;
   (D-SearchForEdge
      edge D-ALLEDGES)
)

(defun D-SearchForEdge (edgename edges)
;;
;;  searches the list of edges for the one with that name
;;
   (cond
      ((null edges) nil)
      ((string-equal (string edgename)
	       (format nil "~s" (D-getedgeNAME (car edges))))
	 (car edges))
      (t (D-SearchForEdge edgename (cdr edges)))
   )
)

(defun D-IsAVertex (vertex)
;;
;;  checks to see if the given name is a vertex returns the vertex
;;  if it exists, null otherwise
;;
   (D-SearchForVertex
      vertex D-ALLVERTICES)
)

(defun D-SearchForVertex (vertexname vertices)
;;
;;  searches the list of vertices for the one with that name
;;
   (cond
      ((null vertices) nil)
      ((string-equal (string vertexname)
	       (format nil "~s" (D-getvertexNAME (car vertices))))
	 (car vertices))
      (t (D-SearchForVertex vertexname (cdr vertices)))
   )
)

(defun D-ShowChart (edges)
;;
;;   Debug code to display the whole chart
;;
   (mapcar
      #'D-ShowEdge
      edges)
)

(defun D-ShowEdge (edge)
;;
;;  This expands an edges to properties to a list
;;
   (princ "Edge ") (princ (D-getedgeNAME edge))
   (princ " from vertex ")
   (princ (D-getvertexNAME (D-getedgeSTART edge)))
   (princ " to vertex ")
   (princ (D-getvertexNAME (D-getedgeEND edge)))
   (cond
      ((D-LexicalEdgeP edge)
	 (terpri)
	 (princ "Lexical Entry: ") (princ (D-getedgeRULENUM edge)))
      (t
         (princ " Rule ")
         (princ (D-getedgeRULENUM edge))))
   (terpri)
   (cond
      ((eq (D-getedgeLABEL edge) 'FAILED)
	 (princ "Failed feature passing conventions"))
      (t
         (princ "Label ")
         (princ (D-getedgeLABEL edge))))
   (terpri)
   (cond
      ((null (D-LexicalEdgeP edge))
	 (princ "Remainder ")
	 (princ (D-getedgeREMAINDER edge))
	 (terpri)
	 (princ "Recognised edges are ")
	 (D-PList
	    (mapcar
	       #'(lambda (daughter)
		     (D-getedgeNAME (cadr daughter)))
	       (D-getedgeRECOG edge)))
	 (terpri)))
   (princ "Bindings ")
   (princ (D-getedgeBIND edge))
   (terpri)
)

(defun D-ShowDebugHelp ()
;;
;;  displays help describing the possible debug commands
;;
   (terpri)
   (princ "h  or ? : help")
   (terpri)
   (princ "e q exit quit or EOF : return to main level ")
   (terpri)
   (princ "dv vertexnumber1 [vertexnumber2 ...]  :  display vertices")
   (terpri)
   (princ "dv ii vn1 [vn2 ...] : display incoming incomplete edges to named")
   (princ " vertices") 
   (terpri)
   (princ "dv co vn1 [vn2 ...] : display outgoing complete edges to named")
   (princ " vertices") 
   (terpri)
   (princ "de edgenumber1 [edgenumber2 ...]  :  display edges")
   (terpri)
   (princ "dc : display number of edges and vertices")
   (terpri)
   (terpri)
)

