; FILE:       graphics.l
; PURPOSE:    Lucid graphics for displaying networks (sg), line graphs (lg)
;             and real-time activation graphs (ag).
; PROGRAMMER: Greg Nelson
; CREATED:    7-20-87
; UPDATED:    5-26-88
; UPDATED:    7-22-88 - made DEVOWEL a bit cleaner
; UPDATED:    9-6-88 - additions to function grapher (file labels, ELIMINATE)
; UPDATED:    9-7-88 - corrections to fg file reading functions
; UPDATED:    10-12-88 - fixed the spidergrapher for the new link format

;; Load in the graphics library
(load "/usr/local/lang/lisp-4-0/windows-x")

;(in-package 'user)

; General global variables
(defvar *grapher-viewport* nil "Pointer to the grapher viewport.")
(defvar *grapher-bitmap* nil "Pointer to the grapher bitmap.")
(defvar *grapher-font* nil "Name of the grapher font.")
(defvar *last-init* nil "Name of grapher last initialized.")
(defvar *debug* nil "Turn on the debugger.")
(defvar *winx* nil "Window width.")
(defvar *winy* nil "Window height.")
(defvar *winxs* nil "Window starting x position.")
(defvar *winys* nil "Window starting y position.")
(defvar *devowel-length* 30 "Length beyond which vowels will be taken out.")
(defvar *time-step* 0 "Added by Paul.")
(defvar *acme-mode* nil "This is defined elsewhere.")
(defvar default_activation .01 "This is defined elsewhere.")
(defvar when_to_print nil "This is defined elsewhere.")

; Global variables specific to network grapher
(defvar *unit-radius* nil "Default size for unit circles.")
(defvar *horiz-spc* nil "X distance between adjacent units.")
(defvar *vert-spc* nil "Y distance between adjacent units.")
(defvar *drawing-mode* 'circle "Determines whether to draw circle or squares.")
(defvar *x-radius* nil "X radius of the circle on which units are plotted.")
(defvar *y-radius* nil "Y radius of the circle on which units are plotted.")
(defvar *second-label* 'actstring "Put activation as a second label at each node.")
(defvar *link-labels* 'weight-link-label
  "The function to print as label on links, when non-nil.")
(defvar *link-width* 'weight-link-width
  "The function which determines widths, when non-nil.")
(defvar *known-functions* nil
  "Variable listing which functions can be graphed.")

; Global variables specific to line grapher
(defvar *x-reg* nil "Most recent x location referenced.")
(defvar *y-reg* nil "Most recent y location referenced.")
(defvar *trace-list* nil "List of units to trace.")
(defvar *var-list* nil "List of temporary variables to store traces.")

; Global variables specific to activation grapher
(defvar *act-type* nil "Type of activation graph: 'circle or 'line")
(defvar *unit-list* nil "List of units currently being graphed.")
(defvar *act-list* nil "Previous activations of current units.")
(defvar *graphs-per-line* 4 "Number of miniature graphs to draw on each line.")

(defun dumpwin (file)
  (shell (format nil "screendump -x ~a -y ~a -X ~a -Y ~a ~a"
		 *winxs* *winys* *winx* *winy* file))
)

; High level startup calls
(defun sg (unit)
  (start-graph 'network)
  (setq *second-label* 'actstring)
  (setq *link-labels* 'weight-link-label)
  (setq *link-width* 'weight-link-width)
  (run-mouse unit)
)
(defun lg () (start-graph 'line nil 700 700))
(defun ag (&optional (at 'line)) (start-graph 'act) (setq *act-type* at))
(defun fg (&optional node file)
  (prog (new)
    (if (and (null file) (null *known-functions*))
	(return "No files have been specified.")
    )
    (start-graph 'network)
    (cond (file
	   (terpri)
	   (princ "Loading file...")
	   (terpri)
	   (construct-children (setq new (calltree file)) file)
	   (construct-parents new)
	  )
    )
    (setq *link-labels* nil)
    (setq *second-label* 'source-file)
    (setq *link-width* 'func-hier-width)
    (if (member node *known-functions*)
	(run-mouse node 'func-child)
        (return "Unknown function.")
    )
  )
)

(defun fhier (&optional node file)
  "Call function hierarchy commands without grapher."
  (prog (new)
    (if (and (null file) (null *known-functions*))
	(return "No files have been specified.")
    )
    (cond (file
	   (terpri)
	   (princ "Loading file...")
	   (terpri)
	   (construct-children (setq new (calltree file)) file)
	   (construct-parents new)
	  )
    )
    (if (member node *known-functions*) (tell node ": " (get node 'children)))
  )
)

; Grapher initialization function
(defun start-graph (&optional (gmode nil) (restart nil)
                              (hei 900) (wid 700) (xloc 450) (yloc 0))
  "Function to open a window for the grapher and set the global variables."
  (cond ((or (null *last-init*) (not (null restart)))
;	 (leave-window-system)
;  Took this out so it doesn't crash the first time.
         (initialize-windows :height hei :width wid
			     :screen-x xloc :screen-y yloc)
; But I think this will make it crash the second, so better not run
; (ag) or (sg) more than once.  If it does, we have to wait for the
; manuals.
	 (setq *winx* wid) (setq *winy* hei)
	 (setq *winxs* xloc) (setq *winys* yloc)
	 (setq *grapher-viewport* (make-viewport :width wid :height hei))
	 (setq *grapher-bitmap* (viewport-bitmap *grapher-viewport*))
	 (setq *grapher-font* "ITALIC")
	)
  )
  (cond ((equal gmode 'network)
	 (setq *unit-radius* 5)
	 (setq *horiz-spc* (round (/ wid 5)))
	 (setq *vert-spc* 45)
	 (setq *x-radius* (round (/ wid 3.5)))
	 (setq *y-radius* (round (/ hei 2.5)))
       )
  )
  (setq *last-init* gmode)
)

;  Network grapher (sg)

(defun run-mouse (unit &optional (child-function 'kidsfunc))
  "Function to start the mouse driven browser starting at a given unit."
  (let (mx my mb cx cy actunit actx acty (centunit unit)
	radius (rdrw t) nu ou omx omy (pc 0)
       )
    (loop
      (setq actx (round (/ (bitmap-width *grapher-bitmap*) 2.0)))
      (setq acty (round (/ (bitmap-height *grapher-bitmap*) 2.0)))
      (setq cx actx) (setq cy acty)
      (if rdrw (draw-around (setq actunit centunit) child-function))
      (setq rdrw nil)
      (setq mx (mouse-x)) (setq my (mouse-y))
      (cond ((and (equal omx mx) (equal omy my))
	     (setq pc (1+ pc))
	     (if (> pc 100)
		 (loop
		   (setq mx (mouse-x)) (setq my (mouse-y))
		   (if (and (equal omx mx) (equal omy my)) (sleep 5) (return))
		 )
	     )
	    )
	    (t (setq pc 0) (setq omx mx) (setq omy my))
      )
      (setq mb (mouse-buttons))
      (setq radius (+ (expt (- mx cx) 2) (expt (- my cy) 2)))
      (setq nu (nearunit centunit mx my child-function))
      (setq actunit (elt (funcall child-function centunit) nu))
      (setq actx (x-rel centunit nu child-function))
      (setq acty (y-rel centunit nu child-function))
      (if (< radius 400) (prog () (setq actunit centunit)
                                 (setq actx cx)
                                 (setq acty cy)))
      (draw-circle *grapher-bitmap* (make-position actx acty) 3 :width 2
                   :operation boole-set)
      (if (not (equal ou nu)) (draw-unit actunit actx acty))
      (draw-circle *grapher-bitmap* (make-position actx acty) 3 :width 2
                   :operation boole-clr)
      (setq ou nu)
      (if (equal mb 4)
          (if (equal actunit centunit)
              (return-from run-mouse) (setq rdrw (setq centunit actunit)))
          (if (equal mb 2)
              (setq rdrw (if (equal *drawing-mode* 'circle)
                             (setq *drawing-mode* 'square)
                             (setq *drawing-mode* 'circle)
                         )
              )
          )
      )
    )
  )
)

(defun nearunit (unit x y &optional child-function)
  "Return position in list of unit."
  (if (equal *drawing-mode* 'circle) 
    (let (adj-lis temp angle delx dely)
      (setq delx (- x (round (/ (bitmap-width *grapher-bitmap*) 2.0))))
      (setq dely (- y (round (/ (bitmap-height *grapher-bitmap*) 2.0))))
      (setq adj-lis (funcall child-function unit))
      (setq angle (/ (* 2 pi) (length adj-lis)))
      (setq temp (atan (/ dely *y-radius*) (/ delx *x-radius*)))
      (if (< temp 0) (setq temp (+ temp (* 2 pi))))
      (setq temp (round (/ temp angle)))
      (if (< temp (length adj-lis)) temp 0)
    )  
    (let (temp col cc delt ncol percol)
      (setq cc (child-count unit child-function))
      (setq delt (round (/ (bitmap-height *grapher-bitmap*)
                           (if (< cc 16) (+ 2 cc) (+ 5 cc)))))
      (setq ncol 1) (setq percol cc)
      (if (< delt *vert-spc*) (prog () (setq delt (* delt 2))
				       (setq percol (round (/ percol 2.0)))
				       (setq ncol (* ncol 2))))
      (if (< delt *vert-spc*) (prog () (setq delt (round (* delt 2)))
	                               (setq percol (round (/ percol 2.0)))
				       (setq ncol (* ncol 2))))
      (if (equal ncol 4) 
        (prog ()
          (setq col 4)
          (if (<= x (* 3.5 *horiz-spc*)) (setq col 1))
          (if (<= x (* 2.5 *horiz-spc*)) (setq col 2))
          (if (<= x (* 1.5 *horiz-spc*)) (setq col 3))
        )
        (if (and (equal ncol 2) (<= x (* 2.5 *horiz-spc*)))
            (setq col 2) (setq col 1))
      )
      (setq temp (round (- (/ y delt)
                           (if (and (oddp col) (equal ncol 4)) 1 0.5))))
      (if (>= temp percol) (setq temp (1- percol)))
      (if (< temp 0) (setq temp 0))
      (round (+ (* (1- col) percol) temp))
    )
  )
)

(defun child-count (unit &optional child-function)
  "Function to return the number of connections to a given unit."
  (length (funcall child-function unit))
)

(defun x-rel (unit num &optional child-function)
  "Calculate x position of a given unit."
  (if (equal *drawing-mode* 'circle) 
    (let (adj-lis angle xcent)
      (setq xcent (round (/ (bitmap-width *grapher-bitmap*) 2.0)))
      (setq adj-lis (funcall child-function unit))
      (setq angle (* num (/ (* 2 pi) (length adj-lis))))
      (round (+ xcent (* *x-radius* (cos angle))))
    )
    (let (cc percol delt col ncol)
      (setq cc (child-count unit child-function))
      (setq delt (round (/ (bitmap-height *grapher-bitmap*)
                           (if (< cc 16) (+ 2 cc) (+ 5 cc)))))
      (setq ncol 1) (setq percol cc)
      (if (< delt *vert-spc*) (prog () (setq delt (* delt 2))
				       (setq percol (round (/ percol 2.0)))
				       (setq ncol (* ncol 2))))
      (if (< delt *vert-spc*) (prog () (setq delt (round (* delt 2)))
				       (setq percol (round (/ percol 2.0)))
				       (setq ncol (* ncol 2))))
      (setq col (- ncol (floor (/ num percol))))
      (cond ((equal col 1) (* 4 *horiz-spc*))
            ((equal col 2) *horiz-spc*)
            ((equal col 3) (* 2 *horiz-spc*))
            ((equal col 4) (* 3 *horiz-spc*)))
    )
  )
)

(defun y-rel (unit num &optional child-function)
  "Calculate y position of a given unit."
  (if (equal *drawing-mode* 'circle)
    (let (adj-lis angle ycent)
      (setq ycent (round (/ (bitmap-height *grapher-bitmap*) 2.0)))
      (setq adj-lis (funcall child-function unit))
      (setq angle (* num (/ (* 2 pi) (length adj-lis))))
      (round (+ ycent (* *y-radius* (sin angle))))
    )
    (let (cc temp delt ncol percol)
      (setq cc (child-count unit child-function))
      (setq delt (round (/ (bitmap-height *grapher-bitmap*)
                           (if (< cc 16) (+ 2 cc) (+ 5 cc)))))
      (setq ncol 1) (setq percol cc)
      (if (< delt *vert-spc*) (prog () (setq delt (* delt 2))
				       (setq percol (round (/ percol 2.0)))
				       (setq ncol (* ncol 2))))
      (if (< delt *vert-spc*) (prog () (setq delt (round (* delt 2)))
	                               (setq percol (round (/ percol 2.0)))
				       (setq ncol (* ncol 2))))
      (setq temp (- ncol (floor (/ num percol))))
      (round (* (+ (mod num percol)
		   (if (and (oddp temp) (equal ncol 4)) 0.5 1)
		) delt))
    )
  )
)
	
(defun draw-around (unit &optional (child-function 'kidsfunc)) 
  "Basic function to draw the connections to a given unit." 
  (let (adj-lis xcent ycent)
    (clear-bitmap *grapher-bitmap*)
    (setq xcent (round (/ (bitmap-width *grapher-bitmap*) 2.0)))
    (setq ycent (round (/ (bitmap-height *grapher-bitmap*) 2.0)))
    (setq adj-lis (funcall child-function unit))
    (do ((tlist adj-lis (cdr tlist))
	 (tunit nil nil)
	 (count 0 (1+ count))
	 (linkw nil nil)
	)
	((null tlist) nil)
	(setq tunit (car tlist))
	(setq linkw (if *link-width* (funcall *link-width* unit tunit) 1))
	(draw-line *grapher-bitmap*
		   (make-position xcent ycent)
		   (make-position  (x-rel unit count child-function)
				   (y-rel unit count child-function))
		   :width linkw
		   :operation boole-set
	)
    )
    (do ((tlist adj-lis (cdr tlist))
	 (tunit nil nil)
	 (count 0 (1+ count))
	 (xpt (x-rel unit 0 child-function)
	      (x-rel unit (1+ count) child-function)
	 )
	 (ypt (y-rel unit 0 child-function)
	      (y-rel unit (1+ count) child-function)
	 )
	 (word nil nil)
	 (str-pix-len nil nil)
	)
	((null tlist) nil)
        (setq tunit (car tlist))
        (draw-unit tunit xpt ypt)
        (cond (*link-labels*
	       (setq word (funcall *link-labels* unit tunit))
	       (setq str-pix-len (* (font-fixed-width *grapher-font*) 
				    (length word)
				 )
	       )
	       (stringblt *grapher-bitmap*
			  (make-position
			   (round (/ (- (+ xpt xcent) str-pix-len) 2.0))
			   (+ (round (/ (+ ypt ycent) 2.0)) 10)
			  )
			  *grapher-font* word
	       )
	      )
        )
    )
    (draw-unit unit xcent ycent)
  )
)

(defun kidsfunc (unit)
  "Default function to determine what is linked to a given unit (for ACME)."
  (mapcar #'car (get unit 'links_from))
)

; might want a different linetype for 0, but not now
(defun weight-link-width (unit1 unit2)
  "Function which determines the width of the line in various cases."
  (let ((linkw (weight_of_link_between unit1 unit2)))
    (cond ((< 0 linkw) 3)
	  ((> 0 linkw) 1)
	  ((equal 0 linkw) 1)
	  ((equal 0.0 linkw) 1)
    )
  )
)

(defun weight-link-label (unit1 unit2)
  "Function which give the label for the links (for ACME)."
  (princ-to-string (/ (round (* (weight_of_link_between unit1 unit2) 1000))
		      1000.0
		   )
  )
)

(defun actstring (unit)
  "Default function to determine the second label for a unit (ACME)."
  (princ-to-string (/ (round (* (act unit) 1000)) 1000.0))
)

(defun func-child (function)
  "Function to determine the children for the function grapher."
  (append (get function 'children) (get function 'parents))
)

(defun source-file (function)
  "Function to print a function's source file as a second label."
  (or (get function 'file) "")
)

(defun construct-children (child-list &optional source-file)
  "Function to note all of the child functions and update *known-functions*."
  (do ((funcs child-list (cddr funcs)))
      ((null funcs) nil)
      (setq *known-functions* (pushnew (car funcs) *known-functions*))
      (put (car funcs) 'children
	   (union (get (car funcs) 'children) (cadr funcs))
      )
      (put (car funcs) 'file source-file)
  )
)

(defun construct-parents (child-list)
  "Function to note all of the parent functions."
  (do ((funcs child-list (cddr funcs)))
      ((null funcs) nil)
; What this psychotic piece of code does is to put (car funcs) on the
; parents property of each member of (cadr funcs), or, in other words,
; to tell the children who their parents are.  Of course if the functions
; beget themselves, it is silly to tell them this, so the if prevents
; recursive functions from being listed as their own parents.
      (mapcar `(lambda (el)
		 (if (not (eq el (quote ,(car funcs))))
		     (put el 'parents
			  (pushnew (quote ,(car funcs)) (get el 'parents))
		     )
		 )
	       )
	      (cadr funcs)
      )
  )
)

(defun eliminate (function)
  (declare (special function))
  (remprop function 'children)
  (remprop function 'file)
  (mapcar '(lambda (func) (put func 'children (remove function (get func 'children)))) (get function 'parents))
  (remprop function 'parents)
)

(defun func-hier-width (function child-or-parent)
  "Function to display whether something is a parent or a child."
  (if (member child-or-parent (get function 'children)) 1 3)
  ; Width 1 if it is a child, otherwise width 3
)

(defun draw-unit (unit x y &optional (highlight nil))
  "Basic unit drawing function."
  (let (str-pix-len word)
    (debug-print (list "drawing circle for" unit))
    (draw-circle *grapher-bitmap* (make-position x y) 3 :width 3
                 :operation (if (null highlight) boole-clr boole-set))
    (draw-circle *grapher-bitmap* (make-position x y) *unit-radius*)
    (setq word (string unit))
    (if (> (length word) *devowel-length*) (setq word (devowel word)))
    (setq str-pix-len (* (font-fixed-width *grapher-font*) (length word)))
    (stringblt *grapher-bitmap*
               (make-position
                 (- x (round (/ str-pix-len 2.0)))
                 (- y (+ *unit-radius* 5)))
               *grapher-font* word)
    (cond (*second-label*
      (setq word (funcall *second-label* unit))
      (setq str-pix-len (* (font-fixed-width *grapher-font*) (length word)))
      (stringblt *grapher-bitmap* 
		 (make-position
		  (- x (round (/ str-pix-len 2.0)))
		  (+ y (+ *unit-radius* 15)))
		 *grapher-font* word)
          )
    )
    (debug-print (list "done drawing" unit))
  )
)

; Line grapher (lg)

; Special functions for acme, echo, etc.
(defun set-trace (lis)
  (setq *trace-list* lis)
  (setq *var-list*
	(do ((ol nil ol)
	     (a (gensym) (gensym))
	     (i (length lis) (1- i)))
	    ((zerop i) ol)
	    (set a nil)
	    (setq ol (append ol (list a)))
	)
  )
)

(defun update-trace ()
  (do ((a (cdr *trace-list*) (cdr a))
       (b (car *trace-list*) (car a))
       (c (cdr *var-list*) (cdr c))
       (d (car *var-list*) (car c))
      )
      ((null b) nil)
      (set d (append (symbol-value d) (list (get b 'activation))))
  )
)

(defun trace-plot ()
  (prog (yr (y-vecs (mapcar #'symbol-value *var-list*)) win-x-save word)
    (setq yr (range (apply #'append y-vecs)))
    (setq win-x-save *winx*)
    (setq *winx* (- *winx* 70))
    (draw-axes 1 (longest y-vecs) (car yr) (cadr yr))
    (do ((x-vec nil nil)
         (tlist (cdr y-vecs) (cdr tlist))
         (y-vec (car y-vecs) (car tlist))
	 (ulist (cdr *var-list*) (cdr ulist))
	 (utemp (car *var-list*) (car ulist))
        )
        ((null y-vec) nil)
	(setq x-vec (iota (length y-vec)))
        (linplot y-vec x-vec)
	(setq word (string utemp))
        (if (> (length word) *devowel-length*) (setq word (devowel word)))
	(stringblt *grapher-bitmap*
		   (make-position *x-reg* *y-reg*)
		   *grapher-font* word)
    )
    (setq *winx* win-x-save)
  )
)

(defun cleanup-trace ()
  (do ((a (cdr *var-list*) (cdr a))
       (b (car *var-list*) (car a))
      )
      ((null b) nil)
      (makunbound (symbol-value b))
  )
  (setq *var-list* nil) (setq *trace-list* nil)
)

; High order functions: lineplot, multiple lineplot, scatterplot, multiple sp
(defun lp (y-vec &optional x-vec)
  (prog (xr yr)
    (if (null x-vec) (setq x-vec (iota (length y-vec))))
    (cond ((not (equal (length x-vec) (length y-vec)))
	   (print "Unequal vectors: cannot graph.")
	   (return nil))
    ) 
    (setq xr (range x-vec)) 
    (setq yr (range y-vec)) 
    (draw-axes (car xr) (cadr xr) (car yr) (cadr yr))
    (linplot y-vec x-vec)
  )
)

(defun mlp (&rest y-vecs)
  (prog (yr)
    (setq yr (range (apply #'append y-vecs)))
    (draw-axes 1 (longest y-vecs) (car yr) (cadr yr))
    (do ((x-vec nil nil)
         (tlist (cdr y-vecs) (cdr tlist))
         (y-vec (car y-vecs) (car tlist))
        )
        ((null y-vec) nil)
	(setq x-vec (iota (length y-vec)))
        (linplot y-vec x-vec)
    )
  )
)

(defun sp (y-vec &optional x-vec)
  (prog (xr yr)
    (if (null x-vec) (setq x-vec (iota (length y-vec))))
    (cond ((not (equal (length x-vec) (length y-vec)))
	   (print "Unequal vectors: cannot graph.")
	   (return nil))
    ) 
    (setq xr (range x-vec)) 
    (setq yr (range y-vec)) 
    (draw-axes (car xr) (cadr xr) (car yr) (cadr yr))
    (scatplot y-vec x-vec)
  )
)

(defun msp (&rest y-vecs)
  (prog (yr)
    (setq yr (range (apply #'append y-vecs)))
    (draw-axes 1 (longest y-vecs) (car yr) (cadr yr))
    (do ((x-vec nil nil)
         (tlist (cdr y-vecs) (cdr tlist))
         (y-vec (car y-vecs) (car tlist))
        )
        ((null y-vec) nil)
	(setq x-vec (iota (length y-vec)))
        (scatplot y-vec x-vec)
    )
  )
)

; Bargraph absolute.
; (bargraph (y-vec x-start width &optional (y-start 0)))

(defun bga (y-vec)
  (prog (yr ym)
    (setq yr (range y-vec))
    (setq ym (if (< (car yr) 0) (car yr) 0))
    (draw-axes 0 (length y-vec) ym (cadr yr))
    (bargraph y-vec 0 1 ym)
  )
)

; Bargraph with counts.

(defun bgc (data &optional (binwidth 1))
  (declare (special binwidth))
  (prog (xr y-vec (halfwid (/ binwidth 2.0)))
    (declare (special halfwid))
    (setq xr (range data))
    (setq y-vec
	  (do ((rdata data)
	       (binend (+ (cadr xr) halfwid) (- binend binwidth))
	       (result nil)
	      )
	      ((< binend (car xr)) result)
	      (declare (special binend))
	      (setq result
		    (cons (count-if
			   '(lambda (el) (and (<= el binend)
					      (> el (- binend binwidth))
				         )
			    )
			   rdata
			  )
			  result
		    )
	      )
	      (setq rdata
		    (remove-if
		     '(lambda (el) (and (<= el binend)
					(> el (- binend binwidth))
				   )
		      )
		     rdata
		    )
	      )
	  )
    )
    (print y-vec)
    (draw-axes (- (car xr) halfwid) (+ (cadr xr) halfwid) 0 (cadr (range y-vec)))
    (bargraph y-vec (- (car xr) halfwid) binwidth 0)
  )
)

; Default definitions to make the compiler happier.
; (These map onto world coordinates (0,0)-(1,1))
(defun xscale (x)
  (+ 70 (round (* (- x 0.0) (- *winx* 100))))
)

(defun yscale (y)
  (- *winy* (round (* (- y 0.0) (- *winy* 80))) 60)
)

; This function draws the axes and redefines the scaling functions so the
; graphing is easier elsewhere.
(defun draw-axes (x-start x-end y-start y-end)
  "Function to draw axes for a lineplot or scatter plot."
  (clear-bitmap *grapher-bitmap*)
  (defun xscale (x) (+ 70 (round (* (/ (- x x-start) (- x-end x-start 0.0)) (- *winx* 100)))))
  (defun yscale (y) (- *winy* (round (* (/ (- y y-start) (- y-end y-start 0.0)) (- *winy* 80))) 60))
  (defun unxscale (x) (+ (* (/ (- x 70) (- *winx* 100.0)) (- x-end x-start)) x-start))
  (defun unyscale (y) (+ (* (/ (- *winy* y 60) (- *winy* 80.0)) (- y-end y-start)) y-start))

  (line-to x-start y-end 'set)
  (line-to x-start y-start)
  (line-to x-end y-start)

  (do ((count x-start (+ count (/ (- x-end x-start) 4.0)))
       (val nil nil)
       (str-pix-len nil nil)
      )
      ((> count x-end) nil)
      (setq val (princ-to-string (/ (round (* count 1000)) 1000.0)))
      (setq str-pix-len (* (font-fixed-width "ITALIC") (length val)))
      (stringblt *grapher-bitmap* 
                 (make-position 
                   (- (xscale count) (round (/ str-pix-len 2.0)))
                   (+ (yscale y-start) 15)
                 )
                 "ITALIC" val
      )
      (draw-line *grapher-bitmap*
                 (make-position (xscale count) (- (yscale y-start) 3))
                 (make-position (xscale count) (+ (yscale y-start) 3)))
  )
  (do ((count y-start (+ count (/ (- y-end y-start) 4.0)))
       (val nil nil)
       (str-pix-len nil nil)
      )
      ((> count y-end) nil)
      (setq val (princ-to-string (/ (round (* count 1000)) 1000.0)))
      (setq str-pix-len (* (font-fixed-width "ITALIC") (length val)))
      (stringblt *grapher-bitmap* 
                 (make-position 10 (+ (yscale count) 8))
                 "ITALIC" val
      )
      (draw-line *grapher-bitmap*
                 (make-position (- (xscale x-start) 3) (yscale count))
                 (make-position (+ (xscale x-start) 3) (yscale count)))
  )
)

(defun linplot (y-vec x-vec)
  (line-to (car x-vec) (car y-vec) 'set)
  (do ((xt (cddr x-vec) (cdr xt))
       (yt (cddr y-vec) (cdr yt))
       (xp (cadr x-vec) (car xt))
       (yp (cadr y-vec) (car yt)))
      ((null xp) nil)
      (line-to xp yp)
  )
)

(defun scatplot (y-vec x-vec)
  (do ((xt (cdr x-vec) (cdr xt))
       (yt (cdr y-vec) (cdr yt))
       (xp (car x-vec) (car xt))
       (yp (car y-vec) (car yt)))
      ((null xp) nil)
      (line-to xp yp 'point)
  )
)

(defun bargraph (y-vec x-start width &optional (y-start 0))
  (do ((xp x-start (+ width xp))
       (yt (cdr y-vec) (cdr yt))
       (yp (car y-vec) (car yt)))
      ((null yp) nil)
      (line-to xp y-start 'set)
      (line-to xp yp)
      (line-to (+ xp width) yp)
      (line-to (+ xp width) y-start)
      (line-to xp y-start)
  )
)

(defun line-to (x y &optional opts)
  (cond ((or (equal opts 'set) (equal opts 'point))
         (setq *x-reg* x) (setq *y-reg* y))
  )
  (cond ((equal opts 'point)
	 (let ((xt (xscale *x-reg*)) (yt (yscale *y-reg*)))
	   (draw-line *grapher-bitmap*
		      (make-position (1- xt) yt)
		      (make-position (1+ xt) yt)
	   )
	   (draw-line *grapher-bitmap*
		      (make-position xt (1- yt))
		      (make-position xt (1+ yt))
	   )
	 )
	)
	((not (equal opts 'set))
	 (draw-line *grapher-bitmap*
		    (make-position (xscale *x-reg*) (yscale *y-reg*))
		    (make-position (xscale x) (yscale y)))
	)
  )
  (cond ((not (equal opts 'stay)) (setq *x-reg* x) (setq *y-reg* y)))
)

; Activation grapher (ag)

(defun reset-act (&optional unit-list no-sort)
  "Prepare the grapher to trace the activations of units in unit-list."
  (prog (x y word (glines (/ *winy* 112)))
    (cond ((and (null unit-list) (null *unit-list*))
	   (print "No previously defined unit-list exists.")
	   (return nil))
          ((null unit-list) (setq unit-list *unit-list*))
    )
    (if (null no-sort) (setq unit-list (sort unit-list #'string<)))
    (cond ((> (length unit-list) (* glines *graphs-per-line*))
	   (print (format nil "Max is ~d units." (* glines *graphs-per-line*)))
	   (return nil)
          )
    )
    (setq *act-list* (rep default_activation (length unit-list)))
    (setq *unit-list* unit-list)
    (setq *time-step* 0)
    (clear-bitmap *grapher-bitmap*)
    (do ((ul (cdr *unit-list*) (cdr ul))
	 (un (car *unit-list*) (car ul))
	 (pos 0 (1+ pos)))
	((null un) nil)
	(setf (get un 'activation) default_activation)
	(setq x (+ (round (* (mod pos *graphs-per-line*)
			     (round (/ 640 *graphs-per-line*)))) 24))
	(setq y (+ (round (* (+ (truncate (/ pos *graphs-per-line*)) 1) 100))
		   (if *acme-mode* (* (mod pos *graphs-per-line*) 16) 0) 16))
	(setq word (string un))
        (if (> (length word) *devowel-length*) (setq word (devowel word)))
	(stringblt *grapher-bitmap*
		   (make-position x (- y 100))
		   *grapher-font* word)
	(cond ((equal *act-type* 'line)
	       (draw-line *grapher-bitmap*
			  (make-position x (- y 90))
			  (make-position x (- y 10))
               )
	       (draw-line *grapher-bitmap*
			  (make-position x (- y 50))
			  (make-position (+ x
					    (round (/ 580 *graphs-per-line*)))
					 (- y 50))
	       )
              )
        )
    )
  )
)

(defun show-act ()
  "The real function to show the activation of the current *unit-list*."
  (let (nal)
    (do ((ul (cdr *unit-list*) (cdr ul))
	 (al (cdr *act-list*) (cdr al))
	 (un (car *unit-list*) (car ul))
	 (ac (car *act-list*) (car al))
	 (pos 0 (1+ pos)))
	((null un) nil)
	(if (equal *act-type* 'line)
	    (draw-delta-line pos (* (act un) -40) (* ac -40))
            (draw-delta-circle pos (1+ (* (1+ (act un)) 15))
			           (1+ (* (1+ ac))))
        )
	(setq nal (append nal (list (act un))))
    )
    (setq *act-list* nal)
    (setq *time-step* (1+ *time-step*))
  )
)

(defun draw-delta-line (pos new-a old-a)
  "Function to redraw the individual units with a new activation."
  (let (x y (xs (/ 560 (* *graphs-per-line* (car (last when_to_print))))))
    (setq x (+ (round (* (mod pos *graphs-per-line*)
			 (/ 640 *graphs-per-line*)))
	       (* *time-step* xs) 24))
    (setq y (+ (round (* (+ (truncate (/ pos *graphs-per-line*)) .5) 100))
	       (if *acme-mode* (* (mod pos *graphs-per-line*) 16) 0) 16))

    (draw-line *grapher-bitmap*
	       (make-position (1+ (round (- x xs))) (round (+ y old-a)))
	       (make-position (round x) (round (+ y new-a)))
    )      
  )
)

(defun draw-delta-circle (pos new-r old-r)
  "Function to redraw the individual units with a new radius."
  (let (x y)
    (setq x (+ (round (* (+ (mod pos *graphs-per-line*) .6)
			 (/ 640 *graphs-per-line*))) 24))
    (setq y (+ (round (* (+ (truncate (/ pos *graphs-per-line*)) .50) 100))
	       (if *acme-mode* (* (mod pos *graphs-per-line*) 16) 0) 15))
    (cond ((< new-r old-r)
	   (do ((i (round old-r) (1- i)))
	       ((not (< new-r i)) nil)
	       (draw-circle *grapher-bitmap* (make-position x y) i
			    :operation boole-clr)
	   )
	  )
	  ((> new-r old-r)
	   (do ((i (round old-r) (1+ i)))
	       ((not (> new-r i)) nil)
	       (draw-circle *grapher-bitmap* (make-position x y) i
			    :operation boole-set)
	   )
	  )
    )
  )
)

(defun act (unit)
  "Function to determine the activation of a unit."
  (get unit 'activation)
)

(defun debug-print (message)
  (if (null *debug*) nil (print message)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; FUNCTIONS FOR THE FUNCTION GRAPHER

; CALLTREE makes a list of lists, which will be (func (call1 call2...))
(defun calltree (filename)
  (do ((file (open filename :if-does-not-exist nil))
       (call-tree nil)
       (lin nil (read file nil '(done)))
       (funcname nil nil)
      )
      ((or (null file) (eq (car lin) 'done))
       (close file)
       call-tree
      )
      (cond ((or (equal (car lin) 'defun) (equal (car lin) 'defmacro))
	     (setq funcname (cadr lin))
	     (setq call-tree
		   (cons funcname
			 (cons (remove-if #'sys-func-p
					  (remove-duplicates (lambdaform lin))
			       )
			       call-tree
			 )
		   )
	     )
	    )
      )
  )
)

; A function to make a list of functions within a file.

(defun funclist (filename)
  (do ((file (open filename :if-does-not-exist nil))
       (funclis nil)
       (lin nil (read file nil '(done)))
      )
      ((or (null file) (eq (car lin) 'done))
       (close file)
       funclis
      )
      (if (or (equal (car lin) 'defun) (equal (car lin) 'defmacro))
	  (setq funclis (cons (cadr lin) funclis))
      )
  )
)

; A function to make a list of variables in a file.
(defun varlist (filename)
  (do ((file (open filename :if-does-not-exist nil))
       (varlis nil)
       (lin nil (read file nil '(done)))
      )
      ((or (null file) (eq (car lin) 'done))
       (close file)
       varlis
      )
      (if (or (equal (car lin) 'setq) (equal (car lin) 'defvar))
	  (setq varlis (cons (cadr lin) varlis))
      )
  )
)

; A function to return a list of the functions called by a given function
; stored in a given file.
(defun func-calls (funcname filename)
  (do ((file (open filename :if-does-not-exist nil))
       (call-list nil)
       (lin nil (read file nil '(done)))
      )
      ((or (null file) (eq (car lin) 'done))
       (close file)
       (remove-duplicates call-list)
      )
      (if (and (or (equal (car lin) 'defun) (equal (car lin) 'defmacro))
	       (equal (cadr lin) funcname)
	  )
	  (setq call-list (lambdaform lin))
      )
  )
)

; EVALUATION is a miniature evaluator which does nothing but keep track of
; what it has evaluated.

(defun evaluation (form)
  (do ((answer nil)
       (tform (cdr form) (cdr tform))
       (cfunc (car form) (car tform))
       (test nil nil)
      )
      ((and (null tform) (null cfunc)) answer)
      (if (listp cfunc) (setq test (car cfunc)))
      (cond ((or (eq test 'let) (eq test 'prog))
	     (setq answer (append answer (letform cfunc)))
	    )
	    ((or (eq test 'lambda) (eq test 'defun) (eq test 'defmacro))
	     (setq answer (append answer (lambdaform cfunc)))
	    )
	    ((eq test 'defvar)
	     (setq answer (append answer (defvarform cfunc)))
	    )
	    ((eq test 'do)
	     (setq answer (append answer (doform cfunc)))
	    )
	    ((eq test 'cond)
	     (setq answer (append answer (condform cfunc)))
	    )
	    ((eq test 'quote)
	     (setq answer (cons 'quote answer))
	    )
	    ((or (eq test 'bq-list) (eq test 'bq-list*))
	     (setq answer
		   (append answer
			   (evaluation (remove-if #'atom (cdr cfunc)))
		   )
	     )
	    )
	    ((atom cfunc) nil) ; Ignore all atoms at this point.
	    (t (setq answer (append answer (genform cfunc))))
      )
  )
)

(defun letform (form)
  (append (vdefform (cadr form)) (evaluation (cddr form)))
)

(defun lambdaform (form)
  (evaluation (if (stringp (car (cdddr form)))
		  (cdr (cdddr form))
		  (cdddr form)
	      )
  )
)

(defun defvarform (form)
  (evaluation (list (caddr form)))
)

(defun genform (form)
  (cons (car form) (evaluation (remove-if #'atom (cdr form))))
)

(defun doform (form)
  (append (list (car form))
	  (vdefform (cadr form))
	  (evaluation (caddr form))
	  (evaluation (cdddr form))
  )
)

(defun condform (form)
  (append (list (car form))
	  (apply 'append (mapcar 'evaluation (cdr form)))
  )
)

(defun vdefform (form)
  (do ((answer nil)
       (tform form (cdr tform))
      )
      ((null tform) answer)
      (setq answer (append answer
			   (if (not (atom (car tform)))
			       (evaluation (cdar tform))
			       nil
			   )
		   )
      )
  )
)

; A predicate to determine if the argument is a system function.

(defun sys-func-p (obj)
  (or (not (symbolp obj))
      (fboundp (read-from-string (format nil "lisp::~A" (symbol-name obj))))
      (null obj)
      (equal obj t)
  )
)

(defun datascape (data times)
  (draw-axes 0 (length data) (car (range data)) (cadr (range data)))
  (dotimes (count times)
    (linplot (runavg (1+ count) data) (runavg (1+ count) (iota (length data))))
  )
)

