%{
/*
 *    Lexical Analysis und Error messages: condela.l
 *
 */

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "util.h"
#include "Error.h"
#include "y_tab.h"

extern BOOL      no_u_ident;
extern YYSTYPE   yylval;
extern int       yynerrs;

void pushstk (int );
void include (void);
int  id_rw   (void);
int  yylook  (void);
int  yywrap  (void);

FILE    *sourcefp[MAXINC];
int     comment   = 0,
        sourcesp  = 0,
        copy_code = 0,
        yylinestack [MAXINC];
STRING  yysource    [MAXINC];

STRING ErrMsg [] =
       {
            "",                                                 /*  0 */
            "Error in Expression",                              /*  1 */
            "Integer Expression Expected",                      /*  2 */
            "Error in Statement",                               /*  3 */
            "",                                                 /*  4 */
            "",                                                 /*  5 */
            "Procedure already defined",                        /*  6 */
            "Duplicate identifier",                             /*  7 */
            "",                                                 /*  8 */
            "Operands are not assignment compatible",           /*  9 */
            "Wrong vector component",                           /* 10 */
            "Parameter must not be a constant",                 /* 11 */
            "Array subscript must not be a rangelist",          /* 12 */
            "Nonexistent record/overlay component",             /* 13 */
            "Undeclared identifier",                            /* 14 */
            "Array subscript not allowed",                      /* 15 */
            "Cannot assign a constant",                         /* 16 */
            "Cannot compare a vector",                          /* 17 */
            "Undeclared layer id",                              /* 18 */
            "Net description expected",                         /* 19 */
            "Illegal recursive Net declaration",                /* 20 */
            "Illegal usage of a network-type",                  /* 21 */
            "Illegal declaration of a network",                 /* 22 */
            "Illegal dynamic network declaration,",             /* 23 */
            "Constant expression expected",                     /* 24 */
            "Undeclared network id",                            /* 25 */
            "Layer has no dimension",                           /* 26 */
            "Layer has no pool component",                      /* 27 */
            "Layer has a smaller dimension of field components",/* 28 */
            "Variable is not of type UNIT",                     /* 29 */
            "",                                                 /* 30 */
            "",                                                 /* 31 */
            "Incomplete description of a Unitselection",        /* 32 */
            "Layer has a smaller dimension",                    /* 33 */
            "Illegal Unitselection",                            /* 34 */
            "",                                                 /* 35 */
            "Unable to open include-file",                      /* 36 */
            "Too many nested includes",                         /* 37 */
            "",                                                 /* 38 */
            "",                                                 /* 39 */
            "",                                                 /* 40 */
            "Illegal nested unitselection",                     /* 41 */
            "",                                                 /* 42 */
            "Illegal preprocessor-directive",                   /* 43 */
          };

STRING WarnMsg [] =
          {
            "",                                                 /*  0 */
            "",                                                 /*  1 */
            "",                                                 /*  2 */
            "",                                                 /*  3 */
            "Identifier already declared",                      /*  4 */
            "Undeclared type identifier",                       /*  5 */
            "",                                                 /*  6 */
            "",                                                 /*  7 */
            "Number of arguments doesn't match description",    /*  8 */
            "",                                                 /*  9 */
            "",                                                 /* 10 */
            "",                                                 /* 11 */
            "",                                                 /* 12 */
            "",                                                 /* 13 */
            "",                                                 /* 14 */
            "",                                                 /* 15 */
            "",                                                 /* 16 */
            "",                                                 /* 17 */
            "",                                                 /* 18 */
            "",                                                 /* 19 */
            "",                                                 /* 20 */
            "",                                                 /* 21 */
            "",                                                 /* 22 */
            "",                                                 /* 23 */
            "",                                                 /* 24 */
            "",                                                 /* 25 */
            "",                                                 /* 26 */
            "",                                                 /* 27 */
            "",                                                 /* 28 */
            "",                                                 /* 29 */
            "Arguments of the init-function are ignored",       /* 30 */
            "Preprocessor-directive ignored",                   /* 31 */
            "",                                                 /* 32 */
            "",                                                 /* 33 */
            "",                                                 /* 34 */
            "Extend not yet implemented...",                    /* 35 */
            "",                                                 /* 36 */
            "",                                                 /* 37 */
            "Unexpected end of comment",                        /* 38 */
            "Illegal character in program;(ignored)",           /* 39 */
            "Operator ignored",                                 /* 40 */
            "",                                                 /* 41 */
            "Unexpected procedure identifier",                  /* 42 */
          };

%}

I       [Ii]
N       [Nn]
C       [Cc]
L       [Ll]
U       [Uu]
D       [Dd]
E       [Ee]
Dig     [0-9]
Exp     [Ee][+-]?{Dig}+
WhSp    [ \t\n]
Letter  [a-zA-Z_$]
Char    [a-zA-Z_$0-9]
PrCh    [\040-\176]
Special [!$%'?@\^`~]

%%

^"%"[ \t]*{I}{N}{C}{L}{U}{D}{E}[ \t]*\"({Char}|"."|";")*\"      {
              if(copy_code) { Error(43); yynerrs++; }
			  else include(); }
^"%"[ \t]*{C}       { if(copy_code)   Warning(31); copy_code=TRUE; }
^"%"[ \t]*{E}{N}{D}{C}  {       if(!copy_code)  Warning(31); copy_code=FALSE; }

"/*"            {       comment++;                                           }
"*/"            {       if(comment) comment--; else Warning(38);             }

"("             {       if(!comment) if(copy_code) ECHO; else return LPAR;   }
")"             {       if(!comment) if(copy_code) ECHO; else return RPAR;   }
"["             {       if(!comment) if(copy_code) ECHO; else return LBRCK;  }
"]"             {       if(!comment) if(copy_code) ECHO; else return RBRCK;  }
"{"             {       if(!comment) if(copy_code) ECHO; else return LCURL;  }
"}"             {       if(!comment) if(copy_code) ECHO; else return RCURL;  }
"|"             {       if(!comment) if(copy_code) ECHO; else return VBAR;   }
"*"             {       if(!comment) if(copy_code) ECHO; else return MUL;    }
"/"             {       if(!comment) if(copy_code) ECHO; else return DIV;    }
"+"             {       if(!comment) if(copy_code) ECHO; else return ADD;    }
"-"             {       if(!comment) if(copy_code) ECHO; else return SUB;    }
"."             {       if(!comment) if(copy_code) ECHO; else return DOT;    }
":"             {       if(!comment) if(copy_code) ECHO; else return COLON;  }
";"             {       if(!comment) if(copy_code) ECHO; else return SEMI;   }
","             {       if(!comment) if(copy_code) ECHO; else return COMMA;  }
">"             {       if(!comment) if(copy_code) ECHO; else return GT;     }
"<"             {       if(!comment) if(copy_code) ECHO; else return LT;     }
">="            {       if(!comment) if(copy_code) ECHO; else return GE;     }
"<="            {       if(!comment) if(copy_code) ECHO; else return LE;     }
"#"             {       if(!comment) if(copy_code) ECHO; else return NE;     }
"="             {       if(!comment) if(copy_code) ECHO; else return EQ;     }
":="            {       if(!comment) if(copy_code) ECHO; else return ASGN;   }
".."            {       if(!comment) if(copy_code) ECHO; else return PERIOD; }
"->"            {       if(!comment) if(copy_code) ECHO; else return ARROW;  }
"&"             {       if(!comment) if(copy_code) ECHO; else return ADROF;  }
{Special}       {       if(!comment) if(copy_code) ECHO; else Warning(39);   }

'({PrCh})'              |
'\\[0-7]'               |
'\\[0-7][0-7]'          |
'\\[0-7][0-7][0-7]'     |
'\\\\'                  |
'\\n'                   |
'\\b'                   |
'\\f'                   |
'\\r'                   |
'\\t'                   {       if(!comment) if(copy_code) ECHO;
				else { pushstk(CHARCONST);
					return CHARCONST; } }

\"([^"\n]|\\["\n])*\"   {       if(!comment) if(copy_code) ECHO; 
				else { pushstk(STRINGCONST);
					return STRINGCONST; } }

{Dig}+                  {       if(!comment) if(copy_code) ECHO;
				else { pushstk(INTCONST);
					return INTCONST; } }

{Dig}*"."{Dig}+({Exp})? |
{Dig}+"."{Dig}*({Exp})?\[^.] |
{Dig}+{Exp}             {       if(!comment) if(copy_code) ECHO;
				else { pushstk(FLOATCONST);
					return FLOATCONST; } }

({Letter})+({Char})*    {       if(!comment) if(copy_code) ECHO;
				else return id_rw(); }

{WhSp}+                 { if(copy_code) ECHO; }

%%

static struct rw_table {
        char * rw_name; /* Text + yylex return-code; must be sorted! */
        int rw_yylex; } rw_table[] =
{
"and",          AND,       "apply",       APPLY,       "array",     ARRAY,
"begin",        BEGN,      "boolean",     BOOLEAN,     "by",        BY,
"case",         CASE,      "char",        CHAR,        "connect",   CONNECT,
"const",        CONST,     "continue",    CONTINUE,    "create" ,   CREATE,
"div",          DIV,       "do",          DO,          "else",      ELSE,
"elsif",        ELSIF,     "end",         END,         "enum",      ENUM,
"exitloop",     EXITLOOP,  "extend",      EXTEND,      "field",     FIELD,
"for",          FOR,       "forever",     FOREVER,     "from",      FROM, 
"if",           IF,        "in",          IN,          "init",      INIT,
"integer",      INTEGER,   "intersect",   INTERSECT,   "is",        IS, 
"join",         JOIN,      "layer",       LAYER,       "less",      LESS,
"load",         LOAD,      "loop",        LOOP,        "mod",       MOD,       
"module",       MODULE,    "network",     NETWORK,     "nil",       NIL,
"noduplicate",  NODUPLICATE, "not",       NOT,         "null",      NUL,
"of",           OF,        "or",          OR,          "overlay",   OVERLAY,
"pointer",      POINTER,   "pool",        POOL,        "procedure", PROCEDURE,
"real",         REAL,      "record",      RECORD,      "repeat",    REPEAT,
"return",       RETURN,    "string",      STRNG,       "save",      SAVE,
"then",         THEN,      "times",       TIMES,       "to",        TO,          
"topology",     TOPOLOGY,  "type",        TYPE,        "unit",      UNIT,
"until",        UNTIL,     "usel",        USEL,        "var",       VAR,
"vector",       VECT,      "weight",      WEIGHT,     "while",      WHILE
};

void include()
{
        STRING s=yytext;
        if (comment)
           return;

        if(sourcesp+1==MAXINC)
        {
            Error(37);
            return;
        }

        yytext [yyleng-1]='\0';

        s+=8;

        while (*s && *s!='"')
           s++;

        s++;

        if((sourcefp[++sourcesp]=fopen(s,"r"))==0)
        {
            sourcesp--;
            Error (36);
            return;
        }
        else
        {
            yyin                    = sourcefp[sourcesp];
            yysource[sourcesp]      = strsave(s);
            yylinestack[sourcesp-1] = yylineno;
            yylineno                = 1;
        }
}

int id_rw()
{
    struct rw_table *low  = rw_table,
                    *high = rw_table-1+sizeof(rw_table)/sizeof(rw_table[0]),
                    *mid;
    int c;
    ST_OBJECT *typ;

    strlwr (yytext);

        /*
         * rw_table contains a list of all keywords of CONDELA-III in
         * alphabetical order. Search the list binary to find a match.
         *
         */

    while(low<=high)
    {
        mid = low+(high-low)/2;
        c   = strcmp(mid->rw_name,yytext);

        if (!c)
            return mid->rw_yylex;

        if(c<0) low=mid+1; else high=mid-1;
    }

        /*
         * yytext does not contain a keyword, so maybe it contains
         * a BOOLEAN-Value
         *
         */

    if(!strcmp("true",yytext) || !strcmp("false",yytext))
    {
        pushstk(BOOLCONST);
        return BOOLCONST;
    }

        /*
         * It is not a BOOLEAN-Value and not a keyword, so we have
         * to deal with an ID
         *
         */

    pushstk(IDENT);

    typ=s_lookup(yytext, S_ENQ);

    if(!no_u_ident && typ && typ->st_type==S_USEL)
        return U_IDENT;

    return IDENT;
}

int yywrap()
{
    if(sourcesp)
    {
        fclose(yyin);
        yyin=sourcefp[--sourcesp];
        yylineno=yylinestack[sourcesp];
        return 0;
    }

    return 1;
}

void pushstk(int tok)
{
    switch(tok)
    {
      case BOOLCONST:
        yylval.cexp.expr = strsave(*yytext=='t' ? "1" : "0");
        break;
      case IDENT:
      case INTCONST:
      case FLOATCONST:
      case CHARCONST:
      case STRINGCONST:
        yylval.cexp.expr=strsave (yytext);
        break;
    }
}

void yywhere()
{
    fprintf(stdout,"%s(%d) : ", yysource[sourcesp], yylineno);
}

int yyerror(STRING s)
{
    yywhere (            );
    fprintf (stdout,    s);
    fputs   ("\n", stdout);

    yynerrs++;
}

int Error (int nErr)
{
    yywhere ();

    fprintf (stdout, "error CON%03d: ", nErr);

    if (nErr <= sizeof ErrMsg/sizeof ErrMsg [0])
        fprintf(stdout, ErrMsg [nErr]);

    fputs("\n",stdout);

    yynerrs++;

    return nErr;
}

int Warning (int nWarn)
{
    yywhere ();

    fprintf (stdout, "warning CON%03d: ", nWarn);

    if (nWarn <= sizeof WarnMsg/sizeof WarnMsg [0])
        fprintf (stdout, WarnMsg [nWarn]);

    fputs("\n",stdout);

    yynerrs++;

    return nWarn;
}

int FatalError (int nErr)
{
    yywhere ();

    fprintf (stdout, "Fatal Error CON%03d: out of memory\n", nErr);

    yynerrs++;

    exit (nErr);
    return nErr;
}
