:- module(foil, [foil/1,foil/2]).

:- (predicate_property(findall(_,_,_),built_in)-> true |
	use_module(findall,library(findall), [findall/3])).

:- use_module(cie,library(cie),[cie_get_fact/1,cie_new_rule/1]).
:- use_module(lists,library(lists),[transpose/2,nth1/3]).
:- use_module(sortedset,library(sortedset),all).
:- use_module(strings,library(strings),[concat_atom/2]).
:- use_module(between,library(between),[between/3]).
:- use_module(itemspec,library(itemspec),[fact_prop/2,fact_ep/2,
	rule_prems/2,rule_concl/2]).
:- use_module(basics,library(basics),[member/2]).

:- use_module(lineio,library(lineio),[fget_line/3]).
:- use_module(ctypes,library(ctypes),all).
:- use_module(caseconv,library(caseconv),[lower/2]).
:- use_module(hci,library(hci),[hci_msgs/1]).
:- use_module(environ,library(environ),[expanded_file_name/2,environ/2]).
:- use_module(hypernews,library(hypernews),[hn_show/1,hn_hide/1]).

foil(Focus):-
	hn_show(stack('Foil')),
	foil(Focus,Focus),
	hn_hide(stack('Foil')).

foil(Focus,Filter):-
	%% determine name of communication files
	expanded_file_name('/tmp/mobal2foil$$',ToFoilFile),
	expanded_file_name('/tmp/foil2mobal$$',ToMobalFile),
	hci_msgs(['Generating Datafile ....',nl]),
	write_foil_file(Focus,ToFoilFile),
	hci_msgs(['Running Foil .....',nl]),
	run_foil(Focus,ToFoilFile,ToMobalFile),
	hci_msgs(['Reading learned Rules ...',nl]),
	read_foil_file(Focus,Filter,ToMobalFile),
	hci_msgs(['Done.',nl]),
        !.

run_foil(_Focus,FromFile,ToFile):-
%	open('File',write,S),
%	close(S),
	%% check if FOIL is in a non-standard place
	(environ('FOIL',_)
           -> expanded_file_name('$FOIL',Foil)
	   ;  expanded_file_name('$MOBALHOME/tools/bin/foil',Foil)),
	concat_atom([Foil,' -v0 < ',
	FromFile,
	' | awk ''($0 ~ /:-/ && $0 !~ /^	/) || $0 ~ /secs/ {print $0 >"',
	ToFile,
	'"; print $0}'''],Command),
	(unix(system(Command))
        ;true).

/*
run_foil(Focus,FromFile,ToFile):-
	%% the two following calls may fail if Focus contains too many predicates
	%% (in this case, the call or the regexp alone are longer than 255 chars,
	%% which is the limit for atoms).
	(environ('FOIL',_)
           -> expanded_file_name('$FOIL',Foil)
	   ;  expanded_file_name('$MOBALHOME/foil/foil5',Foil)),
	(construct_regexp(Focus,RegExp),
	 concat_atom(['xterm -title foil -sb -sl 500 -l -lf "| egrep ''',
	             RegExp,
		     ''' > ',ToFile,'" -e ', Foil,' -v0 < ',FromFile
		    ],Command)
	-> true
        ;  %% use foil in a less verbose mode so we can do away with the egrep
	   concat_atom(['xterm -title foil -sb -sl 500 -l ',
	                '-lf ',ToFile,' ',
			'-e ',FOIL,' -v0',' < ',FromFile,' ;/bin/sh'],Command)),
	hci_msgs([Command,nl]),
	(unix(shell(Command));true).
*/

construct_regexp([Last],LastExp):-
	concat_atom(['(',Last,')'],LastExp),
	!.
construct_regexp([First|Rest],RegExp):-
	construct_regexp(Rest,RestExp),
	concat_atom(['(',First,')|',RestExp],RegExp),
	!.
construct_regexp(Focus,Focus):-
	atom(Focus),
	!.
	
read_foil_file(Focus,Focus,File):-
	!,
	open(File,read,FN),
	findall(Rule,get_rule(FN,Rule),Rules),
	(member(R,Rules),
	 cie_new_rule(R),
	 fail
	;true
	).

read_foil_file(_,Filter,File):-
	open(File,read,FN),
	findall(Rule,get_rule(FN,Rule),Rules),
	(member(R,Rules),
	 %% check if the rule mentions one of the targets in Filter
	 check_rule_against_filter(R,Filter),
	 cie_new_rule(R),
	 fail
	;true
	).

check_rule_against_filter(R,Filter):-	 
	 rule_prems(R,Prems), rule_concl(R,Concl),
	 member(Literal,[Concl|Prems]),
	 (Literal=not(Literal1)->true;Literal1=Literal),
	 functor(Literal1,F,_),
	 member(F,Filter),
	 !.

get_rule(FN,Rule):-
	repeat,
	fget_line(FN,Line,Terminator),
	(is_endfile(Terminator) -> ! | true),
	phrase(foilrule(Rule),Line).


foilrule(rule(_,Premises,Concl,_,_,_,_)) -->
	 foilconcl(Concl),
	 blanks,":-",
	 foilprems(Premises),
	 !.
foilconcl(Concl) -->
	symbol(Pred),
	special_char('('),
	varlist(Vars),
	special_char(')'),
	{Concl =.. [Pred|Vars]},
	!.
foilprems([Prem|Premises]) -->
	foilprem(Prem),
	special_char(','),
	!,
	foilprems(Premises).
foilprems([Prem]) -->
	foilprem(Prem).
foilprem(not(Prem)) -->
	special_char('~'),
	!,
	(special_char('*');{true}),
	symbol(Pred),
	special_char('('),
	varlist(Vars),
	special_char(')'),
	{Prem =.. [Pred|Vars]},
	!.
foilprem(Prem) -->
	blanks,
	("*";{true}),
	symbol(Pred),
	special_char('('),
	varlist(Vars),
	special_char(')'),
	{Prem =.. [Pred|Vars]},
	!.
foilprem(Prem) -->
	blanks,
	"=",
	special_char('('),
	varlist(Vars),
	special_char(')'),
	{Prem =.. [eq|Vars]},
	!.

foilprem(eq(Var1,Var2)) -->
	symbol(UVar1),
	{lower(UVar1,Var1)},
	blanks,
	"=",
	symbol(UVar2),
	{lower(UVar2,Var2)},
	!.

blank --> " ",!.
blank --> [C], {is_space(C)}.

blanks(S0,S1):- var(S0),blank(S0,S1),!.
blanks(S0,S2):- nonvar(S0), blank(S0,S1),!,blanks(S1,S2).
blanks --> [].


digit(D) --> [D], {is_digit(D)},!.

alpha(A) --> [A], {is_alpha(A)},!.

alpa_num(A) --> (digit(A) | alpha(A) | [A], {([A] = "-" | [A] = "_")}),!.

alpa_num_seq(S,I,R):- var(S),var(I),var(R),!,fail.
alpa_num_seq([F|R]) --> alpa_num(F), !,alpa_num_seq(R).
alpa_num_seq([]) --> [].

% Symbol, which does allow any number of blank chars on reading, 
% and prints exactly one on writing:
symbol(S,I,R):- var(S),var(I),var(R),!,fail.
symbol(S) --> {var(S)}, !, blanks, alpha(F), alpa_num_seq(R), {atom_chars(S,[F|R])}.
symbol(S) --> {atom(S),atom_chars(S,[F|R])}, !, blanks, alpha(F), alpa_num_seq(R).

special_char('(') --> blanks, "(".
special_char(')') --> blanks, ")".
special_char('*') --> blanks, "*".
special_char(',') --> blanks, ",".
special_char('~') --> blanks, "~".
special_char('.') --> ".".

varlist([Var|Vars]) -->
	symbol(UVar),
	{lower(UVar,Var)},
	special_char(','),
	!,
	varlist(Vars).
varlist([Var]) -->
	symbol(UVar),
	{lower(UVar,Var)}.

write_foil_file(Focus,File):-
	copy_kb,
	findall(Pred,current_predicate(_,pos:Pred),Preds),
	!,
	build_sorts_for_preds(Preds,[],[],RootPreds,RootTerms),
	open(File,write,FN),
	!,
	write_sort_defs(FN,RootPreds,RootTerms),
	nl(FN),
	!,
	(member(Pred,Preds),
	 write_pred_defs(FN,Pred,Focus,RootPreds,RootTerms),
	 fail
	;true),
	close(FN),
	!.

write_sort_defs(FN,_RootPreds,RootTerms):-
	nth1(N,RootTerms,[FT|Ext]),
	concat_atom([sort_,N],SName),
	write(FN,SName),
	write(FN,': '),
	write(FN,FT),
	(member(T,Ext),write(FN,','),nl(FN),write(FN,T),fail;write(FN,'.')),
	nl(FN),
	fail.
write_sort_defs(_,_,_).

write_pred_defs(FN,Pred,Focus,RootPreds,_RootTerms):-
	functor(Pred,PN,A),
	findall(SName,
		(between(1,A,ArgN),
	         nth1(RN,RootPreds,RPreds),
		 member(PN/ArgN,RPreds),
		 concat_atom([sort_,RN],SName)
		),[PS1|PredSorts]),
	((PN = Focus;member(PN,Focus)) -> true | write(FN,'*')),
	write(FN,PN),
	write(FN,'('),
	write(FN,PS1),
	(member(APS,PredSorts),write(FN,','),write(FN,APS),fail;write(FN,')')),
	nl(FN),
	write_pred_facts(FN,Pred),
	!.

%%% Pred is a *proposition*
write_pred_facts(FN,Pred):-
	pos:clause(Pred,_),
	Pred =.. [_,PT1|PTs],
	write(FN,PT1),
	(member(PT,PTs),write(FN,','),write(FN,PT),fail;nl(FN)),
	fail.
%% 
write_pred_facts(FN,Pred):-
	%% check if there are *any* neg. facts with predicate
	(\+(neg:clause(Pred,_)) -> fail
        ;
	(write(FN,';'),nl(FN), 
	neg:clause(Pred,_),
	Pred =.. [_,PT1|PTs],
	write(FN,PT1),
	(member(PT,PTs),write(FN,','),write(FN,PT),fail;nl(FN)),
	fail)).

write_pred_facts(FN,Pred):-
	(pos:clause(Pred,_);neg:clause(Pred,_)),
	write(FN,'.'),
	nl(FN),
	!.

copy_kb:-
	current_predicate(_,pos:Pred),
	functor(Pred,PN,A),
	abolish(pos:PN,A),
	fail.
copy_kb:-
	current_predicate(_,neg:Pred),
	functor(Pred,PN,A),
	abolish(neg:PN,A),
	fail.
copy_kb:-
	cie_get_fact(FactS),
	fact_prop(FactS,Prop),
	fact_ep(FactS,EP),
	(EP = [1000,_] -> assertz(pos:Prop) | true),
	(EP = [_,1000] -> assertz(neg:Prop) | true),
	fail.
copy_kb.

build_sorts_for_preds([],RootPreds,RootTerms,RootPreds,RootTerms).
build_sorts_for_preds([Pred|Preds],OldRootPreds,OldRootTerms,RootPreds,RootTerms):-
	Pred =.. [PN|Args],
	findall(Args,(pos:clause(Pred,_);neg:clause(Pred,_)),ArgsList),
	transpose(ArgsList,ListofArgs),
	build_arg_sorts(ListofArgs,PN,1,OldRootPreds,OldRootTerms,ZRootPreds,ZRootTerms),
	!,
	build_sorts_for_preds(Preds,ZRootPreds,ZRootTerms,RootPreds,RootTerms).

build_arg_sorts([],_,_,RootPreds,RootTerms,RootPreds,RootTerms).
build_arg_sorts([F|R],PN,A,OldRootPreds,OldRootTerms,RootPreds,RootTerms):-
	sort(F,SortF),
	compute_root_intersections(OldRootPreds,OldRootTerms,
				   [PN/A],SortF,[],[],ZRootPreds,ZRootTerms),
	NA is A+1,
	!,
	build_arg_sorts(R,PN,NA,ZRootPreds,ZRootTerms,RootPreds,RootTerms).
	

/********************************************************************************/
/*  compute_root_intersections                                                  */
/*    (!Old_Names,!Old_WE,!Name,!SE,!New_Names,!NWE,?Names,?WE )                */
/********************************************************************************/

compute_root_intersections([],[],_,[],Name_List,NWE,Name_List,NWE):-
	!.
compute_root_intersections([],[],Name,SE,Name_List,NWE,
	    [Name|Name_List],[SE|NWE]):-
	!.
compute_root_intersections([Sub|Subs],[Ext|Exts],Name,SE,Name_List,NWE,
	    New_Subs,New_WE):-
	s_intersectionset(SE,Ext,[]),
	!,
	compute_root_intersections(Subs,Exts,Name,SE,[Sub|Name_List],[Ext|NWE],
	    New_Subs,New_WE).

compute_root_intersections([Sub|Subs],[Ext|Exts],Name,SE,Name_List,NWE,
	    New_Subs,New_WE):-
	s_unionset(SE,Ext,NSE),
	append(Name,Sub,New_Name),
	compute_root_intersections(Subs,Exts,New_Name,NSE,Name_List,NWE,
	    New_Subs,New_WE).
