%%% ============================================================================
%%% Copyright Notice
%%% ============================================================================
%%% This file is part of the MOBAL system and is NOT in the public domain. It
%%% is available to the MLT consortium partners and the EEC under the conditions
%%% of the MLT contract only and may NOT in any form be used outside MLT nor
%%% made available to any third party without prior written permission from GMD.
%%% (C) 1990-92 GMD (Ges. f. Mathematik und Datenverarbeitung, FIT.KI, 
%%%         Projekt MLT, Pf. 1316, 5205 St. Augustin 1, West Germany).
%%% ============================================================================
%%% The MOBAL program interface:
%%% ============================================================================

:- module(mobalpi,[
%% init MOBAL:
        mobal_initialize/0,
%% Transfer of control to MOBALs HCI:
        mobal/0,
	smobal/0,
%% Typeout message to current output stream:
	mobal_msg/1,
%% MOBALs KB item generators:
	mobal_get_fact/1,
	mobal_get_mfact/1,
	mobal_get_mmfact/1,
	mobal_get_pred/1,
	mobal_get_mpred/1,
	mobal_get_mmpred/1,
	mobal_get_rule/1,
	mobal_get_mrule/1,
	mobal_get_ic/1,
	mobal_get_tnode/1,
	mobal_get_sclass/1,
%% MOBALs KB delete item:
	mobal_delete_fact/1,
	mobal_delete_ic/1,
	mobal_delete_mfact/1,
	mobal_delete_mmfact/1,
	mobal_delete_pred/1,
	mobal_delete_mpred/1,
	mobal_delete_mmpred/1,
	mobal_delete_rule/1,
	mobal_delete_mrule/1,
	mobal_delete_tnode/1,
%% MOBALs KB add new item:
	mobal_new_fact/1,
	mobal_new_ic/1,
	mobal_new_mfact/1,
	mobal_new_mmfact/1,
	mobal_new_pred/1,
	mobal_new_mpred/1,
	mobal_new_mmpred/1,
	mobal_new_rule/1,
	mobal_new_mrule/1,
	mobal_new_tnode/1,
%% MOBALs Parameter:
        mobal_get_parameter/2,
	mobal_set_parameter/2,
%% MOBALs domain operations:
	mobal_load_domain/1,
	mobal_save_domain/1,
	mobal_clear_domain/0,
	mobal_delete_domain/1,
	mobal_read_textfile/1,
	mobal_write_textfile/1,
	mobal_print_domain/1,
	mobal_read_ckrl_file/1,
	mobal_write_ckrl_file/1,
%% MOBALs knowledge revision (KRT)
        mobal_knowledge_revision/1,
%% MOBALs learning tools
	mobal_call_learning_tool/2,
%% MOBALs inference engine
	mobal_get_derivation/2,
	mobal_query_fact/1,
	mobal_query_mfact/1,
	mobal_query_mmfact/1,
%% MOBALs aganda of open problems
	mobal_get_agenda_entry/1,
	mobal_delete_agenda_entry/1,
	mobal_handle_agenda_entry/1,
%% MOBAL events, called by MOBAL - you must not call them, but you could define them.
        mobal_hn_message/3,
	mobal_ic_stored/1,
	mobal_ic_deleted/1,
        mobal_fact_stored/1,
	mobal_fact_deleted/1,
	mobal_fact_changed/1,
	mobal_mfact_stored/1,
	mobal_mfact_deleted/1,
	mobal_mfact_changed/1,
	mobal_mmfact_stored/1,
	mobal_mmfact_deleted/1,
	mobal_mmfact_changed/1,
	mobal_pred_stored/1,
	mobal_pred_deleted/1,
	mobal_pred_changed/1,
	mobal_mpred_stored/1,
	mobal_mpred_deleted/1,
	mobal_rule_stored/1,
	mobal_rule_deleted/1,
	mobal_mrule_stored/1,
	mobal_mrule_deleted/1,
	mobal_tnode_stored/1,
	mobal_tnode_deleted/1,
	mobal_sclass_stored/1,
	mobal_sclass_deleted/1
    ]).
	
%%% ****************************************************************************
%%% Imports
%%% ****************************************************************************
:-use_module(library(compatibility)).

:- use_module(hci,library(hci),[
        hci_command_loop/0,
        hci_serial_command_loop/0,
	hci_initialize/0,
	hci_msgs/1
    ]).

:- use_module(cie,library(cie), [
	cie_interface_independent_initializations/0,
	cie_reinitialize_all/0,
	cie_get_derivation/2,
%% Domain file handling:
	cie_delete_domain_files/1,
	cie_load_specified_domain/1,
	cie_save_domain_as/1,
%% Text file handling:
	cie_read_scratch_file/1,
	cie_print_current_domain/2,
%% Parameter handling:
	mobal_get_param/2,
	mobal_set_param/2,
%% Agenda handling:
	cie_get_agenda_item/1,
	cie_handle_agenda_item/1,
	cie_delete_agenda_item/1,
%% item generators:
	cie_get_fact/1,
	cie_get_mfact/1,
	cie_get_mmfact/1,
	cie_get_pred/1,
	cie_get_mpred/1,
	cie_get_mmpred/1,
	cie_get_rule/1,
	cie_get_mrule/1,
	cie_get_ic/1,
	cie_get_tnode/1,
%% delete item:
	cie_delete_fact/1,
	cie_delete_ic/1,
	cie_delete_mfact/1,
	cie_delete_mmfact/1,
	cie_delete_pred/1,
	cie_delete_mpred/1,
	cie_delete_mmpred/1,
	cie_delete_rule/1,
	cie_delete_mrule/1,
	cie_delete_tnode/1,
%% new item:
	cie_new_fact/1,
	cie_new_ic/1,
	cie_new_mfact/1,
	cie_new_mmfact/1,
	cie_new_pred/1,
	cie_new_mpred/1,
	cie_new_mmpred/1,
	cie_new_rule/1,
	cie_new_mrule/1,
	cie_new_tnode/1,
%% query item:
        cie_query_fact/1,
	cie_query_mfact/1
    ]).

:- use_module(stt,library(stt),[
        stt_get_sort/1
    ]).

:- use_module(ckrl,[
	ckrl_read_file/1,
	ckrl_write_file/1,
	ckrl_translate/0,
	ckrl_retranslate/0,
	ckrl_clear_kb/0
	]).

:- use_module(rdt,library(rdt),[
        rdt/1,
	clt_rdt/1
    ]).

:- use_module(krt,library(krt),[
	krt_knowledge_revision/1
    ]).

:- use_module(basics,library(basics),[member/2]).
:- use_module(itemspec,library(itemspec),all).

%%% ****************************************************************************
%%% mobal_initialize
%%% ****************************************************************************
%%% Initializes all MOBAL datastructures. Does nothing if MOBAL is already
%%% initialized. This call or the call mobal/0 must be the first call to MOBAL.
%%% ****************************************************************************

mobal_initialize:-
	hci_initialize,
	!.

%%% ****************************************************************************
%%% mobal
%%% ****************************************************************************
%%% Initializes all MOBAL datastructures if necessary. 
%%% Starts MOBALs HCI and transfers control to the user.
%%% This call or the call mobal_initialize/0 must be the first call to MOBAL.
%%% ****************************************************************************

mobal:-
	hci_command_loop,
	!.

smobal:-
	hci_serial_command_loop,
	!.

%%% ****************************************************************************
%%% mobal_msg(!MessageList)
%%% ****************************************************************************
%%% Writes the print directives in MessageList to the current output stream.
%%% The default is MOBALs console.
%%% 
%%% Print directives are:
%%%    - MOBAL-items:    Those are pretty-printed using the print-functions
%%%                      in $MOBALHOME/library/itemspec.pl
%%%    - sp, sp(N):      1/N spaces are output.
%%%    - nl or nl(N):    1/N linefeeds are output.
%%%    - Prolog Objects: Those are printed using write
%%% For example: mobal_hci_msg(['Asking MOBAL for Fact',sp,
%%%                             f(responsible(sw,event1),[0,1000],_,_),'.',nl,
%%%                             'Waiting for answer ...',nl]).
%%% Produces:
%%% Asking MOBAL for Fact not(responsible(sw,event1)).
%%% Waiting for answer ...
%%% 
%%% More information about formats can be found in $MOBALHOME/library/msg.pl
%%% ****************************************************************************

mobal_msg(MessageList):-
	hci_msgs(MessageList),
	!.

%%% ****************************************************************************
%%% mobal_get_{item}(?ItemStruc)
%%% ****************************************************************************
%%% The following is a set of calls for asking MOBAL for stored {item}s.
%%% If you want to ask for an inferrable, but not yet inferred, {factitem} 
%%% use mobal_query_{factitem} instead.
%%% {item} <- {factitem} | pred | mpred | mmpred | rule | mrule | ic | tnode | sclass
%%% {factitem} <- fact | mfact | mmfact
%%% See $MOBALHOME/library/itemspec.pl for building and accessing legal ItemStrucs.
%%% It is recommended that you use the calls in $MOBALHOME/library/itemspec.pl  
%%% to build and access ItemStrucs. If you do that your code will be
%%% compatible with future releases of MOBAL.
%%% These calls unify ItemStruc with stored {item}s. They succeed if there is
%%% a matching {item}. If ItemStruc is not fully instantiated, they enumerate all
%%% matching {item}s on backtracking.
%%% ****************************************************************************

mobal_get_fact(FactStruc):-
	cie_get_fact(FactStruc).
mobal_get_mfact(MFactStruc):-
	cie_get_mfact(MFactStruc).
mobal_get_mmfact(MMFactStruc):-
	cie_get_mmfact(MMFactStruc).
mobal_get_pred(PredStruc):-
	cie_get_pred(PredStruc).
mobal_get_mpred(MPredStruc):-
	cie_get_mpred(MPredStruc).
mobal_get_mmpred(MMPredStruc):-
	cie_get_mmpred(MMPredStruc).
mobal_get_rule(RuleStruc):-
	cie_get_rule(RuleStruc).
mobal_get_mrule(MRuleStruc):-
	cie_get_mrule(MRuleStruc).
mobal_get_ic(ICStruc):-
	cie_get_ic(ICStruc).
mobal_get_tnode(TNodeStruc):-
	cie_get_tnode(TNodeStruc).
mobal_get_sclass(SClassStruc):-
	stt_get_sort(SClassStruc).

%%% ****************************************************************************
%%% mobal_get_derivation(!FactItemStruc,?DerivationStruct)
%%% ****************************************************************************

mobal_get_derivation(FactStruc,DerivationStruc):-
        fact_p(FactStruc),
	cie_get_derivation(FactStruc,DerivationStruc).

%%% ****************************************************************************
%%% mobal_delete_{item}(!ItemStruc)
%%% ****************************************************************************
%%% The following is a set of calls for deleting stored MOBAL {item}s.
%%% {item} <- fact | mfact | mmfact | pred | mpred | mmpred | rule | ic | mrule | tnode 
%%% See $MOBALHOME/library/itemspec.pl for building and accessing legal ItemStrucs.
%%% It is recommended that you use these calls in $MOBALHOME/library/itemspec.pl  
%%% to build and access ItemStrucs. If you do that your code will be
%%% compatible with future releases of MOBAL.
%%% These calls behave unpredictably, if ItemStruc doesn't uniquely identify an item.
%%% These calls do not delete an inferred item, they only undo the effect of 
%%% a call to mobal_new_{item}.
%%% ****************************************************************************

mobal_delete_fact(FactStruc):-
	cie_delete_fact(FactStruc).
mobal_delete_mfact(MFactStruc):-
	cie_delete_mfact(MFactStruc).
mobal_delete_mmfact(MMFactStruc):-
	cie_delete_mmfact(MMFactStruc).
mobal_delete_pred(PredStruc):-
	cie_delete_pred(PredStruc).
mobal_delete_mpred(MPredStruc):-
	cie_delete_mpred(MPredStruc).
mobal_delete_mmpred(MMPredStruc):-
	cie_delete_mmpred(MMPredStruc).
mobal_delete_rule(RuleStruc):-
	cie_delete_rule(RuleStruc).
mobal_delete_ic(ICStruc):-
	cie_delete_ic(ICStruc).
mobal_delete_mrule(MRuleStruc):-
	cie_delete_mrule(MRuleStruc).
mobal_delete_tnode(TNodeStruc):-
	cie_delete_tnode(TNodeStruc).

%%% ****************************************************************************
%%% mobal_new_{item}(!ItemStruc)
%%% ****************************************************************************
%%% The following is a set of calls for adding MOBAL {item}s.
%%% {item} <- fact | mfact | mmfact | pred | mpred | mmpred | rule | mrule | ic | tnode 
%%% See $MOBALHOME/library/itemspec.pl for building and accessing legal ItemStrucs.
%%% It is recommended that you use these calls in $MOBALHOME/library/itemspec.pl  
%%% to build and access ItemStrucs. If you do that your code will be
%%% compatible with future releases of MOBAL.
%%% The ID field should be uninstantiated, a unique ID is build by MOBAL. If no 
%%% comment is to be specified, the comment field could be [] or uninstantiated as well
%%% These calls behave unpredictably, if nessessary fields of 
%%% ItemStruc aren't instantiated.
%%% Nessesary are:
%%% fact: prop, ep
%%% mfact: pred, args, supset, ep
%%% mmfact: pred, args, ep
%%% pred: name, arity, usm
%%% mpred: name, args, rulescheme
%%% mmpred: name, args, rulescheme
%%% rule: vars, prems, concl, supset
%%% mrule: vars, prems, concl
%%% ic: prems,concls
%%% tnode: name, predicates, links
%%% ****************************************************************************

mobal_new_fact(FactStruc):-
	cie_new_fact(FactStruc).
mobal_new_ic(ICStruc):-
	cie_new_ic(ICStruc).
mobal_new_mfact(MFactStruc):-
	cie_new_mfact(MFactStruc).
mobal_new_mmfact(MMFactStruc):-
	cie_new_mmfact(MMFactStruc).
mobal_new_pred(PredStruc):-
	cie_new_pred(PredStruc).
mobal_new_mpred(MPredStruc):-
	cie_new_mpred(MPredStruc).
mobal_new_mmpred(MMPredStruc):-
	cie_new_mmpred(MMPredStruc).
mobal_new_rule(RuleStruc):-
	cie_new_rule(RuleStruc).
mobal_new_mrule(MRuleStruc):-
	cie_new_mrule(MRuleStruc).
mobal_new_tnode(TNodeStruc):-
	cie_new_tnode(TNodeStruc).



%%% ****************************************************************************
%%% mobal_get_agenda_entry(?AE_Struc)
%%% ****************************************************************************
%%% This call tries to unify the AE_Struc with an current agenda entry.
%%% It succeeds if there is a matching agenda entry, it fails if not.
%%% IfAE_Struc isn't fully instantiated it enumerates all matching 
%%% agenda entry on backtracking.
%%% ****************************************************************************

mobal_get_agenda_entry(AE_Struc):-
	cie_get_agenda_item(AE_Struc),
	!.

%%% ****************************************************************************
%%% mobal_delete/handle_agenda_entry(!AE_Struc)
%%% ****************************************************************************
%%% You can use these calls to handle or delete agenda entries yoou have 
%%% obtained by a call to mobal_get_agenda_entry.
%%% ****************************************************************************

mobal_delete_agenda_entry(AE_Struc):-
	cie_delete_agenda_item(AE_Struc),
	!.

mobal_handle_agenda_entry(AE_Struc):-
	cie_handle_agenda_item(AE_Struc),
	!.

%%% ****************************************************************************
%%% mobal_get_parameter(?ParameterName,?ParameterValue)
%%% ****************************************************************************
%%% This call tries to unify ParameterName and ParameterValue with 
%%% current settings of MOBAL parameters. It succeeds if there is a matching
%%% ParameterName/ParameterValue combination, it fails if not.
%%% If ParameterName isn't instantiated it enumerates all matching 
%%% parameter/value combinations on backtracking.
%%% ****************************************************************************

mobal_get_parameter(ParamName,ParamValue):-
	mobal_get_param(ParamName,ParamValue).

%%% ****************************************************************************
%%% mobal_set_parameter(!ParameterName,!ParameterValue)
%%% ****************************************************************************
%%% This call tries to set MOBALs Parameter ParameterName to ParameterValue.
%%% It suceeds and sets the parameter to ParameterValue, if ParameterName is a 
%%% MOBAL Parameter and ParameterValue is a legal value for this parameter.
%%% It fails with no effect, if ParameterName isn't a Parameter or ParameterValue  
%%% isn't a possible value for this Parameter.
%%% ****************************************************************************

mobal_set_parameter(ParamName,ParamValue):-
	mobal_set_param(ParamName,ParamValue).

%%% ****************************************************************************
%%% mobal_load_domain (!DomainName)
%%% ****************************************************************************
%%% This call clears MOBALs knowledgebase and loads a MOBAL domain in internal
%%% format from the current MOBAL-Domains directory 
%%% (the mobal_save_directory parameter, default: "$MOBALHOME/domains/").
%%% This should be a domain created by mobal_save_domain 
%%% (or via HCI with the button "Save Domain").
%%% The call ever succeeds and atleast clears the knowledgebase.
%%% All avalaible Unix-Files which belongs to the domain are loaded. 
%%% If there is no domain with Name DomainName in the current 
%%% MOBAL-Domains directory the call does the same as mobal_clear_domain.
%%% (A Domain consist of several files, with unix-file-names:
%%% DomainName.{ks}
%%% {ks} <- clt | cs | globals | ie | krt | mat | rdt | structure | stt | topology
%%% ****************************************************************************

mobal_load_domain(DomainName):-
	cie_load_specified_domain(DomainName),
	!.

%%% ****************************************************************************
%%% mobal_save_domain (!DomainName)
%%% ****************************************************************************
%%% This call saves MOBALs knowledge-base in internal format in the 
%%% current MOBAL-Domains directory. 
%%% (the mobal_save_directory parameter, default: "$MOBALHOME/domains/").
%%% The domain can be reloaded with mobal_load_domain(DomainName).
%%% The call creates several files in the current MOBAL-Domains directory
%%% (A Domain consist of several files, with unix-file-names:
%%% DomainName.{ks}
%%% {ks} <- clt | cs | globals | ie | krt | mat | rdt | structure | stt | topology
%%% Existing Files are overwritten without warning.
%%% ****************************************************************************

mobal_save_domain(DomainName):-
	cie_save_domain_as(DomainName),
	!.

%%% ****************************************************************************
%%% mobal_clear_domain
%%% ****************************************************************************
%%% This call clears MOBALs knowledge-base, 
%%% i.e. it resets MOBAL to the startup state. This includes the resetting of
%%% parameters to there default value. In opposite to mobal_delete_domain it 
%%% doesn't affect any files on disk.
%%% ****************************************************************************

mobal_clear_domain:-
	cie_reinitialize_all,
	!.

%%% ****************************************************************************
%%% mobal_delete_domain (!DomainName)
%%% ****************************************************************************
%%% The call removes the domain-files in the current MOBAL-Domains directory.
%%% (A Domain consist of several files, with unix-file-names:
%%% DomainName.{ks}
%%% {ks} <- clt | cs | globals | ie | krt | mat | rdt | structure | stt | topology
%%% All existing files of that domain are removed from disk. In opposite to  
%%% mobal_clear_domain it doesn't affect the current knowledgebase.
%%% ****************************************************************************

mobal_delete_domain(DomainName):-
	cie_delete_domain_files(DomainName),
	!.

%%% ****************************************************************************
%%% mobal_read_textfile (!FileName)
%%% ****************************************************************************
%%% This reads a file in the same format as the scratchpad of MOBALs HCI.
%%% The knowledge in this file is added as user-input to the current knowledge-base.
%%% ****************************************************************************

mobal_read_textfile(FileName):-
	cie_read_scratch_file(FileName),
	!.

%%% ****************************************************************************
%%% mobal_write_textfile (!FileName)
%%% ****************************************************************************
%%% This writes a file in the same format as the scratchpad of MOBALs HCI.
%%% All of MOBALs knowledge except the inferred knowledge and the sort-lattice
%%% (i.e user-input and learned knowledge) is stored in the file. That means
%%% reading a so created textfile back into MOBAL via mobal_read_textfile will 
%%% result in the same knowledgebase-state as where the save was done,
%%% execpt parameter settings may differ and the knowledgebase contents prior to 
%%% the call to mobal_read_textfile also still belongs to the knowledgebase.
%%% You should use this call if you want to read the knowledge back into MOBAL
%%% or whenever you need to merge knowledgebases.
%%% ****************************************************************************

mobal_write_textfile(FileName):-
	cie_print_current_domain(FileName,write),
	!.

%%% ****************************************************************************
%%% mobal_print_domain (!FileName)
%%% ****************************************************************************
%%% This writes a file in a format similar to the windows of MOBALs HCI, but
%%% it not only writes items that could be read back into MOBAL.
%%% All of MOBALs knowledge items including the user-input, learned knowledge,
%%% inferred knowledge and the sort-lattice is stored in the file.
%%% You cannot read this file back into MOBAL as it is.
%%% ****************************************************************************

mobal_print_domain(FileName):-
	cie_print_current_domain(FileName,print),
	!.

%%% ****************************************************************************
%%% mobal_read_ckrl_file (!FileName)                    
%%% ****************************************************************************
%%% This reads a file in CKRL format.
%%% The knowledge in this file is added as user-input to the current knoledge.
%%% ****************************************************************************

mobal_read_ckrl_file(FileName):-
	ckrl_read_file(FileName),
	ckrl_translate,
	ckrl_clear_kb,
	!.

%%% ****************************************************************************
%%% mobal_write_ckrl_file (!FileName)                   
%%% ****************************************************************************
%%% This writes a file in CKRL format.
%%% All of MOBALs knowledge including the inferred knowledge is stored in the file.
%%% If you read this file back into MOBAL, previously inferred knowledge is 
%%% then handled as user-input knowledge.
%%% ****************************************************************************

mobal_write_ckrl_file(FileName):-
	ckrl_retranslate,
	ckrl_write_file(FileName),
	ckrl_clear_kb,
	!.

%%% ****************************************************************************
%%% mobal_knowledge_revision (Fact)
%%% ****************************************************************************
%%% KRT is called on Fact.  The user is shown the derivation graph, and can
%%% interactively select which items to modify.  These are then appropriately
%%% modified by KRT.  If you want less interactive operation, set KRT/CLT/HCI
%%% parameters appropriately, eg. for totally non-interactive operation:
%%% confirm_modification_choice: no
%%% check_reformulations_manually: no
%%% delete_unreformulated_rules: yes or no
%%% query_individual_clt_attempts: no
%%% continue_search_queries: never
%%% clt_user_queries: no
%%% typeout_on: terminal
%%% ****************************************************************************

mobal_knowledge_revision(Fact):-
	krt_knowledge_revision(Fact).

%%% ****************************************************************************
%%% mobal_call_learning_tool (!Tool,!PredicateSpec)
%%% ****************************************************************************
%%% MOBAL tries to call the learning tool with Predicate as argument.
%%% you can call all current learning tools of MOBAL this way,
%%% and if you write an interface to your own learning tool,
%%% export an predicate Tool/1, to call your tool,
%%% and store it as $MOBALHOME/Tool.pl (.qof) 
%%% you can call it, too.
%%% The current possibilities are:
%%% mobal_call_learning_tool(rdt,!Atom_or_List).
%%%      MOBAL tries to learn rules with Predicate as conclusion using RDT.
%%% mobal_call_learning_tool(clt,!Atom_or_List).
%%%      MOBAL tries to learn rules, Predicate is involved, 
%%%      no only in the conclusion, but also in the premisses of rules using CLT.
%%%      The parameter clt_user_queries [yes/no] determins the interactivity of CLT.
%%% mobal_call_learning_tool(incy,!Atom_or_List).
%%%      MOBAL tries to learn rules with Predicate as conclusion using INCY.
%%% mobal_call_learning_tool(cilgg,!Atom_or_List).
%%%      MOBAL tries to learn rules with Predicate as conclusion using CILGG.
%%% mobal_call_learning_tool(golem,!Atom_or_List).
%%%      MOBAL tries to learn rules with Predicate as conclusion using 
%%%      Muggleton's GOLEM.
%%% mobal_call_learning_tool(mfoil,!Atom_or_List).
%%%      MOBAL tries to learn rules with Predicate as conclusion using 
%%%      Dzeroski's mFOIL.
%%% mobal_call_learning_tool(foil,!Atom_or_List).
%%%      MOBAL tries to learn rules with Predicate as conclusion using 
%%%      Quinlan's FOIL.
%%% ****************************************************************************

mobal_call_learning_tool(Tool,PredicateSpec):-
	functor(ToolCall,Tool,1),
	(current_predicate(Tool,Tool:ToolCall) ->
	    true
        |   on_exception(X,use_module(Tool,library(Tool),[Tool/1]),
			 (print_message(error,X),fail))
	),
	arg(1,ToolCall,Predicate),
	(analyze_predicate_spec(PredicateSpec,Predicate),
	 (call(Tool:ToolCall) -> false)
	;true),
	!.

analyze_predicate_spec(Name/Topology,Predicate):-
	tnode_topology(TNode,Topology),
	tnode_name(TNode,Name),
	cie_get_tnode(TNode),
	tnode_predicates(TNode,Predicates),
	!,
	member(Predicate,Predicates).
analyze_predicate_spec([P|PList],Pred):-
	!,
	member(Pred,[P|PList]).
analyze_predicate_spec(P,P):-
	atom(P), !.
	

%%% ****************************************************************************
%%% mobal_query_{factitem} ()
%%% ****************************************************************************
%%% This call succeeds not only  on currently stored {factitem}s 
%%% (i.e. like mobal_get_{factitem}), but also on inferrable {factitem}s.
%%% It acts similar to a PROLOG-query, that means it fails if there is no
%%% solution, and it enumerates all solutions on backtracking.
%%% ****************************************************************************

mobal_query_fact(FactStruc):-
	cie_query_fact(FactStruc).
mobal_query_mfact(MFactStruc):-
	cie_query_mfact(MFactStruc).
mobal_query_mmfact(MMFactStruc):-
%% mmfacts are not inferrable, so this is the same as mobal_get_mmfact.
	cie_get_mmfact(MMFactStruc).


%%% ****************************************************************************
%%% The following predicates are called by MOBAL. 
%%% You could use them to react to these events generated by MOBAL.
%%% Simply define a clause body for the event you want to handle.
%%% The event handler should not do much computation. It is recommended to
%%% only remember the event within the event handler, and process it, when MOBAL
%%% returns from the original call to MOBAL.
%%%
%%% ATTENTION !!!  YOU MUST NOT CALL MOBAL WITHIN YOUR EVENT HANDLER!!!!
%%% 
%%% MOBAL has not finished its processing, when these 
%%% hooks are called. So inpredictable results 
%%% (including infinite loops and system crashes) can happen if you 
%%% do not respect this.
%%%
%%% These Hooks are not called, if the whole knowledge base is changed, 
%%% e.g. if mobal_clear_domain, or mobal_load_domain is called.
%%% They are called for each item read from text (mobal_read_textfile)
%%% ****************************************************************************

%%% ****************************************************************************
%%% mobal_hn_message(!Object,!Message,!Args)
%%% ****************************************************************************
%%% This is called by mobal, when HyperNeWS sends a message MOBAL don't know.
%%% The arguments are passed as read by hn_read(message(Object,Message,Args))
%%% See the HyperNeWS1.4 manual/online help for a description.
%%% You can use this hook, to handle messages resulting from
%%% interface modifications/additions you have done. This hook should succeed,
%%% on succesfully handled messages, and fail, if you don't want to handle
%%% such an Message.
%%% ****************************************************************************
%%% This is an handler to print HN-Messages not handled by MOBAL:
%%% mobal_hn_message(stack(S),M,A):-
%%%	write('Stack:   '),
%%%	writeq(S),nl,
%%%	write('Message: '), writeq(M), nl,
%%%	write('Args:    '), writeq(A), nl, nl.
%%% mobal_hn_message(S:O,M,A):-
%%%	nl, write('Stack:   '),
%%%	writeq(S),write(',  Object: '), writeq(O), nl,
%%%	write('Message: '), writeq(M), nl,
%%%	write('Args:    '), writeq(A).
%%% ****************************************************************************

mobal_hn_message(_,_,_):- fail.



%%% ****************************************************************************
%%% mobal_fact_{stored | deleted | changed}
%%% ****************************************************************************
%%% These are called by mobal, when the fact knowledge changes.
%%% mobal_fact_stored is called, when the EP changes from [0,0] to something else.
%%% mobal_fact_deleted is called, when the EP changes from something else to [0,0].
%%% mobal_fact_changed is called for all other changes of the EP of this fact.
%%% Changes of other things than the EP are told by a deleted and a stored event.
%%% ****************************************************************************

mobal_fact_stored(_FactStruc).
mobal_fact_deleted(_FactStruc).
mobal_fact_changed(_FactStruc).

%%% ****************************************************************************
%%% mobal_mfact_{stored | deleted | changed}
%%% ****************************************************************************
%%% These are called by mobal, when the meta-fact knowledge changes.
%%% mobal_mfact_stored is called, when the EP changes from [0,0] to something else.
%%% mobal_mfact_deleted is called, when the EP changes from something else to [0,0].
%%% mobal_mfact_changed is called for all other changes of the EP of this mfact.
%%% Changes of other things than the EP are told by a deleted and a stored event.
%%% ****************************************************************************

mobal_mfact_stored(_MFactStruc).
mobal_mfact_deleted(_MFactStruc).
mobal_mfact_changed(_MFactStruc).

%%% ****************************************************************************
%%% mobal_mmfact_{stored | deleted | changed}
%%% ****************************************************************************
%%% These are called by mobal, when the meta-meta-fact knowledge changes.
%%% mobal_mmfact_stored is called, when the EP changes from [0,0] to something else.
%%% mobal_mmfact_deleted is called, when the EP changes from something else to [0,0].
%%% mobal_mmfact_changed is called for all other changes of the EP of this mmfact.
%%% Changes of other things than the EP are told by a deleted and a stored event.
%%% ****************************************************************************

mobal_mmfact_stored(_MMFactStruc).
mobal_mmfact_deleted(_MMFactStruc).
mobal_mmfact_changed(_MMFactStruc).

%%% ****************************************************************************
%%% mobal_pred_{stored | deleted | changed}
%%% ****************************************************************************
%%% These are called by mobal, when the predicate knowledge changes.
%%% mobal_pred_stored is called, when a predicate is first used.
%%% mobal_pred_deleted is called, when a last use of a predicate is deleted.
%%% mobal_pred_changed is called for changes of the user-sort-mask (USM).
%%% ****************************************************************************

mobal_pred_stored(_PredStruc).
mobal_pred_deleted(_PredStruc).
mobal_pred_changed(_PredStruc).

%%% ****************************************************************************
%%% mobal_mpred_{stored | deleted}
%%% ****************************************************************************
%%% These are called by mobal, when the meta-predicate knowledge changes.
%%% mobal_mpred_stored is called, when a metapredicate definition is told to MOBAL.
%%% mobal_mpred_deleted is called, when a metapredicate definition is deleted.
%%% ****************************************************************************

mobal_mpred_stored(_MPredStruc).
mobal_mpred_deleted(_MPredStruc).

%%% ****************************************************************************
%%% mobal_rule_{stored | deleted}
%%% ****************************************************************************
%%% These are called by mobal, when the rule-knowledge changes.
%%% mobal_rule_stored is called, when a rule is learned, told or infered.
%%% mobal_rule_deleted is called, when a rule is deleted.
%%% ****************************************************************************

mobal_rule_stored(_RuleStruc).
mobal_rule_deleted(_RuleStruc).

%%% ****************************************************************************
%%% mobal_mrule_{stored | deleted}
%%% ****************************************************************************
%%% These are called by mobal, when the meta-rule-knowledge changes.
%%% mobal_mrule_stored is called, when a meta-rule is told.
%%% mobal_mrule_deleted is called, when a meta-rule is deleted.
%%% ****************************************************************************

mobal_mrule_stored(_MRuleStruc).
mobal_mrule_deleted(_MRuleStruc).

%%% ****************************************************************************
%%% mobal_ic_{stored | deleted}
%%% ****************************************************************************
%%% These are called by mobal, when the integrity constraints changes.
%%% mobal_ic_stored is called, when an integrity constraint is told.
%%% mobal_ic_deleted is called, when an integrity constraint is deleted.
%%% ****************************************************************************

mobal_ic_stored(_ICStruc).
mobal_ic_deleted(_ICStruc).

%%% ****************************************************************************
%%% mobal_tnode_{stored | deleted}
%%% ****************************************************************************
%%% These are called by mobal, when the topology changes.
%%% mobal_tnode_stored is called, when a topology node is stored.
%%% mobal_tnode_deleted is called, when a topology node is deleted.
%%% ****************************************************************************

mobal_tnode_stored(_TNodeStruc).
mobal_tnode_deleted(_TNodeStruc).

%%% ****************************************************************************
%%% mobal_sclass_{stored | deleted}
%%% ****************************************************************************
%%% These are called by mobal, when the sort lattice changes.
%%% mobal_sclass_stored is called, when a sort lattice class is stored.
%%% mobal_sclass_deleted is called, when a sort lattice class is deleted.
%%% ****************************************************************************

mobal_sclass_stored(_SClassStruc).
mobal_sclass_deleted(_SClassStruc).

