%%% ============================================================================
%%% Copyright Notice
%%% ============================================================================

%%% This file is part of the MOBAL system and is NOT in the public domain. It
%%% is available to the MLT consortium partners and the EEC under the conditions
%%% of the MLT contract only and may NOT in any form be used outside MLT nor 
%%% made available to any third party without prior written permission from GMD.  

%%% (C) 1990-92 GMD (Ges. f. Mathematik und Datenverarbeitung, FIT.KI, 
%%%         Projekt MLT, Pf. 1316, 5205 St. Augustin 1, West Germany).
%%% ============================================================================


:- module(itemspec,[
%% Fact:
	pf/2,
	f/2,
	fact/3,
	fact_p/1,
	fact_pred/2,
	fact_args/2,
	fact_ep/2,
	fact_prop/2,
	fact_id/2,
	fact_comment/2,
	fact_level/2,
	make_fact/5,
	make_fact/4,
	alter_fact_pred/3,
	alter_fact_args/3,
	alter_fact_ep/3,
	alter_fact_id/3,
	print_fact/1,
	print_fact/2,
	print_fact_comment/1,
%% MFact:
	mf/2,
	mfact/3,
	mfact_p/1,
	mfact_pred/2,
	mfact_args/2,
	mfact_prop/2,
	mfact_ep/2,
	mfact_supset/2,
	mfact_id/2,
	mfact_comment/2,
%	mfact_predargs/2,
	get_pred_args/3,
	make_mfact/6,
	make_mfact/5,
	alter_mfact_supset/3,
	alter_mfact_ep/3,
	print_mfact/1,
	print_mfact_comment/1,
%% MMFact:
	mmf/2,
	mmfact/3,
	mmfact_p/1,
	mmfact_pred/2,
	mmfact_args/2,
	mmfact_ep/2,
	mmfact_id/2,
	mmfact_comment/2,
	mmfact_prop/2,
	make_mmfact/4,
	make_mmfact/5,
	alter_mmfact_ep/3,
	print_mmfact/1,
%% Pred:
	p/2,
	pred/3,
	pred_p/1,
	pred_name/2,
	pred_definition/2,
	pred_header/2,
	pred_asm/2,
	pred_usm/2,
	pred_arity/2,
	pred_layer/2,
	pred_comment/2,
	make_pred/7,
	make_pred/6,
	print_pred/1,
	alter_pred_name/3,
%% MPred:
	mp/2,
	mpred/3,
	mpred_p/1,
	mpred_name/2,
	mpred_args/2,
	mpred_mpred/2,
	mpred_header/2,
	mpred_rulescheme/2,
	mpred_prems/2,
	mpred_concl/2,
	mpred_supset/2,
	mpred_comment/2,
	mpred_vars/2,
	make_mpred/4,
	print_mpred/1,
	alter_mpred_rulescheme/3,
	alter_mpred_args/3,
%% MMPred:
	mmp/2,
	mmpred/3,
	mmpred_p/1,
	mmpred_name/2,
	mmpred_args/2,
	mmpred_header/2,
	mmpred_rulescheme/2,
	mmpred_prems/2,
	mmpred_concl/2,
	mmpred_supset/2,
	mmpred_comment/2,
	mmpred_vars/2,
	make_mmpred/4,
	print_mmpred/1,
	alter_mmpred_rulescheme/3,
	alter_mmpred_args/3,
%% Rule:
	rule/2,
	rule/3,
	rule_p/1,
	rule_vars/2,
	rule_prems/2,
	rule_concl/2,
	rule_supset/2,
	rule_fun/2,
	rule_id/2,
	rule_comment/2,
	rule_preds/2,
	alter_rule_prems/3,
	alter_rule_concl/3,
	alter_rule_supset/3,
	alter_rule_vars/3,
	alter_rule_id/3,
	make_rule/8,
	make_rule/7,
	make_rule/6,
	make_rule_from_prems_concl/5,
	print_rule/1,
	print_rule/2,
	read_rule/1,
	find_literal_preds/2,
	pretty_print_rule/1,
	default_supset_p/1,
	default_domains_p/1,
	print_rule_comment/1,
%	print_fun/1,
%% Literals (as structures with target points)
        lit_prop_tp/3,
%% MRule:
	mrule/2,
	mrule/3,
	mrule_p/1,
	mrule_vars/2,
	mrule_prems/2,
	mrule_concl/2,
	mrule_supset/2,
	mrule_id/2,
	alter_mrule_id/3,
	mrule_comment/2,
	make_mrule_from_prems_concl/5,
	make_mrule/6,
	print_mrule/1,
%% Supset:
	su/2,
	supset_p/1,
	supset_domains/2,
	supset_exceptions/2,
	supset_domain_exceptions/3,
	make_supset/3,
	alter_supset_exceptions/3,
	alter_supset_domain_exceptions/4,
	read_supset/1,
	print_supset/1,
	print_supset/2,
%% Topology:
        tnode/2,
	tnode/3,
	tnode_basicnode_p/1,
	tnode_p/1,
	tnode_name/2,
	tnode_id/2,
	tnode_comment/2,
	tnode_predicates/2,
	tnode_links/2,
	tnode_topology/2,
        print_tnode/1,
%% Domain:
	do/2,
	domain_p/1,
	make_domain/3,
	domain_concept/2,
	domain_exceptions/2,
	alter_domain_exceptions/3,
	alter_domain_concept/3,
	print_domain/1,
%% Model:
	model_p/1,
	make_model/4,
	model_name/2,
	model_meta_level/2,
	model_object_level/2,
%% Sorts:
        s/2,
	sclass/3, %% The parsing DCG
	sclass_p/1,
	make_sclass/4,
	make_sclass/6,
	sclass_name/2,
	sclass_sorts/2,
	sclass_ext/2,
	sclass_subs/2,
	sclass_supers/2,
	print_sclass/1,
%% User Sorts:
        us/2,
	usersub/3, % the parsing DCG
	usersub_p/1,
	usersub_sub/2,
	usersub_rel/2,
	usersub_supers/2,
	usersub_super/2,
	make_usersub/4,
	print_usersub/1,
%% Agenda entries:
        ae_p/1,
	ae_type/2,
	ae_arg/2,
	ae_check/2,
	ae_trigger/2,
	ae_handle/2,
	print_ae/1,
%% Stuff:
	print_list/2,
	print_list/3,
	print_tuple/1,
	print_tuple/2,
	print_set/1,
	print_set/2,
	structure_id/2,
	read_object/3,
	parse_string/3,
	denotes_constant/2,
	denotes_variable/2,
	tokenize/2,
	set_variable_syntax/1,
	variable_syntax/1,
%% parse DCGs
        number_string/3,
	int_string/3,
	digit_char/3,
        any/2,
        any/3,
%% Ep:
        ep/2,
	read_ep/1,
	ep_mapping/2,
%% Premises:
	premises/2,
	read_premises/1,
%% generics for items
        item_comment/2,
%% IC:  
        ic/2,
	ic/3,
	ic_p/1,
	ic_vars/2,
	ic_prems/2,
	ic_concls/2,
	ic_comment/2,
	alter_ic_prems/3,
	alter_ic_concls/3,
	print_ic/1,
	pretty_print_ic/1,
	print_ic_comment/1
]).

:- meta_predicate print_list(+,:),
	print_list(+,:,+),
	print_set(+,:),
	print_tuple(+,:).

:-use_module(library(compatibility)).
:-use_module(library(caseconv)).
:- use_module(kitcore,library(kitcore), [
	conc/3,
	member/2,
	codelist/2,
	not/1,
	outterm/1,
	outtermq/1,
	ttyreadterm/1
   ]).

:- use_module(var,library(var), [
	blip_control/2
   ]).

:- use_module(set,library(set), [
	union/3
   ]).

:- use_module(list,library(list), [
	connect_list_with/3,
	position_p/3,
        replace_nth/4
   ]).

:- use_module(lineio,library(lineio),[get_line/2]).

%% Don't use library(msg) !!!!!!!!!! 
%% The quintus compiler has problems with zyklic uses.
%% Please, for ortogonality use only the kitcore predicate outterm 
%% and the built-in predicate tab(N) and nl within printfunctions!!!!
%% (Clearly, you could also use other printfunctions defined here.)



%%% "item-spec.prolog"
%%% Defines the format of item specs and supplies prodecures for
%%% assembling them.

%%% The purpose of introducing a standard item spec format is to
%%% allow the use of generic (e.g., focus) functions on facts, predicates,
%%% etc. and to minimize interface problems between the different system
%%% parts. The ItemSpecs defined here are the external specs that are to
%%% be used whenever items are communicated to other parts of the system,
%%% e.g. when sending an item to be entered into the inference engine, or
%%% sending an item to the interface for display.

%%% Note that by using structures, it is possible to use one format that
%%% may be partially filled in, which is preferable over multi-argument
%%% special-purpose lists.

%%% The slots defined here may change if it turns out that other ways of
%%% storing the items are better (i.e., storing facts as Prolog facts instead
%%% of breaking them up into pred and args). Thus, you should make sure
%%% that you NEVER access the internals of a structure directly - ALWAYS use
%%% the accessors for assembling and disassembling them.
%%% (Changing the internal structure will then only require rewriting the
%%% accessors.)

%%% Each named structure should be "defined" by asserting a descriptor fact:
%%% "<internal-name>(<long-name>,<slot-list>)"
%%% where <internal-name> is the name used in the
%%% object's internal representation,
%%% <long-name> is the user-visible name used for the accessors, <slot-list> is
%%% the list of slots. Also special pretty printing functions
%%% are declared for each structure. These are used in "msg.prolog".


%%% ****************************************************************************
%%% tables about structures
%%% ****************************************************************************

%%% structure print-function table

pf(su,print_supset).
pf(rule,print_rule).
pf(mrule,print_mrule).
pf(abstractedrule,print_rule).
pf(do,print_domain).
pf(f,print_fact).
pf(p,print_pred).
pf(mf,print_mfact).
pf(mmf,print_mmfact).
pf(derivation,print_derivation).
pf(mp,print_mpred).
pf(mmp,print_mmpred).
pf(tnode,print_tnode).
pf(s,print_sclass).
pf(us,print_usersub).
pf(ae,print_ae).
pf(ic,print_ic).				


%%% read procedures are asserted as facts with the structure's functor, eg.
%%% su(read_procedure,read_supset)

%%% ****************************************************************************
%%% Generics (accessors that work on most items)
%%% ****************************************************************************

item_comment(Item,Comment):-
	fact_comment(Item,Comment), !.
item_comment(Item,Comment):-
	mfact_comment(Item,Comment), !.
item_comment(Item,Comment):-
	mmfact_comment(Item,Comment), !.
item_comment(Item,Comment):-
	pred_comment(Item,Comment), !.
item_comment(Item,Comment):-
	mpred_comment(Item,Comment), !.
item_comment(Item,Comment):-
	mmpred_comment(Item,Comment), !.
item_comment(Item,Comment):-
	rule_comment(Item,Comment), !.
item_comment(Item,Comment):-
	tnode_comment(Item,Comment), !.
item_comment(Item,Comment):-
	ic_comment(Item,Comment), !.
						

%%% ****************************************************************************
%%% Fact
%%% ****************************************************************************

%%% Fact
%%%    prop            the proposition
%%%    ep              the evidence point
%%%    id              the internal id
%%%    comment       the comment

f(fact,[prop,ep(ep),id,comment]).

fact_p(f(_,_,_,_)).
fact_pred(f(Prop,_,_,_),Pred):- Prop =.. [Pred|_], !.
fact_pred(Mfact,Pred):- mfact_pred(Mfact,Pred), !.
fact_pred(MMfact,Pred):- mmfact_pred(MMfact,Pred).
fact_args(f(Prop,_,_,_),Args):- Prop =.. [_|Args], !.
fact_args(Mfact,Args):- mfact_args(Mfact,Args), !.
fact_args(MMfact,Args):- mmfact_args(MMfact,Args).
fact_ep(f(_,EP,_,_),EP):- !.
fact_ep(Mfact,Ep):- mfact_ep(Mfact,Ep), !.
fact_ep(MMfact,Ep):- mmfact_ep(MMfact,Ep).
fact_prop(f(Prop,_,_,_),Prop):- !.
fact_prop(MF,Prop):- mfact_prop(MF,Prop), !.
fact_prop(MMF,Prop):- mmfact_prop(MMF,Prop).

fact_id(f(_,_,ID,_),ID):- !.
fact_id(MF,Id):- mfact_id(MF,Id), !.
fact_id(MMF,Id):- mmfact_id(MMF,Id).
fact_comment(f(_,_,_,Comment),Comment):- !.
fact_comment(MF,Comment):- mfact_comment(MF,Comment), !.
fact_comment(MMF,Comment):- mmfact_comment(MMF,Comment).
fact_level(f(_,_,_,_),object_level1):- !.
fact_level(MF,meta_level):- mfact_p(MF), !.
fact_level(MMF,meta_meta_level):- mmfact_p(MMF), !.
%% comment doesn't exist. so this one will do it:
make_fact(f(Prop,EP,ID,[]),Prop,EP,ID,[]):- !.
%%make_fact(f(Prop,EP,ID,RealComment),Prop,EP,ID,Comment):-
%%        var(Comment),
%%        (comment(ID,RealComment),not(var(RealComment));
%%         RealComment=[]), !.
%%make_fact(f(Prop,EP,ID,Comment),Prop,EP,ID,RealComment):-
%%        var(Comment),
%%        (comment(ID,RealComment),not(var(RealComment));
%%         RealComment=[]), !.
make_fact(f(Prop,EP,ID,Comment),Prop,EP,ID,Comment):- !.
make_fact(f(Prop,EP,ID,[]),Prop,EP,ID):- !.

alter_fact_pred(f(Prop,EP,ID,Comment),NewPred,f(NewProp,EP,ID,Comment)):-
        Prop =.. [_|Args],
        NewProp =.. [NewPred|Args].

alter_fact_args(f(Prop,EP,ID,Comment),NewArgs,f(NewProp,EP,ID,Comment)):-
        Prop =.. [Pred|_],
        NewProp =.. [Pred|NewArgs].

alter_fact_ep(f(P,_,I,C),NewEP,f(P,NewEP,I,C)).

alter_fact_id(f(P,E,_,C),NewID,f(P,E,NewID,C)).


print_fact(FactS):-
	fact_prop(FactS,Prop),
        % fact_id(FactS,ID),
	fact_ep(FactS,EP),
% 	(nonvar(ID),
%                 outterm(ID), outterm(': ')
%         ;
%                 true),
	print_fact1(Prop,EP), !.

print_fact1(Prop,[1000,0]):-
	print_prop(Prop), !.
print_fact1(Prop,[0,1000]):-
	print_prop(not(Prop)), !.
print_fact1(Prop,EP):-
	print_prop(Prop), outterm(' - '), outterm(EP), !.

print_prop(Prop):-
	outtermq(Prop).

print_fact(FactS,Indentation):-
        fact_prop(FactS,Fact),
        fact_id(FactS,FactID),
        (fact_ep(FactS,[0,1000]),
                PrintFact = not(Fact)
        ;
                PrintFact = Fact),
        tab(Indentation),
        (nonvar(FactID),
                outterm(FactID)
        ;
                true),
        outterm(': '),outterm(PrintFact),
        (fact_ep(FactS,[1000,1000]),
                outterm(', widerspruechlich')
        ;
                true),
        (blip_control(print_comment_p,t),
        print_fact_comment(FactS)
        ;
                true).

print_fact_comment(FactS):-
        fact_comment(FactS,Comment),
        (not(Comment=[]),
                nl,outterm(Comment),nl
        ;
                true).

%%% the DCG

fact(f(Prop,[0,1000],_,_)) --> ['not','('], ground_proposition(Prop), [')'].
fact(f(Prop,EP,_,_)) --> ground_proposition(Prop), ['-'], ep(EP).
fact(f(Prop,EP,_,_)) --> ground_proposition(Prop), [':'], ep(EP).
fact(f(Prop,[1000,0],_,_)) --> ground_proposition(Prop).


ground_proposition(Prop) --> [Predicate], ['('], ground_arglist(Args), [')'],
	              {Prop =.. [Predicate|Args]}.
ground_arglist([Arg|Rest]) --> ground_argument(Arg), [','], ground_arglist(Rest).
ground_arglist([Arg]) --> ground_argument(Arg).


ground_argument(Arg) --> atom(Arg).
ground_argument(Arg) --> number(Arg).

%% A Hack for accepting Lists as arguments.
ground_argument('[]') --> ['['], [']'].
ground_argument(Arg) --> ['['], ground_arglist(Arg), [']'].

ground_argument(Arg) --> atom(F),['('],ground_arglist(Args),[')'],
	              {Arg =.. [F|Args]}.
proposition(Prop) --> [Predicate], ['('], arglist(Args), [')'],
	              {Prop =.. [Predicate|Args]}.
arglist([Arg|Rest]) --> atomic_object(Arg), [','], arglist(Rest).
arglist([Arg]) --> atomic_object(Arg).


%%% ****************************************************************************
%%% ep (pseudo type)
%%% ****************************************************************************

ep(read_procedure,read_ep).

read_ep(EP):-
	ttyreadterm(Term),
	!,
	ep_mapping(Term,EP).

ep_mapping(not,[0,1000]).
ep_mapping(f,[0,1000]).
ep_mapping(false,[0,1000]).
ep_mapping(t,[1000,0]).
ep_mapping(true,[1000,0]).
ep_mapping(both,[1000,1000]).
ep_mapping(c,[1000,1000]).
ep_mapping(contradictory,[1000,1000]).
ep_mapping(b,[1000,1000]).
ep_mapping([X,Y],[X,Y]).

ep([1000,0]) --> ['true'] | ['t'].
ep([0,1000]) --> ['false'] | ['f'] | ['not'].
ep([1000,1000]) --> ['contradictory'] | ['both'] | ['c'] | ['b'].
ep([X,Y]) --> ['['], integer(X), [','], integer(Y), [']'].

%%% ****************************************************************************
%%% mfact
%%% ****************************************************************************

%%% M[m]fact
%%%    pred, args:           the proposition
%%%    supset:          the support set
%%%    ep:             the evidence point
%%%    id:             internal id
%%%    comment:      the comment for the rule

mf(mfact,[pred,args,su(supset),ep(ep),id,comment]).

mfact_p(mf(_,_,_,_,_,_)).
mfact_pred(mf(Pred,_,_,_,_,_),Pred).
mfact_args(mf(_,Args,_,_,_,_),Args).
mfact_prop(mf(P,A,_,_,_,_),Prop):- Prop =.. [P|A], !.
mfact_ep(mf(_,_,_,EP,_,_),EP).
mfact_supset(mf(_,_,SupSet,_,_,_),SupSet).
mfact_id(mf(_,_,_,_,ID,_),ID).
mfact_comment(mf(_,_,_,_,_,Comment),Comment).



get_pred_args([],[],[]):-!.
get_pred_args([F|R],[_|RR],NR):-
        conc(c,_,F),
        get_pred_args(R,RR,NR).
get_pred_args([_|R],[FF|RR],[FF|NR]):-
        get_pred_args(R,RR,NR).

%% with comment
make_mfact(mf(P,A,S,EP,ID,[]),Prop,S,EP,ID,Comment):-
        var(Comment),
        Prop=..[P|A],
        !.
make_mfact(mf(P,A,S,EP,ID,Comment),Prop,S,EP,ID,[]):-
        var(Comment),
        Prop=..[P|A],
        !.
make_mfact(mf(P,A,S,EP,ID,Comment),Prop,S,EP,ID,Comment):-
        Prop=..[P|A],
        !.

%% without comment
make_mfact(mf(P,A,S,EP,ID,[]),Prop,S,EP,ID):-
        Prop =.. [P|A],
        !.

alter_mfact_supset(mf(Pred,Args,_,EP,ID,Comment),SupSet,
                        mf(Pred,Args,SupSet,EP,ID,Comment)):- !.

alter_mfact_ep(mf(P,A,S,_,I,C),NewEP,mf(P,A,S,NewEP,I,C)).


print_mfact(Mfact):-
        mfact_ep(Mfact,[1000,0]),
        !,
        mfact_prop(Mfact,Prop),
        outterm(Prop),
	print_mfact_supset(Mfact),
	print_mfact_comment(Mfact).
print_mfact(Mfact):-
        mfact_ep(Mfact,[0,1000]),
        !,
        mfact_prop(Mfact,Prop),
        outterm(not(Prop)),
        print_mfact_supset(Mfact),
	print_mfact_comment(Mfact).
print_mfact(Mfact):-
        mfact_ep(Mfact,[1000,1000]),
        !,
        mfact_prop(Mfact,Prop),
        outterm(Prop),outterm(' - contradictory'),
        print_mfact_supset(Mfact),
	print_mfact_comment(Mfact).
print_mfact(Mfact):-
        mfact_ep(Mfact,[0,0]),
        !,
        mfact_prop(Mfact,Prop),
        outterm(Prop),outterm(' - deleted'),
        print_mfact_supset(Mfact),
	print_mfact_comment(Mfact).
print_mfact(Mfact):-
        !,
        mfact_prop(Mfact,Prop),
        mfact_ep(Mfact,EP),
        outterm(Prop),outterm(' - '),outterm(EP),
        print_mfact_supset(Mfact),
	print_mfact_comment(Mfact).

print_mfact_supset(Mfact):-
	mfact_supset(Mfact,Supset),
        (supset_p(Supset),
	 not(default_supset_p(Supset)),
                !,
                outterm(' - '),
                print_supset(Supset)
        ; true).
	
print_mfact_comment(Mfact):-
        blip_control(print_comment_p,t),
        mfact_comment(Mfact,Comment),
        (not(Comment=[]),
            nl,outterm(Comment),nl
        ;
         true).
print_mfact_comment(_).


mfact(mf(Pred,Args,Supset,[0,1000],_,_)) --> ['not','('], ground_proposition(Prop), [')'],
	                              {Prop =.. [Pred|Args]}, supset_attribute(Supset).
mfact(mf(Pred,Args,Supset,EP,_,_)) --> ground_proposition(Prop), ['-'], ep(EP),
	                          {Prop =.. [Pred|Args]}, supset_attribute(Supset).
mfact(mf(Pred,Args,Supset,[1000,0],_,_)) --> ground_proposition(Prop),
	                          {Prop =.. [Pred|Args]}, supset_attribute(Supset).

supset_attribute(Supset) --> ['-'], supset(Supset), !.
supset_attribute(_) --> [].

%%% ****************************************************************************
%%% mmfact
%%% ****************************************************************************

mmf(mmfact,[pred,args,ep(ep),id,comment]).

mmfact_p(mmf(_,_,_,_,_)).
mmfact_pred(mmf(Pred,_,_,_,_),Pred).
mmfact_args(mmf(_,Args,_,_,_),Args).
mmfact_ep(mmf(_,_,EP,_,_),EP).
mmfact_id(mmf(_,_,_,ID,_),ID).
mmfact_comment(mmf(_,_,_,_,Comment),Comment).
mmfact_prop(mmf(Pred,Args,_,_,_),MMFact):- MMFact =.. [Pred|Args], !.

make_mmfact(mmf(P,A,EP,ID,[]),Prop,EP,ID):- Prop =.. [P|A], !.
make_mmfact(mmf(P,A,EP,ID,Comment),Prop,EP,ID,Comment):- Prop =.. [P|A], !.

alter_mmfact_ep(mmf(P,A,_,I,C),NewEP,mmf(P,A,NewEP,I,C)).
alter_mmfact_args(mmf(P,_,E,I,C),NewArgs,mmf(P,NewArgs,E,I,C)).

print_mmfact(Mmfact):-
        mmfact_ep(Mmfact,[1000,0]),
        !,
        mmfact_prop(Mmfact,Prop),
        outterm(Prop).
print_mmfact(Mmfact):-
        mmfact_ep(Mmfact,[0,1000]),
        !,
        mmfact_prop(Mmfact,Prop),
        outterm(not(Prop)).
print_mmfact(Mmfact):-
        mmfact_ep(Mmfact,[1000,1000]),
        !,
        mmfact_prop(Mmfact,Prop),
        outterm(not(Prop)),outterm(' - contradictory').
print_mmfact(Mmfact):-
        mmfact_ep(Mmfact,[0,0]),
        !,
        mmfact_prop(Mmfact,Prop),
        outterm(Prop),outterm(' - deleted').
print_mmfact(Mmfact):-
        !,
        mmfact_prop(Mmfact,Prop),
        mmfact_ep(Mmfact,EP),
        outterm(Prop),outterm(' - '),outterm(EP).

mmfact(mmf(Pred,Args,[0,1000],_,_)) --> ['not','('], ground_proposition(Prop), [')'],
	                                {Prop =.. [Pred|Args]}.
mmfact(mmf(Pred,Args,EP,_,_)) --> ground_proposition(Prop), ['-'], ep(EP),
	                                {Prop =.. [Pred|Args]}.
mmfact(mmf(Pred,Args,[1000,0],_,_)) --> ground_proposition(Prop),
	                                {Prop =.. [Pred|Args]}.
%%% ****************************************************************************
%%% Pred
%%% ****************************************************************************

%%% Pred
%%%   name:         predicate name
%%%   arity:         arity
%%%   usm:          User-specified Sort Mask
%%%   asm:          Automatically-generated Sort Mask
%%%   layer:         layer
%%%   comment:     a comment string/atom

p(pred,[name,arity,usm,asm,layer,comment]).

pred_p(p(_,_,_,_,_,_)).
pred_name(p(Name,_,_,_,_,_),Name).
pred_definition(p(Name,_,USM,_,_,_),Definition):- Definition=..[Name|USM].
pred_header(p(Name,_,USM,_,_,_),Definition):- Definition=..[Name|USM].
pred_asm(p(_,_,_,ASM,_,_),ASM).
pred_usm(p(_,_,USM,_,_,_),USM).
pred_arity(p(_,Arity,_,_,_,_),Arity).
pred_layer(p(_,_,_,_,Layer,_),Layer).
pred_comment(p(_,_,_,_,_,Comment),Comment).

alter_pred_name(p(_,A,USM,ASM,L,C),NewName,p(NewName,A,USM,ASM,L,C)).

make_pred(p(Name,Arity,USM,ASM,NewLayer,Comment),Name,Arity,USM,ASM,
                                                         Layer,Comment):-
        var(Layer),
        NewLayer=[],
        !.
make_pred(p(Name,Arity,USM,ASM,Layer,Comment),
            Name,Arity,USM,ASM,Layer,Comment):- !.

make_pred(p(Name,Arity,USM,ASM,NewLayer,Comment),
                Definition,Arity,ASM,Layer,Comment):-
         var(Layer),
         NewLayer=[],
         Definition=..[Name|USM],
        !.

make_pred(p(Name,Arity,USM,ASM,Layer,Comment),
              Definition,Arity,ASM,Layer,Comment):-
        Definition=..[Name|USM],
        !.

print_pred(Pred):-
        pred_name(Pred,Name),
        pred_usm(Pred,USM),
        pred_arity(Pred,Arity),
        outterm(Name),outterm('/'),outterm(Arity),outterm(': '),
        print_predargs(USM).

print_pred2(Pred):-
	pred_name(Pred,Name),
        pred_usm(Pred,USM),
        outterm(Name),outterm('('),
        print_predargs(USM),outterm(').').

%% If an arg position is [] or unbound, there is no
%% user-defined name yet - just
%% print the brackets (<>):
print_predargs([[]]):-
	%% last element
	outterm('<>'),
	!.
print_predargs([F]):-
	%% last element
	outterm('<'),outterm(F),outterm('>'),
        !.
print_predargs([[]|R]):-
        !,
        outterm('<>'),
	outterm(', '), print_predargs(R), !.
print_predargs([F|R]):-
        outterm('<'),outterm(F),outterm('>'),
        outterm(', '), print_predargs(R), !.

%% We must also be able to read the output format, JUK, 24.8.90
pred(p(Name,Arity,USM,_ASM,_Layer,_)) --> [Name], ['/'], [Arity], [':'], sortlist(USM).

pred(p(Name,Arity,USM,_ASM,_Layer,_)) --> [Name], ['('], sortlist(USM), 
	[')'], {length(USM,Arity)}.

sortlist([Sort|Rest]) --> sort(Sort), [','], sortlist(Rest).
sortlist([Sort]) --> sort(Sort).

sort(Sortname) --> ['<', Sortname, '>'].
sort([]) --> ['<',  '>'].


%%% ****************************************************************************
%%% mpred
%%% ****************************************************************************	

%%% M[m]pred
%%%    name:         predicate name
%%%    args:         arguments in the header, not including the support set arg
%%%    rulescheme:   a rule structure

%%% format of those entries is: structure-name(<slot-list>,<print-function>).
mp(mpred,[name,args,rule(rulescheme)]).

mpred_p(mp(_,_,_)).
mpred_name(mp(Name,_,_),Name).
mpred_args(mp(_,Args,_),Args).
mpred_mpred(mp(Name,Args,_),MPred):- MPred =.. [Name|Args].
mpred_header(mp(Name,Args,_),Header):- Header =.. [Name|Args].
mpred_rulescheme(mp(_,_,Rulescheme), Rulescheme).
mpred_prems(mp(_,_,Rulescheme),Prems):- rule_prems(Rulescheme,Prems).
mpred_concl(mp(_,_,Rulescheme),Concl):- rule_concl(Rulescheme,Concl).
mpred_supset(mp(_,_,Rulescheme),SupSets):- rule_supset(Rulescheme,SupSets).
mpred_comment(mp(_,_,Rulescheme),Comment):-rule_comment(Rulescheme,Comment).
mpred_vars(MP,Vars):-
	mpred_args(MP,Args),
	mpred_rulescheme(MP,RS),
	rule_vars(RS,RVars),
	union(Args,RVars,Vars).

make_mpred(mp(Name,Args,Rulescheme),Header,Prems,Concl):-
                Header =.. [Name|Args],
                make_rule(Rulescheme,_,Prems,Concl,_,_),!.

alter_mpred_rulescheme(mp(Name,Args,_),RS,mp(Name,Args,RS)).
alter_mpred_args(mp(Name,_,RS),Args,mp(Name,Args,RS)).

print_mpred(Mpred):-
        mpred_header(Mpred,Header),
	mpred_vars(Mpred,Vars),
        mpred_prems(Mpred,Prems),
        mpred_concl(Mpred,Concl),
	print_literal(Header,Vars),
	outterm(' : '),
	print_rule_prems(Prems,Vars),
	outterm(' --> '),print_literal(Concl,Vars).


mmp(mmpred,[name,args,rule(rulescheme)]).

mmpred_p(mmp(_,_,_)).
mmpred_name(mmp(Name,_,_),Name).
mmpred_args(mmp(_,Args,_),Args).
mmpred_header(mmp(Name,Args,_),Header):- Header =.. [Name|Args].
mmpred_rulescheme(mmp(_,_,Rulescheme), Rulescheme).
mmpred_prems(mmp(_,_,Rulescheme),Prems):- rule_prems(Rulescheme,Prems).
mmpred_concl(mmp(_,_,Rulescheme),Concl):- rule_concl(Rulescheme,Concl).
mmpred_supset(mmp(_,_,Rulescheme),SupSets):- rule_supset(Rulescheme,SupSets).
mmpred_comment(mmp(_,_,Rulescheme),Comment):-rule_comment(Rulescheme,Comment).
mmpred_vars(MMP,Vars):-
	mmpred_args(MMP,Args),
	mmpred_rulescheme(MMP,RS),
	rule_vars(RS,RVars),
	union(Args,RVars,Vars).

make_mmpred(mmp(Name,Args,Rulescheme),Header,Prems,Concl):-
                Header =.. [Name|Args],
                make_rule(Rulescheme,_,Prems,Concl,_,_),!.

alter_mmpred_rulescheme(mmp(Name,Args,_),RS,mmp(Name,Args,RS)).
alter_mmpred_args(mmp(Name,_,RS),Args,mmp(Name,Args,RS)).

print_mmpred(Mpred):-
        mmpred_header(Mpred,Header),
	mmpred_vars(Mpred,Vars),
        mmpred_prems(Mpred,Prems),
        mmpred_concl(Mpred,Concl),
	print_literal(Header,Vars),
	outterm(' : '),
	print_rule_prems(Prems,Vars),
	outterm(' --> '),print_literal(Concl,Vars).


mpred(mp(Name,Args,Rule)) --> 
	mpheader(Prop), [':'], rule(Rule), {Prop=..[Name|Args]}.
mmpred(mmp(Name,Args,Rule)) --> 
	mpheader(Prop), [':'], rule(Rule), {Prop=..[Name|Args]}.

mpheader(Prop) --> [Predicate], ['('], mparglist(Args), [')'],
	              {Prop =.. [Predicate|Args]}.
mparglist([Arg|Rest]) --> mparg(Arg), [','], mparglist(Rest).
mparglist([Arg]) --> mparg(Arg).

mparg(Term) --> [':', Term], {atom(Term)}.
mparg(Term) --> [':'(Term)], {atom(Term)}.
mparg(Term) --> atom(Term).

%%% ****************************************************************************
%%% Rule
%%% ****************************************************************************

%%% Rule
%%%    vars:        list of variables used in the rule
%%%    prems:       list of premises (a real list, not connected by &)
%%%                    each premise in term format
%%%    concl:       the conclusion, in term format
%%%    supset:      a supset structure
%%%    fun:         the evidence combination function for the rule (usually min)
%%%    id:          internal id
%%%    comment:     an explanation for the rule

rule(rule,[vars,premises(prems),concl,su(supset),fun,id,comment]).

rule_p(rule(_,_,_,_,_,_,_)).
rule_vars(rule(Vars,_,_,_,_,_,_),Vars).
rule_prems(rule(_,Prems,_,_,_,_,_),Prems).
rule_concl(rule(_,_,Concl,_,_,_,_),Concl).
rule_supset(rule(_,_,_,SupSets,_,_,_), SupSets).
rule_fun(rule(_,_,_,_,Fun,_,_), Fun).
rule_id(rule(_,_,_,_,_,ID,_),ID).
rule_comment(rule(_,_,_,_,_,_,Comment),Comment).

%% returns all predicates that are used in a rule
rule_preds(R,Preds):-
	rule_prems(R,Prems),
	rule_concl(R,Concl),
	find_literal_preds([Concl|Prems],Preds).

find_literal_preds([],[]).
find_literal_preds([F|R],[FP|RP]):-
	functor(F,Functor,_),
	(member(Functor,[not,both,contradictory,unknown])
         -> arg(1,F,F1),
	    functor(F1,FP,_)
	 ;  FP = Functor),
	 find_literal_preds(R,RP).

alter_rule_prems(rule(Vars,_,Concl,Supsets,Fun,ID,Comment),Prems,
                   rule(Vars,Prems,Concl,Supsets,Fun,ID,Comment)).

alter_rule_concl(rule(Vars,Prems,_,Supsets,Fun,ID,Comment),Concl,
                   rule(Vars,Prems,Concl,Supsets,Fun,ID,Comment)).

alter_rule_supset(rule(Vars,Prems,Concl,_,Fun,ID,Comment),Supset,
                   rule(Vars,Prems,Concl,Supset,Fun,ID,Comment)).

alter_rule_vars(rule(_,Prems,Concl,Supset,Fun,ID,Comment),Vars,
                   rule(Vars,Prems,Concl,Supset,Fun,ID,Comment)).

alter_rule_id(rule(Vars,Prems,Concl,Supset,Fun,_,Comment),ID,
                   rule(Vars,Prems,Concl,Supset,Fun,ID,Comment)).

%% with comment and evidence function
make_rule(rule(Vars,Premslist,Concl,SupSets,Fun,ID,Comment),
                                Vars,Prems,Concl,SupSets,Fun,ID,Comment):-
        connect_list_with(Premslist,&,Prems).
make_rule(rule(Vars,Prems,Concl,SupSets,Fun,ID,Comment),Vars,Prems,
                        Concl,SupSets,Fun,ID,Comment).

%% with comment
make_rule(rule(Vars,Premslist,Concl,SupSets,_,ID,Comment),
                        Vars,Prems,Concl,SupSets,ID,Comment):-
        connect_list_with(Premslist,&,Prems).
make_rule(rule(Vars,Prems,Concl,SupSets,_,ID,Comment),
                        Vars,Prems,Concl,SupSets,ID,Comment).

%% without comment
make_rule(rule(Vars,Premslist,Concl,SupSets,_,ID,[]),
                        Vars,Prems,Concl,SupSets,ID):-
        connect_list_with(Premslist,&,Prems).
make_rule(rule(Vars,Prems,Concl,SupSets,_,ID,[]),Vars,Prems,Concl,SupSets,ID).

%%%

make_rule_from_prems_concl(Rule,Prems,Concl,Fun,Comment):-
        make_rule_from_prems_concl(Rule,Prems,Concl,Comment),
                        rule_fun(Rule,Fun),
        rule_comment(Rule,Comment).

%% only with prems and concl
make_rule_from_prems_concl(rule(_,Premslist,Concl,_,_,_,Comment),
                        Prems,Concl,Comment):-
        connect_list_with(Premslist,&,Prems), !.
make_rule_from_prems_concl(rule(_,Prems,Concl,_,_,_,Comment),
                        Prems,Concl,Comment).

read_rule(Rule):-
	get_line(RuleString,_),
	parse_string(RuleString,rule,Rule).

print_rule(RuleS):-
        print_rule(RuleS,0).
print_rule(RuleS,Indentation):-
        !,
	rule_vars(RuleS,Vars1), 
	%% must make sure not to bind Vars1 if it is variable !!
	(var(Vars1)
        -> true
        ;  Vars = Vars1),
        rule_prems(RuleS,Prems),
        rule_concl(RuleS,Concl),
        rule_supset(RuleS,Supset),
        rule_fun(RuleS,Fun),
        tab(Indentation),
%	rule_id(RuleS,RuleID),
%        (var(RuleID)
%           %% in this case we have a rulescheme, not a rule
%        ;
%           outterm(RuleID),outterm(': ')
%        ),
	print_rule_prems(Prems,Vars),
	print_rule_arrow(Fun),
	print_literal(Concl,Vars),
        (not(default_supset_p(Supset)),
                !,
                outterm(' - '),
                print_supset(Supset),
                outterm('')
        ; true),
        print_rule_comment(RuleS).

print_rule_prems(Prems):-
	print_rule_prems(Prems,_).
print_rule_prems([],_):-
	!.
print_rule_prems([Last],Vars):-
	!,
	print_literal(Last,Vars).
print_rule_prems([First|Rest],Vars):-
	print_literal(First,Vars),outterm(' & '),
	print_rule_prems(Rest,Vars).


print_literal(:(Literal,EP),Vars):-
	print_literal1(Literal,Vars),
	outterm(':'), outterm(EP).
print_literal(both(Literal),Vars):-
	outterm('both('),
	print_literal1(Literal,Vars),
	outterm(')').
print_literal(not(Literal),Vars):-
	outterm('not('),
	print_literal1(Literal,Vars),
	outterm(')').
print_literal(Literal,Vars):-
	print_literal1(Literal,Vars).

print_literal1(Atom,Vars):-
	atom(Atom),
	!,
	print_literal_arg(Atom,Vars).
print_literal1(Number,_):-
	number(Number),
	!,
	outterm(Number).
print_literal1(Literal,Vars):-
	Literal =.. [Pred|Args],
        (variable_syntax(prolog_switched)
        -> outterm(Pred)
	;  print_literal_arg(Pred,Vars)),
        outterm('('),
	print_literal_args(Args,Vars),
	outterm(')'),
	!.

print_literal_args([],_):- !.
print_literal_args([F],Vars):-
	print_literal(F,Vars),
	!.
print_literal_args([F|R],Vars):-
	print_literal(F,Vars),
	outterm(', '),
	print_literal_args(R,Vars).

print_literal_arg(F,Vars):-
	member(F,Vars),
	!,
	(variable_syntax(prolog),
	    mixed(F,FM),
            outterm(FM)
        ;
	    outterm(F)).
print_literal_arg(F,_Vars):-
	%% a constant
        (variable_syntax(prolog_switched),
	    mixed(F,FM),
            outterm(FM)
        ;
	    outterm(F)).
	

default_supset_p(Supset):-
        supset_exceptions(Supset,[]),
        supset_domains(Supset,Domains),
        default_domains_p(Domains).

default_domains_p([]):- !.
default_domains_p([First|Rest]):-
        domain_concept(First,all),
        domain_exceptions(First,[]),
        default_domains_p(Rest).

print_rule_comment(RuleS):-
        blip_control(print_comment_p,t),
%        rule_id(RuleS,ID),
        rule_comment(RuleS,Comment),
        (not(Comment=[]),
                nl,outterm(Comment),nl
        ;
                true).
print_rule_comment(_).

print_rule_arrow(min):-
	!,
	outterm(' --> ').
print_rule_arrow(V):-
	var(V),
	!,
	outterm(' --> ').
print_rule_arrow(Fun):-
	!,
	outterm(' -('), outterm(Fun), outterm(')-> ').

pretty_print_rule(Rule):-
	rule_prems(Rule,Prems),
	rule_concl(Rule,Concl),
	rule_comment(Rule,Comment),
	rule_supset(Rule,Supset),
	rule_id(Rule,ID),
	rule_fun(Rule,Fun),
	rule_vars(Rule,Vars),
	outterm('Rule ID:   '),outterm(ID),nl,
	outterm('Variables: '),outterm(Vars),nl,nl,
	outterm('    '),pretty_print_rule_prems(Prems),nl,
	outterm('--> '),outterm(Concl),nl,nl,
	outterm('Support Set:       '),outterm(Supset),nl,
	outterm('Evidence Function: '),outterm(Fun),nl,
	outterm('Comment:           '),outterm(Comment),nl,
	!.

pretty_print_rule_prems([]).
pretty_print_rule_prems([Last]):-
	!,
	outterm(Last).
pretty_print_rule_prems([First|Rest]):-
	outterm(First),nl,outterm('  & '),
	pretty_print_rule_prems(Rest).


rule(rule(_Vars,Prems,Concl,Supset,Fun,_,_)) --> premises(Prems), 
	rule_arrow(Fun),
	premise(Concl),
	supset_attribute(Supset).
rule(rule(_Vars,Prems,Concl,Supset,Fun,_,_)) --> [_ID, ':'], premises(Prems), 
	rule_arrow(Fun),
	structure(Concl),
	supset_attribute(Supset).

rule_arrow(min) --> ['--', '>'].
rule_arrow(Fun) --> ['-', '('], atom(Fun), [')', '-', '>']. 


%%% ****************************************************************************
%%% premises (Pseudo type)  / Literals
%%% ****************************************************************************

premises(read_procedure,read_premises).

read_premises(PremsList):-
	ttyreadterm(Prems),
	connect_list_with(PremsList,&,Prems).

premises(P) --> listcore([]), {P = []}.
premises(P) --> premise(P1), {P = [P1]}.
premises(P) --> premise(P1), ['&'], premises(P2),
 			{P = [P1|P2]}.

premise(P) --> structure(P).
premise(:(P,EP)) --> structure(P), [':'], ep(EP).
premise(:(P,EP)) --> structure(P), ['-'], ep(EP).

%%% returns the proposition and tp of a possibly compound literal

lit_prop_tp(Prop1,Prop2,[1000,0]):- var(Prop1), !, Prop1=Prop2.
lit_prop_tp(not(Prop),Prop,[0,1000]):- !.
lit_prop_tp(both(Prop),Prop,[1000,1000]):- !.
lit_prop_tp(:(Prop,TP),Prop,TP):- !.
lit_prop_tp(Prop,Prop,[1000,0]):- !.

%%% ****************************************************************************
%%% mrule
%%% ****************************************************************************

mrule(mrule,[vars,prems,concl,su(supset),id,comment]).
mrule_p(mrule(_,_,_,_,_,_)).
mrule_vars(mrule(Vars,_,_,_,_,_),Vars).
mrule_prems(mrule(_,Prems,_,_,_,_),Prems).
mrule_concl(mrule(_,_,Concl,_,_,_),Concl).
mrule_supset(mrule(_,_,_,SupSets,_,_), SupSets).
mrule_id(mrule(_,_,_,_,ID,_),ID).
mrule_comment(mrule(_,_,_,_,_,Com),Com).

alter_mrule_id(mrule(Vars,Prems,Concl,Supset,_,Comment),ID,
                   mrule(Vars,Prems,Concl,Supset,ID,Comment)).


make_mrule_from_prems_concl(mrule(_,Premslist,Concl,[],_,_),Prems,Concl,_,_):-
        connect_list_with(Premslist,&,Prems).
make_mrule(mrule(Vars,Premslist,Concl,SupSets,ID,_),Vars,Prems,Concl,SupSets,ID):-
        connect_list_with(Premslist,&,Prems).
make_mrule(mrule(Vars,Premslist,Concl,SupSets,ID,Com),Vars,Prems,Concl,SupSets,ID,Com):-
        connect_list_with(Premslist,&,Prems).

print_mrule(MruleS):-
        print_mrule(MruleS,0).
print_mrule(MruleS,Indentation):-
        !,
%        mrule_id(MruleS,RuleID),
	mrule_vars(MruleS,Vars),
        mrule_prems(MruleS,Prems),
        mrule_concl(MruleS,Concl),
	tab(Indentation),
%        (var(RuleID),
%           %% in this case we have a rulescheme, not a rule
%        ;
%           outterm(RuleID),outterm(': ')),
	print_rule_prems(Prems,Vars),
	outterm(' --> '),print_literal(Concl,Vars).

:- dynamic mrule/3.

%%% still to be defined,
%%% this is a first try, juk.
mrule(mrule(_,Prems,Concl,Supset,_,_)) --> opt_mrule_id, premises(Prems), ['--', '>'],
	structure(Concl),
	supset_attribute(Supset).

opt_mrule_id --> [_ID, ':'].
opt_mrule_id --> [].

%%% ****************************************************************************
%%% Supset
%%% ****************************************************************************

%%% a SUPSET consists of DOMAINS of applicability
%%% for the various variables in the rule.

%%% Supset
%%%    domains:      list of domain structures
%%%    exceptions:   list of exception tuples (tuples in list form)

su(supset,[do(domains),exceptions]).
su(read_procedure,read_supset).

supset_p(su(_,_)).
supset_domains(su(D,_),D).
supset_exceptions(su(_,E),E).

make_supset(su(D,E),D,E).

supset_domain_exceptions(S,I,LE):-
	supset_domains(S,Ds),
	position_p(D,Ds,I),
	domain_exceptions(D,LE).

alter_supset_exceptions(su(D,_),E,su(D,E)).

alter_supset_domain_exceptions(su(D,GE),I,LE,su(ND,GE)):-
	position_p(Domain,D,I),
	alter_domain_exceptions(Domain,LE,NewDomain),
	replace_nth(D,I,NewDomain,ND).

read_supset(Supset):-
	get_line(SupsetString,_),
	parse_string(SupsetString,supset,Supset).

print_supset(Supset):-
        print_supset(Supset,0).
print_supset(Supset,Indentation):-
        tab(Indentation),
        supset_domains(Supset,Domains),
	outterm('('),
        print_domains(Domains),
	outterm(')'),
        supset_exceptions(Supset,Exceptions),
        (Exceptions = [],
                !
        ;
                outterm('  \  {'),
                print_tuples(Exceptions),
                outterm('}')),
        !.

print_domains([Last]):-
        !,
        print_domain_with_parentheses(Last).
print_domains([First|Rest]):-
        !,
        print_domain_with_parentheses(First),
        outterm(' * '),
        print_domains(Rest).


print_domain_with_parentheses(Domain):-
        (domain_exceptions(Domain,[]),
                !,
                print_domain(Domain)
        ;
                outterm('('),
                print_domain(Domain),
                outterm(')')).

print_tuples([]):- !.
print_tuples([Last]):-
        print_tuple(Last),!.
print_tuples([First|Rest]):-
        print_tuple(First),
        outterm(','),
        print_tuples(Rest),
        !.


%%% ****************************************************************************
%%% Domain
%%% ****************************************************************************

%%% Domain
%%%    concept          name of the core concept of the domain (an atom)
%%%    exceptions       list of exception terms (atoms)

do(domain,[concept,exceptions]).

domain_p(do(_,_)).
make_domain(do(C,E),C,E).
domain_concept(do(Concept,_),Concept).
domain_exceptions(do(_,Exceptions),Exceptions).

alter_domain_exceptions(do(C,_),E,do(C,E)).
alter_domain_concept(do(_,E),C,do(C,E)).

print_domain(Domain):-
        make_domain(Domain,Concept,Exceptions),
        outterm(Concept),
        (Exceptions = [],
                !
        ;
                outterm('\'),
                print_set(Exceptions)),
        !.


%%% ****************************************************************************
%%% Topology
%%% ****************************************************************************

%%%    id:          internal id
%%%    name:        short name of the topology node
%%%    comment:     an explanation for the topology node
%%%    predicates:  list of all predicates classified into the node
%%%    links:       list of links to other topology nodes

tnode(tnode,[id,name,comment,predicates,links,topology]).

tnode_basicnode_p(tnode(_:basic_node,_,_,_,_)).
tnode_p(tnode(_,_,_,_,_,_)).
tnode_id(tnode(ID,_,_,_,_,_),ID).
tnode_name(tnode(_,Name,_,_,_,_),Name).
tnode_comment(tnode(_,_,C,_,_,_),C).
tnode_predicates(tnode(_,_,_,Preds,_,_),Preds).
tnode_links(tnode(_,_,_,_,Links,_),Links).
tnode_topology(tnode(_,_,_,_,_,Topology),Topology).


print_tnode(NodeStruc):-
	tnode_topology(NodeStruc,Topology),
	tnode_name(NodeStruc,Name),
	tnode_predicates(NodeStruc,Predicates),
	tnode_links(NodeStruc,Links),
	writeq(Name),
	(\+ Topology=system
        -> outterm('/'),
	   writeq(Topology)
        ;  true),
	outterm(' - ['), 
	print_tnode_list(Predicates),outterm(']'),
	(Links = [] 
        -> true
        ;  outterm(' <- ['),
	   print_tnode_list(Links),
           outterm(']')
	).

print_tnode_list([]).
print_tnode_list([Link|Tail]):-
	outtermq(Link),
	(Tail = [] ;
	    outterm(', ')),
	print_tnode_list(Tail).
	
%%% this is the new syntax
tnode(tnode(_,Name,_,Preds,Links,Topology)) --> 
	atom(Name), tnode_name_spec(Topology), tn_preds(Preds), 
	tn_links(Links).

%%% this is the old syntax - for compatibility with old domain text saves
tnode(tnode(_,Name,_,Preds,Links,Topology)) --> atom(Topology), [':'], atom(Name), tn_preds_old(Preds), tn_links_old(Links).

tnode_name_spec(system) --> [].
tnode_name_spec(Name) --> ['/'], atom(Name).

tn_preds(Preds) --> ['-'], !, list(Preds).
tn_preds(_) --> [].

tn_links(Links) --> ['<','-'], !, list(Links).
tn_links(_) --> [].

tn_preds_old(Preds) --> ['-preds', :], !, list(Preds).
tn_preds_old(_) --> [].

tn_links_old(Links) --> ['-links', :], !, list(Links).
tn_links_old(_) --> [].


%%% ****************************************************************************
%%% Model
%%% ****************************************************************************


model_p(mo(_,_,_)).
make_model(mo(N,ML,OL),N,ML,OL).
%% the object_level_world_name is also the name for the model used in the
%% sort-maintanance program to retrieve the current sort-entries and
%% in hypothesis-generation to access the partner-inst-entries
model_name(mo(_,_,Name),Name).
model_meta_level(mo(_,Meta_level_world_name,_),Meta_level_world_name).
model_object_level(mo(_,_,Object_level_world_name),Object_level_world_name).

%%% ****************************************************************************
%%% Sort
%%% ****************************************************************************

s(sort,[name,class,members,subs,supers]).	
	
sclass_p(s(_,_,_,_,_)).				
sclass_name(s(C,_,_,_,_),C).
sclass_sorts(s(_,S,_,_,_),S).
sclass_ext(s(_,_,E,_,_),E).
sclass_subs(s(_,_,_,S,_),S).
sclass_supers(s(_,_,_,_,S),S).
make_sclass(s(C,S,E,_,_),C,S,E).
make_sclass(s(C,S,E,Sub,Super),C,S,E,Sub,Super).

print_sclass(s(class_all,Sorts,Ext,_,_)):-
	outterm(class_all),outterm(': '),
	outterm(Sorts),outterm('  '),outterm(Ext).
print_sclass(s(Class,Sorts,Ext,_,_)):-
	outterm(Class),outterm(': '),
	outterm('['),print_list1(Sorts),outterm(']'),
	outterm('  '),outterm('['),print_list1(Ext),outterm(']').
	
sclass(s(C,S,['...'],_,_)) --> [C,':'],list(S),['[','.','.','.',']'].
sclass(s(C,S,E,_,_)) --> [C,':'],list(S),list(E).

%%% ****************************************************************************
%%%  User Sort
%%% ****************************************************************************

us(usersort,[name,relation,supers]).

usersub_p(us(_,_,_)).
usersub_sub(us(N,_,_),N).
usersub_rel(us(_,Rel,_),Rel).
usersub_supers(us(_,_,Supers), Supers).
usersub_super(us(_,_,Supers),S):- member(S,Supers).

make_usersub(us(N,Relation,Supers),N,Relation,Supers).
%make_usersub(us(N,_),N,_).

print_usersub(us(N,Relation,Supers)):-
	outterm(N),
	outterm(Relation),
	print_list1(Supers).

usersub(us(N,':<',Supers)) --> atom(N), [':'],['<'], listcore(Supers).
usersub(us('NIL',':=',Supers)) --> [:(nil)], [':'],['='], listcore(Supers).
usersub(us(N,':=',Supers)) --> atom(N), [':'],['='], listcore(Supers).


%%% ****************************************************************************
%%% Agenda Entry
%%% ****************************************************************************

%%% Agenda manipulation procedures

%%% agenda entry (ae) accessor/constructor predicates

ae_p(ae(_Type,_Arg,_CheckP,_TriggerP,_HandlingP)).
%%% type can be a compound object, such as "krev(f(p(a),[1000,1000],f12,''))".
ae_type(ae(Type,_Arg,_CheckP,_TriggerP,_HandlingP),Type).
ae_arg(ae(_Type,Arg,_CheckP,_TriggerP,_HandlingP),Arg).
ae_check(ae(_Type,_Arg,CheckP,_TriggerP,_HandlingP),CheckP).
ae_trigger(ae(_Type,_Arg,_CheckP,TriggerP,_HandlingP),TriggerP).
ae_handle(ae(_Type,_Arg,_CheckP,_TriggerP,HandlingP),HandlingP).

print_ae(AE):-
	print_agenda_item(AE).

print_agenda_item(Item):-
	ae_type(Item,Type),
	ae_arg(Item,Arg),
	print_agenda_item1(Type,Arg).

print_agenda_item1(mpredicate_undefined,Name):-
	outterm('Define metapredicate '),
	outterm(Name).
print_agenda_item1(mmpredicate_undefined,Name):-
	outterm('Define metametapredicate '),
	outterm(Name).
print_agenda_item1(enter_metafact_for_rule,[Rule,_]):-
	outterm('Enter mfact for '),
	outterm(Rule).
print_agenda_item1(tnode_undefined,[_NodeStruc,Link]):-
	outterm('Define topology node '),
	outterm(Link).
print_agenda_item1(krev,Fact):-
	outterm('Perform knowledge revision on '),
	outterm(Fact).
print_agenda_item1(icvio,IC):-
	outterm('Integrity constraint violation on '),
	print_ic_on_agenda(IC).

%%% ****************************************************************************
%%% print_list1 (enumeration style without square brackets)
%%% ****************************************************************************

%%% print a list without square brackets
print_list1([]):- !.
print_list1([Last]):-
        outterm(Last),!.
print_list1([First|Rest]):-
        outterm(First),outterm(', '),
        print_list1(Rest),
        !.

%% print items with a supplied print function
print_list1(Items,M:PF):-
	print_list1(Items,M:PF,-1).

print_list1([],_,_):- !.
print_list1(Rest,_,0):- 
	!,
        outterm('... <'), length(Rest,L),
	outterm(L), outterm(' more>'),!.
print_list1([Last],M:PF,_):-
        !,
	Call =.. [PF,Last],
	call(M:Call),
        !.
print_list1([Last],PF,_):-
        !,
	Call =.. [PF,Last],
	call(Call),
        !.
print_list1([First|Rest],M:PF,N):-
	!,
	Call =.. [PF,First],
	call(M:Call), !,outterm(', '),
	N1 is N-1,
        print_list1(Rest,M:PF,N1),
        !.
print_list1([First|Rest],PF,N):-
	!,
	Call =.. [PF,First],
	call(Call), !,outterm(', '),
	N1 is N-1,
        print_list1(Rest,PF,N1),
        !.

%%% print list Prolog style, using the supplied print function for the elements
print_list(List,M:PF):-
	print_list(List,M:PF,-1).
print_list(List,M:PF,N):-
        !,
        outterm('['),
        print_list1(List,M:PF,N),
        outterm(']').

%%% ****************************************************************************
%%% tuple
%%% ****************************************************************************

%%% represented by lists.

print_tuple(Tuple):-
        outterm('('),
        print_list1(Tuple),
        outterm(')').

%%% use PF to print the elements
print_tuple(Tuple,M:PF):-
        outterm('('),
        print_list1(Tuple,M:PF),
        outterm(')').

%%% ****************************************************************************
%%% set
%%% ****************************************************************************

%%% represented by lists.

print_set(Set):-
        outterm('{'),
        print_list1(Set),
        outterm('}').

%%% use PF to print the elements
print_set(Set,M:PF):-
        outterm('{'),
        print_list1(Set,M:PF),
        outterm('}').



%%% ****************************************************************************
%%% general structure predicates
%%% ****************************************************************************

%%% given a structure, return its ID.

structure_id(Fact,ID):- fact_p(Fact), !, fact_id(Fact,ID).
structure_id(Fact,ID):- mfact_p(Fact), !, mfact_id(Fact,ID).
structure_id(Fact,ID):- mmfact_p(Fact), !, mmfact_id(Fact,ID).
structure_id(Rule,ID):- rule_p(Rule), !, rule_id(Rule,ID).




%%% ****************************************************************************
%%% parser procedure for supset strings
%%% ****************************************************************************

supset(Supset) --> ['('], domain_list(DL), [')'], ['\'], global_exceptions(GE),
                        {make_supset(Supset,DL,GE)}.
supset(Supset) --> domain_list(DL), ['\'], global_exceptions(GE),
			{make_supset(Supset,DL,GE)}.
supset(Supset) --> ['('], domain_list(DL), [')'],
                        {make_supset(Supset,DL,[])}.
supset(Supset) --> domain_list(DL),
                        {make_supset(Supset,DL,[])}.

domain_list([D|RD]) --> domain(D), rest_domains(RD).
domain_list([]) --> [].


rest_domains([D|RD]) --> ['*'], domain(D), rest_domains(RD).
rest_domains([]) --> [].


domain(D) --> ['('], domain(D), [')'].
domain(D) --> ['('], concept(C), ['\'], local_exceptions(E), [')'],
                {make_domain(D,C,E)}.
domain(D) --> concept(C), ['\'], local_exceptions(E),
                {make_domain(D,C,E)}.
domain(D) --> concept(C),
                {make_domain(D,C,[])}.
concept(C,[C|Rest],Rest).

local_exceptions(TL) --> ['{'], termlist(TL), ['}'].
termlist([T|RT]) --> term(T), rest_terms(RT).
termlist([]) --> [].
rest_terms([T|RT]) --> [','], term(T), rest_terms(RT).
rest_terms([]) --> [].
term(T,[T|Rest],Rest).

global_exceptions(TL) --> ['{'], tuple_list(TL), ['}'].
tuple_list([T|RT]) --> tuple(T), rest_tuples(RT).
tuple_list([]) --> [].
rest_tuples([T|RT]) --> [','], tuple(T), rest_tuples(RT).
rest_tuples([]) --> [].
tuple(TL) --> ['('], termlist(TL), [')'].

empty([]).




%%% ****************************************************************************
%%% prolog term DCG
%%% ****************************************************************************

%%% only for variable-free expressions!

%% uppercase atoms are especially marked by the parser as ':'(...)
% Hack to accept Lists in Rules
prolog_object(Term) --> ['['], mylistcore(Term), [']'].

prolog_object(Term) --> atomic_object(Term).
prolog_object(Term) --> nonatomic_object(Term).

atomic_object(Term) --> atom(Term).
atomic_object(':'(Term)) --> marked_variable(Term).
atomic_object(Term) --> number(Term).
marked_variable(Term) --> [':', Term], {atom(Term)}.
marked_variable(Term) --> [':'(Term)], {atom(Term)}.
nonatomic_object(Term) --> list(Term).
nonatomic_object(Term) --> structure(Term).

atom(A) --> [A], {atom(A)}.
number(A,[A|Rest],Rest):- number(A).
integer(A,[A|Rest],Rest):- integer(A).


list(Term) --> ['['], listcore(Term), [']'].

listcore([A|R]) --> prolog_object(A), [','], listcore(R).
listcore([A]) --> prolog_object(A).
listcore([]) --> [].


mylistcore([A|R]) --> prolog_object(A), ['|'], ['['], mylistcore(R), [']'].
mylistcore([A|R]) --> prolog_object(A), ['|'], atomic_object(R).
mylistcore([A|R]) --> prolog_object(A), [','], mylistcore(R).
mylistcore([A]) --> prolog_object(A).

%%% if used in metapredicates, functor could be uppercase (marked)
%%% we translate this to lowercase

structure(Term) --> atom_or_var(A), ['('], listcore(L), [')'],
			{Term =.. [A|L]}.
atom_or_var(A) --> marked_variable(A).
atom_or_var(A) --> atom(A).

%%% ****************************************************************************
%%% String DCG components (i.e. tokens are chars here, not atoms as above!
%%% ****************************************************************************

						
number_string(A) --> int_string(A).
number_string(A) --> int_string(A1), ".", int_string(A2), 
                     {[Dot]=".", append(A1,[Dot|A2],A)}.

int_string([C]) --> digit_char(C).
int_string([C|A1]) --> digit_char(C), int_string(A1).
digit_char(C) --> [C], {member(C,"0123456789")}.

any --> [].
any --> [_], any.

any("") --> [].
any([C|S]) --> [C], any(S).


%%% ****************************************************************************
%%% read_object
%%% ****************************************************************************

read_object(StringAtom,ObjectType,Object):-
codelist(StringAtom,CodeList),
parse_string(CodeList,ObjectType,Object).

parse_string(CharList,ObjectType,Object):-
        tokenize(CharList,Tokens),
        %% this calls the parser procedure which returns Object.
        %% obviously, you must have a parser procedure for the object type,
        %% such as SUPSET/3.
        ParserCall =.. [ObjectType,Object,Tokens,[]],
        ParserCall.

%%% ****************************************************************************
%%% textual representation of constants and variables
%%% ****************************************************************************

%%% Determines how constants and variables are distinguished
%%% syntactically.  Prolog syntax is to use lowercase for constants
%%% and uppercase for variables; Prolog switched is to use
%%% lowercase for variables and uppercase for constants. In both
%%% modes, "c(atom)" or "const(atom)" denotes a constant named "atom".

:- dynamic(variable_syntax/1).

variable_syntax(prolog_switched).  %% also allowed "prolog"

set_variable_syntax(Value):-
	(retract(variable_syntax(_));true), 
	assert(variable_syntax(Value)),
	!.

denotes_constant(c(Constant),Constant).
denotes_constant(const(Constant),Constant).
denotes_constant(Constant,Constant):- number(Constant).
denotes_constant(uc(Constant),Constant):-
	variable_syntax(prolog_switched).
denotes_constant(Constant,Constant):-
	atom(Constant),  % special feature for rule-ops.pl (rule entry)
	variable_syntax(prolog).

denotes_variable(uc(Variable),Variable):-
	variable_syntax(prolog).
denotes_variable(V,V):-
	variable_syntax(prolog_switched).
   

						
%%% ****************************************************************************
%%% tokenize
%%% ****************************************************************************

%%% tokenize(Name,Tokenlist):
%%% splits an arbitrary Prolog string ("...") Name into tokens (atoms).
%%% As a special construct,
%%% the tokenizer handles comments which must be prefixed by % and extend
%%% until the end of the line. They are returned as the the token
%%% comment(<CommentAtom>). 
%%% Quoted Prolog Atoms (such as 'a special atom') are handled properly, but
%%% the quote character must not appear inside the quoted atom. 
%%% All atoms that begin with uppercase characters are returned in lowercase
%%% as the token uc(<lowercase var name>).

tokenize([],[]):- !.
tokenize(C,[W|Ws]):-
        gettoken(C,W,C1),
        resttokens(C1,Ws).

resttokens([],[]):- !.
resttokens(C,[W1|Ws]):-
        gettoken(C,W1,C1),
	!,
        resttokens(C1,Ws).
resttokens(_,[]).

gettoken([First|Rest1],W,Rest):-
	%% handle comments
	comment_char([First]),
        !,
	%% read until end of line
	getcomment(Rest1,W,Rest).
gettoken([QuoteChar|Rest1],W,Rest):-
	%% handle quoted Prolog atoms specially
	quote_char([QuoteChar]),
	!,
	get_quoted_token(QuoteChar,Rest1,W,Rest).
gettoken([First|Rest],W,Rest):-
        single_char([First]),
        !,
        name(W,[First]).
gettoken([First,Second|RestToken],Token,Rest):-
        % handle signed numbers
        (name('+',[First])
         ;name('-',[First])),
        in_digit_token(Second,SecondTranslated),
        !,
        restnumbertoken(RestToken,Cs,Rest),
        name(Token,[First,SecondTranslated|Cs]), !.
gettoken([First|RestToken],Token,Rest):-
        % handle unsigned numbers
        in_digit_token(First,FirstTranslated),
        !,
        restnumbertoken(RestToken,Cs,Rest),
        name(Token,[FirstTranslated|Cs]), !.
gettoken([First|RestToken],Token,Rest):-
        in_token(First,FirstTranslated),
        !,
        resttoken(RestToken,Cs,Rest),
        name(W,[FirstTranslated|Cs]),
	%% if this was a variable, mark it
	(uppercase(First), Token = ':'(W) ; Token = W), !.
%% skip anything that's not part of a token and not a separator either
gettoken([_|Rest],W,C2):-
        gettoken(Rest,W,C2).

resttoken([First|Rest],[FirstTranslated|Cs],C2):-
        in_token(First,FirstTranslated),
        !,
        resttoken(Rest,Cs,C2).
resttoken(C,[],C).

% up to now only digits, "." and "E" may still come
restnumbertoken([First|Rest],[FirstTranslated|Cs],C2):-
        in_digit_token(First,FirstTranslated),
        !,
        restnumbertoken(Rest,Cs,C2).
restnumbertoken([First|Rest],[First|Cs],C2):-
        name('.',[First]),
        !,
        restnumbertoken2(Rest,Cs,C2).
restnumbertoken(C,[],C).

% no more "." allowed, but an "E" may follow
restnumbertoken2([First|Rest],[FirstTranslated|Cs],C2):-
        in_digit_token(First,FirstTranslated),
        !,
        restnumbertoken2(Rest,Cs,C2).
restnumbertoken2([First,Second|Rest],[First,Second|Cs],C2):-
        % handle exponent
        (name('E',[First])
         ;name('e',[First])),
        (name('+',[Second])
        ;name('-',[Second])),
        !,
        restnumbertoken3(Rest,Cs,C2).
restnumbertoken2(C,[],C).

% no more "." and "E" allowed
restnumbertoken3([First|Rest],[FirstTranslated|Cs],C2):-
        in_digit_token(First,FirstTranslated),
        !,
        restnumbertoken3(Rest,Cs,C2).
restnumbertoken3(C,[],C).

lasttoken('.').

getcomment(Chars,comment(Comment),Rest):-
	skip_leading_comment_and_space_chars(Chars,Rest1),
	read_until_newline(Rest1,CommentString,Rest),
	atom_chars(Comment,CommentString).

skip_leading_comment_and_space_chars([F|R],Rest):-
	(comment_char([F]) ; space_char([F])),
	!,
	skip_leading_comment_and_space_chars(R,Rest).
skip_leading_comment_and_space_chars(Rest,Rest).


read_until_newline([First|Rest],[],Rest):-
	newline_char([First]).
read_until_newline([First|R],[First|Comment],Rest):-
	read_until_newline(R,Comment,Rest).

%%% read in a quoted token that is quoted by QuoteChar.  End reading upon first
%%% occurence of QuoteChar.

get_quoted_token(QuoteChar,Rest1,Atom,Rest):-
	read_until_quote(QuoteChar,Rest1,AtomString,Rest),
	atom_chars(Atom,AtomString).

read_until_quote(QuoteChar,[QuoteChar|Rest],[],Rest):- !.
read_until_quote(QuoteChar,[First|R],[First|String],Rest):-
	read_until_quote(QuoteChar,R,String,Rest).



single_char("~").	
single_char(",").
single_char("+").
single_char("*").
single_char("/").
single_char("\").
single_char("(").
single_char(")").
single_char("{").
single_char("}").
single_char("[").
single_char("]").
single_char(".").
single_char("&").
single_char(";").
single_char(">").
single_char("<").
single_char("=").
single_char(":").
single_char("|").
comment_char("%").
quote_char("'").
space_char(" ").
newline_char([10]).

%%% modify this in other Prologs, if necessary (char codes might be different)

in_token(C,C):-
	"a" = [CodeSmallA], "z" = [CodeSmallZ], 
	C > CodeSmallA - 1, C < CodeSmallZ + 1.
in_token(C,CTrans):-
	"A" = [CodeA],
	C > CodeA - 1,
	"Z" = [CodeZ],
	 C < CodeZ +1,
	 "a" = [CodeSmallA],
	 Diff is CodeA - CodeSmallA,
	 CTrans is C - Diff.                       % A .. Z
in_token(C,C) :-
	"0" = [Code0], "9" = [Code9],
	C > Code0 - 1, C < Code9 + 1.     % 0 .. 9
in_token(C,C):-
	"'" = [C] ; "-" = [C]; "_" = [C].

in_digit_token(C,C) :-
	"0" = [Code0], "9" = [Code9],
	C > Code0 - 1, C < Code9 + 1.     % 0 .. 9

uppercase(C):-
	"A" = [CodeA],
	C > CodeA - 1,
	"Z" = [CodeZ],
	 C < CodeZ +1.  

%%% ****************************************************************************
%%% IC
%%% ****************************************************************************

%%% Ic
%%%    vars:        list of variables used in the ic
%%%    prems:       list of premises (a real list, not connected by &)
%%%                    each premise in term format
%%%    concls:      list of lists of conclusions 
%%%                    each conclusion in list format (like a list of premises)
%%%    comment:     an explanation for the ic

ic(ic,[vars,icpremises(prems),icconclusions(concls),comment]).

ic_p(ic(_,_,_,_)).
ic_vars(ic(Vars,_,_,_),Vars).
ic_prems(ic(_,Prems,_,_),Prems).
ic_concls(ic(_,_,Concls,_),Concls).
ic_comment(ic(_,_,_,Comment),Comment).

alter_ic_prems(ic(Vars,_,Concls,Comment),Prems,
                   ic(Vars,Prems,Concls,Comment)).

alter_ic_concls(ic(Vars,Prems,_,Comment),Concls,
                   ic(Vars,Prems,Concls,Comment)).


print_ic_on_agenda((ViolatedIC,_OrigIC)):-
	print_ic(ViolatedIC).

print_ic(IcS):-
        !,
	ic_vars(IcS,Vars1), 
	%% must make sure not to bind Vars1 if it is variable !!
	(var(Vars1)
        -> true
        ;  Vars = Vars1),
        ic_prems(IcS,Prems),
        ic_concls(IcS,Concls),
        print_ic_prems(Prems,Vars),
	outterm(' ~> '),
	print_ic_concls(Concls,Vars),
        print_ic_comment(IcS).

print_ic_prems(Prems):-
	print_ic_prems(Prems,_).
print_ic_prems([],_):-
	!.
print_ic_prems([Last],Vars):-
	!,
	print_literal(Last,Vars).
print_ic_prems([First|Rest],Vars):-
	print_literal(First,Vars),outterm(' & '),
	print_ic_prems(Rest,Vars).


print_ic_concls(Concls):-
	print_ic_concls(Concls,_).
print_ic_concls([[]],_).
print_ic_concls([[H|T]|[]],Vars):- 
	outterm('('),
	print_ic_concls([H|T],Vars),
	outterm(')').
print_ic_concls([[H|T]|T2],Vars):- 
	outterm('('),
	print_ic_concls([H|T],Vars),
	outterm(')'),
	outterm(' ; '),
	print_ic_concls(T2,Vars).
print_ic_concls([Last],Vars):-
	print_literal(Last,Vars).
print_ic_concls([First|Rest],Vars):-
	print_literal(First,Vars),
	outterm(' & '),
	print_ic_concls(Rest,Vars).



print_ic_comment(IcS):-
        blip_control(print_comment_p,t),
        ic_comment(IcS,Comment),
        (not(Comment=[]),
                nl,outterm(Comment),nl
        ;
                true).
print_ic_comment(_).


pretty_print_ic(Ic):-
	ic_prems(Ic,Prems),
	ic_concls(Ic,Concls),
	ic_comment(Ic,Comment),
	ic_vars(Ic,Vars),
	outterm('Variables: '),outterm(Vars),nl,nl,
	outterm('    '),pretty_print_ic_prems(Prems),nl,
	outterm('~> '),pretty_print_ic_concls(Concls),nl,nl,
	outterm('Comment:           '),outterm(Comment),nl,
	!.

pretty_print_ic_prems([Last]):-
	!,
	outterm(Last).
pretty_print_ic_prems([First|Rest]):-
	outterm(First),nl,outterm('  & '),
	pretty_print_ic_prems(Rest).


pretty_print_ic_concls([[]]).
pretty_print_ic_concls([[H|T]|[]]):- 
	outterm('('),
	pretty_print_ic_concls([H|T]),
	outterm(')').
pretty_print_ic_concls([[H|T]|T2]):- 
	outterm('('),
	pretty_print_ic_concls([H|T]),
	outterm(')'), nl,
	outterm('  ; '),
	pretty_print_ic_concls(T2).
pretty_print_ic_concls([Last]):-
	outterm(Last).
pretty_print_ic_concls([First|Rest]):-
	outterm(First),
	outterm(' & '),
	pretty_print_ic_concls(Rest).


ic(ic(_Vars,Prems,Concls,_)) --> icpremises(Prems),
	['~', '>'], 
	icconclusions(Concls).

ic(ic(_Vars,_Prems,Concls,_)) --> ['~', '>'], 
	icconclusions(Concls).

ic(ic(_Vars,Prems,_Concls,_)) --> icpremises(Prems),
	['~', '>']. 

icpremises(P) --> icpremise(P1), {P = [P1]}.
icpremises(P) --> ['('], icpremise(P1), [')'], {P = [P1]}.
icpremises(P) --> icpremise(P1), ['&'], icpremises(P2),
 			{P = [P1|P2]}.

icpremise(P) --> structure(P).
icpremise(:(P,EP)) --> structure(P), [':'], ep(EP).
icpremise(:(P,EP)) --> structure(P), ['-'], ep(EP).
icpremise(\+(P)) --> unp, structure(P).
icpremise(\+(P)) --> unp, ['('], structure(P), [')'].
unp --> ['\', '+'].

icconclusion(C) --> icpremises(C1), {C = C1}.
icconclusions(C) --> ['('], icconclusion(C1), [')'], [';'], icconclusions(C2),
 			{C = [C1|C2]}.
icconclusions(C) --> icconclusion(C1), [';'], icconclusions(C2),
 			{C = [C1|C2]}.
icconclusions(C) --> ['('], icconclusion(C1), [')'], {C = [C1]}.
icconclusions(C) --> icconclusion(C1), {C = [C1]}.