%%% ============================================================================
%%% Copyright Notice
%%% ============================================================================

%%% This file is part of the MOBAL system and is NOT in the public domain. It
%%% is available to the MLT consortium partners and the EEC under the conditions
%%% of the MLT contract only and may NOT in any form be used outside MLT nor 
%%% made available to any third party without prior written permission from GMD.  

%%% (C) 1990-92 GMD (Ges. f. Mathematik und Datenverarbeitung, FIT.KI, 
%%%         Projekt MLT, Pf. 1316, 5205 St. Augustin 1, West Germany).
%%% ============================================================================


:- module(derivation, [
	derivation/2,
	support/2,
	derivation_p/1,
	derivation/3,
	derivation_fact/2,
	derivation_supports/2,
	support_p/1,
	support/4,
	make_support/5,
	support_type/2,
	support_input_p/1,
	support_fact/2,
	support_ep/2,
	support_source/2,
	support_antecedents/2,
	support_binding_tuple/2,
	print_derivation/1,
	print_derivation/2
    ]).
:-use_module(library(compatibility)).
:-use_module(kitcore,library(kitcore),[outterm/1]).
:-use_module(basics,library(basics),[member/2]).
:-use_module(list,library(list),[assoc/3]).
:-use_module(conststovars,library(conststovars),[
   make_binding_list_by_pairing/3
]).

:- use_module(itemspec,library(itemspec), [
	print_fact/1,
	print_rule/1,
	rule_id/2,
	fact_pred/2,
	fact_args/2,
	fact_ep/2,
	rule_prems/2,
	rule_vars/2,
	rule_concl/2
  ]).

        
%% Don't use library(msg) !!!!!!!!!! 
%% The quintus compiler has problems with zyklic uses.
%% Please, for ortogonality use only the kitcore predicate outterm 
%% and the built-in predicates tab(N) and nl within printfunctions!!!!
%% (Clearly, you could also use other printfunctions 
%% defined here or in itemspec.)


%%% "derivation.prolog"
%%% defines the derivation structure and its accessor and print routines

%%% ****************************************************************************
%%% derivation (structure), support (structure)
%%% ****************************************************************************

%%% <Derivation>      --> derivation(<Fact>,<Supports>)
%%% <Supports>       --> [Support, Support, ...]
%%% <Support>        --> support(Type,<Fact>,<Source>,<Antecedents>)
%%% <Antecedents>   --> [<Derivation>, <Derivation>, ...]

%%% where <Rule>, <Fact>, <Support>, and <Derivation> are the structures defined
%%% below and in item-spec.prolog.

%%% Structures
%%% derivation(Fact,Supports)
%%% support(Type,EP,Source,Antecedents)

derivation(derivation,[fact,supports]).
support(support,[type,fact,source,antecedents]).

%%% Accessors:

derivation_p(derivation(_,_)).
derivation(derivation(Fact,Supports),Fact,Supports).
derivation_fact(derivation(Fact,_),Fact).
derivation_supports(derivation(_,Supports),Supports).

support_p(support(_,_,_,_)).
%% obsolete - use make_support instead
support(support(EP,Source,Antecedents),EP,Source,Antecedents).
make_support(support(Type,EP,Source,Antecedents),Type,EP,Source,Antecedents).
support_type(support(Type,_,_,_),Type).
support_input_p(support(input,_,_,_)).
support_fact(support(_,Fact,_,_),Fact).
support_ep(support(_,Fact,_,_),EP):- fact_ep(Fact,EP).
support_source(support(_,_,Source,_),Source).
support_antecedents(support(_,_,_,Antecedents),Antecedents).

%%% derived field:
% support_binding_tuple - see below

%%% ****************************************************************************
%%% print_derivation
%%% ****************************************************************************

%%% pretty-print a derivation
print_derivation(Derivation):-
        print_derivation(Derivation,0,-1),
        !.
print_derivation(Derivation,Depth):-
        !,
        print_derivation(Derivation,0,Depth),
        !.

print_derivation(Derivation,Indentation,0):-
        !,
        derivation_fact(Derivation,Fact),
        tab(Indentation),
        print_fact(Fact), 
	outterm(', because '),nl,
	derivation_supports(Derivation,Supports),
        print_supports_short(Supports),
        !.
print_derivation(Derivation,Indentation,_):-
        derivation_supports(Derivation,[Support]),
        support_input_p(Support),
        !,
        %% support is a single input fact, print on same line
        derivation_fact(Derivation,Fact),
        tab(Indentation),
	print_fact(Fact),outterm('Input.'),nl,
        !.
print_derivation(Derivation,Indentation,Depth):-
        !,
        derivation_fact(Derivation,Fact),
        NewIndentation is Indentation+3,
        tab(Indentation),
        print_fact(Fact),
	tab(NewIndentation),outterm('Because:'),nl,
        derivation_supports(Derivation,Supports),
        NewDepth is Depth - 1,
        print_supports(Supports,NewIndentation,NewDepth),
        !.

print_supports([],_,_):- !.
print_supports([FirstSupport|RestSupports],Indentation,Depth):-
        support_input_p(FirstSupport),
        !,
        %% a derivation from input, can't follow that one any further.
        support_source(FirstSupport,FactS),
        tab(Indentation),outterm('Input '),print_fact(FactS),nl,
        print_supports(RestSupports,Indentation,Depth).
print_supports([FirstSupport|RestSupports],Indentation,Depth):-
        %% a support via a rule
        support_source(FirstSupport,Rule),
        NewIndentation is Indentation+3,
        tab(Indentation),
        print_rule(Rule), 
	tab(NewIndentation),outterm('Premises:'),nl,
        support_antecedents(FirstSupport,Antecedents),
        !,
        print_antecedents(Antecedents,NewIndentation,Depth),
        print_supports(RestSupports,Indentation,Depth).

print_supports_short([LastSupport]):-
        support_input_p(LastSupport),
        !,
	outterm('Input.'),nl.
print_supports_short([LastSupport]):-
        %% a support via a rule
        support_source(LastSupport,Rule),
        rule_id(Rule,RuleID),
	outterm('Rule '),outterm(RuleID),outterm('.'),nl.
print_supports_short([FirstSupport|RestSupports]):-
        support_input_p(FirstSupport),
        !,
	outterm('Input.'),nl,
        print_supports_short(RestSupports).
print_supports_short([FirstSupport|RestSupports]):-
        %% a support via a rule
        support_source(FirstSupport,Rule),
        rule_id(Rule,RuleID),
        outterm('Rule '),outterm(RuleID),outterm(', '),
        print_supports_short(RestSupports).


print_antecedents([],_,_):- !.
print_antecedents([FirstAntecedent|RestAntecedents],Indentation,Depth):-
        !,
        print_derivation(FirstAntecedent,Indentation,Depth),
        print_antecedents(RestAntecedents,Indentation,Depth).


%%% ****************************************************************************
%%% constructing the binding from a support
%%% ****************************************************************************

%% this is a derived field - the binding tuple used in the support
%% applicable only for derived supports
support_binding_tuple(Support,Tuple):-
	support_type(Support,derived),
	support_antecedents(Support,Antecedents),
	support_fact(Support,Fact),
	support_source(Support,Rule),
        extract_antecedent_facts(Antecedents,AntecedentFacts),
        rule_prems(Rule,Prems),
        rule_vars(Rule,Vars),
	rule_concl(Rule,Concl),
        find_variable_bindings([Fact|AntecedentFacts],[Concl|Prems],Bindings),
        %% collect the bindings into a list/tuple (ignore all but the first;
        %% all bindings should be the same anyway)
        bagof(Value,Var^Bindings^(member(Var,Vars),assoc(Var,Bindings,[Var|Value])),Tuple),
	!.
support_binding_tuple(Support,Vars):-
	write('support_binding_tuple/2 failed!!'), nl,
	support_source(Support,Rule),
	rule_vars(Rule,Vars).

%%% find_variable_bindings(AntecedentFacts,Prems,Bindings):
%%% find out how the variables in Prems are bound in AntecedentFacts.
%%% ###### !!!!
%%% *** this function relies on the fact that the premises of the rule and
%%% the antecedents returned in a fact's derivation are in the same
%%% order !!! ***
%%% ###### !!!
%%% Bindings is an alist of [Var|Value] pairs which may contain several entries
%%% for a variable

find_variable_bindings([],_,[]):- !.
find_variable_bindings([FAnte|RAntes],[FPrem|RPrems],Bindings):-
	%% try to bind the variables
	find_variable_bindings1(FPrem,FAnte,FirstBindings),
	!,
	find_variable_bindings(RAntes,RPrems,RestBindings),
        append(FirstBindings,RestBindings,Bindings).
find_variable_bindings(Antes,[_|RPrems],Bindings):-
	%% variables could not be bound - skip this premise (should be a
	%% built-in)
	find_variable_bindings(Antes,RPrems,Bindings).
	
find_variable_bindings1(Prem,Antecedent,Bindings):-
        !,
	(Prem = not(Prop)
        -> EP = [_,1000]
        ;  Prop = Prem,
	   EP = [1000,_]),
	Prop =.. [Pred|Vars],
	fact_pred(Antecedent,Pred),
	fact_ep(Antecedent,EP),
	fact_args(Antecedent,Args),
        %% if Header is not equal to the predicate used in Antecedent, we are
        %% matching incorrectly (see next clause of find_variable_bindings1)
        make_binding_list_by_pairing(Vars,Args,Bindings).

extract_antecedent_facts([],[]):- !.
extract_antecedent_facts([Derivation|RestDerivations],[Fact|RestFacts]):-
        derivation_fact(Derivation,Fact),
        extract_antecedent_facts(RestDerivations,RestFacts).


