;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  scroll-bar-dialog-items.lisp
;;
;;
;;  1989, Apple Computer, Inc
;;
;;  the code in this file implements a scroll-bar class of dialog-items
;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Change history
;;
;; 01/28/91 bill event.where -> eventRecord.where
;;

(in-package :ccl)

(export '(scroll-bar-dialog-item scroll-bar-setting
          scroll-bar-min scroll-bar-max scroll-bar-length scroll-bar-width
          scroll-bar-page-size  scroll-bar-scroll-size scroll-bar-scrollee
          set-scroll-bar-setting set-scroll-bar-min set-scroll-bar-max
          set-scroll-bar-length set-scroll-bar-width set-scroll-bar-scrollee
          scroll-bar-changed track-scroll-bar
	  scroll-bar-track-thumb-p set-scroll-bar-track-thumb-p
          pane-splitter split-pane pane-splitter-corners draw-pane-splitter-outline)
        :ccl)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; a few things that need to be around at compile time, but not run time
;;
(eval-when (eval compile)

  ;some constants for tracking the clicks in the scroll-bar
  (defconstant $InUpButton 20)
  (defconstant $InDownButton 21)
  (defconstant $InPageUp 22)
  (defconstant $InPageDown 23)
  (defconstant $InThumb 129)

  ;trap macros compiled away into in-line calls
  (require 'traps))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-dialog-item
;;

(defclass scroll-bar-dialog-item (control-dialog-item)
  ((direction :initarg :direction :reader scroll-bar-direction)  
   (min :initarg :min :reader scroll-bar-min)
   (max :initarg :max :reader scroll-bar-max)
   (setting :initarg :setting)
   (track-thumb-p :initarg :track-thumb-p :initform nil
                  :accessor scroll-bar-track-thumb-p)
   (page-size :initarg :page-size :initform 5 :accessor scroll-bar-page-size)
   (scroll-size :initarg :scroll-size :initform 1 :accessor scroll-bar-scroll-size)
   (scrollee :initarg :scrollee :initform nil :reader scroll-bar-scrollee)
   (pane-splitter :initform nil :accessor pane-splitter)
   (pane-splitter-position :initform nil :initarg :pane-splitter 
                           :reader pane-splitter-position)))

(defclass pane-splitter (simple-view)
  ((scrollee :initarg :scrollee 
             :reader scroll-bar-scrollee)
   (direction :initarg :direction :reader scroll-bar-direction)
   (scroll-bar :initarg :scroll-bar :initform nil :reader scroll-bar)))

; Args would be in wrong order if these were defined as :writer's
(defmethod set-scroll-bar-track-thumb-p ((item scroll-bar-dialog-item) value)
  (setf (scroll-bar-track-thumb-p item) value))

(defmethod set-scroll-bar-scrollee ((view pane-splitter) value)
  (setf (slot-value view 'scrollee) value))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;initialize-instance
;;
;;initargs:
;;   length
;;   width
;;   direction
;;   setting
;;   min
;;   max
;;   page-size
;;   track-thumb-p
;;
;;in addition, the standard dialog-item initargs can be used
;;

(defmethod initialize-instance ((item scroll-bar-dialog-item) &rest initargs
                                &key (min 0) (max 100) (setting 0) (width 16)
                                (direction :vertical) (length 100) scrollee
                                pane-splitter (pane-splitter-length 7)
                                view-position view-container)
  (declare (dynamic-extent initargs))
  (setq max (max min max)
        setting (min (max 0 min) max))
  (when pane-splitter
    (let* ((splitter (make-instance 'pane-splitter 
                                    :direction direction
                                    :width width
                                    :length pane-splitter-length
                                    :scroll-bar item
                                    :scrollee scrollee))
           (size (view-size splitter))
           (h (point-h size))
           (v (point-v size)))
      (setf (pane-splitter item) splitter)
      (if (eq direction :vertical)
        (progn
          (decf length v)
          (when view-position
            (let ((p-h (point-h view-position))
                  (p-v (point-v view-position)))
              (if (eq pane-splitter :top)
                (progn
                  (set-view-position splitter view-position)
                  (setq view-position (make-point p-h (+ p-v v))))
                (progn
                  (set-view-position splitter p-h (+ p-v length)))))))
        (progn
          (decf length h)
          (when view-position
            (let ((p-h (point-h view-position))
                  (p-v (point-v view-position)))
              (if (eq pane-splitter :top)
                (progn
                  (set-view-position splitter view-position)
                  (setq view-position (make-point (+ p-h h) p-v)))
                (progn
                  (set-view-position splitter (+ p-h length) p-v)))))))))
  (apply #'call-next-method
         item
         :min min
         :max max
         :setting setting
         :direction direction
         :length length
         :view-container nil
         :view-position view-position
         :view-size
         (case direction
           (:vertical (make-point width length))
           (:horizontal (make-point length width))
           (t (error "illegal :direction ~a (must be :vertical or :horizontal)."
                     direction)))
         initargs)
  (when (and pane-splitter view-container (not view-position))
    (set-default-size-and-position item view-container)
    (set-view-position item (view-position item)))
  (when view-container
    (set-view-container item view-container))
  (when scrollee
    (add-view-scroll-bar scrollee item)))

(defun view-scroll-bars (view)
  (view-get view 'scroll-bars))

(defun add-view-scroll-bar (view scroll-bar)
  (pushnew scroll-bar (view-get view 'scroll-bars)))

(defun delete-view-scroll-bar (view scroll-bar)
  (setf (view-get view 'scroll-bars)
        (delete scroll-bar (view-get view 'scroll-bars))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;install-view-in-window
;;
;;  this is when we actually create the control (when the item
;;  is added to a window)

(defmethod install-view-in-window :after ((item scroll-bar-dialog-item) view)
  (declare (ignore view))
  (let* ((window (view-window item))
         (my-size (view-size item))
         (my-position (view-position item))
         (setting (scroll-bar-setting item))
         (min (scroll-bar-min item))
         (max (scroll-bar-max item)))
    (when window
      (rlet ((scroll-rect :rect))
        (rset scroll-rect rect.topleft my-position)
        (rset scroll-rect rect.bottomright (add-points my-position my-size))
        (when (dialog-item-handle item)
          (_DisposControl :ptr (dialog-item-handle item)))
        (setf (dialog-item-handle item)
              (_NewControl :ptr (wptr item)   ;window
                           :ptr scroll-rect   ;item rectangle
                           :ptr (%null-ptr)   ;title
                           :word -1           ;visible
                           :word setting      ;initial value
                           :word min  ;min value
                           :word max  ;max value
                           :word 16           ;type of control
                           :long 0            ;refcon
                           :ptr))
        (unless (window-active-p window)
          (view-deactivate-event-handler item))))))

(defmethod remove-view-from-window :before ((item scroll-bar-dialog-item))
  (let ((handle (dialog-item-handle item)))
    (when handle
      (setf (dialog-item-handle item) nil)
      (_DisposControl :ptr handle))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;view-draw-contents
;;
;;this function is called whenever the item needs to be drawn
;;
;;to draw the dialog-item, we just call _Draw1Control
;;

(defmethod view-draw-contents ((item scroll-bar-dialog-item))
  (_Draw1Control :ptr (dialog-item-handle item)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;view-deactivate-event-handler
;;
;;this function is called whenever the scrollbar needs to be deactivated
;;

(defmethod view-deactivate-event-handler ((item scroll-bar-dialog-item))
  (with-focused-view (view-container item)
    (_hilitecontrol :ptr (dialog-item-handle item)
                    :word 255)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;view-activate-event-handler
;;
;;this function is called whenever the scrollbar needs to be activated
;;

(defmethod view-activate-event-handler ((item scroll-bar-dialog-item))
  (when (and (dialog-item-enabled-p item)
             (let ((w (view-window item)))
               (and w (window-active-p w))))
    (with-focused-view (view-container item)
      (_hilitecontrol :ptr (dialog-item-handle item)
                      :word 0))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;dialog-item-enable
;;
;; Need to patch the system-supplied method for control-dialog-item
;; scroll bars are not visibly enabled unless the window they're on
;; is the top window.

(defmethod dialog-item-enable ((item scroll-bar-dialog-item))
  (unless (dialog-item-enabled-p item)
    (setf (dialog-item-enabled-p item) t)
    (view-activate-event-handler item)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-proc
;;
;;this is the hook function which is passed to _TrackControl.  The toolbox
;;  will call this function periodically as the control is clicked.
;;
;; It calls track-scroll-bar every time the ROM calls it.
;; The default version of track-scroll-bat updates the
;; scroll bar position according to the scroll-bar-scroll-size or
;; scroll-bar-page-size and calls dialog-item-action.
;; User's may shadow the default method if they need custom behavior.

(defvar *scroll-bar-item* nil)

(defpascal scroll-bar-proc (:ptr sb-handle :word part)
  "This procedure adjusts the control value, and calls dialog-item-action."
  (let ((item *scroll-bar-item*))
    (track-scroll-bar
     item
     (_GetCtlValue :ptr sb-handle :word)
     (case part
       (#.$InUpButton :in-up-button)
       (#.$InDownButton :in-down-button)
       (#.$InPageUp :in-page-up)
       (#.$InPageDown :in-page-down)
       (#.$InThumb :in-thumb)
       (t nil)))))

;; Unfortunately, the ROM is brain-damaged, so we have to do this ourselves.
(defun track-scroll-bar-thumb (item)
  (let* ((old-setting (scroll-bar-setting item))
         (min (scroll-bar-min item))
         (max (scroll-bar-max item))
         (horizontal? (eq (scroll-bar-direction item) :horizontal))
         (position (view-position item))
         (last-mouse (rref *current-event* :eventRecord.where))
         (size (view-size item))
         width length old-mouse left right mouse)
    (let ((view item))
      (while view                       ; local-to-global
        (setq last-mouse (subtract-points last-mouse (view-position view)))
        (setq view (view-container view))))
    (if horizontal?
      (setq width (point-v size)
            length (- (point-h size) width width width)
            left (+ (round (* width 3) 2) (point-h position))
            old-mouse (point-h last-mouse))
      (setq width (point-h size)
            length (- (point-v size) width width width)
            left (+ (round (* width 3) 2) (point-v position))
            old-mouse (point-v last-mouse)))
    (setq right (+ left length))
    (loop
      (unless (mouse-down-p) (return))
      (setq mouse (view-mouse-position item))
      (unless (eql mouse last-mouse)
        (setq last-mouse mouse)
        (setq mouse (if horizontal? (point-h mouse) (point-v mouse)))
        (track-scroll-bar
         item
         (min max
              (max min
                   (+ old-setting
                      (round (* (- mouse old-mouse) (- max min)) (- right left)))))
         :in-thumb)))))

; Returns the new value for the scroll bar
(defmethod track-scroll-bar ((item scroll-bar-dialog-item) value part)
  (set-scroll-bar-setting 
   item
   (case part
     (:in-up-button (- value (scroll-bar-scroll-size item)))
     (:in-down-button (+ value (scroll-bar-scroll-size item)))
     (:in-page-up (- value (scroll-bar-page-size item)))
     (:in-page-down (+ value (scroll-bar-page-size item)))
     (t value)))
  (dialog-item-action item))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;view-click-event-handler
;;
;;this is the function which is called when the user clicks in the scroll-bar
;;
;;It checks the scroll-bar part, and calls _TrackControl
;;  If appropriate, it passes a hook function to _TrackControl
;;
;;During tracking, dialog-item-action is repeatedly called.
;;

(defmethod view-click-event-handler ((item scroll-bar-dialog-item) where)
  (let* ((sb-handle (dialog-item-handle item))
         (part (_TestControl :ptr sb-handle :long where :word))
         (*scroll-bar-item* item))
    (cond ((eq part #.$InThumb)
           (if (scroll-bar-track-thumb-p item)
             (track-scroll-bar-thumb item)
             (progn (_TrackControl :ptr sb-handle
                                   :long where
                                   :ptr (%int-to-ptr -1)
                                   :word)
                    (dialog-item-action item))))
          ((memq part '(#.$InUpButton #.$InDownButton
                        #.$InPageUp #.$InPageDown))
           (_TrackControl :ptr sb-handle
                          :long where
                          :ptr scroll-bar-proc
                          :word)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;dialog-item-action
;;
;;The default dialog-item-action for a scroll bar calls
;;scroll-bar-changed on the scrollee
;;
(defmethod dialog-item-action ((item scroll-bar-dialog-item))
  (let ((f (dialog-item-action-function item)))
    (if f
      (funcall f item)
      (let ((scrollee (scroll-bar-scrollee item)))
        (when scrollee
          (scroll-bar-changed scrollee item))))))

(defmethod scroll-bar-changed (view scroll-bar)
  (declare (ignore view scroll-bar)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-setting
;;
;;a nice safe Lisp-level function for getting the value of the scroll-bar
;;

(defmethod scroll-bar-setting ((item scroll-bar-dialog-item))
  (let ((handle (dialog-item-handle item)))
    (if handle
      (_GetCtlValue :ptr handle :word)
      (slot-value item 'setting))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;(setf scroll-bar-setting)
;;
;;a nice safe Lisp-level function for changing the value of the scroll-bar
;;

(defmethod (setf scroll-bar-setting) (new-value (item scroll-bar-dialog-item))
  (set-scroll-bar-setting item new-value))

(defmethod set-scroll-bar-setting ((item scroll-bar-dialog-item) new-value)
  (setq new-value (require-type new-value 'fixnum))
  (unless (eql new-value (scroll-bar-setting item))
    (let ((handle (dialog-item-handle item)))
      (when handle
        (with-focused-view (view-container item)
          (_SetCtlValue :ptr handle :word new-value)))
      (setf (slot-value item 'setting) new-value))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-min is a :reader for the class
;;here's the setter
;;
(defmethod (setf scroll-bar-min) (new-value (item scroll-bar-dialog-item))
  (set-scroll-bar-min item new-value))

(defmethod set-scroll-bar-min ((item scroll-bar-dialog-item) new-value)
  (setq new-value (require-type new-value 'fixnum))
  (unless (eql new-value (scroll-bar-min item))
    (let ((handle (dialog-item-handle item)))
      (when handle
        (with-focused-view (view-container item)
          (_SetMinCtl :ptr handle :word new-value)))
      (setf (slot-value item 'min) new-value))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-max is a :reader for the class
;;here's the setter
;;
(defmethod (setf scroll-bar-max) (new-value (item scroll-bar-dialog-item))
  (set-scroll-bar-max item new-value))

(defmethod set-scroll-bar-max ((item scroll-bar-dialog-item) new-value)
  (setq new-value (require-type new-value 'fixnum))
  (unless (eql new-value (scroll-bar-max item))
    (let ((handle (dialog-item-handle item)))
      (when handle
        (with-focused-view (view-container item)
          (_SetMaxCtl :ptr handle :word new-value)))
      (setf (slot-value item 'max) new-value))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-length
;;
;;this is a variation of view-size
;;
;;It only used one dimension, since scroll-bars almost always have a width
;;  of 16 pixels.
;;

(defmethod scroll-bar-length ((item scroll-bar-dialog-item))
  (let* ((size (view-size item))
         (splitter (pane-splitter item))
         (splitter-size (and splitter (view-size splitter))))
    (if (eq (scroll-bar-direction item) :horizontal)
      (+ (point-h size) (if splitter (point-h splitter-size) 0))
      (+ (point-v size) (if splitter (point-v splitter-size) 0)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;set-scroll-bar-length
;;
;;sets the length of the scroll-bar
;;

(defun (setf scroll-bar-length) (new-length scroll-bar-dialog-item)
  (set-scroll-bar-length scroll-bar-dialog-item new-length))

(defmethod set-scroll-bar-length ((item scroll-bar-dialog-item) new-length)
  (let ((splitter (pane-splitter item))
        (direction (scroll-bar-direction item)))
    (when splitter
      (let ((size (view-size splitter)))
        (decf new-length
              (min new-length
                   (if (eq direction :horizontal) (point-h size) (point-v size))))))
    (set-view-size item (if (eq direction :horizontal)
                          (make-point new-length (scroll-bar-width item))
                          (make-point (scroll-bar-width item) new-length)))
    (when splitter
      (set-view-position item (view-position item)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;scroll-bar-width
;;
;; Sometimes you want a different width
;;
(defmethod scroll-bar-width ((item scroll-bar-dialog-item))
  (let ((size (view-size item)))
    (if (eq (scroll-bar-direction item) :horizontal)
      (point-v size)
      (point-h size))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;set-scroll-bar-width
;;
;;sets the width of the scroll-bar
;;

(defun (setf scroll-bar-width) (new-length scroll-bar-dialog-item)
  (set-scroll-bar-width scroll-bar-dialog-item new-length))

(defmethod set-scroll-bar-width ((item scroll-bar-dialog-item) new-width)
  (let ((splitter (pane-splitter item)))
    (if splitter (set-scroll-bar-width splitter new-width)))
  (let ((size (view-size item)))
    (set-view-size item (if (eq (scroll-bar-direction item) :horizontal)
                          (make-point (point-h size) new-width)
                          (make-point new-width (point-v size))))))

(defmethod set-scroll-bar-width ((item scroll-bar-dialog-item) new-width)
  (let ((size (view-size item)))
    (set-view-size item (if (eq (scroll-bar-direction item) :horizontal)
                          (make-point (point-h size) new-width)
                          (make-point new-width (point-v size))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;(setf scroll-bar-scrollee)
;;
;;Change the scrollee of a scroll-bar
;;
(defun (setf scroll-bar-scrollee) (new-scrollee scroll-bar-dialog-item)
  (set-scroll-bar-scrollee scroll-bar-dialog-item new-scrollee))

(defmethod set-scroll-bar-scrollee ((item scroll-bar-dialog-item) new-scrollee)
  (let ((old-scrollee (scroll-bar-scrollee item)))
    (when old-scrollee
      (delete-view-scroll-bar old-scrollee item)))
  (add-view-scroll-bar new-scrollee item)
  (let ((splitter (pane-splitter item)))
    (if splitter (set-scroll-bar-scrollee splitter new-scrollee)))
  (setf (slot-value item 'scrollee) new-scrollee))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; pass set-view-container and set-view-position
;; to the pane-splitter
;;
(defmethod set-view-container ((item scroll-bar-dialog-item) new-container)
  (let ((splitter (pane-splitter item)))
    (when splitter
      (set-view-container splitter new-container))
    (call-next-method)))

(defmethod set-view-position ((item scroll-bar-dialog-item) h &optional v)
  (let ((pos (make-point h v))
        (splitter (pane-splitter item))
        (splitter-position (pane-splitter-position item)))
    (setq h (point-h pos) v (point-v pos))
    (when splitter
      (let ((size (view-size item))
            (s-size (view-size splitter)))
        (if (eq (scroll-bar-direction item) :horizontal)
          (if (eq splitter-position :left)
            (progn (set-view-position splitter pos)
                   (incf h (point-h s-size)))
            (set-view-position splitter (+ h (point-h size)) v))
          (if (eq splitter-position :top)
            (progn (set-view-position splitter pos)
                   (incf v (point-v s-size)))
            (set-view-position splitter h (+ v (point-v size))))))))
  (call-next-method item h v))
    
(defmethod corrected-view-position ((item scroll-bar-dialog-item))
  (let ((splitter (pane-splitter item)))
    (if (and splitter (memq (pane-splitter-position item) '(:top :left)))
      (view-position splitter)
      (view-position item))))

(defmethod set-pane-splitter-position ((item scroll-bar-dialog-item) pos)
  (let ((position (corrected-view-position item)))
    (setf (slot-value item 'pane-splitter-position) pos)
    (set-view-position item position))
  pos)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Methods for pane-splitter
;;
(defmethod initialize-instance ((item pane-splitter) &rest initargs
                                &key (width 16) (length 5) (direction :vertical))
  (declare (dynamic-extent initargs))
  (let ((size (if (eq direction :vertical)
                (make-point width length)
                (make-point length width))))
    (apply #'call-next-method
           item
           :view-size size
           :direction direction
           initargs)))

(defmethod view-draw-contents ((item pane-splitter))
  (let* ((tl (view-position item))
         (br (add-points tl (view-size item))))
    (rlet ((r :rect :topleft tl :botright br))
      (_FillRect :ptr r :ptr *black-pattern*))))

(defmethod view-click-event-handler ((item pane-splitter) where)
  (declare (ignore where))
  (let* ((scrollee (or (scroll-bar-scrollee item) (view-window item)))
         (window (view-window item))
         (scroll-bar (scroll-bar item)))
    (when window
      (multiple-value-bind (s-tl s-br)
                           (pane-splitter-corners scrollee scroll-bar)
        (let ((wait-ticks (max 1 (floor internal-time-units-per-second 30)))
              (direction (scroll-bar-direction item))
              (win-min -20)
              (mouse-pos (view-mouse-position window))
              min max min-pos max-pos drawn time pos pos-accessor line-direction
              win-accessor win-max)
          (if (eq (scroll-bar-direction item) :vertical)
            (setq min (1+ (point-h s-tl))
                  max (- (point-h s-br) 2)
                  min-pos (1+ (point-v s-tl))
                  max-pos (- (point-v s-br) 2)
                  pos (point-v mouse-pos)
                  pos-accessor #'point-v
                  win-accessor #'point-h
                  win-max (+ 20 (point-h (view-size window)))
                  line-direction :horizontal)
            (setq min (1+ (point-v s-tl))
                  max (- (point-v s-br) 2)
                  min-pos (1+ (point-h s-tl))
                  max-pos (- (point-h s-br) 2)
                  pos (point-h mouse-pos)
                  pos-accessor #'point-h
                  win-accessor #'point-v
                  win-max (point-v (view-size window))
                  line-direction :vertical))
          (flet ((draw-line (pos)
                   (draw-pane-splitter-outline
                    scrollee scroll-bar pos min max line-direction)
                   (setq drawn (not drawn)
                         time (get-internal-run-time))))
            (declare (dynamic-extent draw-line))
            (with-focused-view window
              (with-pen-saved
                (_PenPat :ptr *gray-pattern*)
                (_PenMode :word (position :srcxor *pen-modes*))
                (draw-line pos)
                (unwind-protect
                  (loop
                    (unless (mouse-down-p) (return))
                    (let* ((new-mouse (view-mouse-position window))
                           (new-pos (funcall pos-accessor new-mouse))
                           (in-window (<= win-min
                                          (funcall win-accessor new-mouse)
                                          win-max)))
                      (unless (or (eql mouse-pos new-mouse)
                                  (<= (get-internal-run-time) (+ time wait-ticks)))
                        (when (and drawn (or (not (eql new-pos pos)) (not in-window)))
                          (draw-line pos))
                        (setq pos new-pos mouse-pos new-mouse)
                        (when (and (not drawn) (<= min-pos pos max-pos) in-window)
                          (draw-line pos)))))
                  (when drawn 
                    (draw-line pos)
                    (setq drawn t))))))
          (setq pos (funcall pos-accessor (convert-coordinates 
                                           (if (eq direction :horizontal)
                                             (make-point pos 0)
                                             (make-point 0 pos))
                                           window 
                                           scrollee)))
          (split-pane scrollee scroll-bar pos direction drawn))))))

(defmethod draw-pane-splitter-outline (scrollee scroll-bar pos min max direction)
  (declare (ignore scrollee scroll-bar))
  (if (eq direction :horizontal)
    (progn (_MoveTo :word min :word pos)
           (_LineTo :word max :word pos))
    (progn (_MoveTo :word pos :word min)
           (_LineTo :word pos :word max))))

; Some users may want to specialize on this
(defmethod pane-splitter-corners ((scrollee simple-view) scroll-bar)
  (declare (ignore scroll-bar))
  (let* ((window (view-container scrollee))
         (container (view-container scrollee)))
    (multiple-value-bind (tl br) (view-corners scrollee)
      (when (and container (neq container window))
        (setq tl (convert-coordinates tl container window)
              br (convert-coordinates br container window)))
      (values tl br))))

; This is the method that all users will specialize on if they
; want a pane-splitter to do anything but draw a line.
(defmethod split-pane ((scrollee simple-view) scroll-bar pos direction inside-limits)
  (declare (ignore scroll-bar pos direction inside-limits)))


(provide 'scroll-bar-dialog-items)

#|
;; a simple example use

(setq my-dialog (make-instance 'dialog
                       :view-size #@(250 75)
                       :window-title "Scroll Bar Example"))

(setq my-scroll (make-instance
                 'scroll-bar-dialog-item
                 :view-position #@(25 50)
                 :direction :horizontal
                 :length 200
                 :dialog-item-action
                 #'(lambda (item &aux (setting (format nil "~a"
                                                       (scroll-bar-setting item))))
                     (set-dialog-item-text
                      (find-named-sibling item 'display-text)
                      setting)
                     (window-update-event-handler (view-window item)))))

(add-subviews my-dialog
              (make-instance 'static-text-dialog-item
                     :dialog-item-text "000"
                     :view-nick-name 'display-text)
                my-scroll)

(set-part-color my-scroll :frame *purple-color*)
(set-part-color my-scroll :thumb *green-color*)

|#

