;;;____________________________________________________________________________________
;;;                                  GRAPHER
;;;
;;; The code in this file is designed to work in conjunction with FOCL.
;;; Generates graphic representations of FOCL rules, and handles the windows those
;;; graphs are displayed in.
;;;
;;;  Created and designed by Clifford A. Brunk 1991
;;;
;;;  Problems:
;;;
;;;  rewritten in CLOS under MACL 2.0b1 7/31/91
;;;
;;;  In theory the grapher is no longer limited to any set graph size.  Graphs
;;;  start at (20 20) and can grow unboundedly in any direction.  This was accomplished
;;;  by adding a view-left and view-top to the graph-view then subtracting
;;;  (view-left view-top) from all point before performing graphic operations
;;;  thus all manipulations done by QuickDraw are performed in the range (0 0) to
;;;  view-size.  The drawing routines have been modified to perform internal
;;;  clipping.  Thus even with huge graphs only the portion which is on screen will
;;;  ever be drawn.  This approach might decrease the performace of the grapher
;;;  in some circumstances (the overhead of translating points and clipping drawing)
;;;  but on average it will be a win, and a big win because of the capacity to handle
;;;  large graphs.
;;;
;;;____________________________________________________________________________________

(in-package :user)

;;;_____________________________________________________________________________
;;;   parameters

(defparameter *min-window-height* 50)       ;;;  Window Parameters
(defparameter *min-window-width* 100)
(defparameter *scroll-bar-width* 15)
(defparameter *size-box-inner-dimension* 13)
(defparameter *scroll-bar-point* (make-point *scroll-bar-width* *scroll-bar-width*))

(defparameter *graph-boarder* 20)           ;;; Graph Parameters
(defparameter *graph-start-h* *graph-boarder*)
(defparameter *graph-start-v* *graph-boarder*)
(defparameter *and-arc-ratio* .75)

(defparameter *downcase-preds* t)
(defparameter *display-prolog* t)
(defparameter *default-display-deleted-nodes* t)
(defparameter *default-node-selection-constraint* nil)
(defparameter *default-edit-kind* nil)

(defparameter *default-font* '("Monaco" 9 :srcor :plain))
(defparameter *default-orientation* :horizontal)
(defparameter *default-expand* :first-use)
(defparameter *clip-graph-to-view* t)


(defparameter *unoperationalized-pen-size* #@(2 2))
(defparameter *unoperationalized-pen-pattern* *gray-pattern*)
(defparameter *unoperationalized-fill-pattern* *light-gray-pattern*)
(defparameter *unoperationalized-frame-color* *gray-color*)
(defparameter *unoperationalized-fill-color* *light-gray-color*)

(defparameter *normal-pen-size* #@(1 1))
(defparameter *normal-pen-pattern* *black-pattern*)
(defparameter *normal-fill-pattern* *white-pattern*)
(defparameter *normal-frame-color* *black-color*)
(defparameter *normal-fill-color* *white-color*)

(defparameter *extensional-pen-size* #@(2 2))
(defparameter *extensional-pen-pattern* *black-pattern*)
(defparameter *extensional-fill-pattern* *white-pattern*)
(defparameter *extensional-frame-color* *red-color*)
(defparameter *extensional-fill-color* *white-color*)

(defparameter *builtin-pen-size* #@(2 2))
(defparameter *builtin-pen-pattern* *black-pattern*)
(defparameter *builtin-fill-pattern* *white-pattern*)
(defparameter *builtin-frame-color* *pink-color*)
(defparameter *builtin-fill-color* *white-color*)

(defparameter *intensional-pen-size* #@(2 2))
(defparameter *intensional-pen-pattern* *black-pattern*)
(defparameter *intensional-fill-pattern* *white-pattern*)
(defparameter *intensional-frame-color* *orange-color*)
(defparameter *intensional-fill-color* *white-color*)

(defparameter *cliche-pen-size* #@(2 2))
(defparameter *cliche-pen-pattern* *black-pattern*)
(defparameter *cliche-fill-pattern* *white-pattern*)
(defparameter *cliche-frame-color* *blue-color*)
(defparameter *cliche-fill-color* *white-color*)

(defparameter *determinate-pen-size* #@(2 2))
(defparameter *determinate-pen-pattern* *black-pattern*)
(defparameter *determinate-fill-pattern* *white-pattern*)
(defparameter *determinate-frame-color* *green-color*)
(defparameter *determinate-fill-color* *white-color*)

(defparameter *ebl-pen-size* #@(1 1))
(defparameter *ebl-pen-pattern* *black-pattern*)
(defparameter *ebl-fill-pattern* *white-pattern*)
(defparameter *ebl-frame-color* *black-color*)
(defparameter *ebl-fill-color* *white-color*)

(defparameter *error-pen-size* #@(2 2))
(defparameter *error-pen-pattern* *black-pattern*)
(defparameter *error-fill-pattern* *black-pattern*)
(defparameter *error-frame-color* *red-color*)
(defparameter *error-fill-color* *yellow-color*)



;;____________________________________________________________________________________
;;   Utility Fuctions
;;____________________________________________________________________________________

;;____________________________________________________________________________________
;;   window-p

(defun window-p (variable)
  (if (and (ccl::inherit-from-p variable 'window)
           (wptr variable))
    t nil))

;;____________________________________________________________________________________
;;   GRAPH-WINDOW Class and Methods
;;____________________________________________________________________________________


;;____________________________________________________________________________________
;;   graph-window

(defclass graph-window (window)
  ((kind :initarg :kind :initform :graph :accessor kind)
   (edit-kind :initarg :edit-kind :initform *default-edit-kind* :accessor edit-kind)
   (controller :initarg :controller :initform nil :accessor controller)
   ))

;;____________________________________________________________________________________
;;   initialize-instance

(defmethod initialize-instance ((window graph-window) &rest initargs)
  (setf (getf initargs :color-p) t)
  (apply #'call-next-method window initargs )
  (set-view-scroll-position window 0 0)
  (set-window-font window *default-font*)
  (let* ((window-size-h (point-h (view-size window)))
         (window-size-v (point-v (view-size window))))
    (add-subviews window
                  (make-instance 'graph-scroller
                                 :view-position #@(0 0)
                                 :view-size (make-point window-size-h window-size-v)
                                 :view-nick-name :graph-scroller)))
  (setf (graph-window (view-named :graph-view (view-named :graph-scroller window))) window)
  (set-font-dependent-attributes (view-named :graph-view (view-named :graph-scroller window))))

;;____________________________________________________________________________________
;;  set-view-size

(defmethod set-view-size ((window graph-window) h &optional (v nil))
  (call-next-method)
  (reset-view-size (view-named :graph-scroller window) h v))

;;____________________________________________________________________________________
;;   resize-window

(defmethod resize-window ((window graph-window) &optional (h nil) (v nil))
  (let* ((view (view-named :graph-view (view-named :graph-scroller window)))
         (real-h (if h h (min (+ (- (graph-right view) (graph-left view))
                                 *graph-boarder* *graph-boarder* *scroll-bar-width*)
                              (- *screen-width* 6))))
         (real-v (if v v (min (+ (- (graph-bottom view) (graph-top view))
                                 *graph-boarder* *graph-boarder* *scroll-bar-width*)
                              (- *screen-height* 44)))))
    (set-view-size window real-h real-v)))


;;____________________________________________________________________________________
;;   grow-window-if-needed

(defmethod grow-window-if-needed ((window graph-window))
  (let* ((view (view-named :graph-view (view-named :graph-scroller window)))
         (current-h (point-h (view-size window)))
         (current-v (point-v (view-size window)))
         (new-h (max current-h
                     (min (+ (- (graph-right view) (graph-left view))
                             *graph-boarder* *graph-boarder* *scroll-bar-width*)
                          (- *screen-width* 6))))
         (new-v (max current-v
                     (min (+ (- (graph-bottom view) (graph-top view))
                             *graph-boarder* *graph-boarder* *scroll-bar-width*)
                          (- *screen-height* 44)))))
    (if (or (> new-h current-h)
            (> new-v current-v))
      (set-view-size window new-h new-v)
      nil)))

;;____________________________________________________________________________________
;;   window-zoom-event-handler

(defmethod window-zoom-event-handler ((window graph-window) message)
  (declare (ignore message))
  (call-next-method)
  (reset-view-size (view-named :graph-scroller window) (view-size window))
  )



;;____________________________________________________________________________________
;; auto-position-window

(defmethod auto-position-window ((window graph-window) &key (centered nil))
   (let ((old-h (point-h (view-position window)))
         (old-v (point-v (view-position window)))
         (view-size-h (point-h (view-size window)))
         (view-size-v (point-v (view-size window))))
     (if centered
       (set-view-position window
                          (max 3 (round (- *screen-width* view-size-h) 2))
                          (max 41 (round (- *screen-height* view-size-v) 2)))
       (set-view-position window
                          (if (> (+ old-h view-size-h 3) *screen-width*)
                            (max 3 (- *screen-width* view-size-h 3))
                            old-h)
                          (if (> (+ old-v view-size-v 3) *screen-height*)
                            (max 41 (- *screen-height* view-size-v 3))
                            old-v)))))


;;____________________________________________________________________________________
;;   view-key-event-handler

(defmethod view-key-event-handler ((window graph-window) char)
  ;  (format t "CHARACTER : ~A      CHAR-CODE :~A~%" char  (char-code char))
  (let* ((view (view-named :graph-view (view-named :graph-scroller window)))
         (learned-concept (eql (kind window) :learned-concept)))
    (case (char-code char)
      ((69 101)    ;; E e
       (if (fboundp 'edit-node)
         (edit-node (last-node-selected view))))
      ((71 103)    ;; G g
       (regraph-node view))
      ((72 104)    ;; H h
       (hide-antecedents-of-selected-nodes view))
      ((73 105)    ;; I i
       (if learned-concept
         (inspect-last-node-selected-as-literal view)
         (inspect-last-node-selected-as-pred view)))
      ((82 114)    ;; R r
       (force-graph-redraw view))
      ((83 115)    ;; S s
       (show-antecedents-of-selected-nodes view))
      (others
       nil))))


;;____________________________________________________________________________________
;;  graph-setup

(defmethod graph-setup ((window graph-window))
  (let* ((view (view-named :graph-view (view-named :graph-scroller window)))
         (title (window-title window))
         (font (window-font window))
         (orientation (graph-orientation view))
         (expand (graph-expand view))
         (changed nil))
    (multiple-value-bind  
      (new-font
       new-orientation
       new-expand)
      (values-list (get-graph-setup title
                                    font
                                    orientation
                                    expand))
      (erase-view window)
      (unless (equal new-font font)
        (set-window-font window '("Monaco" 12 :plain))  ;; needed to remove previous styles
        (set-window-font window  new-font)
        (set-font-dependent-attributes view)
        (setf changed t))
      
      (if changed
        (resize-all-nodes view))
      
      (unless (equal new-expand expand)
        (setf (graph-expand view) new-expand
              changed t)
        (case new-expand
          (:always (display-all-uses view))
          (:first-use (display-only-first-use view))))
      
      (unless (equal new-orientation orientation)
        (setf (graph-orientation view) new-orientation
              changed t))
      (when changed
        (position-nodes view)
        (resize-window window)
        (auto-position-window window)
        (with-focused-view view
          (position-graph view :centered t)))
      (invalidate-view window))))


;;____________________________________________________________________________________
;; export-graph-picture

(defmethod export-graph-picture ((window graph-window))

  (let* ((view (view-named :graph-view (view-named :graph-scroller window)))
         (view-left (view-left view))
         (view-top (view-top view))
         (port-h (- (rref (wptr view) :window.portRect.right)
                    (rref (wptr view) :window.portRect.left)))
         (port-v (- (rref (wptr view) :window.portRect.bottom)
                    (rref (wptr view) :window.portRect.top)))
         (g-left (- (graph-left view) 1))
         (g-top (- (graph-top view) 1))
         (frame-h (+ (- (graph-right view) (graph-left view)) 2))
         (frame-v (+ (- (graph-bottom view) (graph-top view)) 2)))

    (cond ((or (> frame-h 32000) (> frame-v 32000))
           (message-dialog "The graph is too large to be exported."
                           :position :centered)
           nil)
                     
          (t (without-interrupts
              (rlet ((picture-frame :rect 
                                    :left 0
                                    :top 0
                                    :right frame-h
                                    :bottom frame-v))
                (with-focused-view view
                  (_PortSize :word frame-h :word frame-v)
                  (_ClipRect :ptr picture-frame)
                  (setf (view-left view) g-left
                        (view-top view) g-top)
                  
                  (prog1
                    (_OpenPicture :ptr picture-frame :ptr)
                    (draw-graph view :fast nil :clipped nil)
                    (_ClosePicture)
                    (setf (view-left view) view-left
                          (view-top view) view-top)
                    (_PortSize :word port-h :word port-v)))))))))

;;____________________________________________________________________________________
;; copy

(defmethod copy ((window graph-window))
  (let ((graph-picture (export-graph-picture window)))
    (when graph-picture
      (put-scrap :PICT graph-picture t))))

;;____________________________________________________________________________________
;;  window-hardcopy
;;
;;  If the graph spans multiple pages this code will tile the output,
;;  however it will not to skip blank pages nor will it respond to the
;;  selection of input pages

(eval-when (eval compile)
  (defparameter $PrintErr #x944)
  (defparameter $prJob.bJDocLoop (+ 62 6))
  (defparameter $iPrStatSize 26)
  (defparameter $bSpoolLoop 1)
  (defparameter $err-printer 94)
  (defparameter $err-printer-load 95)
  (defparameter $err-printer-start 97))

(defun check-for-print-error (&optional (errnum $err-printer))
  (let ((print-error (%get-signed-word $PrintErr)))
    (unless (zerop print-error)
      (ccl::%signal-error errnum print-error))))

(defmethod window-hardcopy ((window graph-window) &optional (show-dialog? t))
  (declare (ignore show-dialog?))
  (window-select window)
  (let ((view (view-named :graph-view (view-named :graph-scroller window)))
        (picture (export-graph-picture window))
        (tile-overlap 15))
    (when picture
      (unwind-protect
        (with-cursor *arrow-cursor*
          (#_PrOpen)
          ;        (check-for-print-error $err-printer-load)
          (let ((print-record (get-print-record)))
            (when (_PrJobDialog :ptr print-record :boolean)
              (let ((*hc-page-open-p* nil)
                    (ccl::*inhibit-error* t)
                    err)
                (declare (special *hc-page-open-p* ccl::*inhibit-error*))
                (setq 
                 err 
                 ;              (catch-error-quietly
                 (without-interrupts
                  (let* ((g-left (graph-left view))
                         (g-top (graph-top view))
                         (g-right (graph-right view))
                         (g-bottom (graph-bottom view))
                         (g-size-h (- g-right g-left))
                         (g-size-v (- g-bottom g-top))
                         (hardcopy-ptr (_PrOpenDoc :ptr print-record :long 0 :long 0 :ptr))
                         (page-size-h (rref hardcopy-ptr :grafport.portRect.right))
                         (page-size-v (rref hardcopy-ptr :grafport.portRect.bottom))
                         (pages-h (ceiling g-size-h  page-size-h))
                         (pages-v (ceiling g-size-v page-size-v)))
                    
                    (view-draw-contents window)
                    
                    ;; Adjust number of pages and page-size for tiles when needed.
                    (if (> pages-h 1)
                      (setf page-size-h (- page-size-h tile-overlap)
                            pages-h (ceiling g-size-h page-size-h)))
                    (if (> pages-v 1)
                      (setf page-size-v (- page-size-v tile-overlap)
                            pages-v (ceiling g-size-v page-size-v)))
                    
                    (unwind-protect
                      (with-dereferenced-handles ((ppRec print-record))
                        pprec
                        ;                       (check-for-print-error $err-printer-start)
                        (unwind-protect
                          (dotimes (page-v pages-v)
                            (dotimes (page-h pages-h)
                              (with-port hardcopy-ptr
                                (_PrOpenPage :ptr hardcopy-ptr :long 0)
                                (rset hardcopy-ptr :grafport.txfont (rref (wptr window) :window.txfont))
                                (rset hardcopy-ptr :grafport.txface (rref (wptr window) :window.txface))
                                (rset hardcopy-ptr :grafport.txsize (rref (wptr window) :window.txsize))
                                
                                (let ((new-origin (make-point (- (* page-h page-size-h))
                                                              (- (* page-v page-size-v)))))
                                  (rlet ((frame :rect 
                                                :topleft new-origin
                                                :bottomright (add-points
                                                              new-origin
                                                              (subtract-points
                                                               (rref picture picture.picframe.bottomright)
                                                               (rref picture picture.picframe.topleft)))))
                                    (_DrawPicture :ptr picture :ptr frame)))
                                (_PrClosPage :ptr hardcopy-ptr))))))
                      (_PrClosDoc :ptr hardcopy-ptr)))
                  (when (eq (%hget-byte print-record $prJob.bJDocLoop)
                            $bSpoolLoop)
                    ;                   (check-for-print-error)
                    (%stack-block ((StRec $iPrStatSize))
                      (_PrPicFile :ptr print-record :long 0 :long 0 :long 0 :ptr StRec))
                    ;                   (check-for-print-error)
                    ))
                 ;                )
                 )
                t))))
        (_PrClose)))))

;;____________________________________________________________________________________
;;  get-node

(defmethod get-node ((window graph-window) 
                     graph-view
                     &key (literal nil)
                          (parent nil)
                          (state :normal)
                          (deleted? nil)
                          (expand t))
  (create-node graph-view 
               :literal literal
               :parent parent
               :deleted? deleted?
               :state state
               :expand expand))

;;____________________________________________________________________________________
;;   window-close

(defmethod window-close ((window graph-window))
  (close-graph-view (view-named :graph-view (view-named :graph-scroller window)))
  (call-next-method))


;;____________________________________________________________________________________
;;   GRAPH-SCROLLER Class and Methods
;;____________________________________________________________________________________

(defclass graph-scroller (view) ())

;;____________________________________________________________________________________
;;   initialize-instance

(defmethod initialize-instance ((view graph-scroller) &rest initargs)
  (apply #'call-next-method view initargs )
  (set-view-scroll-position view 0 0)
  (let* ((scrolling-view-size-h (point-h (view-size view)))
         (scrolling-view-size-v (point-v (view-size view)))
         (view-size-h (- scrolling-view-size-h *scroll-bar-width*))
         (view-size-v (- scrolling-view-size-v *scroll-bar-width*)))

    (add-subviews view
                  (make-instance 'scroll-bar-dialog-item
                                 :view-position (make-point -1 view-size-v)
                                 :direction :horizontal
                                 :length (+ view-size-h 2)
                                 :scroll-size 20
                                 :page-size (round view-size-h 2)
                                 :view-nick-name :h-scroll-bar
                                 :dialog-item-action
                                 #'(lambda (item)
                                     (set-view-left (find-named-sibling item :graph-view)
                                                    (scroll-bar-setting item))))
                  (make-instance 'scroll-bar-dialog-item
                                 :view-position (make-point view-size-h -1)
                                 :direction :vertical
                                 :length (+ view-size-v 2)
                                 :scroll-size 20
                                 :page-size (round view-size-v 2)
                                 :view-nick-name :v-scroll-bar
                                 :dialog-item-action
                                 #'(lambda (item)
                                     (set-view-top (find-named-sibling item :graph-view)
                                                   (scroll-bar-setting item))))
                  (make-instance 'graph-view
                                 :view-position #@(0 0)
                                 :view-size (make-point view-size-h view-size-v)
                                 :view-nick-name :graph-view)))
)

;;____________________________________________________________________________________
;;  reset-view-size

(defmethod reset-view-size ((view graph-scroller) h &optional (v nil))
  (if (null v)
    (setf v (point-v h)
          h (point-h h)))
  (let ((graph-view (view-named :graph-view view))
        (h-scroll-bar (view-named :h-scroll-bar view))
        (v-scroll-bar (view-named :v-scroll-bar view))
        (graph-view-size-h (- h *scroll-bar-width*))
        (graph-view-size-v (- v *scroll-bar-width*)))
    
    (set-scroll-bar-length h-scroll-bar (+ graph-view-size-h 2))
    (set-scroll-bar-length v-scroll-bar (+ graph-view-size-v 2))
    (set-view-position h-scroll-bar -1 graph-view-size-v)
    (set-view-position v-scroll-bar graph-view-size-h -1)
    (set-view-size graph-view graph-view-size-h graph-view-size-v)
    (set-view-size view h v)
    (position-graph graph-view)
    ))

;;____________________________________________________________________________________
;;  reset-scroll-bars

(defmethod reset-scroll-bars ((view graph-scroller))
  (let* ((graph-view (view-named :graph-view view))
         (graph-view-h (point-h (view-size graph-view)))
         (graph-view-v (point-v (view-size graph-view)))
         (graph-view-left (view-left graph-view))
         (graph-view-top (view-top graph-view))
         (min-h (- (graph-left graph-view) *graph-boarder*))
         (max-h (- (+ (graph-right graph-view) *graph-boarder*) graph-view-h))
         (min-v (- (graph-top graph-view) *graph-boarder*))
         (max-v (- (+ (graph-bottom graph-view) *graph-boarder*) graph-view-v))
         (h-scroll-bar (view-named :h-scroll-bar view))
         (v-scroll-bar (view-named :v-scroll-bar view)))

    (setf (scroll-bar-page-size h-scroll-bar) (round graph-view-h 2))
    (set-scroll-bar-min h-scroll-bar min-h)
    (set-scroll-bar-max h-scroll-bar max-h)
    (set-scroll-bar-setting h-scroll-bar graph-view-left)
    (if (and (>= min-h graph-view-left)
             (<= max-h graph-view-left))
      (dialog-item-disable h-scroll-bar)
      (dialog-item-enable h-scroll-bar))
    
    (setf (scroll-bar-page-size v-scroll-bar) (round graph-view-v 2))
    (set-scroll-bar-min v-scroll-bar min-v)
    (set-scroll-bar-max v-scroll-bar max-v)
    (set-scroll-bar-setting v-scroll-bar graph-view-top)
    (if (and (>= min-v graph-view-top)
             (<= max-v graph-view-top))
      (dialog-item-disable v-scroll-bar)
      (dialog-item-enable v-scroll-bar))
    ))





;;____________________________________________________________________________________
;;   GRAPH-VIEW Class, Methods and closely related structures and functions
;;____________________________________________________________________________________

;;____________________________________________________________________________________
;; node struct                            

(defstruct (node (:print-function print-node))
  top 
  left
  bottom
  right
  center-h
  center-v
  text-h
  text-v
  literal        ;; the prolog literal (or a negated conjunction) the node prepresents
  text           ;; text in the node graphic (ususally the literal reperesented as a string)
  external-text  ;; text to appear below or behind node (used in operationalization)
  parent         ;; the parent node
  children       ;; an and-or list of child nodes  ((a) (b c d) (e f)) == a + bcd + ef
  selected?      ;; t or nil
  on-screen?     ;; t or nil
  hidden?        ;; t or nil
  deleted?       ;; t or nil
  kind           ;; :intensional, :recursive, :unexpanded :builtin, :extensional, :not, :cut, or :undefined 
  state          ;; :unoperationalized, :normal, :builtin, :cliche, :determinate, :extensional, :intensional
  )

(defun print-node (node stream depth)
  (declare (ignore depth))
  (format stream "~A" (node-text node)))

;;____________________________________________________________________________________
;;   graph-view

(defclass graph-view (view)
  ((last-node-selected :initarg :last-node-selected :initform nil :accessor last-node-selected)
   (last-conjunction-selected :initarg :last-conjunction-selected :initform nil :accessor last-conjunction-selected)
   (node-selection-constraint :initarg :node-selection-constraint :initform *default-node-selection-constraint* :accessor node-selection-constraint)
   (view-left :initarg :view-left :initform 0 :accessor view-left)
   (view-top :initarg :view-top :initform 0 :accessor view-top)
   (graph-root :initarg :graph-root :initform nil :accessor graph-root)
   (graph-left :initarg :graph-left :initform *graph-start-h* :accessor graph-left)
   (graph-top :initarg :graph-top :initform *graph-start-v* :accessor graph-top)
   (graph-right :initarg :graph-right :initform *graph-start-h*  :accessor graph-right)
   (graph-bottom :initarg :graph-bottom :initform *graph-start-v* :accessor graph-bottom)
   (graph-orientation :initarg :graph-orientation :initform *default-orientation* :accessor graph-orientation)
   (graph-expand :initarg :graph-expand :initform *default-expand* :accessor graph-expand)
   (display-deleted-nodes :initarg :display-deleted-nodes :initform *default-display-deleted-nodes* :accessor display-deleted-nodes)
   (substitution :initarg :substitution :initform (list t) :accessor substitution)
   (node-height :initarg :node-height :initform 20 :accessor node-height)
   (node-text-offset :initarg :node-text-offset :initform nil :accessor node-text-offset)
   (cut-node-text-offset :initarg :cut-node-text-offset :initform 4 :accessor cut-node-text-offset)
   (corner :initarg :corner :initform 2 :accessor corner)
   (graph-window :initarg :graph-window :initform nil :accessor graph-window)
   ))

;;____________________________________________________________________________________
;;   graph-view

(defmethod close-graph-view ((view graph-view))
  (setf (last-node-selected view) nil
        (node-selection-constraint view) nil
        (view-left view) nil
        (view-top view) nil
        (graph-root view) nil
        (graph-left view) nil
        (graph-top view) nil
        (graph-right view) nil
        (graph-bottom view) nil
        (graph-orientation view) nil
        (graph-expand view) nil
        (display-deleted-nodes view) nil
        (substitution view) nil
        (node-height view) nil
        (node-text-offset view) nil
        (cut-node-text-offset view) nil
        (corner view) nil
        (graph-window view) nil))

;;____________________________________________________________________________________
;;   position-graph

(defmethod position-graph ((view graph-view) &key (centered nil))
  (without-interrupts
   (let* ((g-left (- (graph-left view) *graph-boarder*))
          (g-top (- (graph-top view) *graph-boarder*))
          (g-right (+ (graph-right view) *graph-boarder*))
          (g-bottom (+ (graph-bottom view) *graph-boarder*))
          (view-size (view-size view))
          (view-left (view-left view))
          (view-top (view-top view))
          (view-h (point-h view-size))
          (view-v (point-v view-size)))
     (if centered
       (let ((center-node (if (node-hidden? (graph-root view))
                            (caar (node-children (graph-root view)))
                            (graph-root view)))
             (g-h (- g-right g-left))
             (g-v (- g-bottom g-top)))
         (setf (view-left view)
               (if (> g-h view-h)
                 (case (graph-orientation view)
                   (:horizontal g-left)
                   (:diagonal g-left)
                   (:vertical (max (- (node-center-h center-node) (round view-h 2)) g-left)))
                 g-left)
               
               (view-top view)
               (if (> g-v view-v)
                 (case (graph-orientation view)
                   (:horizontal (max (- (node-center-v center-node) (round view-v 2)) g-top))
                   (:vertical g-top)
                   (:diagonal g-top))
                 g-top)))
       
       (setf (view-left view)
             (if (and (< g-left view-left) (< g-right (+ view-left view-h)))
               (- view-left (min (- view-left g-left ) (- (+ view-left view-h) g-right )))
               view-left)
             
             (view-top view)
             (if (and (< g-top view-top) (< g-bottom (+ view-top view-v)))
               (- view-top (min (- view-top g-top) (- (+ view-top view-v) g-bottom)))
               view-top)
             ))
     (force-graph-redraw view)
     (reset-scroll-bars (view-container view))
     )))




;;____________________________________________________________________________________
;;  set-view-left

(defmethod set-view-left ((view graph-view) left)
  (without-interrupts
   (let ((delta-h (- (view-left view) left))
         (reg (_newrgn :ptr)))
     (rlet ((view-rect :rect
                       :topleft #@(0 0)
                       :bottomright (view-size view)))
       (_ScrollRect :ptr view-rect
                    :long (make-point delta-h 0)
                    :ptr reg)
       (_InvalRgn :ptr reg)
       (_DisposRgn :ptr reg)
       (setf (view-left view) left)
       (_BeginUpdate :ptr (wptr view))
       (draw-graph view :fast nil :clipped t)
       (_EndUpdate :ptr (wptr view))
       (_ValidRect :ptr (rref (wptr view) :window.portRect))))))

;;____________________________________________________________________________________
;;  set-view-top

(defmethod set-view-top ((view graph-view) top)
  (without-interrupts
   (let ((delta-v (- (view-top view) top))
         (reg (_newrgn :ptr)))
     (rlet ((view-rect :rect
                       :topleft #@(0 0)
                       :bottomright (view-size view)))
       (_ScrollRect :ptr view-rect
                    :long (make-point 0 delta-v)
                    :ptr reg)
       (_InvalRgn :ptr reg)
       (_DisposRgn :ptr reg)
       (setf (view-top view) top)
       (_BeginUpdate :ptr (wptr view))
       (draw-graph view :fast nil :clipped t)
       (_EndUpdate :ptr (wptr view))
       (_ValidRect :ptr (rref (wptr view) :window.portRect))))))

;;____________________________________________________________________________________
;;   set-view-left-top

(defmethod set-view-left-top ((view graph-view) left top)
  (without-interrupts
   (let ((delta-h (- (view-left view) left))
         (delta-v (- (view-top view) top))
         (reg (_newrgn :ptr)))
     (setf (view-left view) left
           (view-top view) top)
     (rlet ((view-rect :rect
                       :topleft #@(0 0)
                       :bottomright (view-size view)))
       (cond ((or (> (abs delta-h) (point-h (view-size view)))
                  (> (abs delta-v) (point-v (view-size view))))
              (_EraseRect :ptr view-rect)
              (_InvalRect :ptr view-rect))
             (t
              (_ScrollRect :ptr view-rect
                           :long (make-point delta-h delta-v)
                           :ptr reg)
              (_InvalRgn :ptr reg)
              (_DisposRgn :ptr reg)
              (_BeginUpdate :ptr (wptr view))
              (draw-graph view :fast nil :clipped t)
              (_EndUpdate :ptr (wptr view))
              (_ValidRect :ptr (rref (wptr view) :window.portRect))))))))

;;____________________________________________________________________________________
;;  view-draw-contents

(defmethod view-draw-contents ((view graph-view))
  (without-interrupts
   (draw-graph view :fast nil :clipped t)))

;;____________________________________________________________________________________
;;  force-graph-redraw

(defmethod force-graph-redraw ((view graph-view))
  (with-focused-view view
    (rlet ((view-rect :rect
                      :topleft #@(0 0)
                      :bottomright (view-size view)))
      (_EraseRect :ptr view-rect)
      (_InvalRect :ptr view-rect))))

;;____________________________________________________________________________________
;;  erase-view

(defmethod erase-view ((view view))
  (with-focused-view view
    (rlet ((view-rect :rect
                      :topleft #@(0 0)
                      :bottomright (view-size view)))
      (_EraseRect :ptr view-rect))))

;;____________________________________________________________________________________
;;  redraw-graph

(defmethod redraw-graph ((view graph-view) &key (fast nil))
  (event-dispatch)
  (with-focused-view view
    (erase-view view)
    (draw-graph view :fast fast :clipped t)))

;;____________________________________________________________________________________
;;  draw-graph

(defmethod draw-graph ((view graph-view) &key (fast nil) (clipped t))
  (without-interrupts
   (with-focused-view view
     (let ((root (graph-root view)))
       (when root
         (if fast
           (fast-draw-node-and-children view root)
           (draw-node-and-children view root :clipped clipped)))))))


;;____________________________________________________________________________________
;;  draw-node-and-children

(defmethod draw-node-and-children ((view graph-view) node &key (clipped t))
  (draw-child-connectors view node :clipped clipped)
  (draw-node view node :clipped clipped)
  (dolist (children (node-children node))
    (dolist (child children)
      (draw-node-and-children view child :clipped clipped))))

;;____________________________________________________________________________________
;;  fast-draw-node-and-children

(defmethod fast-draw-node-and-children ((view graph-view) node &key (clipped t))
  (draw-child-connectors view node :clipped clipped)
  (if (or (node-on-screen? node)
          (node-selected? node))
    (draw-node view node :clipped clipped))
  (dolist (children (node-children node))
    (dolist (child children)
      (fast-draw-node-and-children view child :clipped clipped))))

;;____________________________________________________________________________________
;;  set-font-dependent-attributes

(defmethod set-font-dependent-attributes ((view graph-view))
  (let ((h-text-gap 4)
        (v-text-gap 1))
    (with-port (wptr view)
      (rlet ((font-info :FontInfo))
        (_GetFontInfo :ptr font-info)
        (let ((ascent (rref font-info :FontInfo.ascent))
              (descent (rref font-info :FontInfo.descent)))

          (let* ((max-v 2)
                 (max-h 2))
            
            (setf (node-height view) (+ (* 2 (+ max-v v-text-gap)) ascent descent)
                  (node-text-offset view) (make-point (+ max-h h-text-gap)
                                                        (+ max-v v-text-gap ascent))
                  (cut-node-text-offset view) (make-point 
                                                 (round (- (node-height view) 
                                                           (with-pstrs ((pascal-string "!"))
                                                             (_stringWidth :ptr pascal-string :word)))
                                                          2)
                                                 (+ max-v v-text-gap ascent))
                  (corner view) (round (node-height view) 1.5))))))))

;;____________________________________________________________________________________________
;; anscestor-conjunctive

(defun anscestor-conjunctive (node)
  (if node
    (let ((parent (node-parent node)))
      (if parent
        (let ((sibs (do* ((conjunctions (node-children parent) (cdr conjunctions))
                          (conjunction (car conjunctions) (car conjunctions)))
                         ((member node conjunction) conjunction))))
          (if (cdr sibs)
            t
            (anscestor-conjunctive parent)))
        nil))
    nil))

;;____________________________________________________________________________________________
;; invert-cut-node                                                     

(defmethod invert-cut-node ((view graph-view) node)
  (let* ((left (- (node-left node) (view-left view)))
         (top (- (node-top node) (view-top view)))
         (right (- (node-right node) (view-left view)))
         (bottom (- (node-bottom node) (view-top view)))
         (base (round (node-height view) 3.414213562)))
    (with-focused-view view
      (let ((poly (_OpenPoly :ptr)))
        (_MoveTo :long (make-point (+ left base) top))
        (_LineTo :long (make-point (- right base) top))
        (_LineTo :long (make-point right (+ top base)))
        (_LineTo :long (make-point right (- bottom base)))
        (_LineTo :long (make-point (- right base) bottom))
        (_LineTo :long (make-point (+ left base) bottom))
        (_LineTo :long (make-point left (- bottom base)))
        (_LineTo :long (make-point left (+ top base)))
        (_LineTo :long (make-point (+ left base) top))
        (_ClosePgon)
        (_InvertPoly :ptr poly)
        (_FramePoly :ptr poly)))))

;;____________________________________________________________________________________________
;; invert-node

(defmethod invert-node ((view graph-view) node)
  (rlet ((node-rect :rect
                    :left (- (node-left node) (view-left view))
                    :top (- (node-top node) (view-top view))
                    :right (- (node-right node) (view-left view))
                    :bottom (- (node-bottom node) (view-top view))))

    (with-focused-view view
      (case (node-kind node)
        (:cut
         (invert-cut-node node))
        ((:extensional :undefined)
         (_InverRect :ptr node-rect))
        ((:intensional :unexpanded :recursive :builtin :not)
         (_InverRoundRect :ptr node-rect :word (corner view) :word (corner view)))))))

;;____________________________________________________________________________________________
;;  find-node-clicked-on

(defun find-node-clicked-on (node h v &optional (node-clicked-on nil))
  (when node
    (if (and (node-on-screen? node)
             (not (node-hidden? node))
             (< (node-left node) h)
             (< h (node-right node))
             (< (node-top node) v)
             (< v (node-bottom node)))
      (setf node-clicked-on node))
    (dolist (children (node-children node))
      (dolist (child children)
        (setf node-clicked-on (find-node-clicked-on child h v node-clicked-on)))))
  node-clicked-on)

;;____________________________________________________________________________________________
;;  select-node

(defmethod select-node ((view graph-view) node)
  (when (and (not (node-selected? node))
             (not (node-hidden? node)))
    (setf (node-selected? node) t)
    (invert-node view node)
    (if (eql (node-kind node) :NOT)
      (select-children view node))))

;;____________________________________________________________________________________________
;;  select-children

(defmethod select-children ((view graph-view) node)
  (dolist (children (node-children node))
    (dolist (child children)
      (select-node view child))))

;;____________________________________________________________________________________________
;;  select-subtree

(defmethod select-subtree ((view graph-view) node)
  (cond ((null node) nil)
        ((node-p node)
         (unless (node-hidden? node)
           (unless (node-selected? node)
             (setf (node-selected? node) t)
             (invert-node view node))
           (select-subtree view (node-children node))))
        (t 
         (select-subtree view (car node))
         (select-subtree view (cdr node)))))

;;____________________________________________________________________________________________
;;  deselect-node

(defmethod deselect-node ((view graph-view) node)
  (when (node-selected? node)
    (let ((parent (node-parent node)))
      (setf (node-selected? node) nil)
      (invert-node view node)
      (if (eql (node-kind node) :NOT)
        (deselect-children view node))
      (when (and parent 
                 (eql (node-kind parent) :NOT)
                 (every #'(lambda (c) (not (node-selected? c))) (car (node-children parent))))
        (deselect-node view parent))
      )))

;;____________________________________________________________________________________________
;;  deselect-children

(defmethod deselect-children ((view graph-view) node)
  (dolist (children (node-children node))
    (dolist (child children)
      (deselect-node view child))))

;;____________________________________________________________________________________________
;;  deselect-subtree

(defmethod  deselect-subtree ((view graph-view) node)
  (cond ((null node) nil)
        ((node-p node)
         (deselect-node view node)
         ;(unless (node-hidden? node)
           (deselect-subtree view (node-children node)))
;)
        (t 
         (deselect-subtree view (car node))
         (deselect-subtree view (cdr node)))))

;;____________________________________________________________________________________________
;;  deselect-all

(defmethod deselect-all ((view graph-view))
  (deselect-subtree view (graph-root view)))



;;____________________________________________________________________________________________
;;  view-click-event-handler

(defmethod view-click-event-handler ((view graph-view) where)
  (let ((nodes-where-moved nil)
        (node-selected (find-node-clicked-on (graph-root view) 
                                             (+ (point-h where) (view-left view))
                                             (+ (point-v where) (view-top view)))))
    (setf (last-node-selected view) node-selected)
    (if node-selected
      (case (node-selection-constraint view)
        ((nil :no-drag)
         (cond ((double-click-p)
                (cond ((shift-key-p)
                       (cond ((option-key-p) 
                              (cond ((node-selected? node-selected)
                                     (dolist (node (conjunction-containing-node node-selected))
                                       (select-subtree view node))
                                     (setf nodes-where-moved (drag-selected-nodes view where)))
                                    (t
                                     (dolist (node (conjunction-containing-node node-selected))
                                       (deselect-subtree view node))
                                     (setf (last-node-selected view) nil))))
                             (t
                              (cond ((node-selected? node-selected)
                                     (dolist (node (conjunction-containing-node node-selected))
                                       (select-node view node))
                                     (setf nodes-where-moved (drag-selected-nodes view where)))
                                    (t
                                     (dolist (node (conjunction-containing-node node-selected))
                                       (deselect-node view node)))))))
                      ((option-key-p)
                       (dolist (node (conjunction-containing-node node-selected))
                         (select-subtree view node))
                       (setf nodes-where-moved (drag-selected-nodes view where)))
                      (t
                       (dolist (node (conjunction-containing-node node-selected))
                         (select-node view node))
                       (setf nodes-where-moved (drag-selected-nodes view where)))))
               (t
                (cond ((shift-key-p)
                       (cond ((option-key-p)
                              (cond ((node-selected? node-selected)
                                     (deselect-subtree view node-selected)
                                     (setf (last-node-selected view) nil))
                                    (t
                                     (select-subtree view node-selected)
                                     (setf nodes-where-moved (drag-selected-nodes view where)))))
                             (t
                              (cond ((node-selected? node-selected)
                                     (deselect-node view node-selected)
                                     (setf (last-node-selected view) nil))
                                    (t
                                     (select-node view node-selected)
                                     (setf nodes-where-moved (drag-selected-nodes view where)))))))
                      ((option-key-p)
                       (cond ((node-selected? node-selected)
                              (select-subtree view node-selected)
                              (setf nodes-where-moved (drag-selected-nodes view where)))
                             (t
                              (deselect-all view)
                              (select-subtree view node-selected)
                              (setf nodes-where-moved (drag-selected-nodes view where)))))
                      (t 
                       (cond ((node-selected? node-selected)
                              (setf nodes-where-moved (drag-selected-nodes view where)))
                             (t
                              (deselect-all view)
                              (select-node view node-selected)
                              (setf nodes-where-moved (drag-selected-nodes view where)))))))))

        (:single-not-leaf
         (cond ((node-selected? node-selected)
                (setf nodes-where-moved (drag-selected-nodes view where)))
               (t
                (deselect-all view)
                (when (node-children node-selected)
                  (select-node view node-selected)
                  (setf nodes-where-moved (drag-selected-nodes view where))))))

        (:single-disjunctive-not-leaf
         (cond ((node-selected? node-selected)
                (setf nodes-where-moved (drag-selected-nodes view where)))
               (t
                (deselect-all view)
                (when (and (node-children node-selected)
                           (not (anscestor-conjunctive node-selected)))
                  (select-node view node-selected)
                  (setf nodes-where-moved (drag-selected-nodes view where))))))

        (:no-selection nil)
        )
       
      (deselect-all view))

    (setf (last-conjunction-selected view) nil)
    (when node-selected
      (let ((conjunct (conjunction-containing-node node-selected)))
        (if (every #'node-selected? conjunct)
          (setf (last-conjunction-selected view) conjunct))))

    (let ((graph-window (graph-window view)))
      (when (ccl::inherit-from-p graph-window 'graph-window)
        (when  (edit-kind graph-window)
          (configure-controller graph-window))
        (when (and nodes-where-moved
                   (eql (edit-kind graph-window) :edit-rule)
                   (some #'(lambda (conjunction) (some #'node-selected? conjunction))
                         (node-children (graph-root view))))
      (order-children-based-on-screen-positions (graph-root view))
      (position-nodes view)
      (force-graph-redraw view))))

    (call-next-method)
    (if nodes-where-moved :nodes-where-moved nil)))

;;____________________________________________________________________________________________
;; draw-node                                                      

(defmethod draw-node ((view graph-view) node &key (clipped t))
  (cond ((node-hidden? node) (setf (node-on-screen? node) nil))
        (clipped
         (if (and
              (> (node-bottom node) (view-top view))
              (< (node-top node) (+ (view-top view) (point-v (view-size view))))
              (> (node-right node) (view-left view))
              (< (node-left node) (+ (view-left view) (point-h (view-size view)))))
           (setf (node-on-screen? node) (really-draw-node view node))
           (setf (node-on-screen? node) nil)))
        (t (really-draw-node view node))))

;;____________________________________________________________________________________________
;; really-draw-node                                                      

(defmethod really-draw-node ((view graph-view) node)
  (unless (node-hidden? node)
    (let ((node-left (- (node-left node) (view-left view)))
          (node-top (- (node-top node) (view-top view)))
          (node-right (- (node-right node) (view-left view)))
          (node-bottom (- (node-bottom node) (view-top view)))
          (node-text-h (- (node-text-h node) (view-left view)))
          (node-text-v (- (node-text-v node) (view-top view)))
          (corner (corner view)))
      (multiple-value-bind 
        (fill-pattern frame-color fill-color)
        (set-pen-given-state (node-state node))
      
      (rlet ((node-rect :rect
                        :left node-left
                        :top node-top
                        :right node-right
                        :bottom node-bottom))
        
        (setf (node-on-screen? node) t)
        (case (node-kind node)
          (:undefined
           (draw-undefined-node node-rect fill-pattern frame-color fill-color))
          (:cut
           (draw-cut-node view node fill-pattern frame-color fill-color))
          (:extensional
           (draw-extensional-node node-rect fill-pattern frame-color fill-color))
          (:builtin
           (draw-builtin-node node-rect corner fill-pattern frame-color fill-color))
          (:intensional
           (draw-intensional-node node-rect corner fill-pattern frame-color fill-color))
          (:unexpanded
           (draw-unexpanded-node view node node-rect fill-pattern frame-color fill-color))
          (:recursive
           (draw-recursive-node view node node-rect fill-pattern frame-color fill-color))
          (:not
           (draw-not-node node-rect corner fill-pattern frame-color fill-color))
          (otherwise
           ))

        (_PenPat :ptr *black-pattern*)
        (_PenSize :long #@(1 1))

        (when (and (node-deleted? node)
                   (display-deleted-nodes view))
          (_MoveTo :long (make-point node-left node-top))
          (_LineTo :long (make-point (- node-right 2) (- node-bottom 2)))
          (_MoveTo :long (make-point node-left (- node-bottom 2)))
          (_LineTo :long (make-point (- node-right 2) node-top)))
        
        (_MoveTo :long (make-point node-text-h node-text-v))
        (with-pstrs ((pascal-string (node-text node)))
          (_DrawString :ptr pascal-string))
        
        (when (node-external-text node)
          (if (node-children node)
            (_MoveTo :long (make-point node-text-h (+ node-bottom 9)))
            (_MoveTo :long (make-point (+ node-right 10) node-text-v)))
          (with-pstrs ((pascal-string (node-external-text node)))
            (_DrawString :ptr pascal-string)))
        
        (if (node-selected? node)
          (case (node-kind node)
            (:cut
             (invert-cut-node node))
            ((:extensional :undefined)
             (_InverRect :ptr node-rect))
            (otherwise
             (_InverRoundRect :ptr node-rect :word (corner view) :word (corner view))))))
      t))))

;;____________________________________________________________________________________________
;; set-pen-given-state                                                      

(defun set-pen-given-state (node-state)
  (case node-state
    
    (:unoperationalized
     (_PenSize :long *unoperationalized-pen-size*)
     (_PenPat :ptr *unoperationalized-pen-pattern*)
     (values *unoperationalized-fill-pattern*
             *unoperationalized-frame-color*
             *unoperationalized-fill-color*))
    
    (:normal
     (_PenSize :long *normal-pen-size*)
     (_PenPat :ptr *normal-pen-pattern*)
     (values *normal-fill-pattern*
             *normal-frame-color*
             *normal-fill-color*))
    
    (:extensional
     (_PenSize :long *extensional-pen-size*)
     (_PenPat :ptr *extensional-pen-pattern*)
     (values *extensional-fill-pattern*
             *extensional-frame-color*
             *extensional-fill-color*))
    
    (:builtin
     (_PenSize :long *builtin-pen-size*)
     (_PenPat :ptr *builtin-pen-pattern*)
     (values *builtin-fill-pattern*
             *builtin-frame-color*
             *builtin-fill-color*))
    
    (:intensional
     (_PenSize :long *intensional-pen-size*)
     (_PenPat :ptr *intensional-pen-pattern*)
     (values *intensional-fill-pattern*
             *intensional-frame-color*
             *intensional-fill-color*))
    
    (:cliche
     (_PenSize :long *cliche-pen-size*)
     (_PenPat :ptr  *cliche-pen-pattern*)
     (values *cliche-fill-pattern*
             *cliche-frame-color*
             *cliche-fill-color*))
    
    (:determinate
     (_PenSize :long *determinate-pen-size*)
     (_PenPat :ptr *determinate-pen-pattern*)
     (values *determinate-fill-pattern*
             *determinate-frame-color*
             *determinate-fill-color*))
    
    (:ebl
     (_PenSize :long *ebl-pen-size*)
     (_PenPat :ptr *ebl-pen-pattern*)
     (values *ebl-fill-pattern*
             *ebl-frame-color*
             *ebl-fill-color*))
    
    (:blank
     (_PenSize :long *normal-pen-size*)
     (_PenPat :ptr *white-pattern*)
     (values *white-pattern*
             *white-color*
             *white-color*))
    
    (otherwise                         ;; This is here to help catch errors
     (_PenSize :long *error-pen-size*)
     (_PenPat :ptr *error-pen-pattern*)
     (values *error-fill-pattern*
             *error-frame-color*
             *error-fill-color*))
    
    ))




;;____________________________________________________________________________________________
;; draw-undefined-node                                                     

(defun draw-undefined-node (node-rect fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (_FillRect :ptr node-rect :ptr fill-pattern))
  (with-fore-color frame-color
    (_FrameRect :ptr node-rect)))
          
;;____________________________________________________________________________________________
;; draw-cut-node                                                     

(defmethod draw-cut-node ((view graph-view) node fill-pattern frame-color fill-color)
  (let* 
    ((left (- (node-left node) (view-left view)))
     (top (- (node-top node) (view-top view)))
     (right (- (node-right node) (view-left view)))
     (bottom (- (node-bottom node) (view-top view)))
     (base (round (node-height view) 3.414213562)))

    (with-focused-view view
      (let ((poly (_OpenPoly :ptr)))
        (_MoveTo :long (make-point (+ left base) top))
        (_LineTo :long (make-point (- right base) top))
        (_LineTo :long (make-point right (+ top base)))
        (_LineTo :long (make-point right (- bottom base)))
        (_LineTo :long (make-point (- right base) bottom))
        (_LineTo :long (make-point (+ left base) bottom))
        (_LineTo :long (make-point left (- bottom base)))
        (_LineTo :long (make-point left (+ top base)))
        (_LineTo :long (make-point (+ left base) top))
        (_ClosePgon)
        (with-fore-color fill-color
          (_FillPoly :ptr poly :ptr fill-pattern))
        (with-fore-color frame-color
          (_FramePoly :ptr poly))
        (_KillPoly :ptr poly))
      
      (let ((poly (_OpenPoly :ptr)))
        (_MoveTo :long (make-point (+ left base 1) (+ top 2)))
        (_LineTo :long (make-point (- right base 1) (+ top 2)))
        (_LineTo :long (make-point (- right 2)(+ top base 1)))
        (_LineTo :long (make-point (- right 2) (- bottom base 1)))
        (_LineTo :long (make-point (- right base 1) (- bottom 2)))
        (_LineTo :long (make-point (+ left base 1) (- bottom 2)))
        (_LineTo :long (make-point (+ left 2) (- bottom base 1)))
        (_LineTo :long (make-point (+ left 2) (+ top base 1)))
        (_LineTo :long (make-point (+ left base 1) (+ top 2)))
        (_ClosePgon)
        (with-fore-color frame-color
          (_FramePoly :ptr poly))
        (_KillPoly :ptr poly)))))


;;____________________________________________________________________________________________
;; draw-extensional-node                                                     

(defun draw-extensional-node (node-rect fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (_FillRect :ptr node-rect :ptr fill-pattern))
  (with-fore-color frame-color
    (_FrameRect :ptr node-rect)))

          
;;____________________________________________________________________________________________
;; draw-recursive-node                                                     

(defun draw-recursive-node (view node node-rect fill-pattern frame-color fill-color)
  (let* ((corner (corner view))
         (top (view-top view))
         (left (view-left view))
         (node-bottom (- (node-bottom node) top))
         (node-right (- (node-right node) left))
         (node-center-h (- (node-center-h  node) left))
         (node-center-v (- (node-center-v node) top)))
    (with-fore-color fill-color
      (_FillRoundRect :ptr node-rect :word corner :word corner :ptr fill-pattern))
    (with-fore-color frame-color
      (_FrameRoundRect :ptr node-rect :word corner :word corner))
  (_PenSize :long #@(1 1))
  (with-pstrs ((pascal-string ""))
    (if (eql (graph-orientation view) :vertical)
      (_MoveTo :long (make-point (- node-center-h 
                                    (round (with-focused-view view 
                                             (_stringWidth :ptr pascal-string :word))) 2)
                                 (+ node-bottom 4)))
      (_MoveTo :long (make-point (+ node-right 3) (+ node-center-v 1))))
    (_DrawString :ptr pascal-string))))


;;____________________________________________________________________________________________
;; draw-unexpanded-node                                                     

(defun draw-unexpanded-node (view node node-rect fill-pattern frame-color fill-color)
  (let* ((corner (corner view))
         (top (view-top view))
         (left (view-left view))
         (node-bottom (- (node-bottom node) top))
         (node-right (- (node-right node) left))
         (node-center-h (- (node-center-h  node) left))
         (node-center-v (- (node-center-v node) top)))
    (with-fore-color fill-color
      (_FillRoundRect :ptr node-rect :word corner :word corner :ptr fill-pattern))
    (with-fore-color frame-color
      (_FrameRoundRect :ptr node-rect :word corner :word corner))
  (_PenSize :long #@(1 1))
  (with-pstrs ((pascal-string ""))
    (if (eql (graph-orientation view) :vertical)
      (_MoveTo :long (make-point (- node-center-h 
                                    (round (with-focused-view view 
                                             (_stringWidth :ptr pascal-string :word))) 2)
                                 (+ node-bottom 4)))
      (_MoveTo :long (make-point (+ node-right 3) (+ node-center-v 1))))
    (_DrawString :ptr pascal-string))))


;;____________________________________________________________________________________________
;; draw-intensional-node                                                     

(defun draw-intensional-node (node-rect corner fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (_FillRoundRect :ptr node-rect :word corner :word corner :ptr fill-pattern))
  (with-fore-color frame-color
    (_FrameRoundRect :ptr node-rect :word corner :word corner)))

;;____________________________________________________________________________________________
;; draw-builtin-node                                                     

(defun draw-builtin-node (node-rect corner fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (_FillRoundRect :ptr node-rect :word corner :word corner :ptr fill-pattern))
  (with-fore-color frame-color
    (_FrameRoundRect :ptr node-rect :word corner :word corner)))

;;____________________________________________________________________________________________
;; draw-not-node                                                     

(defun draw-not-node (node-rect corner fill-pattern frame-color fill-color)
  (with-fore-color fill-color
    (_FillRoundRect :ptr node-rect :word corner :word corner :ptr fill-pattern))
  (with-fore-color frame-color
    (_FrameRoundRect :ptr node-rect :word corner :word corner)))


;;____________________________________________________________________________________________
;; outline-cut-node                                                    

(defmethod outline-cut-node ((view graph-view) node)

  (let* 
    ((left (- (node-left node) (view-left view)))
     (top (- (node-top node) (view-top view)))
     (right (- (node-right node) (view-left view)))
     (bottom (- (node-bottom node) (view-top view)))
     (base (round (node-height view) 3.414213562)))

    (with-focused-view view
      (let ((poly (_OpenPoly :ptr)))
        (_MoveTo :long (make-point (+ left base) top))
        (_LineTo :long (make-point (- right base) top))
        (_LineTo :long (make-point right (+ top base)))
        (_LineTo :long (make-point right (- bottom base)))
        (_LineTo :long (make-point (- right base) bottom))
        (_LineTo :long (make-point (+ left base) bottom))
        (_LineTo :long (make-point left (- bottom base)))
        (_LineTo :long (make-point left (+ top base)))
        (_LineTo :long (make-point (+ left base) top))
        (_ClosePgon)
        (_FramePoly :ptr poly)
        (_KillPoly :ptr poly)))))

;;____________________________________________________________________________________________
;; draw-outline-of-selected-nodes                                                     

(defmethod draw-outline-of-selected-nodes ((view graph-view) &optional (node (graph-root view)))
  (with-focused-view view
    (if (node-selected? node)
      (rlet ((node-rect :rect
                        :left (- (node-left node) (view-left view))
                        :top (- (node-top node)  (view-top view))
                        :right (- (node-right node) (view-left view))
                        :bottom (- (node-bottom node) (view-top view))))
        (case (node-kind node)
          (:cut
           (outline-cut-node node))
          ((:extensional :undefined)
           (_FrameRect :ptr node-rect))
          ((:intensional :unexpanded :recursive :builtin :not)
           (_FrameRoundRect :ptr node-rect :word (corner view) :word (corner view)))))))
  (dolist (children (node-children node))
    (dolist (child children)
      (draw-outline-of-selected-nodes view child))))


;;____________________________________________________________________________________________
;; drag-selected-nodes

(defmethod drag-selected-nodes ((view graph-view) where)
  (if (eql (node-selection-constraint view) :no-drag)
    nil
    (with-focused-view view
      (let ((view-size-h (point-h (view-size view)))
            (view-size-v (point-v (view-size view)))
            offset)
        
        (rlet ((limit-rect :rect 
                           :left 0
                           :top 0
                           :right view-size-h
                           :bottom view-size-v)
               (slop-rect :rect 
                          :left -30
                          :top -30
                          :right (+ view-size-h 30)
                          :bottom (+ view-size-v 30)))
          
          (let ((moving-region (_NewRgn :ptr)))
            (_OpenRgn)
            (draw-outline-of-selected-nodes view)
            (_CloseRgn :ptr moving-region)
            
            (_PenPat :ptr *black-pattern*)
            (_PenSize :long #@(1 1))
            
            (setf offset (_DragGrayRgn :ptr moving-region
                                       :long where
                                       :ptr limit-rect
                                       :ptr slop-rect
                                       :word 0            ; 0 any, 1 horizontal only, 2 vertical only
                                       :ptr (%null-ptr)   ; no action procedure
                                       :long))
            (_DisposRgn :ptr moving-region)))
        
        (let ((offset-h (point-h offset))
              (offset-v (point-v offset)))
          (if (and (> offset-h -32768)
                   (or (> offset-h 2)
                       (< offset-h -2)
                       (> offset-v 2)
                       (< offset-v -2)))
            (progn
              (move-selected-nodes (graph-root view) offset-h offset-v)
              (order-siblings-of-selected-nodes (graph-root view))
              (redraw-graph view :fast t)
              (reset-graph-size view)
              (reset-scroll-bars (view-container view))
              t)
            nil))))))

;;____________________________________________________________________________________________
;; move-selected-nodes

(defun move-selected-nodes (node offset-h offset-v)
  (when (node-selected? node)
    (incf (node-left node) offset-h)
    (incf (node-top node) offset-v)
    (incf (node-right node) offset-h)
    (incf (node-bottom node) offset-v)
    (incf (node-center-h node) offset-h)
    (incf (node-center-v node) offset-v)
    (incf (node-text-h node) offset-h)
    (incf (node-text-v node) offset-v))
  (dolist (children (node-children node))
    (dolist (child children)
      (move-selected-nodes child offset-h offset-v))))


;;____________________________________________________________________________________________
;; order-siblings-of-selected-nodes

(defun order-siblings-of-selected-nodes (node)
  (if (node-selected? node)
    (order-siblings node))
  (dolist (children (node-children node))
    (dolist (child children)
      (order-siblings-of-selected-nodes child))))

;;____________________________________________________________________________________
;; order-siblings

(defun convert-to-polar (ax ay px py)
  (let* ((a-px (- ax px))
         (a-py (- ay py))
         (degrees-pre-radian 57.29577951)
         (a-p-radius (sqrt (+ (* a-px a-px) (* a-py a-py))))
         (a-p-asin (if (= a-p-radius 0)
                     0
                     (round (* degrees-pre-radian (asin (/ a-py a-p-radius))))))
         (a-p-angle (cond 
                     ((and (<= a-px 0) (<= a-py 0))    ;; in first quadrant
                      (- a-p-asin))
                     ((and (<= 0 a-px) (<= a-py 0))    ;; in second quadrant
                      (+ 180 a-p-asin))
                     ((and (<= 0 a-px) (<= 0 a-py))     ;; in third quadrant
                      (+ 180 a-p-asin))
                     ((and (<= a-px 0) (<= 0 a-py))     ;; in fourth quadrant
                      (- 360 a-p-asin)))))
    (values a-p-angle a-p-radius)))

(defun preceeds-in-clockwise-order (node1 node2 px py child-connector-top?)
  (let (node1-x node1-y node2-x node2-y)
    (if child-connector-top?
      (setf node1-x (node-center-h node1)
            node1-y (node-top node1)
            node2-x (node-center-h node2)
            node2-y (node-top node2))
      (setf node1-x (node-left node1)
            node1-y (node-center-v node1)
            node2-x (node-left node2)
            node2-y (node-center-v node2)))
    (let ((node1-p-angle (convert-to-polar node1-x node1-y px py))
          (node2-p-angle (convert-to-polar node2-x node2-y px py)))
      (if (< node1-p-angle node2-p-angle)
        t
        nil))))

(defun order-siblings (node)
  (let ((parent (node-parent node)))
    (if parent 
      (let (px py (child-connector-top? (some-child-left-of-center parent)))
        (case (node-kind parent)
          (:not (setf px (node-center-h parent)
                      py (node-center-v parent)))
          (otherwise (if (some-child-above-bottom parent)
                       (setf px (node-right parent)
                             py (node-center-v parent))
                       (setf px (node-center-h parent)
                             py (node-bottom parent)))))

        (do* ((conjunctions (node-children parent) (cdr conjunctions))
              (conjunction (car conjunctions) (car conjunctions)))
             ((null conjunctions))
          (when (member node conjunction)
            (cond
             ((null (cdr conjunction)) nil)                               ;; single node conjuction
             ((every #'(lambda (n) (node-selected? n)) conjunction) nil)  ;; entire conjunction selected
             (t (rplaca conjunctions
                        (sort conjunction
                              #'(lambda (node1 node2) 
                                  (preceeds-in-clockwise-order node1 node2 px py child-connector-top?))))))))))))

;;____________________________________________________________________________________
;;  reset-graph-size

(defmethod reset-graph-size ((view graph-view))
  (let ((node (if (node-hidden? (graph-root view))
                (car (car (node-children (graph-root view))))
                (graph-root view))))
    (setf (graph-left view) (node-left node)
          (graph-top view) (node-top node)
          (graph-right view) (node-right node)
          (graph-bottom view) (node-bottom node))
    (set-graph-size view (graph-root view))))

;;____________________________________________________________________________________
;;  set-graph-size

(defmethod set-graph-size ((view graph-view) node)
  (when node
    (unless (node-hidden? node)
      (setf (graph-left view) (min (graph-left view) (node-left node))
            (graph-top view) (min (graph-top view) (node-top node))
            (graph-right view) (max (graph-right view) (node-right node))
            (graph-bottom view) (max (graph-bottom view) (node-bottom node))))
    (dolist (children (node-children node))
      (dolist (child children)
        (set-graph-size view child)))))

;;____________________________________________________________________________________________
;; first-visible-node

(defun first-visible-node (node &optional (visible-node nil))
  (when node
    (if (node-hidden? node)
      (do* ((conjunctions (node-children node) (cdr conjunctions))
            (conjunction (car conjunctions) (car conjunctions)))
           ((or (null conjunctions) visible-node) visible-node)
        (do* ((nodes conjunction (cdr nodes))
              (n (car nodes) (car nodes)))
             ((or (null nodes) visible-node) visible-node)
          (setf visible-node (first-visible-node n visible-node))))
      node)))

;;____________________________________________________________________________________________
;; return-literal-string

(defun return-literal-string (pred args)
  (cond ((and *display-prolog* *downcase-preds*) (format nil "~(~S~)(~{~S~^, ~})" pred args))
        (*display-prolog* (format nil "~S(~{~S~^, ~})" pred args))
        (*downcase-preds* (format nil "(~(~S~)~{ ~S~})" pred args))
        (t (format nil "(~S~{ ~S~})" pred args))))

;;____________________________________________________________________________________________
;; return-pred-string

(defun return-pred-string (pred)
  (cond ((null pred) "")
        (*downcase-preds* (format nil "~(~S~)" pred))
        (t (format nil "~S" pred))))

;;____________________________________________________________________________________________
;; create-node

(defmethod create-node ((view graph-view)  &key (literal nil)
                                                (parent nil)
                                                (children nil)
                                                (external-text nil)
                                                (state :normal)
                                                (selected? nil)
                                                (on-screen? nil)
                                                (hidden? nil)
                                                (deleted? nil)
                                                (left 0)
                                                (top 0)
                                                (expand t)
                                                )
  (let ((new-node (make-node :external-text external-text
                             :parent parent
                             :children children
                             :selected? selected?
                             :on-screen? on-screen?
                             :hidden? hidden?
                             :deleted? deleted?
                             :state state
                             :top top
                             :left left)))
    
    (set-node-literal view new-node literal expand)
    new-node))


;;____________________________________________________________________________________________
;; inspect-last-node-selected-as-pred

(defmethod inspect-last-node-selected-as-pred ((view graph-view))
  (when (last-node-selected view)
    (case (node-kind (last-node-selected view))
      ((:intensional :recursive :unexpanded :builtin :extensional)
       (inspect (get-pstruct (car (node-literal (last-node-selected view))))))
      (:not
       (format t "~%~%Trying to inspect ~A." (node-literal (last-node-selected view)))
       (format t "~%Inspecting a NOT~%"))
      (:cut
       (format t "~%~%Inspecting a prolog CUT~%"))
      (:undefined
       (format t "~%~%Trying to inspect ~A." (node-literal (last-node-selected view)))
       (format t "~%This node is undefined, thus has no associated p-structure.~%")))))

;;____________________________________________________________________________________________
;; some-child-above-bottom

(defun some-child-above-bottom (parent)
  (let ((return-value nil))
    (dolist (children (node-children parent))
      (dolist (child children)
        (if (<= (node-center-v child) (node-bottom parent) )
          (setf return-value t))))
    return-value))

;;____________________________________________________________________________________________
;; some-child-left-of-center

(defun some-child-left-of-center (parent)
  (let ((return-value nil))
    (dolist (children (node-children parent))
      (dolist (child children)
        (if (<= (node-left child) (node-center-h parent))
          (setf return-value t))))
    return-value))

;;;____________________________________________________________________________________________
;;;  draw-child-connectors

(defmethod draw-child-connectors ((view graph-view) parent &key (clipped t))
    (unless (or (node-hidden? parent)
                (eql (node-kind parent) :unexpanded))
      
      (let ((child-connector-top? (some-child-left-of-center parent))
            (view-left (view-left view))
            (view-top (view-top view))
            px py cx cy ax ay last-ax last-ay state last-state)
        (case (node-kind parent)
          (:not (setf px (- (node-center-h parent) view-left)
                      py (- (node-center-v parent) view-top)))
          (otherwise (if (some-child-above-bottom parent)
                       (setf px (- (node-right parent) view-left)
                             py (- (node-center-v parent) view-top))
                       (setf px (- (node-center-h parent) view-left)
                             py (- (node-bottom parent) view-top)))))
        
        (dolist (children (node-children parent))
          (cond 
           
           ((null (cdr children))                      ;;  OR-ed Child
            (let ((child (car children)))
              (unless (node-hidden? child)
                (if child-connector-top?
                  (draw-line view px py 
                             (- (node-center-h child) view-left)
                             (- (node-top child) view-top)
                             :state (node-state child)
                             :clipped clipped)
                  (draw-line view px py 
                             (- (node-left child) view-left)
                             (- (node-center-v child) view-top)
                             :state (node-state child)
                             :clipped clipped)))))
           
           (t                                          ;;  AND-ed Child
            (let ((first t))
              (dolist (child children)
                (setf state (node-state child))
                (unless (node-hidden? child)
                  (if child-connector-top?
                    (setf cx (- (node-center-h child) view-left)
                          cy (- (node-top child) view-top))
                    (setf cx (- (node-left child) view-left)
                          cy (- (node-center-v child) view-top)))
                  (setf ax (round (+ (* (- cx px) *and-arc-ratio*) px))
                        ay (round (+ (* (- cy py) *and-arc-ratio*) py)))
                  (draw-line view px py cx cy :state state :clipped clipped)
                  (if first
                    (setf first nil)
                    (draw-line view last-ax last-ay 
                               ax ay 
                               :state state
                               :clipped clipped))
                  (setf last-ax ax
                        last-ay ay
                        last-state state))))))))))

;;;____________________________________________________________________________________________
;;;  move-first-point-on-screen

(defun  move-first-point-on-screen (x1 y1 x2 y2 left top right bottom)
  (let ((delta-x (- x2 x1))
        (delta-y (- y2 y1)))

    (cond ((= delta-x 0)
           (values x1 (if (< y1 y2) top bottom)))
          ((= delta-y 0)
           (values (if (< x1 x2) left right) y1))
          (t (let ((left-intercept nil)
                   (top-intercept nil)
                   (right-intercept nil)
                   (bottom-intercept nil))

               (if (< x1 x2)
                 (setf left-intercept (+ (round (* delta-y (- left x1)) delta-x) y1))
                 (setf right-intercept (+ (round (* delta-y (- right x1)) delta-x) y1))
                 )

               (if (< y1 y2)
                 (setf top-intercept (+ (round (* delta-x (- top y1)) delta-y) x1))
                 (setf bottom-intercept (+ (round (* delta-x (- bottom y1)) delta-y) x1))
                 )

               (cond
                ((and left-intercept (<= top left-intercept) (<= left-intercept bottom))
                 (values left left-intercept))
                ((and top-intercept (< left top-intercept) (< top-intercept right))
                 (values top-intercept top))
                ((and right-intercept (<= top right-intercept) (<= right-intercept bottom))
                 (values right right-intercept))
                ((and bottom-intercept (< left bottom-intercept) (< bottom-intercept right))
                 (values bottom-intercept bottom))))))))

;;;____________________________________________________________________________________________
;;;  draw-line

(defmethod draw-line ((view graph-view) x1 y1 x2 y2 &key (state :normal) (clipped t))
  (if clipped
    (let* ((left 0)
           (top 0)
           (right (+ left (point-h (view-size view))))
           (bottom (+ top (point-v (view-size view))))
           (x1-left (< x1 left))
           (x2-left (< x2 left))
           (x1-right (> x1 right))
           (x2-right (> x2 right))
           (y1-top (< y1 top))
           (y2-top (< y2 top))
           (y1-bottom (> y1 bottom))
           (y2-bottom (> y2 bottom))
           (P1-on-screen (not (or x1-left x1-right y1-top y1-bottom)))
           (P2-on-screen (not (or x2-left x2-right y2-top y2-bottom))))
      
      (unless (or (and x1-left x2-left)
                  (and x1-right x2-right)
                  (and y1-top y2-top)
                  (and y1-bottom y2-bottom))
        (unless P1-on-screen
          (multiple-value-setq (x1 y1)
            (move-first-point-on-screen x1 y1 x2 y2 left top right bottom)))
        (when (and x1 y1)
          (unless (or P2-on-screen)
            (multiple-value-setq (x2 y2)
              (move-first-point-on-screen x2 y2 x1 y1 left top right bottom)))
          (when (and x2 y2)
            (really-draw-line view x1 y1 x2 y2 state)))))
    (really-draw-line view x1 y1 x2 y2 state)))
                 

;;;____________________________________________________________________________________________
;;;  really-draw-line

(defmethod really-draw-line ((view graph-view) x1 y1 x2 y2 &optional (state :normal))
  (multiple-value-bind (fill-pattern frame-color fill-color)
                       (set-pen-given-state state)
    (declare (ignore fill-pattern fill-color))
    (with-fore-color frame-color
      (_MoveTo :long (make-point x1 y1))
      (_LineTo :long (make-point x2 y2))))
  t)

;;;____________________________________________________________________________________
;;;  conjunction-containing-node

(defun conjunction-containing-node (node)
  (let ((parent (node-parent node))
        (conjunction-containing-node (list node)))
    (when parent
      (dolist (conjunction (node-children parent))
        (if (member node conjunction)
          (setf conjunction-containing-node conjunction))))
    conjunction-containing-node))

;;;____________________________________________________________________________________
;;;  literals-correspond
;;;
;;;  Literals correspond if they had the same head and the same number of arguments.
;;;  This is a hack.  It should see if they unify?

(defun literals-correspond (literal1 literal2)
  (if (= (length literal1) (length literal2))
    (cond
     ((and (eql (car literal1) 'not)
           (eql (car literal2) 'not)) (clauses-correspond (cdr literal1) (cdr literal2)))
     ((eql (car literal1) (car literal2)) t)
     (t nil))
    nil))

;;;____________________________________________________________________________________
;;;  literal-corresponds-to-some-literal

(defun literal-corresponds-to-some-literal (literal clause)
  (let ((corresponds nil))
    (dolist (clause-literal clause)
      (if (literals-correspond literal clause-literal)
        (setf corresponds t)))
    corresponds))

;;;____________________________________________________________________________________
;;;  clauses-correspond

(defun clauses-correspond (clause1 clause2)
  (let ((corresponds nil))
    (when (= (length clause1) (length clause2))
      (setf corresponds t)
      (dolist (literal clause1)
        (unless (literal-corresponds-to-some-literal literal clause2)
          (setf corresponds nil))))
    corresponds))

;;;_______________________________________________________________________________
;;; node-is-an-ancestor-of-itself
;;;
;;; used to detect recursion

(defun node-is-an-ancestor-of-itself (node &key (ancestor (node-parent node)))
  (cond ((null ancestor) nil)
        ((literals-correspond (node-literal node) (node-literal ancestor)) t)
        (t (node-is-an-ancestor-of-itself node :ancestor (node-parent ancestor)))))

;;;_______________________________________________________________________________
;;; create-all-unique-vars
;;;
;;;  Given:   A integer, NUMBER.
;;;  Returns: A list of all unique pc-vars of length NUMBER 
;;;           (no two elements of this list are the same)

(defun create-all-unique-vars (NUMBER &key (use-gensyms *use-gensyms*))
  (let ((all-unique-var-list nil))
    (dotimes (i NUMBER)
      (push (make-pcvar :id (if use-gensyms
                              (gensym "v")
                              i))               ;;;; XXXX This may not be cool!
            all-unique-var-list))
    (nreverse all-unique-var-list)))

;;;_______________________________________________________________________________
;;; all-antecedents   [This is a potential memory hog.]

(defun all-antecedents (graph-view prolog-literal)
  (let ((rule (get (first prolog-literal) 'rule)))
    (if (rule-p rule)
      (mapcar #'(lambda (clause)
                  (let* ((parameters (clause-parameters clause))
                         (body (clause-body clause))
                         (new-vars (clause-new-vars clause))
                         (users-new-vars (mapcar #'(lambda (var)
                                                       (find-user-name-for-variable body var))
                                                   new-vars))
                         (bindings (unify-list new-vars users-new-vars)))
                    (if new-vars
                      (setf (substitution graph-view) (unify-list new-vars 
                                                                  users-new-vars
                                                                  (substitution graph-view))))
                    (substitute-vars body 
                                     (unify-list parameters (rest prolog-literal) bindings))

                    ))
              (rule-clauses rule))
      nil)))

(defun find-user-name-for-variable (clause variable)
  (let* ((literal (first-literal-containing-variable clause variable))
         (p-struct (get-pstruct (car literal))))
    (if (and p-struct (p-vars p-struct))
      (elt (p-vars p-struct) (position variable (cdr literal)))
      variable)))

(defun first-literal-containing-variable (clause variable)
  (let ((literal-containing-variable nil))
    (do* ((literals clause (cdr literals))
          (literal (car literals) (car literals)))
         ((or (null literals) literal-containing-variable) nil)
      (setf literal-containing-variable
            (if (eql (car literal) 'not)
              (first-literal-containing-variable (cdr literal) variable)
              (if (member variable (cdr literal))
                literal))))
    literal-containing-variable))

;;;_______________________________________________________________________________
;;;  connect-clauses

(defmethod connect-clauses ((view graph-view) parent-node clauses source expand)
  (mapcar #'(lambda (clause) 
              (connect-clause view parent-node clause source expand))
          clauses))

;;;_______________________________________________________________________________
;;;  connect-clause

(defmethod connect-clause ((view graph-view) parent-node clause source expand)
  (if (literal-p  clause)
    (let ((conjunction nil))
      (do ((literal clause (literal-next literal)))
          ((null literal) conjunction)
        (setf conjunction (nconc conjunction
                                 (list (connect-literal view parent-node literal source expand))))))

    (mapcar #'(lambda (literal) 
                (connect-literal view parent-node literal source expand))
            clause)))

;;;_______________________________________________________________________________
;;;  connect-literal

(defmethod connect-literal ((view graph-view) parent-node literal source expand)
  (when literal
    (let* ((prolog-literal (if (literal-p literal)
                             (convert-literal-to-prolog-regardless literal)
                             literal))
           (deleted? (if (literal-p literal)
                       (literal-deleted? literal)
                       nil))
           (state (if (literal-p literal)
                    (derivation-type (literal-derivation literal))
                    source))
           (node (get-node (graph-window view)
                           view
                           :literal prolog-literal
                           :parent parent-node
                           :expand expand
                           :deleted? deleted?
                           :state state)))
 
      (case (node-kind node)
        (:intensional
         (unless deleted?
           (setf (node-children node) 
                 (connect-clauses view node (all-antecedents view prolog-literal) state expand))))
        (:not
         (if (literal-p literal)
           (cond (deleted?
                  (setf (node-children node) 
                        (list (connect-clause view node (literal-negated-literals literal) source nil)))
                  (mark-node-and-all-descedents-as-deleted node))
                 (t 
                  (setf (node-children node) 
                        (list (connect-clause view node (literal-negated-literals literal) source expand)))))
           (cond (deleted?
                  (setf (node-children node) 
                        (list (connect-clause view node (cdr prolog-literal) source nil)))
                  (mark-node-and-all-descedents-as-deleted node))
                 (t 
                  (setf (node-children node) 
                        (list (connect-clause view node (cdr prolog-literal) source expand)))))))
        (otherwise nil))
      node)))

;;;____________________________________________________________________________________
;;; mark-node-and-all-descedents-as-deleted

(defun mark-node-and-all-descedents-as-deleted (node)
  (when node
    (setf (node-deleted? node) t)
    (dolist (children (node-children node))
      (dolist (child children)
        (mark-node-and-all-descedents-as-deleted child)))))

;;;_______________________________________________________________________________
;;;  set-node-literal

(defmethod  set-node-literal ((view graph-view) node literal &optional (expand t))
  (when node
    (setf (node-literal node) literal)

    (cond ((null literal)
           (setf (node-text node) ""
                 (node-kind node) :undefined))
          
          ((cut? literal)
           (setf (node-text node) "!"
                 (node-kind node) :cut))
          
          ((atom literal)
           (setf (node-text node) (format nil "~A" literal)
                 (node-kind node) :undefined))
          
          ((is? literal)
           (setf (node-text node) (return-literal-string (car literal) (cdr literal))
                 (node-kind node) :builtin))
          
          ((equality? literal)
           (setf (node-text node) (return-literal-string (car literal) (cdr literal))
                 (node-kind node) :builtin))
          
          ((builtin? literal)
           (setf (node-text node) (return-literal-string (car literal) (cdr literal))
                 (node-kind node) :builtin))
          
          ((extensional? literal) 
           (setf (node-text node) (return-literal-string (car literal) (cdr literal))
                 (node-kind node) :extensional))
          
          ((negation? literal)
           (setf (node-text node) (if *downcase-preds* "not" "NOT")
                 (node-kind node) :not))

          ((intensional? literal)
           (setf (node-text node) (return-literal-string (car literal) (cdr literal))
                 (node-kind node) (cond ((and expand (node-is-an-ancestor-of-itself node)) :recursive)
                                        (expand :intensional)
                                        (t :unexpanded))))
          
          ((undefined? literal)
           (setf (node-text node) (return-literal-string (car literal) (cdr literal))
                 (node-kind node) :undefined))
          )
    (size-node view node)
    )
   node)






;;;_______________________________________________________________________________
;;;  size-node

(defmethod size-node ((view graph-view) node)
  (let ((kind (node-kind node))
        (left (node-left node))
        (top (node-top node)))
    (setf (node-right node) (+ left 
                               (case kind
                                 (:cut (node-height view))
                                 (otherwise 
                                  (+ (with-focused-view view
                                       (with-pstrs ((pascal-string (node-text node)))
                                         (_stringWidth :ptr pascal-string :word))) 
                                     (* 2 (point-h (node-text-offset view)))))))
          (node-bottom node) (+ top (node-height view))
          (node-center-h node) (floor (+ left (node-right node)) 2)
          (node-center-v node) (floor (+ top (node-bottom node)) 2)
          (node-text-h node) (+ left (point-h (case kind
                                                (:cut (cut-node-text-offset view))
                                                (otherwise (node-text-offset view)))))
          (node-text-v node) (+ top (point-v (case kind
                                               (:cut (cut-node-text-offset view))
                                               (otherwise (node-text-offset view))))))))

;;;_______________________________________________________________________________
;;;  place-node

(defun place-node (node new-h new-v)
  (place-node-h-components node new-h)
  (place-node-v-components node new-v))

(defun place-node-h-components (node new-h)
  (let ((offset-h (- new-h (node-left node))))
    (incf (node-left node) offset-h)
    (incf (node-right node) offset-h)
    (incf (node-center-h node) offset-h)
    (incf (node-text-h node) offset-h)))
  
(defun place-node-v-components (node new-v)
  (let ((offset-v (- new-v (node-top node))))
    (incf (node-top node) offset-v)
    (incf (node-bottom node) offset-v)
    (incf (node-center-v node) offset-v)
    (incf (node-text-v node) offset-v)))

;;;_______________________________________________________________________________
;;; tree-depth

(defun tree-depth (node)
  (if node
    (case (node-kind node)
      ((:intensional :not)
       (let ((max-depth 0))
         (dolist (children (node-children node) max-depth)
           (dolist (child children max-depth)
             (setf max-depth (max max-depth (+ 1 (tree-depth child))))))))
      (otherwise 1))))


;;;_______________________________________________________________________________
;;; position-nodes

(defmethod position-nodes ((view graph-view))

  (let ((next-node-x-position *graph-start-h*)
        (next-node-y-position *graph-start-v*)
        (x-table (make-array (tree-depth (graph-root view)) :initial-element *graph-start-h*))
        h-node-gap
        v-node-gap
        conjunction-break-gap)

    (labels

       ;;___________________________
       ;; Position Horizontal Graphs

      ((position-horizontal-graph-v (node)
         (when (or (not (node-hidden? node))
                   (null (node-parent node)))
           (dolist (children (node-children node))
             (dolist (child children)
               (position-horizontal-graph-v child))
             (incf next-node-y-position conjunction-break-gap))
           (cond
            ((and (node-children node)
                  (not (eq (node-kind node) :unexpanded)))
             (place-node-v-components 
              node 
              (+ (floor
                  (+ (node-top (first (first (node-children node))))
                     (node-bottom (first (last (first (last (node-children node)))))))
                  2)
                 (- (node-top node) (node-center-v node)))))
             (t
              (place-node-v-components node next-node-y-position)
              (setf next-node-y-position (+ (node-bottom node) v-node-gap))))))
         
      (position-horizontal-graph-h (node h-position)
         (place-node-h-components node h-position)
         (let* ((top-child (first (first (node-children node))))
                (bottom-child (first (last (first (last (node-children node))))))
                (childrens-h-position (+ (node-right node)
                                         (if (eq top-child bottom-child) 
                                           20
                                           (max 20
                                                (floor (- (node-center-v bottom-child)
                                                          (node-center-v top-child))
                                                       4))))))
           (dolist (children (node-children node))
             (dolist (child children)
               (position-horizontal-graph-h child childrens-h-position)))))

       (position-horizontal-graph (node)
         (position-horizontal-graph-v node)
         (position-horizontal-graph-h node *graph-start-v*))

       ;;___________________________
       ;; Position Vertical Graphs

       (offset-all-descendants-h (node offset-h)
          (incf (node-left node) offset-h)
          (incf (node-right node) offset-h)
          (incf (node-center-h node) offset-h)
          (incf (node-text-h node) offset-h)
          (setf (aref x-table (/ (- (node-top node) *graph-start-v*) (+ (node-height view) v-node-gap)))
                (+ (node-right node) h-node-gap))
          (unless (or (null (node-children node))
                      (eql (node-kind node) :unexpanded))
            (dolist (children (node-children node))
              (dolist (child children)
                (offset-all-descendants-h child offset-h))
              (incf (aref x-table (+ (/ (- (node-top node) *graph-start-v*) (+ (node-height view) v-node-gap) 1)))
                   conjunction-break-gap))))

       (position-vertical-graph (node y-position)
          (place-node-v-components node y-position)
          (let ((childrens-y-position (+ (node-bottom node) v-node-gap)))
            (when (and (not (eql (node-kind node) :unexpanded))
                       (node-children node))
              (let ((y-index (/ (- childrens-y-position *graph-start-v*) (+ (node-height view) v-node-gap))))
                (dolist (children (node-children node))
                  (dolist (child children)
                    (position-vertical-graph child childrens-y-position))
                  (incf (aref x-table y-index) conjunction-break-gap)))))
          (let ((y-index (/ (- (node-top node) *graph-start-v*) (+ (node-height view) v-node-gap))))
            (cond 
             ((or (null (node-children node))
                  (eql (node-kind node) :unexpanded))
              (place-node-h-components node (aref x-table y-index))
              (setf (aref x-table y-index) (+ (node-right node) h-node-gap)))
             (t
              (let ((x-pos (+ (round 
                               (+ (node-left (first (first (node-children node))))
                                  (node-right (first (last (first (last (node-children node))))))) 
                               2)
                              (-  (node-left node) (node-center-h node)))))
                (place-node-h-components node x-pos)
                (when (< x-pos (aref x-table y-index))
                  (offset-all-descendants-h node (- (aref x-table y-index) x-pos)))
                (setf (aref x-table y-index) (+ (node-right node) h-node-gap)))))))

       ;;________________________
       ;; Position Diagonal Graph

       (position-diagonal-graph-v (node)
        (when (or (not (node-hidden? node))
                  (null (node-parent node)))
          (place-node-v-components node next-node-y-position)
          (unless (eq (node-kind node) :not)
            (setf next-node-y-position (+ (node-bottom node) v-node-gap)))
          (unless (eq (node-kind node) :unexpanded)
            (dolist (children (node-children node))
              (dolist (child children)
                (position-diagonal-graph-v child))
              (incf next-node-y-position conjunction-break-gap)))))
       
       (position-diagonal-graph-h (node)
        (when (or (not (node-hidden? node))
                  (null (node-parent node)))
          (let* ((top (node-bottom node))
                 (bottom-node (first (last (first (last (node-children node))))))
                 (child-h-position (if bottom-node
                                     (+ (node-center-h node)
                                        (max 30
                                             (round (- (node-center-v bottom-node) top) 4))))))
            (dolist (children (node-children node))
              (dolist (child children)
                (place-node-h-components child child-h-position)
                (position-diagonal-graph-h child))))))
       
       (position-diagonal-graph (node)
         (position-diagonal-graph-v node)
         (position-diagonal-graph-h node)))


      (ccase (graph-orientation view)
             
             (:horizontal
              (setf h-node-gap (max (* 2 (node-height view)) 30)
                    v-node-gap (max (round (node-height view) 5) 1)
                    conjunction-break-gap v-node-gap)
              (position-horizontal-graph (graph-root view))
              (reset-graph-size view))
             
             (:vertical
              (setf h-node-gap (max (round (node-height view) 5) 2)
                    v-node-gap (max (* 2 (node-height view)) 20)
                    conjunction-break-gap h-node-gap)
              (position-vertical-graph (graph-root view) next-node-y-position)
              (reset-graph-size view))
             
             (:diagonal
              (setf v-node-gap (max (round (node-height view) 5) 1)
                    h-node-gap (max (* 2 (node-height view)) 20)
                    conjunction-break-gap v-node-gap)
              (place-node (graph-root view) next-node-x-position next-node-y-position)
              (position-diagonal-graph (graph-root view))
              (reset-graph-size view))))))


;;;________________________________________________________________________________
;;; hi-lite-subtree

(defun hi-lite-subtree (node state)
  (when node
    (setf (node-state node) state)
    (dolist (children (node-children node))
      (dolist (child children)
        (hi-lite-subtree child state)))))

;;;_______________________________________________________________________________
;;;  show-antecedents-of-selected-nodes

(defmethod show-antecedents-of-selected-nodes ((view graph-view))
  (mark-hidden-nodes-to-show-and-select (graph-root view))
  (show-and-select-marked-nodes (graph-root view))
  (position-nodes view)
  (if (grow-window-if-needed (graph-window view))
    (auto-position-window (graph-window view))
    (reset-scroll-bars (view-container view)))
  (force-graph-redraw view))

;;;_______________________________________________________________________________
;;;  mark-hidden-nodes-to-show-and-select

(defun mark-hidden-nodes-to-show-and-select (node)
  (when (and (eql (node-kind node) :unexpanded)
             (node-selected? node))
    (setf (node-kind node) :intensional))
  (when (and (node-hidden? node)
             (node-parent node)
             (node-selected? (node-parent node)))
    (setf (node-hidden? node) :show-and-select)
    (if (eql (node-kind node) :not)
      (setf (node-selected? node) t)))    ;; <- this will force the not's children to be shown!
    (dolist (children (node-children node))
      (dolist (child children)
        (mark-hidden-nodes-to-show-and-select child))))

;;;_______________________________________________________________________________
;;;  show-and-select-marked-nodes

(defun show-and-select-marked-nodes (node)
  (when (eql (node-hidden? node) :show-and-select)
    (setf (node-selected? node) t
          (node-hidden? node) nil))
  (dolist (children (node-children node))
    (dolist (child children)
      (show-and-select-marked-nodes child))))


;;;_______________________________________________________________________________
;;;  display-all-uses

(defmethod display-all-uses ((view graph-view) &optional (node (graph-root view)))
  (if (eql (node-kind node) :unexpanded)
    (setf (node-kind node) :intensional))
  (when (and (node-hidden? node) (not (eql (node-kind node) :hidden)))
    (setf (node-hidden? node) nil)
    (when (node-parent node)
      (setf (node-selected? node) (node-selected? (node-parent node)))))
  (dolist (children (node-children node))
    (dolist (child children)
      (display-all-uses view child))))


;;;_______________________________________________________________________________
;;;  hide-all-nodes

(defmethod hide-all-nodes ((view graph-view) &optional (node (graph-root view)))
  (setf (node-hidden? node) t)
  (dolist (children (node-children node))
    (dolist (child children)
      (hide-all-nodes view child))))

;;;_______________________________________________________________________________
;;;  hide-antecedents-of-selected-nodes

(defmethod hide-antecedents-of-selected-nodes ((view graph-view))
  (dolist (children (node-children (graph-root view)))
    (dolist (child children)
      (really-hide-antecedents-of-selected-nodes child)))
  (position-nodes view)
  (if (grow-window-if-needed (graph-window view))
    (auto-position-window (graph-window view))
    (reset-scroll-bars (view-container view)))
  (force-graph-redraw view))

;;;_______________________________________________________________________________
;;;  really-hide-antecedents-of-selected-nodes

(defun really-hide-antecedents-of-selected-nodes (node)
  (if (node-selected? node)
    (hide-all-descendents node))
  (dolist (children (node-children node))
    (dolist (child children)
      (really-hide-antecedents-of-selected-nodes child))))

;;;_______________________________________________________________________________
;;;  hide-all-descendents

(defun hide-all-descendents (node)
  (case (node-kind node)
    ((:extensional :builtin :undefined :cut :unexpanded :recursive) nil)
    ((:intensional)
     (setf (node-kind node) :unexpanded)
     (dolist (children (node-children node))
       (dolist (child children)
         (setf (node-hidden? child) t
               (node-selected? child) nil)
         (hide-all-descendents child)
         (when (eql (node-kind child) :not)
           (dolist (g-children (node-children child))
             (dolist (g-child g-children)
               (setf (node-hidden? g-child) t
                     (node-selected? g-child) nil)))))))
    ((:not)
     (dolist (children (node-children node))
       (dolist (child children)
         (hide-all-descendents child))))))

;;;_______________________________________________________________________________
;;;  display-only-first-use

(defmethod display-only-first-use ((view graph-view) )
  (let ((expanded-literals nil))
    (labels
       ((only-once (node)
            (if node
              (case (node-kind node)
                ((:extensional :builtin :undefined :cut) nil)
                ((:not)
                 (dolist (children (node-children node))
                   (dolist (child children)
                     (only-once child))))
                ((:intensional)
                 (cond
                  ((literal-corresponds-to-some-literal (node-literal node) expanded-literals)
                   (hide-all-descendents node))
                  (t
                   (setf expanded-literals (push (node-literal node) expanded-literals))
                   (dolist (children (node-children node))
                     (dolist (child children)
                       (only-once child))))))))))

      (only-once (graph-root view)))))

;;;_______________________________________________________________________________
;;;  generate-graph

(defmethod generate-graph ((view graph-view) prolog-literal)
  (when prolog-literal
    (setf (graph-root view) (connect-literal view nil prolog-literal :normal t))
    (if (eql (graph-expand view) :first-use)
      (display-only-first-use view))
    (position-nodes view)))

;;____________________________________________________________________________________
;;  regraph-node

(defmethod regraph-node ((view graph-view))
  (when (last-node-selected view)
    (let ((literal (node-literal (last-node-selected view))))
      (create-graph-window literal))))

;;____________________________________________________________________________________
;;  create-graph-window

(defun create-graph-window (prolog-literal)
  (let* ((window (make-instance 'graph-window :window-show nil))
         (view (view-named :graph-view (view-named :graph-scroller window))))
    (generate-graph view prolog-literal)
    (resize-window window)
    (position-graph view :centered t)
    (auto-position-window window :centered t)
    (set-window-title window (case (node-kind (graph-root view))
                               (:cut "CUT")
                               (:undefined "Undefined")
                               (otherwise (node-text (graph-root view)))))
    (window-select window)))

;;____________________________________________________________________________________
;;  graph-rule

(defun graph-rule (&optional (rule nil))
  (unless rule
    (let ((r (catch-cancel (select-item-from-list 
                            (sort (mapcar #'car *intensional-preds*)
                                  #'(lambda(x y) (string< (symbol-name x)
                                                          (symbol-name y))))
                            :window-title "Select A Rule To Be Graphed..."))))
      (unless (eq r :cancel)
        (setf rule (first r)))))
  (when rule
    (let* ((rule-pstruct (get-pstruct rule))
           (literal (cons rule 
                          (or (p-vars rule-pstruct)
                              (create-all-unique-vars (p-arity rule-pstruct) :use-gensyms nil)))))
      (create-graph-window literal))))

;;____________________________________________________________________________________
;; get-graph-setup

(defun get-graph-setup (graph-title
                        font
                        orientation
                        expand)

  (let* ((orientation-cluster 0)
         (expand-cluster 1)
         (setup-dialog 
          (make-instance 'dialog
                         :window-show nil
                         :window-type :double-edge-box
                         :view-position :centered
                         :view-size #@(360 270)
                         :close-box-p nil
                         :view-font '("Chicago" 12 :srcor :plain)
                         :view-subviews
                         (list 
                          (make-dialog-item 'static-text-dialog-item
                                            #@(15 7) #@(330 18)
                                            (format nil "~A Setup..." graph-title) nil)
                          
                          
                          (make-dialog-item 'static-text-dialog-item 
                                            #@(15 30) #@(50 20) "Font:" nil)
                          (make-dialog-item  'sequence-dialog-item
                                             #@(15 50) #@(160 80) "Untitled" nil
                                             :cell-size #@(145 16)
                                             :table-hscrollp NIL
                                             :table-vscrollp T
                                             :table-sequence *font-list*
                                             :selection-type :single
                                             :view-nick-name :font-selector)
                          (make-dialog-item 'static-text-dialog-item
                                            #@(15 142) #@(100 20) "Font Size:" nil)
                          (make-dialog-item 'editable-text-dialog-item 
                                            #@(100 142) #@(32 16) 
                                            (format nil "~A" (find-if #'numberp font))
                                            nil
                                            :allow-returns nil
                                            :view-nick-name :font-sizer)
                          (make-dialog-item 'static-text-dialog-item
                                            #@(15 172) #@(100 20) "Font Sytle:" nil)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(17 192) #@(150 16) "Plain"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :bold))
                                                (check-box-uncheck (find-named-sibling item :Italic))
                                                (check-box-uncheck (find-named-sibling item :Underline))
                                                (check-box-uncheck (find-named-sibling item :Outline))
                                                (check-box-uncheck (find-named-sibling item :Shadow))
                                                (check-box-uncheck (find-named-sibling item :Condense))
                                                (check-box-uncheck (find-named-sibling item :Extend)))
                                            :view-font '("Geneva" 12 :srcor :plain)
                                            :check-box-checked-p (member :plain font)
                                            :view-nick-name :plain)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(17 208) #@(150 16) "Bold"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain)))
                                            :view-font '("Geneva" 12 :srcor :bold)
                                            :check-box-checked-p (and (member :bold font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :bold)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(17 224) #@(150 16) "Italic"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain)))
                                            :view-font '("Geneva" 12 :srcor :italic)
                                            :check-box-checked-p (and (member :italic font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :italic)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(17 240) #@(150 16) "Underline"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain)))
                                            :view-font '("Geneva" 12 :srcor :underline)
                                            :check-box-checked-p (and (member :underline font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :underline)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(100 192) #@(150 16) "Outline"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain)))
                                            :view-font '("Geneva" 12 :srcor :outline)
                                            :check-box-checked-p (and (member :outline font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :outline)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(100 208) #@(150 16) "Shadow"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain)))
                                            :view-font '("Geneva" 12 :srcor :shadow)
                                            :check-box-checked-p (and (member :shadow font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :shadow)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(100 224) #@(150 16) "Condense"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain)))
                                            :view-font '("Geneva" 12 :srcor :condense)
                                            :check-box-checked-p (and (member :condense font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :condense)
                          (make-dialog-item 'check-box-dialog-item
                                            #@(100 240) #@(150 16) "Extend"
                                            #'(lambda (item)
                                                (check-box-uncheck (find-named-sibling item :plain))) 
                                            :view-font '("Geneva" 12 :srcor :extend)
                                            :check-box-checked-p (and (member :extend font) 
                                                                      (not (member :plain font)))
                                            :view-nick-name :extend)

                          (make-dialog-item 'static-text-dialog-item
                                            #@(200 30) #@(80 20) "Orientation:"
                                            nil)
                          (make-dialog-item 'radio-button-dialog-item
                                            #@(210 50) #@(150 16) "Horizontal"
                                            nil
                                            :radio-button-cluster orientation-cluster
                                            :radio-button-pushed-p
                                            (eql orientation :horizontal)
                                            :view-nick-name :horizontal)
                          (make-dialog-item 'radio-button-dialog-item
                                            #@(210 66) #@(150 16) "Diagonal"
                                            nil
                                            :radio-button-cluster orientation-cluster
                                            :radio-button-pushed-p 
                                            (eql orientation :diagonal)
                                            :view-nick-name :diagonal)
                          (make-dialog-item 'radio-button-dialog-item
                                            #@(210 82) #@(150 16) "Vertical"
                                            nil
                                            :radio-button-cluster orientation-cluster
                                            :radio-button-pushed-p
                                            (eql orientation :vertical)
                                            :view-nick-name :vertical)
                          
                          
                          (make-dialog-item 'static-text-dialog-item
                                            #@(200 107) #@(150 20) "Expand Node:"
                                            nil)
                          (make-dialog-item 'radio-button-dialog-item
                                            #@(210 127) #@(150 16) "Always"
                                            nil
                                            :radio-button-cluster expand-cluster
                                            :radio-button-pushed-p
                                            (eql expand :always)
                                            :view-nick-name :always)
                          (make-dialog-item 'radio-button-dialog-item
                                            #@(210 143) #@(150 16) "First Use"
                                            nil
                                            :radio-button-cluster expand-cluster
                                            :radio-button-pushed-p
                                            (eql expand :first-use)
                                            :view-nick-name :first-use)
                          
                          
                          
                          (make-dialog-item 'button-dialog-item
                                            #@(210 240) #@(60 20) " OK "
                                            #'(lambda (item)
                                                (let ((new-font nil)
                                                      (font-sizer (find-named-sibling item :font-sizer))
                                                      (font-selector (find-named-sibling item :font-selector))
                                                      (dialog (view-container item)))
                                                  (if (check-box-checked-p (find-named-sibling item :Bold))
                                                    (setf new-font (push :bold new-font)))
                                                  (if (check-box-checked-p (find-named-sibling item :Italic))
                                                    (setf new-font (push :italic new-font)))
                                                  (if (check-box-checked-p (find-named-sibling item :Underline))
                                                    (setf new-font (push :underline new-font)))
                                                  (if (check-box-checked-p (find-named-sibling item :Outline))
                                                    (setf new-font (push :outline new-font)))
                                                  (if (check-box-checked-p (find-named-sibling item :Shadow))
                                                    (setf new-font (push :shadow new-font)))
                                                  (if (check-box-checked-p (find-named-sibling item :Condense))
                                                    (setf new-font (push :condense new-font)))
                                                  (if (check-box-checked-p (find-named-sibling item :Extend))
                                                    (setf new-font (push :extend new-font)))
                                                  (if (null new-font)
                                                    (setf new-font (list :plain)))
                                                  (push :srcor new-font)
                                                  (push (read-from-string (dialog-item-text font-sizer)) new-font)
                                                  (push (cell-contents font-selector (car (selected-cells font-selector))) new-font)
                                                  (return-from-modal-dialog 
                                                   (list new-font
                                                         (view-nick-name (pushed-radio-button dialog orientation-cluster))
                                                         (view-nick-name (pushed-radio-button dialog expand-cluster))))))
                                            :default-button t)
                          
                          (make-dialog-item 'button-dialog-item
                                            #@(285 240) #@(60 20) " Cancel "
                                            #'(lambda (item)
                                                (declare (ignore item))
                                                (return-from-modal-dialog 
                                                 (list font
                                                       orientation
                                                       expand)))
                                            :default-button nil)

                         ))))

    (let ((font-pos (position (find-if #'stringp font) *font-list* :test #'string-equal))
          (font-selector (view-named :font-selector setup-dialog)))
      (scroll-to-cell font-selector 0 font-pos)
      (cell-select font-selector 0 font-pos))
                  
    (modal-dialog setup-dialog t)))

;;____________________________________________________________________________________
;; default-setup

(defun default-setup ()
  (multiple-value-setq 
    (*default-font*
     *default-orientation*
     *default-expand*)
    (values-list (get-graph-setup "Default"
                                  *default-font*
                                  *default-orientation*
                                  *default-expand*))))

;;____________________________________________________________________________________
;; GRAPHER MENU

(defparameter *grapher-menu*
  (let ((graph-goal-concept-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Graph Goal Concept"
                        :disabled t
                        :menu-item-action #'(lambda () (eval-enqueue '(graph-rule *top-level-call*)))))

        (graph-rule-menu-item
         (make-instance 'menu-item 
                        :menu-item-title "Graph Rule"
                        :menu-item-action #'(lambda () (eval-enqueue '(graph-rule)))))

        (graph-setup-menu-item
         (make-instance 'window-menu-item
                        :menu-item-title "Graph Setup"
                        :menu-item-action 'graph-setup))

        (default-setup-menu-item
          (make-instance 'menu-item
                         :menu-item-title "Default Setup"
                         :menu-item-action #'(lambda () (eval-enqueue '(default-setup)))))
        )

    (set-menu-item-update-function graph-goal-concept-menu-item
                                   #'(lambda (item) (if *top-level-call*
                                                      (menu-item-enable item)
                                                      (menu-item-disable item))))
    (make-instance 'menu
                   :menu-title "Grapher"
                   :menu-items
                   (list graph-goal-concept-menu-item
                         graph-rule-menu-item
                         (make-instance 'menu-item :menu-item-title "-" :disabled t)
                         graph-setup-menu-item
                         default-setup-menu-item))))

;;;____________________________________________________________________________________
;;;  resize-all-nodes

(defun resize-all-nodes (graph-view &optional (node (graph-root graph-view)))
  (when node
    (size-node graph-view node)
    (dolist (children (node-children node))
      (dolist (child children)
        (resize-all-nodes graph-view child)))))

;;;____________________________________________________________________________________
;;;  clear-external-text

(defun clear-external-text (graph-view &optional (node (graph-root graph-view)))
  (when node
    (setf (node-external-text node) nil)
    (dolist (children (node-children node))
      (dolist (child children)
        (clear-external-text graph-view child)))))

;;;__________________________________________________________________________________
;;; return-selected-nodes
;;;
;;; Searches the tree depth first stops searching down a brach when the first selected
;;; node is encountered.  Returns a list of these first selected nodes.

(defun return-selected-nodes (node &optional (selected-nodes nil))
  (if (node-selected? node)
    (setf selected-nodes (nconc selected-nodes (list node)))
    (dolist (children (node-children node) selected-nodes)
      (dolist (child children)
        (setf selected-nodes (return-selected-nodes child selected-nodes))))))

;;;__________________________________________________________________________________
;;; return-deleted-nodes

(defun return-deleted-nodes (node &optional (deleted-nodes nil))
  (if (node-deleted? node)
    (setf deleted-nodes (nconc deleted-nodes (list node))))
  (dolist (children (node-children node) deleted-nodes)
    (dolist (child children)
      (setf deleted-nodes (return-deleted-nodes child deleted-nodes)))))

;;;__________________________________________________________________________________
;;; some-node-selected

(defun some-node-selected (node &optional (selected? nil))
  (cond
   (selected? t)
   ((node-selected? node) t)
   (t (dolist (children (node-children node) selected?)
        (dolist (child children)
          (setf selected? (some-node-selected child selected?)))))))

;;;__________________________________________________________________________________
;;; some-deleted-node-selected

(defun some-deleted-node-selected (node &optional (selected? nil))
  (cond
   (selected? t)
   ((and (node-selected? node) (node-deleted? node)) t)
   (t (dolist (children (node-children node) selected?)
        (dolist (child children)
          (setf selected? (some-deleted-node-selected child selected?)))))))


;;;__________________________________________________________________________________
;;; some-undeleted-node-selected

(defun some-undeleted-node-selected (node &optional (selected? nil))
  (cond
   (selected? t)
   ((and (node-selected? node) (not (node-deleted? node))) t)
   (t (dolist (children (node-children node) selected?)
        (dolist (child children)
          (setf selected? (some-undeleted-node-selected child selected?)))))))


;;;____________________________________________________________________________________
;;;  conjunction-corresponds-to-clause-p

(defun conjunction-corresponds-to-clause-p (conjunction prolog-clause)
  (let ((corresponds nil))
    (when (= (length prolog-clause) (length conjunction))
      (setf corresponds t)
      (do* ((rest conjunction (cdr rest))
            (node (car rest) (car rest)))
           ((or (null rest) (null corresponds)) nil)
        (if (eql (node-kind node) :not)
          (unless (not-node-corresponds-to-some-literal-p node prolog-clause)
            (setf corresponds nil))
          (unless (member (node-literal node) prolog-clause :test #'equal)
            (setf corresponds nil)))))
    corresponds))

;;;____________________________________________________________________________________
;;;  not-node-corresponds-to-some-literal-p

(defun not-node-corresponds-to-some-literal-p (not-node prolog-clause)
  (let ((corresponds nil))
      (dolist (literal prolog-clause)
        (when (eql (car literal) 'not)
          (dolist (conjunction (node-children not-node))
            (if (conjunction-corresponds-to-clause-p conjunction (cdr literal))
              (setf corresponds t)))))
      corresponds))

;;;____________________________________________________________________________________
;;;  find-node-conjunction-corresponding-to-clause

(defmethod find-node-conjunction-corresponding-to-clause ((view graph-view) prolog-clause frontier)
  (let ((corresponding-conjunction nil)
        (clause (substitute-vars prolog-clause (substitution view))))
    (do* ((rest frontier (cdr rest))
          (conjunction (car rest) (car rest)))
         ((or corresponding-conjunction (null rest)) nil) 
      (when (conjunction-corresponds-to-clause-p conjunction clause)
        (setf corresponding-conjunction conjunction)))
    corresponding-conjunction))



;;;____________________________________________________________________________________
;;;  less-colorful

(defun less-colorful ()
  (setf *unoperationalized-pen-size* #@(2 2))
  (setf *unoperationalized-pen-pattern* *gray-pattern*)
  (setf *unoperationalized-fill-pattern* *light-gray-pattern*)
  (setf *unoperationalized-frame-color* *gray-color*)
  (setf *unoperationalized-fill-color* *light-gray-color*)
  
  (setf *normal-pen-size* #@(1 1))
  (setf *normal-pen-pattern* *black-pattern*)
  (setf *normal-fill-pattern* *white-pattern*)
  (setf *normal-frame-color* *black-color*)
  (setf *normal-fill-color* *white-color*)
  
  (setf *extensional-pen-size* #@(2 2))
  (setf *extensional-pen-pattern* *black-pattern*)
  (setf *extensional-fill-pattern* *white-pattern*)
  (setf *extensional-frame-color* *red-color*)
  (setf *extensional-fill-color* *white-color*)
  
  (setf *builtin-pen-size* #@(2 2))
  (setf *builtin-pen-pattern* *black-pattern*)
  (setf *builtin-fill-pattern* *white-pattern*)
  (setf *builtin-frame-color* *pink-color*)
  (setf *builtin-fill-color* *white-color*)
  
  (setf *intensional-pen-size* #@(2 2))
  (setf *intensional-pen-pattern* *black-pattern*)
  (setf *intensional-fill-pattern* *white-pattern*)
  (setf *intensional-frame-color* *orange-color*)
  (setf *intensional-fill-color* *white-color*)
  
  (setf *cliche-pen-size* #@(2 2))
  (setf *cliche-pen-pattern* *black-pattern*)
  (setf *cliche-fill-pattern* *white-pattern*)
  (setf *cliche-frame-color* *blue-color*)
  (setf *cliche-fill-color* *white-color*)
  
  (setf *determinate-pen-size* #@(2 2))
  (setf *determinate-pen-pattern* *black-pattern*)
  (setf *determinate-fill-pattern* *white-pattern*)
  (setf *determinate-frame-color* *green-color*)
  (setf *determinate-fill-color* *white-color*)
  
  (setf *ebl-pen-size* #@(1 1))
  (setf *ebl-pen-pattern* *black-pattern*)
  (setf *ebl-fill-pattern* *white-pattern*)
  (setf *ebl-frame-color* *black-color*)
  (setf *ebl-fill-color* *white-color*)

  (map-windows #'invalidate-view))


;;;____________________________________________________________________________________
;;;  more-colorful

(defun more-colorful ()
  (setf *unoperationalized-pen-size* #@(2 2))
  (setf *unoperationalized-pen-pattern* *black-pattern*)
  (setf *unoperationalized-fill-pattern* *black-pattern*)
  (setf *unoperationalized-frame-color* *gray-color*)
  (setf *unoperationalized-fill-color* *light-gray-color*)
  
  (setf *normal-pen-size* #@(1 1))
  (setf *normal-pen-pattern* *black-pattern*)
  (setf *normal-fill-pattern* *black-pattern*)
  (setf *normal-frame-color* *black-color*)
  (setf *normal-fill-color* *white-color*)
  
  (setf *extensional-pen-size* #@(2 2))
  (setf *extensional-pen-pattern* *black-pattern*)
  (setf *extensional-fill-pattern* *black-pattern*)
  (setf *extensional-frame-color* *black-color*)
  (setf *extensional-fill-color* *yellow-color*)
  
  (setf *builtin-pen-size* #@(2 2))
  (setf *builtin-pen-pattern* *black-pattern*)
  (setf *builtin-fill-pattern* *black-pattern*)
  (setf *builtin-frame-color* *black-color*)
  (setf *builtin-fill-color* *tan-color*)
  
  (setf *intensional-pen-size* #@(2 2))
  (setf *intensional-pen-pattern* *black-pattern*)
  (setf *intensional-fill-pattern* *black-pattern*)
  (setf *intensional-frame-color* *black-color*)
  (setf *intensional-fill-color* *orange-color*)
  
  (setf *cliche-pen-size* #@(2 2))
  (setf *cliche-pen-pattern* *black-pattern*)
  (setf *cliche-fill-pattern* *black-pattern*)
  (setf *cliche-frame-color* *black-color*)
  (setf *cliche-fill-color* *light-blue-color*)
  
  (setf *determinate-pen-size* #@(2 2))
  (setf *determinate-pen-pattern* *black-pattern*)
  (setf *determinate-fill-pattern* *black-pattern*)
  (setf *determinate-frame-color* *black-color*)
  (setf *determinate-fill-color* *pink-color*)
  
  (setf *ebl-pen-size* #@(1 1))
  (setf *ebl-pen-pattern* *black-pattern*)
  (setf *ebl-fill-pattern* *black-pattern*)
  (setf *ebl-frame-color* *black-color*)
  (setf *ebl-fill-color* *white-color*)

  (map-windows #'invalidate-view)
)




(provide :grapher)

