;;;____________________________________________________________________________________
;;;                          Animate Gain
;;;
;;; The code in this file is designed to work in conjunction with grapher.lisp
;;; to display a prolog object and its associated gain process of FOCL.
;;;
;;;  Created and designed by Clifford A. Brunk 10/20/91
;;;
;;;  Problems:
;;;____________________________________________________________________________________

(require 'grapher)

(in-package :user)

;;;_______________________________________________________________________________
;;;  gain-window

(defclass gain-window (window)
  (
   (not-node :initarg :not-node :initform nil :accessor not-node)
   (simple-node :initarg :simple-node :initform nil :accessor simple-node)
   (simple-literal :initarg :simple-literal :initform (cons 'a 'b) :accessor simple-literal)
   (node-list :initarg :node-list :initform nil :accessor node-list)
   (nodes-hidden :initarg :nodes-hidden :initform nil :accessor nodes-hidden)
   ))

;;;_______________________________________________________________________________
;;;  initialize-instance

(defmethod initialize-instance ((window gain-window) &rest initargs)
  (setf (getf initargs :color-p) t)
  (apply #'call-next-method window initargs )
  (set-view-scroll-position window 0 0)
  (set-window-font window *default-font*)
  (let* ((window-size-h (point-h (view-size window)))
         (window-size-v (point-v (view-size window)))
         (display-font '("Chicago" 12 :SRCOR :PLAIN))
         (y 2))
    (add-subviews 
     window
     (make-instance 'graph-scroller
                    :view-position #@(0 20)
                    :view-size (make-point window-size-h (- window-size-v 20))
                    :view-nick-name :graph-scroller)
     
     (make-dialog-item 'static-text-dialog-item
                       (make-point 7 y) (make-point 100 16)
                       ""
                       nil
                       :view-font display-font
                       :view-nick-name :source)
     
     (make-dialog-item 'static-text-dialog-item
                       (make-point 115 y) (make-point 80 16)
                       ""
                       nil
                       :view-font display-font
                       :view-nick-name :gain)
     
     (make-dialog-item 'static-text-dialog-item
                       (make-point 220 y) (make-point 60 16)
                       ""
                       nil
                       :view-font display-font
                       :view-nick-name :pos)
     
     (make-dialog-item 'static-text-dialog-item
                       (make-point 280 y) (make-point 60 16)
                       ""
                       nil
                       :view-font display-font
                       :view-nick-name :neg)
     )
    (let ((graph-view (view-named :graph-view (view-named :graph-scroller window))))
      (setf (graph-window graph-view) window)
      (set-font-dependent-attributes graph-view)
      (let ((node-v (round (- (point-v (view-size graph-view)) (node-height graph-view)) 2)))
        (setf (not-node window) (create-node graph-view
                                             :literal '(not)
                                             :left 20
                                             :top node-v
                                             :hidden? t)
              (simple-node window) (create-node graph-view
                                                :parent (not-node window)
                                                :literal '(simple-node)
                                                :left 65
                                                :top node-v
                                                :hidden? t)
              (node-children (not-node window)) (list (list (simple-node window)))
              (graph-root graph-view) (not-node window)
              (node-selection-constraint graph-view) :no-drag))
      (reset-scroll-bars (view-named :graph-scroller window))
      )))
    

;;;_______________________________________________________________________________
;;;   set-view-size

(defmethod set-view-size ((window gain-window) h &optional (v nil))
  (if (null v)
    (setf v (point-v h)
          h (point-h h)))
  (reset-view-size (view-named :graph-scroller window) h (- v 20))
  (call-next-method))

;;;_______________________________________________________________________________
;;;   view-draw-contents

(defmethod view-draw-contents ((window gain-window))
  (with-focused-view window
    (call-next-method)
    (with-focused-view window
      (_PenSize :long #@(1 1))
      (_PenPat :ptr *black-pattern*)
      (with-fore-color *black-color*
        (_MoveTo :long (make-point -1 19))
        (_LineTo :long (make-point 3000 19))))))


;;;_______________________________________________________________________________
;;;   window-zoom-event-handler

(defmethod window-zoom-event-handler ((window gain-window) message)
  (declare (ignore message))
  (call-next-method)
  (reset-view-size (view-named :graph-scroller window) (subtract-points (view-size window) #@(0 20)))
  )


;;;_______________________________________________________________________________
;;;   view-key-event-handler

(defmethod view-key-event-handler ((window gain-window) char)
  (let* ((view (view-named :graph-view (view-named :graph-scroller window))))
    (case (char-code char)
      ((71 103)    ;; G g
       (regraph-node view))
      ((73 105)    ;; I i
       (inspect-last-node-selected-as-pred view))
      ((82 114)    ;; R r
       (force-graph-redraw view))
      (others
       nil))))

;;;_______________________________________________________________________________
;;;  hide-nodes

(defmethod hide-nodes ((window gain-window))
  (dolist (node (node-list window))
    (setf (node-hidden? node) t
          (node-children node) nil
          (node-selected? node) nil))
  (let ((graph-view (view-named :graph-view (view-named :graph-scroller window))))
    (setf (view-top graph-view) 0
          (view-left graph-view) 0)
    (place-node (not-node window) 20 20)
    (place-node (simple-node window) 70 20)
    (setf (node-hidden? (not-node window)) t
          (node-hidden? (simple-node window)) t
          (node-selected? (not-node window)) nil
          (node-selected? (simple-node window)) nil
          (nodes-hidden window) t)))

;;;_______________________________________________________________________________
;;;  get-node

(defmethod get-node ((window gain-window) 
                     graph-view
                     &key (literal nil)
                     (parent nil)
                     (state :normal)
                     (deleted? nil)
                     (expand t))
  (do* ((nodes (node-list window) (cdr nodes))
        (node (car nodes) (car nodes)))
       ((or (null node) (node-hidden? node))
        (cond ((null node)
               (setf node (create-node graph-view 
                                       :literal literal
                                       :parent parent
                                       :deleted? deleted?
                                       :state state
                                       :expand expand)
                     (node-list window) (push node (node-list window)))
               node)
              (t
               (setf (node-hidden? node) nil
                     (node-parent node) parent
                     (node-deleted? node) deleted?
                     (node-state node) state)
               (set-node-literal graph-view node literal expand)
               node)))))

;;;_______________________________________________________________________________
;;;  display-gain-caption

(defmethod display-gain-caption ((window gain-window) 
                                 &key (source *status*) (gain nil) (pos nil) (neg nil))
  (let (g p n)
    (if (gain-p gain)
      (setf g (gain-gain gain)
            p (gain-pp gain) 
            n (gain-nn gain))
      (setf g gain
            p pos
            n neg))
    (set-dialog-item-text (view-named :source window) (format nil "~A" source))
    (set-dialog-item-text (view-named :gain window) (cond ((numberp g) (format nil "gain: ~5F" g))
                                                          ((null g) (format nil "gain: ----"))
                                                          (t (format nil "gain: ~A" g))))
    (set-dialog-item-text (view-named :pos window) (format nil "~5@A +" p))
    (set-dialog-item-text (view-named :neg window) (format nil "~5@A -" n))))


;;;____________________________________________________________________________________
;;;  setup-BEST-GAIN-WINDOW

(defun setup-BEST-GAIN-WINDOW ()
  (unless (window-p *BEST-GAIN-WINDOW*)
    (setf *BEST-GAIN-WINDOW*
          (make-instance 'gain-window
                         :window-show nil
                         :window-title "Best Gain"
                         :view-font *default-font*
                         :view-size #@(340 115)
                         :view-position (make-point 2 (- *screen-height* 118))
                         :close-box-p t))
    (set-font-dependent-attributes (view-named :graph-view 
                                               (view-named :graph-scroller 
                                                           *BEST-GAIN-WINDOW*))))
  (clear-gain *BEST-GAIN-WINDOW*))


;;;____________________________________________________________________________________
;;;  setup-CURRENT-GAIN-WINDOW

(defun setup-CURRENT-GAIN-WINDOW ()
  (unless (window-p *CURRENT-GAIN-WINDOW*)
    (setf *CURRENT-GAIN-WINDOW*
          (make-instance 'gain-window
                         :window-show nil
                         :window-title "Current Gain"
                         :view-font *default-font*
                         :view-size #@(340 115)
                         :view-position (if (window-p *BEST-GAIN-WINDOW*)
                                          (make-point 2 (- *screen-height* 256))
                                          (make-point 2 (- *screen-height* 118)))
                         :close-box-p t))
    (set-font-dependent-attributes (view-named :graph-view 
                                               (view-named :graph-scroller 
                                                           *CURRENT-GAIN-WINDOW*))))
  (clear-gain *CURRENT-GAIN-WINDOW*))


;;;_______________________________________________________________________________
;;; convert-cliche-to-prolog
;;;
;;;  Assume that cliches will always be conjunctive and that everything is instancuated
;;;  except for the last position.

(defun convert-cliche-to-prolog (cliche last-pred last-vars negated?)
  (let ((prolog-cliche nil)
        (cliche-literal nil))
    (dolist (literal-info (butlast cliche))
      (setf cliche-literal (cons (pred-name (literal-info-pred literal-info))
                                 (literal-info-variabilization literal-info)))
      (if (literal-info-negated? literal-info)
        (setf cliche-literal (list 'not cliche-literal)))
      (setf prolog-cliche (nconc prolog-cliche (list cliche-literal))))
    (setf cliche-literal (cons last-pred last-vars))
    (if negated?
      (setf cliche-literal (list 'not cliche-literal)))
    (setf prolog-cliche (nconc prolog-cliche (list cliche-literal)))
    prolog-cliche))
      

;;;_______________________________________________________________________________
;;; return-literal-and-depth

(defun return-literal-and-depth (entity)
  (if (and (consp entity)
           (not (symbolp (car entity))))
    (multiple-value-bind (literal depth)
                         (return-literal-and-depth (car entity))
      (values literal (incf depth)))
    (values entity 0)))

;;;_______________________________________________________________________________
;;;  return-simplified-entity-and-kind

(defun return-simplified-entity-and-kind (entity)
  (multiple-value-bind (literal depth)
                       (return-literal-and-depth entity)
    (case depth
      (0 (cond ((null literal) (values literal :empty))
               ((p-p literal) (values literal :p-struct))
               ((literal-p literal) (if (literal-negated? literal)
                                      (let ((negated-literal (literal-negated-literals literal)))
                                        (if (or (literal-negated? negated-literal)
                                                (literal-next negated-literal)
                                                (literal-next literal))
                                          (values literal :conjunction-of-literal-structs)
                                          (values negated-literal :negated-literal-struct)))
                                      (if (literal-next literal)
                                        (values literal :conjunction-of-literal-structs)
                                        (values literal :literal-struct))))
               ((listp literal)
                (if (eq (first literal) 'not)
                  (if (rest (rest literal))
                    (values literal :negated-conjunction-of-prolog-literals)
                    (values literal :negated-prolog-literal))
                  (values literal :prolog-literal)))
               (t (values entity :unknown))  ))

      (1 (let ((other-literals (rest entity)))
           (cond ((literal-p literal) (if other-literals
                                        (values entity :disjunction-of-literal-structs)
                                        (if (or (literal-next literal)
                                                (literal-negated-literals literal))
                                          (values literal :conjunction-of-literal-structs)
                                          (values literal :literal-struct))))

                 ((listp literal)
                  (let ((other-literals (rest entity)))
                    (if other-literals
                      (values entity :conjunction-of-prolog-literals)
                      (if (eq (first literal) 'not)
                        (if (rest (rest literal))
                          (values literal :negated-conjunction-of-prolog-literals)
                          (values literal :negated-prolog-literal))
                        (values literal :prolog-literal)))))
                 (t (values entity :unknown) ))))

      (2 (if (listp literal)
           (let ((other-conjunctions (rest entity))
                 (other-literals (rest (first entity))))
             (if other-conjunctions
               (values entity :disjunction-of-prolog-literals)
               (if other-literals
                 (values (first entity) :conjunction-of-prolog-literals)
                 (if (eq (first literal) 'not)
                   (if (rest (rest literal))
                     (values literal :negated-conjunction-of-prolog-literals)
                     (values literal :negated-prolog-literal))
                   (values literal :prolog-literal)))))
           (values entity :unknown) ))

      (otherwise (values entity :unknown))
      )
    )
  )



;;;_______________________________________________________________________________
;;;_______________________________________________________________________________
;;;  Function to be used by FOCL code to interface to Gain Windows
;;;_______________________________________________________________________________
;;;_______________________________________________________________________________

;;;_______________________________________________________________________________
;;;  clear-gain
;;;
;;;  window    the gain-window which will be cleared

(defmethod clear-gain ((window gain-window))
  (when (window-p window)
    (set-dialog-item-text (view-named :source window) "")
    (set-dialog-item-text (view-named :gain window) "")
    (set-dialog-item-text (view-named :pos window) "")
    (set-dialog-item-text (view-named :neg window) "")
    (hide-nodes window)
    (erase-view (view-named :graph-view (view-named :graph-scroller window)))))


;;;_______________________________________________________________________________
;;;  display-pred-vars-gain
;;;
;;;  window    the gain-window which will contain the graph
;;;  pred      the name of the relation used in the literal
;;;  vars      a list of the variables used in the literal
;;;  negated?  t if literal is negated, nil otherwsie
;;;  source    :ebl, :extensional, :builtin, :intensional, :cliche, or :determinate
;;;  gain      the gain structure associated with literal

(defmethod display-pred-vars-gain ((window gain-window) pred vars negated?
                                   &key (source *status*) (gain nil) (pos nil) (neg nil) (update-gain? t))
  (when (window-p window)
    (unless (nodes-hidden window) (hide-nodes window))
    (let* ((graph-scroller (view-named :graph-scroller window))
           (graph-view (view-named :graph-view graph-scroller))
           (simple-literal (simple-literal window))
           (simple-node (simple-node window))
           (not-node (not-node window)))
      (rplaca simple-literal pred)
      (rplacd simple-literal vars)
      (set-node-literal graph-view simple-node simple-literal t)
      (setf (node-state not-node) source
            (node-state simple-node) source
            (node-kind simple-node) (if (eq (node-kind simple-node) :intensional)
                                      :unexpanded
                                      (node-kind simple-node))
            (node-hidden? not-node ) (not negated?)
            (node-hidden? simple-node) nil
            (node-selected? (not-node window)) nil
            (node-selected? (simple-node window)) nil)
      (if update-gain?
        (display-gain-caption window :source source :gain gain :pos pos :neg neg))
      (reset-scroll-bars graph-scroller)
      (force-graph-redraw graph-view)
      ))
  (values))


;;;_______________________________________________________________________________
;;;  display-entity-gain
;;;
;;;  window       the gain-window which will contain the graph
;;;  entity       a p-structure, a literal-structure, a list of literal-structures, or prolog
;;;  vars         a list of the variables used in entity
;;;  negated?     t if entity is negated, nil otherwsie
;;;  source       :ebl, :extensional, :builtin, :intensional, :cliche, :determinate, or :normal
;;;  gain         the gain structure associated with entity, or a number, or nil
;;;  pos          a number or nil (if gain is a structure gain-pp used instead)
;;;  neg          a number or nil (if gain is a structure gain-nn used instead)
;;;  update-gain? t or nil determines if gain should be changed when graph is redrawn
;;;  root-label   text that will be displayed as the root of enitity if it is a
;;;               literal-structure or prolog-disjunction

(defmethod display-entity-gain ((window gain-window) entity
                                &key (vars nil) (negated? nil) (source *status*)
                                     (gain nil) (pos nil) (neg nil) (update-gain? t)
                                     (graph t) (instantiated-cliche nil))
  (let ((root-label nil))
    (when (and (window-p window) graph)
      (when (eq graph :if-needed-ebl)
        (setf entity (mapcar #'literal-part* (expand-single-clause-literals entity)))
        )
      
      (when instantiated-cliche
        (setf root-label "clich"
              entity (convert-cliche-to-prolog instantiated-cliche 
                                               (if (p-p entity) 
                                                 (p-name entity) entity)
                                               vars
                                               negated?)))      
      (multiple-value-bind (literal kind)
                           (return-simplified-entity-and-kind entity)
        (case kind
          (:empty (values))
          (:p-struct (display-pred-vars-gain window
                                             (p-name literal)
                                             vars
                                             negated?
                                             :source source
                                             :gain gain
                                             :pos pos
                                             :neg neg
                                             :update-gain? update-gain?))
          (:literal-struct (display-pred-vars-gain window
                                                   (literal-predicate-name literal)
                                                   (literal-variablization literal)
                                                   nil
                                                   :source source
                                                   :gain gain
                                                   :pos pos
                                                   :neg neg
                                                   :update-gain? update-gain?))
          (:negated-literal-struct (display-pred-vars-gain window
                                                           (literal-predicate-name literal)
                                                           (literal-variablization literal)
                                                           t
                                                           :source source
                                                           :gain gain
                                                           :pos pos
                                                           :neg neg
                                                           :update-gain? update-gain?))
          (:prolog-literal (display-pred-vars-gain window
                                                   (first literal)
                                                   (rest literal)
                                                   negated?
                                                   :source source
                                                   :gain gain
                                                   :pos pos
                                                   :neg neg
                                                   :update-gain? update-gain?))
          
          (:negated-prolog-literal (let ((unnegated-literal (cadr literal)))
                                     (display-pred-vars-gain window
                                                             (car unnegated-literal)
                                                             (cdr unnegated-literal)
                                                             t
                                                             :source source
                                                             :gain gain
                                                             :pos pos
                                                             :neg neg
                                                             :update-gain? update-gain?)))
          (otherwise
           (without-interrupts 
            (unless (nodes-hidden window) (hide-nodes window))
            (let* ((graph-scroller (view-named :graph-scroller window))
                   (graph-view (view-named :graph-view graph-scroller))
                   (simple-node (simple-node window)))
              (setf (nodes-hidden window) nil
                    (node-hidden? (not-node window)) t
                    (node-hidden? (simple-node window)) nil
                    (node-selected? (not-node window)) nil
                    (node-selected? (simple-node window)) nil)
              (if update-gain?
                (display-gain-caption window :source source :gain gain :pos pos :neg neg))
              
              (case kind
                (:negated-conjunction-of-prolog-literals
                 (set-node-literal graph-view simple-node literal)
                 (setf (node-state simple-node) source
                       (node-children simple-node) (list (connect-clause graph-view
                                                                         simple-node
                                                                         (cdr literal)
                                                                         source
                                                                         nil))))
                ((:conjunction-of-prolog-literals :conjunction-of-literal-structs)
                 (setf (node-literal simple-node) nil
                       (node-state simple-node) :blank
                       (node-kind simple-node) :intensional
                       (node-text simple-node) (if root-label root-label "")
                       (node-children simple-node) (list (connect-clause graph-view
                                                                         simple-node
                                                                         literal
                                                                         source
                                                                         nil))))
                ((:disjunction-of-prolog-literals :disjunction-of-literal-structs)
                 (setf (node-literal simple-node) nil
                       (node-state simple-node) :blank
                       (node-kind simple-node) :intensional
                       (node-text simple-node) (if root-label root-label "")
                       (node-children simple-node) (connect-clauses graph-view
                                                                    simple-node
                                                                    literal
                                                                    source
                                                                    nil)))
                
                (otherwise (display-pred-vars-gain window
                                                   'undecipherable
                                                   'object
                                                   nil
                                                   :source source
                                                   :gain gain
                                                   :pos pos
                                                   :neg neg
                                                   :update-gain? update-gain?))
                )
                
             
              (size-node graph-view simple-node)
              (position-nodes graph-view)
              (position-graph graph-view :centered t)
              (reset-scroll-bars graph-scroller)
              (force-graph-redraw graph-view) 
              )))
          )
        
        (values)))))








