;;;____________________________________________________________________________________
;;;                               KR-FOCL-INTERFACE
;;;
;;; The code in this file is designed to work in conjunction with grapher.lisp and
;;; to provide a interface which will facilitate the revision of a rule base.
;;;
;;;  Created and designed by Clifford A. Brunk 05/28/91
;;;
;;;  Problems:
;;;____________________________________________________________________________________

(require 'grapher)

(in-package :user)

(defvar *THEORY-WINDOW* nil)
(defvar *INDUCED-WINDOW* nil)
(defvar *KR-DIALOG* nil)

(defparameter *window-window-offset* 30)
(defparameter *attachment-notes* nil)

;;;_______________________________________________________________________________
;;; *attach-induced-literal-dialog*

(defclass KR-DIALOG (windoid)
  ((done :initarg :done :initform nil :accessor done)))

;;;_______________________________________________________________________________
;;; create-attach-induced-literal-dialog

(defun create-attach-induced-literal-dialog (induced-literal operationalized-clause)
  (let* ((h 375)
         (v 175)
         (x 10)
         (x1 20)
         (w (- h x x))
         (y 5)
         (button-size #@(60 20))
         (literal-text-v 12)
         (literal-font '("monaco" 9 :plain))
         (static-text-v 16)
         (static-font '("chicago" 12 :plain)))
    (setf *KR-DIALOG*
          (make-instance 
           'KR-DIALOG
           :window-title "Attach Induced Literals"
           :view-size (make-point h v)
           :close-box-p nil
           :window-show t
           :view-position '(:bottom 6)
           :view-subviews
            (list
            (make-dialog-item 
             'static-text-dialog-item (make-point x y) (make-point w static-text-v)
             "During learning the induced literals:" nil
             :view-font static-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x1 (incf y 20)) (make-point w literal-text-v)
             (format nil "~A" induced-literal) nil
             :view-font literal-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x (incf y 18)) (make-point w static-text-v)
             "were added to the operationalized clause:" nil
             :view-font static-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x1 (incf y 20)) (make-point w literal-text-v)
             (format nil "~A" operationalized-clause) nil
             :view-font literal-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x (incf y 18)) (make-point w (* 3 static-text-v))
             (format nil "This indicates that the induced literals~%should be added somewhere along the~%operationalized path.")
             nil
             :view-font static-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x (incf y 56)) (make-point w (* 2 static-text-v))
             (format nil "Select the overly general clause and the~%induced literals to attach to it.") nil
             :view-font static-font)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 70) (- v 60)) button-size
             " Attach "
             #'(lambda (item) (declare (ignore item)) (do-literal-attachment))
             :view-nick-name :attach
             :default-button nil
             :dialog-item-enabled-p nil)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 70) (- v 30)) button-size
             " Done "
             #'(lambda (item)
                 (window-close (view-container item))
                 (setf (done (view-container item)) t))
             :view-nick-name :attach
             :dialog-item-enabled-p t))))))

;;;_______________________________________________________________________________
;;; create-attach-induced-clause-dialog

(defun create-attach-induced-clause-dialog (induced-clause)
  (let* ((h 375)
         (v 175)
         (x 10)
         (x1 20)
         (w (- h x x))
         (y 5)
         (button-size #@(60 20))
         (literal-text-v 12)
         (literal-font '("monaco" 9 :plain))
         (static-text-v 16)
         (static-font '("chicago" 12 :plain)))
    (setf *KR-DIALOG*
          (make-instance 
           'KR-DIALOG
           :window-title "Attach Induced Clause"
           :view-size (make-point h v)
           :close-box-p nil
           :window-show t
           :view-position '(:bottom 6)
           :view-subviews
           (list
            (make-dialog-item 
             'static-text-dialog-item (make-point x y) (make-point w static-text-v)
             "During learning the induced clause:" nil
             :view-font static-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x1 (incf y 20)) (make-point w literal-text-v)
             (format nil "~A" induced-clause) nil
             :view-font literal-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x (incf y 18)) (make-point w static-text-v)
             "was added as a disjunct at the top level." nil
             :view-font static-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x (incf y 26)) (make-point w (* 2 static-text-v))
             (format nil "This clause can be attached as a~%disjunct anywhere in the rule base.") nil
             :view-font static-font)
            (make-dialog-item 
             'static-text-dialog-item (make-point x (incf y 40)) (make-point w (* 3 static-text-v))
             (format nil "Select the overly specific rule and~%the portion of the clause to be~%attached to it as a new disjunct.") nil
             :view-font static-font)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 70) (- v 60)) button-size
             " Attach "
             #'(lambda (item) (declare (ignore item)) (do-clause-attachment))
             :view-nick-name :attach
             :default-button nil
             :dialog-item-enabled-p nil)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 70) (- v 30)) button-size
             " Done "
             #'(lambda (item)
                 (window-close (view-container item))
                 (setf (done (view-container item)) t))
             :view-nick-name :attach
             :dialog-item-enabled-p t))))))

;;;_______________________________________________________________________________
;;; create-delete-dialog

(defun create-delete-dialog ()
  (let* ((h 375)
         (v 115)
         (x 15)
         (w (- h x x))
         (y 5)
         (button-size #@(60 20))
         (static-text-v 16)
         (static-font '("chicago" 12 :plain)))

    (setf *KR-DIALOG*
          (make-instance 
           'KR-DIALOG
           :window-title "Delete Clauses and Literals"
           :view-size (make-point h v)
           :close-box-p nil
           :window-show t
           :view-position '(:bottom 6)
           :view-subviews
           (list
            (make-dialog-item 
             'static-text-dialog-item (make-point x y) (make-point w (* 3 static-text-v))
             (format nil "Literals and clauses which decrease~%the accuracy of the goal concept~%have been marked as deleted.")
             nil
             :view-font static-font)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 90) (- v 100)) #@(80 20)
             " Delete "
             #'(lambda (item) (declare (ignore item))
                (delete-selected-nodes (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*))))
             :view-nick-name :delete
             :dialog-item-enabled-p nil)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 90) (- v 70)) #@(80 20)
             " Un-Delete "
             #'(lambda (item) (declare (ignore item))
                (undelete-selected-nodes (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*))))
             :view-nick-name :un-delete
             :dialog-item-enabled-p nil)

            (make-dialog-item 
             'button-dialog-item
             (make-point (- h 80) (- v 30)) button-size
             " Done "
             #'(lambda (item)
                 (window-close (view-container item))
                 (setf (done (view-container item)) t))
             :view-nick-name :attach
             :default-button t
             :dialog-item-enabled-p t))))))

;;;_______________________________________________________________________________
;;;  learned-via
;;;
;;;  Returns :combination if the clause was learned by a combination of ebl and induction.
;;;          :induction   if the clause was learned by induction alone
;;;          :ebl         if the clause was learned by ebl alone

(defun learned-via (clause)
  (let ((induced nil)
        (operationalized nil))
    (do* ((literal clause (literal-next literal)))
         ((null literal))
      (unless (literal-deleted? literal)
        (if (eql (derivation-type (literal-derivation literal)) :ebl)
          (setf operationalized t)
          (setf induced t))))
    (cond ((and induced operationalized) :combination)
          (induced :induction)
          (operationalized :ebl)
          (t nil))))

;;;_______________________________________________________________________________
;;;  external-duplicate-tree

(defun external-duplicate-tree (node &optional (parent nil))
  (when node
    (let ((copy (copy-node node)))
      (setf (node-parent copy) parent
            (node-selected? copy) nil
            (node-children copy) (mapcar 
                                  #'(lambda (children)
                                      (mapcar
                                       #'(lambda (child)
                                           (external-duplicate-tree child copy))
                                       children))
                                  (node-children copy)))
      copy)))

;;;_______________________________________________________________________________
;;;  nodes-in-tree

(defun nodes-in-tree (root)
  (let ((list-of-nodes nil))
    (labels 
      ((list-em (node)
                (cond ((null node) nil)
                      (t (setf list-of-nodes (push node list-of-nodes))
                         (dolist (children (node-children node))
                           (dolist (child children)
                             (list-em child)))))))
      (list-em root)
      list-of-nodes)))



;;;_______________________________________________________________________________
;;;  clause-string

(defun clause-string (clause &key (ebl? nil) (induction? nil) (both? nil))
  (let ((c-string ""))
    (do* ((literal clause (literal-next literal)))
         ((null literal))
      (cond ((literal-deleted? literal))
            ((and (literal-negated? literal)
                  (or both?
                      (and ebl? (eql (derivation-type (literal-derivation literal)) :ebl))
                      (and induction? (not (eql (derivation-type (literal-derivation literal)) :ebl)))))
             
             (setf c-string (concatenate 'string
                                         c-string
                                         "(NOT "
                                         (clause-string (literal-negated-literals literal)
                                                        :ebl? ebl?
                                                        :induction? induction?
                                                        :both? both?)
                                         ") "
                                         )))
            ((or both?
                 (and ebl? (eql (derivation-type (literal-derivation literal)) :ebl))
                 (and induction? (not (eql (derivation-type (literal-derivation literal)) :ebl))))
             (setf c-string
                   (if (string-equal c-string "")
                     (return-literal-string (literal-predicate-name literal) 
                                            (literal-variablization literal))
                     (concatenate 'string c-string ", " 
                                  (return-literal-string (literal-predicate-name literal) 
                                                         (literal-variablization literal))))))
            (t )))
    (if induction?
      c-string
      (concatenate 'string c-string "."))))

;;;_______________________________________________________________________________
;;;  clauses-string

(defun clauses-string (clauses &key (ebl? nil) (induction? nil) (both? nil))
  (format nil "~{~A~^; ~}"
          (mapcar #'(lambda (clause)
                      (clause-string clause :ebl? ebl? :induction? induction? :both? both?))
                  clauses)))


;;;________________________________________________________________________________
;;; Knowledge Revision Heuristics
;;;________________________________________________________________________________

;;;________________________________________________________________________________
;;;  KR-Heuristic 0 - If it isn't broken don't fix it.  If the clause was
;;;                   learned using ebl alone then it is ok unless it can be shown
;;;                   to be incorrect elsewhere.
;;;________________________________________________________________________________


;;;________________________________________________________________________________
;;;  KR-Heuristic 1 - When a clause is learned by operationalizing a portion of the
;;;                   domain theory and then inducing some predicates to further
;;;                   specialize the clause, it may be a sign that one or more of
;;;                   the operationalized clauses need to be modified by adding an
;;;                   extra literal from the induced predicates.
;;;________________________________________________________________________________

;;;________________________________________________________________________________
;;; ebl-portion-equal
;;; Returns true if the ebl portions of two clauses are equal.  The order of the
;;; literals in the cluases does not effect their equality.

(defun ebl-portion-equal (clause1 clause2)
  (let ((equal? t))
    (do ((l1 clause1 (literal-next l1)))
        ((null l1) equal?)
      (when (eql (derivation-type (literal-derivation l1)) :EBL)
        (do ((l2 clause2 (literal-next l2)))
            ((or (null l2)
                 (and (eql (derivation-type (literal-derivation l2)) :EBL)
                      (eql (literal-predicate-name l2) (literal-predicate-name l1))
                      (eql (literal-variablization l2) (literal-variablization l2))))
             (unless l2 (setf equal? nil))))))))

;;;________________________________________________________________________________
;;;  KR-Heuristic-1 (learned-concept-description)
;;;
;;;  Isolates the clauses of learned-concept-description learned by :EBL followed
;;;  by some inductive method. Then calls get-induced-literal-attachment which asks
;;;  the user where along the operationalized path to attach the induced literals.

(defun KR-Heuristic-1 (Learned-Concept-Description)
  (let ((combo-clauses nil)
        (clauses-with-same-ebl nil))
    (dolist (clause Learned-Concept-Description)
      (when (eql (learned-via clause) :COMBINATION)
        (setf combo-clauses (nconc combo-clauses (list clause)))))
    (if combo-clauses
      (do* ((cs combo-clauses (cdr cs))
            (c (car cs) (car cs)))
           ((null cs) nil)
        (setf clauses-with-same-ebl (list c))
        (do* ((rcs cs (cdr rcs))
              (r (cadr rcs) (cadr rcs)))
             ((null (cdr rcs)) nil)
          (when (ebl-portion-equal c r)
            (setf clauses-with-same-ebl (nconc clauses-with-same-ebl (list r)))
            (rplacd rcs (cddr rcs))))
        (setf *attachment-notes* nil)
        (get-induced-literal-attachment clauses-with-same-ebl)
        (internalize-literal-attachment *attachment-notes*))
      (message-dialog (format nil "Heuristic 1 does not apply.~%~%No operationalized clauses needed to be specialized using induction.")
                      :position :centered
                      :size #@(350 115)))))

;;;________________________________________________________________________________
;;;  KR-Heuristic 2 - When a clause is learned exclusively by induction, it may be
;;;                   a sign that some unoperationalized clauses need to be changed.
;;;                   In this case KR-FOCL may suggest two types of revisons.
;;;                   KR-FOCL may suggest replacing a clause in the rule base which
;;;                   meets either of the following two conditions:
;;;
;;;                   a. The induced predicates subsume an unoperationalized clause.
;;;                      This may be a sign that the clause can be replaced by the
;;;                      induced predicates. In effect, this modification removes
;;;                      superfluous literals from a clause.
;;;                      [GENERALIZATION]
;;;
;;;                   b. The induced predicates are subsumed by an unoperationalized
;;;                      clause. This also can be a sign that the clause can be
;;;                      replaced by the induced predicates. In effect, this 
;;;                      modification adds extra literals to a clause.
;;;                      [SPECIALIZATION]
;;;
;;;                   Alternatively when a clause is learned exclusively by induction,
;;;                   it could be a sign that the induced predicates may be another
;;;                   clause for the top level predicate or for a predicate that is
;;;                   part of an unoperationalized clause.
;;;________________________________________________________________________________
;;;
;;;  KR-Heuristic-2 (learned-concept-description unoperationalized-clauses)
;;;
;;;  Isolates the clauses of learned-concept-description learned exclusively by
;;;  inductive methods. Then calls get-induced-clause-attachment which asks the
;;;  the user where to attach the induced clauses, following the suggestions
;;;  outlined above.
;;;  
;;;________________________________________________________________________________

(defun KR-Heuristic-2 (Learned-Concept-Description)
  (let ((applies? nil))
    (dolist (induced-clause (remove-if-not #'(lambda (C) (eql (learned-via C) :induction))
                                           Learned-Concept-Description))
      (setf applies? t
            *attachment-notes* nil)
      (get-induced-clause-attachment induced-clause)
      (internalize-clause-attachment *attachment-notes*))
    (unless applies?
      (message-dialog (format nil "Heuristic 2 does not apply.~%~%No clauses were learned using induction alone.")
                      :position :centered
                      :size #@(350 115)))))

;;;________________________________________________________________________________
;;;  KR-Heuristic 3 - When a clause is not operationalized during the entire 
;;;                   learning process, it is a sign ther the clause can be 
;;;                   eliminated.
;;;________________________________________________________________________________
;;;  KR-Heuristic-3 (unoperationalized-rules)
;;;
;;;  Calls delete-unoperationalized-clauses which alerts the user to any
;;;  modification of an unoperationalized-clause and then asks the users whether
;;;  or not to delete each unoperationalized-clause.
;;;
;;;________________________________________________________________________________

(defun KR-Heuristic-3 ()
  (let ((applies? nil)
        (theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*))))
    (setf applies? (mark-and-select-deleted-and-unoperationalized-nodes (graph-root theory-view)))
    (force-graph-redraw theory-view)
    (cond
     (applies?
      (get-deletions)
      (internalize-deletions))
    (t
     (message-dialog (format nil "Heuristic 3 does not apply.~%~%There are no extraneous literals or~%clauses which decrease the accuracy~%of the goal concept.")
                     :position :centered
                     :size #@(350 115))))))

;;;________________________________________________________________________________
;;; mark-and-select-deleted-and-unoperationalized-nodes

(defun mark-and-select-deleted-and-unoperationalized-nodes (node &optional (applies? nil))
  (setf (node-selected? node) nil)
  (if (or (node-deleted? node) 
          (eql (node-state node) :unoperationalized))
    (setf applies? t
          (node-deleted? node) t
          (node-selected? node) t))
  (dolist (children (node-children node) applies?)
    (dolist (child children)
      (setf applies? (mark-and-select-deleted-and-unoperationalized-nodes child applies?)))))


;;;________________________________________________________________________________
;;; traverse-a-derivation-path
;;;
;;;  This code is specially ineffecient  (but hey it works)

(defun traverse-a-derivation-path (graph-view clause root state function)
  (let (derivation
        parent
        context
        clause-struct
        actual-parameters
        actual-clause)
    (do ((literal clause (literal-next literal)))
        ((null literal))
      (setf parent root
            context (list parent)
            derivation (literal-derivation literal))
      (when (eql (derivation-type derivation) state)
        (funcall function parent)
        (dolist (step (butlast (derivation-path derivation)))
          (setf clause-struct (car step)
                parent (find-if #'(lambda (node) (equal (node-literal node) (cdr step))) context)
                actual-parameters (rest (node-literal parent))
                actual-clause (substitute1 (clause-body clause-struct)
                                           (unify-list (clause-parameters clause-struct) 
                                                       actual-parameters))
                context (find-node-conjunction-corresponding-to-clause graph-view
                                                                       actual-clause
                                                                       (node-children parent)))

          (dolist (node context)
            (when (eq (node-kind node) :not)
              (setf context (append context (car (node-children node))))))

          (dolist (node context)
            (funcall function node)))

          (when (literal-deleted? literal)                                    ;;;  This is a hack
            (let ((deleted-literal (car (last (derivation-path derivation)))))
              (dolist (node context)
                (when (literals-correspond (node-literal node) deleted-literal)
                  (setf (node-deleted? node) t)))))))))

;;;________________________________________________________________________________
;;; operationalize-ebl-path

(defun operationalize-ebl-path (graph-view clause)
  (setf (last-node-selected graph-view) nil)
  (unoperationalize-and-deselect (graph-root graph-view))
  (traverse-a-derivation-path graph-view
                              clause 
                              (graph-root graph-view)
                              :ebl
                              #'(lambda (node) (setf (node-state node) :ebl))))

;;;________________________________________________________________________________
;;; unoperationalize-and-deselect

(defun unoperationalize-and-deselect (node)
  (setf (node-selected? node) nil)
  (if (or (eql (node-state node) :ebl)
          (eql (node-state node) :normal))
    (setf (node-state node) :unoperationalized))
  (dolist (children (node-children node))
    (dolist (child children)
      (unoperationalize-and-deselect child))))

;;;________________________________________________________________________________
;;; hi-lite-operationalize-path-as-intensional 

(defun hi-lite-operationalize-path-as-intensional (graph-view subtree clause)
  (traverse-a-derivation-path graph-view
                              clause 
                              subtree
                              :intensional
                              #'(lambda (node) (setf (node-state node) :intensional))))

;;;________________________________________________________________________________
;;; operationalize-all-ebl-paths

(defun operationalize-all-ebl-paths (graph-view Learned-Concepted-Description)
  (dolist (clause Learned-Concepted-Description)
    (let (derivation
          parent
          context
          clause-struct
          actual-parameters
          actual-clause)
      (do ((literal clause (literal-next literal)))
          ((null literal))
        (setf parent (graph-root graph-view)
              context (list parent)
              derivation (literal-derivation literal))
        (when (eql (derivation-type derivation) :ebl)
          (setf (node-state parent) :ebl)
          (dolist (step (butlast (derivation-path derivation)))
            (setf clause-struct (car step)
                  parent (find-if #'(lambda (node) (equal (node-literal node) (cdr step))) context)
                  actual-parameters (rest (node-literal parent))
                  actual-clause (substitute1 (clause-body clause-struct)
                                             (unify-list (clause-parameters clause-struct) 
                                                         actual-parameters))
                  context (find-node-conjunction-whose-ebl-component-corresponds-to-clause
                           graph-view
                           actual-clause
                           (node-children parent)))

            (dolist (node context)
              (when (eq (node-kind node) :not)
                (setf context (append context (car (node-children node))))))

            (dolist (node context)
              (setf (node-state node) :ebl)))
          
          (when (literal-deleted? literal)                                    ;;;  This is a hack
            (let ((deleted-literal (car (last (derivation-path derivation)))))
              (dolist (node context)
                (when (literals-correspond (node-literal node) deleted-literal)
                  (setf (node-deleted? node) t))))))))))

;;;____________________________________________________________________________________
;;;  find-node-conjunction-whose-ebl-component-corresponds-to-clause

(defmethod find-node-conjunction-whose-ebl-component-corresponds-to-clause ((view graph-view) prolog-clause frontier)
  (let ((corresponding-conjunction nil)
        (clause (substitute-vars prolog-clause (substitution view))))
    (do* ((rest frontier (cdr rest))
          (conjunction (remove-if #'(lambda (node) (or (eql (node-state node) :extensional)
                                                       (eql (node-state node) :builtin)
                                                       (eql (node-state node) :intensional)
                                                       (eql (node-state node) :cliche)
                                                       (eql (node-state node) :determinate))) (car rest))
                       (remove-if #'(lambda (node) (or (eql (node-state node) :extensional)
                                                       (eql (node-state node) :builtin)
                                                       (eql (node-state node) :intensional)
                                                       (eql (node-state node) :cliche)
                                                       (eql (node-state node) :determinate))) (car rest))))
         ((or corresponding-conjunction (null rest)) nil)
      (when (conjunction-corresponds-to-clause-p conjunction clause)
        (setf corresponding-conjunction conjunction)))
    corresponding-conjunction))


;;___________________________________________________________________________
;; gen-rels-name
;; this function is designed to create a name for a newly created rule.
 
(defun gen-rel-name (node)
  (let ((name ""))
    (dolist (children (butlast (node-children node)))
      (dolist (child (butlast children))
        (setf name (format nil "~A~A-AND-" name (node-n child))))
      (setf name (format nil "~A~A" name (node-n (car (last children)))))
      (setf name (format nil "~A-OR-" name)))

    (let ((children (car (last (node-children node)))))
      (dolist (child (butlast children))
        (setf name (format nil "~A~A-AND-" name (node-n child))))
      (setf name (format nil "~A~A" name (node-n (car (last children)))))
    name)))

(defun node-n (node)
  (if (eql (node-kind node) :NOT)
    (format nil "NOT-~A" (gen-rel-name node))
    (car (node-literal node))))
  

;;___________________________________________________________________________
;; gen-rel-vars
;; this function is designed to create an agrument list for an newly
;; created rule.  Currently all the variables in the children are added
;; to the parent.  This should be modified to check mode ...
 
(defun node-vars (node)
  (if (eql (node-kind node) :NOT)
    (let ((vars nil))
      (dolist (children (node-children node))
        (dolist (child children)
          (dolist (v (node-vars child))
            (setf vars (nconc vars (list v))))))
      vars)
    (remove-duplicates (cdr (node-literal node)))))

(defun gen-rel-vars (node)
  (let ((vars nil))
    (dolist (children (node-children node))
      (dolist (child children)
        (dolist (v (node-vars child))
          (setf vars (nconc vars (list v))))))
    (sort vars #'string<)))


;;;_______________________________________________________________________________
;;;  graph-induced-portion-of-clauses
;;;
;;;  given a linked list of literal stuctures this fuction returns a graph
;;;  of the induced literals.  The top-level node and conjunction are
;;;  hidden (maybe they should be made explicit and given new selection
;;;  semantics.

(defun graph-induced-portion-of-clauses (graph-view clauses)
  (setf (graph-root graph-view) (create-node graph-view
                                             :literal '(top)
                                             :hidden? t)
        
        (node-kind (graph-root graph-view)) :intensional)
  (size-node graph-view (graph-root graph-view))

  (let ((derivation nil)
        (conjunction nil)
        (disjunction nil)
        (next-literal nil)
        (new-node nil))
    (dolist (clause clauses)
      (setf conjunction nil)
       (do ((literal clause (literal-next literal)))
          ((null literal) nil)
        (setf derivation (literal-derivation literal))
        (case (derivation-type derivation)
          (:ebl nil)
          ((:extensional :builtin :determinate :cliche)
           (setf next-literal (literal-next literal)
                 (literal-next literal) nil
                 new-node (connect-literal graph-view 
                                           (graph-root graph-view)
                                           (car (convert-to-prolog literal))
                                           -1
                                           t)
                 conjunction (nconc conjunction (list new-node))
                 (literal-next literal) next-literal)
           (hi-lite-subtree new-node (derivation-type derivation)))

          (:intensional
           (let ((prolog-literal (cdar (derivation-path derivation)))
                 (subtree nil))
             (dolist (node conjunction)
               (if (eql prolog-literal (node-literal node))
                 (setf subtree node)))
             (unless subtree
               (setf subtree (connect-literal graph-view
                                              (graph-root graph-view)
                                              prolog-literal
                                              -1
                                              t)
                     conjunction (nconc conjunction (list subtree)))
               (hi-lite-subtree subtree :unoperationalized))
             (hi-lite-operationalize-path-as-intensional graph-view subtree clause)))
          ))
      (setf disjunction (nconc disjunction (list conjunction))))
    (setf (node-children (graph-root graph-view)) disjunction))

  (when (cdr clauses)
    (let ((name (gen-rel-name (graph-root graph-view)))
          (vars (gen-rel-vars (graph-root graph-view))))
      (setf (node-hidden? (graph-root graph-view)) nil
            (node-literal (graph-root graph-view)) (cons name vars)
            (node-text (graph-root graph-view)) (return-literal-string name vars))
    (size-node graph-view (graph-root graph-view))))

  (position-nodes graph-view)
  (resize-window (graph-window graph-view))
  (position-graph graph-view :centered t)
  (force-graph-redraw graph-view))


;;;________________________________________________________________________________
;;;  KR applies Heuristic 1 2 3

(defun KR (Learned-Concept-Description TOP-LEVEL-LITERAL)

  (format t "~%~%Applying Knowledge Revision Heuristics~%")
  (setf *error-output* *terminal-io*)
  (window-hide *top-listener*)

  (setf *KR-APPLICABLE* nil)
  (when (window-p *INDUCED-WINDOW*)
    (window-close *INDUCED-WINDOW*))
  (setf *INDUCED-WINDOW* (make-instance 'graph-window
                                        :window-show nil
                                        :close-box-p nil
                                        :window-type :document
                                        :view-font '("Monaco" 9 :plain)
                                        :edit-kind nil
                                        :window-do-first-click t))
  (let ((induced-view (view-named :graph-view (view-named :graph-scroller *INDUCED-WINDOW*))))
    (setf (graph-orientation induced-view) :horizontal
          (node-selection-constraint induced-view) :no-drag
          (graph-expand induced-view) :always)

    (when (window-p *THEORY-WINDOW*)
      (window-close *THEORY-WINDOW*))
    (setf *THEORY-WINDOW* (make-instance 'graph-window
                                         :view-position #@(2 41)
                                         :window-type :document-with-zoom
                                         :window-show nil
                                         :close-box-p nil
                                         :window-title "Domain Theory"
                                         :view-font '("Monaco" 9 :plain)
                                         :window-do-first-click t))
    (let ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*))))
      (setf (graph-orientation theory-view) :horizontal
            (node-selection-constraint theory-view) :no-drag
            (graph-expand theory-view) :always)
      
      (generate-graph theory-view TOP-LEVEL-LITERAL)
      
      (KR-Heuristic-1 Learned-Concept-Description)
      
      (setf (last-node-selected induced-view) nil
            (last-conjunction-selected induced-view) nil
            (last-node-selected theory-view) nil
            (last-conjunction-selected theory-view) nil)
      (unoperationalize-and-deselect (graph-root theory-view))
      (operationalize-all-ebl-paths theory-view Learned-Concept-Description)

      (KR-Heuristic-2 Learned-Concept-Description)
      
      (window-close *INDUCED-WINDOW*)

      (setf (last-node-selected theory-view) nil
            (last-conjunction-selected theory-view) nil)
      (deselect-all theory-view)
      
      (KR-Heuristic-3)
      (window-close *THEORY-WINDOW*)

      (setf *THEORY-WINDOW* (make-instance 'graph-window
                                           :view-position #@(2 41)
                                           :window-type :document-with-zoom
                                           :window-show nil
                                           :window-title "Revised Domain Theory"
                                           :view-font '("Monaco" 9 :plain)
                                           :window-do-first-click t)
            theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*))
            (graph-orientation theory-view) :horizontal
            (graph-expand theory-view) :always)
      (generate-graph theory-view TOP-LEVEL-LITERAL)
      (resize-window *THEORY-WINDOW*)
      (position-graph theory-view :centered t)
      (window-select *THEORY-WINDOW*)

      (setf *error-output* CCL::*pop-up-terminal-io*)
      (window-select *top-listener*)
      (format t "~%~%Domain Thoery Revised~%")

      (values))))


;;;______________________________________________________________________________
;;;______________________________________________________________________________
;;;  get-induced-literal-attachment (prolog-root-literal clause) 
;;;
;;;  For each clause in learned-clauses get the correct attachment point along the
;;;  operationalized path for the literals learned using induction.
;;;
;;;  Display entire path learned by :EBL (this will be a connected graph [probably an and-tree])
;;;  Display the literals learned by :INDUCTION (these will be isolated nodes)
;;;  Display the literals learned by :CONSTRUCTIVE-INDUCTION (these will be disjoint connected 
;;;                                                            graphs [probably and-trees])
;;;
;;;  Allow the user to draw and connectors from any :EBL node to any induced 
;;;  literal and vice-versa.  Allow the user to stop at any point, but notify
;;;  the user that correct classification is not ensured if a fronteer of the
;;;  indeuced literals have not been connected.
;;;
;;;  DO THIS -->  show the default attachment point and selection

(defun get-induced-literal-attachment (clauses)
  (let ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*)))
        (induced-view (view-named :graph-view (view-named :graph-scroller  *INDUCED-WINDOW*))))

    (unoperationalize-and-deselect (graph-root theory-view))
    (operationalize-ebl-path theory-view (car clauses))
    (resize-window *THEORY-WINDOW*)
    (position-graph theory-view :centered t)
    (set-view-position *THEORY-WINDOW* (round (- *screen-width* (point-h (view-size *THEORY-WINDOW*))) 2) 42)
    (force-graph-redraw theory-view)
    (window-select *THEORY-WINDOW*)
    
    (set-window-title *INDUCED-WINDOW* "Induced Literals")
    (graph-induced-portion-of-clauses induced-view clauses)
    (set-view-position *INDUCED-WINDOW* '(:bottom 201))
    (window-select *INDUCED-WINDOW*)

    (create-attach-induced-literal-dialog (clauses-string clauses :induction? t) 
                                          (clause-string (car clauses) :ebl? t))
    (do ()
        ((done *KR-DIALOG*)
         (window-hide *INDUCED-WINDOW*)
         (values))
      (event-dispatch)
      (if (and (last-conjunction-selected theory-view)
               (some-node-selected (graph-root theory-view)))
        (let ((attach (view-named :attach *KR-DIALOG*)))
          (set-default-button *KR-DIALOG* attach)
          (dialog-item-enable attach))
        (let ((attach (view-named :attach *KR-DIALOG*)))
          (set-default-button *KR-DIALOG* nil)
          (dialog-item-disable attach))))))

;;;_______________________________________________________________________________
;;;  do-literal-attachment

(defun do-literal-attachment ()
  (let* ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*)))
         (induced-view (view-named :graph-view (view-named :graph-scroller *INDUCED-WINDOW*)))
         (conjunction (last-conjunction-selected theory-view))
         (head-node (node-parent (car conjunction)))
         (added-nodes (return-selected-nodes (graph-root induced-view)))
         (copy-of-added-nodes nil)
         (duplicate nil))
    (when (and head-node added-nodes)
      (dolist (node added-nodes)
        (setf duplicate (external-duplicate-tree node)
              (node-parent duplicate) head-node
              copy-of-added-nodes (push duplicate copy-of-added-nodes)))
      (nconc conjunction (nreverse copy-of-added-nodes))
      (setf *attachment-notes* (nconc *attachment-notes* (list (list head-node conjunction))))
      (position-nodes theory-view)
      (grow-window-if-needed *THEORY-WINDOW*)
      (force-graph-redraw theory-view)

      (window-select *INDUCED-WINDOW*))))


;;;_______________________________________________________________________________
;;;  internalize-literal-attachment

(defun internalize-literal-attachment (attachment-notes)
  (dolist (attachment attachment-notes)
    (let* ((head-node (first attachment))
           (body-nodes (second attachment))
           (clause-number (position body-nodes (node-children head-node)))
           (prolog-head (node-literal head-node))
           (old-prolog-clause (nconc (list prolog-head)
                                     (mapcan #'(lambda (node) 
                                                 (if (eql (node-state node) :ebl)
                                                   (list (node-literal node))))
                                             body-nodes)))
           (new-prolog-clause (nconc (list prolog-head)
                                     (mapcar #'node-literal body-nodes)))
           

           (rule-name (first prolog-head))
           (rule (get rule-name 'rule))

           (original-clauses (get rule-name 'clauses))
           (originial-clause (elt original-clauses clause-number))
           (sub-from-original-to-graph (unify originial-clause old-prolog-clause))
           (invert-sub (mapcar #'(lambda (sub) (if (consp sub)
                                                 (list (second sub)(first sub))
                                                 sub))
                               sub-from-original-to-graph))
           (new-clause (substitute-vars new-prolog-clause invert-sub)))


      (format t "~%Clause Number :    ~A" clause-number)
      (format t "~%originial-clause : ~A" originial-clause)
      (format t "~%new-clause :       ~A" new-clause)



      (setf (elt original-clauses clause-number) new-clause)
      (re-def-rule rule :clauses original-clauses))))


;;;________________________________________________________________________________
;;;  get-induced-clause-attachment (learned-clauses unoperationalized-clauses)
;;;
;;;  For each clause in learned-clauses get the correct attachment point for the 
;;;  clause. Search all unoperationalized-clause for a clause which is either
;;;  more general or more specific than the learned-clause. Suggest replacing the
;;;  unoperationalized-clause with the learned clause. If no such suggestion is 
;;;  accepted (or exists) suggest attaching the learned-clause as a disjunct for
;;;  the target-concept or for a rule that is part of an unoperationalized-clause.
;;;
;;;  Yah, right!

(defun get-induced-clause-attachment (clause)
  (let ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*)))
        (induced-view (view-named :graph-view (view-named :graph-scroller *INDUCED-WINDOW*))))
    (set-view-position *THEORY-WINDOW* (round (- *screen-width* (point-h (view-size *THEORY-WINDOW*))) 2) 42)
    (force-graph-redraw theory-view)
    (window-select *THEORY-WINDOW*)
        
    (set-window-title *INDUCED-WINDOW* "Induced Clause")
    (graph-induced-portion-of-clauses induced-view (list clause))
    (set-view-position *INDUCED-WINDOW* '(:bottom 201) )
    (window-select *INDUCED-WINDOW*)
    
    (create-attach-induced-clause-dialog (clause-string  clause :both? t))
    (do ()
        ((done *KR-DIALOG*)
         (window-hide *INDUCED-WINDOW*)
         (values))
      (event-dispatch)
      (if (and (last-node-selected theory-view)
               (or (eql (node-kind (last-node-selected theory-view)) :intensional)
                   (eql (node-kind (last-node-selected theory-view)) :unexpanded))
               (some-node-selected (graph-root induced-view)))
        (let ((attach (view-named :attach *KR-DIALOG*)))
          (set-default-button *KR-DIALOG* attach)
          (dialog-item-enable attach))
        (let ((attach (view-named :attach *KR-DIALOG*)))
          (set-default-button *KR-DIALOG* nil)
          (dialog-item-disable attach))))))


;;;________________________________________________________________________________
;;;  do-clause-attachment

(defun do-clause-attachment ()
  (let* ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*)))
         (induced-view (view-named :graph-view (view-named :graph-scroller *INDUCED-WINDOW*)))
         (head-node (last-node-selected theory-view))
         (clause-nodes (return-selected-nodes (graph-root induced-view)))
         (copy-of-clause-nodes nil)
         (duplicate nil)
         (new-v 0))
    (when (and head-node clause-nodes)
      (dolist (node clause-nodes)
        (setf duplicate (external-duplicate-tree node)
              (node-parent duplicate) head-node
              copy-of-clause-nodes (push duplicate copy-of-clause-nodes)))
      (setf (node-children head-node) (nconc (node-children head-node)
                                             (list (nreverse copy-of-clause-nodes))))
      (setf *attachment-notes* (nconc *attachment-notes* (list (list head-node copy-of-clause-nodes))))
      (position-nodes theory-view)
      (grow-window-if-needed *THEORY-WINDOW*)
      (force-graph-redraw theory-view)
      (setf new-v (+ *window-window-offset* 
                     (point-v (view-position *THEORY-WINDOW*))
                     (point-v (view-size *THEORY-WINDOW*))))
      (set-view-position *INDUCED-WINDOW* 
                         (point-h (view-position *INDUCED-WINDOW*)) 
                         (min new-v (- *screen-height* (point-v (view-size *INDUCED-WINDOW*)) 20))))))

;;;_______________________________________________________________________________
;;;  internalize-clause-attachment

(defun internalize-clause-attachment (attachment-notes)
  (dolist (attachment attachment-notes)
    (let* ((head-from-graph (node-literal (first attachment)))
           (rule-name (car head-from-graph))
           (rule (get rule-name 'rule))
           (node-conjunction-from-graph (second attachment))
           (new-clause-from-graph
            (nconc (list head-from-graph)
                   (mapcar #'node-literal node-conjunction-from-graph))))

      (setf (get rule-name 'clauses) (nconc (get rule-name 'clauses) (list new-clause-from-graph)))
      (re-def-rule rule :clauses (get rule-name 'clauses)))))

;;;________________________________________________________________________________
;;;  get-deletions 

(defun get-deletions ()
  (let ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*))))
    (resize-window *THEORY-WINDOW*)
    (position-graph theory-view :centered t)
    (set-view-position *THEORY-WINDOW*
                       (round (- *screen-width* (point-h (view-size *THEORY-WINDOW*))) 2) 42)
    (force-graph-redraw theory-view)
    (window-select *THEORY-WINDOW*)
      
    (create-delete-dialog)
    (let ((delete (view-named :delete *KR-DIALOG*))
          (un-delete (view-named :un-delete *KR-DIALOG*)))
    (do ()
        ((done *KR-DIALOG*)
         (values))
      (event-dispatch)
      (if (some-deleted-node-selected (graph-root theory-view))
        (dialog-item-enable un-delete)
        (dialog-item-disable un-delete))
      (if (some-undeleted-node-selected (graph-root theory-view))
        (dialog-item-enable delete)
        (dialog-item-disable delete))))))

;;;_______________________________________________________________________________
;;;  internalize-deletions
;;;
;;;  This relies on the literal in the clause and in the graph being in the same order!

(defun internalize-deletions ()
  (let* ((theory-view (view-named :graph-view (view-named :graph-scroller *THEORY-WINDOW*)))
         (nodes-needing-revision (delete-duplicates (mapcar #'node-parent 
                                                            (return-deleted-nodes 
                                                             (graph-root theory-view))))))
    (dolist (node nodes-needing-revision)
      (let* ((rule-name (car (node-literal node)))
             (clauses (get rule-name 'clauses))
             (rest clauses))
        (dolist (conjunction (node-children node))
          (cond ((every #'node-deleted? conjunction)
                 (rplaca rest nil))
                ((some #'node-deleted? conjunction)
                 (let ((body (cdar rest)))
                   (dolist (child conjunction)
                     (if (node-deleted? child)
                       (rplaca body nil))
                     (setf body (cdr body)))
                   (rplaca rest (delete nil (car rest)))))
                (t nil))
          (setf rest (cdr rest)))
        (setf clauses (delete nil clauses))
        (re-def-rule (get rule-name 'rule) :clauses clauses)))))

;;;__________________________________________________________________________________
;;; apply-heuristics

(defun apply-heuristics ()
  (when (and *learned-concept-description* *goal-concept*)
    (catch-cancel 
     (KR *learned-concept-description* *goal-concept*))))


;;;__________________________________________________________________________________
;;;__________________________________________________________________________________
;;; Needed by KR but not supplied when graphic editor and ES not loaded


(defun re-def-rule (r &key (name (rule-name r)) 
                      (clauses nil) 
                      (type (rule-type r))
                      (constraint (rule-constraint r))
                      (mode (rule-mode r))
                      (commutative (rule-commutative r))
                      (induction (rule-induction r))
                      (vars nil)
                      (questions nil))
  (if (null clauses)
    (setf clauses (if (null vars)
                    (get name 'clauses)
                    (mapcar #'(lambda(c) (cons (cons name vars) 
                                               (nsublis (mapcar #'cons (cdr(car c)) vars)
                                                        (cdr c)
                                                        :test #'equalp)))
                            (get name 'clauses)))))
  (if (null questions)
    (setq questions (if (null vars)
                      (rule-questions r)
                      (nsublis (mapcar #'cons (rule-vars r) vars)
                               (rule-questions r)
                               :test #'equalp))))
  (if (null vars)
    (setq vars (rule-vars r)))
                      
  
  (eval `(def-rule ,name 
           :clauses ,clauses
           :type ,type
           :constraint ,constraint
           :mode ,mode
           :commutative ,commutative
           :induction ,induction
           :vars ,vars
           :questions ,questions))
  (setq *rules-changed* t)
;  (show-rule-def name)
  name)




;;;_______________________________________________________________________________
;;;  Delete (d D delete)  is   delete-selected-nodes
;;;  Undelete (clear)     is   undelete-selected-nodes
;;;  Really-Delete (control-delete) is  remove-all-deleted-nodes
;;;  Undelete-all (control-clear)  is  undelete-all-deleted-nodes
;;;_______________________________________________________________________________

;;____________________________________________________________________________________
;; remove-child-from-parent

(defmethod remove-child-from-parent ((view graph-view) node)
  (let ((parent (node-parent node)))
    (when parent
      (do* ((conjunctions (node-children parent) (cdr conjunctions))
            (conjunction (car conjunctions) (car conjunctions)))
           ((member node conjunction)
            (rplaca conjunctions (delete node conjunction))))
      (setf (node-children parent) (delete nil (node-children parent)))
      (if (null (node-children parent))
        (setf (node-kind parent) :undefined)))))

;;;_______________________________________________________________________________
;;; delete-selected-nodes

(defmethod delete-selected-nodes ((view graph-view))
  (if (node-selected? (graph-root view))
    (case (catch-cancel 
           (y-or-n-dialog (format nil "~%Do you want to delete the rule ~A from the rule base?"
                                  (car (node-literal (graph-root view))))))
      ((nil) nil)
      (:cancel nil)
      (t (format t "~%The rule ~A should be deleted." (car (node-literal (graph-root view))))
         (window-close (graph-window view)))))
    (mark-selected-nodes-and-hide-descendents (graph-root view))
    (position-nodes view)
    (force-graph-redraw view))


;;;_______________________________________________________________________________
;;; mark-selected-nodes-and-hide-descendents

(defun mark-selected-nodes-and-hide-descendents (node)
  (cond ((node-selected? node)
         (setf (node-deleted? node) t)
         (hide-all-descendents node))
        (t (dolist (children (node-children node))
             (dolist (child children)
               (mark-selected-nodes-and-hide-descendents child))))))
     

;;;_______________________________________________________________________________
;;; undelete-selected-nodes

(defmethod undelete-selected-nodes ((view graph-view) &optional (node (graph-root view)))
  (when (node-selected? node)
    (setf (node-deleted? node) nil)
    (rlet ((rect :rect
                 :left (node-left node)
                 :top (node-top node)
                 :right (node-right node)
                 :bottom (node-bottom node)))
      (with-focused-view view
        (_EraseRect :ptr rect)
        (draw-node view node))))
  (dolist (children (node-children node))
    (dolist (child children)
      (undelete-selected-nodes view child))))

;;;_______________________________________________________________________________
;;; undelete-all-deleted-nodes

(defmethod undelete-all-deleted-nodes ((view graph-view))
  (undelete-deleted-nodes (graph-root view))
  (force-graph-redraw view))

(defun undelete-deleted-nodes (node)
  (if (node-deleted? node)
    (setf (node-deleted? node) nil))
  (dolist (children (node-children node))
    (dolist (child children)
      (undelete-deleted-nodes child))))





(provide :KR-FOCL)
