
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)
;;judge the accuracy of the definition of illegal
(defun judge(n &aux p a i)
  (cond ((= n 0) 0)
	(t (setq p (generate-position))
	   (setq a (prove-function?  #'learned-concept/6 nil nil p))
	   (setq i (illegal-position p))
	   (+ (cond ((or (and i a)(and (not i)(not a))) 1)(t 0)) 
	      (judge (- n 1))))))

;;______________________________________________________________________________
;; judge-examples (examples)
;;    judge the accuracy of the definition of illegal on a given set of examples

(defun judge-examples (examples &optional (rule 'illegal)
				&aux actual-class
				theory-class
				(correct 0)
				(neg-classified-pos 0)
				(pos-classified-neg 0)
                                (rule-fun (rule-prolog-function (get rule 'rule))))

  (dolist (example examples correct)
	  (setq actual-class (illegal-position example))
	  (setq theory-class (prove-function?  rule-fun nil nil example))
	  (cond ((and actual-class theory-class) (incf correct))
		((and (not actual-class) (not theory-class)) (incf correct))
		((and actual-class (not theory-class)) (incf pos-classified-neg))
		((and (not actual-class) theory-class) (incf neg-classified-pos))))
  (/ correct (length examples)))


;;______________________________________________________________________________
;; generate-example-set (n pp)
;;     generates n examples, pp percent of which are positive.

(defun generate-example-set (n pp &aux (examples nil))
  (dotimes (i  n examples)
    (if (< (random 100) pp)              
        (push (return-positive-example) examples)
        (push (return-negative-example) examples))))


;;______________________________________________________________________________
;; return-negative-example
;;     returns a negative example

(defun return-negative-example ()
  (do ((example (generate-position))) 
      ((not(illegal-position example)) example)
    (setq example (generate-position))))    
    


;;______________________________________________________________________________
;; return-positive-example
;;     returns a positive example

(defun return-positive-example ()
  (do ((example (generate-position)))
      ((illegal-position example) example)
    (setq example (generate-position))))     

;;generate a random board position
(defun generate-position (&aux (result nil))
  (dotimes (i 6 result) (push (+ 1 (random 8)) result)))

(defun adjacent (i j k l)
  (or ( and (eq i k) (near j l)) 
       (and (eq j l) (near i k))
       (and (near i k) (near j l))))


(defun near (i j)
  (or (eq i (+ 1 j)) (eq i (- j 1))))

(defun between (a b c d e f)
   (or (and (= a c e)(or (< d b f)(> d b f)))
     (and (= b d f)(or (< c a e) (> c a e)))))


(defun same-line (i j k l)
  (or (eq i k) (eq j l)))

;;returns T if position is illegal
(defun illegal-position (L)
   (cond ((and (eq (first L) (third L))(eq (second L) (fourth L)) ) t)
         ((and (eq (first L) (fifth L))(eq (second L)(sixth L))) t)
         ((and (eq (third L) (fifth L))(eq (fourth L) (sixth L)))t) ;; same position
         ((adjacent (first L) (second L) (fifth L) (sixth L)) t) ;; king capture king
         ((and(same-line (third L) (fourth L) (fifth L) (sixth L))  ;; rook capture
           (not (between (first L) (second L) (third L) (fourth L)
                  (fifth L) (sixth L))) t))))

;generates definition of illegal with N examples
;;if type is true, uses :column row, else :column

(defun illegal-generator (n                      ; number of examples to generate

                          &optional 
			  (pp  50)                   ; percent positive
                             (noise 0)           ; percent error rate
                             (type? nil)         ; if true uses :column :row, else :column
                             (print-training-set? nil)
                          &aux
                             (p-noise noise)     ; rate of mis-classification of positive examples
                             (n-noise noise)     ; rate of mis-classification of negative examples
                             (in nil)            ; list of positive examples (containing noise)
                             (out nil))          ; list of negative examples (containing noise)
  (dotimes (i  n)
    (if (< (random 100) pp)                    ; Choose class of example (pos or neg)
      (if (< (random 100) p-noise)             ; Introduce noise into positive examples
        (push (return-negative-example) in)
        (push (return-positive-example) in))
      (if (< (random 100) n-noise)             ; Introduce noise into negative examples
        (push (return-positive-example) out)
        (push (return-negative-example) out))))

  (if print-training-set?
    (progn
      (format t "~%~%Training Set")
      (format t "~%Pos: ~a" in)
      (format t "~%Neg: ~a" out)))

  (if type? 
    (eval `(def-pred illegal 
             :induction nil 
             :type (:row :column :row :column :row :column)
             :pos ,in 
             :neg ,out))
    (eval `(def-pred illegal 
             :induction nil
             :pos ,in 
             :neg ,out))))


;;judge accuracy of original theory (of illegal-board
(defun judge-original(n &aux p a i)
  (cond ((= n 0) 0)
	(t (setq p (generate-position))
	   (setq a (prove-function?  #'illegal-board/6 nil nil p))
	   (setq i (illegal-position p))
	   (+ (cond ((or (and i a)(and (not i)(not a))) 1)(t 0)) 
	      (judge-original (- n 1))))))

;;converst result of focl to executable prolog rule
(defun make-prolog-rule(pred arity clauses
		      &aux (old-vars (do ( (i (- arity 1) (decf i))
					   (result nil))
					 ((< i 0) result)
					 (push (make-pcvar :id i) result) ))
		      prolog-def)
  (setq prolog-def
	(mapcar #'(lambda(x)(cons (cons pred old-vars)
				  (convert-to-prolog x)))
		clauses))
  (setf (get pred  'pred) nil)
  (eval `(def-rule ,pred :clauses ,prolog-def)))
  
 
;;learns illegal and tests on 100 examples
(defun run-illegal-test(n)
  (illegal-generator n  50 0 t)
  (make-prolog-rule 'learned-concept 6 (focl 'illegal))
  (judge 100))


(defun test-mutatator-v (&optional (mutations '(0 1 2 4 6 8 10 12 14 16 20 24)) &aux
				(foil-acc nil) foil-tuples domain-acc
				saved-examples
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq saved-examples (generate-example-set 1000 36))
  (load-source-test-file "typed-641")
  (test-illegal)
  (setq foil-tuples *variablizations-checked*)
  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (mutate-theory))
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (test-illegal-ebl)
	  (setq focl-tuples *variablizations-checked*)
	  (format t "~%~a~a~a~a~4F~a~4F"
		  m #\tab 0 #\tab foil-tuples #\tab domain-acc)
	  (format t "~%~a~a~a~a~4F~a~4f"
		  m #\tab 1 #\tab focl-tuples #\tab domain-acc)))

(defun test-mutatator (&optional (examples 80) (mutations '(0 1 2 4 6 8 10 12 14 16 20 24)) &aux
				foil-acc foil-tuples domain-acc
				saved-examples
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq  *conditions-for-extensional-threshold-conjunctions*
	 #'(lambda(&rest x) x nil))
  (setq saved-examples (generate-example-set 1000 50))
  (load-source-test-file "untyped-between-equal-near")
  (illegal-generator examples 50 0 nil)
  
  (load-source-test-file "untyped-chess-domain")
  (format t ";Mutations DT?  tuples learned-accuracy original-accuracy~%")
  (make-prolog-rule 'learned-concept 6 (print(focl 'illegal :save-examples nil
					     :constructive-induction nil)))



  (setq foil-acc (judge-examples saved-examples 'learned-concept))
  (setq foil-tuples *variablizations-checked*)

  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (mutate-theory))
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (make-prolog-rule 'learned-concept 6 (print(test-illegal-ebl)))
	  (setq focl-acc (judge-examples saved-examples 'learned-concept))
	  (setq focl-tuples *variablizations-checked*)
	  (format t "~%~a~a~a~a~4F~a~4f~a~4f~%"
		  m #\tab 0 #\tab foil-tuples #\tab foil-acc #\tab domain-acc)
	  (format t "~a~a~a~a~4F~a~4f~a~4f~%"
		  m #\tab 1 #\tab focl-tuples #\tab focl-acc #\tab domain-acc)))

(defun mutate-theory (&optional 
				&aux choice)
  
  (setq choice (random 4))		; 1..4
  (format t "~&;make-n-random-mutations: selecting choice ~a~%" choice)
  (cond ((= choice 1) (delete-random-term))
	((= choice 2) (delete-random-clause ))
	((= choice 3) (add-random-term ))
	((= choice 0) (add-random-clause ))))


(defun jack-knife(concept &rest keys &key the-ignored-key &allow-other-keys)
  (declare (ignore the-ignored-key))
  (let* ((s (get-pstruct concept))
         (p (pred-pos s))
         (n (pred-neg s))
	 (r nil)
         (arity (pred-arity s))
         (total 0)
         (correct 0)
         (rule-fun nil))
    (dolist (pos p)
	    (incf total)
	    (destroy-fact concept pos)
	    (setq r (apply #'focl concept keys))
	    (when r (make-prolog-rule 'learned-concept arity r)
		  (setq rule-fun (rule-prolog-function (get 'learned-concept 'rule)))
		  (when (prove-function?  rule-fun nil nil pos)
			(incf correct)))
	    (insert-new-fact concept pos))
    (dolist (neg n)
	    (setf (pred-neg s)(delete neg (pred-neg s)))
	    (setq r (apply #'focl concept keys))
	    (when r (make-prolog-rule 'learned-concept arity r)
		  (setq rule-fun (rule-prolog-function (get 'learned-concept 'rule))))
	    (incf total)
	    (unless (and r (prove-function?  rule-fun nil nil neg))
		    (incf correct))
	    (push  neg  (pred-neg s)))
    (/ CORRECT TOTAL)))


(defun test-op(&aux saved-examples domain-acc mutations)
  (reset-preds)
  (load-source-test-file  "untyped-between-equal-near")
  (dotimes (j 2)
    (format t "~%;;================ ~a ================" (+ 50 (* 25 j))) 
    (dotimes (k 12)
      (setq saved-examples (generate-example-set 1000 50))
      (load-source-test-file  "untyped-chess-domain")
      (setq mutations 0)
      (dolist (i '(1 1 2 2 3 4 5))
        (setq domain-acc (judge-examples saved-examples 'illegal-board))
        (illegal-generator (+ 50 (* 25 j)) 50 0 nil)
        (make-prolog-rule 'learned-concept 6 (test-illegal-ebl  :frontier))
        (format t "~%info  frontier~a~a~a~a~a~4F~a~4f~a~4f"
		  #\tab (+ 50 (* 25 j)) #\tab mutations #\tab *variablizations-checked*  
                  #\tab (judge-examples saved-examples 'learned-concept) #\tab domain-acc)

	        (make-prolog-rule 'learned-concept 6 (test-illegal-ebl  :frontier :ratio))
        (format t "~%ratio frontier~a~a~a~a~a~4F~a~4f~a~4f"
		  #\tab (+ 50 (* 25 j)) #\tab mutations #\tab *variablizations-checked*  
                  #\tab (judge-examples saved-examples 'learned-concept) #\tab domain-acc)
        (make-prolog-rule 'learned-concept 6 (test-illegal-ebl  :leaves))
        (format t "~%info  leaves  ~a~a~a~a~a~4F~a~4f~a~4f"
		  #\tab (+ 50 (* 25 j)) #\tab mutations #\tab *variablizations-checked*  
                  #\tab (judge-examples saved-examples 'learned-concept) #\tab domain-acc)
        (dotimes (j i)
          (incf mutations)
          (mutate-theory))))))



(defun test-mutatator-v2 (mutator &optional(mutations '(0 1 2 3 4 6 8 10 12))
				(foil-acc nil) foil-tuples domain-acc
				saved-examples
				focl-acc focl-tuples (m 0))
  (reset-preds)
  (setq saved-examples (generate-example-set 1000 36))
  (load-source-test-file "typed-641")
  (dolist (mm mutations)
	  (do ()
	      ((= m mm))
	      (incf m)
	      (funcall mutator))
	  (setq domain-acc (judge-examples saved-examples 'illegal-board))
	  (test-illegal-ebl)
	  (setq focl-tuples *variablizations-checked*)
	  (format t "~%~a~a~a~a~4F~a~4f"
		  m #\tab 1 #\tab focl-tuples #\tab domain-acc)))