
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)
(provide "loop")

(proclaim '(inline mappend mklist flatten random-elt member-equal
                   starts-with compose last1 length=1
                   rest2 rest3 symbol old-symbol reuse-cons))

(defun mappend (fn list)
  "Append the results of calling fn on each element of list.
  Like mapcon, but uses append instead of nconc."
  (apply #'append (mapcar fn list)))

(defun mklist (x) 
  "If x is a list return it, otherwise return the list of x"
  (if (listp x) x (list x)))

(defun flatten (exp)
  "Get rid of imbedded lists (to one level only)."
  (mappend #'mklist exp))

(defun random-elt (seq) 
  "Pick a random element out of a sequence."
  (elt seq (random (length seq))))

;;;

(defun member-equal (item list)
  (member item list :test #'equal))

(defun starts-with (list x)
  "Is x a list whose first element is x?"
  (and (consp list) (eql (first list) x)))


(defun keep-if (predicate sequence &rest keywords)
  "Keep only those elements of sequence that satisfy predicate."
  (apply #'remove-if-not predicate sequence keywords))


;;

(defmacro lambda (args &rest body)
  "Use (lambda args body...) instead of #'(lambda args body...)"
  `(function (lambda ,args . ,body)))

;;;

(defun compose (&rest functions)
  (lambda (x)
    (reduce #'funcall functions :from-end t :initial-value x)))

;;;


;;;; Once-only:



(defun side-effect-free? (exp)
  "Is exp a constant, variable, or function,
  or of the form (THE type x) where x is side-effect-free?"
  (or (atom exp) (constantp exp)
      (starts-with exp 'function)
      (and (starts-with exp 'the)
           (side-effect-free? (third exp)))))

;;;; Other:
(defun symbol (&rest args)
  "Concatenate symbols or strings to form an interned symbol"
  (intern (format nil "~{~a~}" args)))

(defun new-symbol (&rest args)
  "Concatenate symbols or strings to form an uninterned symbol"
  (format nil "~{~a~}" args))

(defun last1 (list)
  "Return the last element (not last cons cell) of list"
  (first (last list)))

(defun sort* (seq pred &key key) 
  "Sort without altering the sequence"
  (sort (copy-seq seq) pred :key key))

(defun reuse-cons (x y x-y)
  "Return (cons x y), or reuse x-y if it is equal to (cons x y)"
  (if (and (eql x (car x-y)) (eql y (cdr x-y)))
      x-y
      (cons x y)))

(defmacro read-time-case (first-case &rest other-cases)
  "Do the first case, where normally cases are
  specified with #+ or possibly #- marks."
  (declare (ignore other-cases))
  first-case)

;;;

(defun length=1 (x) 
  "Is x a list of length 1?"
  (and (consp x) (null (cdr x))))

(defun rest2 (list)
  "The rest of a list after the first TWO elements."
  (cddr list))

(defun rest3 (list)
  "The rest of a list after the first THREE elements."
  (cdddr list))

;;;

(defun unique-find-if-anywhere (predicate tree
                                &optional found-so-far)
  "Return a list of leaves of tree satisfying predicate,
  with duplicates removed."
  (if (atom tree)
      (if (funcall predicate tree)
          (adjoin tree found-so-far)
          found-so-far)
      (unique-find-if-anywhere
        predicate
        (first tree)
        (unique-find-if-anywhere predicate (rest tree)
                                 found-so-far))))

(defun find-if-anywhere (predicate tree)
  "Does predicate apply to any atom in the tree?"
  (if (atom tree)
      (funcall predicate tree)
      (or (find-if-anywhere predicate (first tree))
          (find-if-anywhere predicate (rest tree)))))

(defun find-anywhere (item tree)
  "Does item occur anywhere in tree?"
  (if (atom tree)
      (if (eql item tree) tree)
      (or (find-anywhere item (first tree))
          (find-anywhere item (rest tree)))))




;;;



(defun not-null (x) (not (null x)))

(defun first-or-nil (x)
  "The first element of x if it is a list; else nil."
  (if (consp x) (first x) nil))

(defun first-or-self (x)
  "The first element of x, if it is a list; else x itself."
  (if (consp x) (first x) x))

;;;

(defun permute (bag)
  "Return a random permutation of the given input list."
  (if (null bag)
      nil
      (let ((e (random-elt bag)))
        (cons e (permute (remove e bag :count 1 :test #'eq))))))


;;;; Map-into



(defstruct loop
  "A structure to hold parts of a loop as it is built."
  (vars nil) (prologue nil) (body nil) (steps nil)
  (epilogue nil) (result nil) (name nil))

(let ( #+:APPLE (*Warn-if-redefine-kernel* nil) )
  (defmacro loop (&rest exps)
    "Supports both ANSI and simple LOOP."
    (if (not (symbolp (first exps)))
      `(block nil (tagbody loop ,@exps (go loop)))
      (let ((l (make-loop)))
        (parse-loop-body l exps)
        `(let* ,(nreverse (loop-vars l))
           (block ,(loop-name l)
             ,@(nreverse (loop-prologue l))
             (tagbody
               loop
               ,@(nreverse (loop-body l))
               ,@(nreverse (loop-steps l))
               (go loop)
               end
               ,@(nreverse (loop-epilogue l))
               (return ,(loop-result l)))))))))

(defun add-body (l exp) (push exp (loop-body l)))

(defun add-test (l test)
  "Put in a test for loop termination."
  (push `(if ,test (go end)) (loop-body l)))

(defun add-var (l var init &optional (update nil update?))
  "Add a variable, maybe including an update step."
  (unless (assoc var (loop-vars l))
    (push (list var init) (loop-vars l)))
  (when update? 
    (push `(setq ,var ,update) (loop-steps l))))

(defun parse-loop-body (l exps)
  "Parse the exps based on the first exp being a keyword.
  Continue until all the exps are parsed."
  (unless (null exps)
    (parse-loop-body l (call-loop-fn l (first exps) (rest exps)))))

(defun call-loop-fn (l key exps)
  "Return the loop parsing function for this keyword"
  (if (and (symbolp key) (get key 'loop-fn))
      (funcall (get key 'loop-fn) l (first exps) (rest exps))
      (error "Unknown loop key: ~a" key)))

(defmacro defloop (key args &rest body)
  "Define a new LOOP keyword."
  ;; If the args do not have a third arg, one is supplied.
  ;; Also, we can define an alias with (defloop key other-key)
  `(setf (get ',key 'loop-fn)
         ,(cond ((and (symbolp args) (null body))
                 `(lambda (l x y)
                    (call-loop-fn l ',args (cons x y))))
                ((and (listp args) (= (length args) 2))
                 `(lambda (,@args -exps-) ,@body -exps-))
                (t `(lambda ,args ,@body)))))

;;;; 26.6. Iteration Control

(defloop repeat (l times)
  "(LOOP REPEAT n ...) does loop body n times" 
  (let ((i (gensym "REPEAT")))
    (add-test l `(<= ,i 0))
    (add-var l i times `(- ,i 1))))

(defloop as for)  ;; AS is the same as FOR

(defloop for (l var exps)
  "4 of the 7 cases for FOR are covered here:
  (LOOP FOR i FROM s TO e BY inc ...) does arithemtic iteration
  (LOOP FOR v IN l ...) iterates for each element of l
  (LOOP FOR v ON l ...) iterates for each tail of l
  (LOOP FOR v = expr [THEN step]) initializes and iterates v"
  (let ((key (first exps))
        (source (second exps))
        (rest (rest2 exps)))
    (ecase key
      ((from downfrom upfrom to downto upto by)
       (loop-for-arithmetic l var exps))
      (in (let ((v (gensym "IN")))
            (add-var l v source `(cdr ,v))
            (add-var l var `(car ,v) `(car ,v))
            (add-test l `(null ,v))
            rest))
      (on (add-var l var source `(cdr ,var))
          (add-test l `(null ,var))
          rest)
      (= (if (eq (first rest) 'then)
             (progn
               (pop rest)
               (add-var l var source (pop rest)))
             (progn
               (add-var l var nil)
               (add-body l `(setq ,var ,source))))
         rest)
      ;; ACROSS, BEING clauses omitted
      )))

(defun loop-for-arithmetic (l var exps)
  "Parse loop expressions of the form:
  (LOOP FOR var [FROM|DOWNFROM|UPFROM exp1] [TO|DOWNTO|UPTO exp2]
        [BY exp3]"
  ;; The prepositions BELOW and ABOVE are omitted
  (let ((exp1 0)
        (exp2 nil)
        (exp3 1)
        (down? nil))
    ;; Parse the keywords:
    (when (member (first exps) '(from downfrom upfrom))
      (setf exp1 (second exps)
            down? (eq (first exps) 'downfrom)
            exps (rest2 exps)))
    (when (member (first exps) '(to downto upto))
      (setf exp2 (second exps)
            down? (or down? (eq (first exps) 'downto))
            exps (rest2 exps)))
    (when (eq (first exps) 'by)
      (setf exp3 (second exps)
            exps (rest2 exps)))
    ;; Add variables and tests:
    (add-var l var exp1
             `(,(if down? '- '+) ,var ,(maybe-temp l exp3)))
    (when exp2
      (add-test l `(,(if down? '< '>) ,var ,(maybe-temp l exp2))))
    ;; and return the remaining expressions:
    exps))

(defun maybe-temp (l exp)
  "Generate a temporary variable, if needed."
  (if (constantp exp)
      exp
      (let ((temp (gensym "TEMP")))
        (add-var l temp exp)
        temp)))

;;;; 26.7. End-Test Control

(defloop until (l test) (add-test l test))

(defloop while (l test) (add-test l `(not ,test)))

(defloop always (l test)
  (setf (loop-result l) t)
  (add-body l `(if (not ,test) (return nil))))

(defloop never (l test)
  (setf (loop-result l) t)
  (add-body l `(if ,test (return nil))))

(defloop thereis (l test) (add-body l `(return-if ,test)))

(defmacro return-if (test)
  "Return TEST if it is non-nil"
  (once-only (test)
    `(if ,test (return ,test))))

(defmacro loop-finish () `(go end))

;;;

(defmacro def-cons-struct (cons car cdr &optional inline?)
  "Define aliases for cons, car and cdr."
  `(progn (proclaim '(,(if inline? 'inline 'notinline)
                      ,car ,cdr ,cons))
          (defun ,car (x) (car x))
          (defun ,cdr (x) (cdr x))
          (defsetf ,car (x) (val) `(setf (car ,x) ,val))
          (defsetf ,cdr (x) (val) `(setf (cdr ,x) ,val))
          (defun ,cons (x y) (cons x y))))

;;;



;;;

(def-cons-struct make-q q-last q-contents t)

(defun tconc (item q)
  "Insert item at the end of the queue."
  (setf (q-last q)
        (if (null (q-last q))
            (setf (q-contents q) (cons item nil))
            (setf (rest (q-last q))
                  (cons item nil)))))
(defun empty-q ()
  "Build a new queue, with no elements."
  (let ((q (make-q nil nil)))
    (setf (q-last q) q)))

(defun q-insert (item q)
  "Insert item at the end of the queue."
  (setf (q-last q)
        (setf (rest (q-last q))
              (cons item nil))))

(defun q-nconc (q list)
  "Add the elements of LIST to the end of the queue."
  (setf (q-last q)
        (last (setf (rest (q-last q)) list))))

;;;

;;;; 26.8 Value Accumulation
;;;  ges modified so we can load loop more than once

(if (not (boundp '*acc*)) (defconstant *acc* (gensym "ACC")))

;;; INTO preposition is omitted

(defloop collect (l exp)
  (add-var l *acc* '(empty-q))
  (add-body l `(q-insert ,exp ,*acc*))
  (setf (loop-result l) `(q-contents ,*acc*)))

(defloop nconc (l exp)
  (add-var l *acc* '(empty-q))
  (add-body l `(q-nconc ,*acc* ,exp))
  (setf (loop-result l) `(q-contents ,*acc*)))

(defloop append (l exp exps)
  (call-loop-fn l 'nconc `((copy-list ,exp) .,exps)))

(defloop count (l exp)
  (add-var l *acc* 0)
  (add-body l `(when ,exp (incf ,*acc*)))
  (setf (loop-result l) *acc*))

(defloop sum (l exp)
  (add-var l *acc* 0)
  (add-body l `(incf ,*acc* ,exp))
  (setf (loop-result l) *acc*))

(defloop maximize (l exp)
  (add-var l *acc* nil)
  (add-body l `(setf ,*acc*
                     (if ,*acc*
                         (max ,*acc* ,exp)
                         ,exp)))
  (setf (loop-result l) *acc*))

(defloop minimize (l exp)
  (add-var l *acc* nil)
  (add-body l `(setf ,*acc*
                     (if ,*acc*
                         (min ,*acc* ,exp)
                         ,exp)))
  (setf (loop-result l) *acc*))

(defloop collecting collect)
(defloop nconcing   nconc)
(defloop appending  append)
(defloop counting   count)
(defloop summing    sum)
(defloop maximizing maximize)
(defloop minimizing minimize)


(defvar *queue*)


(defun collect (item &optional (queue *queue*))
  (q-insert item queue))

(defmacro with-collection ((&optional (queue '*queue*))
                           &body body)
  `(let ((,queue (empty-q)))
     ,@body
     (q-contents ,queue)))

;;;

;;;; 26.9. Variable Initializations ("and" omitted)

(defloop with (l var exps)
  (let ((init nil))
    (when (eq (first exps) '=)
      (setf init (second exps)
            exps (rest2 exps)))
    (add-var l var init)
    exps))

;;;; 26.10. Conditional Execution ("and" and "end" omitted)

(defloop when (l test exps) 
  (loop-unless l `(not ,(maybe-set-it test exps)) exps))

(defloop unless (l test exps)
  (loop-unless l (maybe-set-it test exps) exps))

(defun maybe-set-it (test exps)
  "Return value, but if the variable IT appears in exps,
  then return code that sets IT to value."
  (if (find-anywhere 'it exps)
      `(setq it ,test)
      test))

(defloop if when)

(defun loop-unless (l test exps)
  (let ((label (gensym "L")))
    (add-var l 'it nil)
    ;; Emit code for the test and the THEN part
    (add-body l `(if ,test (go ,label)))
    (setf exps (call-loop-fn l (first exps) (rest exps)))
    ;; Optionally emit code for the ELSE part
    (if (eq (first exps) 'else)
        (progn
          (let ((label2 (gensym "L")))
            (add-body l `(go ,label2))
            (add-body l label)
            (setf exps (call-loop-fn l (second exps) (rest2 exps)))
            (add-body l label2)))
        (add-body l label)))
    exps)
;;; 26.11. Unconditional Execution

(defloop do (l exp exps)
  (add-body l exp)
  (loop (if (symbolp (first exps)) (RETURN exps))
        (add-body l (pop exps))))

(defloop return (l exp) (add-body l `(return ,exp)))

;;;; 26.12. Miscellaneous Features
;; (Data Types and Destructuring omitted)

(defloop initially (l exp exps)
  (push exp (loop-prologue l))
  (loop (if (symbolp (first exps)) (RETURN exps))
        (push (pop exps) (loop-prologue l))))

(defloop finally (l exp exps)
  (push exp (loop-epilogue l))
  (loop (if (symbolp (first exps)) (RETURN exps))
        (push (pop exps) (loop-epilogue l))))

(defloop named (l exp) (setf (loop-name l) exp))


(defvar *check-invariants* t 
  "Should VARIANT and INVARIANT clauses in LOOP be checked?")

(defloop invariant (l exp)
  (when *check-invariants*
     (add-body l `(assert ,exp () "Invariant violated."))))

(defloop variant (l exp)
  (when *check-invariants*
    (let ((var (gensym "INV")))
      (add-var l var nil)
      (add-body l `(setf ,var (update-variant ,var ,exp))))))

(defun update-variant (old new)
  (assert (or (null old) (< new old)) () 
          "Variant is not monotonically decreasing")
  (assert (> new 0) () "Variant is no longer positive")
  new)



;;; The Debugging Output Facility:

(defvar *dbg-ids* nil "Identifiers used by dbg")


(defun debug (&rest ids)
  "Start dbg output on the given ids."
  (setf *dbg-ids* (union ids *dbg-ids*)))

(defun undebug (&rest ids)
  "Stop dbg on the ids.  With no ids, stop dbg altogether."
  (setf *dbg-ids* (if (null ids) nil
                      (set-difference *dbg-ids* ids))))



(defun dbg (id indent format-str &rest args)
  "Print debugging info if (DEBUG ID) has been specified."
  ;; INDENT should be the number of spaces to indent,
  ;; but if it is not supplied, 0 is used.
  (when (member id *dbg-ids*)
    (if (numberp indent)
        (format *debug-io* "~&~V@T~?" indent format-str args)
        (apply #'dbg id 0 indent format-str args))))


(defmacro once-only (variables &rest body)
  "Returns the code built by BODY.  If any of VARIABLES
  might have side effects, they are evaluated once and stored
  in temporary variables that are then passed to BODY."
  (assert (every #'symbolp variables))
  (let ((temps (loop repeat (length variables) collect (gensym))))
    `(if (every #'side-effect-free? (list .,variables))
         (progn .,body)
         (list 'let
               ,`(list ,@(mapcar (lambda (tmp var)
                                   `(list ',tmp ,var))
                                 temps variables))
               (let ,(mapcar (lambda (var tmp) `(,var ',tmp))
                             variables temps)
                 .,body)))))

(defmacro define-enumerated-type (type &rest elements)
  "Represent an enumerated type with integers 0-n."
  `(progn
     (deftype ,type () '(integer 0 ,(- (length elements) 1)))
     (defun ,(symbol type '->symbol) (,type)
       (elt ',elements ,type))
     (defun ,(symbol 'symbol-> type) (symbol)
       (position symbol ',elements))
     ,@(loop for element in elements
             for i from 0
             collect `(defconstant ,element ,i))))