
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)


;;;_______________________________________________________________________________
;;;  FOCL
;;;
;;;  A relational learner which uses a separate and conquer approach guided by an 
;;;  information based heuristic to produce a concept description that covers all the 
;;;  positive examples and excludes all the negative examples.  Given a collection of 
;;;  classified examples of a concept, a set of extensionally defined predicates, and 
;;;  a possibly empty set of intensionally defined rules, one of may be distinguished 
;;;  as the goal concept, FOCL produces a Horn-clause description of the concept in 
;;;  terms of the extensional predicates.
;;; 
;;;  FOCL can be viewed as having two operators: find-a-clause which start a new 
;;;  clause with the body true, and find-a-literal which adds a literal to the end of 
;;;  the current clause body.  FOCL performs the second operator until no negative 
;;;  examples are covered by the clause, and performs the first operator adding new 
;;;  clauses until all positive examples are covered by some clause.  FOCL computes 
;;;  the information gain of the variabilizations (i.e., the orderings of existing 
;;;  and new variables)  of each extensionally defined predicate in order to 
;;;  determine which literal to add to the end of a clause.
;;; 
;;;  revisions
;;;  rv  who    date      reason
;;;  00  glenn  11/28/90  extended to deal with partial domain theories 
;;;                       (i.e., deleting literals if it improves info-gain,
;;;                              this is done even if info-gain is 0 if
;;;                              *partial-dt-0-gain* is set)
;;;  01  glenn  11/28/90  added in a batch-mode flag (*focl-batch-mode*)
;;;                       which will catch 0 gain errors (and possibly others)
;;;                       by returning a string like (error "<errmsg>")
;;;  02  mike   12/23/90  added  a flag (*save-examples*) which defaults to T
;;;                       and determines whether to save training examples in
;;;                       and out of each literal set to nil to save space.
;;;  03  cliff   3/13/91  noise-tolerance and *stopping-criteria-enabled* are now
;;;                       multi-valued flags
;;;                         :FOIL           uses FOIL's stopping criteria
;;;                         :RECURSIVE      uses our recursive stopping criteria
;;;                         {anything else} uses no stopping criteria
;;;  04  cliff   3/13/91  stopping-tables (*number-of-predicates*, and  
;;;                       *variablizations-of-new-slots-table*) are initialized
;;;                       when *stopping-criteria-enabled*.
;;;  05  cliff   3/14/91  parameter clauses has been replaced with goal-concept-name.
;;;                       nil  indicates there is no partial definition of the 
;;;                            concept being learned.
;;;                       rule-name indicates that the predicate rule-name on
;;;                                 *intensional-preds* should be used as the
;;;                                 goal concept.
;;;                       goal-concept-name defaults to pred-name, the name of the
;;;                       predicate being learned.
;;; 06  mike     4/26/91  My brother's birthday, and I didn't get a card
;;;                       changed goal-concept to be a pair or literal (for return opeartional literal)
;;;                       and rule structure (for compiled predicate, hash tables etc.)
;;; 07  cliff    4/29/91  No longer push bits around.  Stores the needed information
;;;                       in a non-local repository defined in stopping.lisp.
;;;                       number-of-initial-preds is no longer pased to find-a-clause.
;;; 08  glenn    5/04/91  added try-cliches-first back in - will cause cliches to be applied at the
;;;                       same level as everything else (i.e., not done as a last resort)
;;; 09  glenn    5/09/91  added *cliche-variabilizations-checked* for accounting of cliche work
;;; 10  mp       5/18/91  aded *refinement* :leaves or :frontier to control operationalization
;;; 11  cliff    5/20/91  Added capacity to graph while learning.
;;; 12  cliff    6/10/91  pause -> pause-break -> soft-pause-break.
;;; 13  mp       7/10/91  added *prefer-deletions* and *prefer-children* to control :frontier
;;; 14 mp        7/21/91  set *batch* to t during learning so "expert system"
;;;                       doesn't ask questions during learning.
;;; 15  cliff    7/25/91  tweeked graphic stuff
;;; 16 mp       7/26/91   added max-new-cliche-vars
(defun focl (pred-name                           ;; predicate (structure) to learn

             &key
             (clauses nil)
	     (goal-concept-name                  ;; CB(05) name of rule to use as the goal concept
              (intern (format nil "~a-rule" (car(first(first clauses))))))  
             (reset-hash-tables t) 
             (max-new-variables 0)
             (operationalize-constructive t) 
             (constructive-induction nil) 
             (trace nil) 
             (simplify-clauses t)
             (reset-statistics t)  
             (save-examples t)  
             (pred (get pred-name 'pred))
             (prefer-theory t)
	     (prefer-deletions nil)
	     (prefer-children t)
             (proof-vars-available t)
             (noise-tolerance nil)   ;; CB(03) :FOIL, :RECURSIVE, or NIL (everything else)
             (partial-dt-0-gain nil) ; ges explore parital domain theories even when dt has 0 gain
	     (simplify-operationalizations nil)
             (batch-mode nil) ; will suppress errors and just return them as strings
	                  ;currently can't try cliches first 
             (try-all-conjunctions nil) ; ges 2/16 implements strawman approach of trying all 2 element conjs
             (perform-pruning-III t) ; ges 2/22 whether or not to do III-pruning
	     (builtin-threshold-only nil) ; ges 3/15
             (use-cliches t)
             (refinement :leaves);; mp :leaves or :frontier
	     (gain-function :information);; :information or :ratio
             (available-cliches '(threshold-comparator partof))
             (try-cliches-first t) ; ges 5/4 will try cliches at the same level as everything else
             (graph-learning nil)    ;; CB(10)
             (max-new-cliche-vars nil)
            
             &allow-other-keys       ;; CB 6/12/91  this ought to make debugging difficult !!!!

             &aux 
	     (*batch* t)   ;;MP don't ask questions about predicate values during learning
             (old-vars (do ((i (- (pred-arity pred) 1) (decf i))  ;; old variables are the variables
                            (result nil))                         ;; that match the tuples of pred
                           ((< i 0) result)                       ;; ?0 matches the first variable,
                         (push (make-pcvar :id i) result)))       ;; ?1 the second etc
             (pos-tuples (pred-pos pred))
             (neg-tuples (pred-neg pred))
             (continue t)        ;;set to nil (value of find-a-clause) if no clause can be found
             (all-clauses nil)   ;; list of all clauses, in reverse order
             (last-clause nil)   ;; set to value of find a clause
             (goal-concept nil)) ;; CB(05) a prolog literal to be used as the goal concept 
  
  (when clauses
	(eval `(def-rule ,goal-concept-name :type ,(pred-type pred)
		 :clauses ,(subst goal-concept-name (car(first(first clauses))) clauses)
		 :induction nil)))
  (setq *predicate-being-learned* (pred-name pred)  ;used by prolog
        *operationalize-constructive* operationalize-constructive  
        *constructive-induction* constructive-induction 
	*save-examples* save-examples
        *focl-trace-level* trace
        *simplify-clauses* simplify-clauses
        *stopping-criteria-enabled* (case noise-tolerance                     ;; CB(03)
                                      ((:FOIL :RECURSIVE) noise-tolerance)    ;;
                                      (otherwise nil))                        ;;
        *proof-vars-available* proof-vars-available
        *theory-mode*  prefer-theory	
        *max-new-cliche-vars* (or max-new-cliche-vars max-new-variables)
        *prefer-children*  prefer-children
	*prefer-deletions*  prefer-deletions
	*simplify-operationalizations* simplify-operationalizations
        *partial-dt-0-gain* partial-dt-0-gain ; ges
        *focl-batch-mode* batch-mode
        *try-cliches-first* try-cliches-first ; ges 5/4 cliches will no longer require 0-gain
        *try-all-conjunctions* try-all-conjunctions ; ges 2/16
        *perform-pruning-III* perform-pruning-III ; ges 2/22
	*builtin-threshold-only* builtin-threshold-only
        *use-relational-cliches* use-cliches
        *refinement* refinement
	*gain-function* gain-function
        *available-relational-cliches* available-cliches ; ges
        *graph-learning* graph-learning    ;; CB(10)
        )

  (let ((goal-rule (get goal-concept-name 'rule)))                            ;; CB(05)
    (if (rule-p goal-rule)                                                    ;; 
      (setq goal-concept                                                      ;;
            (cons (list (cons goal-concept-name (copy-list old-vars)))        ;;
                  goal-rule))))                                               ;; (06) MP
  (if *stopping-criteria-enabled*                                             ;; CB(04)(05)(07)
    (init-stopping-tables goal-concept pos-tuples neg-tuples))                ;;
  (when *graph-learning*                                                      ;; CB(10)
    (init-learning-graph-windows *graph-learning*                             ;;
                                 (cons pred-name old-vars)                    ;;
                                 goal-concept)                                ;;
    (let ((pos (length pos-tuples))                                           ;;
          (neg (length neg-tuples)))                                          ;;
      (display-initial-uncovered-examples pos neg)         ;;
    ))                                                                        ;;

  (if reset-hash-tables (reset-hash-tables))
  (if reset-statistics 
    (setq *variablizations-checked* 0
          *cliche-variabilizations-checked* 0)) ; ges 5/10 better measure of work in cliches

  (do ()
      ((not (and pos-tuples continue)))
    ;;until there are no positive tuples uncovered
    ;;(or find-a-clause returned nil last time)
    (when (member :ct *focl-trace-level*)
      (format t "~%~%================================================================~%")
      (format t "Positive Tuples: ")
      (print pos-tuples)
      (format t "~%Negative Tuples: ")
      (print neg-tuples)
      (format t "~%"))
    
    (if *stopping-criteria-enabled*                                                 ;; CB(07)
      (set-clause-training-set-values-for-stopping-criteria pos-tuples neg-tuples)) ;;

    (when *graph-learning*                            ;; CB(10)
      (let ((pos (length pos-tuples))                 ;;  
            (neg (length neg-tuples)))                ;;
        (display-uncovered-examples pos neg)          ;;
        (display-covered-examples pos neg)            ;;
        ))                                            ;;

    (setf continue
          (setf last-clause (find-a-clause pred
                                           old-vars
                                           (pred-type pred)
                                           goal-concept     ;; CB(05) 
                                           pos-tuples
                                           neg-tuples
                                           max-new-variables
                                           ':first-time)))
;;; deal with no gain error
    (if (and (listp last-clause)
             (eql (car last-clause) 'error))
      (return-from focl last-clause))
    (when last-clause
      (multiple-value-setq (last-clause pos-tuples)
                          (simplify-clause last-clause
                                           pos-tuples
                                           neg-tuples
                                           old-vars))
      ;;all clauses is in reverse order, it's okay, we don't need it till the end
      (push last-clause all-clauses)
      ;;reduce positive tuples and learn the next clause
      (when (member :c *focl-trace-level*)
        (format t "~%~%*********New clause: ~a" last-clause))

      (when *graph-learning*                                 ;; CB(11)
        (store-learned-definition all-clauses)               ;;
        (when (member :pause-after-clause *graph-learning*)  ;;
          (soft-pause-break)))                               ;;

      ))
  (when clauses 
    (delete-rule goal-concept-name))
  (nreverse all-clauses))



;;;_______________________________________________________________________________
;;; FIND-A-CLAUSE
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  added special variable *covered-all-pos-tuples* for pruning II note
;;;                       that this var should only be set by as few procedures as possible
;;;                       and only by a call to the macro check-pruning-III-and-prune. Currently
;;;                       find-max-literal and find-literal-builtin-thresh set this var.
;;;  01  mike   12/05/90  set clauses to nil if literal learned by ebl
;;;                       never learn conjunction of target concept.                  
;;;  02  cliff   3/13/91  only perform bit computations when *stopping-criteria-enabled*
;;;                       modified to correctly handle both our recursive stopping criteria
;;;                       (:RECURSIVE), and FOIL's stopping criteria (:FOIL).
;;;  03  cliff   3/14/91  parameter clauses has been replaced with goal-concept.
;;;  04  mike    4/26/91  goal concept is now (prolog-literal-list . rule-structure)
;;;  05  cliff   4/29/91  No longer recieves number-of-initial-tuples, nor computes bit-available.
;;;                       Stores the needed information in a non-local env in stopping.lisp.
;;;  06  cliff   5/20/91  Added capacity to graph while learning.

(defun find-a-clause (pred
                      old-vars
                      type
                      goal-concept        ;; CB(03)
                      pos-tuples
                      neg-tuples
                      max-new-variables 
                      use-hash-tables

                      &aux
                      (vars old-vars)                      ;vars grows, old-vars stays the same
                      (continue t)                         ;nil if find-literal fails
                      new-vars                             ;additional variables,
                      new-neg-tuples 
                      new-pos-tuples
                      new-type                             ;additional type information (of new-vars)
                      (first-literal nil)                  ;start of clause
                      (new-literal nil)                    ;value of find-literal
                      (last-literal nil))                  ;end of clause (place to add next literal)

  (do ((*covered-all-pos-tuples*))       ;; ges note defined only in context of finding a clause
      ((not(and neg-tuples continue)))
    (declare (special *covered-all-pos-tuples*)) ; ges
    (setq *covered-all-pos-tuples* nil)
    ;;some negative couples are covered (or the last attamept at covering positive failed)
    
    (multiple-value-setq  (new-literal
                           new-vars
                           new-type
                           new-pos-tuples
                           new-neg-tuples)
                          ;;find-a-literal returns the next state for finding the remaining literals
                          ;;new-literal is nil if find-a-literal fails
                          (find-a-literal (I-content (length pos-tuples) (length neg-tuples))
                                          pred 
                                          vars
                                          type
                                          goal-concept   ;; CB(03)
                                          pos-tuples
                                          neg-tuples 
                                          max-new-variables
                                          old-vars
                                          use-hash-tables))
    ;;make new literal first (if no first)
    ;;or have last literal point at new-literal
    (when (setf continue new-literal)
      (cond                                                      ;; CB(06)
       (first-literal                                            ;;
        (setf (literal-next last-literal) new-literal)           ;;
        (when *graph-learning*                                   ;;
          (add-literal-to-last-clause-in-lcd-graph new-literal)  ;;
          (display-covered-examples (length new-pos-tuples)      ;;
                                    (length new-neg-tuples)))    ;;
        )                                                        ;;
       (t                                                        ;;
        (setf first-literal new-literal)                         ;;
        (when *graph-learning*                                   ;;
          (add-new-clause-to-lcd-graph new-literal)              ;;
          (display-covered-examples (length new-pos-tuples)      ;;
                                    (length new-neg-tuples)))    ;;
        ))                                                                        ;;
      (setf (literal-prev new-literal) last-literal) ;;last literal may be nil (if first clause) 
      (setf last-literal (last-literal new-literal)) ;;update last-literal
      (setf pos-tuples new-pos-tuples)
      (setf neg-tuples new-neg-tuples)
      (setf use-hash-tables nil)   ;;after first literal, negative tuples may be reduced, 
                                   ;;so we can't use hash taples that count # of matching neg-tuples
      (setq vars (append vars new-vars)) ;update vars and type
      (setq type (append type new-type))

      (when (eq :ebl (derivation-type (literal-derivation new-literal))) ;; [????]
        (setq goal-concept nil))                                         ;; CB(02)

      (when (member :l *focl-trace-level*)
        (format t "~%++++++New literal: ~a " new-literal))

      (if *stopping-criteria-enabled*                                                ;; CB(05)
        (update-clause-values-for-stopping-criteria new-literal pos-tuples type))))  ;; CB(05)
  
  first-literal)  ;value returned is first literal of clause






;;;_______________________________________________________________________________
;;; FIND-A-LITERAL
;;;
;;; returns 5 values
;;;
;;; 1. new-literal      -literal structure with maximum gain, suitble for adding to end of clause
;;; 2. new-vars         -names of newvariables (renamed to be old variables)
;;; 3. new-types        -types of new variables
;;; 4. new-pos-tuples   -next set of positive tuples
;;; 5. new-neg-tuples   -next set of negative tuples
;;;
;;;  Dispatches to specific find-literal functions
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  altered control flow to deal with pruning-III
;;;  01  cliff   3/14/91  parameter clauses has been replaced with goal-concept.
;;;  02  cliff  04/29/91  no longer accepts bits-available, nor returns literal-bits
;;;  03  glenn  05/04/91  added a call to find-literal-cliches
;;;  04  cliff  05/23/91  resets induction graphic when *graph-learning*

(defun find-a-literal (orig-info            ;I(Ti)
                       pred                 ;predicate structure of pred being learned
                       vars                 ;current set of bound variables
                       type                 ;type of bound vars
                       goal-concept         ;; CB (01) goal concept (if any) to operationlize for EBL
                       pos-tuples           
                       neg-tuples 
                       max-new-variables 
                       old-vars             ;original vars (for recursion detection)
                       use-hash-tables)

  (when *graph-learning*                          ;; CB(04)
    (clear-current-literal-in-induction-graphic)  ;;
    (clear-best-literal-in-induction-graphic))    ;;

  (let (ebl-literal ebl-vars ebl-type ebl-pos-tuples ebl-neg-tuples ebl-gain
                    sbl-literal sbl-vars sbl-type sbl-pos-tuples sbl-neg-tuples sbl-gain
                    ci-literal  ci-vars  ci-type  ci-pos-tuples  ci-neg-tuples  ci-gain
                    bi-literal bi-vars bi-type bi-pos-tuples bi-neg-tuples bi-gain
                    cl-literal  cl-vars  cl-type  cl-pos-tuples  cl-neg-tuples  cl-gain
                    cl-name cl-head-vars) ; cliches stuff
    
    
    (when (and (member :lt *focl-trace-level*)
               (not use-hash-tables)) ;i.e. not first literal
      (format t "~%~%________________________________________________________________~%")
      (format t "Positive Tuples: ")
      (print pos-tuples)
      (format t "~%Negative Tuples: ")
      (print neg-tuples))
    
    (multiple-value-setq  (ebl-literal
                           ebl-vars
                           ebl-type
                           ebl-pos-tuples
                           ebl-neg-tuples
                           ebl-gain)
                          ;; ebl-literal is :fail if no literal with positive gain is found.
                          ;; ebl-literal is nil when all literal encoding lengths are too long.
                          (find-literal-from-goal-concept orig-info
                                                          goal-concept         ;; CB(01)
                                                          vars
                                                          type
                                                          pos-tuples 
                                                          neg-tuples
                                                          use-hash-tables))
    (if (or (null ebl-literal)
            (eq ebl-literal :fail))
      (setq ebl-gain 0))
    (cond 
     ((and *theory-mode* (> ebl-gain 0))
      (values ebl-literal
              ebl-vars
              ebl-type
              ebl-pos-tuples
              ebl-neg-tuples))
     (t (multiple-value-setq  (sbl-literal
                               sbl-vars
                               sbl-type
                               sbl-pos-tuples
                               sbl-neg-tuples
                               sbl-gain)
                              ;; sbl-literal is :fail if no literal with positive gain is found.
                              ;; sbl-literal is nil when all literal encoding lengths are too long.
                              (find-literal-extensional orig-info 
                                                        pred
                                                        vars 
                                                        type
                                                        max-new-variables
                                                        pos-tuples 
                                                        neg-tuples
                                                        old-vars 
                                                        use-hash-tables
                                                        ebl-gain))
        (if (or (null sbl-literal) 
                (eq sbl-literal :fail))
          (setq sbl-gain 0))
        (if *covered-all-pos-tuples* ; ges found the best literal to complete clause
          (setq bi-literal :fail)
          (multiple-value-setq  (bi-literal 
                                 bi-vars 
                                 bi-type 
                                 bi-pos-tuples 
                                 bi-neg-tuples 
                                 bi-gain)
                                ;;find-a-literal returns the next state for finding the remaining literals
                                ;;bi-literal is nil if find-a-literal fails
                                (find-literal-builtin orig-info
                                                      pred
                                                      vars 
                                                      type ; glenn added
                                                      pos-tuples
                                                      neg-tuples 
                                                      old-vars ; glenn added
                                                      use-hash-tables ; glenn added
                                                      (max ebl-gain sbl-gain)))) 
        (if (or (null bi-literal)
                (eq bi-literal :fail))
          (setq bi-gain 0))
        (if (and *constructive-induction* (not *covered-all-pos-tuples*)) ; ges see above
          (multiple-value-setq  (ci-literal
                                 ci-vars
                                 ci-type
                                 ci-pos-tuples
                                 ci-neg-tuples
                                 ci-gain)
                                ;; ci-literal is :fail if no literal with positive gain is found.
                                ;; ci-literal is nil when all literal encoding lengths are too long.
                                (find-literal-intensional orig-info 
                                                          pred 
                                                          vars 
                                                          type
                                                          max-new-variables
                                                          pos-tuples 
                                                          neg-tuples
                                                          old-vars 
                                                          use-hash-tables
                                                          (max ebl-gain sbl-gain bi-gain)))
          (setq ci-literal :fail)) ; ges make sure ci-literal is fail if we don't try
        (if (or (null ci-literal)
                (eq ci-literal :fail))
          (setq ci-gain 0))
        
        (if (or *try-cliches-first*
                (and (zerop sbl-gain) (zerop ebl-gain) (zerop bi-gain) (zerop ci-gain))) ; nothing so far
          (multiple-value-setq  (cl-literal
                                 cl-vars
                                 cl-type
                                 cl-pos-tuples
                                 cl-neg-tuples
                                 cl-gain
                                 cl-name
                                 cl-head-vars)
                                ;; sbl-literal is :fail if no literal with positive gain is found.
                                ;; sbl-literal is nil when all literal encoding lengths are too long.
                                (find-literal-cliches orig-info 
                                                      pred vars 
                                                      type
                                                      max-new-variables
                                                      pos-tuples 
                                                      neg-tuples
                                                      old-vars 
                                                      use-hash-tables
                                                      (max sbl-gain bi-gain ci-gain ebl-gain))))
        
        (if (or (null cl-literal)
                (eq cl-literal :fail))
          (setq cl-gain 0))
        
        ;;find-a-literal returns the next state for finding the remaining literals
        ;;find-a-literal fails if ebl (find-a-literal-from-clause),
        ;;                        sbl (find-a-literal-extensional), and
        ;;                        ci  (find-a-literal-intensional) fail.
        (cond
         ((> ci-gain 0)            ;;ci-gain > 0  =>  ci-gain > (cl-gain max sbl-gain ebl-gain)
          (values ci-literal 
                  ci-vars
                  ci-type
                  ci-pos-tuples
                  ci-neg-tuples))
         ((> cl-gain 0)            ;;cl-gain > 0  =>  cl-gain > (max sbl-gain ebl-gain)
          (create-pred-from-cliche cl-literal cl-name cl-head-vars) ; create named or unnamed predicate if cache? flag set
          (values cl-literal 
                  cl-vars
                  cl-type
                  cl-pos-tuples
                  cl-neg-tuples))
         ((> bi-gain 0) ; glenn added
          (values bi-literal
                  bi-vars
                  bi-type
                  bi-pos-tuples 
                  bi-neg-tuples ))
         ((> sbl-gain 0)           ;;sbl-gain > 0  =>  sbl-gain > ebl-gain
          (values sbl-literal
                  sbl-vars
                  sbl-type
                  sbl-pos-tuples
                  sbl-neg-tuples))
         ((> ebl-gain 0)
          (values ebl-literal
                  ebl-vars
                  ebl-type
                  ebl-pos-tuples
                  ebl-neg-tuples))
         ((and (eq  ci-literal :fail)
               (eq sbl-literal :fail)
               (eq bi-literal :fail)
               (eq ebl-literal :fail))
          (error "No gain greater than 0")))))))




;;;_______________________________________________________________________________
;;; FIND-A-LITERAL-FROM-GOAL-CONCEPT
;;;
;;; returns 6 values
;;;
;;; 1. new-literal      -literal structure with maximum gain, suitble for adding to end of clause
;;; 2. new-vars         -names of newvariables (renamed to be old variables)
;;; 3. new-types        -types of new variables
;;; 4. new-pos-tuples   -next set of positive tuples
;;; 5. new-neg-tuples   -next set of negative tuples
;;; 6. max-gain         -maximum information gain (to compare aginst intensional if necessary)
;;;
;;;   Note:  EBL operationalizes in a greedy fashion and as a result it may
;;;          not always operationalize the branch which yields the highest gain.
;;;          When complete, *check-all-definitions* :always would force the 
;;;          selection of the "clause" either operational or non-operational
;;;          with the highest gain. *operationalize-constructive* true would
;;;          force consideration of only the operational clauses.
;;;          
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   3/16/91  created to replace find-a-literal-from-clause I think
;;;                       *partial-dt-0-gain* functionality is maintained since
;;;                       I don't check the information gain of goal-concept
;;;                       before I pass it to return-operational-literal.
;;;  01  cliff  04/29/91  no longer accepts bits-available, nor returns literal-bits
;;;  02  cliff  06/12/91  stores the number of pos and neg tuples for display in ebl

(defun find-literal-from-goal-concept (original-info
                                       goal-concept
                                       variables
                                       variables-type
                                       pos-tuples
                                       neg-tuples
                                       use-hash-tables)
  (when *graph-learning*
    (store-info-gain-values-for-display (length pos-tuples) (length neg-tuples) "-"))
  (if (null goal-concept)
    :fail                                              ;; with no goal concept EBL fails
    (return-operational-literal (car goal-concept)     ;; try greedy EBL on goal-cocnept
                                variables-type
                                variables        ;; parameters to rule
                                variables        ;; bound variables
                                pos-tuples
                                neg-tuples 
                                original-info
                                :EBL
                                use-hash-tables)))


;;;_______________________________________________________________________________
;;; OPERATIONALIZE
;;;
;;;  Operationalizes a prolog clause body (ie. a list of literals) using a greedy
;;;  method which takes the path with maximum gain at each step.
;;;
;;;  returns 7 values
;;;
;;; 1. operational-body   -
;;; 2. all-new-vars       -  
;;; 3. all-new-types      -
;;; 4. all-new-var-alist  -
;;; 5. pos-tuples         -
;;; 6. neg-tuples         -
;;;
;;;  returns nil if unable to find an operationalization
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  extended to deal with is predicates
;;;  01  glenn  11/28/90  extended to allow for a partial operationalization when the current
;;;                       literal added causes 0 positive tuple matches (i.e., by skipping over
;;;                       the current literal).  Note this is not as secure as other methods
;;;                       employed so it is controlled by the flag
;;;  02  mike    ????     ????
;;;  03  cliff  05/20/91  Added capacity to graph operationalization process.
;;;                       seperated extensional and builtin conditions

(defun operationalize (body 
                       vars 
                       type 
                       pos-tuples 
                       neg-tuples 
                       orig-info
                       use-hash-tables

                       &optional 
                       (all-new-vars nil)      ;; not used in first call, so inited to nil
                       (all-new-var-alist nil) ;; alist used to convert derivation variables
                       (all-new-types nil) 
                       (first-literal nil)
                       (last-literal nil)

                       &aux
                       new-vars 
                       literal-vars            ;; free variables are converted to old variables here
                       new-pos-tuples 
                       new-neg-tuples 
                       new-types
                       (new-var-alist nil)
                       (new-literal nil)
                       (literal (car body))    ;; literal is a prolog one, not a literal structure
                       (old-body body)
                       (old-all-new-var-alist all-new-var-alist))

  (when *graph-learning*                         ;; CB(03)
    (graph-clause-operationalization body))      ;;

  (cond ;((null pos-tuples)  nil)   ;; operationalization doesn't cover any positive, return failure
        ((null literal)             ;; we are finished operationalizing a clause
         (values first-literal      ;; return first-literal and info to update state
                 all-new-vars 
                 all-new-types 
                 all-new-var-alist 
                 pos-tuples 
                 neg-tuples)) 
        (t (cond
            ((eq (car literal) 'is) ; ges added 
             (setq new-pos-tuples 
                   (extend-tuples-is pos-tuples vars (second literal)(third literal)))
             (setq new-neg-tuples 
                   (extend-tuples-is neg-tuples vars (second literal)(third literal)))
             (setq new-vars (list (make-pcvar :id (length vars)))) ; is adds a new var at the end of the old vars
             (setq new-types (list :number)) ; do something here
             ;; (get-pstruct (car (third literal))) ;;+
             ;; added these three because they're used in extensional and builtin preds
             (setq literal-vars (cons (car new-vars) (cddr literal)))
             (setf new-var-alist (list (cons (second literal) (car new-vars))))
             (setf body (sublis new-var-alist body :test #'equalp))
             (setf all-new-var-alist (nconc all-new-var-alist new-var-alist))
             (setf new-literal (make-literal :predicate-name (car literal)
                                             :variablization literal-vars
                                             :derivation (make-derivation 
                                                          :path (list (cons (car literal)
                                                                            literal-vars)))))
             )

            ((extensional? literal)
             ;;it is operational, so just rename free variables to old variables
             ;;because this clause binds variables
             (multiple-value-setq (literal-vars new-vars new-types new-var-alist)
                                  ;literal-vars is the variables in call with free vars
                                  ;renamed to next old-var
                                  ;new-vars is list of "new" old-vars
                                  ;new-types is types of new vars
                                  ;new-var-alist is (old-name . bound-name) for free vars
                                  (transfer-literal-vars (cdr literal) ;variables in call
                                                         (p-type (get-pstruct (car literal)))
                                                         ;;type of variables
                                                         vars ;;bound variables of clause
                                                         (length vars)))
             (when new-var-alist  ;make new var an old var in the rest of the body
               (setf body (sublis new-var-alist body :test #'equalp))
               ;update all-new-var-alist which will be passed back to keep derivation
               (setf all-new-var-alist (nconc all-new-var-alist new-var-alist)))
             
             ;;create new literal structure for this variabilization
             (setf new-literal 
                   (make-literal :predicate-name (car literal)
                                 :variablization literal-vars
                                 :derivation (make-derivation 
                                              :path (list (cons (car literal) literal-vars)))))
             (setf new-pos-tuples 
                   (extend-tuples-extensional (get-pstruct (car literal))
                                              pos-tuples
                                              literal-vars
                                              new-vars))
             ;;extend the positive and negative tuples
             (setf new-neg-tuples (extend-tuples-extensional (get-pstruct (car literal))
                                                             neg-tuples
                                                             literal-vars
                                                             new-vars))

             ;;save positive and negative literals (before and after)
             ;;they may prove useful for revising theories
             (setf (literal-pos new-literal) pos-tuples)
             (setf (literal-neg new-literal) neg-tuples)
             (setf (literal-new-pos new-literal) new-pos-tuples)
             (setf (literal-new-neg new-literal) new-neg-tuples)
             )


            ((builtin? literal)
             ;;it is operational, so just rename free variables to old variables
             ;;because this clause binds variables
             (multiple-value-setq (literal-vars new-vars new-types new-var-alist)
                                  ;literal-vars is the variables in call with free vars
                                  ;renamed to next old-var
                                  ;new-vars is list of "new" old-vars
                                  ;new-types is types of new vars
                                  ;new-var-alist is (old-name . bound-name) for free vars
                                  (transfer-literal-vars (cdr literal) ;variables in call
                                                         (p-type (get-pstruct (car literal)))
                                                         ;;type of variables
                                                         vars ;;bound variables of clause
                                                         (length vars)))
             (when new-var-alist  ;make new var an old var in the rest of the body
               (setf body (sublis new-var-alist body :test #'equalp))
               ;update all-new-var-alist which will be passed back to keep derivation
               (setf all-new-var-alist (nconc all-new-var-alist new-var-alist)))
             
             ;;create new literal structure for this variabilization
             (setf new-literal 
                   (make-literal :predicate-name (car literal)
                                 :variablization literal-vars
                                 :derivation (make-derivation 
                                              :path (list (cons (car literal) literal-vars)))))

             (setf new-pos-tuples 
                   (extend-tuples-builtin (get-pstruct (car literal))
                                          pos-tuples
                                          literal-vars))
             ;;extend the positive and negative tuples
             (setf new-neg-tuples 
                   (extend-tuples-builtin (get-pstruct (car literal))
                                          neg-tuples
                                          literal-vars))

             ;;save positive and negative literals (before and after)
             ;;they may prove useful for revising theories
             (setf (literal-pos new-literal) pos-tuples)
             (setf (literal-neg new-literal) neg-tuples)
             (setf (literal-new-pos new-literal) new-pos-tuples)
             (setf (literal-new-neg new-literal) new-neg-tuples)
             )

            ((negation? literal) ;looks like (not (foo x) (bar y z)) - an implicit conjunction
             ;(usually (not (foo x))

             (when *graph-learning*                                                          ;; CB(03)
               (replace-consequent-with-antecedents-in-operationalization-frontier literal)) ;;

             (multiple-value-setq  (new-literal new-vars new-types new-var-alist 
                                                new-pos-tuples new-neg-tuples)
                                   ;operationalize body of negation
                                   (operationalize (cdr literal)
                                                   vars type neg-tuples 
                                                   pos-tuples orig-info nil))
             (setf new-vars nil)      ;negations don't create new variables
             (setf new-types nil) 
             (setf new-var-alist nil)
             ;;make a negated literal
             (setf new-literal 
                   (make-literal :negated? t
                                 :negated-literals new-literal
                                 :derivation (make-derivation 
                                              :path (list (cons 'not
                                                                (convert-to-prolog new-literal))))))
             ;update-tuples (uses full
             (setq new-pos-tuples (extend-tuples new-literal pos-tuples vars nil))
             (setq new-neg-tuples (extend-tuples new-literal neg-tuples vars nil))
             
             ;save positive and negative literals (before & after)
                          (when *save-examples* 
		   (setf (literal-pos new-literal) pos-tuples)
		   (setf (literal-neg new-literal) neg-tuples)
		   (setf (literal-new-pos new-literal) new-pos-tuples)
		   (setf (literal-new-neg new-literal) new-neg-tuples))
             )
            
            (t ;head of term is the name of a rule

             (when *graph-learning*                                                          ;; CB(03)
               (replace-consequent-with-antecedents-in-operationalization-frontier literal)) ;;

             (multiple-value-setq  (new-literal new-vars new-types new-var-alist 
                                                new-pos-tuples new-neg-tuples) 
                                   ;really operationalize does the work of finding
                                   ;which clause of the rule to operationalize
                                   (really-operationalize literal vars type pos-tuples 
                                                          neg-tuples orig-info use-hash-tables))
             )
            )

           (cond ((and new-literal ;new-pos-tuples
                       ) ; still covered some pos-tuples
             ;;;chain new literals together with first
                  (if first-literal 
                    (setf (literal-next last-literal) new-literal)
                    (setf first-literal new-literal))
                  (setf (literal-prev new-literal) last-literal)
                  (setf last-literal (last-literal new-literal))
                  
                  ;compute information of new-pos-tuples and new-neg-tuples
                  (setf orig-info (if (null  new-pos-tuples) 0
                        (I-content (length new-pos-tuples) (length new-neg-tuples))))
                  
                  ;;recuse on remainder of body-
                  ;;if it can't be operationalized, this fails also
                  (operationalize (cdr body) (append vars new-vars) (append type new-types)
                                  new-pos-tuples new-neg-tuples orig-info nil
                                  (append all-new-vars new-vars) all-new-var-alist
                                  (append all-new-types new-types)
                                  first-literal
                                  last-literal))
                 (*partial-dt-0-gain* 
                  ; skip over literal
                  (operationalize (cdr old-body) vars type pos-tuples neg-tuples orig-info nil
                                  all-new-vars old-all-new-var-alist all-new-types 
                                  first-literal last-literal))
                 (t nil)))))



;;;_______________________________________________________________________________
;;; TRANSFER-LITERAL-VARS
;;;
;;;  makes a copy of the argument to a literal, replacing new vars 
;;;  (i.e., not member of vars) with new bound vars returns a the replaced
;;;  argument, new vars, types of new vars, and an alist of (free . bound)
;;;  name for variable- bound names is 1 + number of current bound variables
;;;
;;;  revisions
;;;  rv  who    date      reason

(defun transfer-literal-vars (literal-parameters types vars no-vars &aux new new-vars new-types)
  (let* ((alist nil)
         (vars  (mapcar #'(lambda(p type) 
                            (cond ((member p vars :test #'equalp) p)
                                     ((cdr(assoc p alist :test #'equalp)))
                                     ((not (pcvar-p p))  ;;could be a constant
                                      p)
                                     (t (setf new (make-pcvar :id no-vars)) ;;free variable becomes bound
                                        (incf no-vars)
                                        (push (cons p new) alist)
                                        (push new new-vars)
                                        (push type new-types)
                                        new)))
                        literal-parameters types)))
    (values vars (nreverse new-vars) (nreverse new-types) alist)))


;;;_______________________________________________________________________________
;;; REALLY-OPERATIONALIZE
;;;
;;;  rv  who    date      reason
;;;  00  cliff  05/15/91  added capacity to dispay the gain of each disjunct being
;;;                       considered for operationalization

(defun really-operationalize (literal              ;(name arg1 arg2)
                              vars                 ;current list of bound vars
                              type                 ;type of bound vars
                              pos-tuples           ;positive tuples 
                              neg-tuples           ;negative tuples 
                              orig-info            ;info in tuples
                              use-hash-tables)
  (let (clause 
        new-literal 
        all-new-vars 
        all-new-types
        all-new-var-alist 
        new-pos-tuples 
        new-neg-tuples
        clause-gain                                             ;; CB(00)
        (clauses (rule-clauses (get (car literal) 'rule)))
        )

    ;;2. If there is one clause, it has the maximum information gain
    ;;   otherwise, sort clauses by information gain
    (cond ((null (cdr clauses))
           (setq clauses (list (cons 'this-is-not-used (car clauses)))))
          (t (setq clauses 
                   (sort (mapcar #'(lambda(clause)
                                     (setf clause-gain                              ;; CB(00)
                                           (info-gain-prove (clause-body clause)
                                                            (clause-prolog-function clause)
                                                            orig-info
                                                            pos-tuples
                                                            neg-tuples 
                                                            (cdr literal) 
                                                            (when use-hash-tables 
                                                              (clause-neg-tuples-hash clause))))
                                     (when *graph-learning*                         ;; CB(00)
                                       (display-clause-pos-neg-gain                 ;;
                                        (substitute1 (clause-body clause)           ;;
                                                     (unify-list                    ;;
                                                      (clause-parameters clause)    ;;
                                                      (cdr literal)))))             ;;
                                     (cons clause-gain clause))                     ;;
                                 clauses)
                         #'(lambda(x y)
                             (> (car x)(car y)))))))
    ;;3. attempt to operationalize each clause in order of info gain
    ;;;attempt may fail if it doesn't cover any positive examples
    ;;;can this really happen?  I doubt it, (but maybe with pruning heuristics)
    ;;;return values of first one to operationalize
    (when (some #'(lambda(c)
                    (setq clause (cdr c)) ;car is info-gain
                    (multiple-value-setq (new-literal 
                                          all-new-vars
                                          all-new-types
                                          all-new-var-alist 
                                          new-pos-tuples
                                          new-neg-tuples)
                                         (operationalize (substitute1 (clause-body clause)
                                                                      (unify-list (clause-parameters clause)
                                                                                  (cdr literal)))
                                                         vars type pos-tuples neg-tuples orig-info use-hash-tables))
                    
                    ;;insert proof trace stores a derivation in the literal
                    (when new-literal
                      (insert-proof-trace new-literal literal all-new-var-alist clause)
                      t))
                clauses)
      (values new-literal
              all-new-vars
              all-new-types
              all-new-var-alist 
              new-pos-tuples
              new-neg-tuples))))



;;;===============================================================================
;;; Move The following functions to  Literal-Strutures-Utilities.Lisp  [????]
;;;===============================================================================

;;;_______________________________________________________________________________
;;; INSERT-PROOF-TRACE
;;;
;;;  push derivation onto the front of every operationalized literal
;;;  derivation of the form (clause-structure . prolog-body)
;;;  the clause-structure tells waht clause was used
;;;  the prolog body uses the same variables as the head of the clause and the literal
;;;  the clause is maintained for possible revisions
;;;
;;;  rv  who    date      reason

(defun insert-proof-trace (new-literal
                           prolog-literal
                           var-alist clause)
  (setf prolog-literal (sublis var-alist prolog-literal :test #'equalp))
  (do ((l new-literal (literal-next l)))
      ((null l))
    (push (cons clause prolog-literal) 
          (derivation-path (literal-derivation l)))))


;;;_______________________________________________________________________________
;;;  LAST-LITERAL
;;;
;;;  chases pointers until it finds the one with a null pointer
;;;
;;;  rv  who    date      reason

(defun last-literal(l &aux (next (literal-next l)))
  (if next (last-literal next)
      l))

;;;===============================================================================
;;; Move The following functions to Variable-Utilities.Lisp  [????]
;;;===============================================================================

#|
;;;_______________________________________________________________________________
;;;  EXTRACT-NEW-VARS
;;;
;;;  rv  who    date      reason

(defun extract-new-vars (literal-parameters clause-parameters)
  (remove-if #'(lambda(x) (member x clause-parameters :test #'equalp))
             literal-parameters))
|#


;;;===============================================================================
;;; Doesn't the following belong in load?  [????]
;;;===============================================================================

#+:ccl(defun full-trace()
  (setq *trace-print-level* (setq *trace-print-length* nil)))


