
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

;;;__________________________________________________________________________________________
;;; FIND-MAXIMUM-LITERAL
;;;
;;; returns the predicate variabilization of pred with the maximum information gain
;;; works for intension and extensional pred
;;;
;;; returns 4 values
;;; 1. predicate-with-maximum-gain       - pred structure
;;; 2. variabilization-with-maximum-gain - variables ?0 thru ?n are old, ?-j indicates new
;;; 3. max-negated?                      - t if negation of predicate has maximum gain
;;; 4. max-gain                          - the amount of information gain
;;;
;;; returns :fail if no variabilization has gain greater than 0, or
;;;         nil   if all variablizarion with non zero require too many bits to encode
;;;
;;;  Note if pruning III occurs this is indicated by the special variable
;;;       *covered-all-pos-tuples* (this variable should only be set by the
;;;       macro check-pruning-III-and-prune) to facilitate the proper control flow.
;;;
;;;  rv  who    date       reason
;;;  00  glenn  10/04/90   changed max-gain from 0
;;;  01  glenn  10/04/90   changed initialization of gain from (gain gain-to-beat)
;;;  02  kamal  10/09/90   added call to get-variabilizations
;;;                        added 3 kinds of pruning
;;;  03  glenn  11/01/90   added extensional/threshold conjunctions - killed this
;;;  04  glenn  11/01/90   adjusted control flow for type III pruning and
;;;                        made pruning III check a macro
;;;  05  glenn  02/09/91   added in key-word param variabilization-restrictions for cliches
;;;  06  glenn  02/16/91   added key-word try-cliches - to implement iterative approach
;;;                        (i.e., try without cliches first)
;;;  07  cliff  03/19/91   adjusted control flow when *stopping-criteria-enable* is nil
;;;  08  cliff  04/29/91   make stopping criteria reflect FOIL's.  No longer accepts
;;;                        bits-available nor returns literal bits
;;;  09  glenn  05/02/91   deleted conjunction stuff and added cliche info as keyword args
;;;                        (instantiated-cliche, position-in-cliche, var-restrictions, and
;;;                        pred-restrictions)
;;;  10  glenn  05/02/91   updated varzn restriction filter and added pred restriction filter
;;;  11  cliff  05/08/91   removed ":old-variables variables" from calls to info-gain.
;;;  12  cliff  05/20/91   added capacity to graph while learning.
;;;  13  cliff  07/23/91   modified graphics stuff

(defun find-maximum-literal
       (original-info              ;passed to info-gain
        predicate-being-learned    ;used to detect recursion
        variables                  ;list of old variables
        variable-types             ;types of old variables
        maximum-new-vars           ;maximum number of new variables allowed
        pos-tuples                 ;positive tuples
        neg-tuples                 ;negative
        original-vars              ;used to detect recursion
        use-hash-tables            ;used by info-gain
        all-preds                  ;list of predicates to try constructing literals from
        gain-to-beat               ;don't consider predicates with less than this much gain

        &key
        (var-restrictions nil) ; ges 5/2 renamed
        (pred-restrictions nil) ; ges 5/2
        (instantiated-cliche nil) ; ges 5/2
        (position-in-cliche nil) ; ges 5/2
        (try-cliches nil))                 ; ges 2/16

  (let
    ((epsilon 0.001)              ;if a variabilization achieves within epsilon of its max possible gain
                                  ;prune away specializations of that variabilization
     (max-gain-exceeded-by-negative nil)
     (covered-all-pos-tuples nil)) ;dont look at any more variabilizations for any predicate
                                  ;when a variabilization covers all remaining tuples for this clause

    (when *graph-learning*                                                               ;; CB 13
      (if instantiated-cliche                                                            ;;
        (display-partially-instanciated-cliche-in-induction-graphic instantiated-cliche) ;;
        (display-cliche-name-in-induction-graphic nil)))                                 ;;

    ;;loop thru all predicates
    (do* ((preds (apply-pred-restrictions all-preds pred-restrictions maximum-new-vars)
                 (cdr preds)) ; ges 5/2
          (pred (cdr (car preds)) (cdr(car preds))) ;structure is cdr of alist
          (max-gain (max 0 gain-to-beat))
          (gain  )
          (predicate-with-maximum-gain nil)
          (variabilization-with-maximum-gain)
          (best-negative-variabilization nil) ;ka
          (gain-from-negative nil)            ;ka
          (variabilization nil)               ;ka
          (max-possible-gain nil)             ;ka
          (max-negated? nil))

         ;;return values when preds is exhausted
         ((null preds)
          (cond ((and (null predicate-with-maximum-gain)
                      *stopping-criteria-enabled*
                      (some-literal-required-too-many-bits)) nil)
                ((null predicate-with-maximum-gain) :fail)
                ; ges 5/2 deleted conj-info-struct stuff
                (t (progn
                     (when (member :lt *focl-trace-level*)
                       (format t "~&old-vars: ~a winning predicate: ~a winning variabilization: ~a negated? ~a ~%~%" variables
                               (p-name predicate-with-maximum-gain)
                               variabilization-with-maximum-gain
                               max-negated?))
                     (values predicate-with-maximum-gain
                             variabilization-with-maximum-gain
                             max-negated?
                             max-gain)))))

      (cond
       ((not (p-induction pred)) nil)
       ((not (any-variabilizations? (p-type pred) variable-types maximum-new-vars)) nil)
       ((and *stopping-criteria-enabled*
             (predicate-requires-too-many-bits pred)) nil)
       (t

        (when *graph-learning*                                     ;; CB(12)
          (update-predicate-in-induction-graphic (p-name pred)))   ;;

        ;; CHECK POSITIVE VARIABLIZATIONS
        (do* ((variabilizations
               (apply-variabilization-restrictions ; ges 5/2 updated varzn restr check
                pred
                var-restrictions
                (get-variabilizations pred
                                      maximum-new-vars
                                      variables
                                      predicate-being-learned
                                      variable-types
                                      original-vars)
                variables
                original-vars
                instantiated-cliche
                position-in-cliche)
               (cdr variabilizations))
              (current-struct (car variabilizations) (car variabilizations)))

        ((null variabilizations))  ;exit when no more variabilizations
     (setq variabilization
           (variabilization-struct-variabilization current-struct))
          (when (variabilization-struct-look-at-positive? current-struct)
            (multiple-value-setq
             (gain max-possible-gain covered-all-pos-tuples)
             (info-gain pred
                        variabilization
                        nil
                        pos-tuples
                        neg-tuples
                        original-info
                        use-hash-tables
                        :instantiated-cliche instantiated-cliche ; to print out something more appropriate
                        :position-in-cliche position-in-cliche)) ; ges

              (when (member :i *focl-trace-level*)
                ; ges 5/2 updated to print out cliche-info
               (if instantiated-cliche
                  (print-cliche instantiated-cliche position-in-cliche)))

              (when *graph-learning*                                               ;; CB(12)
                (update-variablization-in-induction-graphic variabilization        ;;
                                                            instantiated-cliche    ;;
                                                            position-in-cliche))   ;;

              (when (>= gain max-gain)   ;; want most specific of the positive preds with same gain

                (when *graph-learning*                         ;; CB(12)
                  (update-best-literal-in-induction-graphic))  ;;

                (setf max-gain gain
                      predicate-with-maximum-gain pred
                      variabilization-with-maximum-gain variabilization
                      max-negated? nil)

                ;; pruning type III: dont look at any more varzns for any predicate
                (check-pruning-III-and-prune
                 covered-all-pos-tuples find-maximum-literal predicate-with-maximum-gain
                 variables variabilization-with-maximum-gain
                 (values predicate-with-maximum-gain variabilization-with-maximum-gain
                         max-negated? max-gain nil)))

              (when
                (and (>= 0 gain)                                  ;only try negative if positives <= to 0
                     (> 2 (count-if #'new-var? variabilization))) ;not more than 1 free var [????]
                (setf (variabilization-struct-look-at-negative? current-struct) t))

              ;; now see if we can prune some variabilizations out
              ;; 1 stands for type I pruning ... normal quinlan
              ;; 2 stands for type II pruning ... when a varzn's info-gain is within epsilon of
              ;;                                  max-possible gain for that varzn, prune
              ;;                                  specializations of that varzn
              (when (find-if #'new-var? variabilization)
                (if (< max-possible-gain max-gain)
                  (prune variabilization (cdr variabilizations) 1)
                  (when (< (- max-possible-gain gain) epsilon)
                    (prune variabilization (cdr variabilizations) 2))))))


        ;; CHECK NEGATIVE VARIABLIZATIONS
        ;; if max-negated==T, gain-from-negative guaranteed to be better than max-gain
        ;; if max-negated==nil, disregard other 2 returned values
        ;; even with super-pruning, still need to visit this set of varzns
          (setq max-gain-exceeded-by-negative nil)
          (multiple-value-setq (gain-from-negative
                                best-negative-variabilization
                                max-gain-exceeded-by-negative
                                covered-all-pos-tuples) ; ges
                               (look-at-negative-variabilizations
                                ;; cliches aren't appropriate with negative variablizations so
                                ;; don't generate them when processing cliches (but ok with try
                                ;; all conjunctions)
                                (if (or *try-all-conjunctions* (not try-cliches)) ; note can't try cliches first anymore
                                  (reverse (get-variabilizations pred maximum-new-vars variables
                                                                 predicate-being-learned
                                                                 variable-types original-vars))
                                  nil)
                                max-gain
                                pred
                                pos-tuples
                                neg-tuples
                                original-info
                                use-hash-tables
                                :instantiated-cliche instantiated-cliche
                                :position-in-cliche position-in-cliche)) ; ges 5/2

          (when max-gain-exceeded-by-negative
            (setf max-gain gain-from-negative
                  predicate-with-maximum-gain pred
                  max-negated? t
                  variabilization-with-maximum-gain best-negative-variabilization))

          (check-pruning-III-and-prune
           covered-all-pos-tuples find-maximum-literal predicate-with-maximum-gain
           variables variabilization-with-maximum-gain
           (values predicate-with-maximum-gain
                   variabilization-with-maximum-gain
                   max-negated?
                   max-gain))

          (reset-variabilization-flags (gethash (* 100 maximum-new-vars (length variables))
                                                (p-variabilizations pred))))))))


;;;__________________________________________________________________________________________
;;; LOOK-AT-NEGATIVE-VARIABLIZATIONS
;;;
;;; these variabilizations have already been checked for induction, mode etc.
;;; they are guaranteed to have at most one new (unbound) variable
;;; varzn is an abbreviation for variabilization
;;;
;;; returns: 1. new max-gain over all predicates so far for current literal
;;;          2. best variabilization (unchanged if max-gain wasnt topped)
;;;          3. max-negated?: t if max-gain was topped (by a negative variabilization)
;;;          4. covered-all-pos-tuples - to tell find-max-literal if pruning III is appropriate
;;;
;;;  note 2/16/91 currently cliches will not be tried for negative variabilizations
;;;
;;;  revisions
;;;  rv  who    date       reason
;;;  00  glenn  11/01/90   added covered-all-pos-tuples as a return var to tell
;;;                        find-max-literal if pruning III has occured.
;;;                        Also made pruning III check a macro call.
;;;  01  cliff  04/29/91   make stopping criteria reflect FOILs
;;;  02  glenn  05/02/91   deleted conjunction stuff and added cliche info (instantiated-cliche and
;;;                        position-in-cliche
;;;  03  cliff  05/08/91   removed ":old-variables variables" from calls to info-gain.
;;;  04  cliff  05/20/91   Added capacity to graph while learning.

(defun look-at-negative-variabilizations (struct-list
                                          max-gain
                                          pred
                                          pos-tuples
                                          neg-tuples
                                          original-info
                                          use-hash-tables

                                          &key
                                          (instantiated-cliche nil) ; ges 5/2
                                          (position-in-cliche nil)) ; ges 5/2

  (let (gain
        best-varzn
        current-varzn
        max-possible-gain
        (epsilon 0.001)
        covered-all-pos-tuples
        max-negated?) ; ges

    (when *graph-learning*                                           ;; CB(04)
      (negate-predicate-in-induction-graphic (p-name pred)))         ;;

    (do* ((remaining-list struct-list (cdr remaining-list))
          (current-struct (car struct-list) (car remaining-list)))

         ;; terminating criterion
         ((null current-struct) (values max-gain best-varzn max-negated?))

      (setq current-varzn (variabilization-struct-variabilization current-struct))
      (when (variabilization-struct-look-at-negative? current-struct)
        (multiple-value-setq (gain max-possible-gain covered-all-pos-tuples)
                             (info-gain pred
                                        current-varzn
                                        t
                                        pos-tuples
                                        neg-tuples
                                        original-info
                                        use-hash-tables
                                        :instantiated-cliche instantiated-cliche ; ges 5/2
                                        :position-in-cliche position-in-cliche))
        (when *graph-learning*                                                ;; CB(04)
          (update-variablization-in-induction-graphic current-varzn           ;;
                                                      instantiated-cliche     ;;
                                                      position-in-cliche))    ;;

        (when (< max-gain gain) ; ges don't want to override a positive pred
          (setf max-gain gain
                best-varzn current-varzn
                max-negated? t)

          (when *graph-learning*                          ;; CB(04)
            (update-best-literal-in-induction-graphic))   ;;

          ;; pruning III
          (check-pruning-III covered-all-pos-tuples look-at-negative-variabilizations
                             (values max-gain best-varzn t   ; t is for max-negated? == t
                                     covered-all-pos-tuples))) ; ges so find-max-literal knows why it exited

        ;; otherwise try the other 2 types of pruning ...

        (when (find-if #'new-var? current-varzn)
          (if (< max-possible-gain max-gain )                       ; pruning I
            (prune-negations current-varzn (cdr remaining-list) 1)
            (when (< (- max-possible-gain gain) epsilon)            ; pruning II
              (prune-negations current-varzn (cdr remaining-list) 2))))))))

;;;===============================================================================
;;; Is ANY-VARIABLIZATIONS? needed for FOCL to function correctly when
;;; *stopping-criteria-enabled* is nil?  If not remove it and let cliff
;;; worry about it when stopping breaks.
;;;===============================================================================

;;;__________________________________________________________________________________________
;;; ANY-VARIABLIZATIONS?
;;;
;;;  determine if any variabilizations are possible
;;;  rv  who    date     reason
;;;  00  glenn  11/02/90 added to fix problem with trying to encode the number of bits

(defun any-variabilizations? (pred-type tuple-type maximum-new-vars)
  (let ((type-mismatches 0)
        types-processed)
    (dolist (type pred-type (<= type-mismatches maximum-new-vars))
      (cond ((not (member type types-processed))
             (setq types-processed (cons type types-processed))
             (setq type-mismatches
                   (+ (max (- (count-if #'(lambda (t2) (eql type t2)) pred-type)
                              (count-if #'(lambda (t2) (eql type t2)) tuple-type))
                           0)
                      type-mismatches)))
            (t nil)))))



;;;__________________________________________________________________________________________
;;; FIND-LITERAL-EXTENSIONAL
;;;
;;; finds extensional literal with maximum gain-  see find-max for details of params
;;;
;;;  returns 7 values
;;;  1 new-literal             -literal structure with maximum gain
;;;                            -suitble for adding to end of clause
;;;  2 new-vars                -names of newvariables (renamed to be old variables)
;;;  3 new-types               -types of new variables
;;;  4 new-pos-tuples          -next set of positive tuples
;;;  5 new-neg-tuples          -next set of negative tuples
;;;  6 max-gain                -maximum information gain (to compare aginst intensional if necessary)
;;;
;;; returns nil if no literal has positive gain
;;;
;;;  revisions
;;;  rv  who    date       reason
;;;  00  glenn  11/01/90   extended to deal with conjunctions of extensional preds and other preds
;;;                        i.e., thresholded  builtins
;;;  01  glenn  02/16/91   added keyword try-cliches
;;;  02  cliff  04/07/91   no longer accepts bits-available, nor returns literal-bits

(defun find-literal-extensional (original-info             ;passed to info-gain
                                 predicate-being-learned   ;used to detect recursion
                                 variables
                                 variable-types
                                 maximum-new-vars
                                 pos-tuples
                                 neg-tuples
                                 original-vars             ;used to detect recursion
                                 use-hash-tables           ;used by info-gain
                                 gain-to-beat              ;previous high for minimum-gain

                                 &key
                                 try-cliches)

  ;;find-maximum-literal does the real work- its passed alist *extensional-preds*
  (multiple-value-bind (predicate-with-maximum-gain
                        variabilization-with-maximum-gain
                        max-negated?
                        max-gain)
                       (find-maximum-literal original-info
                                             predicate-being-learned
                                             variables
                                             variable-types
                                             maximum-new-vars
                                             pos-tuples
                                             neg-tuples
                                             original-vars
                                             use-hash-tables
                                             *extensional-preds*
                                             gain-to-beat
                                             :try-cliches try-cliches) ; ges 2/16

    (cond ((null predicate-with-maximum-gain) nil)               ;; not enough bits left
          ((eq predicate-with-maximum-gain :fail) :fail)         ;; no predicate has enough gain
          ;;create-literals constructs all the return arguments
          (t (create-literal variables
                             pos-tuples
                             neg-tuples
                             predicate-with-maximum-gain
                             variabilization-with-maximum-gain
                             max-negated?
                             max-gain)))))



;;;__________________________________________________________________________________________
;;; FIND-LITERAL-INTENSIONAL
;;;
;;;  finds intensional literal with maximum gain-  see find-max for details of params
;;;
;;; returns 6 values
;;; 1. new-literal     -literal structure with maximum gain, suitable for adding to end of clause
;;;                    -may be operational conjunction of litearals
;;;                         (if *operationalize-constructive is true) or non-operational
;;; 2. new-vars        -names of newvariables (renamed to be old variables)
;;; 3. new-types       -types of new variables
;;; 4. new-pos-tuples  -next set of positive tuples
;;; 5. new-neg-tuples  -next set of negative tuples
;;; 6. max-gain        -maximum information gain (to compare aginst intensional if necessary)
;;;
;;;  returns nil if no literal has positive gain
;;;
;;;  Note: there is a problem when *stopping-criteria-enabled* and *operationalize-constructive*
;;;        find-literal-intensional may fail even though there is an intensional literal with
;;;        positive gain which could be encoded.  The problem arises because cliff's info-gain
;;;        doesn't account for *operationalize-constructive* when determining the bits to encode
;;;        a literal.  Perhaps operationalize could be more closely intergrated with
;;;        find-maximum-literal when *operationalize-constructive* is true.  Find-maximum-literal
;;;        might return "winning" operational disjunct when *operationalize-constructive* is true.
;;;
;;;  revisions
;;;  rv  who    date       reason
;;;  00  cliff  04/29/91   no longer accepts bits-available, nor returns literal-bits

(defun find-literal-intensional (original-info              ;; passed to info-gain
                                 predicate-being-learned    ;; used to detect recursion
                                 variables
                                 variable-types
                                 maximum-new-vars
                                 pos-tuples
                                 neg-tuples
                                 original-vars              ;; used to detect recursion
                                 use-hash-tables            ;; used by info-gain
                                 gain-to-beat)

  (multiple-value-bind (predicate-with-maximum-gain
                        variabilization-with-maximum-gain
                        max-negated?
                        max-gain)
                       (find-maximum-literal original-info
                                             predicate-being-learned
                                             variables
                                             variable-types
                                             maximum-new-vars
                                             pos-tuples
                                             neg-tuples
                                             original-vars
                                             use-hash-tables
                                             *intensional-preds*
                                             gain-to-beat)
    (cond ((null predicate-with-maximum-gain) nil)               ;;not enough bits left
          ((eq predicate-with-maximum-gain :fail) :fail)         ;;no predicate has enough gain
          ((null *operationalize-constructive*)                  ;;just like extensional
           (create-literal variables
                           pos-tuples
                           neg-tuples
                           predicate-with-maximum-gain
                           variabilization-with-maximum-gain
                           max-negated?
                           max-gain))
          (t ;;similar to operationalization for ebl (except clause is constructed differently)
           ;;;;;return-operational-literal constructs all the return arguments
           (return-operational-literal (list
                                        (if max-negated?
                                          (list 'not (cons (rule-name predicate-with-maximum-gain)
                                                           variabilization-with-maximum-gain))
                                          (cons (rule-name predicate-with-maximum-gain)
                                                variabilization-with-maximum-gain)))
                                       (rule-type predicate-with-maximum-gain)
                                       variabilization-with-maximum-gain
                                       variables
                                       pos-tuples
                                       neg-tuples
                                       original-info
                                       :constructive-induction
                                       use-hash-tables)))))


;;;_______________________________________________________________________________
;;; RETURN-OPERATIONAL-LITERAL
;;;
;;; returns 6 values
;;;
;;; 1. operational-clause  -literal structure with maximum gain- suitable for adding to end of clause
;;;                        -may be operational conjunction of litearals
;;;                             (if *operationalize-constructive is true) or non-operational
;;; 2. new-vars            -names of new variables (renamed to be old variables)
;;; 3. new-types           -types of new variables
;;; 4. new-pos-tuples      -next set of positive tuples
;;; 5. new-neg-tuples      -next set of negative tuples
;;; 6. max-gain            -information gain
;;;
;;;  returns :FAIL if no operationalization can be found with positive gain using the greedy method
;;;  returns NIL if the best operationalization requires too many bits to encode.
;;;
;;;   Note:  Stopping Criteria interacts with EBL.
;;;          The problem when *stopping-criteria-enabled* is that we look for the disjunct
;;;          with the highest gain and operationalize it using a greedy method.  If the
;;;          number of bits required to encode that operationalization is greated than
;;;          bits-available this function fails even though a clause with lower info-gain
;;;          might have required fewer bits to encode, and could have succeeded.
;;;          bits-available might it be used to as selection criteria for clauses, and
;;;          this would be the place to try to get the next best operationalization
;;;          if the current one requires too many bits.
;;;
;;;   Note:  if *operationalize-constructive* is nil the operationalization process
;;;          should stop as soon as operationalization can not improve things. (i.e., the
;;;          gain doesn't increase or the number of negatives excluded doesn't go up...)
;;;
;;;   Note:  return-operational-literal will fail when there aren't enough bits and
;;;          *stopping-criteria-enabled* is not nil.  Perhaps in this situation it should
;;;          really return nil.  This would require modifying operationalize, and perhaps
;;;          the behavior or info gain.
;;;
;;;  revisions
;;;  rv  who    date       reason
;;;  00  glenn  11/28/90   extended to allow for simplification of operationalized clause by deleting
;;;                        literals to improve info-gain. Note that since when *partial-dt-0-gain*
;;;                        theories with 0 gain could potentially be operationalized so
;;;                        return-operational literal now must return fail when after simplifying
;;;                        there is still 0 gain
;;;  01  cliff   3/12/91   adjusted control flow when *stopping-criteria-enable* is nil.
;;;  02  cliff  04/29/91   no longer accepts bits-available, nor returns literal-bits
;;;                        no longer accepts old-types
;;;  03  mike   05/20/91   added operationalize-if-needed capacity
;;;  04  cliff  05/20/91   added capacity to graph while learning. Corrected derivation type insertion.

(defun return-operational-literal (clause          ;; clause to operationalize
                                   param-types     ;; types of parameters to clause
                                   variablization  ;; parameters to clause
                                   old-variables   ;; bound variables
                                   pos-tuples
                                   neg-tuples
                                   original-info   ;; current information
                                   derivation      ;; :ebl or :constructive induction
                                   use-hash-tables)

  (if (eq *refinement* :frontier)                              ;; mp(03)
    (return-operational-literal-if-needed clause               ;;
                                          variablization       ;;
                                          old-variables        ;;
                                          pos-tuples           ;;
                                          neg-tuples           ;;
                                          original-info        ;;
                                          derivation)          ;;


    (let (info-gain
          new-info-gain
          fun
          simplified-clause)

      (when *graph-learning*                                       ;; CB(04)
        (reset-operationalization-frontier clause derivation))     ;;

      (multiple-value-bind  (operational-clause
                             new-vars
                             new-types
                             new-var-alist
                             new-pos-tuples
                             new-neg-tuples)

                            ;;operationalize clause
                            ;;new-var-alist isn't needed here
                            ;;the rest are passed back to find-a-literal
                            (operationalize clause
                                            old-variables
                                            param-types
                                            pos-tuples
                                            neg-tuples
                                            original-info
                                            use-hash-tables)
        new-var-alist  ;; [????]

        (insert-derivation-type-of-every-literal operational-clause derivation)  ;; CB(04)

        (cond ((null operational-clause) :fail) ; had to delete everything
              (t (setq fun (convert-to-prolog-function operational-clause old-variables))
                 (if (equal variablization old-variables)
                   (setq info-gain
                         (info-gain-prove-immediate operational-clause
                                                    fun
                                                    original-info
                                                    pos-tuples
                                                    neg-tuples
                                                    old-variables
                                                    nil ;no cacheing of neg-tuples since theres no clause
                                                    ))
                   (setq info-gain
                         (info-gain-prove operational-clause fun
                                          original-info
                                          pos-tuples
                                          neg-tuples
                                          old-variables
                                          nil ;no cacheing of neg-tuples since theres no clause
                                          )))
                 ;;; simplify and try to improve info-gain
                 (multiple-value-setq
                  (simplified-clause
                   new-pos-tuples
                   new-neg-tuples
                   new-vars
                   new-types
                   new-info-gain)
                  (simplify-operational-clause operational-clause
                                               pos-tuples
                                               neg-tuples
                                               old-variables
                                               new-pos-tuples
                                               new-neg-tuples
                                               new-vars
                                               new-types
                                               info-gain
                                               original-info
                                               use-hash-tables))
                 (if (> new-info-gain 0)
                   (values simplified-clause
                           new-vars
                           new-types
                           new-pos-tuples
                           new-neg-tuples
                           ;recompute gain of operational clause
                           ;it may cover fewer positive
                           new-info-gain)
                   :fail)))))))



;;;===============================================================================
;;; Move The following to globals.lisp  [????]
;;;===============================================================================

(defvar *variabilization-hash-array* (make-hash-table :test #'equal :size 100))


;;;===============================================================================
;;; Move All the functions below to appropriate utility files.  [????]
;;;
;;;   Variable-Utilities.Lisp
;;;   Literal-Structures-Utilities.Lisp
;;;   P-Structure-Utilities.Lisp
;;;   Hash-Table-Utilites.Lisp          (Data-Base-Utilities.lisp ?)
;;;===============================================================================

;;;_______________________________________________________________________________
;;; INSERT-DERIVATION-TYPE-OF-EVERY-LITERAL
;;;
;;;  rv  who    date       reason
;;;  00  cliff  05/20/91   created because derivation type of negated literals was
;;;                        not being set correctly.
;;;  01  cliff  05/20/91   moved from find-literal.lisp

(defun insert-derivation-type-of-every-literal (clause derivation)
  (do ((L clause (literal-next L)))
      ((null L))
    (setf (derivation-type (literal-derivation L)) derivation)
    (if (literal-negated? L)
      (insert-derivation-type-of-every-literal (literal-negated-literals L) derivation))))


;;;_______________________________________________________________________________
;;; CONSTRUCT-LITERAL
;;;
;;;  rv  who    date       reason
;;;  00  glenn  05/04/91   constructs a literal with supplied derivation type

(defun construct-literal (max-negated?
                          predicate-with-maximum-gain
                          literal-vars
                          derivation-type)
  (cond ((is-op-p predicate-with-maximum-gain)
         (construct-literal max-negated?
                            (get-pstruct 'is)
                            (list (car literal-vars)
                                  (cons (is-op-arithmetic-op predicate-with-maximum-gain)
                                        (cdr literal-vars)))
                            derivation-type))
        (max-negated?
         (make-literal
          :negated? t
          :negated-literals (make-literal
                             :predicate-name (p-name predicate-with-maximum-gain)
                             :variablization literal-vars
                             :derivation (make-derivation :type derivation-type))
          :derivation (make-derivation :type derivation-type)))
        (t (make-literal
            :predicate-name (p-name predicate-with-maximum-gain)
            :variablization literal-vars
            :derivation (make-derivation :type derivation-type)))))



;;;_______________________________________________________________________________
;;; CREATE-LITERAL
;;;
;;; returns 6 values
;;; 1. new-literal             -literal structure with maximum gain
;;;                           -suitable for adding to end of clause
;;;                           -may be operational conjunction of litearals
;;;                           (if *operationalize-constructive is true) or non-operational
;;; 2. new-vars                -names of newvariables (renamed to be old variables)
;;; 3. new-types               -types of new variables
;;; 4. new-pos-tuples          -next set of positive tuples
;;; 5. new-neg-tuples          -next set of negative tuples
;;; 6. max-gain                -maximum information gain
;;;
;;;  rv  who    date       reason
;;;  00  cliff  ??/??/??   add literal-bits
;;;  01  cliff  04/29/91   remove literal-bits

(defun create-literal (variables
                       pos-tuples
                       neg-tuples
                       predicate-with-maximum-gain 
                       variabilization-with-maximum-gain
                       max-negated?
                       max-gain)
  
  (let (new-literal 
        new-vars            ; the list of "new" variables renamed to old-vars
        new-types           ; the types of new vars
        new-pos-tuples
        new-neg-tuples
        literal-vars        ; the variables in call with free vars renamed to next old-var
        alist-ignore)       ; new-var-alist is (old-name . bound-name) for free vars

    (multiple-value-setq 
     (literal-vars
      new-vars
      new-types
      alist-ignore)
     (transfer-literal-vars variabilization-with-maximum-gain    ;; variables in call
                            (p-type predicate-with-maximum-gain) ;; type of variables
                            variables                            ;; bound variables
                            (length variables)))
    (setq new-literal 
          (construct-literal max-negated? predicate-with-maximum-gain literal-vars
                             (if (builtin-p predicate-with-maximum-gain)
                               :induction   ;was builtin, who knows why
                               :induction)))
    (setq new-pos-tuples 
	  (generalized-extend-tuples predicate-with-maximum-gain
				     pos-tuples
				     literal-vars
				     max-negated?
				     new-vars
				     variables))
    (setq new-neg-tuples
	  (generalized-extend-tuples predicate-with-maximum-gain
				     neg-tuples
				     literal-vars
				     max-negated?
				     new-vars
				     variables))
	  
    ;;insert before and after, maybe useful later on
    (when *save-examples*    
      (setf (literal-pos new-literal) pos-tuples)
      (setf (literal-neg new-literal) neg-tuples)
      (setf (literal-new-pos new-literal) new-pos-tuples)
      (setf (literal-new-neg new-literal) new-neg-tuples))
    
    (values new-literal
            new-vars
            new-types
            new-pos-tuples
            new-neg-tuples
            max-gain)))



;;;_______________________________________________________________________________
;;; MAKE-NEW-VARS
;;;
;;;  creates n new variables
;;;
;;;  rv  who    date       reason

(defun make-new-vars (n)
  (if (= n 0) nil
      (cons (make-pcvar :id (- n))
            (make-new-vars (- n 1)))))

;;;_______________________________________________________________________________
;;; INFINITE-RECURSIVE-CALL
;;;
;;;   infinite recursion if every var is an original one or a new one (and same pred name)
;;;
;;;  rv  who    date       reason
;;;  00  cliff  05/20/91   moved from find-literal.lisp

(defun infinite-recursive-call (pred predicate-being-learned variabilization original-vars)
       (and (eq pred predicate-being-learned) ;same name
            (every #'(lambda(v)
                       (or (member v original-vars :test #'var-eq)
                           (new-var? v)))
                   variabilization)))



;;;_______________________________________________________________________________
;;; ALL-TYPED-VARIABLIZATIONS
;;;
;;;  rv  who    date       reason

(defun all-typed-variabilizations (bound type new new-type)
  (let ((key (list bound type new new-type)))
    (or (gethash key *variabilization-hash-array*)
        (setf (gethash key *variabilization-hash-array*)
              (all-typed-variabilizations1 bound type new new-type)))))

;;;_______________________________________________________________________________
;;; ALL-TYPED-VARIABLIZATIONS1
;;;
;;;  rv  who    date       reason

(defun all-typed-variabilizations1 (bound type new new-type)
  (let ((old (mapcar #'cons bound type)))
    (all-images #'(lambda(x) (and (not (redundant x new))
                                  (typed-agree new-type x)
                                  (some #'(lambda(x)(not(new-var? (if (consp x) (car x) x)))) x)
                                  (mapcar #'(lambda(x)
                                              (if (consp x) (car x) x)) x)))
                (all-tuples (length new-type)
                            (append new old)))))

;;;_______________________________________________________________________________
;;; ALL-TUPLES
;;;
;;;  rv  who    date       reason

(defun all-tuples (n v)
   (if (= n 0) '(())
      (let ((short (all-tuples (- n 1) v)))
      (mapcan #'(lambda(new)
            (mapcar #'(lambda(old)
                  (cons new old))
               short))
            v))))

;;;_______________________________________________________________________________
;;; REDUNDANT
;;;
;;;  rv  who    date       reason

(defun redundant (vars ordered-new-vars &aux (exit nil) (r nil))
  (do ((v  (reverse ordered-new-vars) (cdr v)))
      ((or exit (null (cdr v))) r)
    (if (and (member (car v) vars :test #'equalp)
             (or (every #'(lambda(x) (member x (member (car v) vars :test #'equalp)
                                             :test #'equalp))
                        (cdr v))
                 (not (every #'(lambda(x) (member x vars :test #'equalp))  (cdr v)))))
      (setq exit t r t))))

;;;_______________________________________________________________________________
;;; TYPED-AGREE
;;;
;;;  rv  who    date       reason

(defun typed-agree (type args &optional (bound nil))
  (if (null args) t
      (if (consp (car args))
        (and (eq (cdr (car args)) (car type))
             (typed-agree  (cdr type) (cdr args) bound))
        (if (assoc (car args) bound :test #'var-eq)
            (and (eq (cdr (assoc (car args) bound :test #'var-eq)) (car type))
                 (typed-agree (cdr type) (cdr args) bound))
            (typed-agree (cdr type) (cdr args)
                         (cons (cons (car args) (car type)) bound))))))






