
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

(defvar *facts-changed*)
(defparameter *compile-with-rule-trace* nil)

;; Clauses are stored on the predicate's plist
(defun get-clauses (pred) (get pred 'clauses))
(defun predicate (relation) (first relation))

(defvar *db-predicates* nil
  "A list of all predicates stored in the database.")

(defvar *rule-trace-string* 
  (make-array '(200)  :adjustable t :fill-pointer 0 :element-type 'character))


(defmacro <- (&rest clause)
  "Add a clause to the data base."
  `(add-clause ',clause))


;;;

(defun clear-db ()
  "Remove all clauses (for all predicates) from the data base."
  (mapc #'clear-predicate *db-predicates*))

(defun clear-predicate (predicate)
  "Remove the clauses for a single predicate."
  (setf (get predicate 'clauses) nil))



(defconstant unbound "Unbound")
(defvar *var-counter* 0)

(defstruct (var (:constructor ? ())
                (:print-function print-var))
  (id (incf *var-counter*))
  (binding unbound))

(defun bound-p (var) (not (eq (var-binding var) unbound)))

(defmacro deref (exp)
  "Follow pointers for bound variables."
  `(progn (loop while (and (var-p ,exp) (bound-p ,exp))
             do (setf ,exp (var-binding ,exp)))
          ,exp))

;;;

(defun unify! (x y)
  "Destructively unify two expressions"
  (cond ((eql (deref x) (deref y)) t)
        ((var-p x) (set-binding! x y))
        ((var-p y) (set-binding! y x))
        ((and (consp x) (consp y))
         (and (unify! (first x) (first y))
              (unify! (rest x) (rest y))))
        (t nil)))


(defparameter *trail* (make-array 100 :fill-pointer 0 :adjustable t))

(defun set-binding! (var value)
  "Set var's binding to value, after saving the variable
  in the trail.  Always returns t."
  (unless (eq var value)
    (vector-push-extend var *trail*)
    (setf (var-binding var) value))
  t)

(defun undo-bindings! (old-trail)
  "Undo all bindings back to a given point in the trail."
  (loop until (= (fill-pointer *trail*) old-trail)
     do (setf (var-binding (vector-pop *trail*)) unbound)))

;;;


(defvar *uncompiled* nil 
        "Prolog symbols that have not been compiled.")

(defun add-clause (clause)
  "Add a clause to the data base, indexed by head's predicate."
  ;; The predicate must be a non-variable symbol.
  (let ((pred (predicate (clause-head clause))))
    (pushnew pred *db-predicates*)
    (pushnew pred *uncompiled*)                         
    (setf (get pred 'clauses)
          (nconc (get-clauses pred) (list clause)))
    pred))

;;;


  "Are the two arguments EQUAL with no unification,
  but with dereferencing?  If so, succeed."

(defun deref-exp (exp)
  "Build something equivalent to EXP with variables dereferenced."
  (if (atom (deref exp))
      exp
      (reuse-cons
        (deref-exp (first exp))
        (deref-exp (rest exp))
        exp)))




(defun prolog-compile-symbols (&optional (symbols *uncompiled*))
  "Compile a list of Prolog symbols.
  By default, the list is all symbols that need it."
    (mapc #'prolog-compile symbols)
    (setf *uncompiled* (set-difference *uncompiled* symbols)))



(defun variables-in (exp)
  "Return a list of all the variables in EXP."
  (unique-find-if-anywhere #'non-anon-variable-p exp))
;;; replaced ? with ? for anon vars
(defun non-anon-variable-p (x)
  (and (variable-p x) (not (eq x '?))))


(defconstant no-bindings '((T . T)))
(defun binding-val (binding) (rest binding))
(defun get-vbinding (var bindings) (assoc var bindings))
(defun extend-bindings (var val bindings)
  (cons (cons var val)
        (if (and (eq bindings no-bindings) (not (null no-bindings)))
            nil
            bindings)))


;;;  added ? for anon-vars
(defun variable-p (x)
  "Is x a variable (a symbol beginning with `?' or '?')?"
  (and (symbolp x) (eql (elt (symbol-name x) 0) #\?)))

;(defun variable-p (x)
;  "Is x a variable (a symbol beginning with `?' or '?')?"
;  (or (and (symbolp x) (equal (elt (symbol-name x) 0) #\?))
;      (pcvar-p x)))

(defmacro >- (&rest goals) `(top-level-prove ',goals))

(defun clauses-with-arity (clauses arity)
  "Return all clauses whose head has given arity."
  (remove-if-not (lambda (clause)
                        (= arity (relation-arity (clause-head clause))))
                 clauses))

(defun relation-arity (relation)
  "The number of arguments to a relation.
  Example: (relation-arity '(p a b c)) => 3"
  (length (args relation)))

(defun args (x) "The arguments of a relation" (rest x))


(defun deref-equal (x y)
  "Are the two arguments EQUAL with no unification,
  but with dereferencing?"
  (or (eql (deref x) (deref y))
      (and (consp x)
           (consp y)
           (deref-equal (first x) (first y))
           (deref-equal (rest x) (rest y)))))

;;;
(defmacro with-undo-bindings (&body body)
  "Undo bindings after each expression in body except the last."
  (if (length=1 body)
      (first body)
      `(let ((old-trail (fill-pointer *trail*)))
         ,(first body)
         ,@(loop for exp in (rest body)
                 collect '(undo-bindings! old-trail)
                 collect exp))))

;;; added ? as an anon var
(defun deref-copy (exp)
  "Copy the expression, replacing variables with new ones.
  The part without variables can be returned as is."
  (sublis (mapcar (lambda (var) (cons var (?)))
                  (unique-find-if-anywhere #'var-p exp))
          exp))

(defun unbound-var-p (exp)
  "Is EXP an unbound var?"
  (and (var-p exp) (not (bound-p exp))))

;;;



(defun prolog-compiler-macro (name)
  "Fetch the compiler macro for a Prolog predicate."
  ;; Note NAME is the raw name, not the name/arity
  (get name 'prolog-compiler-macro))

(defmacro def-prolog-compiler-macro (name arglist &body body)
  "Define a compiler macro for Prolog."
  `(setf (get ',name 'prolog-compiler-macro)
          #'(lambda ,arglist .,body)))

#| These are replaced later, but this section can be evaled
   To get a non-optimizing compiler
(def-prolog-compiler-macro = (goal body cont bindings)
  (let ((args (args goal)))
    (if (/= (length args) 2)
        :pass
        `(if ,(compile-unify (first args) (second args))
             ,(compile-body body cont bindings)))))

(defun compile-unify (x y)
  "Return code that tests if var and term unify."
  
  `(unify! ,(compile-arg x) ,(compile-arg y )))

(defun non-anon-variable-p (x)
  "Hack so anonymous variables are declared"
  (and (variable-p x) ))


(defun compile-arg (arg &rest ignore)
  "Generate code for an argument to a goal in the body."
  (cond ((variable-p arg) arg)
        ((not (has-variable-p arg)) `',arg)
        ((proper-listp arg)
         `(list .,(mapcar #'compile-arg arg)))
        (t `(cons ,(compile-arg (first arg))
                  ,(compile-arg (rest arg))))))

|#

(defun compile-unify (x y bindings)
  "Return 2 values: code to test if x and y unify,
  and a new binding list."
  (cond
    ;; Unify constants and conses:                       ; Case
    ((not (or (has-variable-p x) (has-variable-p y)))    ; 1,2
     (values (equal x y) bindings))
    ((and (consp x) (consp y))                           ; 3
     (multiple-value-bind (code1 bindings1)
         (compile-unify (first x) (first y) bindings)
       (multiple-value-bind (code2 bindings2)
           (compile-unify (rest x) (rest y) bindings1)
         (values (compile-if code1 code2) bindings2))))
    ;; Here x or y is a variable.  Pick the right one:
    ((variable-p x) (compile-unify-variable x y bindings))
    (t              (compile-unify-variable y x bindings))))

(def-prolog-compiler-macro not (goal body cont bindings rule-trace)
  (let ((args (args goal)))
    (if (and (length=1  args)
             *compile-with-rule-trace*)
      :pass ;;macro doesn't do augment explanation
      (let ((and-code (compile-body (args goal) 
                                    '(lambda(rt)(declare (ignore rt))
                                       (throw 'not-fails t))
                                    bindings rule-trace)))
            `(unless (catch 'not-fails ,and-code)
               ,(if body (compile-body  body cont bindings rule-trace)
                    `(funcall cont 
			     ,(if *compile-with-rule-trace*
				   `(augment-rule-trace Rule-trace
						       (cons 'not (deref-exp ',args))")")
				 'rule-trace))))))))

(defun compile-if (pred then-part)
  "Compile a Lisp IF form. No else-part allowed."
  (case pred
    ((t) then-part)
    ((nil) nil)
    (otherwise `(if ,pred ,then-part))))

;;; anon vars changed to ?

(defun compile-unify-variable (x y bindings)
  "X is a variable, and Y may be."
  (let* ((xb (follow-binding x bindings))
         (x1 (if xb (cdr xb) x))
         (yb (if (variable-p y) (follow-binding y bindings)))
         (y1 (if yb (cdr yb) y)))
    (cond                                                 ; Case:
      ((or (eq x '?) (eq y '?)) (values t bindings))      ; 12
      ((not (and (equal x x1) (equal y y1)))              ; deref
       (compile-unify x1 y1 bindings))
      ((find-anywhere x1 y1) (values nil bindings))       ; 11
      ((consp y1)                                         ; 7,10
       (values `(unify! ,x1 ,(compile-arg y1 bindings))
               (bind-variables-in y1 bindings)))
      ((not (null xb))
       ;; i.e. x is an ?arg variable
       (if (and (variable-p y1) (null yb))
           (values 't (extend-bindings y1 x1 bindings))   ; 4
           (values `(unify! ,x1 ,(compile-arg y1 bindings))
                   (extend-bindings x1 y1 bindings))))    ; 5,6
      ((not (null yb))
       (compile-unify-variable y1 x1 bindings))
      (t (values 't (extend-bindings x1 y1 bindings)))))) ; 8,9

;;; anon vars changed to ?
(defun compile-arg (arg bindings)
  "Generate code for an argument to a goal in the body."
  (cond ((eq arg '?) '(?))
        ((variable-p arg)
         (let ((binding (get-vbinding arg bindings)))
           (if (and (not (null binding))
                    (not (eq arg (binding-val binding))))
             (compile-arg (binding-val binding) bindings)
             arg)))
        ((not (find-if-anywhere #'variable-p arg)) `',arg)
        ((proper-listp arg)
         `(list .,(mapcar (lambda (a) (compile-arg a bindings))
                          arg)))
        (t `(cons ,(compile-arg (first arg) bindings)
                  ,(compile-arg (rest arg) bindings)))))


(defun has-variable-p (x)
  "Is there a variable anywhere in the expression x?"
  (find-if-anywhere #'variable-p x))

(defun proper-listp (x)
  "Is x a proper (non-dotted) list?"
  (or (null x)
      (and (consp x) (proper-listp (rest x)))))

(defun bind-variables-in (exp bindings)
  "Bind all variables in exp to themselves, and add that to
  bindings (except for variables already bound)."
  (loop for var in (variables-in exp) do
        (unless (get-vbinding var bindings)
          (setf bindings (extend-bindings var var bindings))))
  bindings)

(defun follow-binding (var bindings)
  "Get the ultimate binding of var according to bindings."
  (let ((b (get-vbinding var bindings)))
    (if (eq (car b) (cdr b))
        b
        (or (follow-binding (cdr b) bindings)
            b))))

;;;

(defvar *predicate* nil
  "The Prolog predicate currently being compiled")

(defvar *last-defun*)

;;; changed anon vars to ?

(defun bind-unbound-vars (parameters exp)
  "If there are any variables in exp (besides the parameters)
  then bind them to new vars."
  (let ((exp-vars (set-difference (variables-in exp)
                                  parameters)))
    (if exp-vars
        `(let ,(mapcar (lambda (var) `(,var (?)))
                       exp-vars)
           ,exp)
        exp)))

(defun self-cons (x) (cons x x))
(defun make-parameters (arity)
  "Return the list (?arg1 ?arg2 ... ?arg-arity)"
  (loop for i from 1 to arity
        collect (symbol '?arg i)))

(defun make-predicate (symbol arity)
  "Return the symbol: symbol/arity"
  (or (get symbol 'prolog-predicate-name)
      (setf (get symbol 'prolog-predicate-name)
	    (symbol symbol '/ arity))))

(defun bind-new-variables (bindings goal)
  "Extend bindings to include any unbound variables in goal."
  (let ((variables (remove-if (lambda (v) (assoc v bindings))
                              (variables-in goal))))
    (nconc (mapcar #'self-cons variables) bindings)))

(defvar *prolog-trace-indent* 0)

(defun print-var (var stream depth)
  (deref var)
  (if (or (and *print-level*
               (>= depth *print-level*))
          (var-p var))
      (format stream "?~a" (var-id var))
      (write var :stream stream)))




;;; maintain a list of the traced predicates

(defvar *traced-predicates* nil)

;;;  accepts one or more predicates to trace (accumulated into predicates) and
;;;  recompiles each of the predicates to add in the necessary code for tracing
;;;  *traced-predicates* is updated to record any new predicates that are traced

(defun p-trace (&rest predicates)
  (dolist (pred predicates)
    (pushnew pred *traced-predicates*)))

;;;  accepts one or more predicates to untrace (accumulated into predicates) and
;;;  recompiles each of them normally to remove the excess trace code.  Note if
;;;  no predicates are supplied then all predicates are untraced (1)

(defun p-untrace (&rest predicates)
  (if (null predicates) (setq  *traced-predicates* nil) ; (1)
      (setq *traced-predicates* (set-difference *traced-predicates* predicates))))

;;;  accepts the name of a predicate (symbol) and optionally the remaining clauses
;;;  associated with the predicate (clauses are processed by arity) and compiles
;;;  the clauses with the same arity of the first clause adding in the necessary 
;;;  code to perform a trace and then recursively compiles the rest of the clauses.
;;;  compile-predicate adds in the code to do the actual tracing

(defun prolog-compile (symbol &optional
                              (clauses (get-clauses symbol)))
  "Compile a symbol; make a separate function for each arity."
  (unless (null clauses)
    (let* ((arity (relation-arity (clause-head (first clauses))))
           (clauses-with-arity (clauses-with-arity clauses arity))
           (other-clauses (remove-if #'(lambda (x) (member x clauses-with-arity))
                                     clauses)))
      ;; Compile the clauses with this arity
      (compile-predicate
        symbol arity clauses-with-arity)
      ;; Compile all the clauses with any other arity
      (prolog-compile symbol other-clauses))))




;;;  code to deal with maintaining rule trace

;;;  note all compiled predicates will have an additional variable for the rule trace so
;;;  we need to adjust the top-level-call (1)

(defun run-prolog (procedure cont rule-trace)
  "Run a 0-ary prolog procedure with a given continuation."
  ;; First compile anything else that needs it
  (prolog-compile-symbols)
  ;; Reset the trail and the new variable counter
  (setf (fill-pointer *trail*) 0)
  (setf *var-counter* 0)
  ;; Finally, call the query
  (format t "~%running ~a " procedure)
  (catch 'top-level-prove
    (funcall procedure cont rule-trace))) ; (1) start off rule trace

;;;  add in rule trace
(defun show-prolog-vars/2 (var-ids vars cont rule-trace) ; (1)
  "Display the variables, and prompt the user to see
  if we should continue.  If not, return to the top level."
  (if (null vars)
      (format t "~&Yes")
      (loop for name in var-ids
            for var in vars do
            (format t "~&~a = ~a" name (deref-exp var))))
  (if (continue-p rule-trace)
      (funcall cont rule-trace) ; (2)
      (throw 'focl-prove nil)))

;;; add in rule trace as a parameter to the continuation function

(defun top-level-prove (goals)
  "Prove the list of goals by compiling and calling it."
  ;; First redefine top-level-query
  (clear-predicate 'top-level-query)
  (let ((vars (delete '? (variables-in goals))))
    (add-clause `((top-level-query)
                  ,@goals
                  (show-prolog-vars ,(mapcar #'symbol-name vars)
                                    ,vars))))
  ;; Now run it
  (run-prolog 'top-level-query/0 
              #'(lambda (rt) rt nil)
              (init-trace)) 
  (format t "~&No.")
  (values))

;;;  add in rule-trace to all pre-defined predicates


(defun call/1 ( cont rule-trace goal) ; (1)
  "Try to prove goal by calling it."
  (deref goal)
  (apply (make-predicate (first goal) 
                         (length (args goal)))
         cont
         rule-trace
         (args goal)
         ))




(defun not/1 (cont rule-trace relation) ; (1)
  "Negation by failure: If you can't prove G, then (not G) true."
  ;; Either way, undo the bindings.
  (with-undo-bindings
    (call/1  #'(lambda (rt) (declare (ignore rt)) (return-from not/1 nil)) rule-trace relation)
    (funcall cont (augment-rule-trace rule-trace 
                                     (list 'not relation)))))

(defun setof/3 (cont rule-trace exp goal result)
  "Find all unique solutions to GOAL, and for each solution,
  collect the value of EXP into the list RESULT."
  ;; Ex: Assume (p 1) (p 2) (p 3).  Then:
  ;;     (setof ?x (p ?x) ?l) ==> ?l = (1 2 3)
  (let ((answers nil)
        (saved-traces (make-array '(200)  :adjustable t :fill-pointer 0 :element-type 'character)))
    (call/1 (lambda (rt) 
              (let ((answer (deref-copy (deref-exp exp))))
                (unless (member answer answers :test #'deref-equal)
                  (when *maintain-prolog-rule-trace* 
                    (augment-rule-trace rt ")")
                    (with-output-to-string (s saved-traces)
                      (prin1 (fix-rule-trace (read-from-string (subseq *rule-trace-string* rule-trace)))
                             s)
                      ))
                  (push answer answers)
                  )))
            rule-trace  goal)
    (if (and (not (null answers))
             (unify! result answers))
      (funcall cont (augment-rule-trace rule-trace "(" (list 'setof exp goal result)
                                        saved-traces ")")))))

(defun is/2 (cont rule-trace var exp)
  ;; Example: (is ?x (+ 3 (* ?y (+ ?z 4))))
  ;; Or even: (is (?x ?y ?x) (cons (first ?z) ?l))
  (let ((exit-cont #'(lambda (rt) (prolog-trace 'exit 'is var exp) (funcall cont rt))))
    (prolog-trace 'call 'is var exp)
    (if (and (not (find-if-anywhere #'unbound-var-p exp))
             (unify! var (eval (deref-exp exp))))
      (funcall exit-cont
               (augment-rule-trace rule-trace
                                   " ("
                                   (list 'is (deref-exp var) (deref-exp exp))
                                   ")")))
    (prolog-trace 'fail 'is var exp)))

;;;





;;;  check prolog compiler macros and see if they need additions later for now just pass
;;;  the rule trace through

(def-prolog-compiler-macro internal-= (goal body cont bindings rule-trace) ; (1)
  "Compile a goal which is a call to =."
  (let ((args (args goal)))
    (if (/= (length args) 2)
        :pass ;; decline to handle this goal
        (multiple-value-bind (code1 bindings1)
            (compile-unify (first args) (second args) bindings)
          (compile-if
           code1
           (compile-body body cont bindings1 
                         rule-trace)))))) ; (2)

(defun make-= (x y) `(internal-= ,x ,y))




;;;

;;;  accepts the name of a predicate (symbol) its arity and a list of the clauses 
;;;  associated with the predicate that have arity as its arity and produces and compiles
;;;  a defun to execute the predicate.  The additions made to add in the trace feature 
;;;  are mainly in maybe-add-undo-bindings.  The changes here are
;;;   (1) passed exit-cont as the name of the continuation used by compile-clause
;;;   (2) passed args symbol and parameters for maybe-add-undo-bindings

(defun compile-predicate (symbol arity clauses)
  "Compile all the clauses for a given symbol/arity
  into a single LISP function."
  (let* ((*predicate* (make-predicate symbol arity))  ;***
         (parameters (make-parameters arity)))
    (terpri)
    (compile
      (eval (setq *last-defun*
       (write
       `(defun ,*predicate* (cont rule-trace . ,parameters)
	          #+:allegro (declare (ignore-if-unused .,parameters))
             .,(maybe-add-undo-bindings
               (mapcar (lambda (clause)
                         (bind-unbound-vars 
                           parameters
                           (compile-clause parameters clause
                                           'exit-cont))) 
                       clauses)
               symbol parameters)) ; (2)
       :pretty t)
       )))))

;;;  accepts a set of compiled expressions (compiled-exps) the name of the predicate being
;;;  compiled (symbol) and the parameters of the predicate (parameters) and creates the
;;;  body of the defun that will be the compilation of the predicate with calls to 
;;;  undo-bindings! and prolog-trace where appropriate.  The additions necessary to 
;;;  implement the trace are:
;;;  (1) created a lambda for the exit-cont which performs a call to prolog-trace before
;;;     funcalling the continuation (cont)
;;;  (2) added a "call" prolog-trace before the first set of unifies
;;;  (3) added a "redo" prolog-trace just after each undo-bindings! (i.e., before each 
;;;     set of unifications starting with the second unification set
;;;  (4) added a "fail" prolog-trace at the end of the body of the let

;;; mp 04/27/91  Used *compile-with-rule-trace to not include trace information
(defun maybe-add-undo-bindings (compiled-exps symbol parameters)
  "Undo any bindings that need undoing.
  If there are any, bind the trail before we start."
  (if (length=1 compiled-exps)
    (if *compile-with-rule-trace*
      `((let ((cont #'(lambda (rt) (prolog-trace 'exit ',symbol ,@parameters)
                       (funcall cont rt)))
              (prolog-trace-indent *prolog-trace-indent*))
          (prolog-trace 'call ',symbol ,@parameters)
          ,@compiled-exps
          (setq *prolog-trace-indent* prolog-trace-indent)
          (prolog-trace 'redo ',symbol ,@parameters)))
      compiled-exps)
    (if *compile-with-rule-trace*
      `((let ((old-trail (fill-pointer *trail*))
              (cont #'(lambda (rt) (prolog-trace 'exit ',symbol ,@parameters)
                       (funcall cont rt)))
              (prolog-trace-indent *prolog-trace-indent*))
          (prolog-trace 'call ',symbol ,@parameters)
          ,(first compiled-exps)
          ,@(loop for exp in (rest compiled-exps)
                  collect '(undo-bindings! old-trail)
                  collect '(setq *prolog-trace-indent* prolog-trace-indent)
                  collect `(prolog-trace 'redo ',symbol ,@parameters) ; (3)
                  collect exp)))
      `((let ((old-trail (fill-pointer *trail*)))
          ,(first compiled-exps)
          ,@(loop for exp in (rest compiled-exps)
                  collect '(undo-bindings! old-trail)
                  collect exp))))))


(defun update-rule-trace (clause &optional rule-trace parms)
  (let ((clause-head (if (listp (car clause)) (car clause) clause)))
    (add-to-rule-trace rule-trace 
                          " (" 
                          `(list ',(car clause-head) 
                                 ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                           parms)))))

(defun update-rule-trace-exit (clause rule-trace parms)
  (let ((clause-head (if (listp (car clause)) (car clause) clause)))
    (add-to-rule-trace rule-trace 
                          " (" 
                          `(list ',(car clause-head) 
                                 ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                           parms))
                          ")")))

(defparameter *maintain-prolog-rule-trace* nil)

(defun add-to-rule-trace (rule-trace &rest args)
  (if *compile-with-rule-trace*
    `(augment-rule-trace ,@(if (and (listp rule-trace) 
                                    (eql (car rule-trace) 'augment-rule-trace))
                        (cdr rule-trace)
                        (list rule-trace))
                    ,@args)
    rule-trace))

(defun augment-rule-trace (rt &rest args)
  (when *maintain-prolog-rule-trace*
    (setf (fill-pointer *rule-trace-string*) rt)
    (mapc #'(lambda(string)
              (if (stringp string)
                (dotimes (i (length string))
                  (vector-push-extend (char string i) *rule-trace-string*))
                (with-output-to-string (s *rule-trace-string*)
                  (prin1 string s))))
          args)
    (fill-pointer *rule-trace-string*)))

;;;  added symbol to facilitate maintainence of rule-trace

(defun compile-clause (parms clause cont)
  "Transform away the head, and compile the resulting body."
  (compile-body
   (nconc
    (mapcar #'make-= parms (args (clause-head clause)))
    (clause-body clause))
   cont
   (mapcar #'self-cons parms)
   (update-rule-trace clause 'rule-trace parms)))

(defun compile-body (body cont bindings rule-trace)
  "Compile the body of a clause."
  (cond
   ((null body)
    (if *compile-with-rule-trace* 
	`(lambda (rt)
	   (funcall ,cont ,(add-to-rule-trace 'rt ")")))
      cont))				; (1) 
   ((eq (first body) '!)		;*** 
    `(progn ,(compile-body (rest body) cont bindings rule-trace)
	    (return-from ,*predicate* nil)))
   (t (let* ((goal (first body))
	     (macro (prolog-compiler-macro (predicate goal)))
	     (macro-val 
	      (if macro 
		  (funcall macro goal (rest body) cont bindings rule-trace))))
	(if (and macro (not (eq macro-val :pass)))
	    macro-val
	  `(,(make-predicate (predicate goal)
			     (relation-arity goal))

	    ,(if (null (rest body))
		 (if *compile-with-rule-trace* 
		     `(lambda (rt)
			(funcall ,cont ,(add-to-rule-trace 'rt ")")))
		   cont)
	       `(lambda (rt)
		  ,(compile-body 
		    (rest body) cont
		    (bind-new-variables bindings goal)
		    'rt)))
	    ,rule-trace
	    . ,(mapcar #'(lambda (arg)
                           (compile-arg arg bindings))
                       (args goal))))))))

(defun continue-p (trace)
  "Ask user if we should continue looking for solutions."
  (princ " ")
  (case (read-char)
    (#\? (print-trace trace) (continue-p trace))
    (#\; t)
    (#\. nil)
    (#\newline (continue-p trace))
    (otherwise
      (format t " Type ; to see more or . to stop or ? to trace")
      (continue-p trace))))

(defun print-trace (trace)
  (augment-rule-trace trace ")")
  (mapc #'(lambda(tr)
            (print-rule-trace (fix-rule-trace tr)))
        (read-from-string *rule-trace-string*)))

(defun print-rule-trace (trace &optional (indent 1) (level 1) &aux tr)
  (cond ((listp (car trace)) (setq tr (instantiate-english(car trace)))
         (when tr (format t "~&~VT~a. ~{~a ~}" indent level tr))
         (mapc #'(lambda (tr) (print-rule-trace tr (+ indent 3) (+ level 1))) (cdr trace)))
        ((eq (car trace) 'not) 
         (setq tr (instantiate-english (second trace)))
         (when tr (format t "~&~VT~a. NOT ~a. ~{~a ~}" indent level (+ level 1) tr))
         (mapc #'(lambda(g)(print-rule-trace (list g) (+ indent (if (> level 9) 8 7)) (+ level 1)))
               (cddr trace)))
        (t (format t "~&~VT~a. ~a" indent level trace)))) 

(defun instantiate-english(literal)
  (let* ((head (car literal))
        (args (cdr literal))
        (s (get-pstruct head))
        (vars (cond (s (pred-vars s))(t '(?X ?Y))))
        (english (cond ((eq head 'is) '(?X is equal to ?y))
                       ((eq head '=) nil)
                  ((or (builtin-p s)(rule-p s))
                   (pred-questions s))
                  ((pred-p s)
                   (cadr (assoc :fact (pred-questions s)))))))
    
        (mapc #'(lambda(old new)
                  (setq english (subst new old english :test #'equalp)))
              vars args)
        english))
        

(defun prolog-trace (kind predicate &rest args)
  (cond ((or (eq *traced-predicates* :all)
             (member predicate *traced-predicates*))
         (if (member kind '(call redo))
           (incf *prolog-trace-indent* 3))
         (format t "~&~VT~a. ~a ~a:~{ ~a~}"
                 *prolog-trace-indent* (/ (1- *prolog-trace-indent*) 3) kind predicate args)
         (if (member kind '(fail exit))
           (decf *prolog-trace-indent* 3))
         )
        (t nil))
  nil)

;;; search backwards for bindings

(defun get-vbinding-from-trail (var &optional (fill-pointer (fill-pointer *trail*)) 
                                   &aux cv)
  (cond ((zerop fill-pointer) nil)
        ((and (var-p (setq cv (aref *trail* (1- fill-pointer)))) (eql (var-id cv) var))
         (var-binding cv))
        (t (get-vbinding-from-trail var (1- fill-pointer)))))

(defun fix-rule-trace (trace)
  
  (cond ((and (symbolp trace) (eql (position #\? (symbol-name trace)) 0))
         (or (get-vbinding-from-trail (read-from-string (subseq (symbol-name trace) 1)))
             trace))
        ((not (consp trace))
         trace)
        ((proper-listp trace)
         (mapcar #'fix-rule-trace trace))
        (t (reuse-cons (fix-rule-trace (car trace)) (fix-rule-trace (cdr trace)) trace))))


;;; supplied code
(DEFUN =/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL '= ?ARG1 ?ARG2)
    (when (UNIFY! ?ARG2 ?ARG1)
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST '= (DEREF-EXP ?ARG1) (DEREF-EXP ?ARG2))
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '= ?ARG1 ?ARG2)))

;;; supplied code

(DEFUN ==/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  "Are the two arguments EQUAL with no unification,
  but with dereferencing?  If so, succeed."
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL '== ?ARG1 ?ARG2)
    (when (DEREF-EQUAL ?ARG2 ?ARG1)
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST '== (DEREF-EXP ?ARG1) (DEREF-EXP ?ARG2))
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '== ?ARG1 ?ARG2)))



(DEFUN >/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '> ?ARG1 ?ARG2) (FUNCALL CONT RT))))
        (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '> ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (> ?arg1 ?arg2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST '> ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '> ?ARG1 ?ARG2)))


(DEFUN math-=/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  "Is arg1 = arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT 'math-= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '> ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (=  ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST 'math-= ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL 'math-= ?ARG1 ?ARG2)))


(DEFUN fail/0 (CONT RULE-TRACE)
  cont rule-trace
  (PROLOG-TRACE 'CALL 'FAIL)
  (PROLOG-TRACE 'FAIL 'FAIL))


(DEFUN </2 (CONT RULE-TRACE ?ARG1 ?ARG2)
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '< ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '< ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (< ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST '< ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL '< ?ARG1 ?ARG2)))

(DEFUN <=/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '<= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL '<= ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (<= ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST '<=  ?ARG1 ?ARG2)
                              ")"))
)
    (PROLOG-TRACE 'FAIL '<= ?ARG1 ?ARG2)))

(DEFUN >=/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  "Is arg1 >arg2"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT '>= ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL '>= ?ARG1 ?ARG2)
    (when (and (numberp ?arg1)(numberp ?arg2)
               (>=  ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST '>= ?ARG1 ?ARG2)
                              ")"))
)
    (PROLOG-TRACE 'FAIL '>= ?ARG1 ?ARG2)))


;; 5/13/91 -ges  added these builtin functions to support non-numeric builtin functionality

;;;  move this to the cruel-compiler.lisp

;;; supplied code
(DEFUN eql/2 ( CONT RULE-TRACE ?ARG1 ?ARG2)
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT 'eql ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (PROLOG-TRACE 'CALL 'eql ?ARG1 ?ARG2)
    (when (UNIFY! ?ARG2 ?ARG1)
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST 'EQL (DEREF-EXP ?ARG1) (DEREF-EXP ?ARG2))
                              ")"))
      )
    (PROLOG-TRACE 'FAIL 'eql ?ARG1 ?ARG2)))


;;; a non-numeric relation builtin

(DEFUN string-lessp/2 (CONT RULE-TRACE ?ARG1 ?ARG2)
  "Is (string-lessp arg1 arg2)"
  (LET ((CONT #'(LAMBDA (RT) (PROLOG-TRACE 'EXIT 'string-lessp ?ARG1 ?ARG2) (FUNCALL CONT RT))))
    (setq ?arg1 (deref-exp ?ARG1) ?arg2 (deref-exp ?ARG2))
    (PROLOG-TRACE 'CALL 'string-lessp ?ARG1 ?ARG2)
    (when (and (stringp ?arg1)(stringp ?arg2)
               (string-lessp ?ARG1 ?ARG2))
      (FUNCALL CONT
               (AUGMENT-RULE-TRACE RULE-TRACE
                              " ("
                              (LIST 'string-lessp ?ARG1 ?ARG2)
                              ")"))
      )
    (PROLOG-TRACE 'FAIL 'string-lessp ?ARG1 ?ARG2)))



;;; got to figure out a way to get rid of this eval if possible



;;;  accepts a list of goals with vars either in focl format or in norvig format (i.e., 
;;;  ?a and ?a are both acceptable

(defun prove (goals &optional (pred nil))
  (cond((cdr goals)
        (let* ((parameter-bindings (collect-parameter-bindings goals nil))
               (i (sublis parameter-bindings goals :test #'var-eq))
               (*prolog-trace-indent* 1)
               (cont
                #'(lambda (rt)
                    (throw 'reg-prove 
                           (if pred t
                               (values (deref-exp i)
                                       (augment-rule-trace rt ")")))))))
	  (setf (fill-pointer *trail*) 0)
	  (setf *var-counter* 0)
          (catch 'reg-prove
            (eval (build-args-and-cont cont goals (init-trace) parameter-bindings)))))
       (t (prove-goal (car goals) pred))))

(defun build-args-and-cont (cont goals rt parameter-bindings)
  (let* ((goal (car goals))
         (goal-predicate (make-predicate (car goal) (length (cdr goal)))))
    `(,goal-predicate 
      ',(if (null (cdr goals))
	    cont
	  `(lambda (rt) 
	     ,(build-args-and-cont cont (cdr goals) 'rt parameter-bindings)))
      ,rt
      . ,(sublis parameter-bindings (mapcar #'build-arg (cdr goal)) :test #'var-eq))))

(defun prove-goal(goal &optional (pred nil))
  (let* ((parameter-bindings (collect-parameter-bindings goal nil))
         (*prolog-trace-indent* 1)
         (goal-predicate (symbol-function(make-predicate (car goal) (length (cdr goal)))))
         (i (sublis parameter-bindings goal :test #'var-eq))
         (cont
          #'(lambda (rt) 
               (throw 'reg-prove 
                      (if pred t
                      (values (deref-exp i)
                              (augment-rule-trace rt ")")))))))
      (setf (fill-pointer *trail*) 0)
      (setf *var-counter* 0)
    (catch 'reg-prove
      (apply goal-predicate cont (init-trace)  (cdr i) ))))



(defun prove-function? (function arg-list negation &optional (vlist (create-parameter-bindings arg-list nil)))
    (setf (fill-pointer *trail*) 0)
    (setf *var-counter* 0)
  (if negation (not (catch 'reg-prove
		       (apply function #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t))
			      nil  vlist)))
    (catch 'reg-prove
      (apply function #'(lambda (rt) (declare (ignore rt)) (throw 'reg-prove t))
	     nil  vlist))))

(defun setof-function (function vlist outlist)
  (let ((the-answer nil))
    (apply function #'(lambda (rt) (declare (ignore rt)) (pushnew (deref-exp outlist) the-answer :test #'equal))
	   nil  vlist)
    the-answer))



(defun init-trace()
  (when *maintain-prolog-rule-trace* 
  (setf (fill-pointer *rule-trace-string*) 0)
  (vector-push #\( *rule-trace-string*)
  (fill-pointer *rule-trace-string*)))


(defun insert-derefs (exp)
  (cond ((variable-p exp) (list 'deref exp))
        ((proper-listp exp) (cons 'list (mapcar #'insert-derefs exp)))
        ((consp exp) (reuse-cons (insert-derefs (car exp)) (insert-derefs (cdr exp)) exp))
        (t `',exp)))


(defun build-arg (exp)
  (cond ((variable-p exp) exp)
        ((proper-listp exp) (cons 'list (mapcar #'build-arg exp)))
        ((consp exp) (reuse-cons (build-arg (car exp)) (build-arg (cdr exp)) exp))
        (t `',exp)))


;;; collects bindings of variables in an arg list.  Note variables may appear nested
;;; in the arg list, i.e., it works sort of like unify only on a single list and it 
;;; augments the bindings by creating a new variable and associating it with each 
;;; variable it sees for the first time

(defun collect-parameter-bindings (args bindings)
  (cond ((variable-p args) (if (assoc args bindings) bindings (cons (cons args (?)) bindings)))
        ((consp args)
         (collect-parameter-bindings (cdr args)
                               (collect-parameter-bindings (car args) bindings)))
        (t bindings)))



;;; like collect except arglist is guarenteed to be a tuple
;;;returns new-binding list

(defun create-parameter-bindings (args bindings &aux temp)
  (mapcar #'(lambda(arg)
	      (if (variable-p arg)
		  (if (setq temp (assoc arg bindings))
		      (cdr temp)
		    (prog1  (setq temp (?))(push (cons arg temp) bindings)))
		arg))
	  args))

(defun create-parameter-bindings-from-tuple (args tuple &optional (bindings nil) &aux temp out)
  (setq out (mapcar #'(lambda(arg)
			(if (variable-p arg)
			    (if (new-var? arg)
				(if (setq temp (assoc arg bindings))
				    (cdr temp)
				  (prog1 (setq temp (?))(push (cons arg temp) bindings)))
			      (nth (pcvar-id arg) tuple))
			  arg))
		    args)
	)
  (values out bindings))
  


(defun compile-focl-all ()
  (compile-focl-rules)
  (compile-focl-facts))

;;;  compile all the def-rules

(defun compile-focl-rules ()
  (focl-prolog-compile-symbols (mapcar #'car *intensional-preds*)))

(defun focl-prolog-compile-symbols (&optional (symbols *uncompiled*))
  "Compile a list of Prolog symbols.
  By default, the list is all symbols that need it."
    (mapc #'focl-prolog-compile symbols)
    (setf *uncompiled* (set-difference *uncompiled* symbols)))


;;; version that deals with def-rule
;;; clauses will contain focl clause structures

(defun focl-prolog-compile (symbol &optional
                              (clauses (get-focl-clauses symbol)))
  "Compile a symbol; make a separate function for each arity."
  (unless (null clauses)
    (let* ((arity (focl-clause-arity (first clauses)))
	   )
      ;; Compile the clauses with this arity
      (focl-compile-predicate
        symbol arity clauses))))

;(defun get-focl-clauses (rule) (rule-clauses (get rule 'rule)))

(defun get-focl-clauses (rule) (get rule 'clauses))

;;;  note these are like norvigs clause-head and clause-body but focl uses a clause structure

(defun focl-clause-head (clause)
  (car clause))

(defun focl-clause-body (clause)
  (cdr clause))


(defun focl-clause-arity (clause)
  (length (cdr (focl-clause-head clause))))

(defun focl-clause-parameters (clause)
  (cdr (focl-clause-head clause)))


;;; modified to deal with focl clause structures

(defun focl-clauses-with-arity (clauses arity)
  "Return all clauses whose head has given arity."
  (remove-if-not 
   #'(lambda (clause) (= arity (focl-clause-arity clause)))
   clauses))

(defun compile-unless-macl(x)
 #+:ccl x
 #-:ccl (compile x)
)

(defun focl-compile-predicate (symbol arity clauses &optional (*predicate* (make-predicate symbol arity)))
  "Compile all the clauses for a given symbol/arity
  into a single LISP function."
  (let* ((parameters (make-parameters arity))
         #+:ccl (*warn-if-redefine-kernel* nil)
         #+:ccl (*warn-if-redefine* nil)
         #+:ccl (*compile-definitions* t)
	 #+:ccl (*compiler-warnings* nil)
	 #+:ccl (*fasl-compiler-warnings* nil)
         #+:ccl (*fasl-save-local-symbols* nil)
         #+:ccl (*save-local-symbols* nil)	
         )
    (compile-unless-macl(eval (setq *last-defun*
                                    `(defun ,*predicate* (cont rule-trace . ,parameters)
				       #+:allegro (declare (ignore-if-unused .,parameters))
                                       ,@(maybe-add-undo-bindings
                                          (mapcar (lambda (clause)
                                                    (bind-unbound-vars 
                                                     parameters
                                                     (focl-compile-clause parameters clause
                                                                          'cont))) 
                                                  clauses)
                                          symbol parameters)
                                       ,@(if *compile-with-rule-trace*
                                           `((prolog-trace 'fail ',symbol ,@parameters))) ; (2)
                                       
                                       ))))
     (symbol-function *predicate*)))




(defun focl-compile-clause-function (clause arity)
  "Compile all the clauses for a given symbol/arity
  into a single LISP function."
  (let* ((parameters (make-parameters arity))
	 (*predicate* 'dummy)
         #+:ccl (*warn-if-redefine-kernel* nil)
	 #+:ccl (*compiler-warnings* nil)
	 #+:ccl (*fasl-compiler-warnings* nil)
         #+:ccl (*warn-if-redefine* nil)
         #+:ccl (*compile-definitions* t)
         #+:ccl (*fasl-save-local-symbols* nil)
         #+:ccl (*save-local-symbols* nil)	
         )
    (compile nil  (setq *last-defun*
                                    `(lambda (cont rule-trace . ,parameters)
				       #+:allegro (declare (ignore-if-unused .,parameters))
				       (block dummy
                                       ,@(maybe-add-undo-bindings
					  (list (bind-unbound-vars parameters
								   (focl-compile-clause parameters clause
											'cont))) 
                                          *predicate* parameters))
                                       )))))

(defun compile-focl-facts (&optional (*extensional-preds* *extensional-preds*))
  (focl-compile-symbol-facts (mapcar #'car *extensional-preds*)))

(defvar *uncompiled-facts* nil)

(defun focl-compile-symbol-facts (&optional (symbols *uncompiled-facts*))
  "Compile a list of Prolog symbols.
  By default, the list is all symbols that need it."
    (mapc #'focl-compile-facts symbols)
    (setf *uncompiled-facts* (set-difference *uncompiled-facts* symbols)))


(defun retrieve-superset-of-matching-tuples-compiled  (pred vs 
                                                            &aux (position 0) (var nil) key 
                                                            (first-const nil))
  (mapc  #'(lambda(i)
             (if (var-p i)
               (setf var t)
               (unless first-const (setf first-const position)))
             (incf position)
             i)
         vs)
  (setq key 
        (if (null var)
          :all-bound
          (if first-const 
            first-const
            :all-vars)))
  (case key
    (:all-vars (pred-pos pred))
    (:all-bound (gethash vs (pred-pos-hash pred)))
    (otherwise (gethash (nth key vs) (return-argument-hash-table key pred)))))
 

(defun focl-compile-facts (symbol &optional (arity (p-arity (get-pstruct symbol))))
  "Compile all the clauses for a given symbol/arity
  into a single LISP function."
  (let* ((*predicate* (make-predicate symbol arity))  ;***
         (parameters (make-parameters arity))
         #+:ccl (*warn-if-redefine-kernel* nil)
	 #+:ccl (*compiler-warnings* nil)
	 #+:ccl (*fasl-compiler-warnings* nil)
         #+:ccl (*warn-if-redefine* nil)
         #+:ccl (*compile-definitions* t)
         #+:ccl (*fasl-save-local-symbols* nil)
         #+:ccl (*save-local-symbols* nil)
         )
    (compile-unless-macl (eval (setq *last-defun*
                                     `(defun ,*predicate* (cont rule-trace . ,parameters)
					#+:allegro (declare (ignore-if-unused .,parameters))
                                        (let* ((old-trail (fill-pointer *trail*))
                                               ,@(if *compile-with-rule-trace* 
						     `((cont #'(lambda (rt) (prolog-trace 'exit ',symbol ,@parameters)
                                                                 (funcall cont rt)))) nil)
                                               (parameters (list ,@parameters))
                                               (iparm (mapcar #'(lambda(i)(deref i)) parameters))
					       ,@(if *compile-with-rule-trace* '((matched nil)))
                                               (matching-tuples 
                                                (retrieve-superset-of-matching-tuples-compiled  
                                                 (get-pstruct ',symbol) iparm)))
                                          ,@(if *compile-with-rule-trace* `((prolog-trace 'call ',symbol parameters)))
                                          (mapc 
                                           #'(lambda (tuple)
                                               (when (unify! parameters tuple)
                                                 (funcall cont 
                                                          ,(add-to-rule-trace 'rule-trace "(" 
                                                                              `(cons ',symbol tuple)
                                                                              ")"))
                                                 ,@(if *compile-with-rule-trace* 
                                                     `((prolog-trace 'redo ',symbol parameters)
                                                 (setq matched t)))
                                                 )
                                               (undo-bindings! old-trail))
                                           matching-tuples)
                                          
                                          ,@(if *compile-with-rule-trace* 
                                              `(
                                                (loop (setq matching-tuples (ask-user-for-values  (cons ',symbol iparm)  matched))
                                                      (unless matching-tuples (return))
                                                      (when (unify! parameters (cdr matching-tuples))
                                                        (funcall cont 
                                                                 ,(add-to-rule-trace 'rule-trace "(" 
                                                                                     `(cons ',symbol (cdr matching-tuples))
                                                                                     ")"))
                                                        (prolog-trace 'redo ',symbol parameters)
                                                        (setq matched t)
                                                        )
                                                      (undo-bindings! old-trail))                 
                                                (prolog-trace 'fail ',symbol ,@parameters))
                                              '(nil))
                                              )))))))




(defun focl-compile-clause (parms clause cont)
  "Transform away the head, and compile the resulting body."
  (compile-body
   (nconc
    (mapcar #'make-= parms (focl-clause-parameters clause))
    (focl-clause-body clause))
   cont
   (mapcar #'self-cons parms)
   (focl-update-rule-trace clause 'rule-trace parms)))


(defun focl-update-rule-trace (clause &optional rule-trace parms)
  (add-to-rule-trace rule-trace 
                        " (" 
                        `(list ',(car (focl-clause-head clause))
                               ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                         parms))))

(defun focl-update-rule-trace-exit (clause rule-trace parms)
  (add-to-rule-trace rule-trace 
                        " (" 
                        `(list ',(car (focl-clause-head clause)) 
                               ,@(mapcar #'(lambda (p) `(deref-exp ,p))
                                         parms))
                        ")"))



(defun ask-user-for-values (form from-database)
  #-:ccl (declare (ignore form from-database))
  #-:ccl nil
  #+:ccl (unless (or *batch* (never-ask-again? form))
           (let* ((p (car form))
                  (s (get-pstruct p))
                  (qs (pred-questions s))
                  (vars (pred-vars s))
                  (types (pred-type s))
                  vars-in-form
                  vars-not-in-form 
                  q)
             (when qs
               (do ((args (cdr form)(cdr args))
                    (vars vars (cdr vars)))
                   ((null vars))
                 (if (var-p (car args))
                   (push (car vars) vars-in-form)
                   (push (car vars) vars-not-in-form)))
               (setq vars-not-in-form (nreverse vars-not-in-form))
               (setq vars-in-form (nreverse vars-in-form))
               (cond ((null vars-in-form) 
                      (if from-database nil
                          (when (okay-to-ask-y-n-question vars s form qs)
                            (ask-y-n-question s vars form (cadr (assoc :question qs))))))
                     (t (let ((q-alist (mapcar #'(lambda(v)
                                                   (setq q (some #'(lambda(vs.q.m?)
                                                                     (let ((vs (first vs.q.m?))
                                                                           )
                                                                       (if (subsetp vs vars-not-in-form
                                                                                    :test #'equalp)
                                                                         vs.q.m?)))
                                                                 
                                                                 (cdr(assoc v qs :test #'equalp))))
                                                   (if q (cons v q)
                                                       (return-from ask-user-for-values nil)))
                                               vars-in-form)))
                          (ask-value-question s vars types form q-alist from-database)))))))
  )

#+:ccl(defun ask-y-n-question(s vars form q)
  (let ((iq (sublis (mapcar #'cons vars (cdr form)) q :test #'equalp)))
    (cond ((y-or-n-dialog (format nil "~{ ~a~}?" iq)
                       :cancel-text nil)
           (insert-pos form) form)
          (t (insert-neg s  form)
             nil))))

#+:ccl(defun ask-value-question(s vars type form q-alist db)
  (unless (and db (some #'(lambda(v.vs.q.m?)
                            (eq :single-valued (third (cdr v.vs.q.m?))))
                        q-alist))
    (let* ((v-alist (mapcar #'cons vars (cdr form)))
           (t-alist (mapcar #'cons vars type))
           (q t)
           v
           (new-vals
            (mapcar #'(lambda(v.vs.q.m?) 
                        (setq v (get-value-from-user (car v.vs.q.m?)
                                                     (cdr (assoc (car v.vs.q.m?) t-alist
                                                                 :test #'equal))
                                                     (sublis v-alist (second (cdr v.vs.q.m?))
                                                             :test #'equalp)
                                                     q
                                                     s
                                                     form))
                        (cond ((null v)
                               (never-ask-again form)
                               (return-from ask-value-question nil))
                              (t (setq q nil)
                                 (cons (car v.vs.q.m?) v))))
                    q-alist))
           (new-form (cons (car form)
                     (mapcar #'(lambda(param arg)
                                 (if (var-p param)
                                   (cdr(assoc arg new-vals :test #'equalp))
                                   param))
                             (cdr form)
                             vars))))
      (insert-pos new-form)
      new-form)))

#+:ccl(defun get-value-from-user(v type q ask? s form &aux c)
  (cond ((or (eq :numeric type)(eq :anything type))
         (get-value-from-user-typein v type q ask? s form))
        ((and (not(eq '|type in| 
                      (setq c (catch-cancel (car(select-item-from-list (cons '|type in| (cons '|Nothing| (get-type-instances type)))
                                                                   :window-title
                                                                   (format nil "~{ ~a~}?" q)
                                                                   :selection-type :disjoint))))))
              (not (eq c :cancel)))
         (unless (eq c '|Nothing|)
           c))
        ((eq c :cancel) nil)
        (t (get-value-from-user-typein v type q ask? s form))))

#+:ccl(defun get-value-from-user-typein(v type q ask? s form)
  (declare (ignore v ask?))
  (let ((v (catch-cancel (get-string-from-user (format nil "~{ ~a~}?" q) ""
                                               :cancel-text "Nothing"))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (atom value)))
                (format t "~%~a ill-formed value" value)
                (get-value-from-user-typein v type q nil s form))
               ((and (eq :numeric type)
                     (not (numberp value)))
                (format t "~%~a ill-formed value" value)
                (get-value-from-user-typein v type q nil s form))
               ((or (eq :numeric type)(eq :anything type)) value)
               (t (pushnew value (get-type-instances type))
                  (setq *facts-changed* t)
                  value))))))

(defvar *New-facts* nil)
(defvar *New-negs* nil)
(defun insert-pos(form)
  (insert-new-fact (car form)(cdr form))
  (push form *new-facts*)
  (setq *facts-changed* t))
(defun insert-neg(s form)
  (push form *new-negs*)
  (push (cdr form)(pred-neg s))
  (setq *facts-changed* t))

(defun convert-to-focl-vars (exp)
  (cond ((and (var-p exp)
              (not (var-p (deref exp)))) ; i.e., bound but not to another var
         (deref exp))
        ((var-p exp)
        (make-pcvar :id (var-id exp)))
        ((not (consp exp)) exp)
        ((proper-listp exp)
         (mapcar #'convert-to-focl-vars exp))
        (t (reuse-cons (convert-to-focl-vars (car exp))
                       (convert-to-focl-vars (cdr exp))
                       exp))))

(defun never-ask-again(form)
  (insert-neg (get-pstruct (car form)) (convert-to-focl-vars form)))

(defun never-ask-again?(form &aux (tuple (cdr form))(negs (pred-neg(get-pstruct (car form)))))
  (some #'(lambda(neg)(unify-list tuple neg)) negs))


(defun unify (a b &optional (bindings (list t)))
  ;;; Return a most general binding list which unifies a & b
  (cond ((eql a b) bindings)
	((pcvar-p a) (var-unify a b bindings))
	((pcvar-p b) (var-unify b a bindings))
	((or (atom a)(atom b)) nil)
	((setf bindings (unify (first a)(first b) bindings))
         (unify (rest a) (rest b) bindings))))

(defun substitute-vars (form bindings)
  ;;; Substitute the variables in form for their ultimate values specified
  ;;; in the bindings
  (if (rest bindings)
      (substitute1 form bindings)
      form))

(defun substitute1 (form bindings)
  (cond ((null form) nil)
	((pcvar-p form)
	 (let ((binding (get-binding form bindings)))
	   (if binding
	       (substitute1 (second binding) bindings)
	       form)))
	((atom form) form)
	(t (reuse-cons (substitute1 (first form) bindings)
		 (substitute1 (rest form) bindings)
                 form))))

(defun bound-to-p (var1 var2 bindings)
  ;;; Check if var1 is eventually bound to var2 in the bindings
  (cond ((equalp var1 var2) t)
	((let ((val (second (get-binding var1 bindings))))
	   (and val (pcvar-p val) (bound-to-p val var2 bindings))))))
        
(defun unify-list (la lb &optional (bindings (list t)))
  ;;; Return a most general binding list which unifies a & b
  (when (every #'(lambda(a b)
                   (cond ((eql a b) t)
                         ((pcvar-p a) (var-unify a b bindings))
                         ((pcvar-p b) (var-unify b a bindings))))
               la lb)
    bindings))

(defun var-unify (var b bindings)
  ;;; Unify a variable with a wff, if must bind variable and *occur-check*
  ;;; flag is set then check for occur-check violation
  (if (and (pcvar-p b) (var-eq var b))
      bindings
      (let ((binding (get-binding var bindings)))
	(cond (binding (unify (second binding) b bindings))
	      ((and (pcvar-p b)(bound-to-p b var bindings)) bindings)
	      (t (add-binding var b bindings))))))

(defun var-eq (var1 var2)
  ;;; Return T if the two variables are equal
  (eql var1 var2))

(defun get-binding (var bindings)
  ;;; Get the variable binding for var
  (assoc var (rest bindings) ))

(defun add-binding (var val bindings)
  ;;; Add the binding of var to val to the existing set of bindings
  (setf (rest bindings) (cons (list var val) (rest bindings)))
  bindings)
(defun uniquify-variables (form)
  ;;; Make all the variables in form "unique" gensymed variables
  (let ((new-names (rename-list form nil)))
    (if (null new-names)
	form
	(rename-variables form new-names))))

(defun rename-list (form &optional new-names)
  (cond ((pcvar-p form)
	 (let ((id (pcvar-id form)))
	   (if (assoc id new-names)
	       new-names
	       (cons (list id (make-pcvar :id (if *use-gensyms*
						  (gensym)
						  (copy-symbol id))))
		     new-names))))
	((consp form) (rename-list (rest form)
				   (rename-list (first form) new-names)))
	(t new-names)))

(defun rename-variables (form new-names)
  (cond ((pcvar-p form)
	 (let ((entry (assoc (pcvar-id form) new-names)))
	   (if entry (second entry) form)))
	((atom form) form)
	(t (cons (rename-variables (first form) new-names)
		 (rename-variables (rest form) new-names)))))
#+:ccl (defun okay-to-ask-y-n-question (vars s form qs)
         (cond ((cdr vars)
                (null (every #'(lambda(v) 
                                 (or (null (cdr(assoc v qs :test #'equalp))) ;no question
                                     (and (every #'(lambda(v.vs.q.m?)
                                                     (eq :single-valued (third v.vs.q.m?)))
                                                 (cdr(assoc v qs :test #'equalp)))
                                          (has-some-value s v vars (cdr form)))))
                             vars)))
               (t (not (has-a-value s (cdr form))
                       ))))

#+:ccl (defun has-some-value (s v vars form)
	 (or (some #'(lambda(neg)(unify-list form neg)) (pred-neg s))
	     (let* ((n (position v vars))
		    (vform (append (butlast form (- (length vars) n))
				   (list v)
				   (nthcdr (+ n 2) form))))
	       (some #'(lambda(pos)(unify-list vform pos)) (pred-pos s)))))

#+:ccl (defun has-a-value (s form)
	 (or (some #'(lambda(neg)(unify-list form neg)) (pred-neg s))
             (some #'(lambda(pos)(unify-list form pos)) (pred-pos s))))


           
                                   
                              
