
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

;;;________________________________________________________________________________
;;; INFO-GAIN
;;;
;;; All-purpose general infomation gain function
;;; pred is either a pred structure, a built-in structure, a rule structure, or a clause structure
;;; used by induction, constructive induction, and ebl respecitively
;;; variabilization is the variabilization to be tested
;;; negative? tests negation of clause ; shouldn't be used by ebl
;;; it's less expensive to compute original info once, outside of this
;;; since it's not a function of the variabilization
;;; use-negative-hash-tables is T if hash tables should be used
;;; they store the # of matches of negative tuples
;;; it should only be used for the first literal of a clause (since the neg-tuples are the same)
;;; and avoid recomputation of this number for subsequent clauses
;;;
;;; returns four values
;;;
;;; 1. Info gain
;;; 2. Max possible gain for any specialized version
;;; 3. Stop- a flag to determine if this is the best you can do
;;; 4. A structure (an instance of conj-info) that is non-nil for extensional preds whose
;;;    variabilization's are appropriate for computing conjunctions
;;;
;;;  Note: to save space we could pass in and recycle a conj-info structure
;;;        (from find-max-literal, etc.)
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  to allow for extensional conjunctions
;;;  01  glenn  02/20/91  fix it so that when there are no pos-tuples it returns an i-gain of 0
;;;  02  glenn  03/21/91  ????
;;;  03  glenn  05/02/91  removed conjunction stuff
;;;  04  glenn  05/02/91  added instantiated-cliche and position-in-cliche to facilitate the
;;;                       proper printing of cliches - may want to make these special vars
;;;  05  cliff  05/08/91  moved back to count.
;;;  06  cliff  05/15/91  store pos-matches and neg-matches for graphing

(defun info-gain (pred                              ;; If you change any parameters
                  variablization                    ;; here also modify the definition
                  negative?                         ;; of info-gain in stopping.lisp.
                  pos-tuples                        ;; Thanks.
                  neg-tuples                        ;;
                                                    ;;
                  &optional                         ;;
                  (original-info nil)               ;;
                  (use-negative-hash-table nil)     ;;
                                                    ;;
                  &key                              ;;
                  instantiated-cliche               ;;
                  position-in-cliche)               ;;

  (let (pos-matches
        gain
        orig-pos-matches
        (lpos (length pos-tuples))
        (lneg (length neg-tuples))
        neg-matches
        orig-neg-matches
        max-possible-gain)

    (when (null original-info)
      (setq original-info (i-content lpos lneg)))

    (incf *variablizations-checked*)
  (multiple-value-setq (pos-matches
                        orig-pos-matches)
                       (generalized-count-matches pred
                                                  pos-tuples
                                                  variablization
                                                  negative?))

  (cond
   ((= pos-matches 0) (setq gain 0)  ;it's probably less than 0, but we won't choose it anymay
    (setq max-possible-gain 0)
    (when *graph-learning*                                          ;; CB(06)
      (store-info-gain-values-for-display "-" "-" 0))               ;;

    )
   (t
    (cond
     ((and use-negative-hash-table
           (setq neg-matches (gethash variablization (neg-hash-table pred negative?)))))
     (t (multiple-value-setq (neg-matches
                              orig-neg-matches)
                             (generalized-count-matches pred
                                                        neg-tuples
                                                        variablization
                                                        negative?))
        (when use-negative-hash-table
          (setf (gethash variablization (neg-hash-table pred negative?))
                neg-matches))))

    (setq max-possible-gain (* pos-matches original-info))
    (setq gain (if (= 0 original-info)
                 pos-matches
                 (I-gain original-info
                         orig-pos-matches
                         pos-matches
                         neg-matches)))

    (when *graph-learning*                                         ;; CB(06)
      (store-info-gain-values-for-display pos-matches neg-matches gain))  ;;
    ))

  ;;; ges 5/2 - added printing of cliche - note if conjunction is too long this will mess up the indenting
  (when (member :i *focl-trace-level*)
    (if instantiated-cliche
      (print-cliche instantiated-cliche position-in-cliche))
    (format t "~%~a~a~a ~30Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
            (if negative? "~" " ")
            (if (p-p pred)
              (p-name pred)
              (clause-body pred))
            variablization pos-matches orig-pos-matches (length pos-tuples)
            neg-matches orig-neg-matches lneg gain))
  (values gain
          max-possible-gain
          (and (= orig-pos-matches lpos)
               (or (null neg-matches)
                   (= 0 neg-matches)))
          )))


;;;________________________________________________________________________________
;;; GENERALIZED-COUNT-MATCHES
;;;
;;;  rv  who    date      reason
;;;  00  glenn  05/02/91  counts matches for any pred that satisfies pred-p, builtin-p, or rule-p
;;;  01  cliff  05/08/91  moved to count

(defun generalized-count-matches (pred
                                  tuples
                                  variablization
                                  negative?)
  (cond ((null tuples)
         (values 0 0))
        ((pred-p pred)                                            ;; extensional induction
         (if negative?
           (count-negation-matches pred tuples variablization)
           (count-matches pred tuples variablization)))
        ((builtin-p pred)                                         ;; builtin induction
         (if negative?
           (count-builtin-negation-matches (builtin-function pred)
                                           tuples variablization)
           (count-builtin-matches (builtin-function pred)
                                  tuples variablization)))
        ((rule-p pred)                                            ;; constructive-induction
         (count-prove (rule-prolog-function pred)
                      variablization
                      tuples negative?))
        (t                                                        ;; ebl
         (count-matches-clause (clause-body pred)
                               tuples
                               variablization))))


;;;________________________________________________________________________________
;;; INFO-GAIN-PROVE
;;;
;;;  Compute info gain (but T++ = pos)
;;;
;;;  rv  who    date     reason
;;;  00  cliff  05/20/91 store pos-matches, neg-matches, and gain for graphing.

(defun info-gain-prove (name
                        function
                        orig-info
                        pos-tuples
                        neg-tuples
                        args
                        negative-hash-table)
  (let (pos-matches
        neg-matches
        gain)

    (incf *variablizations-checked*)
    (setq pos-matches (count-prove function args pos-tuples))
    (cond ((and negative-hash-table
                (setq neg-matches (gethash args negative-hash-table))))
          (t (setq neg-matches (count-prove function args neg-tuples))
             (when negative-hash-table
               (setf (gethash args negative-hash-table) neg-matches))))
    (setq gain (if (= 0 orig-info)
                 pos-matches
                 (I-gain orig-info
                         pos-matches
                         pos-matches
                         neg-matches)))

    (when (member :i *focl-trace-level*)
      (format t "~% ~a~a~% ~30Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
              name args
              pos-matches pos-matches (length pos-tuples)
              neg-matches neg-matches (length neg-tuples) gain))

    (when *graph-learning*                                                ;; CB(00)
      (store-info-gain-values-for-display pos-matches neg-matches gain))  ;;

    (values gain
            pos-matches
            neg-matches)))


;;;________________________________________________________________________________
;;; INFO-GAIN-PROVE-IMMEDIATE
;;;
;;;  rv  who    date     reason
;;;  00  cliff  05/20/91 store pos-matches, neg-matches, and gain for graphing.

(defun info-gain-prove-immediate (name
                                  function
                                  orig-info
                                  pos-tuples
                                  neg-tuples
                                  args
                                  negative-hash-table)
  (let (pos-matches
        neg-matches
        gain)

    (incf *variablizations-checked*)
    (setq pos-matches (count-prove-immediate function pos-tuples))
    (cond ((and negative-hash-table
                (setq neg-matches (gethash args negative-hash-table))))
          (t (setq neg-matches (count-prove-immediate function neg-tuples))
             (when negative-hash-table
               (setf (gethash args negative-hash-table) neg-matches))))
    (setq gain (if (= 0 orig-info)
                 pos-matches
                 (I-gain orig-info
                         pos-matches
                         pos-matches
                         neg-matches)))
    (when (member :i *focl-trace-level*)
      (format t "~% ~a~a~% ~30Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
              name args
              pos-matches pos-matches (length pos-tuples)
              neg-matches neg-matches (length neg-tuples) gain))

    (when *graph-learning*                                                ;; CB(00)
      (store-info-gain-values-for-display pos-matches neg-matches gain))  ;;

    (values gain
            pos-matches
            neg-matches)))





#|
(defun count-matches (pred tuples variablization)
  (let ((constraints (find-new-equality-constraints variablization))
        (matches 0)
        (key (classify-variabilization-for-induction variablization))
        arguments)
    (dolist (tuple tuples)
      (setf arguments (mapcar #'(lambda(v)
                                  (if (new-var? v)
                                    v
                                    (nth (pcvar-id v) tuple)))
                              variablization))
      (incf matches (count arguments
                           (retrieve-superset-of-matching-tuples pred arguments key)
                           :test  #'(lambda(args new-tuple)
                                      (and (every #'(lambda(a n)
                                                      (or (pcvar-p a)
                                                          (equalp a n)))
                                                  args new-tuple)
                                           (every #'(lambda(p1.p2)
                                                      (equalp (nth (car p1.p2) new-tuple)
                                                             (nth (cdr p1.p2) new-tuple)))
                                                  constraints))))))
    matches))
|#

;;;________________________________________________________________________________
;;; COUNT-MATCHES
;;;
;;;  this is slightly slower than the above, but so much clearer
;;;  returns 2 values
;;;   1 number of extended tuples matches
;;;   2 number of original tuples matched
;;;  pred is  a pred structure
;;;  tuples is a lsit of tuples (assumes tht first var is ?0, second ?1 etc
;;;  variabilization is a list of variables where ?-n indicates new, ?n indicates
;;;  nth constant of tuple
;;;
;;;  rv  who    date     reason

(defun count-matches (pred
                      tuples
                      variablization)
  (let ((matches 0)
        (original-matches 0)
        (key (classify-variabilization-for-induction variablization))
        arguments
        new-matches)
    (dolist (tuple tuples)
      (setf arguments (mapcar #'(lambda(v)
                                  (if (new-var? v)
                                    v
                                    (nth (pcvar-id v) tuple)))
                              variablization))   ;;arguments holds instantiated pattern
                                                 ;;the only variables are new
      (setf new-matches (count arguments
                               (retrieve-superset-of-matching-tuples pred arguments key)
                               :test  #'(lambda(args new-tuple)
                                          (unify-list args new-tuple))))
      (unless (= 0 new-matches)
        (incf original-matches) ;increment 1 if any extended tuples that matches
        (incf matches new-matches))) ;increment n ifn extended tuples match
    (values matches original-matches)))


;;;________________________________________________________________________________
;;; COUNT-NEGATION-MATCHES
;;;
;;;  a negation never binds variables, instead it's true for each tuple is there
;;;  is no fact that unifies with that tuple
;;;
;;;  rv  who    date     reason

(defun count-negation-matches (pred
                               tuples
                               variablization)
  (let ((key (classify-variabilization-for-induction variablization))
        arguments
        matches)
    (setq matches (count-if #'(lambda(tuple)
                                (setf arguments (mapcar #'(lambda(v)
                                                            (if (new-var? v)
                                                              v
                                                              (nth (pcvar-id v) tuple)))
                                                        variablization))
                                (not (some #'(lambda(new-tuple)
                                               (unify-list arguments new-tuple))
                                           (retrieve-superset-of-matching-tuples pred
                                                                                 arguments
                                                                                 key))))
                            tuples))
    ;;only counts those tuples for which no pattern unifies
    (values matches matches)))


;;;________________________________________________________________________________
;;; EXTEND-TUPLES-EXTENSIONAL
;;;
;;;  like count tuples, but instantiates new vars and append to end of teach tuple
;;;  note that new-vars are now indicated by a list of new-variables
;;;  all-unique-images dousn't add duplicates
;;;
;;;  rv  who    date     reason

(defun extend-tuples-extensional (pred
                                  tuples
                                  variablization

                                  &optional
                                  (new-vars (compute-new variablization)))
  (if new-vars
      (let ((key (classify-variabilization-for-induction variablization new-vars))
	    arguments
	    bindings)
	(mapcan #'(lambda(tuple)
		    (setf arguments (mapcar #'(lambda(v)
						(if (or (new-var? v new-vars);;is it a member
							(not (pcvar-p v)))
						    v
						  (nth (pcvar-id v) tuple)))
					    variablization))
		    (all-unique-images #'(lambda(new-tuple)
					   (when (setf bindings (unify-list arguments new-tuple))
						 (append tuple (substitute1 new-vars bindings))))
				       (retrieve-superset-of-matching-tuples pred arguments key)))
		tuples))
    (filter-tuples pred tuples variablization)))


;;;________________________________________________________________________________
;;; EXTEND-NEGATION-TUPLES
;;;
;;;  negations dont extend, they just filter (using remove if not))
;;;
;;;  rv  who    date     reason

(defun extend-negation-tuples (pred
                               tuples
                               variablization

                               &optional
                               (new-vars (compute-new variablization)))
  (let ((key (classify-variabilization-for-induction variablization new-vars))
        arguments)
    (remove-if #'(lambda(tuple)
                  (setf arguments (mapcar #'(lambda(v)
                                              (if (or (new-var? v new-vars)  ;;is it a member
                                                    (not (pcvar-p v)))
                                                v
                                                (nth (pcvar-id v) tuple)))
                                          variablization))
                  (some #'(lambda(new-tuple)
                                 (unify-list arguments new-tuple))
                             (retrieve-superset-of-matching-tuples pred arguments key)))
              tuples)))


;;;________________________________________________________________________________
;;; FILTER-TUPLES
;;;
;;;  returns the subset of tuples that are s
;;;
;;;  rv  who    date     reason

(defun filter-tuples (pred
                      tuples
                      variablization)
  (let ((the-tuple (copy-list variablization))
	(hash-table (pred-pos-hash pred)))
    (remove-if-not #'(lambda(tuple)
		       (do ((tu the-tuple (cdr tu));;destroy the-tuple
			    (v variablization (cdr v)))
			   ((null v))
			   (setf (car tu)
				 (if (pcvar-p (car v))
				     (nth (pcvar-id (car v)) tuple)
				   (car v))))
		       (gethash the-tuple hash-table nil))
		   tuples)))


;;;________________________________________________________________________________
;;; FILTER-TUPLES
;;;
;;;  general purpose tuple extender - uses prolog use prove if possible
;;;
;;;  rv  who    date     reason

(defun extend-tuples (new-literal
                      tuples
                      variablization
                      new-vars)				
  (cond (new-vars
	 (let* ((v-list (create-parameter-bindings new-vars nil))
		(the-tuple (append variablization v-list))
		(body-fun (convert-to-prolog-function new-literal
                                                      (append variablization new-vars))))
	   (mapcan #'(lambda(tuple)
		       (do ((tu the-tuple (cdr tu));;destroy the-tuple into argument
			    (v tuple (cdr v)))
			   ((null v))
			   (setf (car tu)(car v)))
		       (mapcar #'(lambda(new)
				   (append tuple new))
			       (setof-function body-fun the-tuple v-list)))
		   tuples)))
	(t (filter-unproved-tuples (convert-to-prolog-function new-literal variablization) tuples))))
	

;;;________________________________________________________________________________
;;; COUNT-PROVE
;;;
;;;  counts the number of tuples that are true for a variablization of a predicate
;;;  return two values pos-matches and original-pos-matches
;;;  these are equal since we are not expanding tuples
;;;
;;;  rv  who    date     reason

(defun count-prove (fun args tuples
                    &optional (negative nil) &aux (matches 0))
  (dolist (tuple tuples)
    (if (prove-function? fun args negative (create-parameter-bindings-from-tuple args tuple))
      (incf matches)))
  (values matches matches))


;;;________________________________________________________________________________
;;; COUNT-PROVE-IMMEDIATE
;;;
;;;  like count-prove, but tuples is in order of argument of function
;;;
;;;  rv  who    date     reason

(defun count-prove-immediate (fun tuples)
  (let ((matches 0))
    (dolist (tuple tuples)
      (if (prove-function? fun nil nil tuple)
        (incf matches)))
    (values matches matches)))


;;;________________________________________________________________________________
;;; EXTEND-TUPLES-INTENSIONAL
;;;
;;;  like extend-tuples
;;;
;;;  rv  who    date     reason

(defun extend-tuples-intensional (pred
                                  tuples
                                  variablization

                                  &optional
                                  (negation nil)
                                  (new-vars (compute-new variablization)))
  (let (argument
        (bindings nil)
        (new-prolog-vars nil)
        (pred-fun (rule-prolog-function pred)))
    (cond ((and new-vars (not negation))
           (mapcan #'(lambda(tuple)
                       (multiple-value-setq (argument bindings)
                                            (create-parameter-bindings-from-tuple variablization tuple bindings))
                       (when (null new-prolog-vars)
                         (setq new-prolog-vars (mapcar #'(lambda(x)(cdr (assoc x bindings))) new-vars)))
                       (mapcar #'(lambda(new) (append tuple new))
                               (setof-function  pred-fun argument new-prolog-vars)))
                   tuples))
          (negation (filter-proved-tuples-with-variablization variablization pred-fun tuples))
          (t (filter-unproved-tuples-with-variablization variablization pred-fun tuples)))))


;;;________________________________________________________________________________
;;; EXTEND-TUPLES-IS-OP
;;;
;;;  rv  who    date      reason
;;;  00  glenn  05/02/91  added to deal with arithmetic operators

(defun extend-tuples-is-op (pred tuples varzn vars)
  (let ((is-var (car varzn))
        (is-expr (cons (is-op-arithmetic-op pred) (cdr varzn))))
    (extend-tuples-is tuples vars is-var is-expr)))


;;;________________________________________________________________________________
;;; GENERALIZED-EXTEND-TUPLES
;;;
;;;  extends both positive and negative tuples and returns the result note pred
;;;  is a structure here might want to use this in find-literal
;;;
;;;  rv  who    date      reason
;;;  00  glenn  05/02/91  added to provide a uniform procedure for extending tuples

(defun generalized-extend-tuples (predicate-with-maximum-gain
                                  tuples
                                  literal-vars
                                  max-negated?
                                  new-vars

                                  &optional
                                  vars)
  (cond ((pred-p predicate-with-maximum-gain)
         (if max-negated?
           (extend-negation-tuples predicate-with-maximum-gain
                                   tuples
                                   literal-vars
                                   new-vars)
           (extend-tuples-extensional predicate-with-maximum-gain
                                      tuples
                                      literal-vars
                                      new-vars)))
        ((builtin-p predicate-with-maximum-gain)
         (if max-negated?
           (extend-negation-tuples-builtin predicate-with-maximum-gain
                                           tuples
                                           literal-vars)
           (extend-tuples-builtin predicate-with-maximum-gain
                                  tuples
                                  literal-vars)))
        ((is-op-p predicate-with-maximum-gain)
         (if max-negated?
           nil
           (extend-tuples-is-op predicate-with-maximum-gain
                                tuples
                                literal-vars
                                vars)))
        (t
         (extend-tuples-intensional predicate-with-maximum-gain
                                    tuples
                                    literal-vars
                                    max-negated?
                                    new-vars))))


;;;________________________________________________________________________________
;;; FILTER-PROVED-TUPLES-WITH-VARIABLIZATION
;;;
;;;  rv  who    date      reason

(defun filter-proved-tuples-with-variablization (variablization clause tuples)				
  (remove-if #'(lambda(tuple)
                        (prove-function? clause tuple nil
                                         (create-parameter-bindings-from-tuple variablization tuple))
                        )
             tuples))


;;;________________________________________________________________________________
;;; FILTER-UNPROVED-TUPLES-WITH-VARIABLIZATION
;;;
;;;  rv  who    date      reason

(defun filter-unproved-tuples-with-variablization (variablization clause tuples)				
  (remove-if-not #'(lambda(tuple)
                        (prove-function? clause tuple nil
                                         (create-parameter-bindings-from-tuple variablization tuple))
                        )
             tuples))


;;;________________________________________________________________________________
;;; COUNT-AND-EXTEND-MATCHES
;;;
;;;  note the functions below will actually collect and count the matches this is
;;;  a cross between count-matches and extend-tuples.  It accepts a predicate (pred), a set of
;;;  tuples (tuples), a variabilization (variablization), the variables which have been bound
;;;  been bound so far (old-variables) and optional a list of unbound variables (new-vars).
;;;
;;;  Three values are returned:
;;;  1) the number of matches
;;;  2) the number of original matches
;;;  3) the tuples that matched (new-tuples)
;;;  4) the new variabilization (literal-vars)
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  added

(defun count-and-extend-matches (pred
                                 tuples
                                 variablization
                                 old-variables

                                 &optional
                                 (new-vars (compute-new variablization)))
  (let ((key (classify-variabilization-for-induction variablization new-vars))
        arguments
        bindings
        new-matches
        new-tuples
        (original-matches 0)
        literal-vars
        new-literal-vars
        new-types
        alist-ignore)

    (multiple-value-setq (literal-vars new-literal-vars new-types alist-ignore)
					;literal-vars is the variables in call with free vars
					;renamed to next old-var
					;new-vars is list of "new" variables renamed to old-vars
					;new-types is types of new vars
					;new-var-alist is (old-name . bound-name) for free vars
                         (transfer-literal-vars variablization ;variables in call
                                                (p-type pred)
                                                ;;type of variables
                                                old-variables ;bound variables
                                                (length old-variables)))
    (setq new-tuples
          (mapcan
           #'(lambda(tuple)
               (setf arguments (mapcar #'(lambda(v)
                                           (if (or (new-var? v new-vars)  ;;is it a member
                                                   (not (pcvar-p v)))
                                             v
                                             (nth (pcvar-id v) tuple)))
                                       variablization))
               (setq new-matches
                     (all-unique-images
                      #'(lambda(new-tuple)
                          (when (setf bindings (unify-list arguments new-tuple))
                            (append tuple (substitute1 new-vars bindings))))
                      (retrieve-superset-of-matching-tuples pred arguments key)))
               (if new-matches (incf original-matches))
               new-matches)
           tuples))
    (values (length new-tuples) original-matches new-tuples literal-vars
            new-types)))


;;;________________________________________________________________________________
;;; COUNT-AND-EXTEND-NEGATION-MATCHES
;;;
;;;  again this is a cross between count-negation-matches and extend-negation tuples
;;;  three values are returned:
;;;  1) the number of matches
;;;  2) the number of original matches (same for negation)
;;;  3) the tuples that matched (new-tuples)
;;;  4) the new variabilization (unchanged, i.e. same as the original variabilization since
;;;     negation matches don't bind variables
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  added

(defun count-and-extend-negation-matches (pred
                                          tuples
                                          variablization

                                          &optional
                                          (new-vars (compute-new variablization)))

  (let ((key (classify-variabilization-for-induction variablization (compute-new variablization)))
        arguments
        matches)
    (setq matches
          (remove-if #'(lambda(tuple)
                         (setf arguments (mapcar #'(lambda(v)
                                                     (if (or (new-var? v new-vars)  ;;is it a member
                                                             (not (pcvar-p v)))
                                                       v
                                                       (nth (pcvar-id v) tuple)))
                                                 variablization))
                         (some #'(lambda(new-tuple)
                                   (unify-list arguments new-tuple))
                               (retrieve-superset-of-matching-tuples pred arguments key)))
                     tuples))
    (values (length matches)
            (length matches)
            matches
            variablization)))


;;;________________________________________________________________________________
;;; I-CONTENT
;;;
;;;  rv  who    date     reason

(defun I-content (pos-tuples
                  neg-tuples)
  (log (/ (+ pos-tuples neg-tuples)
          pos-tuples)
       2))


;;;________________________________________________________________________________
;;; I-GAIN
;;;
;;;  Low level I-gain (pseudo information gain) computation
;;;
;;;  rv  who    date     reason
;;;  00  mike   ????     ????
;;;  01  mike   04/26/91 added the test for 0 the hack is there for EBL.  If a general
;;;                      node covers all positives, then the operationalization have
;;;                      no-igain.  The 1 favors those that cover more positive

(defun I-gain (original-I-content
               original-pos-matches
               pos-matches
               neg-matches)
  (if (= pos-matches 0) 0
      (* pos-matches (if (= original-I-content 0) 1  ;;<- HACK
                         (- original-I-content
                            (log (/ (+ pos-matches neg-matches)
                                    original-pos-matches)
                                 2))))))


;;;________________________________________________________________________________
;;; FILTER-PROVED-TUPLES
;;;
;;;  rv  who    date     reason

(defun filter-proved-tuples (clause tuples)
  (remove-if #'(lambda (tuple)
                     (prove-function? clause tuple nil tuple))
                 tuples))


;;;________________________________________________________________________________
;;; COUNT-PROVED-TUPLES
;;;
;;;  rv  who    date     reason

(defun count-proved-tuples (clause tuples)
  (count-if #'(lambda (tuple)
                     (prove-function? clause tuple nil tuple))
                 tuples))


;;;________________________________________________________________________________
;;; FILTER-UNPROVED-TUPLES
;;;
;;;  rv  who    date     reason

(defun filter-unproved-tuples (clause tuples)
  (remove-if-not #'(lambda (tuple)
                     (prove-function? clause tuple nil tuple))
                 tuples))


;;;________________________________________________________________________________
;;; ANY-PROVED-TUPLES
;;;
;;;  returns T if any tuples are satisfied by a clause
;;;
;;;  rv  who    date     reason

(defun any-proved-tuples (clause tuples)
  (some #'(lambda (tuple)
	    (prove-function? clause tuple nil tuple))
	tuples))


;;;________________________________________________________________________________
;;; COUNT-AND-EXTEND-BUILTIN-MATCHES
;;;
;;;  builtins don't introduce new variables so we can just extend them normally
;;;  (i.e., don't need to keep track of things like orig-tuples etc.)
;;;
;;;  rv  who    date      reason
;;;  00  glenn  03/21/91  changed

(defun count-and-extend-builtin-matches (pred tuples variablization)
  (let ((new-tuples (extend-tuples-builtin pred tuples variablization)))
    (values (length new-tuples) (length new-tuples) new-tuples variablization)))


;;;________________________________________________________________________________
;;; COUNT-AND-EXTEND-BUILTIN-NEGATED-MATCHES
;;;
;;;  builtins don't introduce new variables so we can just extend them normally
;;;  (i.e., don't need to keep track of things like orig-tuples etc.)
;;;
;;;  rv  who    date      reason
;;;  00  glenn  03/21/91  changed

(defun count-and-extend-builtin-negation-matches (pred tuples variablization)
  (let ((new-tuples (extend-negation-tuples-builtin pred tuples variablization)))
    (values (length new-tuples) (length new-tuples) new-tuples variablization)))







;;;===============================================================================
;;; Move The following to globals.lisp  [????]
;;;===============================================================================

(defvar *domain*)


;;;===============================================================================
;;; Move All the functions below to appropriate utility files.  [????]
;;;
;;;   Variable-Utilities.Lisp
;;;   Literal-Structures-Utilities.Lisp
;;;   P-Structure-Utilities.Lisp
;;;   Hash-Table-Utilites.Lisp          (Data-Base-Utilities.lisp ?)
;;;   Clause-Structure-Utilities.Lisp
;;;===============================================================================

;;;________________________________________________________________________________
;;; RESET-PREDS
;;;
;;;  Used to change the context of learning (ie. the background knowledge)
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/09/90  to clean up predicate definitions when resetting
;;;  01  mike   11/12/90  remove brules property also (for prolog)

(defun reset-preds ()
  (remove-pred-defs *extensional-preds* 'pred)
  (setq *extensional-preds* nil)
  (remove-pred-defs *builtin-preds* 'builtin)
  (setq *builtin-preds* nil)
  (remove-pred-defs *intensional-preds* 'rule)
  (remove-pred-defs *intensional-preds* 'brules)
  (setq *intensional-preds* nil)
  ; clean up cliche variables
  (setq *cliches-to-be-named* nil)
  (setq *named-cliches* nil)
  (setq *anonymous-cliches* nil))


;;;________________________________________________________________________________
;;; DELETE-RULE
;;;
;;;  if there are any more properties, they should be deleted too
;;;
;;;  rv  who    date      reason

(defun delete-rule (r)
  (setf *intensional-preds* (delete r *intensional-preds* :key #'car))
  (setf (get r 'rule) nil)
  (setf (get r 'brules) nil))


;;;________________________________________________________________________________
;;; REMOVE-PRED-DEFS
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/09/90  to handle the clean up predicate definitions when resetting

(defun remove-pred-defs (pred-def-list type)
  (dolist (pred-def pred-def-list)
    (setf (get (car pred-def) type) nil)))


;;;________________________________________________________________________________
;;; PRINT-LITERAL
;;;
;;;  Print literal structure to look just like prolog.  (Well almost)
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  to print out predicates using is properly (e.g. A is (* B (+ C D)))

(defun print-literal (l stream depth &optional (print-dot t))
  (cond ((null l) )
        ((and (literal-deleted? l)
              (cond(*print-deleted-literals* (format stream "DELETED ") nil)  ;;causes and to fail, rest is executed
                   (t t)))  nil)  ;otherwise don't print anything
        ((literal-negated? l)
         (if (literal-next (literal-negated-literals l)) ;negation of conjunct
           (format stream "not((~a))" (with-output-to-string (string)
                                        (print-literal (literal-negated-literals l)
                                                       string depth nil)))
           (format stream "not(~a)" (with-output-to-string (string)
                                      (print-literal (literal-negated-literals l)
                                                     string depth nil)))))
        ((eq (literal-predicate-name l) 'is) ; ges handle is builtins
         (format stream "~a is ~a"
                 (var-name (first (literal-variablization l)))
                 (substitute-var-names (second (literal-variablization l)))))
        (t
         (cond((and (get-pstruct (literal-predicate-name l))
                        (p-infix (get-pstruct (literal-predicate-name l))))
                   (format stream "~a~(~a~)~a"
                           (var-name (first (literal-variablization l)))
                           (literal-predicate-name l)
                           (var-name (second (literal-variablization l)))))
                  (t (format stream "~(~a~)~a" (literal-predicate-name l)
                             (append (mapcar #'(lambda(x)
                                             (var-name x t))
                                             (butlast(literal-variablization l)))
                                     (list (var-name (car (last (literal-variablization l)))))))))))

  (cond((null l) (format stream "."))
       ((literal-next l)
        (unless (and (literal-deleted? l) (not *print-deleted-literals*))
          (format stream ","))
        (print-literal (literal-next l) stream depth))
       (print-dot (format stream "."))))


;;;________________________________________________________________________________
;;; SUBSTITUTE-VAR-NAMES
;;;
;;;  Accepts an expression and substitutes the variable names for the variables
;;;  in the expression.  This is meant to be used in conjunction with print-literal
;;;  where vars like ?0 get substited with names like A.
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/01/90  added

(defun substitute-var-names (exp)
  (cond ((pcvar-p exp) (var-name exp))
        ((atom exp) exp)
        (t (mapcar #'substitute-var-names exp))))


;;;________________________________________________________________________________
;;; VAR-NAME
;;;
;;;  Returns name for variable (if it is one)
;;;
;;;  rv  who    date      reason

(defun var-name (x &optional (comma nil))
 (format nil "~a~a"
   (if (pcvar-p x)
     (convert-to-letter (pcvar-id x))
     x)
   (if comma "," "")))


;;;________________________________________________________________________________
;;; CONVERT-TO-LETTER
;;;
;;;  rv  who    date      reason

(defun convert-to-letter (n)
  (if (and (numberp n)(>= n 0)(< n 26))
    (nth n '(A B C D E F G H I J K L M N O P Q R S T U V W Y Z))
    (format nil "_~a" n))
  )


;;;________________________________________________________________________________
;;; CONVERT-TO-PROLOG
;;;
;;;  convert a linked list of literal structures to a prolog-term that can be
;;;  sent to prove note that it does not copy deleted literals into the list.
;;;
;;;  rv  who    date      reason

(defun convert-to-prolog (literal)
  (if (null literal) nil
      (if (literal-deleted? literal)
        (convert-to-prolog (literal-next literal))
        (if (literal-negated? literal)
          (cons (cons 'not (convert-to-prolog (literal-negated-literals literal)))
                (convert-to-prolog (literal-next literal)))
          (cons (cons (literal-predicate-name literal)
                      (literal-variablization literal))
                (convert-to-prolog (literal-next literal)))))))

;;;________________________________________________________________________________
;;; CONVERT-TO-PROLOG-FUNCTION
;;;
;;;  rv  who    date     reason

(defun convert-to-prolog-function (l v &optional (clause (convert-to-prolog l)))
  (when clause
    (focl-compile-clause-function (cons (cons 'dummy-prolog-predicate v)
                                        clause)
                                  (length v))))

;;;________________________________________________________________________________
;;; CONVERT-xxxxx-TO-PROLOG
;;;
;;;  convert a linked list of literal structures to a prolog-term that can be
;;;  sent to prove note that it does not copy deleted literals into the list.
;;;
;;;  rv  who    date      reason
;;;  00  cliff  05/20/91  needed to be able to convert deleted literals too.

(defun convert-clause-to-prolog (clause)
  (cond ((null clause) nil)
        ((literal-deleted? clause) (convert-clause-to-prolog (literal-next clause)))
        (t (cons (convert-literal-to-prolog clause) 
                 (convert-clause-to-prolog (literal-next clause))))))

(defun convert-literal-to-prolog (literal)
  (cond ((null literal) nil)
        ((literal-deleted? literal) nil)
        ((literal-negated? literal) 
         (cons 'not (convert-clause-to-prolog (literal-negated-literals literal))))
        (t (cons (literal-predicate-name literal)
                 (literal-variablization literal)))))

(defun convert-literal-to-prolog-regardless (literal)
  (cond ((null literal) nil)
        ((literal-negated? literal) 
         (cons 'not (convert-clause-to-prolog (literal-negated-literals literal))))
        (t (cons (literal-predicate-name literal)
                 (literal-variablization literal)))))

;;;________________________________________________________________________________
;;; ALL-TYPED-TUPLES
;;;
;;;  rv  who    date     reason
;;;  00  cliff  05/20/91  moved from count.lisp

(defun all-typed-tuples (type)
   (if (null type) '(())
      (let ((short (all-typed-tuples (cdr type))))
      (mapcan #'(lambda(new)
            (mapcar #'(lambda(old)
                  (cons new old))
               short))
            (all-elements (car type))))))

;;;________________________________________________________________________________
;;; ALL-ELEMENTS
;;;
;;;  rv  who    date     reason
;;;  00  cliff  05/20/91  moved from count.lisp

(defun all-elements (type)
  (cdr (assoc type *domain*)))


;;;________________________________________________________________________________
;;; MAKE-CLAUSE-STRUCTURE
;;;
;;;  make-clause-structure converts a prolog style rule (i.e., a list of prolog clauses)
;;;  to a list of clause-structures
;;;
;;;  rv  who    date     reason

(defun make-clause-structure (clauses &aux (clauseno 0))
  (mapcar #'(lambda(clause &aux (new (uniquify-variables clause)))
              (prog1 (make-clause :body (cdr new)
                           :number clauseno
                           :neg-tuples-hash (make-hash-table :test #'equal
                                                             :size 2)
	                    ;the negative hash table is small, since there
                            ;is actually only 1 variabilization tested for each clause
                           :head (car (car new))
                           :new-vars (compute-new-vars
                                      (cdr new)
                                      (cdr (car new)))
                           :parameters (cdr (car new)))
                (incf clauseno)))
          clauses))

;;;________________________________________________________________________________
;;; COMPUTE-NEW-VARS
;;;
;;;  returns variables in clause that are not in head
;;;
;;;  rv  who    date     reason

(defun compute-new-vars (body old-vars &optional (new-vars nil))
  (cond ((pcvar-p body)
         (if (or (member body old-vars :test #'var-eq)
                 (member body new-vars :test #'var-eq))
           new-vars
           (cons body new-vars)))
        ((consp body)
         (compute-new-vars (car body) old-vars
                           (compute-new-vars (cdr body) old-vars new-vars)))
        (t new-vars)))


;;;________________________________________________________________________________
;;; MAKE-POS-HASH
;;;
;;;  makes the equal hash table that returns a list of the positive example if
;;;  the example (used as a key), is positive
;;;
;;;  rv  who    date     reason

(defun make-pos-hash (pos)
  (let*
    ((table (make-hash-table :test #'equal :size (+ (length pos) 1))))
    (dolist (tuple pos table)
      (add-to-pos-hash tuple table))))


;;;________________________________________________________________________________
;;; ADD-TO-POS-HASH
;;;
;;;  rv  who    date     reason

(defun add-to-pos-hash(tuple table &optional (delete nil))
  (setf (gethash tuple table) (if delete nil (list tuple))))


;;;________________________________________________________________________________
;;; MAKE-SLOT-VALUE-HASH
;;;
;;;  Makes an array of hash tables, one for each parameter.  Each hash table stores
;;;  a list of positive tuples that have a particular value for a key.
;;;
;;;  rv  who    date     reason

(defun make-slot-value-hash (arity pos)
  (let ((array (make-array (list arity))))
    (dotimes (i arity)
      (setf (aref array i) (make-hash-table :test #'equal  ;;changed to equal so that 1.0 equal 1
                                            :size *init-slot-table-size*
                                            :rehash-size *slot-table-rehash-size*
                                            :rehash-threshold *slot-table-rehash-threshold*)))
    (dolist (tuple pos)
      (add-to-slot-value-hash array tuple arity))
    array))


;;;________________________________________________________________________________
;;; ADD-TO-SLOT-VALUE-HASH
;;;
;;;  rv  who    date     reason

(defun add-to-slot-value-hash(array tuple arity &optional (delete nil))
  (dotimes (i arity)
      (if delete
        (setf (gethash (nth i tuple) (aref array i))
              (delete tuple (gethash (nth i tuple) (aref array i)) :test #'equal))
        (pushnew tuple (gethash (nth i tuple) (aref array i))))))

;;;________________________________________________________________________________
;;; INSERT-NEW-FACT
;;;
;;;  rv  who    date     reason

(defun insert-new-fact (p tuple)
  (let* ((arity (length tuple))
         (s (get-pstruct p))
         (array (pred-slot-value-hash s))
         (table (pred-pos-hash s)))
    (pushnew tuple (pred-pos s))
    (add-to-slot-value-hash array tuple arity)
    (add-to-pos-hash tuple table)
    tuple))

;;;________________________________________________________________________________
;;; DESTROY-FACT
;;;
;;;  rv  who    date     reason

(defun destroy-fact(p tuple)
  (let* ((arity (length tuple))
         (s (get-pstruct p))
         (array (pred-slot-value-hash s))
         (table (pred-pos-hash s)))
    (setf (pred-pos s) (delete tuple (pred-pos s) :test #'equal))
    (add-to-slot-value-hash array tuple arity t)
    (add-to-pos-hash tuple table t)
    tuple))

;;;________________________________________________________________________________
;;; NEW-VAR?
;;;
;;;  return T if var is new (i.e., an number < 0)
;;;  alteratively, if variables is a member of new-vars
;;;  The later is used by extend-tuples
;;;  The former by count-matches
;;;  sorry- but new variables have to become old sooner or later
;;;  (KA, dec. 11): added test for numberp, and if it fails, var is
;;;  considered old.
;;;  during dt mutation, some pcvar id's are not numbers. called via
;;;  generate-variabilizations via add-random-term in dt-mutator.lisp
;;;
;;;  rv  who    date     reason

(defun new-var?(var &optional (new-vars nil))
  (when (pcvar-p var)
  (if new-vars
    (member var new-vars :test #'var-eq)
      (if (numberp (pcvar-id var))
          (> 0  (pcvar-id var))
          nil ))))

;;;________________________________________________________________________________
;;; CLASSIFY-ARGUMENTS-FOR-PROLOG
;;;
;;;  input    a argument of the form (?1 ?2 a b)
;;;  returns :all-bound (if no vars) or an integer representing the position of
;;;                     the first constants (where 0 is the first variable), or
;;;          :all-vars  (if there are no constants)
;;;
;;;  rv  who    date     reason

(defun classify-arguments-for-prolog  (vs &aux (position 0) (first-const nil) (var nil))
  (dolist (i vs)
    (if (pcvar-p i)
      (setf var t)
      (unless first-const (setf first-const position)))
    (incf position))
  (if (null var)
    :all-bound
    (if first-const
      first-const
      :all-vars)))

;;;________________________________________________________________________________
;;; CLASSIFY-VARIABLIZATIONS-FOR-INDUCTION
;;;
;;;  like above, but it works on variabilizatiosn rather than arguments
;;;  in a variabilization $n indicates an old var, and $-n indicates a new var
;;;  Therefore return :all-bound if no new vars, or the position of the first old-var
;;;  (if there are new) or :all-vars if all new vars (used by (six ?x) in loan examples)
;;;
;;;  rv  who    date     reason

(defun classify-variabilization-for-induction  (vs &optional (new-vars nil))
  (let ((position 0)
        (first-const nil)
        (new nil))
    (dolist (i vs)
      (if (new-var? i new-vars)
        (setf new t)
        (unless first-const (setf first-const position)))
      (incf position))
    (if (null new)
      :all-bound
      (if first-const
        first-const
        :all-vars))))

;;;________________________________________________________________________________
;;; RETURN-ARGUMENT-HASH-TABLE
;;;
;;;  return hash table for argument (position) of pred)
;;;
;;;  rv  who    date     reason

(defun return-argument-hash-table (position pred)
  (aref (pred-slot-value-hash pred) position))


;;;________________________________________________________________________________
;;; RETRIEVE-SUPERSET-OF-MATCHING-TUPLES
;;;
;;;  pred-  a pred struct
;;;  vs-    an instantiated variabilization (a b ?-1 ?-1 %-3)
;;;  key-   an positional index into vs (for positional value), or
;;;         :all-vars (only for prolog), or
;;;         :all-bound
;;;
;;;  NOTE: that equal hash array for all bound returns a list of one tuple
;;;         returns tuples of pred that may unify with vs
;;;
;;;  rv  who    date     reason

(defun retrieve-superset-of-matching-tuples (pred vs key)
  (case key
    (:all-vars (pred-pos pred))
    (:all-bound (gethash vs (pred-pos-hash pred)))
    (otherwise (gethash (nth key vs) (return-argument-hash-table key pred)))))

#|
;;;________________________________________________________________________________
;;; FIND-NEW-EQULAITY-CONSTRAINT
;;;
;;;  takes in a variables list (?1 ?2 ?-1 ?-2 ?-1 ?-1 ?-2 ?-2)
;;;  returns an index into list of all new vars (ie., negative) that appear twice (or more)
;;;  answ ((2 . 4) (4 . 5)(6 . 7))
;;;  note 2 . 5, although true, isn't needed
;;;
;;;  rv  who    date     reason

(defun find-new-equality-constraints (variabilizations &aux (p1 0) p2 (ret nil))
  (dolist (v variabilizations)
    (when (new-var? v)
      (when (setq p2 (position-if #'(lambda(x) (eq (pcvar-id v)
                                                   (pcvar-id x)))
              variabilizations
              :start (+ 1 p1)))
        (push (cons p1 p2) ret)))
    (incf p1))
  ret)
|#


;;;________________________________________________________________________________
;;; COMPUTE-NEW
;;;
;;;   returns the new variables in variabilization, sorted by pcvar-id
;;;
;;;  rv  who    date     reason

(defun compute-new (variablization)
  (sort (remove-duplicates (remove-if-not #'new-var? variablization))
        #'<
        :key #'pcvar-id))


;;;________________________________________________________________________________
;;; NEG-HASH-TABLE
;;;
;;;   returns the appropriate hash table to store # of matches
;;;
;;;  rv  who    date     reason

(defun neg-hash-table (pred negative?)
  (if (clause-p pred)
      (clause-neg-tuples-hash pred) ;;negative shouldn't be true for clauses
    (if negative?
      (p-neg-tuples-hash-for-negated pred)
      (p-neg-tuples-hash pred))
    ))

;;;________________________________________________________________________________
;;; MEMBCAR
;;;
;;;  accepts an element (e) and a list l and returns true e is the car of any element of the list
;;;
;;;  rv  who    date      reason
;;;  00  glenn  11/09/90  generally a useful function
;;;  01  glenn  05/02/91  now deals with lists with non-cons elements

(defun membcar (e l) (member e l :test #'(lambda (e1 e2) (and (consp e2) (eql e1 (car e2))))))

;;;________________________________________________________________________________
;;; UNDEFINED?    returns T if the head of a prolog literal has no corresponding p-structure
;;; EXTENSIONAL?  returns T if the head of a prolog literal is an extensional predicate
;;; INTENSIONAL?  returns T if the head of a prolog literal is an intensional predicate
;;; BUILTIN?      returns T if the head of a prolog literal is the name of a builtin predicate
;;; NEGATION?     returns T if the prolog literal begins with a not
;;; EQUALITY?     returns T if the head of a prolog literal is =
;;; IS?           returns T if the head of a prolog literal is IS
;;; CUT?          returns T if the literal is a cut symbol, !.
;;;
;;;  rv  who    date      reason
;;;  00  cliff  06/04/91  added some test useful which are useful when graphing

(defun undefined? (literal)
  (null (get-pstruct (car literal))))

(defun extensional? (literal)
  (and (pred-p (get (car literal) 'pred))
       (not (rule-p (get (car literal) 'rule)))))

(defun intensional? (literal)
  (rule-p (get (car literal) 'rule)))

(defun builtin? (literal)
  (builtin-p (get (car literal) 'builtin)))

(defun negation? (literal)
  (eql (car literal) 'not))

(defun equality? (literal)
  (eql (car literal) '=))

(defun is? (literal)
  (eql (car literal) 'is))

(defun cut? (literal)
  (eql literal '!))


;;;________________________________________________________________________________
;;; RESET-HASH-TABLES
;;;
;;;  used by subsequent runs of FOCL to re-initialize negative hash tables
;;;
;;;  rv  who    date      reason

(defun reset-hash-tables()
  (mapc #'(lambda(name.pred)
            (setf (p-neg-tuples-hash (cdr name.pred))
                  (make-hash-table :test #'equal
                                   :size (* (pred-arity (cdr name.pred)) 10)))
            (setf (p-neg-tuples-hash-for-negated (cdr name.pred))
                  (make-hash-table :test #'equal
                                   :size (* (pred-arity (cdr name.pred)) 10))))
        *extensional-preds*)

  (mapc #'(lambda(name.rule)
            (setf (p-neg-tuples-hash (cdr name.rule))
                  (make-hash-table :test #'equal
                                   :size (* (rule-arity (cdr name.rule)) 10)))
            (setf (p-neg-tuples-hash-for-negated (cdr name.rule))
                  (make-hash-table :test #'equal
                                   :size (* (rule-arity (cdr name.rule)) 10))))
        *intensional-preds*)

    (mapc #'(lambda(name.builtin)
            (setf (p-neg-tuples-hash (cdr name.builtin))
                  (make-hash-table :test #'equal
                                   :size (* (builtin-arity (cdr name.builtin)) 10)))
            (setf (p-neg-tuples-hash-for-negated (cdr name.builtin))
                  (make-hash-table :test #'equal
                                   :size (* (builtin-arity (cdr name.builtin)) 10))))
        *builtin-preds*))

