
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

(defun eval-comp (comp comp-val var-pos pos neg)
  (values
   (all-images
    #'(lambda (pos-tuple)
	(if (funcall comp (nth var-pos pos-tuple) comp-val)
            pos-tuple))
    pos)
   (all-images
    #'(lambda (neg-tuple)
	(if (funcall comp (nth var-pos neg-tuple) comp-val)
            neg-tuple))
    neg)))


;;  rv  who    date       reason
;;  00  glenn  05/13/91   revised to deal with non-numeric builtins

(defun count-builtin-matches (comp tuples variabilization)
  (let* ((var1 (first variabilization))
         (var2 (second variabilization))
         (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
         (var-pos2 (if (pcvar-p var2) (pcvar-id var2)))
         (matches
          (cond (var-pos1
                 (if var-pos2
                   (count-if #'(lambda (tuple) 
                                 (funcall comp (nth var-pos1 tuple) (nth var-pos2 tuple)))
                             tuples)
                   (count-if #'(lambda (tuple) 
                                 (funcall comp (nth var-pos1 tuple) var2)) tuples)))
                (var-pos2
                 (count-if #'(lambda (tuple) 
                               (funcall comp var1 (nth var-pos2 tuple)))
                           tuples))
                (t 
                 (count-if #'(lambda (tuple) 
                               (funcall comp var1 var2))
                           tuples)))))
    (values matches matches)))


(defun count-builtin-negation-matches (comp tuples variabilization)
  (let* ((var1 (first variabilization))
         (var2 (second variabilization))
         (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
         (var-pos2 (if (pcvar-p var2) (pcvar-id var2)))
         (matches
          (cond (var-pos1
                 (if var-pos2
                   (count-if-not #'(lambda (tuple) 
                                     (funcall comp (nth var-pos1 tuple) (nth var-pos2 tuple)))
                                 tuples)
                   (count-if-not #'(lambda (tuple) 
                                     (funcall comp (nth var-pos1 tuple) var2)) tuples)))
                (var-pos2
                 (count-if-not #'(lambda (tuple) 
                                   (funcall comp var1 (nth var-pos2 tuple)))
                               tuples))
                (t 
                 (count-if-not #'(lambda (tuple) 
                                   (funcall comp var1 var2))
                               tuples)))))
    (values matches matches)))


(defun compute-midpt (a b) (/ (+ a b) 2.0))


;;  rv  who    date       reason
;;  00  glenn  02/20/91   revised to deal with nil for pos-points or neg-points
;;  00  glenn  05/13/91   generalized to deal with non-numerics

(defun compute-boundary-points (pos-points neg-points &optional (sort-fn #'<) 
                                           &aux (numeric? (numberp (car pos-points))))
  (if numeric?
    (do ((pos pos-points)
         (neg neg-points)
         (pos-is-less 
          (and pos-points neg-points) (< (car pos-points) (car neg-points)))
         (boundary-pts)
         (mid-point))
        ((or (null pos) (null neg)) (nreverse boundary-pts))
      (cond (pos-is-less
             (cond ((or (null (cadr pos))
                        (not (< (cadr pos) (car neg)))) ; not still <
                    (setq pos-is-less nil) 
                    (setq mid-point (/ (+ (car pos) (car neg)) 2))
                    (pushnew mid-point boundary-pts))
                   (t nil))
             (setq pos (cdr pos)))
            (t
             (cond ((or (null (cadr neg))
                        (not (< (cadr neg) (car pos)))) ; not still <
                    (setq pos-is-less t)
                    (setq mid-point (/ (+ (car pos) (car neg)) 2))
                    (pushnew mid-point boundary-pts))
                   (t nil))
             (setq neg (cdr neg)))))
    ;; separated because want to process numbers more efficiently
    (do ((pos pos-points)
         (neg neg-points)
         (pos-is-less 
          (and pos-points neg-points) (funcall sort-fn (car pos-points) (car neg-points)))
         (boundary-pts))
        ((or (null pos) (null neg)) (nreverse boundary-pts))
      (cond (pos-is-less
             (cond ((or (null (cadr pos))
                        (not (funcall sort-fn (cadr pos) (car neg)))) ; not still <
                    (setq pos-is-less nil) 
                    (pushnew (car pos) boundary-pts)
                    (pushnew (car neg) boundary-pts))
                   (t nil))
             (setq pos (cdr pos)))
            (t
             (cond ((or (null (cadr neg))
                        (not (funcall sort-fn (cadr neg) (car pos)))) ; not still <
                    (setq pos-is-less t)
                    (pushnew (car pos) boundary-pts)
                    (pushnew (car neg) boundary-pts))
                  (t nil))
             (setq neg (cdr neg)))))))


;;  rv  who    date       reason
;;  00  glenn  05/13/91   added misc. functions to compute subsets of builtins that
;;    satisfy various properties


;; the following are functions useful for retrieving subsets of builtins that satisfy
;; various properties (i.e., equality, non-numeric, numeric, relational (i.e., the 
;; non-equality builtins))


;;; determine if type is a numeric type

(defun number-type (type)
  (or (equal type :integer) (equal type :real) (equal type :numeric)(equal type :number)))


;;; for builtins typically concerned with type of first arg (should be same for all
;;; comparator builtins in general)

(defun get-builtin-type (builtin)
  (car (p-type builtin)))

;;; accepts 
(defun overall-builtin-type (builtins)
  (let ((numeric-type nil)
        (non-numeric-type nil))
    (dolist (b builtins)
      (if (number-type (get-builtin-type (cdr b)))
        (setq numeric-type t)
        (setq non-numeric-type t)))
    (if numeric-type
      (if non-numeric-type 
        :anything
        'numeric-type)
      (if non-numeric-type
        'non-numeric-type))))

;;; retrieve those builtins that support a threshold

(defun thresh-builtins (builtins )
  (all-images #'(lambda (b) (if (builtin-one-variable-comp (cdr b)) b)) builtins))

;;;  not of numeric type

(defun non-numeric-type (type)
  (not (number-type type)))

;;; note this distinction (numeric vs. non-numeric) may not be useful at the moment
;;; but may be in the future (there may be some merit in separating the processing of
;;; numeric and non-numeric builtins)

(defun numeric-thresh-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (number-type (car (p-type (cdr b))))
                          (builtin-one-variable-comp (cdr b))))
                 builtins))

(defun non-numeric-thresh-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (non-numeric-type (car (p-type (cdr b))))
                          (p-builtin-one-variable-comp (cdr b))))
                 builtins))

;;;  returns those non-equality builtins that support a threshold
(defun relational-thresh-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (builtin-one-variable-comp (cdr b))
                          (not (builtin-equality? (cdr b)))))
                 builtins))

;;;  returns those equality builtins that support the use of a constant
(defun equality-constant-builtins (builtins)
  (remove-if-not #'(lambda (b) 
                     (and (builtin-one-variable-comp (cdr b))
                          (builtin-equality? (cdr b))))
                 builtins))

;;; this is used in cliches and probably should be moved there

(defun pred-type-restriction-supports-threshold (pred-type-restriction)
  (or (eql pred-type-restriction 'pred) (eql pred-type-restriction 'thresh)))


; note: bring back code that checks the negated literal  - also figure out what to do about
; variabilizations


;;__________________________________________________________________________________________
;; FIND-LITERAL-BUILTIN
;;
;;  finds builtin literal with maximum gain-  see find-max for details of params
;;
;;  returns 6 values
;;  1 new-literal             -literal structure with maximum gain
;;                            -suitble for adding to end of clause
;;  2 new-vars                -names of newvariables (renamed to be old variables)
;;  3 new-types               -types of new variables
;;  4 new-pos-tuples          -next set of positive tuples
;;  5 new-neg-tuples          -next set of negative tuples
;;  6 max-gain                -maximum information gain 
;;                            (to compare aginst intensional if necessary)
;;
;;  returns nil if no literal has positive gain 
;;
;;  rv  who    date       reason
;;  00  cliff  04/07/91   no longer accepts bits-available, nor returns literal-bits
;;                        [????]   What is the 0 in values for?
;;  01  glenn  05/02/91   got rid of conjunction baggage
 
(defun find-literal-builtin (original-info ;passed to info-gain
                             predicate-being-learned   ; used to detect recursion
                             variables
                             variable-types
                             pos-tuples
                             neg-tuples
                             original-vars
                             use-hash-tables
                             gain-to-beat)              ; previous high for minimum-gain
  
  (let (thresh-predicate-with-maximum-gain
        thresh-variabilization-with-maximum-gain 
        thresh-max-negated?
        thresh-max-gain
        covered-all-pos-tuples) ; do something with this
    
    ;;; try variable pairs first
    (multiple-value-bind (predicate-with-maximum-gain
                          variabilization-with-maximum-gain 
                          max-negated?
                          max-gain)
                         (if *builtin-threshold-only* ; ges 3/15
                           (values nil nil nil 0 nil nil)
                           (find-maximum-literal original-info
                                                 predicate-being-learned
                                                 variables
                                                 variable-types
                                                 0
                                                 pos-tuples
                                                 neg-tuples 
                                                 original-vars
                                                 use-hash-tables
                                                 *builtin-preds* 
                                                 gain-to-beat)) 
      (if (not *covered-all-pos-tuples*)
        (multiple-value-setq (thresh-predicate-with-maximum-gain
                              thresh-variabilization-with-maximum-gain 
                              thresh-max-negated?
                              thresh-max-gain
                              covered-all-pos-tuples)
                             (find-literal-builtin-thresh original-info
                                                          variables
                                                          variable-types
                                                          pos-tuples
                                                          neg-tuples 
                                                          *builtin-preds* 
                                                          (if (numberp max-gain) (max max-gain gain-to-beat) 
                                                              gain-to-beat ))))
      
      (cond ((and (eq predicate-with-maximum-gain :fail) (eq thresh-predicate-with-maximum-gain :fail))
             :fail) ;;no predicate has enough gain
            ;;;create-literals constructs all the return arguments
            ((and thresh-predicate-with-maximum-gain
                  (not (eq thresh-predicate-with-maximum-gain :fail))); threshold wins
             
             (create-literal variables 
                             pos-tuples 
                             neg-tuples 
                             thresh-predicate-with-maximum-gain 
                             thresh-variabilization-with-maximum-gain 
                             thresh-max-negated? 
                             thresh-max-gain))
            ((and predicate-with-maximum-gain
                  (not (eq predicate-with-maximum-gain :fail))); two vars wins wins 
             (create-literal variables 
                             pos-tuples 
                             neg-tuples 
                             predicate-with-maximum-gain 
                             variabilization-with-maximum-gain 
                             max-negated? 
                             max-gain))
            (t nil))))) ; at least one of the two ran out of bits

;;  this is a generally useful function and should be moved to a utilities file
;;  01  glenn  05/05/91  used when given a subset of two lists in correspondence and you want
;;    the corresponding set of the second list

(defun retrieve-corresponding-elements (new-list1 old-list1 old-list2)
  (if (equal new-list1 old-list1)
      old-list2
    (mapcar #'(lambda (e1) (nth (position e1 old-list1) old-list2)) new-list1)))

;;__________________________________________________________________________________________
;; find-literal-builtin-thresh
;;
;; returns the predicate variabilization of pred with the maximum information gain
;; works for intension and extensional pred
;;
;; returns 4 values 
;; 1. predicate-with-maximum-gain       - builtin predicate (not a structure like other routines)
;; 2. variabilization-with-maximum-gain -variabilization ?0 thru ?n are old ?-n indicates new
;; 3. max-negated?                      -t if negation of predicate has maximum gain
;; 4. max-gain                          -the amount of information gain
;;
;; returns nil if no variabilization has gain greater than 0
;;
;;  rv  who    date       reason
;;  00  cliff  04/07/91   no longer accepts bits-available, nor returns literal-bits
;;  01  glenn  05/02/91   got rid of conjunction baggage - added in cliche info
;;  02  glenn  05/05/91   moved loop over builtin-predicates to compute-threshold-gain to
;;    make it more efficient 
;;  03  glenn  05/05/91   added call to apply-variabilization-restrictions - not had to dummy
;;    up the variabilizations by making the variables into lists - see (1) below
;;  04  glenn  05/12/91   consolidated compute-threshold-gain into this function and macros
;;    broke up builtins into equality and relational-preds, and fixed it so that it can
;;    compute constants for equality preds as well as thresholds for relational preds
;;  05  glenn  05/13/91   extend builtins to include non-numeric builtins (e.g., eql
;;     and string-lessp), both numeric and non-numeric builtins are handled uniformly
;;     note - we may want to consider keeping the builtins sorted with respect to their
;;     sort-fn as this will make the computation of the boundary points more efficient
;;     (as we're processing builtins, as long as their sort fn is the same we don't need 
;;      to resort the tuples for thresholds)
;;  06  glenn  05/13/91   note two fields were added to builtins to achieve points 04 and
;;    05, they are equality? and sort-fn.

;; (1) apply-variabilization-restrictions wants variabilizations - since we don't 
;;     know what the thresholds are yet - the variabilizations are simply the 
;;     variables themselves each in an individual list - taking the car of the
;;     varzns returned gets us back to variables

(defun find-literal-builtin-thresh (original-info       ;passed to info-gain
                                    variables           ;list of old variables
                                    variable-types      ;types of old variables
                                    pos-tuples          ;positive tuples
                                    neg-tuples          ;negative
                                    builtins
                                    gain-to-beat
                                    
                                    &key
                                    var-restrictions
                                    pred-restrictions
                                    instantiated-cliche 
                                    position-in-cliche)
  (let (covered-all-pos-tuples
        gain
        (builtins-that-satisfy-restrictions
         (apply-pred-restrictions builtins pred-restrictions))
        relational-builtins
        equality-builtins
        (max-gain gain-to-beat)
        (comp-with-max-gain nil)
        (variabilization-with-maximum-gain nil)
        variabilization
        negative?
        (max-negated? nil)
        max-possible-gain)
    (setq relational-builtins 
          (relational-thresh-builtins builtins-that-satisfy-restrictions))
    (setq equality-builtins
          (equality-constant-builtins builtins-that-satisfy-restrictions))

    (when *graph-learning*                                                               ;; CB 13
      (if instantiated-cliche                                                            ;;
        (display-partially-instanciated-cliche-in-induction-graphic instantiated-cliche) ;;
        (display-cliche-name-in-induction-graphic nil)))                                 ;;

    (process-threshold-variabilizations 
       relational-builtins
       var-restrictions
       variables
       variable-types
       pos-tuples
       neg-tuples
       variabilization
       instantiated-cliche
       ; exit condition for loop over-variables
       nil
       ; body of inner loop
       ;;;  try built-ins with a single variable and a threshold value first

       (when *graph-learning*                                      ;; CB(7/27/91)
         (if negative?                                             ;;
           (negate-predicate-in-induction-graphic (p-name comp))   ;;
           (update-predicate-in-induction-graphic (p-name comp)))) ;;

       (multiple-value-setq 
        (gain max-possible-gain covered-all-pos-tuples)
        (info-gain comp                               ;;CB(03)
                   variabilization 
                   nil 
                   pos-tuples 
                   neg-tuples 
                   original-info
                   nil 
                   :instantiated-cliche instantiated-cliche 
                   :position-in-cliche position-in-cliche))

       (when *graph-learning*                                               ;; CB(7/27/91)
         (update-variablization-in-induction-graphic variabilization        ;;
                                                     instantiated-cliche    ;;
                                                     position-in-cliche))   ;;

       (when (<= max-gain gain)

         (when *graph-learning*                         ;; CB(7/27/91)
           (update-best-literal-in-induction-graphic))  ;;

         (setf max-gain gain
               variabilization-with-maximum-gain variabilization
               comp-with-max-gain comp
               max-negated? nil)
         (check-pruning-III covered-all-pos-tuples find-literal-builtin-thresh 
                            (values comp-with-max-gain
                                    variabilization-with-maximum-gain 
                                    max-negated?
                                    max-gain
                                    covered-all-pos-tuples)))
       (when (>= 0 gain)

       (when *graph-learning*                                      ;; CB(7/27/91)
         (if negative?                                             ;;
           (update-predicate-in-induction-graphic (p-name comp))   ;;
           (negate-predicate-in-induction-graphic (p-name comp)))) ;;

         (multiple-value-setq 
          (gain max-possible-gain covered-all-pos-tuples)
          (info-gain comp                               ;;CB(03)
                     variabilization 
                     t 
                     pos-tuples 
                     neg-tuples 
                     original-info
                     nil 
                     :instantiated-cliche instantiated-cliche 
                     :position-in-cliche position-in-cliche))

       (when *graph-learning*                                               ;; CB(7/27/91)
         (update-variablization-in-induction-graphic variabilization        ;;
                                                     instantiated-cliche    ;;
                                                     position-in-cliche))   ;;

         (when (< max-gain gain)

           (when *graph-learning*                         ;; CB(7/27/91)
             (update-best-literal-in-induction-graphic))  ;;

           (setf max-gain gain
                 variabilization-with-maximum-gain variabilization
                 comp-with-max-gain comp
                 max-negated? t)
           (check-pruning-III covered-all-pos-tuples find-literal-builtin-thresh 
                              (values comp-with-max-gain
                                      variabilization-with-maximum-gain 
                                      max-negated?
                                      max-gain
                                      covered-all-pos-tuples)))))
    ; handle non-numeric builtins - i.e., generating constants
    (process-equality-constant-varzns 
       equality-builtins
       var-restrictions
       variables
       variable-types
       pos-tuples
       neg-tuples
       variabilization
       negative?
       instantiated-cliche
       ; exit condition for loop over-variables
       nil
       ; body of inner loop
       ;;;  try built-ins with a single variable and a threshold value first

       (when *graph-learning*                                                               ;; CB 13
         (if instantiated-cliche                                                            ;;
           (display-partially-instanciated-cliche-in-induction-graphic instantiated-cliche) ;;
           (display-cliche-name-in-induction-graphic nil)))                                 ;;

       (when *graph-learning*                                      ;; CB(7/27/91)
         (if negative?                                             ;;
           (negate-predicate-in-induction-graphic (p-name comp))   ;;
           (update-predicate-in-induction-graphic (p-name comp)))) ;;

       (multiple-value-setq 
        (gain max-possible-gain covered-all-pos-tuples)
        (info-gain comp                               ;;CB(03)
                   variabilization 
                   negative? 
                   pos-tuples 
                   neg-tuples 
                   original-info
                   nil 
                   :instantiated-cliche instantiated-cliche 
                   :position-in-cliche position-in-cliche))

       (when *graph-learning*                                               ;; CB(7/27/91)
         (update-variablization-in-induction-graphic variabilization        ;;
                                                     instantiated-cliche    ;;
                                                     position-in-cliche))   ;;

       (when (if negative? (< max-gain gain) (<= max-gain gain))

         (when *graph-learning*                         ;; CB(7/27/91)
           (update-best-literal-in-induction-graphic))  ;;

         (setf max-gain gain
               variabilization-with-maximum-gain variabilization
               comp-with-max-gain comp
               max-negated? negative?)
         (check-pruning-III covered-all-pos-tuples find-literal-builtin-thresh 
                            (values comp-with-max-gain
                                    variabilization-with-maximum-gain 
                                    max-negated?
                                    max-gain
                                    covered-all-pos-tuples))))
    (cond ((and (null comp-with-max-gain)
                *stopping-criteria-enabled*
                (some-literal-required-too-many-bits)) nil)
          ((null comp-with-max-gain) :fail)
          (t 
           (values comp-with-max-gain
                   variabilization-with-maximum-gain 
                   max-negated?
                   max-gain
                   covered-all-pos-tuples)))
))



;;;  for builtin predicates computes the information gain of comp using a single variable 
;;;  (var) and a threshold value for the comparison.  The threshold with the best info-gain is
;;;  returned provided it beats gain-to-beat.  A triple is returned consisting of the gain
;;;  of the best threshold, the threshold value (nil signifies that we couldn't beat gain-to-beat),
;;;  and the max-negated? flag. 

;;__________________________________________________________________________________________
;; compute-threshold-gain
;;
;;  for builtin predicates computes the information gain of all builtins (note this should be
;;  the subset of builtins that have their one-variable-comp flag set) using a single variable 
;;  (var) and a threshold value for the comparison.  The threshold with the best info-gain is
;;  returned provided it beats gain-to-beat.  4 values are returned consisting of the gain of 
;;  the best threshold and builtin, the best builtin (comp-with-max-gain), the threshold value 
;;  (nil signifies that we couldn't beat gain-to-beat), and the max-negated? flag.
;;
;;  rv  who    date       reason
;;  00  cliff  04/07/91   no longer accepts bits-available, nor returns literal-bits
;;  01  glenn  05/02/91   got rid of conjunction baggage
;;  02  glenn  05/05/91   moved loop over builtin predicates into compute-threshold gain to
;;                        avoid resorting the pts etc. for each predicate
;;  03  cliff  05/08/91   replaced "real-info-gain" with "info-gain"

(defun compute-threshold-gain (builtins
                               var
                               pos-tuples
                               neg-tuples
                               gain-to-beat
                               original-info

                               &key
                               instantiated-cliche
                               position-in-cliche)
  (let ((max-gain gain-to-beat)
        (pos (sort (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) pos-tuples) #'<))
        (neg (sort (mapcar #'(lambda (tuple) (nth (pcvar-id var) tuple)) neg-tuples) #'<))
        (threshold nil)
        (comp-with-max-gain nil)
        (gain 0)
        (max-negated? nil)
        (max-possible-gain nil)
        (covered-all-pos-tuples nil)
        (boundary-pts nil))
    (setq boundary-pts (compute-boundary-points pos neg))
    (do* ((comps builtins (cdr comps))
          (comp (cdr (car comps)) (cdr (car comps))))
         ((null comps))
      (do* ((boundary-points boundary-pts (cdr boundary-points))
            (boundary-point (car boundary-points) (car boundary-points)))
           ((null boundary-points)) 
        ; fix so it computes boundary points
        (multiple-value-setq 
         (gain max-possible-gain covered-all-pos-tuples)
         (info-gain comp                               ;;CB(03)
                    (list var boundary-point) 
                    nil 
                    pos-tuples 
                    neg-tuples 
                    original-info
                    nil 
                    :instantiated-cliche instantiated-cliche 
                    :position-in-cliche position-in-cliche))
        (when (<= max-gain gain)
          (setf max-gain gain
                threshold boundary-point
                comp-with-max-gain comp
                max-negated? nil))
        (check-pruning-III covered-all-pos-tuples compute-threshold-gain 
                           (values max-gain comp-with-max-gain threshold max-negated? 
                                   covered-all-pos-tuples))
        (when (>= 0 gain)
          (multiple-value-setq 
           (gain max-possible-gain covered-all-pos-tuples)
           (info-gain comp                             ;;CB(03)
                           (list var boundary-point) 
                           t 
                           pos-tuples 
                           neg-tuples 
                           original-info
                           nil 
                           :instantiated-cliche instantiated-cliche 
                           :position-in-cliche position-in-cliche))
          (when (< max-gain gain)
            (setf max-gain gain
                  threshold boundary-point
                  comp-with-max-gain comp
                  max-negated? t))
          (check-pruning-III covered-all-pos-tuples compute-threshold-gain 
                             (values max-gain comp-with-max-gain threshold max-negated? 
                                     covered-all-pos-tuples)))))
    (values max-gain comp-with-max-gain threshold max-negated? covered-all-pos-tuples)))


;;;  computing info-gain - note we may want to splice this into info gain

(defun builtin-info-gain (comp var negative? pos neg comp-val
			       &optional (original-info (I-content (length pos) (Length neg)))
			       &aux pos-matches gain neg-matches)

  (setq pos-matches
        (if negative?
	    (count-builtin-negation-matches comp comp-val pos)
          (count-builtin-matches comp comp-val pos)))
  (cond ((= pos-matches 0) (setq gain 0)) ;it's probably less than 0, but we won't choose it anymay
        (t (setq neg-matches 
                 (if negative?
		     (count-builtin-negation-matches comp comp-val neg)
                   (count-builtin-matches comp comp-val neg)))
;;;MP USE I-GAIN
           (setq gain 
;                 (* pos-matches 
;                    (- original-info (- (log (/ pos-matches (+ pos-matches neg-matches)) 2)))))))
                 (I-gain original-info pos-matches pos-matches neg-matches))))
                
					; note check on orig-pos-matches and orig-neg-matches
  (when (member :i *focl-trace-level*)
	(format t "~%~a(~a ~a ~a) ~30Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~68T gain: ~5f"
		(if negative? "~" " ")
		comp var comp-val pos-matches pos-matches (length pos)
		neg-matches neg-matches (length neg) gain))
  gain)



;;;  computing info-gain for built-ins comparing two variables 

(defun builtin-info-gain-var (comp var1 var2 negative? pos neg 
				   &optional (original-info (I-content (length pos) (Length neg)))
                       &aux pos-matches gain neg-matches)
  (setq pos-matches
        (if negative?
          (count-builtin-var-negation-matches comp (pcvar-id var1) (pcvar-id var2) pos)
          (count-builtin-var-matches comp (pcvar-id var1) (pcvar-id var2) pos)))
  (cond ((= pos-matches 0) (setq gain 0))  ;it's probably less than 0, but we won't choose it anymay
        (t (setq neg-matches 
                 (if negative?
                   (count-builtin-var-negation-matches comp (pcvar-id var1) (pcvar-id var2) neg)
                   (count-builtin-var-matches comp (pcvar-id var1) (pcvar-id var2) neg)))

;;;MP USE I-GAIN
           (setq gain 
;                 (* pos-matches 
;                    (- original-info (- (log (/ pos-matches (+ pos-matches neg-matches)) 2)))))))
                 (I-gain original-info pos-matches pos-matches neg-matches))))
                
; note check on orig-pos-matches and orig-neg-matches
  (when (member :i *focl-trace-level*)
    (format t "~%~a(~a ~a ~a) ~30Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~68T gain: ~5f"
            (if negative? "~" " ")
             comp var1 var2 pos-matches pos-matches (length pos)
            neg-matches neg-matches (length neg) gain))
  gain)



(defun bound-var? (var)
  (or (numberp var) (and (pcvar-p var) (> (pcvar-id var) -1))))

;;  rv  who    date       reason
;;  00  glenn  05/13/91   revised to deal with non-numeric builtins

;;; like count-builtin matches, but returns those tuples that match
;;; note variablization currently should be a list with only two elements - first element 
;;; is the variable undergoing the comparison and the second var is the comp val
(defun extend-tuples-builtin (pred tuples variabilization)
  (let* ((var1 (first variabilization))
         (var2 (second variabilization))
         (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
         (var-pos2 (if (pcvar-p var2) (pcvar-id var2)))
         (comp (builtin-function pred)))
    (cond (var-pos1
           (if var-pos2
             (all-images
              #'(lambda (tuple)
                  (if (funcall comp (nth var-pos1 tuple) (nth var-pos2 tuple))
                    tuple))
              tuples)
             (all-images
              #'(lambda (tuple)
                  (if (funcall comp (nth var-pos1 tuple) var2)
                    tuple))
              tuples)))
          (var-pos2
           (all-images
            #'(lambda (tuple)
                (if (funcall comp var1 (nth var-pos2 tuple))
                  tuple))
            tuples))
          (t 
           (if (funcall comp var1 var2) 
             tuples 
             nil)))))


;;  rv  who    date       reason
;;  00  glenn  05/13/91   revised to deal with non-numeric builtins

;;; filters those tuples that satisfy built-in predicate pred
;;; note variablization currently should be a list with only two elements - first element 
;;; is the variable undergoing the comparison and the second var is the comp val
(defun extend-negation-tuples-builtin (pred tuples variabilization)
  (let* ((var1 (first variabilization))
         (var2 (second variabilization))
         (var-pos1 (if (pcvar-p var1) (pcvar-id var1)))
         (var-pos2 (if (pcvar-p var2) (pcvar-id var2)))
         (comp (builtin-function pred)))
    (cond (var-pos1
           (if var-pos2
             (remove-if
              #'(lambda (tuple)
                  (if (funcall comp (nth var-pos1 tuple) (nth var-pos2 tuple))
                    tuple))
              tuples)
             (remove-if
              #'(lambda (tuple)
                  (if (funcall comp (nth var-pos1 tuple) var2)
                    tuple))
              tuples)))
          (var-pos2
           (remove-if
            #'(lambda (tuple)
                (if (funcall comp var1 (nth var-pos2 tuple))
                  tuple))
            tuples))
          (t 
           (if (funcall comp var1 var2) 
             nil 
             tuples)))))
  

