
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstien
;;;; and Kamal Ali.  

(in-package :user)

;;;______________________________________________________________________________
;;; CLAUSE-CONTAINS-UNBOUND-VARS
;;;
;;;  Given a prolog literal (clause-head) and a literal-structure (clause-body)
;;;  This function determines if there are any variables in the clause which
;;;  are unbound.  A mode of (:+ or :?) for a variable is assumed to mean that
;;;  occurance is a definition of the variable.  A mode of (:-) for a variable
;;;  means that occurance is a use and must have a balancing definition elsewhere
;;;  in the clause.
;;;                 
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/23/91  Created to replace literal-causes-unbound-vars in simplify

(defun clause-contains-unbound-vars (clause-head    ;; prolog literal
                                     clause-body)   ;; linked list of literal structures

  (multiple-value-bind (def-list use-list)
                       (def-and-use-from-prolog-literal clause-head)
    (multiple-value-setq (def-list use-list)
                         (def-and-use-from-clause clause-body def-list use-list))
    (if (every #'(lambda (use-var) (member use-var def-list :test #'eql)) use-list)
      nil
      t)))

;;;______________________________________________________________________________
;;; DEF-AND-USE-FROM-PROLOG-LITERAL
;;;                 
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/23/91  Created

(defun def-and-use-from-prolog-literal (literal &optional (def-list nil) (use-list nil))
  (let ((pstruct (get-pstruct (first literal))))
    (when pstruct
      (do* ((vars (rest literal) (cdr vars))
            (var (car vars) (car vars))
            (modes (or (p-mode pstruct) (make-list (p-arity pstruct) :initial-element :?)) (cdr modes)))
           ((null vars))
      (if (variable-p var)
        (case (car modes)
          ((:? :+) (setf def-list (pushnew var def-list)))
          ((:-) (setf use-list (pushnew var use-list)))))))
    (values def-list use-list)))

;;;______________________________________________________________________________
;;; DEF-AND-USE-FROM-CLAUSE
;;;                 
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/23/91  Created

(defun def-and-use-from-clause (clause &optional (def-list nil) (use-list nil))
  (do* ((literal clause (literal-next literal)))
       ((null literal))
    (if (literal-negated? literal)
      (multiple-value-setq (def-list use-list)
                           (def-and-use-from-clause (literal-negated-literals literal) def-list use-list))
      (let ((pstruct (get-pstruct (literal-predicate-name literal))))
        (do* ((vars (literal-variablization literal) (cdr vars))
              (var (car vars) (car vars))
              (modes (or (p-mode pstruct) (make-list (p-arity pstruct) :initial-element :?)) (cdr modes)))
             ((null vars))
          (if (variable-p var)
            (case (car modes)
              ((:? :+) (setf def-list (pushnew var def-list)))
              ((:-) (setf use-list (pushnew var use-list)))))))))
  (values def-list use-list))

;;;______________________________________________________________________________
;;; CREATE-PRUNE-SET-FROM-PRED-STRUCT
;;;
;;;  Finds the pred-struct associated with pred-name and removes percent-prune-set
;;;  of the positive and percent-prune-set of the negative examples at random.
;;;  The sets of removed examples are returned as the prune set.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   7/14/91  Created

(defun create-prune-set-from-pred-struct (pred-name &key (percent-prune-set 33.33))
  (let* ((pred (get pred-name 'pred))
         (example-set-pos (pred-pos pred))
         (example-set-neg (pred-neg pred))
         (length-example-set-pos (length example-set-pos))
         (length-example-set-neg (length example-set-neg))
         (prune-set-pos nil)
         (prune-set-neg nil)
         (length-prune-set-pos (round (* length-example-set-pos percent-prune-set) 100))
         (length-prune-set-neg (round (* length-example-set-neg percent-prune-set) 100))
         (cons-cell nil)
         (position 0))

      ;; Extract Positive Prune Set
      (dotimes (i length-prune-set-pos)
        (setf position (- (random length-example-set-pos) 1))
        (cond ((= position -1)
               (setf prune-set-pos (push (car example-set-pos) prune-set-pos)
                     example-set-pos (cdr example-set-pos)))
              (t
               (setf cons-cell (nthcdr position example-set-pos)
                     prune-set-pos (push (cadr cons-cell) prune-set-pos))
               (rplacd cons-cell (cddr cons-cell))))
        (decf length-example-set-pos 1))

      ;; Extract Negative Prune Set
      (dotimes (i length-prune-set-neg)
        (setf position (- (random length-example-set-neg) 1))
        (cond ((= position -1)
               (setf prune-set-neg (push (car example-set-neg) prune-set-neg)
                     example-set-neg (cdr example-set-neg)))
              (t
               (setf cons-cell (nthcdr position example-set-neg)
                     prune-set-neg (push (cadr cons-cell) prune-set-neg))
               (rplacd cons-cell (cddr cons-cell))))
        (decf length-example-set-neg 1))

      ;; Redefine pred-struct with training-set = (example-set - prune-set)
      (eval `(def-pred ,pred-name
               :pos ,example-set-pos
               :neg ,example-set-neg
               :type ,(pred-type pred)
               :constraint ,(pred-constraint pred)
               :mode ,(pred-mode pred)
               :commutative ,(pred-commutative pred)
               :induction ,(pred-induction pred)
               :infix ,(pred-infix pred)
               :vars ,(pred-vars pred)
               :questions ,(pred-questions pred)))

      (values prune-set-pos prune-set-neg)))

;;;______________________________________________________________________________
;;; RESTORE-ORIGINAL-PRED-STRUCT
;;;
;;;  Adds the examples in prune-set to the examples in pred-name's pred-struct.
;;;  Note: Applying CREATE-PRUNE-SET-FROM-PRED-STRUCT followed by
;;;        RESTORE-ORIGINAL-PRED-STRUCT will restore the examples in the
;;;        orginal pred-struct (assuming the proper arguments are given),
;;;        but the order will probably have changed.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   7/14/91  Created

(defun restore-original-pred-struct (pred-name prune-set-pos prune-set-neg)
  (let* ((pred (get pred-name 'pred))
         (example-set-pos (pred-pos pred))
         (example-set-neg (pred-neg pred)))

    ;; Redefine pred-struct with example-set = (training-set + prune-set)
    (eval `(def-pred ,pred-name
             :pos ,(nconc example-set-pos prune-set-pos)
             :neg ,(nconc example-set-neg prune-set-neg)
             :type ,(pred-type pred)
             :constraint ,(pred-constraint pred)
             :mode ,(pred-mode pred)
             :commutative ,(pred-commutative pred)
             :induction ,(pred-induction pred)
             :infix ,(pred-infix pred)
             :vars ,(pred-vars pred)
             :questions ,(pred-questions pred)))))

;;;______________________________________________________________________________
;;; MAKE-CONCEPT-FUNCTION
;;;                 
;;;   Converts a concept definition learned by FOCL into an executable lisp function
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Created

(defun make-concept-function (concept-name arity concept-definition)
  (let* ((old-vars (do ((i (- arity 1) (decf i))
                        (result nil))
                       ((< i 0) result)
                     (push (make-pcvar :id i) result) ))
         (prolog-clauses (mapcar #'(lambda (clause-body)
                                     (cons (cons concept-name old-vars)
                                           (convert-to-prolog clause-body)))
                                 concept-definition)))
    (focl-compile-predicate concept-name arity prolog-clauses)))


;;;______________________________________________________________________________
;;; NUMBER-CORRECT
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Created

(defun number-correct (concept-description pos-set neg-set)
  
  (let ((correct 0)
        (concept-function (make-concept-function 'concept-description 
                                                 (length (or (first pos-set) (first neg-set)))
                                                 concept-description)))
    (dolist (p pos-set)
      (if (prove-function? concept-function nil nil p)
        (incf correct)))
    (dolist (n neg-set)
      (if (not (prove-function? concept-function nil nil n))
        (incf correct)))
    correct))


;;;______________________________________________________________________________
;;; REDUCED-ERROR-PRUNING
;;;
;;;   Destructively prunes concept-description to increase its accuracy
;;;   on the set { pos-set neg-set }.  During one pass the pruning algorithm
;;;   independently applies each operator in all posisble ways and retains
;;;   the single modification which leads to the greatest improvement in
;;;   accuracy.  Mulitple passes are made over the concept-description until
;;;   all operators, if applied, would result in a decrease in accuracy on
;;;   the pruning set.  At this point pruning terminates and the pruned 
;;;   concept-description is returned.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Completely rewritten

(defun Reduced-Error-Pruning (&key concept-description
                                   pos-set
                                   neg-set
                                   (operators (list #'delete-last-literal
                                                    #'drop-any-clause)))

  (let ((clause-head (cons *predicate-being-learned*
                           (do ((i (- (p-arity (get-pstruct *predicate-being-learned*)) 1) (decf i)) 
                                (result nil))
                               ((< i 0) result)
                             (push (make-pcvar :id i) result)))))

  (do ((number-to-beat (- (number-correct concept-description pos-set neg-set) 1)
                        (decf number-to-beat))
       (best-operator nil nil)
       (terminate nil))
      (terminate concept-description)

    (setf terminate t)
    (dolist (operator operators)
      (let ((correct (funcall operator clause-head concept-description pos-set neg-set :test)))
        (if (> correct number-to-beat)
          (setf number-to-beat correct
                best-operator operator
                terminate nil))))

    (if (not terminate)
      (setf concept-description 
            (funcall best-operator clause-head concept-description pos-set neg-set :apply))))))



;;;______________________________________________________________________________
;;;  REDUCED-ERROR-PRUNING OPERATORS
;;;______________________________________________________________________________

;;;______________________________________________________________________________
;;; delete-last-literal
;;;
;;;  :test Independently marks the last undeleted literal of each clause in
;;;        concept-description as deleted and determines the number of examples
;;;        correctly classified.  Records the deletion which leads to the most
;;;        correctly classified examples, and returns that number.
;;;
;;;  :apply (called after :test) Deletes the literal from concept-description
;;;         which leads to the most correctly classified examples.  Returns
;;;         the modified concept-description.
;;;
;;; NOTE: This might cause problems if the last undeleted literal is the only
;;;       undeleted literal.  In theory this would cause the body to be empty
;;;       (ie, no antecedents) and therefore be monotonically true.  This
;;;       would cause the concept description to always return true.
;;;
;;; NOTE: If the last literal of a clause is a negated conjunctions of literals,
;;;       delete-last-literal will delete the entire conjunction
;;;
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Created

(let ((best-literal-to-delete nil))
  (defun delete-last-literal (clause-head concept-description pos-set neg-set op)
    (declare (ignore clause-head))
    (labels 
      ((last-undeleted-literal (L &optional (last nil))
        (cond ((null L) last)
              ((literal-deleted? L) (last-undeleted-literal (literal-next L) last))
              (t (last-undeleted-literal (literal-next L) L)))))

      (case op
        (:test (let ((most-correct 0)
                     (current-correct 0)
                     (last-literal nil))
                 (dolist (clause-body concept-description)

                   (setf last-literal  (last-undeleted-literal clause-body))
                   (when last-literal
                     (setf (literal-deleted? last-literal) t
                          current-correct (number-correct concept-description pos-set neg-set))
                     (if (> current-correct most-correct)
                       (setf most-correct current-correct
                             best-literal-to-delete last-literal))
                     (setf (literal-deleted? last-literal) nil)))
                 most-correct))
        
        (:apply (setf (literal-deleted? best-literal-to-delete) t)
                concept-description))))
  )

;;;______________________________________________________________________________
;;; delete-any-literal
;;;
;;;  :test Independently marks the each undeleted literal in concept-description
;;;        as deleted and determines the number of examples correctly classified.
;;;        Records the deletion which leads to the most correctly classified
;;;        examples, and returns that number.
;;;
;;;  :apply (called after :test) Deletes the literal from concept-description
;;;         which leads to the most correctly classified examples.  Returns
;;;         the modified concept-description.
;;;
;;; NOTE: If the last literal of a clause is a negated conjunctions of literals,
;;;       delete-last-literal will delete the entire conjunction
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Created

(let ((best-literal-to-delete nil))
  (defun delete-any-literal (clause-head concept-description pos-set neg-set op)
    (case op
      (:test (let ((most-correct 0)
                   (current-correct 0))
               (dolist (clause-body concept-description)
                 (do ((literal clause-body (literal-next literal)))
                     ((null literal) nil)
                   (when (not (literal-deleted? literal))
                     (setf (literal-deleted? literal) t)
                     (unless (clause-contains-unbound-vars clause-head clause-body)
                       (setf current-correct (number-correct concept-description pos-set neg-set))
                       (if (> current-correct most-correct)
                         (setf most-correct current-correct
                               best-literal-to-delete literal)))
                     (setf (literal-deleted? literal) nil))))
               most-correct))
      
      (:apply (setf (literal-deleted? best-literal-to-delete) t)
              concept-description)))
  )

;;;______________________________________________________________________________
;;; drop-any-clause
;;;
;;; drop-any-clause marks the last undeleted literal of a clause as
;;;       deleted.
;;;  :test Independently drops each clause in concept-description and
;;;        determines the number of examples correctly classified.
;;;        Records the modification which leads to the most correctly
;;;        classified examples, and returns that number.
;;;
;;;  :apply (called after :test) Drops the clause from concept-description
;;;         which leads to the most correctly classified examples.  Returns
;;;         the modified concept-description.
;;;
;;; NOTE: This might cause problems if the last undeleted literal is the
;;;       only undeleted literal.  In theory this would cause the concept-
;;;       description to be empty and therefore be monotonically false.
;;;
;;;  revisions
;;;  rv  who    date      reason
;;;  00  cliff   5/02/91  Created

(let ((best-clause-to-drop nil))
  (defun drop-any-clause (clause-head concept-description pos-set neg-set op)
    (declare (ignore clause-head))
    (case op
      (:test (let ((most-correct 0)
                   (current-correct 0))
               (dolist (clause concept-description)
                 (setf current-correct
                       (number-correct (remove clause concept-description) pos-set neg-set))
                 (if (> current-correct most-correct)
                   (setf most-correct current-correct
                         best-clause-to-drop clause)))
               most-correct))
      
      (:apply (delete best-clause-to-drop concept-description))))
  )

(provide :REP)