
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

(defparameter *gain-window-width* 340)
(defparameter *gain-window-height* 115)

;;;===============================
;;;  gain-window

(defclass gain-window (learning-window)
  ((not-node :initarg :not-node :initform nil :accessor not-node)
   (node :initarg :node :initform nil :accessor node)
   (not-cell :initarg :not-cell :initform nil :accessor not-cell)
   (cell :initarg :cell :initform nil :accessor cell)
   (s-graph :initarg :s-graph :initform nil :accessor s-graph)
   (s-root :initarg :s-root :initform nil :accessor s-root)
   (c-graph :initarg :c-graph :initform nil :accessor c-graph)
   (c-root :initarg :c-root :initform nil :accessor c-root)
   ))

;;;_______________________________
;;;  initialize-instance

(defmethod initialize-instance ((window gain-window) &rest initargs)
  (apply #'call-next-method window initargs )
  (let ((font '("Chicago" 12 :SRCOR :PLAIN)))
    (add-subviews 
     window
     (make-dialog-item 'static-text-dialog-item #@(7 2) #@(100 16) "" nil :view-font font :view-nick-name :source)
     (make-dialog-item 'static-text-dialog-item #@(115 2) #@(80 16) "" nil :view-font font :view-nick-name :gain)
     (make-dialog-item 'static-text-dialog-item #@(220 2) #@(60 16) "" nil :view-font font :view-nick-name :pos)
     (make-dialog-item 'static-text-dialog-item #@(280 2) #@(60 16) "" nil :view-font font :view-nick-name :neg)
     )
    (let* ((view (graph-view window))
           (c-graph (create-graph))
           (s-graph (create-graph))
           (s-root (graph-root s-graph))
           (c-root (graph-root c-graph))
           (not-node (get-node s-graph :r-struct (get-r-struct 'not) :kind :not))
           (node (get-node s-graph))
           (not-cell (get-cell view :node not-node :text "not" :hidden? t))
           (cell (get-cell view :node node :hidden? t)))
      (get-cell view :node s-root :hidden? t)
      (setf (graph view) s-graph
            (root view) s-root
            (s-graph window) s-graph
            (c-graph window) c-graph
            (s-root window) s-root
            (c-root window) c-root
            (not-node window) not-node
            (node window) node
            (not-cell window) not-cell
            (cell window) cell
            (node-antecedents s-root) (list (list not-node))
            (node-antecedents not-node) (list (list node))
            (node-selection-constraint view) :no-drag))))

;;;_______________________________
;;;   view-key-event-handler

(defmethod view-key-event-handler ((window gain-window) char)
  (let* ((view (graph-view window)))
    (case (char-code char)
      ((71 103) (redisplay-last-node-selected view))   ;;  G g
      ((73 105) (inspect-last-node-selected view))     ;;  I i
      ((82 114) (invalidate-view window t))            ;;  R r
      ((84 116) (show-tuples-last-node-selected view)) ;;  T t
      (others nil))))

;;;_______________________________
;;;  hide-cells

(defmethod hide-cells ((window gain-window))
  (without-interrupts
   (let ((view (graph-view window)))
     (free-node (node-antecedents (c-root window)) (c-graph window) t)
     (setf (graph view) (s-graph window)
           (root view) (s-root window)
           (node-antecedents (c-root window)) nil
           (node-selected? (not-node window)) nil
           (node-selected? (node window)) nil
           (cell-hidden? (not-cell window)) t
           (cell-hidden? (cell window)) t
           (view-top view) 0
           (view-left view) 0)
     (place-cell (not-cell window) 20 (default-top window))
     (place-cell (cell window) 70 (default-top window)))))

;;;_______________________________
;;;  clear-gain-window

(defun clear-gain-window (window)
  (without-interrupts 
   (when (window-open? window)
     (let ((scroller (graph-scroller window)))
       (dialog-item-disable (view-named :h-scroll-bar scroller))
       (dialog-item-disable (view-named :v-scroll-bar scroller))
       (clear-caption window)
       (hide-cells window)
       (invalidate-view window t)))))

;;;_______________________________
;;;  setup-BEST-GAIN-WINDOW

(defun setup-BEST-GAIN-WINDOW ()
  (unless (window-open? *BEST-GAIN-WINDOW*)
    (setf *BEST-GAIN-WINDOW*
          (make-instance 'gain-window
                         :window-show nil
                         :window-title "Best Gain"
                         :view-font *default-font*
                         :view-size (make-point *gain-window-width* *gain-window-height*)
                         :view-position (make-point 2 (- *screen-height* *gain-window-height* 3))
                         :close-box-p t)))
  (clear-gain-window *BEST-GAIN-WINDOW*)
  (window-select *BEST-GAIN-WINDOW*))

;;;_______________________________
;;;  setup-CURRENT-GAIN-WINDOW

(defun setup-CURRENT-GAIN-WINDOW ()
  (unless (window-open? *CURRENT-GAIN-WINDOW*)
    (setf *CURRENT-GAIN-WINDOW*
          (make-instance 'gain-window
                         :window-show nil
                         :window-title "Current Gain"
                         :view-font *default-font*
                         :view-size (make-point *gain-window-width* *gain-window-height*)
                         :view-position (make-point 2 (- *screen-height* *gain-window-height* *gain-window-height* 26))
                         :close-box-p t)))
  (clear-gain-window *CURRENT-GAIN-WINDOW*)
  (window-select *CURRENT-GAIN-WINDOW*))

;;;_______________________________
;;;  display-gain-caption

(defun display-gain-caption (window &key (source *status*) (gain nil) (pos nil) (neg nil))
  (without-interrupts
   (multiple-value-bind (g p n) (if (gain-p gain) (values (gain-gain gain) (gain-pp gain) (gain-nn gain)) (values gain pos neg))
     (display-caption window
                      (format nil "~A" source)
                      (if (numberp g) (format nil "gain: ~4F" g) (format nil "gain: ~A" (or g "----")))
                      (format nil "~5@A +" p)
                      (format nil "~5@A -" n)))))

;;;_______________________________
;;;  display-struct-vars-gain

(defmethod display-struct-vars-gain ((window gain-window) r-struct vars negated?
                                     &key (literal nil) (source *status*)
                                          (gain nil) (pos nil) (neg nil) (update-gain? t))
  (without-interrupts
   (when (window-open? window)
     (let ((view (graph-view window))
           (not-node (not-node window))
           (node (node window))
           (not-cell (not-cell window))
           (cell (cell window)))
       (unless (eq (graph view) (s-graph window)) (hide-cells window))
       (setf (node-selected? not-node) nil
             (node-selected? node) nil
             (node-state not-node) source
             (node-state node) source
             (node-vars node) vars
             (node-aux node) literal
             (node-r-struct node) r-struct
             (node-kind node) (r-kind r-struct)
             (cell-hidden? not-cell) (not negated?)
             (cell-hidden? cell) nil
             (cell-text cell) (node-string node))
       (size-cell cell)
       (when update-gain? (display-gain-caption window :source source :gain gain :pos pos :neg neg))
       (setf (graph-left view) (cell-left not-cell)
             (graph-top view) (cell-top cell)
             (graph-right view) (cell-right cell)
             (graph-bottom view) (cell-bottom cell))
       (reset-scroll-bars (graph-scroller window))
       (invalidate-view view t)
       ))))

;;;_______________________________
;;;  display-winner-gain

(defun display-winner-gain (window winner
                                   &key (vars nil) (negated? nil) (source (current-status))
                                   (gain nil) (pos nil) (neg nil) (update-gain? t)
                                   (instantiated-cliche nil))
  (without-interrupts 
   (when (window-open? window)
     (if (and (r-p winner) (null instantiated-cliche))
       (display-struct-vars-gain window winner vars negated? :source source :gain gain :pos pos :neg neg :update-gain? update-gain?)
       (let ((view (graph-view window))
             (scroller (graph-scroller window))
             (graph (c-graph window))
             (root (c-root window))
             (from-cliche nil)
             nodes)

         (when (winner-p winner)
           (setf vars (winner-vars winner)
                 negated? nil
                 source (winner-source winner)
                 gain winner
                 winner (winner-literal winner)))
         
         (when (or (eq source :cliche) instantiated-cliche)
           (setf from-cliche t))

         (when update-gain? (display-gain-caption window :source source :gain gain :pos pos :neg neg))
         (hide-cells window)
         (setf (graph view) graph
               (root view) root
               nodes (reduce-nodes (cond
                                    (from-cliche (convert-cliche-to-nodes instantiated-cliche winner vars negated? graph))
                                    ((node-p winner) (duplicate-node winner nil graph))
                                    ((conjunction-p winner) (duplicate-node winner nil graph))
                                    ((disjunction-p winner) (duplicate-node winner nil graph))
                                    ((literal-p winner) (convert-literals-to-nodes winner graph))
                                    ((literal-disjunction-p winner) (convert-literals-to-nodes winner graph))
                                    (t (convert-prolog-to-nodes winner source graph)))))

         (unless (and (node-p nodes) (node-not? nodes) (not from-cliche))
           (let* ((base (get-node graph :consequent root :kind :intensional :state (if (node-p nodes) :blank-no-lines :blank) :r-struct instantiated-cliche)) )
             (setf (node-antecedents root) (list (list base))
                   root base)
             (get-cell view :node base :text (if from-cliche "clich" "   "))))

         (cond ((node-p nodes)
                (setf (node-antecedents root) (list (list nodes))
                      (node-consequent nodes) root))
               ((conjunction-p nodes)
                (setf (node-antecedents root) (list nodes))
                (dolist (node nodes)
                  (setf (node-consequent node) root)))
               ((disjunction-p nodes)
                (setf (node-antecedents root) nodes)
                (dolist (conjunction nodes)
                  (dolist (node conjunction)
                    (setf (node-consequent node) root)))))

         (display-tree-cells view)
         (size-all-cells view)
         (position-cells view)
         (position-graph view :centered t)
         (reset-scroll-bars scroller)
         (invalidate-view view t)
         )))))



;;;===============================
;;;  winners-window

(defclass winners-window (gain-window)
  ((winners :initarg :winners :accessor winners-window-winners)
   (current :initarg :current :accessor winners-window-current)
   (selection :initarg :selection :accessor winners-window-selection)))

;;;_______________________________
;;; user-choose-a-winner

(defun user-choose-a-winner (winners)
  (cond ((= (winners-number winners) 1)
         (best-winner winners))
        (t
         (let ((x 350) (y 2) (button-size #@(55 15)) (button-font '("Chicago" 12 :plain))
               (window (make-instance 'winners-window
                         :winners winners
                         :current 0
                         :selection nil
                         :window-show t
                         :window-title "Choose Winner"
                         :view-font *default-font*
                         :view-size #@(540 300)
                         :view-position :centered
                         :close-box-p nil))
               winner)
           (clear-gain-window window)
           (add-subviews
            window
            (make-dialog-item
             'scroll-bar-dialog-item (make-point x y) nil ""
             #'(lambda (item) 
                 (let ((window (view-container item))
                       (number (scroll-bar-setting item)))
                   (unless (= (winners-window-current window) number)
                     (display-winner-gain window (nth number (winners-all-winners (winners-window-winners window))))
                     (setf (winners-window-current window) number))))
             :direction :horizontal :width 15 :length 100 :scroll-size 1 :page-size 1
             :min 0 :max (- (winners-number winners) 1) :view-nick-name :selector)
            (make-dialog-item
             'button-dialog-item (make-point (- 540 70) y) button-size " Select "
             #'(lambda (item)
                 (let ((window (view-container item)))
                   (setf (winners-window-selection window)
                         (nth (winners-window-current window) (winners-all-winners (winners-window-winners window))))))
             :view-font button-font :view-nick-name :select))
           (display-winner-gain window (first (winners-all-winners winners)))
           (setf winner (do () ((winners-window-selection window) (winners-window-selection window)) (event-dispatch)))
           (window-close window)
           winner))))