
;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; The code contained in this file was written by Cliff Brunk.

(in-package :user)

;;; To make the code deal with invalid user input
;;; strengthen the following two functions.

;;;_______________________________________
;;;  value-from-dialog-item-text

(defun value-from-dialog-item-text (dialog-item)
  (read-from-string (dialog-item-text dialog-item) nil nil))

;;;_________________________________________________________________
;;; number-from-dialog-item-text

(defun number-from-dialog-item-text (item)
  (let ((value (value-from-dialog-item-text item)))
    (if (numberp value) value 0)))

;;;_________________________________________________________________
;;; select-pop-up-menu-item

(defun select-pop-up-menu-item (pop-up-menu menu-item-title)
  (setf (ccl::pop-up-menu-default-item pop-up-menu)
        (+ (or (position menu-item-title (menu-items pop-up-menu) :test #'string-equal :key #'menu-item-title) 0) 1))
  (invalidate-view pop-up-menu))



;;;=================================================================
;;; Graph Window Setup Dialog
;;;=================================================================

;;;_________________________________________________________________
;;; graph-window-setup-dialog

(defun graph-window-setup-dialog (graph-title font orientation expand expand-depth)
  (let*
    ((styles '(:bold :italic :underline :outline :shadow :condense :extend))
     (font-face (find-if #'stringp font))
     (font-size (find-if #'numberp font))
     (not-plain (not (member :plain font)))
     (orientation-cluster 0)
     (expand-cluster 1)
     (dialog
      (make-instance
       'dialog
       :window-show nil
       :window-type :double-edge-box
       :view-position :centered
       :view-size #@(360 270)
       :view-font '("Chicago" 12 :srcor :plain)
       :view-subviews
       (list 
        (make-dialog-item 'static-text-dialog-item #@(15 7) nil
                          (format nil "~A Setup..." graph-title) nil)
        (make-dialog-item 'static-text-dialog-item #@(15 30) nil "Font:" nil)
        (make-dialog-item 'sequence-dialog-item  #@(15 50) #@(160 80) "Font Selector" nil
                          :cell-size #@(145 16)
                          :table-hscrollp NIL
                          :table-vscrollp T
                          :table-sequence *font-list*
                          :selection-type :single
                          :view-nick-name :font-selector)
        (make-dialog-item 'static-text-dialog-item #@(15 142) nil "Font Size:" nil)
        (make-dialog-item 'editable-text-dialog-item  #@(100 142) #@(32 16)
                          (format nil "~A" font-size) nil
                          :allow-returns nil :view-nick-name :font-sizer)
        (make-dialog-item 'static-text-dialog-item #@(15 172) nil "Font Sytle:" nil)
        (make-dialog-item 'check-box-dialog-item #@(17 192) nil "Plain"
                          #'(lambda (item)
                              (dolist (style styles)
                                (check-box-uncheck (find-named-sibling item style))))
                          :check-box-checked-p (not not-plain)
                          :view-font '("Geneva" 12 :srcor :plain) :view-nick-name :plain)
        (make-dialog-item 'check-box-dialog-item #@(17 208) nil "Bold"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :bold font))
                          :view-font '("Geneva" 12 :srcor :bold) :view-nick-name :bold)
        (make-dialog-item 'check-box-dialog-item #@(17 224) nil "Italic"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :italic font))
                          :view-font '("Geneva" 12 :srcor :italic) :view-nick-name :italic)
        (make-dialog-item 'check-box-dialog-item #@(17 240) nil "Underline"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :underline font))
                          :view-font '("Geneva" 12 :srcor :underline) :view-nick-name :underline)
        (make-dialog-item 'check-box-dialog-item #@(100 192) nil "Outline"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :outline font))
                          :view-font '("Geneva" 12 :srcor :outline) :view-nick-name :outline)
        (make-dialog-item 'check-box-dialog-item #@(100 208) nil "Shadow"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :shadow font))
                          :view-font '("Geneva" 12 :srcor :shadow) :view-nick-name :shadow)
        (make-dialog-item 'check-box-dialog-item #@(100 224) nil "Condense"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :condense font))
                          :view-font '("Geneva" 12 :srcor :condense) :view-nick-name :condense)
        (make-dialog-item 'check-box-dialog-item #@(100 240) nil "Extend"
                          #'(lambda (item) (check-box-uncheck (find-named-sibling item :plain)))
                          :check-box-checked-p (and not-plain (member :extend font))
                          :view-font '("Geneva" 12 :srcor :extend) :view-nick-name :extend)
        (make-dialog-item 'static-text-dialog-item #@(200 30) nil "Orientation:" nil)
        (make-dialog-item 'radio-button-dialog-item #@(210 50) nil "Horizontal" nil
                          :radio-button-cluster orientation-cluster
                          :radio-button-pushed-p (eq orientation :horizontal)
                          :view-nick-name :horizontal)
        (make-dialog-item 'radio-button-dialog-item #@(210 66) nil "Diagonal" nil
                          :radio-button-cluster orientation-cluster
                          :radio-button-pushed-p (eq orientation :diagonal)
                          :view-nick-name :diagonal)
        (make-dialog-item 'radio-button-dialog-item  #@(210 82) nil "Vertical" nil
                          :radio-button-cluster orientation-cluster
                          :radio-button-pushed-p (eq orientation :vertical)
                          :view-nick-name :vertical)
        (make-dialog-item 'static-text-dialog-item #@(200 107) nil "Expand:" nil)
        (make-dialog-item 'radio-button-dialog-item #@(210 127) nil "Completely" nil
                          :radio-button-cluster expand-cluster
                          :radio-button-pushed-p (or (eq expand :completely) (eq expand t))
                          :view-nick-name :completely)
        (make-dialog-item 'radio-button-dialog-item #@(210 143) nil "Every Use" nil
                          :radio-button-cluster expand-cluster
                          :radio-button-pushed-p (eq expand :every-use)
                          :view-nick-name :every-use)
        (make-dialog-item 'radio-button-dialog-item #@(210 159) nil "First Use" nil
                          :radio-button-cluster expand-cluster
                          :radio-button-pushed-p (eq expand :first-use)
                          :view-nick-name :first-use)
        (make-dialog-item 'radio-button-dialog-item #@(210 175) nil "Never" nil
                          :radio-button-cluster expand-cluster
                          :radio-button-pushed-p (or (eq expand :never) (eq expand nil))
                          :view-nick-name :Never)

        (make-dialog-item 'static-text-dialog-item #@(200 200) nil "Expand Depth:" nil)
        (make-dialog-item 'editable-text-dialog-item  #@(300 200) #@(32 16)
                          (format nil "~A" expand-depth) nil
                          :allow-returns nil :view-nick-name :expand-depth)

        (make-dialog-item 'button-dialog-item #@(210 240) #@(60 20) " OK "
                          #'(lambda (item)
                              (let* ((dialog (view-container item))
                                     (font-selector (find-named-sibling item :font-selector))
                                     (font-face (cell-contents font-selector (first (selected-cells font-selector))))
                                     (font-size (number-from-dialog-item-text (find-named-sibling item :font-sizer)))
                                     (orientation (view-nick-name (pushed-radio-button dialog orientation-cluster)))
                                     (expand (view-nick-name (pushed-radio-button dialog expand-cluster)))
                                     (expand-depth (number-from-dialog-item-text (find-named-sibling item :expand-depth)))
                                     (new-font nil))
                                (dolist (style styles)
                                  (when (check-box-checked-p (find-named-sibling item style))
                                    (setf new-font (push style new-font))))
                                (if (null new-font) (setf new-font (list :plain)))
                                (push :srcor new-font)
                                (push font-size new-font)
                                (push font-face new-font)
                                (return-from-modal-dialog 
                                 (list new-font orientation expand expand-depth))))
                          :default-button t)
        (make-dialog-item 'button-dialog-item #@(285 240) #@(60 20) " Cancel "
                          #'(lambda (item) item (return-from-modal-dialog :cancel))
                          :default-button nil))))
     (font-selector (view-named :font-selector dialog))
     (font-postion (position font-face *font-list* :test #'string-equal)))
    (scroll-to-cell font-selector 0 font-postion)
    (cell-select font-selector 0 font-postion)
    dialog))

;;;_________________________________________________________________
;;; get-window-setup

(defun get-window-setup (graph-title font orientation expand expand-depth)
  (modal-dialog (graph-window-setup-dialog graph-title font orientation expand expand-depth) t))

;;;_________________________________________________________________
;;; default-setup

(defun default-setup ()
  (catch-cancel
   (multiple-value-setq (*default-font* *default-orientation* *default-expand* *default-expand-depth*)
     (values-list (modal-dialog
                   (graph-window-setup-dialog
                    "Default" *default-font* *default-orientation* *default-expand* *default-expand-depth*)
                   t)))
   (when (user-monitor-p *user-monitor*)
     (incf (user-monitor-default-setup *user-monitor*))))
  (values))


;;;=================================================================
;;; Learning Settings Dialog
;;;=================================================================

(defparameter *trace-level-list* (list (cons "Information Gain" :i)
                                       (cons "New Clauses" :c)
                                       (cons "Clause Tuples" :ct)
                                       (cons "New literals" :l)
                                       (cons "Literal Tuples" :lt)
                                       (cons "Bits"  :b)
                                       (cons "Operationalization" :o)
                                       (cons "Work" :w)))
(defparameter *display-level-list* (list (cons "Current Gain" :current-gain)
                                         (cons "Best Gain" :best-gain)
                                         (cons "EBL" :ebl)
                                         (cons "Learned Definition" :learned-description)
                                         (cons "Work"  :work)))
(defparameter *pause-level-list* (list (cons "Pause before Learning" :pause-before-learning)
                                       (cons "Pause after Clause" :pause-after-clause)
                                       (cons "Pause after Literal" :pause-after-literal)))
(defvar *focl-trace-level-item* nil)
(defvar *focl-display-level-item* nil)


;;;_________________________________________________________________
;;; create-pred-pop-up-menu

(defun create-pred-pop-up-menu (x y)
  (let ((pred-menu
         (make-dialog-item 
          'pop-up-menu (make-point x y) #@(255 22) "Learning:" nil :view-nick-name :pred-name
          :menu-items
          (mapcar #'(lambda (name) (make-instance 'menu-item :menu-item-title (format nil "~(~A~)" name)))
                  (mapcan #'(lambda (r) (if (pred-p r) (list (r-name r)))) *r-structs*)))))
    (dolist (item (menu-items pred-menu))
      (set-menu-item-action-function
       item #'(lambda () (update-settings-from-pred-choice (view-container pred-menu)))))
    pred-menu))

;;;_________________________________________________________________
;;; create-rule-pop-up-menu

(defun create-rule-pop-up-menu (x y)
  (let ((rule-menu
         (make-dialog-item 
          'pop-up-menu (make-point x y) #@(290 22) "Goal Concept:" nil :view-nick-name :rule-name
          :menu-items
          (append (list (make-instance 'menu-item :menu-item-title "none")
                        (make-instance 'menu-item :menu-item-title "-"))
                  (mapcar #'(lambda (name) (make-instance 'menu-item :menu-item-title (format nil "~(~A~)" name)))
                          (mapcan #'(lambda (r) (if (rule-p r) (list (r-name r)))) *r-structs*))))))
    (dolist (item (menu-items rule-menu))
      (set-menu-item-action-function
       item #'(lambda () (update-settings-from-rule-choice (view-container rule-menu)))))
    rule-menu))

;;;_________________________________________________________________
;;;  update-settings-from-pred-choice

(defun update-settings-from-pred-choice (dialog)
  (disable-invalid-goal-concepts dialog)
  (multiple-value-bind (pred-name rule-name) (pred-name-and-rule-name-from-dialog dialog)
    (setf *predicate-being-learned* pred-name
          *goal-concept* (if rule-name
                           (if (eql rule-name (first *goal-concept*))
                             *goal-concept*
                             (cons rule-name (make-old-vars (r-arity (get-r-struct rule-name)))))))
    (hilite-items-with-predefined-settings pred-name rule-name dialog)
    (setting-from-predefined-focl-problem pred-name rule-name dialog)))

;;;_________________________________________________________________
;;;  update-settings-from-rule-choice

(defun update-settings-from-rule-choice (dialog)
  (multiple-value-bind (pred-name rule-name) (pred-name-and-rule-name-from-dialog dialog)
    (setf *goal-concept* (if rule-name
                           (if (eql rule-name (first *goal-concept*))
                             *goal-concept*
                             (cons rule-name (make-old-vars (r-arity (get-r-struct rule-name)))))))
    (hilite-items-with-predefined-settings pred-name rule-name dialog)
    (setting-from-predefined-focl-problem pred-name rule-name dialog)))


;;;_________________________________________________________________
;;;  pred-name-and-rule-name-from-dialog

(defun pred-name-and-rule-name-from-dialog (dialog)
  (let* ((pred-menu (view-named :pred-name dialog))
         (pred-name-item (selected-item pred-menu))
         (pred-name (relation-name-from-menu-item pred-name-item))
         (rule-menu (view-named :rule-name dialog))
         (rule-name-item (selected-item rule-menu))
         (rule-name (relation-name-from-menu-item rule-name-item)))
    (if (equalp rule-name 'none) (setf rule-name nil))
    (values pred-name rule-name)))

;;;_________________________________________________________________
;;;  relation-name-from-menu-item

(defun relation-name-from-menu-item (menu-item)
  (when menu-item
    (let ((relation-name (read-from-string (menu-item-title menu-item))))
      (if (equalp relation-name 'none) (setf relation-name nil))
      relation-name)))
    
;;;_________________________________________________________________
;;;  hilite-items-with-predefined-settings

(defun hilite-items-with-predefined-settings (pred-name rule-name dialog)
  (dolist (menu-item (menu-items (view-named :rule-name dialog)))
    (if (find-predefined-focl-problem pred-name (relation-name-from-menu-item menu-item))
      (set-menu-item-check-mark menu-item t)
      (set-menu-item-check-mark menu-item nil)))
  (dolist (menu-item (menu-items (view-named :pred-name dialog)))
    (if (find-predefined-focl-problem (relation-name-from-menu-item menu-item) rule-name )
      (set-menu-item-check-mark menu-item t)
      (set-menu-item-check-mark menu-item nil))))

;;;_________________________________________________________________
;;;  disable-invalid-goal-concepts

(defun disable-invalid-goal-concepts (dialog)
  (multiple-value-bind (pred-name rule-name) (pred-name-and-rule-name-from-dialog dialog)
    (declare (ignore rule-name))
    (let* ((pred (get-pred pred-name))
           (pred-type (r-type pred))
           (pred-arity (r-arity pred))
           (rule-menu (view-named :rule-name dialog))
           (some-rule-enabled? nil)
           r r-name)
      (dolist (menu-item (menu-items rule-menu))
        (setf r-name (relation-name-from-menu-item menu-item)
              r (get-r-struct r-name))
        (cond ((null r) nil)
              ((and (= pred-arity (r-arity r)) (type-equal pred-type (r-type r)))
               (setf some-rule-enabled? t)
               (menu-item-enable menu-item))
              (t (menu-item-disable menu-item))))
      (unless (menu-item-enabled-p (selected-item rule-menu))
        (select-pop-up-menu-item rule-menu "none"))
      (if some-rule-enabled?
        (dialog-item-enable rule-menu)
        (dialog-item-disable rule-menu)))))

;;;_________________________________________________________________
;;;  find-predefined-focl-problem

(defun find-predefined-focl-problem (pred-name rule-name)
  (find-if  #'(lambda (settings)
                (and (eql (first settings) pred-name)
                     (eql (getf (rest settings) :goal-concept-name) rule-name)))
            *focl-problems*))

;;;_________________________________________________________________
;;;  setting-from-predefined-focl-problem

(defun setting-from-predefined-focl-problem (pred-name rule-name dialog)
  (let ((settings (find-predefined-focl-problem pred-name rule-name)))
    (if settings
      (apply #'set-learning-parameters (first settings) (rest settings))
      (if rule-name
        (setf *goal-concept* (cons rule-name (make-old-vars (r-arity (get-rule rule-name))))
              *use-goal-concept* t)
        (setf *goal-concept* nil
              *use-goal-concept* nil
              *theory-mode* nil)))
    (update-dialog-from-globals dialog)))

;;;_________________________________________________________________
;;;  initialize-dialog-settings

(defun initialize-dialog-settings (dialog)
  (let ((pred-name (predicate-being-learned))
        (rule-name (or (goal-concept-name) 'none)))
    (select-pop-up-menu-item (view-named :pred-name dialog) pred-name)
    (select-pop-up-menu-item (view-named :rule-name dialog) rule-name)
    (update-settings-from-pred-choice dialog)))

;;;_________________________________________________________________
;;; setup-focl-trace-level-pop-up-menu

(defun setup-focl-trace-level-pop-up-menu (x y)
  (unless *focl-trace-level-item*
    (setf *focl-trace-level-item*
          (make-instance
           'pop-up-menu :view-size #@(120 22) :view-position (make-point x y)
           :auto-update-default nil :item-display "Trace Level"
           :menu-items
           (append
            (mapcar
             #'(lambda (t-i)
                 (let ((menu-item (make-instance 'menu-item :menu-item-title (first t-i))))
                   (set-menu-item-action-function
                    menu-item
                    #'(lambda ()
                        (if (menu-item-check-mark menu-item)
                          (set-menu-item-check-mark menu-item nil)
                          (set-menu-item-check-mark menu-item t))))
                   menu-item))
             *trace-level-list*)
              (list
               (make-instance 'menu-item :menu-item-title "-" :disabled t)
               (let ((menu-item (make-instance 'menu-item :menu-item-title "Full Trace")))
                 (set-menu-item-action-function
                  menu-item
                  #'(lambda ()
                      (let ((menu (menu-item-owner menu-item)))
                        (dolist (t-i *trace-level-list*)
                          (set-menu-item-check-mark (find-menu-item menu (first t-i)) t)))))
                 menu-item)
               (let ((menu-item (make-instance 'menu-item :menu-item-title "No Trace")))
                 (set-menu-item-action-function
                  menu-item
                  #'(lambda ()
                      (let ((menu (menu-item-owner menu-item)))
                        (dolist (t-i *trace-level-list*)
                          (set-menu-item-check-mark (find-menu-item menu (first t-i)) nil))
                        (check-box-uncheck (find-named-sibling menu :trace-learning?))
                        (dialog-item-disable menu))))
                 menu-item)))
             :view-nick-name :focl-trace-level)))
  (dolist (t-i *trace-level-list*)
    (if (member (rest t-i) *focl-trace-level*)
      (set-menu-item-check-mark (find-menu-item *focl-trace-level-item* (first t-i)) t)
      (set-menu-item-check-mark (find-menu-item *focl-trace-level-item* (first t-i)) nil)))
  (if *trace-learning?*
    (dialog-item-enable *focl-trace-level-item*)
    (dialog-item-disable *focl-trace-level-item*))
  *focl-trace-level-item*)


;;;_________________________________________________________________
;;; setup-focl-display-level-pop-up-menu

(defun setup-focl-display-level-pop-up-menu (x y)
  (unless *focl-display-level-item*
    (setf *focl-display-level-item*
          (make-instance
           'pop-up-menu :view-size #@(120 22) :view-position (make-point x y)
           :auto-update-default nil :item-display "Display Level"
           :menu-items
             (append 
              (mapcar
               #'(lambda (t-i)
                   (let ((menu-item (make-instance 'menu-item :menu-item-title (first t-i))))
                     (set-menu-item-action-function
                      menu-item #'(lambda ()
                                    (if (menu-item-check-mark menu-item)
                                      (set-menu-item-check-mark menu-item nil)
                                      (set-menu-item-check-mark menu-item t))))
                     menu-item))
               *display-level-list*)
              (list (make-instance 'menu-item :menu-item-title "-" :disabled t)
                    (let ((menu-item (make-instance 'menu-item :menu-item-title "Full Display")))
                      (set-menu-item-action-function
                       menu-item
                       #'(lambda ()
                           (let ((menu (menu-item-owner menu-item)))
                             (dolist (t-i *display-level-list*)
                               (set-menu-item-check-mark (find-menu-item menu (first t-i)) t))
                             (check-box-uncheck (find-named-sibling menu :trace-learning?))
                             (dialog-item-disable (find-named-sibling menu :focl-trace-level)))))
                       menu-item)
                    (let ((menu-item (make-instance 'menu-item :menu-item-title "No Display")))
                      (set-menu-item-action-function
                       menu-item
                       #'(lambda ()
                           (let ((menu (menu-item-owner menu-item)))
                             (dolist (t-i *display-level-list*)
                               (set-menu-item-check-mark (find-menu-item menu (first t-i)) nil))
                             (dolist (t-i *pause-level-list*)
                               (set-menu-item-check-mark (find-menu-item menu (first t-i)) nil))
                             (check-box-uncheck (find-named-sibling menu :display-learning?))
                             (dialog-item-disable menu))))
                      menu-item)
                    )
              (list (make-instance 'menu-item :menu-item-title "-" :disabled t))
              (mapcar
               #'(lambda (t-i)
                   (let ((menu-item (make-instance 'menu-item :menu-item-title (first t-i))))
                     (set-menu-item-action-function
                      menu-item #'(lambda ()
                                    (if (menu-item-check-mark menu-item)
                                      (set-menu-item-check-mark menu-item nil)
                                      (set-menu-item-check-mark menu-item t))))
                     menu-item))
               *pause-level-list*))
             :view-nick-name :focl-display-level)))
  (dolist (t-i *display-level-list*)
    (if (member (rest t-i) *focl-display-level*)
      (set-menu-item-check-mark (find-menu-item *focl-display-level-item* (first t-i)) t)
      (set-menu-item-check-mark (find-menu-item *focl-display-level-item* (first t-i)) nil)))
  (dolist (t-i *pause-level-list*)
    (if (member (rest t-i) *focl-display-level*)
      (set-menu-item-check-mark (find-menu-item *focl-display-level-item* (first t-i)) t)
      (set-menu-item-check-mark (find-menu-item *focl-display-level-item* (first t-i)) nil)))
  (if *display-learning?*
    (dialog-item-enable *focl-display-level-item*)
    (dialog-item-disable *focl-display-level-item*))
  *focl-display-level-item*)


;;;_________________________________________________________________
;;;  learning-settings-dialog

(defun learning-settings-dialog (&optional (modify-button-name " Set "))
  (let*
    ((window-h 590) (window-v 290) (y 5) (y2 5) (y3 5)
     (x 15) (x-o 25) (x-o2 50) (x2 210) (x2-o 220) (x2-o1 238) (x3 380) (x3-o 385) (x3-o2 400)
     (gain-function-cluster 0) (selection-function-cluster 1) (refinement-cluster 2)
     (edit-box-size #@(18 16))
     (dialog
      (make-instance
       'dialog
       :window-show nil
       :window-type :double-edge-box
       :view-position :centered
       :view-size (make-point window-h window-v)
       :close-box-p nil
       :view-font '("Chicago" 12 :srcor :plain)
       :view-subviews
       (list 
        (make-dialog-item
         'static-text-dialog-item (make-point 15 y) #@(250 16) "FOCL Parameter Settings..." nil)
        (create-pred-pop-up-menu 15 (incf y 25))
        (create-rule-pop-up-menu 280 y)

        ;;; Empirical Learning ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x (setf y3 (setf y2 (incf y 30)))) nil "Empirical Learning" nil)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 16)) nil "extensional induction"
         #'(lambda (item)
             (setf *extensional-induction* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :extensional-induction)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 15)) nil "builtin induction"
         #'(lambda (item)
             (setf *builtin-induction* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :builtin-induction)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 15)) nil "intensional induction"
         #'(lambda (item)
             (setf *intensional-induction* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :intensional-induction)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 15)) nil "clich instantiation"
         #'(lambda (item)
             (setf *use-relational-cliches* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :use-relational-cliches)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 15)) nil "determinate addition"
         #'(lambda (item)
             (setf *enable-determinate-literals* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :enable-determinate-literals)

        (make-dialog-item
         'editable-text-dialog-item (make-point x-o (incf y 24)) edit-box-size ""
         #'(lambda (item) (setf *max-winners* (number-from-dialog-item-text item)))
         :allow-returns nil :view-nick-name :max-winners)
        (make-dialog-item
         'static-text-dialog-item (make-point x-o2 y) nil "max-winners" nil
         :view-nick-name :max-winners-title)
        (make-dialog-item
         'editable-text-dialog-item (make-point x-o (incf y 25)) edit-box-size ""
         #'(lambda (item) (setf *max-new-variables* (number-from-dialog-item-text item)))
         :allow-returns nil :view-nick-name :max-new-variables)
        (make-dialog-item
         'static-text-dialog-item (make-point x-o2 y) nil "max-new-variables" nil
         :view-nick-name :max-new-variables-title)
        (make-dialog-item
         'editable-text-dialog-item (make-point x-o (incf y 25)) edit-box-size ""
         #'(lambda (item) (setf *max-new-cliche-vars* (number-from-dialog-item-text item)))
         :allow-returns nil :view-nick-name :max-new-cliche-vars)
        (make-dialog-item
         'static-text-dialog-item (make-point x-o2 y) nil "max-new-cliche-vars" nil
         :view-nick-name :max-new-cliche-vars-title)
        (make-dialog-item
         'editable-text-dialog-item (make-point x-o (incf y 25)) edit-box-size ""
         #'(lambda (item) (setf *max-determinacy* (number-from-dialog-item-text item)))
         :allow-returns nil :view-nick-name :max-determinacy)
        (make-dialog-item
         'static-text-dialog-item (make-point x-o2 y) nil "max-determinacy" nil
         :view-nick-name :max-determinacy-title)

#|
        ;;; Noise Tolerance ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x (incf y 27)) nil "Noise Tolerance" nil)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 16)) nil "encoding length"
         #'(lambda (item)
             (check-box-uncheck (find-named-sibling item :REP))
             (setf *stopping-criteria-enabled* (if (check-box-checked-p item) :FOIL nil)))
         :view-nick-name :FOIL)
        (make-dialog-item
         'check-box-dialog-item (make-point x-o (incf y 15)) nil "error pruning"
         #'(lambda (item)
             (check-box-uncheck (find-named-sibling item :FOIL))
             (setf *stopping-criteria-enabled* (if (check-box-checked-p item) :REP nil)))
         :view-nick-name :REP)
|#

        ;;; Gain-Function ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x2 y2) nil "Gain Function" nil)
        (make-dialog-item
         'radio-button-dialog-item (make-point x2-o (incf y2 16)) nil "information"
          #'(lambda (item) item (setf *gain-function* :information))
         :radio-button-cluster gain-function-cluster
         :view-nick-name :information)
        (make-dialog-item
         'radio-button-dialog-item (make-point x2-o (incf y2 15)) nil "ratio"
         #'(lambda (item) item (setf *gain-function* :ratio))
         :radio-button-cluster gain-function-cluster
         :view-nick-name :ratio)

        ;;; Selection-Function ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x2 (incf y2 23)) nil "Selection Function" nil)
        (make-dialog-item
         'radio-button-dialog-item (make-point x2-o (incf y2 16)) nil "maximum"
          #'(lambda (item) item (setf *selection-function* :maximum))
         :radio-button-cluster selection-function-cluster
         :view-nick-name :maximum)
        (make-dialog-item
         'radio-button-dialog-item (make-point x2-o (incf y2 15)) nil "probabilistic"
         #'(lambda (item) item (setf *selection-function* :probabilistic))
         :radio-button-cluster selection-function-cluster
         :view-nick-name :probabilistic)
        (make-dialog-item
         'radio-button-dialog-item (make-point x2-o (incf y2 15)) nil "manual"
         #'(lambda (item) item (setf *selection-function* :manual))
         :radio-button-cluster selection-function-cluster
         :view-nick-name :manual)

        ;;; Miscellaneous ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x2 (incf y2 23)) nil "Miscellaneous" nil)
        (make-dialog-item
         'check-box-dialog-item (make-point x2-o (incf y2 16)) nil "save learned def"
         #'(lambda (item) (setf *define-rule-for-learned-description* (check-box-checked-p item)))
         :view-nick-name :save-learned-definition)

#|
        (make-dialog-item
         'check-box-dialog-item (make-point x2-o (incf y2 16)) nil "stop at first cover"
         #'(lambda (item) (setf *stop-when-all-pos-covered* (check-box-checked-p item)))
         :view-nick-name :stop-when-all-pos-covered)
|#
        (make-dialog-item
         'check-box-dialog-item (make-point x2-o (incf y2 16)) nil "save examples"
         #'(lambda (item) (setf *save-examples* (check-box-checked-p item)))
         :view-nick-name :save-examples)

        ;;; Trace & Display ___________________
        (make-dialog-item
         'check-box-dialog-item (make-point x2-o (incf y2 23)) #@(16 16) ""
         #'(lambda (item)
             (if (setf *trace-learning?* (check-box-checked-p item))
               (dialog-item-enable (find-named-sibling item :focl-trace-level))
               (dialog-item-disable (find-named-sibling item :focl-trace-level))))
         :view-nick-name :trace-learning?)
        (setup-focl-trace-level-pop-up-menu x2-o1 (decf y2 2))
        (make-dialog-item
         'check-box-dialog-item (make-point x2-o (incf y2 28)) #@(16 16) ""
         #'(lambda (item)
             (if (setf *display-learning?* (check-box-checked-p item))
               (dialog-item-enable (find-named-sibling item :focl-display-level))
               (dialog-item-disable (find-named-sibling item :focl-display-level))))
         :view-nick-name :display-learning?)
        (setup-focl-display-level-pop-up-menu x2-o1 (decf y2 2))


        ;;; Analytical Learning ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x3 y3) nil "Analytical Learning" nil)
        (make-dialog-item
         'check-box-dialog-item (make-point x3-o (incf y3 16)) nil "use goal concept"
         #'(lambda (item)
             (setf *use-goal-concept* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :use-goal-concept)
        (make-dialog-item
         'check-box-dialog-item (make-point x3-o2 (incf y3 15)) nil "prefer theory"
         #'(lambda (item)
             (setf *theory-mode* (check-box-checked-p item))
             (update-dialog-from-globals (view-container item)))
         :view-nick-name :theory-mode)
        (make-dialog-item
         'radio-button-dialog-item (make-point x3-o (incf y3 18)) nil "leaf refinement"
         #'(lambda (item)
             (setf *refinement* :leaves)
             (update-dialog-from-globals (view-container item)))
         :radio-button-cluster refinement-cluster
         :view-nick-name :leaves)
        (make-dialog-item
         'radio-button-dialog-item (make-point x3-o (incf y3 15)) nil "frontier refinement"
         #'(lambda (item)
             (setf *refinement* :frontier)
             (update-dialog-from-globals (view-container item)))
         :radio-button-cluster refinement-cluster
         :view-nick-name :frontier)
        (make-dialog-item
         'check-box-dialog-item(make-point x3-o2 (incf y3 16)) nil "prefer children"
         #'(lambda (item) (setf *prefer-children* (check-box-checked-p item)))
         :view-nick-name :prefer-children)
        (make-dialog-item
         'check-box-dialog-item (make-point x3-o2 (incf y3 15)) nil "prefer deletions"
         #'(lambda (item) (setf *prefer-deletions* (check-box-checked-p item)))
         :view-nick-name :prefer-deletions)
        (make-dialog-item
         'check-box-dialog-item (make-point x3-o (incf y3 18)) nil "operationalize intensional"
         #'(lambda (item) (setf *operationalize-intensional* (check-box-checked-p item)))
         :view-nick-name :operationalize-intensional)

        ;;; Simplify ___________________
        (make-dialog-item
         'static-text-dialog-item (make-point x3 (incf y3 25)) nil "Simplify" nil)
        (make-dialog-item
         'check-box-dialog-item (make-point x3-o (incf y3 16)) nil "clauses"
         #'(lambda (item) (setf *simplify-clauses* (check-box-checked-p item)))
         :view-nick-name :simplify-clauses)
        (make-dialog-item
         'check-box-dialog-item (make-point x3-o (incf y3 15)) nil "operationalizations"
         #'(lambda (item) (setf *simplify-operationalizations* (check-box-checked-p item)))
         :view-nick-name :simplify-operationalizations)

        ;;; Exit Buttons ___________________
        (make-dialog-item
         'button-dialog-item
         (make-point (- window-h 145) (- window-v 27)) #@(60 20) modify-button-name
         #'(lambda (item)
             (setf *focl-trace-level* nil
                   *focl-display-level* nil)
             (let ((trace-menu (view-named :focl-trace-level (view-container item))))
               (dolist (t-i *trace-level-list*)
                 (if (menu-item-check-mark (find-menu-item trace-menu (first t-i)))
                   (setf *focl-trace-level* (push (rest t-i) *focl-trace-level*)))))
             (let ((display-menu (view-named :focl-display-level (view-container item))))
               (dolist (t-i *display-level-list*)
                 (if (menu-item-check-mark (find-menu-item display-menu (first t-i)))
                   (setf *focl-display-level* (push (rest t-i) *focl-display-level*))))
               (dolist (t-i *pause-level-list*)
                 (if (menu-item-check-mark (find-menu-item display-menu (first t-i)))
                   (setf *focl-display-level* (push (rest t-i) *focl-display-level*)))))
             (multiple-value-bind (pred parameters) (collect-learning-parameters)
               (return-from-modal-dialog (apply #'set-focl-problem pred parameters))))
         :default-button t :view-nick-name :modify-button)
        (make-dialog-item
         'button-dialog-item
         (make-point (- window-h 70) (- window-v 27)) #@(60 20) " Cancel "
         #'(lambda (item) item (return-from-modal-dialog :cancel))
         :default-button nil)))))
    (initialize-dialog-settings dialog)
    dialog))

;;;_________________________________________________________________
;;;  update-dialog-from-globals

(defun update-dialog-from-globals (dialog)
  (let* ((extensional-induction-possible? (some #'(lambda (r) (and (r-induction r) (pred-p r))) *r-structs*))
         (builtin-induction-possible? (some #'(lambda (r) (and (r-induction r) (or (builtin-p r) (builtin-fn-p r)))) *r-structs*))
         (intensional-induction-possible? (some #'(lambda (r) (and (r-induction r) (rule-p r))) *r-structs*))
         (extensional-induction? (and extensional-induction-possible? *extensional-induction*))
         (builtin-induction? (and builtin-induction-possible? *builtin-induction*))
         (intensional-induction? (and intensional-induction-possible? *intensional-induction*))
         (induction? (or extensional-induction? builtin-induction? intensional-induction?))
         (cliche-instantiation? (and *available-relational-cliches* *use-relational-cliches*))
         (determinate-addition? (and extensional-induction-possible? *enable-determinate-literals*))
         
         (goal-concept-name (first *goal-concept*))
         (theory-based-learning-possible? (and *goal-concept* (get-rule goal-concept-name)))
         (use-goal-concept? (and *use-goal-concept* theory-based-learning-possible?))
         (theory-mode? (and use-goal-concept? *theory-mode*))
         (analytical-learning-possible? (or use-goal-concept? cliche-instantiation? intensional-induction?))
         )      

  ;;; Empirical Learning ___________________

    (let ((item (view-named :extensional-induction dialog)))
      (if extensional-induction? (check-box-check item) (check-box-uncheck item))
      (if extensional-induction-possible? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :builtin-induction dialog)))
      (if builtin-induction? (check-box-check item) (check-box-uncheck item))
      (if builtin-induction-possible? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :intensional-induction dialog)))
      (if intensional-induction? (check-box-check item) (check-box-uncheck item))
      (if intensional-induction-possible? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :use-relational-cliches dialog)))
      (if cliche-instantiation? (check-box-check item) (check-box-uncheck item))
      (if *available-relational-cliches* (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :enable-determinate-literals dialog)))
      (if determinate-addition? (check-box-check item) (check-box-uncheck item))
      (if extensional-induction-possible? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-new-variables dialog)))
      (set-dialog-item-text item  (if induction? (format nil "~a" *max-new-variables*) ""))
      (if induction? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-new-variables dialog)))
      (set-dialog-item-text item (if induction? (format nil "~a" *max-new-variables*) ""))
      (if induction? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-new-variables-title dialog)))
      (if induction? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-new-cliche-vars dialog)))
      (set-dialog-item-text item (if cliche-instantiation? (format nil "~a" *max-new-cliche-vars*) ""))
      (if cliche-instantiation? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-new-cliche-vars-title dialog)))
      (if cliche-instantiation? (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-determinacy dialog)))
      (set-dialog-item-text item (if determinate-addition? (format nil "~a" *max-determinacy*) ""))
      (if determinate-addition?  (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :max-determinacy-title dialog)))
      (if determinate-addition? (dialog-item-enable item) (dialog-item-disable item)))

  ;;; Trace & Display ___________________
    (let ((item (view-named :trace-learning? dialog)))
      (if *trace-learning?* (check-box-check item) (check-box-uncheck item)))
    (let ((item (view-named :focl-trace-level dialog)))
      (if *trace-learning?* (dialog-item-enable item) (dialog-item-disable item)))
    (let ((item (view-named :display-learning? dialog)))
      (if *display-learning?* (check-box-check item) (check-box-uncheck item)))
    (let ((item (view-named :focl-display-level dialog)))
      (if *display-learning?* (dialog-item-enable item) (dialog-item-disable item)))

#|
  ;;; Noise Tolerance ___________________
  (let ((foil (view-named :foil dialog))
        (rep (view-named :rep dialog)))
    (case *stopping-criteria-enabled*
      (:FOIL (check-box-check foil)
             (check-box-uncheck rep))
      (:REP (check-box-check rep)
            (check-box-uncheck foil))
      (otherwise (check-box-uncheck foil)
                 (check-box-uncheck rep))))
|#

  ;;; Gain-Function ___________________
  (case *gain-function*
    (:information (radio-button-push (view-named :information dialog)))
    (:ratio (radio-button-push (view-named :ratio dialog))))

  ;;; Selection-Function ___________________
  (case *selection-function*
    (:maximum (radio-button-push (view-named :maximum dialog)))
    (:probabilistic (radio-button-push (view-named :probabilistic dialog)))
    (:manual (radio-button-push (view-named :manual dialog))))

  ;;; Miscellaneous ___________________
  (let ((item (view-named :save-learned-definition dialog)))
    (if *define-rule-for-learned-description* (check-box-check item) (check-box-uncheck item)))
  (let ((item (view-named :save-examples dialog)))
    (if *save-examples* (check-box-check item) (check-box-uncheck item)))
  (let ((item (view-named :max-winners dialog)))
    (set-dialog-item-text item (format nil "~a" *max-winners*)))
  
  ;;; Analytical Learning ___________________
  (let ((item (view-named :use-goal-concept dialog)))
    (if use-goal-concept? (check-box-check item) (check-box-uncheck item))
    (if theory-based-learning-possible? (dialog-item-enable item) (dialog-item-disable item)))

  (let ((item (view-named :theory-mode dialog)))
    (if theory-mode? (check-box-check item) (check-box-uncheck item))
    (if use-goal-concept? (dialog-item-enable item) (dialog-item-disable item)))
  
  (let ((leaves (view-named :leaves dialog))
        (frontier (view-named :frontier dialog))
        (prefer-children (view-named :prefer-children dialog))
        (prefer-deletions (view-named :prefer-deletions dialog)))
    (cond (analytical-learning-possible?
           (dialog-item-enable leaves)
           (dialog-item-enable frontier)
           (case *refinement*
             (:leaves (radio-button-push leaves)
                      (check-box-uncheck prefer-children)
                      (check-box-uncheck prefer-deletions)
                      (dialog-item-disable prefer-children)
                      (dialog-item-disable prefer-deletions))
             (:frontier (radio-button-push frontier)
                        (dialog-item-enable prefer-children)
                        (dialog-item-enable prefer-deletions)
                        (if *prefer-children* (check-box-check prefer-children) (check-box-uncheck prefer-children))
                        (if *prefer-deletions* (check-box-check prefer-deletions) (check-box-uncheck prefer-deletions)))))
          (t (radio-button-unpush leaves)
             (radio-button-unpush frontier)
             (check-box-uncheck prefer-children)  
             (check-box-uncheck prefer-deletions)
             (dialog-item-disable leaves)
             (dialog-item-disable frontier)
             (dialog-item-disable prefer-children)
             (dialog-item-disable prefer-deletions))))
  
  (let ((item (view-named :operationalize-intensional dialog)))
    (if (and *operationalize-intensional*
             (or cliche-instantiation? intensional-induction?)) (check-box-check item) (check-box-uncheck item))
    (if (or cliche-instantiation? intensional-induction?) (dialog-item-enable item) (dialog-item-disable item)))

  ;;; Simplify ___________________
  (let ((item (view-named :simplify-clauses dialog)))
    (if *simplify-clauses* (check-box-check item) (check-box-uncheck item)))
  (let ((item (view-named :simplify-operationalizations dialog)))
    (if (and analytical-learning-possible? *simplify-operationalizations*) (check-box-check item) (check-box-uncheck item))
    (if analytical-learning-possible? (dialog-item-enable item) (dialog-item-disable item)))
  ))

;;;_________________________________________________________________
;;;  collect-learning-parameters

(defun collect-learning-parameters ()
  (values *predicate-being-learned*
          (list :goal-concept-name (first *goal-concept*)
                :use-goal-concept *use-goal-concept*

                :prefer-theory *theory-mode*
                :refinement *refinement*
                :active-frontier-operators *active-frontier-operators*
                :prefer-deletions *prefer-deletions*
                :prefer-children *prefer-children*

                :gain-function *gain-function*
                :max-new-variables *max-new-variables*

                :extensional-induction *extensional-induction*
                :builtin-induction *builtin-induction*
                :intensional-induction *intensional-induction* 
                :operationalize-intensional *operationalize-intensional*

                :trace *focl-trace-level*
                :display *focl-display-level*
                :simplify-clauses *simplify-clauses*
                :define-rule-for-learned-description *define-rule-for-learned-description*
                :save-examples *save-examples*
                :noise-tolerance *stopping-criteria-enabled*
                :simplify-operationalizations *simplify-operationalizations*
                :stop-when-all-pos-covered *stop-when-all-pos-covered*

                :use-cliches *use-relational-cliches*
                :always-try-cliches *always-try-cliches*
                :available-cliches *available-relational-cliches*
                :cliches-can-have-negated-components? *cliches-can-have-negated-components?*
                :create-preds-from-cliches *create-preds-from-cliches*
                :max-new-cliche-vars *max-new-cliche-vars*
                :try-all-conjunctions *try-all-conjunctions*

                :enable-determinate-literals *enable-determinate-literals*
                :max-determinacy *max-determinacy*
                :max-determinate-depth *max-determinate-depth*
                :max-winners *max-winners*
                :selection-function *selection-function*
                :define-rule-for-learned-description *define-rule-for-learned-description*
                :revise-theory *revise-theory*


                :reset-hash-tables *reset-hash-tables*
                :reset-statistics *reset-statistics*
                :overfit-ratio *overfit-ratio*
                :proof-vars-available *proof-vars-available*
                :partial-dt-0-gain *partial-dt-0-gain*
                :batch-mode *focl-batch-mode*
                :builtin-threshold-only *builtin-threshold-only*
                
                ;;; Non-User Modifiable Function Parameters
                ; :literal-better-function *literal-better-function*
                ; :ebl-better-function *ebl-better-function*
                ; :delete-better-function *delete-better-function*
                )))



;;;=================================================================
;;; Set Preferences Dialog
;;;=================================================================

(defun set-preferences ()
  (let* ((h 400) (v 370) (x 10) (x2 30) (y 2)
         (expert-mode *expert-mode*)
         (color-level *color-level*)
         (dialog
          (make-instance
            'dialog
            :window-show nil
            :window-type :double-edge-box
            :view-position :centered
            :view-size (make-point h v)
            :view-font '("Chicago" 12 :srcor :plain)
            :view-subviews
            (list
             (make-dialog-item 'static-text-dialog-item (make-point x y) nil "Learning Preferences")
             
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "compile with rule trace"
              #'(lambda (item) (setf *compile-with-rule-trace* (check-box-checked-p item)))
              :check-box-checked-p *compile-with-rule-trace*)

            #|
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "fast operationalization"
              #'(lambda (item) (setf *fast-operationalization* (check-box-checked-p item)))
              :check-box-checked-p *fast-operationalization*)
            |#

             (make-dialog-item 'static-text-dialog-item (make-point x (incf y 18)) nil "Interface Preferences")
             
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "expert menus"
              #'(lambda (item) (setf *expert-mode* (check-box-checked-p item)))
              :check-box-checked-p *expert-mode*)

             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "suppress compiler warnings"
              #'(lambda (item) (setf ccl::*suppress-compiler-warnings* (check-box-checked-p item)
                                     *es-suppress-compiler-warnings* (check-box-checked-p item)))
              :check-box-checked-p ccl::*suppress-compiler-warnings*)


             (make-dialog-item 'static-text-dialog-item (make-point x (incf y 18)) nil "Editing Options")

             #|
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "use menus when entering new clauses"
              #'(lambda (item) (setf *use-menu* (check-box-checked-p item)))
              :check-box-checked-p *use-menu*)
             |#
             
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "enable graphic editor"
              #'(lambda (item) (setf *edit-graphically* (check-box-checked-p item)))
              :check-box-checked-p *edit-graphically*)

             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "maintain examples when editing"
              #'(lambda (item) (setf *maintain-examples-when-editing* (check-box-checked-p item)))
              :check-box-checked-p *maintain-examples-when-editing*)

             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "retain analysis while editing"
              #'(lambda (item) (setf *retain-analysis-while-editing* (check-box-checked-p item)))
              :check-box-checked-p *retain-analysis-while-editing*)
             
             
             (make-dialog-item 'static-text-dialog-item (make-point x (incf y 18)) nil "Example & Tuple Display Options")
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "use example templates when possible"
              #'(lambda (item) (setf *use-templates-when-possible* (check-box-checked-p item)))
              :check-box-checked-p *use-templates-when-possible*)
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "map extended tuples back to examples"
              #'(lambda (item) (setf *map-tuples-back-to-examples* (check-box-checked-p item)))
              :check-box-checked-p *map-tuples-back-to-examples*)
             
             (make-dialog-item
              'check-box-dialog-item (make-point x2 (incf y 15)) nil "re-analyze automatically after editing"
              #'(lambda (item) (setf *reanalyze-automatically* (check-box-checked-p item)))
              :check-box-checked-p *reanalyze-automatically*)
             
             (make-dialog-item 'static-text-dialog-item (make-point x (incf y 18)) nil "Permanently Record User Responses")
             
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 15)) nil "auto assert"
              #'(lambda (item) item (setf *permanently-record-user-responses* t))
              :radio-button-pushed-p (eq *permanently-record-user-responses* t)
              :radio-button-cluster 2)
             
             (make-dialog-item
              'radio-button-dialog-item (make-point (+ x2 110) y)  nil "ask user"
              #'(lambda (item) item (setf *permanently-record-user-responses* :ask))
              :radio-button-pushed-p (eq *permanently-record-user-responses* :ask)
              :radio-button-cluster 2)
             
             (make-dialog-item
              'radio-button-dialog-item (make-point (+ x2 200) y) nil "auto retract"
              #'(lambda (item) item (setf *permanently-record-user-responses* nil))
              :radio-button-pushed-p (eq *permanently-record-user-responses* nil)
              :radio-button-cluster 2)

             (make-dialog-item 'static-text-dialog-item (make-point x (incf y 18)) nil "Copy Graphs")
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 15)) nil "PICT - for export to other documents"
              #'(lambda (item) item (setf *copy-graph-to* :pict))
              :radio-button-pushed-p (eq *copy-graph-to* :pict)
              :radio-button-cluster 1)
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 15)) nil "TEXT - selected nodes"
              #'(lambda (item) item (setf *copy-graph-to* :selected-nodes))
              :radio-button-pushed-p (eq *copy-graph-to* :selected-nodes)
              :radio-button-cluster 1)
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 15)) nil "TEXT - retain structure"
              #'(lambda (item) item (setf *copy-graph-to* :prolog))
              :radio-button-pushed-p (eq *copy-graph-to* :prolog)
              :radio-button-cluster 1)

             (make-dialog-item 'static-text-dialog-item (make-point x (incf y 18)) nil "Color Level")
             
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 15))  nil "b&w"
              #'(lambda (item) item (setf *color-level* :bw))
              :radio-button-pushed-p (eq *color-level* :bw)
              :radio-button-cluster 3)
             (make-dialog-item
              'radio-button-dialog-item (make-point (+ x2 60) y) nil "b&w compatible color"
              #'(lambda (item) item (setf *color-level* nil))
              :radio-button-pushed-p (eq *color-level* nil)
              :radio-button-cluster 3)
             (make-dialog-item
              'radio-button-dialog-item (make-point (+ x2 230) y) nil "color"
              #'(lambda (item) item (setf *color-level* :color))
              :radio-button-pushed-p (eq *color-level* :color)
              :radio-button-cluster 3)

             
             (make-dialog-item
              'button-dialog-item (make-point (- h 70) (- v 30)) #@(60 20) " OK "
              #'(lambda (item) item (return-from-modal-dialog nil))
              :default-button t)))))
    (modal-dialog dialog t)
    (unless (eql expert-mode *expert-mode*) (es))
    (unless (eql color-level *color-level*) (set-color-level *color-level*))))


;;;_____________________________________
;;; specify-relation-kind

(defun specify-relation-kind (name)
  (let* ((h 350) (v 120) (x 10) (x2 30) (y 5)
         (definition :extensional)
         (dialog
          (make-instance
            'dialog
            :window-show nil
            :window-type :double-edge-box
            :view-position :centered
            :view-size (make-point h v)
            :view-font '("Chicago" 12 :srcor :plain)
            :view-subviews
            (list 
             (make-dialog-item 'static-text-dialog-item (make-point x y) nil (format nil "Indicate how ~S will be defined." name))
             
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 25)) nil "Extensionally  - as a Fact"
              #'(lambda (item) item (setf definition :extensional))
              :radio-button-pushed-p t)
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 18)) nil "Functionally    - as a Builtin"
              #'(lambda (item) item (setf definition :builtin)) )
             (make-dialog-item
              'radio-button-dialog-item (make-point x2 (incf y 18)) nil "Intensionally  - as a Rule"
              #'(lambda (item) item (setf definition :intensional)))
             
             (make-dialog-item
              'button-dialog-item (make-point (- h 140) (- v 30)) #@(60 20) " Cancel "
              #'(lambda (item) item (return-from-modal-dialog :cancel)))
             (make-dialog-item
              'button-dialog-item (make-point (- h 70) (- v 30)) #@(60 20) " OK "
              #'(lambda (item) item (return-from-modal-dialog definition))
              :default-button t)
             
             ))))
    (modal-dialog dialog t)))

;;;_______________________________________
;;;  builtin-flags

(defun builtin-flags ()
  (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-change-builtin-flags *user-monitor*)))
  (let* ((h 280) (v 100) (y 2) (x1 10) (x2 130) (x3 215)
         (dialog-font '("chicago" 12 :plain))
         (dialog (make-instance 
                   'window
                   :window-type :document
                   :close-box-p nil
                   :window-title "Built-In Relations"
                   :window-show nil
                   :view-font dialog-font
                   :view-size (make-point h v)
                   :view-position :centered)))
    (add-subviews
     dialog
     (make-dialog-item 'static-text-dialog-item (make-point 10 y) nil(format nil "~%relations") nil)
     (make-dialog-item 'static-text-dialog-item (make-point 110 y) nil(format nil "~%induction") nil)
     (make-dialog-item 'static-text-dialog-item (make-point 190 y) nil (format nil "     try~%constants") nil))
    (incf y 24)
    (dolist (bucket *builtin-preds*)
      (add-subviews
       dialog
       (make-dialog-item 
        'static-text-dialog-item (make-point x1 (+ (incf y 16) 2)) nil (format nil "~(~A~)" (first bucket)) nil
        :view-font '("monaco" 9 :plain))
       (make-dialog-item 
        'check-box-dialog-item (make-point x2 y) nil ""
        #'(lambda (item) (setf (r-induction (rest (assoc (view-nick-name item) *builtin-preds*))) (check-box-checked-p item)))
        :check-box-checked-p (r-induction (rest bucket))
        :view-nick-name (first bucket))
       (make-dialog-item 
        'check-box-dialog-item (make-point x3 y) nil ""
        #'(lambda (item) (setf (r-try-constants (rest (assoc (view-nick-name item) *builtin-preds*))) (check-box-checked-p item)))
        :check-box-checked-p (r-try-constants (rest bucket))
        :view-nick-name (first bucket))
       ))
    (add-subviews
     dialog
     (make-dialog-item
      'button-dialog-item (make-point 140 (incf y 35)) #@(80 20) " OK "
      #'(lambda (item) item (return-from-modal-dialog nil))
      :default-button t))
    (set-view-size dialog h (incf y 30))
    (modal-dialog dialog t)))

;;;_______________________________________
;;;  avaliable-cliches

(defun avaliable-cliches ()
  (let* ((h 280) (v 100) (y 2) (x1 10) (x2 200)
         (dialog-font '("chicago" 12 :plain))
         (dialog (make-instance 
                   'window
                   :window-type :document
                   :close-box-p nil
                   :window-title "Relational Clichs"
                   :window-show nil
                   :view-font dialog-font
                   :view-size (make-point h v)
                   :view-position :centered)))
    (add-subviews
     dialog
     (make-dialog-item 'static-text-dialog-item (make-point 10 y) nil(format nil "~%Clich") nil)
     (make-dialog-item 'static-text-dialog-item (make-point 170 y) nil(format nil "~%Available") nil))
    (incf y 24)
    (dolist (bucket *all-cliches*)
      (add-subviews
       dialog
       (make-dialog-item 
        'static-text-dialog-item (make-point x1 (+ (incf y 16) 2)) nil (format nil "~(~A~)" (first bucket)) nil
        :view-font '("monaco" 9 :plain))
       (make-dialog-item 
        'check-box-dialog-item (make-point x2 y) nil ""
        #'(lambda (item) (setf *available-relational-cliches*
                               (if (check-box-checked-p item)
                                 (pushnew (view-nick-name item) *available-relational-cliches*)
                                 (delete (view-nick-name item) *available-relational-cliches*))))
        :check-box-checked-p (member (first bucket) *available-relational-cliches*)
        :view-nick-name (first bucket))
       ))
    (add-subviews
     dialog
     (make-dialog-item
      'button-dialog-item (make-point 140 (incf y 35)) #@(80 20) " OK "
      #'(lambda (item) item (return-from-modal-dialog nil))
      :default-button t))
    (set-view-size dialog h (incf y 30))
    (modal-dialog dialog t)))