
;;;; copyright (c) 1990, 1991 by the university of california, irvine. 
;;;; this program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  this program
;;;; may not be sold or incorporated into another product to be sold withou
;;;; written permission from the regents of the university of california.
;;;; this program was written by michael pazzani, cliff brunk, glenn silverstein
;;;; and kamal ali.  

(in-package :user)

(defvar *saved-menubar* nil)
(defmacro catch-nil(&rest body)
  `(let ((cancel-value (catch-cancel ,@body)))
     (if (eq cancel-value :cancel) nil
         cancel-value)))

(defvar *last-goal-typed-in* "")
(defvar *=* '(= . =))
(defparameter *type-files* nil)
(defvar *max-l* 6)
(setq *compile-with-rule-trace* t)
(setq *user-monitor* (make-user-monitor))

(defun already-defined(p &aux r)
  (and (setq r (get-r-struct p))
       (not (eq :undefined (r-kind r)))))
 
(defun trim-list (x &optional (m *max-l*) (end nil) &aux (length (length x) ))
  (if m
    (if (> length m)
      (append (butlast x (- length m)) end)
      x)
    x))

(defmacro notify-error (&rest message)
  `(let ((*print-escape* nil))
     (message-dialog (format nil . ,message)  :size #@(400 200)  :position :centered)))

(defun abort-operation (message)
  (not (y-or-n-dialog message :yes-text "Continue" :no-text  "Abort" :size #@(400 150) :position :centered :cancel-text nil)))

(defun clear-kb ()
  (when (and (or *rules-changed* *facts-changed*)
             (y-or-n-dialog "Do you want to save the knowledge base file?") :position :centered)
    (dump-kb-to-file-as))
  (reset-relations)
  (setf *kb-file* nil))

(defun initialize-everything ()
  (when (user-monitor-p *user-monitor*)
    (setf *user-monitor* (make-user-monitor)))
  (unless (and (or *goal-concept* *predicate-being-learned* *intensional-preds* *extensional-preds*)
               (abort-operation  "A Knowledge Base has already been initialized.  If you continue, this knowledge base will be deleted"))
    (reset-facts)
    (reset-rules)
    (setq *focl-problem* nil)
    (setq *focl-problems* nil)
    (let* ((p (get-atom-from-user "What is the name of the top level goal to prove?")))
      (when p
        (let ((a (get-args "What arguments should be used. Suggestion: (?example) for binary classifications or (?example ?class) for multi-class classifications" nil))
              (ap (intern (format nil "~a-FACT" p))))
          (when a
            (let ((types (get-types p a)))
              (when types
                (let ((q (get-question (cons p a) nil)))
                  (when q
                    (eval `(def-rule ,p :clauses nil :questions ,q :vars ,a :type ,types))
                    (eval `(def-pred ,ap :questions ((:FACT ,q)(:question ())) :vars ,a :type ,types))
                    (set-focl-problem ap :goal-concept-name p)
                    (format t "~%")
                    (setq *goal-concept* (cons p a))
                    (setq *predicate-being-learned* ap)
                    (setq *rules-changed* t)
                    (setq *facts-changed* t)
                    (setq *kb-file* nil)
                    ))))))))))

(defun lisp-file-name(f)
  (merge-pathnames (make-pathname :type "lisp") f))

(defun load-kb (&optional (reset t))
  (catch-cancel 
    (let ((f (choose-file-dialog :button-string "Load"))
          (set-time reset))
      (when (or (null *kb-file*)
                (null reset)
                (not (setf set-time (abort-operation "A Knowledge Base file has already been loaded.  If you continue, the two knowledge bases will be merged. If you merge the two knowledge bases and then use \"Save Knowledge Base\" or \"Save Knowledge Base As\" the merged knowledge base will be stored under the name of the last knowledge base loaded.  If you don't desire this, you should select \"Clear Knowledge Base\" first"))))
        (when reset (setq *kb-file* (lisp-file-name f)))
        (when set-time (setq *load-time* (get-time)))
        (setq *rules-changed* nil
              *facts-changed* nil)
        (eval-enqueue  `(progn (load ',f :print t) (format t "~%KNOWLEDGE BASE LOADED.~%")))
        (values)))))

#|
(defun load-kb (&optional (reset t) &aux f)
  (unless (eq :cancel (setq f (catch-cancel (choose-file-dialog :button-string "Load"))))
    (when (or (null *kb-file*)
              (null reset)
              (not (abort-operation "A Knowledge Base file has already been loaded.  If you continue, the two knowledge bases will be merged. If you merge the two knowledge bases and then use \"Save Knowledge Base\" or \"Save Knowledge Base As\" the merged knowledge base will be stored under the name of the last knowledge base loaded.  If you don't desire this, you should select \"Clear Knowledge Base\" first")))
      (when reset (setq *kb-file* (lisp-file-name f)))
      (setq *rules-changed* nil
            *facts-changed* nil)
      (eval-enqueue  `(progn (load ',f :print t) (update-relations) (format t "~%KNOWLEDGE BASE LOADED.~%")))
      (values))))
|#

(defun merge-kb ()
  (catch-cancel
    (let ((f (choose-file-dialog :button-string "Merge")))
      (eval-enqueue `(progn (merge-kb-file ',f) (update-relations) (format t "~%KNOWLEDGE BASE MERGED~%")))
      (setq *kb-file* nil)
      (values))))

(defun merge-kb-file (f)
  (with-open-file (s f)
    (do ((form (read s nil :eof)(read s nil :eof)))
        ((eq form :eof))
      (case (first form)
        (def-type (merge-type form))
        (def-rule (merge-rule form))
        (def-pred (merge-pred form))
        ((def-example-template def-focl-problem set-builtin-flags))
        (set-user-monitor form)              ;; <- new
      (t (format t "~%~A ignored" form))))))

(defun merge-type(f) 
  (let ((type (second f))
        (vals (cddr f)))
    (if (equal vals (get-type-instances type))
      (format t "~%~a old=new" type)
      (when (or (null (get-type-instances type))
                (progn (format t "~%~%~a~%Old Type: ~a~%New Type: ~a"
                               type (get-type-instances type) vals)
                       (yes-or-no-p "Use New")))
        (eval f)))))

                                    
                                   
(defun merge-rule(f)
  (let ((r (second f))
        (clauses (cadr (member :clauses f))))
    (if (equal clauses (get r 'clauses))
      (format t "~%~a old=new" r)
      (when (or (null (get r 'clauses))
          (progn (format t "~%~%~a~%Old Rule: ~a~%New Rule: ~a"
                         r (get r 'clauses) clauses)
                 (yes-or-no-p "Use New")))
          (eval f)))))



(defun merge-pred(f)
  (let* ((r (second f))
        (def (cons (cadr (member :pos f))
                   (cadr (member :neg f))))
        (old (if (pred-p (get-r-struct r))
               (cons (r-pos (get-r-struct r)) 
                     (r-neg (get-r-struct r))))))
    (if (equal def old)
      (format t "~%~a old=new" r)
      (when (or (null old)
                (progn (format t "~%~%~a~%Old Pos: ~a~%Old Neg: ~a~%New Pos: ~a~%New Neg: ~a"
                               r (first old)(rest old)(first def)(rest def))
                       (yes-or-no-p "Use New")))
        (eval f)))))

(defun compile-a-file (&aux f)
  (unless (eq :cancel (setq f (catch-cancel (choose-file-dialog :mac-file-type 'text :button-string "Compile"))))
    (eval-enqueue  `(compile-file ',f))))

(defun quit-es ()
  (map-windows #'(lambda (w)
                   (when (changed w) (window-select w))
                   (window-close w))
               :class 'examples-window :include-invisibles t)
  (map-windows #'(lambda (w) (window-close w)) :include-invisibles t)
  (unless (eq :cancel
              (catch-cancel 
                (when (and (or *rules-changed* *facts-changed*)
                           (y-or-n-dialog "Do you want to save the knowledge base file?" :position :centered))
                  (dump-kb-to-file-as))))
    (ccl:quit)))

(defun explain-last ()
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-explain-last-proof *user-monitor*)))
  (create-show-window (with-output-to-string (out) (print-trace *last-explanation* out))
                      (format nil "Explain Last Proof")))

(defun get-new-instance-for-type (type)
  (let ((name (get-atom-from-user (format nil "Enter a new instance of the type ~a" type) "No More")))
    (when name 
      (setq *facts-changed* t)
      (pushnew-type-instance type name))))

(defun select-relations (r-structs &optional (message "Select relations") (title "Select Relations"))
  (select-item r-structs
               :window-title title
               :selection-type :disjoint
               :message message
               :table-print-function #'pretty-print-r-struct-name
               :table-name-function #'(lambda (r) (symbol-name (r-name r)))))

(defun trace-all ()
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-trace-all *user-monitor*)))
  (setq *traced-predicates* :all))

(defun untrace-all ()
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-untrace-all *user-monitor*)))
  (setq *traced-predicates* nil))

(defun add-trace ()
  (catch-cancel
    (let ((relations (select-relations (remove-if #'(lambda (x) (member (r-name x) *traced-predicates*)) *r-structs*)
                                       "Select relations to trace" "Add Trace")))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-add-trace *user-monitor*)))
      (when relations
        (setq *traced-predicates* (nconc *traced-predicates* (mapcar #'r-name relations)))))))

(defun remove-trace ()
  (catch-cancel
    (let ((relations (select-relations (mapcar #'get-r-struct (setf *traced-predicates* (sort *traced-predicates* #'universal<)))
                                       "Select relations to untrace" "Remove Trace")))
      (when relations
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-remove-trace *user-monitor*)))
        (setq *traced-predicates* (set-difference *traced-predicates* (mapcar #'r-name relations)))))))

(defun add-spy ()
  (catch-cancel
    (let ((relations (select-relations (remove-if #'(lambda (x) (or (not (rule-p x))
                                                                    (member (r-name x) *spy-preds*))) *r-structs*)
                                       "Select rule to spy" "Add Spy")))
      (when relations
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-add-spy *user-monitor*)))
        (setq *spy-preds* (nconc *spy-preds* (mapcar #'r-name relations)))))))

(defun remove-spy ()
  (catch-cancel
    (let ((relations (select-relations (mapcar #'get-r-struct (setf *spy-preds* (sort *spy-preds* #'universal<)))
                                       "Select rule to unspy" "Remove Spy")))
      (when relations
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-remove-spy *user-monitor*)))
        (setq *spy-preds* (set-difference *spy-preds* (mapcar #'r-name relations)))))))


;;;  TYPE MANIPULATION FUNCTIONS

(defun list-type (name &optional (stream t))
  (when name
    (format stream "~%TYPE ~(~s~)" name)
    (cond ((member name *special-types*) (format stream "  <--  DEFINED INTERNALLY"))
          ((member name *all-types*) (mapc #'(lambda (instance) (format stream "~%     ~a" instance)) (get-type-instances name)))
          (t (format stream "  <--  UNDEFINED") ))))

(defun list-a-type (&optional (name nil))
  (catch-cancel
    (unless name
      (setf name (select-a-type (user-defined-types) "Select a type to show" "Show Type"))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-show-type *user-monitor*))))
    (create-show-window (with-output-to-string (out) (list-type name out))
                        (format nil "Show Type - ~(~s~)" name))))

(defun list-types ()
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-show-all-types *user-monitor*)))
  (create-show-window (with-output-to-string (out) (mapc #'(lambda (X) (list-type x out) (format out "~%")) (user-defined-types)))
                      "Show All Types"))

(defun new-type ()
  (let ((name (get-atom-from-user "Enter a name for the new type: ")))
    (when name
      (cond ((or (member name *all-types*)
                 (member name *special-types*))
             (message-dialog (format nil "A type named ~(~S~) is already defined." name) :position :centered :size #@(400 100)))
            (t
             (when (user-monitor-p *user-monitor*)
               (incf (user-monitor-menu-new-type *user-monitor*)))
             (pushnew name *all-types*)
             (setq *facts-changed* t)
             (do ((new (get-new-instance-for-type name)
                       (get-new-instance-for-type name)))
                 ((null new)))
             (list-a-type name)
             (update-types))))))

(defun rename-type (&optional (old-name nil))
  (catch-cancel
    (let ((user-defined-types (user-defined-types)))
      (unless old-name
        (setf old-name (select-a-type user-defined-types "Select a type to rename" "Rename Type")))
      (let ((new-name (type-in-predicate (format nil "Enter a new type name for ~(~S~)" old-name)))
            (doit nil))
        (unless (or (null new-name)
                    (eql old-name new-name))
          (cond ((or (member new-name *special-types*)
                     (member new-name *all-types*))
                 (when (y-or-n-dialog (format nil "The type ~(~S~) is already defined.  ~%The definition of ~(~S~) will not be renamed, but~%all uses of ~(~S~) will be changed to ~(~S~)." new-name old-name old-name new-name)
                                      :position :centered :cancel-text nil :yes-text " OK " :no-text " Cancel " :size #@(450 120))
                   (setf doit t)))
                (t
                 (setf *all-types* (delete old-name *all-types*)
                       *all-types* (sort (pushnew new-name *all-types*) #'universal<)
                       (get new-name 'focl-instances) (get-type-instances old-name)
                       (get old-name 'fold-instances) nil
                       doit t)))
          (when doit
            (dolist (r *r-structs*)
              (unless (member r *special-r-structs*)
                (setf (r-type r) (nsubst new-name old-name (r-type r)))))
            (update-types)
            (update-relations)))))))

(defun copy-type (&optional (name nil))
  (catch-cancel
    (unless name
      (setf name (select-a-type *all-types* "Select a type to copy" "Copy Type")))
    (let ((copy-name (type-in-predicate (format nil "Enter a name for the copy of ~(~s~)." name))))
      (when copy-name
        (cond ((or (member copy-name *all-types*)
                   (member copy-name *special-types*))
               (message-dialog (format nil "A type named ~(~S~) is already defined." copy-name) :position :centered :size #@(400 100)))
              (t (setf *all-types* (sort (pushnew copy-name *all-types*) #'universal<)
                       (get copy-name 'focl-instances) (copy-list (get-type-instances name)))
                 (update-types)))))))

(defun delete-type (&optional (type nil))
  (catch-cancel
    (when (and (or type (setf type (select-a-type *all-types* "Select a type to delete" "Delete Type")))
               (or *expert-mode*
                   (y-or-n-dialog (format nil "Do you really want to delete ~a?" (pretty-print-type-name type nil)) :yes-text "Yes, Delete" :no-text  "No" :cancel-text nil)))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-menu-delete-type *user-monitor*)))
      (set-type-instances type nil)
      (mapc #'(lambda(p) (setf (get type p) nil)) *type-properties*)
      (setf  *facts-changed* t
             *all-types* (delete type *all-types*))
      (update-types))))

(defun add-to-type (&optional (type nil))
  (catch-cancel
    (when (or type (setf type (select-a-type *all-types* "Select a type to add to" "Add To Type")))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-add-to-type *user-monitor*)))
      (do ((new (get-new-instance-for-type type) (get-new-instance-for-type type)))
          ((null new)))
      (list-a-type type))))

(defun delete-from-type (&optional (type nil) &aux c)
  (catch-cancel
    (when (and (or type (setf type (select-a-type *all-types* "Select a type to delete from" "Delete From Type"))) 
               (not (eq :cancel (setq c (select-item (get-type-instances type)
                                                     :window-title "Delete From Type"
                                                     :message (format nil "Select element(s) of ~a to be deleted" (pretty-print-type-name type nil))
                                                     :selection-type :disjoint))))
               c)
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-delete-from-type *user-monitor*)))
      (set-type-instances type (set-difference (get-type-instances type) c))
      (setq *facts-changed* t)
      (list-a-type type))))

#|
(defun new-rule ()
  (let ((p (type-in-predicate "Type name of predicate to be concluded.")))
    (when p 
      (if (already-defined p)
        (notify-existing-relation-error p)
        (let ((a (get-arguments p)))
          (when a
            (let ((types (get-types p a)))
              (when types
                (let ((q (get-question (cons p a) nil)))
                  (when q
                    (let ((c (create-new-clause p a)))
                      (when c
                        (when (user-monitor-p *user-monitor*)
                          (incf (user-monitor-new-rule *user-monitor*)))
                        (eval `(def-rule ,p :clauses (,c) :questions ,q :vars ,a :type ,types))
                        (setq *rules-changed* t) 
                        (show-rule-def p)))))))))))))
|#

(defun new-rule ()
  (catch-cancel
    (let* ((head (create-new-relation "Enter name and arguments of new rule.
For example, (related ?X ?Y)"))
           (name (first head))
           (args (rest head)))
      (if (already-defined name)
        (notify-existing-relation-error name)
        (let ((types (get-types name args)))
          (when types
            (let ((q (get-question head nil)))
              (when q
                (let ((c (create-new-clause name args)))
                  (when c
                    (when (user-monitor-p *user-monitor*)
                      (incf (user-monitor-new-rule *user-monitor*)))
                    (eval `(def-rule ,name :clauses (,c) :questions ,q :vars ,args :type ,types))
                    (setq *rules-changed* t)
                    (show-rule (get-rule name))))))))))))

(defun create-new-relation (&optional (message "Create new relation")
                                      (initial-string ""))
  (let ((string (get-string-from-user message
                                      :initial-string initial-string
                                      :position :centered)))
  (multiple-value-bind (value error) (catch-error-quietly (read-from-string string))
    (unless error
      (setf error
            (cond ((null value)
                   "A literal must contain relation name and arguments.")
                  ((not (consp value))
                   "A literal must be enclosed in parentheses.")
                  ((not (every #'(lambda (v) (pcvar-p v)) (rest value)))
                   "All arguments must be variables.")
                  ((null (rest value))
                   "A relation must have at least one argument.")
                  ((not (atom (first value)))
                   "A literal must contain a relation name.")
                  (t nil))))
    (cond (error
           (notify-error  "~%~a~%   ~a is ill-formed." error string)
           (create-new-relation message string))
          (t value)))))

;;;  Shouldn't there be a way to change the goal concept and predicate independently?

(defun change-top ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to be the new goal concept" "Change Top Level Predicate"))
           (rule-name (r-name rule))
           (fact-name (intern (format nil "~a-FACT" rule-name)))
           (arguments (r-vars rule))
           (types (r-type rule))
           (questions (r-questions rule)) )
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-change-top-level-predicate *user-monitor*)))
      (setq *predicate-being-learned* fact-name)
      (setq *goal-concept* (cons rule-name arguments))
      (unless (get-r-struct fact-name)
        (eval `(def-pred ,fact-name :questions ((:FACT ,questions)(:question ())) :vars ,arguments :type ,types)))
      (set-focl-problem fact-name :goal-concept-name rule-name)
      (setq *rules-changed* t)
      (setq *facts-changed* t))))

(defun copy-a-rule ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to copy" "Copy Rule"))
           (name (r-name rule)))
      (when (rule-p rule)
        (let ((copy-name (type-in-predicate (format nil "Enter a name for the copy of ~(~s~).~%" name))))
          (when (and copy-name (not (eql copy-name name)))
            (cond ((already-defined copy-name)
                   (notify-existing-relation-error copy-name))
                  (t
                   (when (user-monitor-p *user-monitor*)
                     (incf (user-monitor-copy-rule *user-monitor*)))
                   (re-def-rule rule
                                :name copy-name
                                :vars (copy-list (r-vars rule))
                                :type (copy-list (r-type rule))
                                :mode (copy-list (r-mode rule))
                                :clauses (subst copy-name name (get-clauses name))
                                :deterimancy (copy-list (r-determinacy rule))
                                :questions (copy-list (r-questions rule)))
                   (setf *rules-changed* t)))))))))


(defun rename-a-rule ()
  (catch-cancel
    (let ((rule (select-a-rule "Select a rule to rename" "Rename Rule")))
      (when (rule-p rule)
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-rename-rule *user-monitor*)))
        (rename-relation rule)))))

(defun rename-a-fact ()
  (catch-cancel
    (let ((pred (select-a-relation (delete-if #'(lambda (p) (eq (r-name p) (predicate-being-learned))) (remove-if-not #'pred-p *r-structs*))
                                   "Select a fact to rename" "Rename Fact")))
      (when (pred-p pred)
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-rename-fact *user-monitor*)))
        (rename-relation pred)))))

(defun rename-relation (r-struct)
  (when (r-p r-struct)
    (cond ((member r-struct *special-r-structs*)
           (special-r-struct-message r-struct))
          ((or (rule-p r-struct) (pred-p r-struct))
           (let* ((old-name (r-name r-struct))
                  (new-name (type-in-predicate (format nil "Enter a new name for ~(~s~).~%" old-name) (format nil "~(~s~)" old-name))))
             (when (and new-name (not (eql old-name new-name)))
               (if (already-defined new-name)
                 (notify-existing-relation-error new-name)
                 (case (r-kind r-struct)
                   (:intensional (re-def-rule r-struct :name new-name :clauses (subst new-name old-name (get-clauses old-name)))
                                 (delete-r-struct r-struct)
                                 (revise-callers old-name new-name)
                                 (update-focl-problems old-name new-name)
                                 (setf *rules-changed* t))
                   (:extensional (setf (r-name (get-r-struct old-name)) new-name)
                                 (revise-callers old-name new-name)
                                 (update-focl-problems old-name new-name)
                                 (set-r-struct new-name r-struct)
                                 (set-r-struct old-name nil)
                                 (setf *facts-changed* t)
                                 (setf *extensional-preds* (mapcar #'(lambda(x)
                                                                       (if (eq (first x) old-name)
                                                                         (cons new-name (rest x))
                                                                         x))
                                                                   *extensional-preds*))
                                 ;(focl-compile-facts new-name)  
                                 ;No need since defun is on r-struct
                                 (setf *r-structs* (sort *r-structs* #'r-name<))
                                 (update-relations)))))))
           (t (notify-error "Only predicates and rules can be renamed.")))))


(defun update-focl-problems (old-relation-name new-relation-name)
  (if (eql *predicate-being-learned* old-relation-name)
    (setf *predicate-being-learned* new-relation-name))
  (if (eql (first *goal-concept*) old-relation-name)
    (rplaca *goal-concept* new-relation-name))
  (dolist (problem *focl-problems*)
    (when (eql (first problem) old-relation-name)
      (rplaca problem new-relation-name))
    (when (eql (getf (rest problem) :goal-concept-name) old-relation-name)
      (setf (getf (rest problem) :goal-concept-name) new-relation-name))))

(defun new-fact ()
  (catch-cancel
    (let* ((head (create-new-relation "Enter name and arguments of new fact.
For example, (married ?X ?Y)"))
           (name (first head))
           (args (rest head)))
      (if (already-defined name)
        (notify-existing-relation-error name)
        (let ((types (get-types name args)))
          (when types
            (let ((s (get-question head nil)))
              (when s 
                (let ((q (get-question head nil 1)))
                  (when q
                    (let ((qs (when (rest args) (mapcar #'(lambda(x)(cons x (get-variable-questions head x (remove x args)))) args))))
                      (when (user-monitor-p *user-monitor*)
                        (incf (user-monitor-new-fact *user-monitor*)))
                      (eval `(def-pred ,name :questions ((:fact ,s) (:question ,q) .,qs)
                               :vars ,args :type ,types))
                      (setq *facts-changed* t)
                      (show-fact (get-pred name)))))))))))))

(defun change-var-type ()
  (catch-cancel
    (let* ((fact (select-a-fact "Select a fact to modify" "Change Variable Type"))
           (name (r-name fact)))
      (when (pred-p fact)
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-change-variable-type *user-monitor*)))
        (let* ((vars (r-vars fact))
               (types (r-type fact))
               n v new)
          (setq v (catch-cancel (first (select-item vars
                                                    :window-title "Change Variable Type"
                                                    :message "What variable do you want to change the type of"))))
          (when (and v (not (eq v :cancel)))
            (setq n (position v vars))
            (setq new (catch-cancel (get-type-from-user name v)))
            (when (and n (not (eq new :cancel)))
              (setf (elt types n) new)
              (setq *facts-changed* t))))))))
          
(defun describe-fact (name-or-struct &optional (stream t))
  (let (pred name)
    (if (pred-p name-or-struct)
      (setf pred name-or-struct
            name (r-name pred))
      (setf name name-or-struct
            pred (get-r-struct name)))
    (when (pred-p pred)
      (format stream "~%FACT DEFINITION: ~a   Variable Types: ~s" (name-vars-string name (r-vars pred)) (r-type pred))
      (when (r-pos pred)
        (format stream "~%  Positive Examples")
        (mapc #'(lambda(x) (format stream "~%    ~a" (name-vars-string name x))) (trim-list (r-pos pred)))
        (when (and *max-l* (> (length (r-pos pred)) *max-l*))
          (format stream "~%")))
      (when (r-neg pred)
        (format stream "~%~%  Negative Examples")
        (mapc #'(lambda(x) (format stream "~%    ~a" (name-vars-string name x))) (trim-list(r-neg pred)))
        (when (and *max-l* (> (length (r-neg pred)) *max-l*))
          (format stream "~%")))
      (when (r-questions pred)
        (format stream "~%English Translations")
        (mapc #'(lambda(x)
                  (cond ((pcvar-p (first x))
                         (format stream "~%   To ask about ~s~:{~%~6T if ~s is known, use ~s ~s~}"
                                 (first x) (rest x)))
                        (t (format stream "~%  ~s ~s" (first x)(cadr x)))))
              (r-questions pred))
        (format stream "~%~%")))))

(defun delete-entire-fact ()
  (catch-cancel
    (let* ((fact (select-a-fact "Select a fact to delete" "Delete Fact"))
           (name (r-name fact)))
      (when (and (pred-p fact) (y-or-n-dialog (format nil "Do you really want to delete the predicate ~a with the following positive examples: ~{ ~a~}" name
                                                      (trim-list (r-pos fact) *max-l* '()))
                                              :yes-text "Yes, Delete"
                                              :no-text  "No"
                                              :cancel-text nil
                                              :size #@(480 200)))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-delete-fact *user-monitor*)))
      (setf *facts-changed* t)
      (delete-r-struct fact)
      (format t "~%~a deleted~%" name)))))

(defun delete-fact ()
  (catch-cancel
    (let* ((fact (select-a-fact "Select a fact to retract an assertion" "Delete Positive Fact"))
           (name (r-name fact)))
      (when (pred-p fact)
        (let ((v (catch-cancel (select-item (r-pos fact)
                                            :window-title "Delete Positive Fact"
                                            :message "Select the assertions to delete"
                                            :selection-type :disjoint))))
          (unless (or (null v)(eq v :cancel)
                      (not (y-or-n-dialog (format nil "Do you really want to delete~{ ~a~} from ~a" v name)
                                          :yes-text "Yes, Delete"
                                          :no-text  "No"
                                          :cancel-text nil
                                          :size #@(480 200))))
            (when (user-monitor-p *user-monitor*)
              (incf (user-monitor-delete-positive-fact *user-monitor*)))
            (setq *facts-changed* t)
            (mapc #'(lambda (c) (retract-fact (cons name c))) v)
            (show-fact fact)
            ))))))

(defun delete-negative-fact ()
  (catch-cancel
    (let* ((fact (select-a-fact "Select a fact to retract an assertion" "Delete Negative Fact"))
           (name (r-name fact)))
      (when (pred-p fact)
        (let ((v (catch-cancel (select-item (r-neg fact)
                                            :window-title "Delete Negative Fact"
                                            :message "Select the assertions to delete"
                                            :selection-type :disjoint))))
          (unless (or (null v)(eq v :cancel)
                      (not (y-or-n-dialog (format nil "Do you really want to delete~{ ~a~} from negative examples of ~a" v name)
                                          :yes-text "Yes, Delete"
                                          :no-text  "No"
                                          :cancel-text nil
                                          :size #@(480 200))))
            (when (user-monitor-p *user-monitor*)
              (incf (user-monitor-delete-negative-fact *user-monitor*)))
            (setq *facts-changed* t)
            (mapc #'(lambda (as) (setf (r-neg fact) (delete as (r-neg fact) :test #'equalp))) v)
            (show-fact fact)
            ))))))
      

(defun get-variable-questions(f var vars &optional (r nil))
  (unless (and  r
           (null (rest vars)))
  (when (y-or-n-dialog (format nil "Are there any (more) ways to ask the user about ~a" var)
                         :yes-text "Yes"
                         :no-text  "No"
                         :position :centered
                         :cancel-text nil)
    (let ((q (get-var-q f var vars)))
      (when q
        (cons q (get-variable-questions f var vars t)))))))

(defun get-var-q (f v vars)
  (let ((kvars (catch-cancel (if (rest vars)
                               (select-item vars
                                            :window-title "Variable Values"
                                            :message (format nil "Select the variable(s) that must be known to ask for the value of ~a" v)
                                            :selection-type :disjoint)
                               vars))))
    (unless (eq kvars :cancel)
      (let ((multi (y-or-n-dialog (format nil "For a given value of the variable(s) ~a can there be more than one value of ~a" kvars v)
                                  :yes-text "Yes"
                                  :no-text  "No"
                                  :position :centered
                                  :cancel-text nil))
            (q (get-question f nil 3 kvars v)))
        (when q (list kvars q (if multi :multi-valued :single-valued)))))))


(defun changes-vars-and-questions ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to modify" "Change Vars/English"))
           (name (r-name rule)))
      (when (rule-p rule)
        (let ((a (get-arguments name nil (r-vars rule))))
          (when a
            (let ((types (if (not (= (length (r-vars rule))
                                     (length a)))
                           (get-types name a)
                           (r-type rule))))
              (when types
                (when (user-monitor-p *user-monitor*)
                  (incf (user-monitor-change-vars-english *user-monitor*)))
                (let ((q (get-question (cons name a) (r-questions rule))))
                  (when q
                    (cond ((and (eq name (first *goal-concept*))
                                (not (= (length (r-vars rule))
                                        (length a))))
                           (unless(abort-operation (format nil
                                                           "You have changed the number of variables used by the top level predicate.  If you do not abort, the definition
~a will be reset.  This means the ~a example(s) you have defined will be deleted
since they are not valid. " *predicate-being-learned* (length (r-pos (get-r-struct *predicate-being-learned*)))))
                             (eval `(def-pred , *predicate-being-learned*  :questions ((:fact ,q))
                                      :vars ,a :type ,types))
                             (re-def-rule rule :questions q :vars a :type types :mode nil)))
                          (t (re-def-rule rule :questions q :vars a :type types :mode nil)))))))))))))


(defun change-questions-fact ()
  (catch-cancel
    (let* ((fact (select-a-fact "Select a fact to change translation" "Change English"))
           (name (r-name fact)))
      (when (pred-p fact)
        (let ((q (get-vars-for-question (cons name (r-vars fact))
                                        (mapcar #'car (r-questions fact))
                                        (r-questions fact))))
          (when (user-monitor-p *user-monitor*)
            (incf (user-monitor-change-english *user-monitor*)))
          (when q 
            (setq *facts-changed* t)
            (describe-fact fact)))))))

(defun delete-questions-fact ()
  (catch-cancel
    (let ((fact (select-a-relation (delete-if-not #'(lambda (p) (and (r-questions p) (rest (r-vars p)))) (remove-if-not #'pred-p *r-structs*))
                                   "Select a fact to delete translation" "Delete English")))
      (when (pred-p fact)
        (let ((q (delete-vars-for-question (delete-if-not #'pcvar-p (mapcar #'first (r-questions fact))) (r-questions fact))))
          (when q 
            (setq *facts-changed* t)
            (describe-fact fact)))))))

(defun delete-vars-for-question (vars qs &aux x)
  (setq x (catch-cancel
            (let ((v (first (select-item vars
                                         :window-title "Delete English"
                                         :message "What do you want to delete the English of?"))))
              (when (user-monitor-p *user-monitor*)
                (incf (user-monitor-delete-english *user-monitor*)))
              (let* ((q (assoc v qs :test #'equalp))
                     (kvars (first (select-item (rest q)
                                                :window-title "Delete English"
                                                :message "Which question do you want to delete?"
                                                :selection-type :single))))
                (when kvars
                  (setf (rest q) (delete kvars (rest q)))
                  t)))))
  (unless (eq x :cancel) x))

(defun add-questions-fact ()
  (catch-cancel
    (let ((fact (select-a-relation (delete-if-not #'(lambda (r) (rest (r-vars r))) (remove-if-not #'pred-p *r-structs*))
                                   "Select a fact to add translation" "Add English")))
      (when (pred-p fact)
        (let ((q (add-vars-for-question (r-name fact) (r-vars fact) (r-questions fact) )))
          (when (user-monitor-p *user-monitor*)
            (incf (user-monitor-add-english *user-monitor*)))
          (when q
            (setq *facts-changed* t)
            (describe-fact fact)))))))

(defun change-var-type-rule ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to modify" "Change Variable Type"))
           (name (r-name rule)))
      (when (rule-p rule)
        (let* ((vars (r-vars rule))
               (types (r-type rule))
               n v new)
          (setq v (catch-cancel (first (select-item vars
                                                    :window-title "Change Variable Type"
                                                    :message "What variable do you want to change the type of"))))
          (when (and v (not (eq v :cancel)))
            (setq n (position v vars))
            (setq new (catch-cancel (get-type-from-user name v)))
            (when (and n (not (eq new :cancel)))
              (setf (elt types n) new)
              (setq *rules-changed* t))))))))


(defun add-vars-for-question (p vars qs &aux x b)
  (setq x (catch-cancel
            (let* ((v (first (select-item vars
                                          :window-title "Add Variables for Question"
                                          :message "What variable do you want to add a question for?")))
                   (q (get-var-q (cons p vars) v (remove v vars))))
              (when q
                (setq b (assoc v qs :test #'equalp))
                (cond ((null b)(nconc qs (list (cons v (list q)))))
                      (t (nconc b (list q))))))))
  (unless (eq x :cancel) x))


(defun get-vars-for-question (r vars qs &aux x)
  (setq x (catch-cancel
            (let ((v (first (select-item vars
                                         :window-title "Get Variables for Question"
                                         :message "What variable do you want to change the English for?"))))
              (unless (eq v :cancel)
                (let* ((q (assoc v qs :test #'equalp))
                       (kvars nil))
                  (cond ((pcvar-p v)
                         (setq q 
                               (first (select-item (rest q)
                                                   :window-title "Get Variables for Question"
                                                   :message "Which Question do you want to Change")))
                         (setq kvars (remove-if-not #'pcvar-p (cadr q)))
                         (let ((multi 
                                (y-or-n-dialog (format nil "For a given value of the variable(s) ~a can there be more than one value of ~a" 
                                                       kvars v)
                                               :yes-text "Yes"
                                               :no-text  "No"
                                               :position :centered
                                               :cancel-text nil))
                               (new-q (get-question r (cadr q) 3 kvars v)))
                           (when new-q
                             (setf (cadr q) new-q)
                             (setf (caddr q) (if multi :multi-valued :single-valued)))))
                        
                        (t (let ((new-q (get-question r  (cadr q) (if (eq v :fact) 2 1))))
                             (when new-q
                               (setf (cadr q) new-q))))))))))
  (unless (eq x :cancel) x))


(defun get-question (r d &optional (question 2) (a (rest r))(v nil) &aux
                       (string (case  question
                                 (1 (format nil 
                                            "How should a question about ~a be expressed?~
                                             ~%For example, (AGE ?PERSON ?N) is~
                                             ~%(is ?PERSON ?N years old)" r))
                                 (2 (format nil "How should ~a be expressed?~
                                                 ~%For example, (AGE ?PERSON ?N) is~
                                                 ~%(?PERSON is ?N years old)" r))
                                 (3 (format nil "How should a question for ~a of ~a ~%be expressed using ~a~
                                                 ~%For example, To ask about ?N of (AGE ?PERSON ?N):~
                                                 ~%(How old is ?PERSON)" v r a)))))
  (let ((s (catch-cancel (get-text-from-user string :initial-string (if d (if (stringp d) d (format nil "~s" d)) "")))))
    (unless (eq s :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string s))
        (cond  ((or error
                    (not (consp value)))
                (notify-error "~%~a- ~a ill-formed question" error s)
                (get-question r s question a v))
               ((some #'(lambda(x) (and (pcvar-p x) (not (member x value :test #'equalp)))) a)
                (notify-error "~%~a: Every variable in ~a must be used.~%~%" value a)
                (get-question r s question a v))
               ((some #'(lambda(x)(and (pcvar-p x)
                                       (not (member x a :test #'equalp))))
                      value)
                (notify-error "~%An unknown variable cannot be used in ~a~
                               ~%Only variables in ~a can be used.~%~%" value a)
                (get-question r s question a v))
               (t value))))))

(defun add-clause-to-rule ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to modify" "Add Clause To Rule"))
           (name (r-name rule)))
      (when (rule-p rule)
        (let ((position (select-a-clause (cons '(head |New clause should be first|) (get-clauses name)) 
                                              "Select a clause to insert the new clause after" "Add Clause To Rule"))
              (c (create-new-clause name))
              new-clauses rest)
          (when c
            (cond ((equal (rest position) '(|New clause should be first|))
                   (setf new-clauses (cons c (get-clauses name))))
                  (t (setf rest (rest (member position (get-clauses name))))
                     (setf new-clauses
                           (nconc (ldiff (get-clauses name) rest)
                                  (list c)
                                  rest))))
            (when (user-monitor-p *user-monitor*)
              (incf (user-monitor-add-clause-to-rule *user-monitor*)))
            (re-def-rule rule :clauses new-clauses)))))))

  
(defun add-literal-to-clause ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to modify" "Add Literal To Clause"))
           (name (r-name rule)))
      (when (rule-p rule)
        (let ((c (select-a-clause (get-clauses name) "Select a clause to modify" "Add Literal To Clause")))
          (when c
            (let ((position (select-a-literal (cons '|New literal should be first literal| (rest c))
                                            "Select a literal to insert the new literal after" "Add Literal To Clause")))
              (when position
                (let ((l (create-new-literal c)) rest new-clause)
                  (when l
                    (cond ((eq position '|New literal should be first literal|)
                           (setf new-clause
                                 (cons (first c)
                                       (cons l (rest c)))))
                          (t (setf rest (rest (member position c)))
                             (setf new-clause
                                   (nconc (ldiff c rest)
                                          (list l)
                                          rest))))
                    (when (user-monitor-p *user-monitor*)
                      (incf (user-monitor-add-literal-to-clause *user-monitor*)))
                    (re-def-rule (get-rule name) :clauses (nsubstitute new-clause c (get-clauses name)))))))))))))
                  
                                        
(defun create-new-literal (c  &optional (negation nil))
  (unless negation 
    (format t "~%Create a new literal for~%IF ~a~%THEN ~a~%~%"
            (rest c)(first c)))
  (let ((l (catch-cancel
             (let ((p (prompt-for-predicate negation)))
               (when p
                 (if (eq p '!)
                   p
                   (let* ((s (get-r-struct p))
                          (all-vars (collect-vars c))
                          (vars (when s (r-vars s)))
                          (default-vars (if (and vars (subsetp vars all-vars)) vars))
                          (a (get-arguments p negation default-vars c)))
                     (when a (cons p a)))))))))
    (if (eq l :cancel)  nil l)))

(defun create-new-clause (r &optional (a nil))
  (let ((v (or a 
               (r-vars (get-rule r))
               (progn (format t "~%Type arguments for THEN part~%") 
                      (get-arguments r)))))
    (when v
      (let ((c (list (cons r v))))
        (create-clause-with-head-and-body c (user-create-literals nil c))))))

(defun create-clause-with-head-and-body (head x)
  (cond ((null x) x)
        ((consp (car x)) (cons head x))
        (t (list head x))))
  


(defun get-arguments(p &optional (negation nil) (d nil)(c nil))
  (cond ((eq p 'not) (let ((l (create-new-literal c t))) (when l (list l))))
        ((eq p 'is) (get-args "Enter 2 argument(s) for is. For example (?x (+ ?y 7))" 2  d))
        (t (let ((s (get-r-struct p)))
            (get-args (format nil "Enter ~a argument(s) for ~a~a~%For example ~a" 
                     (if (or (null s) d) "" (r-arity s))
                     (if negation "not " "")
                     p 
                     (if (or (null s)
                             (null (r-vars s)))
                       "(?x ...)" 
                       (r-vars s)))
                      (if s (r-arity s))
                      d)))))

(defun get-types(p a &aux c)
  (setq c (catch-cancel
  (mapcar #'(lambda(a)
              (get-type-from-user p a))
          a)))
  (unless (eq c :cancel) c))

(defun get-type-from-user (p a)
  (let ((type (first (select-item (sort (nconc (copy-list *all-types*) (list :anything :number)) #'universal<)
                                  :window-title "Get Type From User"
                                  :message (format nil "Select a type for ~a of ~a" a p)
                                  :ok-text "OK"
                                  :cancel-text "Cancel"
                                  :type-in-text "Type In"
                                  :table-print-function #'pretty-print-type-name))))
    (cond ((null type) :anything)
          ((eq type :anything) type)
          ((eq type :number) type)
          ((member type *all-types*) type)
          (t (setf *all-types* (insert-inorder type *all-types* #'universal<))
             (format t "~%~s : New Type Defined~%" type)
             type))))

(defun prompt-for-predicate (&optional (negation nil)
                                         (message (if negation "Select a relation to be negated" "Select a relation")))
    (let* ((name-or-struct
            (first (select-item *r-structs*
                                :window-title "Select Relation"
                                :message message
                                :selection-type :single
                                :table-print-function #'pretty-print-r-struct-name
                                :table-name-function #'(lambda (r) (symbol-name (r-name r)))
                                :type-in-text "Type In")))
           (pred-name (if (r-p name-or-struct) (r-name name-or-struct) name-or-struct)))
      (cond ((symbolp pred-name) pred-name)
            (t (message-dialog (format nil "~a is not a valid predicate name." pred-name) :position :centered)
               (prompt-for-predicate negation message)))))

(defun get-args (string arity &optional (d nil))
  (let ((v (catch-cancel (get-string-from-user string :initial-string (if d (format nil "~a" d) "") :position :centered))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) (catch-error-quietly (read-from-string v))
        (cond ((or error (not (consp value)))
               (notify-error "~%~a- ~a ill-formed arguments" error v)
               (get-args string arity v))
              ((and arity (null d) (not (eq arity (length value))))
               (notify-error "You typed ~a.  ~a argument(s) are required~%" value arity)
               (get-args string arity v))
              (t value))))))

(defun type-in-predicate (&optional (message nil) (initial-string ""))
  (let ((v (catch-cancel (get-string-from-user (cond ((null message) "Type a relation name")
                                                     ((eq message t) "Type a relation name (to be negated)")
                                                     ((stringp message) message)
                                                     (t "Type a relation name"))
                                               :initial-string initial-string
                                               :position :centered))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) (catch-error-quietly (read-from-string v))
        (cond  ((or error (not (symbolp value)))
                (notify-error "~%~a- ~a ill-formed predicate" error v) nil)
               (t value))))))

(defun delete-literal-from-clause ()
    (catch-cancel
      (let* ((rule (select-a-rule "Select a rule to modify" "Delete Literal From Clause"))
             (name (r-name rule)))
        (when (rule-p rule)
          (let ((c (select-a-clause (get-clauses name) "Select a clause to modify" "Delete Literal From Clause")))
            (when c
              (let ((l (select-a-literal (rest c) "Select a literal to delete" "Delete Literal From Clause")))
                (when (and l
                           (y-or-n-dialog (format nil "Do you really want to delete~% ~a from~%IF ~a~%THEN ~a"
                                                  l (rest c) (first c))
                                          :yes-text "Yes, Delete"
                                          :no-text  "No"
                                          :cancel-text nil
                                          :position :centered
                                          :size #@(480 200)))
                  
                  (setq *rules-changed* t)
                  (delete l c :test #'equal) ;note destructive delete always works because car is head of list
                  (when (user-monitor-p *user-monitor*)
                    (incf (user-monitor-delete-literal-from-clause *user-monitor*)))
                  (re-def-rule rule :clauses (get-clauses name))
                  ))))))))

(defun delete-clause-from-rule ()
    (catch-cancel
      (let* ((rule (select-a-rule "Select a rule to modify" "Delete Clause From Rule"))
             (name (r-name rule)))
        (when (rule-p rule)
          (let ((c (select-a-clause (get-clauses name) "Select a clause to delete" "Delete Clause From Rule")))
            (when (and c
                       (y-or-n-dialog (format nil "Do you really want to delete~%IF ~a~%THEN ~a"  (rest c) (first c))
                                      :yes-text "Yes, Delete"
                                      :no-text  "No"
                                      :cancel-text nil
                                      :position :centered
                                      :size #@(480 200)))
              (setq *rules-changed* t)
              (when (user-monitor-p *user-monitor*)
                (incf (user-monitor-delete-clause-from-rule *user-monitor*)))
              (re-def-rule rule :clauses (delete c (get-clauses name) :test #'equal))
              ))))))

(defun text-edit-clause ()
  (catch-cancel
    (let* ((rule (select-a-rule "Select a rule to modify" "Text Edit Clause"))
           (name (r-name rule)))
      (when (rule-p rule)
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-text-edit-clause *user-monitor*)))
        (prog* ((c (select-a-clause (get-clauses name) "Select a clause to edit" "Text Edit Clause"))
                (newc c))
          (unless c (return))
          (when (setq newc (create-clause-with-head-and-body (car newc)
                                                             (user-create-literals (cdr newc) (car newc))))
          
          (setq *rules-changed* t)
          (re-def-rule (get-rule name) :clauses (substitute newc c (get-clauses name)))
          ))))))

(defun delete-the-rule ()
  (catch-cancel
    (let* ((rule (select-a-relation (delete-if #'(lambda (r) (eql (r-name r) (goal-concept-name))) (remove-if-not #'rule-p *r-structs*))
                                    "Select a rule to delete" "Delete Rule"))
           (name (r-name rule)))
      (when (and (rule-p rule)
                 (or *expert-mode*
                     (y-or-n-dialog (format nil "Do you really want to delete ~a" name)
                                    :yes-text "Yes, Delete"
                                    :no-text  "No"
                                    :position :centered
                                    :cancel-text nil)))
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-delete-rule *user-monitor*)))
        (setf *rules-changed* t)
        (delete-r-struct rule)
        (format t "~%~a deleted" name)))))

(defclass show-window (fred-window) ())

(defmethod window-needs-saving-p ((window show-window)) window nil)

(defun create-show-window (&optional (contents "")
                                     (title "Show Window")
                                     (position :centered)
                                     (size #@(500 200)))
  (let ((window (find-window title)))
    (cond ((window-open? window)
           (select-all window)
           (clear window))
          (t
           (setf window (make-instance 'show-window
                          :window-title title
                          :view-position position
                          :view-size size))))
    (buffer-insert (fred-display-start-mark window) contents)
    (fred-update window)
    (window-select window)
    (set-mini-buffer window "")
    window))

(defun show-rule (&optional (rule nil))
  (catch-cancel
    (unless (rule-p rule)
      (setf rule (select-a-rule "Select a rule to display" "Show Rule"))
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-show-rule *user-monitor*))))
    (when (rule-p rule) 
      (create-show-window (with-output-to-string (out) (show-rule-def rule out))
                          (format nil "Show Rule - ~(~s~)" (r-name rule))))))

(defun show-rules ()
  (when (user-monitor-p *user-monitor*)
    (incf (user-monitor-show-all-rules *user-monitor*)))
  (create-show-window (with-output-to-string (out) (mapc #'(lambda(r) (when (rule-p r) (show-rule-def r out))) *r-structs*))
                      (format nil "Show All Rules")))

(defun collect-example-facts (example)
  (let ((pos-facts nil)
        (neg-facts nil))
    (dolist (r *r-structs*)
      (when (pred-p r)
        (dolist (p (r-pos r))
          (when (member example p) (push (cons (r-name r) p) pos-facts)))
        (dolist (n (r-neg r))
          (when (member example n) (push (cons (r-name r) n) neg-facts)))))

        (when *find-related-facts*
          (let* ((name (format nil "~s-" example))
                 (namel (length name)))
          (dolist (r *r-structs*)
            (when (pred-p r)
              (dolist (p (r-pos r))
                (when (find-if  #'(lambda(x)
                              (and (symbolp x)
                                (let* ((xname (symbol-name x)) (xlength (length xname)))
                                  (and (> xlength namel)
                                       (string-equal name xname :end1  namel  :end2  namel)))))
                          p)
                  (push (cons (r-name r) p) pos-facts)))
              (dolist (n (r-neg r))
                (when (find-if  #'(lambda(x)
                              (and (symbolp x)
                                (let* ((xname (symbol-name x)) (xlength (length xname)))
                                (and (> xlength namel)
                                     (string-equal name xname :end1 namel :end2 namel))))) 
                          n)
                  (push (cons (r-name r) n) neg-facts)))))))
        (values (nreverse pos-facts) (nreverse neg-facts))))

(defun show-facts-for-example (&optional (delete nil))
  (catch-cancel 
    (let* ((examples (sort (nconc (mapcar #'car (r-pos (get-r-struct *predicate-being-learned*)))
                                  (mapcar #'car (r-neg (get-r-struct *predicate-being-learned*))))
                           #'universal<))
           (example (first (select-item examples
                                        :window-title (if delete "Delete Example" "Show Example")
                                        :message (if delete "Select an example to delete" "Select an example to display")))))
      (when example
        (if delete
          (delete-example-facts example)
          (multiple-value-bind (pos-facts neg-facts) (collect-example-facts example)
            (when (user-monitor-p *user-monitor*)
              (incf (user-monitor-show-example *user-monitor*)))
            (create-show-window
             (with-output-to-string (out)
               (when pos-facts
                 (format out "~%Positive Facts")
                 (mapc #'(lambda (f) (format out "~%  ~a" (literal-string f))) pos-facts))
               (when neg-facts
                 (format out "~%~%Negative Facts")
                 (mapc #'(lambda (f) (format out "~%  ~a" (literal-string f))) neg-facts)))
            (format nil "Show Example - ~S" example)))))))
    (values))

(defun show-all-examples ()
  (let ((r-struct (get-r-struct *predicate-being-learned*)))
    (when (r-p r-struct)
      (create-show-window  
       (with-output-to-string (out)
         (dolist (example (sort (nconc (mapcar #'car (r-pos r-struct)) (mapcar #'car (r-neg r-struct))) #'universal<))
           (when (user-monitor-p *user-monitor*)
             (incf (user-monitor-show-example *user-monitor*)))
           (multiple-value-bind (pos-facts neg-facts) (collect-example-facts example)
             (format out "~S" example)
             (when pos-facts
               (format out "~%  Positive Facts")
               (mapc #'(lambda (f) (format out "~%    ~a" (literal-string f))) pos-facts))
             (when neg-facts
               (format out "~%  Negative Facts")
               (mapc #'(lambda (f) (format out "~%    ~a" (literal-string f))) neg-facts))
             (format out "~%~%"))))
       (format nil "Examples of ~A" *predicate-being-learned*))))
  (values))

(defun delete-example-facts (example)
  (multiple-value-bind (pos-facts neg-facts) (collect-example-facts example)
    (when (or pos-facts neg-facts)
      (when (user-monitor-p *user-monitor*)
        (incf (user-monitor-delete-example *user-monitor*)))
      (without-interrupts 
       (let* ((facts (nconc pos-facts (mapcar #'(lambda (n) (list 'not n)) neg-facts)))
              (window (make-instance 
                        'selector-dialog
                        :window-title (format nil "Delete Example - ~A" example)
                        :window-show nil
                        :view-position :centered
                        :close-box-p nil
                        :view-size #@(400 160)))
              (all-button (make-dialog-item
                           'button-dialog-item #@(0 0) #@(60 20) "All"
                           #'(lambda (item) item (return-from-modal-dialog :all))
                           :dialog-item-enabled-p t))
              (selected-button (make-dialog-item
                                'button-dialog-item #@(0 0) #@(60 20) "Selected"
                                #'(lambda (item) item
                                   (let ((item-selector (find-named-sibling item :item-selector)))
                                     (return-from-modal-dialog 
                                      (mapcar #'(lambda (cell) (cell-contents item-selector cell)) (selected-cells item-selector)))))
                                :dialog-item-enabled-p nil))
              (none-button (make-dialog-item
                            'button-dialog-item #@(0 0) #@(60 20) "None"
                            #'(lambda (item) item (return-from-modal-dialog :none))
                            :dialog-item-enabled-p t)))
         (add-buttons-and-selector   
          window
          facts
          :message (format nil "The following are all the facts about ~A.~%Select those that should be deleted." example)
          :selection-type :disjoint
          :buttons (list all-button selected-button none-button)
          :enable-only-when-selected (list selected-button)
          :print-function #'(lambda (l s) (format s "~A" (literal-string l)))
          :name-function nil)
         (let ((assertions-to-retract (modal-dialog window t)))
           (case assertions-to-retract
             (:all (mapc #'retract-fact facts))
             (:none nil)
             (:closed nil)
             (otherwise (mapc #'retract-fact assertions-to-retract)))))))))

(defun delete-all-examples ()
  (labels ((delete-example (example)
             (multiple-value-bind (pos-facts neg-facts) (collect-example-facts (first example))
               (mapc #'retract-fact pos-facts)
               (mapc #'(lambda (f) (retract-fact f t)) neg-facts))))
    (let ((r-struct (get-r-struct *predicate-being-learned*)))
      (when (r-p r-struct)
        (when (y-or-n-dialog (format nil "Do you want to delete all the examples of ~A and all the assertions associated with those examples?" *predicate-being-learned*) :position :centered :cancel-text nil)
          (mapc #'delete-example (r-pos r-struct))
          (mapc #'delete-example (r-neg r-struct)))))))


(defun record-changes-to-facts? ()
  (when (or *new-facts-pos* *new-facts-neg*)
    (cond ((eq *permanently-record-user-responses* :ask)
           (without-interrupts 
            (let* ((window (make-instance 
                             'selector-dialog
                             :window-title "Record New Facts"
                             :window-show nil
                             :view-position :centered
                             :view-size #@(400 160)))
                   (all-button (make-dialog-item
                                'button-dialog-item #@(0 0) #@(60 20) "All"
                                #'(lambda (item) item (return-from-modal-dialog :none))
                                :dialog-item-enabled-p t))
                   (selected-button (make-dialog-item
                                     'button-dialog-item #@(0 0) #@(60 20) "Selected"
                                     #'(lambda (item) item
                                        (let ((item-selector (find-named-sibling item :item-selector)))
                                          (return-from-modal-dialog
                                           (set-difference (table-sequence item-selector)
                                                           (mapcar #'(lambda (cell) (cell-contents item-selector cell))
                                                                   (selected-cells item-selector))
                                                           :test #'equal))))
                                     :dialog-item-enabled-p nil))
                   (none-button (make-dialog-item
                                 'button-dialog-item #@(0 0) #@(60 20) "None"
                                 #'(lambda (item) item (return-from-modal-dialog :all))
                                 :dialog-item-enabled-p t)))
              (add-buttons-and-selector   
               window
               (nconc (copy-list *new-facts-pos*) (mapcar #'(lambda (n) (list 'not n)) *new-facts-neg*))
               :message (format nil "The following facts were recently asserted.~%Select those that should be permanently recorded")
               :selection-type :disjoint
               :buttons (list all-button selected-button none-button)
               :enable-only-when-selected (list selected-button)
               :print-function #'(lambda (l s) (format s "~A" (literal-string l)))
               :name-function nil)
              (let ((assertions-to-retract (modal-dialog window t)))
                (case assertions-to-retract
                  (:all (mapc #'retract-fact *new-facts-pos*)
                        (mapc #'(lambda (f) (retract-fact f t)) *new-facts-neg*)
                        (when (user-monitor-p *user-monitor*)
                          (incf (user-monitor-retract-new-facts *user-monitor*))))
                  (:none nil)
                  (:closed nil)
                  (otherwise (mapc #'retract-fact assertions-to-retract)
                             (when (user-monitor-p *user-monitor*)
                               (incf (user-monitor-retract-new-facts *user-monitor*)))))))))
          ((not *permanently-record-user-responses*)
           (mapc #'retract-fact *new-facts-pos*)
           (mapc #'(lambda (f) (retract-fact f t)) *new-facts-neg*)))
    (setq *new-facts-pos* nil
          *new-facts-neg* nil)))

(defun show-fact (&optional (fact nil))
  (catch-cancel
    (let ((*max-l* nil))
      (unless (pred-p fact)
        (setf fact (select-a-fact "Select a fact to display" "Show Fact"))
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-show-fact *user-monitor*))))
      (when (pred-p fact)
        (create-show-window (with-output-to-string (out) (describe-fact fact out))
                            (format nil "Show Fact - ~(~s~)" (r-name fact)))))))

(defun show-facts ()
  (let ((*max-l* nil))
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-show-all-facts *user-monitor*)))
    (create-show-window (with-output-to-string (out) (mapc #'(lambda(r) (when (pred-p r) (describe-fact r out))) *r-structs*))
                        (format nil "Show All Facts"))))


;;; [this function is flawed]
;;; Not isn't handled correctly, while and and or are not handled at all.

(defun check-rules (&optional (edit-it nil))
  (when (user-monitor-p *user-monitor*)
    (if edit-it
      (incf (user-monitor-edit-rule-warnings *user-monitor*))
      (incf (user-monitor-check-rules *user-monitor*))))
  (let ((stream (make-string-output-stream))
        (errors 0)
        v type edit-me so-far n fl lr rl)
    (unless edit-it
      (format stream "~%Errors and Warnings...~%~%"))
    (mapc #'(lambda (s)
              (when (rule-p s)
                (setq n (r-name s))
                (setq edit-me nil)
                (setq so-far errors)
                (mapc #'(lambda (clause)
                          (setq v (mapcar #'cons (r-vars s) (r-type s)))
                          (mapc #'(lambda (literal)
                                    (cond ((listp literal)
                                           (if (eq (first literal) 'not)
                                             (setq literal (cadr literal)))
                                           (setq fl (first literal)
                                                 rl (rest literal)
                                                 lr (get-r-struct fl))
                                           (cond ((not (already-defined fl))
                                                  (unless (equal literal '(fail))
                                                    (when (not (r-p lr))
                                                      (setf lr (make-r :name fl :vars rl :arity (length rl) :kind :undefined))
                                                      (set-r-struct fl lr)
                                                      (add-r-struct lr))
                                                    (setf (r-type lr) (mapcar #'(lambda (var) (or (rest (assoc var v :test #'equalp)) :anything)) rl))
                                                    (incf errors)
                                                    (setq edit-me n)
                                                    (format stream ";Undefined Term:            ~a in ~(~s~)~%" (literal-string literal) n)))
                                                 ((not (= (length (rest literal))
                                                          (length (r-type lr))))
                                                  (incf errors)(setq edit-me n)
                                                  (format stream ";Wrong Number of Arguments: ~a in ~(~s~) uses ~(~s~)~%" (literal-string literal) n
                                                          (r-type lr)))                                                 
                                                 (t (mapc #'(lambda (arg vtype)
                                                              (cond ((pcvar-p arg)
                                                                     (setf type (rest (assoc arg v :test #'equalp)))
                                                                     (cond ((null type) (push (cons arg vtype) v))
                                                                           ((type-eq type vtype))
                                                                           (t (incf errors) (setq edit-me n)
                                                                              (format stream ";Variable Type Mismatch:    ~s should be a ~s: ~a in ~(~s~)~%" arg vtype (literal-string literal) n))))
                                                                    (t
                                                                     (cond ((constant-type-eq arg vtype))
                                                                           (t (incf errors) (setq edit-me n)
                                                                              (format stream ";Constant Type Mismatch:    ~s should be a ~s: ~a in ~(~s~)~%" arg vtype (literal-string literal) n))))
                                                                    ))
                                                          (rest literal) (r-type lr)))))
                                          ((not (eq literal '!)) (setq edit-me n)
                                           (format stream ";Undefined Literal:         ~a in ~(~s~)~%" (literal-string literal) n)
                                           (incf errors))))
                                (rest clause))
                          
                          (mapc #'(lambda(v)
                                    (incf errors) (setq edit-me n)
                                    (format stream ";Singleton Variable:        ~a in (~a)~%" v (clause-body-string clause)))
                                (singleton-vars clause)))
                      (get-clauses n))
                
                (when (and (> errors so-far)
                           (not edit-it))
                  (format stream "~%"))
                (when (and edit-me edit-it) 
                  (text-edit-relation s :header (string-downcase (get-output-stream-string stream))
                                      :extra-lines (- errors so-far)))
                ))
          *r-structs*)
    (unless edit-it
      (create-show-window (format nil "~A~%~a problems found" (get-output-stream-string stream) errors) (format nil "Check Rules"))))
  (update-relations))

(defun revise-callers(p new-p)
  (mapc #'(lambda(n.r)
            (when (some #'(lambda(clause)
                            (some #'(lambda(literal)
                                      (cond ((listp literal)
                                             (cond((eq (first literal) 'not)
                                                   (setq literal (cadr literal))))
                                             (eq (first literal) p))))
                                  (rest clause)))
                        (get-clauses (first n.r)))
              (format t "~%Changing ~a to use ~a instead of ~a ~%" (first n.r) new-p p)
              (re-def-rule (rest n.r) :clauses (subst new-p p (get-clauses (first n.r))))))
        *intensional-preds*))

(defun who-calls (&optional (relation-name nil))
  (catch-cancel 
    (let ((from-menu? (not relation-name)))
      (unless relation-name
        (setq relation-name (r-name (select-a-relation *r-structs* "Select a relation" "Who Calls"))))
      (when relation-name
        (when (user-monitor-p *user-monitor*)
          (if from-menu?
            (incf (user-monitor-who-calls-menu *user-monitor*))
            (incf (user-monitor-who-calls-relation-window *user-monitor*))))
        (create-who-calls-window (rules-using-relation relation-name) relation-name)))))
    
(defun create-who-calls-window (callers callee-name)
  (let ((window (find-window "Callers"))
        (message (format nil "Rules that use ~(~s~)." callee-name)))
    (if (window-open? window)
      (let* ((selector (view-named :item-selector window))
             (sp (scroll-position  selector)))
        (dolist (cell (selected-cells selector))
          (cell-deselect selector cell))
        (set-table-sequence selector callers)
        (set-dialog-item-text (view-named :message window) message)
        (scroll-to-cell selector sp)
        (update-selector-dialog window selector))
      (setf window (create-relation-manipulation-window :r-structs callers
                                                        :title "Callers"
                                                        :message message
                                                        :view-position :centered
                                                        :view-size #@(450 200))))
    (window-select window)
    window))

(defun singleton-vars (clause)
  (let* ((vars (collect-vars clause))
         (uvars (remove-duplicates vars)))
    (delete-if #'(lambda(v)(or (not (= (count v vars :test #'equalp) 1))
                               (and (symbolp (pcvar-id v))
                                    (eq #\_ (char (symbol-name (pcvar-id v)) 0)))))
               uvars)))

(defun collect-vars(l)
  (if (consp l)
    (nconc (collect-vars (first l)) (collect-vars (rest l)))
    (if (pcvar-p l) (list l))))
    
                           
(defun run-rules-on-examples (&optional (type-in))
  (catch-cancel 
    (let* ((r (get-r-struct *predicate-being-learned*))
           (answers  (r-pos r))
           (examples (nconc (mapcar #'(lambda(x)(list (first x))) answers)
                            (mapcar #'(lambda(x)(list (first x))) 
                                    (r-neg (get-r-struct *predicate-being-learned*)))))
           (*batch* nil)
           (*maintain-prolog-rule-trace* t)
           a e)
      (setq *new-facts-pos* nil
            *new-facts-neg* nil
            e (if type-in 
                (if (eq (first (r-type r)) 'example)
                  (gentemp "EXAMPLE-")
                  (get-atom-from-user "Type a name for the example"))
                (first (first (select-item examples
                                           :window-title "Run On Old Example"
                                           :message "Select an old example")))))
      (when e
        (when (user-monitor-p *user-monitor*)
          (if type-in
            (incf (user-monitor-run-on-new-example *user-monitor*))
            (incf (user-monitor-run-on-old-example *user-monitor*))))
        (setq e (add-extra-variables e (first *goal-concept*)))
        (cond ((and type-in (member e examples :test #'equal))
               (notify-error "There is already an example named ~a" e))
              (t
               (when e
                 (pushnew-type-instance (first (r-type (get-r-struct (first *goal-concept*)))) (first e))
                 (multiple-value-setq  (a *last-explanation*) (prove-goal `(,(first *goal-concept*)  . ,e)))
                 (if type-in
                   (cond ((null a)
                          (supposed-to-fail? e))
                         (t (supposed-to-succeed? e a)))
                   (cond ((null a)
                          (format t "~%~a was not proved to be true~%" (cons (first *goal-concept*)  e))
                          (when (setq a (first (member e answers :test #'equal)))
                            (format t "Previously, it was indicated that ~a should be inferred" a)))
                         (t (format t "~%~a was infered~%" a)
                            (unless (first (member (rest a) answers :test #'equal))
                              (format t "Previously, it was not indicated that ~a should be inferred" a)))))
                 (record-changes-to-facts?))))))))

(defun bagof-on-example ()
  (catch-cancel 
    (let* ((answers  (r-pos (get-r-struct *predicate-being-learned*)))
           (examples (nconc (mapcar #'(lambda(x)(list (first x))) answers)
                            (mapcar #'(lambda(x)(list (first x))) 
                                    (r-neg (get-r-struct *predicate-being-learned*)))))
           (*batch* nil)
           (*maintain-prolog-rule-trace* t)
           a e)
      (setq *new-facts-pos* (setq *new-facts-neg* nil))
      (setq e (first (first (select-item examples
                                         :window-title "Find All Proofs"
                                         :message "Select An Old Example"))))
      (when e
        (when (user-monitor-p *user-monitor*)
          (incf (user-monitor-find-all-proofs *user-monitor*)))
        (setq e (add-extra-variables e (first *goal-concept*)))
        (multiple-value-setq  (a *last-explanation*)
          (prove-goal `(bagof ,(rest e) (,(first *goal-concept*)  . ,e)  ?bag)))
        (if a (format t "~%~a was proved: ~a" e a)
            (format t "~%~a was not proved" e))
        (record-changes-to-facts?)))))

(defun supposed-to-fail? (e)
  (format t "~%~{~a ~}was not proved to be true~%" (instantiate-english (cons (first *goal-concept*)  e)))
  (when *permanently-record-user-responses*
    (if (y-or-n-dialog (format nil  "~{~a ~} failed.~% Was this intended?"  (instantiate-english (cons (first *goal-concept*)  e)))
                       :cancel-text nil
                       :position :centered
                       :size #@(400 150))
      (assert-fact (cons *predicate-being-learned* e) t)
      (get-desired-answer *predicate-being-learned* e))))

(defun supposed-to-succeed? (e a)
  (format t  "~%~{~a ~} was infered.~%" (instantiate-english a))
  (when *permanently-record-user-responses*
   
  (if (y-or-n-dialog (format nil  "~{~a ~} was infered.~% Is this the intended answer?" (instantiate-english a))
                     :cancel-text nil
                     :position :centered
                     :size #@(400 150))
    (assert-fact (cons *predicate-being-learned* (rest a)))
    (get-desired-answer *predicate-being-learned* e))))

(defun get-desired-answer (p e)
  (let ((*batch* nil)
        (*maintain-prolog-rule-trace* nil))
    ;;side effect of proof gets asks user questions
    ;;*maintain-prolog-rule-trace* is set to nil to avoid distirburing "real" proof
    (prove (list (cons p e)))))


(defun get-atom-from-user (string &optional cancel)
  (let ((v (catch-cancel (if cancel
                           (get-string-from-user string :cancel-text cancel :position :centered)
                           (get-string-from-user string :position :centered)))))
    (unless (eq v :cancel)
      (multiple-value-bind (value error) 
                           (catch-error-quietly (read-from-string v))
        (cond  ((or error
                    (not (atom value)))
                (notify-error "~%~a- ~a ill-formed symbol" error v)
                nil)
               (t value))))))

(defun get-new-clause (c &optional string)
  (cons (first c)
        (if (rest c)
          (get-conjunction-from-user (format nil "Type a new body for ~a" (literal-string (first c)))
                                     (or string (format nil "(~a)" (clause-body-string (rest c)))))
          (get-conjunction-from-user (format nil "Type a new clause for ~a" (literal-string (first c))) "" ))))

(defun get-conjunction-from-user (message initial-string)
  (let* ((string (get-text-from-user message  :initial-string initial-string))
         (clause (read-conjunction-from-string string)))
    (if (eq clause :invalid)
      (get-conjunction-from-user message string)
      clause)))
        
(defun read-conjunction-from-string (string)
  (multiple-value-bind (value error) (catch-error-quietly (read-from-string string))
    (unless (or error (consp value))
      (setf error "Not a valid conjunction of literals."))
    (cond (error
           (notify-error "~%~a~%  ~a ill-formed conjunction" error string)
           :invalid)
          (t
           value))))

(defun validate-conjunction (conjunction)
  (multiple-value-bind (undefined-terms wrong-args) (invalid-literals conjunction)
    (cond ((and undefined-terms wrong-args)
           (notify-error "Undefined relations:~%   ~{~a~^, ~}~%Literals with the wrong number of arguments:~%   ~{~a~^, ~}" undefined-terms wrong-args)
           :invalid)
          (undefined-terms
           (notify-error "Undefined relations:~%   ~{~a~^, ~}" undefined-terms)
           :invalid)
          (wrong-args
           (notify-error "Literals with the wrong number of arguments:~%   ~{~a~^, ~}" wrong-args)
           :invalid)
          (t
           conjunction))))

(defun invalid-literals (conjunction)
  (let ((undefined-terms nil)
        (wrong-args nil)
        relation vars r-struct)
    (dolist (literal conjunction)
      (cond ((consp literal)
             (setf relation (first literal)
                   vars (rest literal)
                   r-struct (get-r-struct relation))
             (case relation
               ((and or not)
                (multiple-value-bind (ut wa) (invalid-literals vars)
                  (setf undefined-terms (nconc undefined-terms ut)
                        wrong-args (nconc wrong-args wa))))
               (fail nil)
               (otherwise
                (cond ((not (r-p r-struct)) (push relation undefined-terms))
                      ((not (= (r-arity r-struct) (length vars))) (push literal wrong-args))))))
            (t
             (cond ((eq literal '!))
                   ((eq literal 'fail))
                   (t (push literal undefined-terms))))))
    (values undefined-terms
            wrong-args)))

(defun test-on-all-examples ()
    (when (user-monitor-p *user-monitor*)
      (incf (user-monitor-test-on-all-examples *user-monitor*)))
    (judge-rules-on-examples)
    (record-changes-to-facts?))

;;;_____________________________________
;;;  create-nodes-window

(defclass prove-goal-window (window) ())

;;;_____________________________________
;;;  set-view-size

(defmethod set-view-size ((window prove-goal-window) h &optional v)
  (without-interrupts
   (call-next-method window h v)
   (update-prove-goal-window window)))

;;;_____________________________________
;;;  window-zoom-event-handler

(defmethod window-zoom-event-handler ((window prove-goal-window) message)
  (without-interrupts
   (call-next-method window message)
   (update-prove-goal-window window)))

(defmethod update-prove-goal-window ((window prove-goal-window)) 
  (without-interrupts
   (with-focused-view window
     (rlet ((view-rect :rect :topleft #@(0 0) :bottomright (view-size window)))
       (#_eraserect view-rect)
       (#_beginupdate (wptr window))
       (let* ((length (point-v (view-size window)))
              (width (point-h (view-size window)))
              (edit-length (round (- length 100) 2))
              (edit-width (- width 22))
              (goal (view-named :goal window))
              (result-caption (view-named :result-caption window))
              (result (view-named :result window))
              (non-batch (view-named :non-batch window))
              (prove (view-named :prove window))
              (y 2))

         (set-view-position result-caption 10 y)
         (set-view-position goal 3 (incf y 14))
         (set-view-size goal edit-width edit-length)

         (set-view-position result-caption 10 (incf y (+ edit-length 17)))
         (set-view-position result 3 (incf y 14))
         (set-view-size result edit-width edit-length)

         (set-view-position non-batch 10 (- length 25))
         (set-view-position prove (- width 100) (- length 27))
         
         (#_endupdate (wptr window))
         (#_invalrect view-rect))))))

  
(defun create-prove-goal-window ()
  (let ((x 500)
        (y 223)
        (prove-goal-window (find-window "Prove Goal")))
    (unless (window-open? prove-goal-window)
      (setf prove-goal-window (make-instance 'prove-goal-window
                                :view-size (make-point x y)
                                :view-position :centered
                                :window-title "Prove Goal"
                                :window-type :document-with-zoom
                                :window-show nil))
      (add-subviews
       prove-goal-window
       (make-dialog-item 'static-text-dialog-item nil nil "Enter a goal to prove   For example,  ((= ?x 1)(is ?y (+ 3 ?x)))"
                         nil :view-font '("monaco" 9 :bold) :view-nick-name :goal-caption)
       (make-dialog-item 'ccl::scrolling-fred-dialog-item #@(0 0) nil *last-goal-typed-in* nil
                         :allow-tabs t :allow-returns t :view-font '("monaco" 9 :plain) :view-nick-name :goal)
       (make-dialog-item 'static-text-dialog-item nil nil "Result" nil :view-font '("monaco" 9 :bold) :view-nick-name :result-caption)
       (make-dialog-item 'ccl::scrolling-fred-dialog-item #@(0 0) nil "" nil
                         :allow-tabs t :allow-returns t :view-font '("monaco" 9 :plain) :view-nick-name :result)
       (make-dialog-item 'check-box-dialog-item nil nil "query user while attempting to prove goal" nil
                         :view-font '("monaco" 9 :bold)  :check-box-checked-p nil :view-nick-name :non-batch)
       (make-dialog-item 'button-dialog-item nil #@(70 20) "Prove"
                         #'(lambda (item)
                             (let* ((string (dialog-item-text (find-named-sibling item :goal)))
                                    (goal (read-conjunction-from-string string)))
                               (unless (eq goal :invalid)
                                 (setf goal (validate-conjunction goal))
                                 (unless (eq goal :invalid)
                                   (setf *last-goal-typed-in* string)
                                   (let ((*batch* (not (check-box-checked-p (find-named-sibling item :non-batch))))
                                         (*maintain-prolog-rule-trace* t)
                                         (fred-result (find-named-sibling item :result))
                                         (result nil))
                                     (when (user-monitor-p *user-monitor*)
                                       (if *batch*
                                         (incf (user-monitor-prove-goal-batch *user-monitor*))
                                         (incf (user-monitor-prove-goal-interactive *user-monitor*))))
                                     (multiple-value-setq (result *last-explanation*) (compile-and-prove goal))
                                     (record-changes-to-facts?)
                                     (select-all fred-result)
                                     (clear fred-result)
                                     (buffer-insert (fred-display-start-mark fred-result)
                                                    (cond ((null result) (format nil "Not Proved"))
                                                          ((consp (first result)) (format nil "Proved:~% (~a)" (clause-body-string result)))
                                                          (t (format nil "Proved:~% (~a)" (clause-body-string (list result))))))
                                     (fred-update fred-result))))))
                         :default-button t
                         :view-nick-name :prove))
      (update-prove-goal-window prove-goal-window))
    (window-select prove-goal-window)))


(defun ES ()
  (setf *menubar-frozen* t
        ccl::*suppress-compiler-warnings* *es-suppress-compiler-warnings*
        *error-output* *terminal-io*)
  (unless *saved-menubar* (setf *saved-menubar* (rest (menubar))))
  (mapc #'menu-deinstall (menubar))
  (menu-install (if *expert-mode* *es-expert-file-menu* *es-file-menu*))
  (menu-install *edit-menu*)
  (if *expert-mode* (menu-install *eval-menu*))
  (if *expert-mode* (menu-install *tools-menu*))
  (menu-install *es-rules-menu*)
  (menu-install *es-facts-menu*)
  (menu-install *es-type-menu*)
  (menu-install *es-run-menu*)
  (menu-install *es-windows-menu*)
  (menu-install *es-display-menu*)
  (menu-install *es-learn-menu*)
  (setf *menubar-frozen* nil)
  (draw-menubar-if)
  (values))

(defun reset-menubar ()
  (setf *menubar-frozen* t
        *error-output* ccl::*pop-up-terminal-io*
        ccl::*suppress-compiler-warnings* *saved-warnings*)
  (mapc #'menu-deinstall (menubar))
  (mapc #'menu-install *saved-menubar*)
  (setf *saved-menubar* nil
        *menubar-frozen* nil)
  (draw-menubar-if)
  nil)


(defun notify-existing-relation-error (name)
  (notify-error "There is already a relation named ~(~s~).~%Delete it first, or use a different name.~%~%" name))

(defun set-user-monitor (um)
  (cond
   ((not (user-monitor-p um)))
   ((not (user-monitor-p *user-monitor*)) (setf *user-monitor* um))
   (t
    (incf (user-monitor-show-rule *user-monitor*) (user-monitor-show-rule um))
    (incf (user-monitor-show-all-rules *user-monitor*) (user-monitor-show-all-rules um))
    (incf (user-monitor-new-rule *user-monitor*) (user-monitor-new-rule um))
    (incf (user-monitor-copy-rule *user-monitor*) (user-monitor-copy-rule um))
    (incf (user-monitor-rename-rule *user-monitor*) (user-monitor-rename-rule um))
    (incf (user-monitor-delete-rule *user-monitor*) (user-monitor-delete-rule um))
    (incf (user-monitor-add-clause-to-rule *user-monitor*) (user-monitor-add-clause-to-rule um))
    (incf (user-monitor-add-literal-to-clause *user-monitor*) (user-monitor-add-literal-to-clause um))
    (incf (user-monitor-delete-clause-from-rule *user-monitor*) (user-monitor-delete-clause-from-rule um))
    (incf (user-monitor-delete-literal-from-clause *user-monitor*) (user-monitor-delete-literal-from-clause um))
    (incf (user-monitor-text-edit-clause *user-monitor*) (user-monitor-text-edit-clause um))
    (incf (user-monitor-text-edit-rule *user-monitor*) (user-monitor-text-edit-rule um))
    (incf (user-monitor-change-vars-english *user-monitor*) (user-monitor-change-vars-english um))
    (incf (user-monitor-show-fact *user-monitor*) (user-monitor-show-fact um))
    (incf (user-monitor-show-all-facts *user-monitor*) (user-monitor-show-all-facts um))
    (incf (user-monitor-show-example *user-monitor*) (user-monitor-show-example um))
    (incf (user-monitor-new-fact *user-monitor*) (user-monitor-new-fact um))
    (incf (user-monitor-rename-fact *user-monitor*) (user-monitor-rename-fact um))
    (incf (user-monitor-delete-fact *user-monitor*) (user-monitor-delete-fact um))
    (incf (user-monitor-delete-positive-fact *user-monitor*) (user-monitor-delete-positive-fact um))
    (incf (user-monitor-delete-negative-fact *user-monitor*) (user-monitor-delete-negative-fact um))
    (incf (user-monitor-delete-example *user-monitor*) (user-monitor-delete-example um))
    (incf (user-monitor-retract-new-facts *user-monitor*) (user-monitor-retract-new-facts um))
    (incf (user-monitor-text-edit-fact *user-monitor*) (user-monitor-text-edit-fact um))
    (incf (user-monitor-add-english *user-monitor*) (user-monitor-add-english um))
    (incf (user-monitor-change-english *user-monitor*) (user-monitor-change-english um))
    (incf (user-monitor-delete-english *user-monitor*) (user-monitor-delete-english um))
    (incf (user-monitor-change-variable-type *user-monitor*) (user-monitor-change-variable-type um))
    (incf (user-monitor-show-type *user-monitor*) (user-monitor-show-type um))
    (incf (user-monitor-show-all-types *user-monitor*) (user-monitor-show-all-types um))
    (incf (user-monitor-menu-new-type *user-monitor*) (user-monitor-menu-new-type um))
    (incf (user-monitor-menu-delete-type *user-monitor*) (user-monitor-menu-delete-type um))
    (incf (user-monitor-add-to-type *user-monitor*) (user-monitor-add-to-type um))
    (incf (user-monitor-delete-from-type *user-monitor*) (user-monitor-delete-from-type um))
    (incf (user-monitor-run-on-new-example *user-monitor*) (user-monitor-run-on-new-example um))
    (incf (user-monitor-run-on-old-example *user-monitor*) (user-monitor-run-on-old-example um))
    (incf (user-monitor-test-on-all-examples *user-monitor*) (user-monitor-test-on-all-examples um))
    (incf (user-monitor-prove-goal-interactive *user-monitor*) (user-monitor-prove-goal-interactive um))
    (incf (user-monitor-prove-goal-batch *user-monitor*) (user-monitor-prove-goal-batch um))
    (incf (user-monitor-find-all-proofs *user-monitor*) (user-monitor-find-all-proofs um))
    (incf (user-monitor-who-calls-menu *user-monitor*) (user-monitor-who-calls-menu um))
    (incf (user-monitor-who-calls-relation-window *user-monitor*) (user-monitor-who-calls-relation-window um))
    (incf (user-monitor-check-rules *user-monitor*) (user-monitor-check-rules um))
    (incf (user-monitor-edit-rule-warnings *user-monitor*) (user-monitor-edit-rule-warnings um))
    (incf (user-monitor-explain-last-proof *user-monitor*) (user-monitor-explain-last-proof um))
    (incf (user-monitor-change-top-level-predicate *user-monitor*) (user-monitor-change-top-level-predicate um))
    (incf (user-monitor-trace-all *user-monitor*) (user-monitor-trace-all um))
    (incf (user-monitor-untrace-all *user-monitor*) (user-monitor-untrace-all um))
    (incf (user-monitor-add-trace *user-monitor*) (user-monitor-add-trace um))
    (incf (user-monitor-remove-trace *user-monitor*) (user-monitor-remove-trace um))
    (incf (user-monitor-add-spy *user-monitor*) (user-monitor-add-spy um))
    (incf (user-monitor-remove-spy *user-monitor*) (user-monitor-remove-spy um))
    (incf (user-monitor-display-top-level-examples *user-monitor*) (user-monitor-display-top-level-examples um))
    (incf (user-monitor-new-template *user-monitor*) (user-monitor-new-template um))
    (incf (user-monitor-edit-template *user-monitor*) (user-monitor-edit-template um))
    (incf (user-monitor-delete-template *user-monitor*) (user-monitor-delete-template um))
    (incf (user-monitor-new-extensional-relation *user-monitor*) (user-monitor-new-extensional-relation um))
    (incf (user-monitor-new-builtin-relation *user-monitor*) (user-monitor-new-builtin-relation um))
    (incf (user-monitor-new-intensional-relation *user-monitor*) (user-monitor-new-intensional-relation um))
    (incf (user-monitor-edit-relation *user-monitor*) (user-monitor-edit-relation um))
    (incf (user-monitor-rename-relation *user-monitor*) (user-monitor-rename-relation um))
    (incf (user-monitor-delete-relation *user-monitor*) (user-monitor-delete-relation um))
    (incf (user-monitor-who-calls-relation *user-monitor*) (user-monitor-who-calls-relation um))
    (incf (user-monitor-display-relation *user-monitor*) (user-monitor-display-relation um))
    (incf (user-monitor-new-type *user-monitor*) (user-monitor-new-type um))
    (incf (user-monitor-edit-type *user-monitor*) (user-monitor-edit-type um))
    (incf (user-monitor-delete-type *user-monitor*) (user-monitor-delete-type um))
    (incf (user-monitor-new-cliche *user-monitor*) (user-monitor-new-cliche um))
    (incf (user-monitor-edit-cliche *user-monitor*) (user-monitor-edit-cliche um))
    (incf (user-monitor-delete-cliche *user-monitor*) (user-monitor-delete-cliche um))
    (incf (user-monitor-use-cliche *user-monitor*) (user-monitor-use-cliche um))
    (incf (user-monitor-dont-use-cliche *user-monitor*) (user-monitor-dont-use-cliche um))
    (incf (user-monitor-display-goal-concept *user-monitor*) (user-monitor-display-goal-concept um))
    (incf (user-monitor-display-domain-theory *user-monitor*) (user-monitor-display-domain-theory um))
    (incf (user-monitor-display-learned-description *user-monitor*) (user-monitor-display-learned-description um))
    (incf (user-monitor-window-setup *user-monitor*) (user-monitor-window-setup um))
    (incf (user-monitor-default-setup *user-monitor*) (user-monitor-default-setup um))
    (incf (user-monitor-change-learning-parameters *user-monitor*) (user-monitor-change-learning-parameters um))
    (incf (user-monitor-change-builtin-flags *user-monitor*) (user-monitor-change-builtin-flags um))
    (incf (user-monitor-analyze-coverage *user-monitor*) (user-monitor-analyze-coverage um))
    (incf (user-monitor-learn *user-monitor*) (user-monitor-learn um))
    (incf (user-monitor-revise *user-monitor*) (user-monitor-revise um))
    (incf (user-monitor-edit-translation-define *user-monitor*) (user-monitor-edit-translation-define um))
    (incf (user-monitor-edit-translation-cancel *user-monitor*) (user-monitor-edit-translation-cancel um))
    (incf (user-monitor-example-window-display-using-template *user-monitor*) (user-monitor-example-window-display-using-template um))
    (incf (user-monitor-example-window-display-without-template *user-monitor*) (user-monitor-example-window-display-without-template um))
    (incf (user-monitor-example-window-define-from-template *user-monitor*) (user-monitor-example-window-define-from-template um))
    (incf (user-monitor-example-window-define *user-monitor*) (user-monitor-example-window-define um))
    (incf (user-monitor-example-window-delete-column *user-monitor*) (user-monitor-example-window-delete-column um))
    (incf (user-monitor-example-window-delete-row *user-monitor*) (user-monitor-example-window-delete-row um))
    (incf (user-monitor-example-window-delete-var *user-monitor*) (user-monitor-example-window-delete-var um))
    (incf (user-monitor-example-window-delete-type *user-monitor*) (user-monitor-example-window-delete-type um))
    (incf (user-monitor-example-window-delete-mode *user-monitor*) (user-monitor-example-window-delete-mode um))
    (incf (user-monitor-example-window-delete-datum *user-monitor*) (user-monitor-example-window-delete-datum um))
    (incf (user-monitor-example-window-new-column *user-monitor*) (user-monitor-example-window-new-column um))
    (incf (user-monitor-example-window-new-row *user-monitor*) (user-monitor-example-window-new-row um))
    (incf (user-monitor-example-window-new-var *user-monitor*) (user-monitor-example-window-new-var um))
    (incf (user-monitor-example-window-new-type *user-monitor*) (user-monitor-example-window-new-type um))
    (incf (user-monitor-example-window-new-mode *user-monitor*) (user-monitor-example-window-new-mode um))
    (incf (user-monitor-example-window-new-datum *user-monitor*) (user-monitor-example-window-new-datum um))
    (incf (user-monitor-rule-editor-edit-definition *user-monitor*) (user-monitor-rule-editor-edit-definition um))
    (incf (user-monitor-rule-editor-create-literal *user-monitor*) (user-monitor-rule-editor-create-literal um))
    (incf (user-monitor-rule-editor-copy *user-monitor*) (user-monitor-rule-editor-copy um))
    (incf (user-monitor-rule-editor-negate *user-monitor*) (user-monitor-rule-editor-negate um))
    (incf (user-monitor-rule-editor-attach *user-monitor*) (user-monitor-rule-editor-attach um))
    (incf (user-monitor-rule-editor-replace *user-monitor*) (user-monitor-rule-editor-replace um))
    (incf (user-monitor-rule-editor-delete *user-monitor*) (user-monitor-rule-editor-delete um))
    (incf (user-monitor-rule-editor-show *user-monitor*) (user-monitor-rule-editor-show um))
    (incf (user-monitor-rule-editor-hide *user-monitor*) (user-monitor-rule-editor-hide um))
    (incf (user-monitor-rule-editor-analyze *user-monitor*) (user-monitor-rule-editor-analyze um))
    (incf (user-monitor-rule-editor-cancel *user-monitor*) (user-monitor-rule-editor-cancel um))
    (incf (user-monitor-rule-editor-define *user-monitor*) (user-monitor-rule-editor-define um))
    (setf (user-monitor-file-names *user-monitor*) (nconc (user-monitor-file-names *user-monitor*) (user-monitor-file-names um)))
    (setf (user-monitor-times *user-monitor*) (nconc (user-monitor-times *user-monitor*) (user-monitor-times um)))))
  (values))

#|
(defun get-time ()
  (rlet ((sec :integer))
    (#_GetDateTime sec)
    (%get-unsigned-long sec)))
|#

(defun salvage ()
  (let ((kb-file *kb-file*)
        (facts-changed *facts-changed*)
        (rules-changed *rules-changed*))
    (dump-kb-to-file  "home:SALVAGED-KB.LISP")
    (setf *kb-file* kb-file
          *facts-changed* facts-changed 
          *rules-changed* rules-changed)
    (mapc #'menu-enable (menubar))
    (message-dialog (format  nil "SALVAGE SUCCESSFUL...~%    Knowledge Base stored as SALVAGED-KB.LISP~%    in the directory containing FOCL-1-2-3 v1.2.") :size (make-point 400 100) :position :centered)))