;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk

(in-package :user)

(defvar *initially-bound-frontier-vars* nil)


;;;__________________________________
;;;  REDEFINE-PROLOG-FUNCTIONS-OF-RULES-ALLOWING-DELETION

(defun redefine-prolog-functions-of-rules-allowing-deletion ()
  (let ((*compile-allowing-deletions* t))
    (dolist (r *r-structs*)
      (when (rule-p r)
        (format t "~%;; Redefining prolog function for ~A" (r-name r))
        (setf (r-prolog-function r) (focl-create-prolog-function (r-name r) (r-arity r) (get-clauses (r-name r))))))))

#|
;;;__________________________________
;;;   

(defun return-best-frontier (graph input-vars input-type input-pos input-neg source state-value)
  (let ((best-evaluation (evaluate-frontier graph state-value input-pos input-neg source)))
    (when (and *trace-learning?* (member :o *focl-trace-level*))
      (format t "~%Frontier   : ~S" (convert-graph-to-prolog graph))
      (format t "~%Evaluation : ~S~%" best-evaluation))
    (multiple-value-setq (graph best-evaluation)
      (find-best-frontier graph input-vars input-type input-pos input-neg source state-value best-evaluation))
    (when *EXPAND-BEST-FRONTIER*
      (let (frontier)
        (do ((continue t))
            ((not continue))
          (when (and *trace-learning?* (member :o *focl-trace-level*))
            (format t "~%~%Expanding Best Frontier : ~A" (convert-tree-to-prolog (graph-base graph))))
          (multiple-value-setq (graph frontier continue) (expand-frontier graph))
          (when (and *trace-learning?* (member :o *focl-trace-level*))
            (format t "~%Expanded Best Frontier : ~A" (convert-tree-to-prolog (graph-base graph)))
            (format t "~%Evaluation : ~A~%" best-evaluation))
          (when continue
            (multiple-value-setq (graph best-evaluation continue)
              (find-best-frontier graph input-vars input-type input-pos input-neg source state-value best-evaluation)))
          (unless continue
            (unexpand-frontier graph frontier)))))
    (values graph best-evaluation)))

;;;__________________________________
;;;  EXPAND-FRONTIER

(defun expand-frontier (graph)
  (let ((frontier (selected-nodes graph))
        (antecedents nil)
        (something-expanded nil))
    (dolist (n frontier)
      (when (setq antecedents (get-antecedents-quick n graph))
        (setq something-expanded t)
        (deselect-node n)
        (select-node antecedents)))
    (values graph frontier something-expanded)))

;;;__________________________________
;;;  UNEXPAND-FRONTIER

(defun unexpand-frontier (graph frontier)
  (deselect-node (graph-root graph) t)
  (select-node frontier))
|#

;;;__________________________________
;;; RETURN-BEST-FRONTIER

(defun return-best-frontier (graph input-vars input-type input-pos input-neg source state-value)
  (let ((best-evaluation (evaluate-frontier graph state-value input-pos input-neg source)))
    (when (and *trace-learning?* (member :o *focl-trace-level*))
      (format t "~%Frontier   : ~S" (convert-graph-to-prolog graph))
      (format t "~%Evaluation : ~S~%" best-evaluation))
    (multiple-value-setq (graph best-evaluation)
      (find-best-frontier graph input-vars input-type input-pos input-neg source state-value best-evaluation))
    (when *EXPAND-BEST-FRONTIER*
      (do ((continue t))
          ((not continue))
        (when (and *trace-learning?* (member :o *focl-trace-level*))
          (format t "~%~%Expanding Best Frontier : ~A" (convert-tree-to-prolog (graph-base graph))))
        (multiple-value-setq (graph continue) (expand-frontier graph))
        (when (and *trace-learning?* (member :o *focl-trace-level*))
          (format t "~%Expanded Best Frontier : ~A" (convert-tree-to-prolog (graph-base graph)))
          (format t "~%Evaluation : ~A~%" best-evaluation))
        (when continue
          (setq *expanding-best-frontier* t)
          (multiple-value-setq (graph best-evaluation continue)
            (find-best-frontier graph input-vars input-type input-pos input-neg source state-value best-evaluation))
          (setq *expanding-best-frontier* nil))
        (unexpand-frontier graph)))
    (values graph best-evaluation)))

(defvar *expanded-nodes* nil)

;;;__________________________________
;;;  EXPAND-FRONTIER

(defun expand-frontier (graph)
  (setq *expanded-nodes* (selected-nodes graph t))
  (let ((antecedents nil)
        (something-expanded nil))
    (dolist (n *expanded-nodes*)
      (when (setq antecedents (get-antecedents-quick n graph))
        (setq something-expanded t)
        (deselect-node n)
        (select-node antecedents)))
    (values graph something-expanded)))

;;;__________________________________
;;;  UNEXPAND-FRONTIER

(defun unexpand-frontier (graph)
  (let ((something-expanded nil))
    (dolist (node *expanded-nodes*)
      (if (some #'(lambda (conjunction)
                       (some #'(lambda (node)
                                 (or (not (node-selected? node))
                                     (node-deleted? node)))
                             conjunction))
                   (node-antecedents node))
        (setq something-expanded t)
        (progn
          (deselect-node node t)
          (select-node node))))
    (setq *expanded-nodes* nil)
    (values graph something-expanded)))


;;;__________________________________
;;;  FIND-BEST-FRONTIER

(defun find-best-frontier (graph input-vars input-type input-pos input-neg source state-value best-evaluation)
  (declare (ignore input-vars input-type))
  (let ((modified? nil))
    (unless (and *operationalize-only-when-inital-frontier-has-positive-gain*
                 (not (evaluation-good-enough-to-continue best-evaluation)))
      (do ((terminate nil)
           (best-operator nil nil)
           (best-node nil nil)
           (best-modification nil nil))
          (terminate)
        (multiple-value-setq (best-operator best-node best-modification best-evaluation)
          (return-best-operator graph state-value input-pos input-neg source best-operator best-node best-modification best-evaluation))
        (cond ((and best-operator (evaluation-good-enough-to-continue best-evaluation))
               (funcall (symbol-function best-operator) graph best-node best-modification :apply)
               (setq modified? t
                     *expanding-best-frontier* nil)
               (when (and *trace-learning?* (member :o *focl-trace-level*))
                 (format t "~%~%Selected Frontier : ~S" (convert-graph-to-prolog graph))
                 (format t "~%Evaluation :           ~S~%" best-evaluation))
               (when (and *display-learning?* (eq source :ebl) (window-open? *EBL-WINDOW*))
                 (display-gain-caption *EBL-WINDOW* :source "Frontier" :gain best-evaluation)
                 (dolist (node (graph-used-nodes graph)) (when (node-selected? node) (setf (node-state node) source)))))
              (t
               (setf terminate t)))))
    (values graph best-evaluation modified?)))


;;;__________________________________
;;;  RETURN-BEST-OPERATOR

(defun return-best-operator (graph state-value input-pos input-neg source best-operator best-node best-modification best-evaluation)
  (let (evaluation)
    (do ((node (next-selected-node (graph-base graph) :include-self t) (next-selected-node node)))
        ((null node))
      (dolist (operator *ACTIVE-FRONTIER-OPERATORS*)
        (do ((modification (funcall (symbol-function operator) graph node nil :next) (funcall (symbol-function operator) graph node modification :next)))
            ((null modification))
          (setf evaluation (evaluate-frontier graph state-value input-pos input-neg source))
          (when (and *trace-learning?* (member :o *focl-trace-level*))
            (format t "~%Frontier   : ~S" (convert-graph-to-prolog graph))
            (format t "~%Evaluation : ~S~%" evaluation))
          (when (evaluation-operator< best-evaluation (real-operator best-operator best-modification) evaluation (real-operator operator modification))
            (setf best-operator operator
                  best-node node
                  best-modification modification
                  best-evaluation evaluation)))))
    (values best-operator best-node best-modification best-evaluation)))

;;;__________________________________
;;; SET-DELETED-CLAUSES-AND-LITERALS-FROM-TREE
;;;
;;;  Traverse sub-tree below node setting the hash table *deleted-clauses-and-literals*
;;;  based on the selected nodes (nodes that are not selected are considered to be
;;;  deleted.
;;;
;;;  Returns:
;;;   t      - node or some antecedent of node is selected
;;;   nil    - neither node nor any antecedent of node is selected
;;;   :fail  - encountered a frontier configuration that can not be emulated using
;;;            the hash table approach because some immediate antecedent of an
;;;            AND, OR, or NOT was deleted.
;;;
;;; Note: There might be a problem when multiple uses of the same relation appear in a frontier together.
;;;       It is NOT the case that one over writes the other.  Deletions will be additive from both uses,
;;;       a literal must be undeleted in every use for it to appear in the definition. 

(defun set-deleted-clauses-and-literals-from-tree (node)
  (when (node-p node)
    (cond ((node-deleted? node) nil)
          ((node-selected? node) t)
          ((next-selected-node-below node)
           (let* ((name (r-name (node-r-struct node)))
                  (some-literal-deleted? nil)
                  (every-clause-deleted? t)
                  (every-literal-deleted? t))
             (do ((conjunctions (node-antecedents node) (rest conjunctions))
                  (clause-index 0 (incf clause-index)))
                 ((null conjunctions))
               (setq every-literal-deleted? t)
               (do ((nodes (first conjunctions) (rest nodes))
                    (literal-index 0 (incf literal-index)))
                   ((null nodes))
                 (cond ((set-deleted-clauses-and-literals-from-tree (first nodes))
                        (setq every-literal-deleted? nil))
                       (t
                        (setq some-literal-deleted? t)
                        (delete-literal name clause-index literal-index))))
               (if every-literal-deleted?
                 (delete-clause name clause-index)
                 (setq every-clause-deleted? nil)))
             (if (and some-literal-deleted?
                      (not every-clause-deleted?)
                      (node-second-order? node))
               (throw :fail :failed)
               (not every-clause-deleted?)))))))

;;;__________________________________
;;; EVALUATE-FRONTIER

(defun evaluate-frontier (graph current-state-value pos-tuples neg-tuples source)
  (let* ((base (graph-base graph))
         (r-struct (node-r-struct base))
         (variablization (node-vars base))
         (types (r-type r-struct))
         (function (r-prolog-function r-struct))
         (frontier nil)
         (gain :fail)
         (valid? :fail))
    (unless *MULTIPLE-CALLS*
      (clear-clause-and-literal-deletions)
      (setq valid? (catch :fail (set-deleted-clauses-and-literals-from-tree base))))
    (when (eql valid? :fail)
      (clear-clause-and-literal-deletions)
      (setq frontier (convert-tree-to-prolog base))
      (if frontier
        (let ((*compile-allowing-deletions* nil))
          (setq function (focl-compile-clause-function (list (cons 'dummy variablization) frontier) (length variablization))
                valid? t))
        (setq valid? nil)))
    (when valid?
      (when (and *trace-learning?* (member :odt *focl-trace-level*))
        (format t "~%~%Deletion Hash Table ____________________________")
        (maphash #'(lambda (key value) (format t "~%~40A   ~A" key value)) *deleted-clauses-and-literals*)
        (format t "~%_________________________________________________"))
      (setq gain (info-gain-prove (or frontier graph) function current-state-value pos-tuples neg-tuples variablization types))
      (when (and *display-learning?* (eq source :ebl) (window-open? *EBL-WINDOW*))
        (display-gain-caption *EBL-WINDOW* :source "Frontier" :gain gain)))
    gain))

#|(defun evaluate-frontier (graph state-value input-pos input-neg source)
  (let* ((base (graph-base graph))
         (variables (node-vars base))
         (frontier (convert-tree-to-prolog base))
         (gain :fail))
    (when frontier
      (let* ((function (focl-compile-clause-function (list (cons 'dummy variables) frontier) (length variables)))
             (p (count-prove function variables input-pos))
             (n (count-prove function variables input-neg)))
        (setf gain (gain-metric state-value p p n n))
        
        (when (and *trace-learning?* (member :o *focl-trace-level*))
          (format t "~% ~a~% ~40Tpos: ~a:~40T~a/~a~49Tneg: ~a:~60T~a/~a ~67T gain: ~4f"
                  frontier
                  p p (length input-pos)
                  n n (length input-neg) (gain-gain gain)))

        (when (and *display-learning?* (eq source :ebl) (window-open? *EBL-WINDOW*))
          (display-gain-caption *EBL-WINDOW* :source "Frontier" :gain gain))))
    gain))
|#

;;;__________________________________
;;; EVALUATION-GOOD-ENOUGH-TO-CONTINUE

(defun evaluation-good-enough-to-continue (evaluation)
  (and (gain-p evaluation)
       (> (gain-pp evaluation) 0)
       (> (gain-gain evaluation) 0)))

;;;__________________________________
;;; EVALUATION-OPERATOR<

(defun evaluation-operator< (gain-1 operator-1 gain-2 operator-2)
  (cond
   ((and (null operator-1) (eql operator-2 'FRONTIER-OP_leaf_operationalization)) t)                ;;;  <-- HACK to perfrom leaf operationalization
   (t (cond
       ((not (gain-p gain-1)) t)                                                                    ;;;  1st criterion existance of gain
       ((not (gain-p gain-2)) nil)
       (t (let ((g1 (gain-gain gain-1))                                                             ;;;  2nd criterion value of gain
                (g2 (gain-gain gain-2)))
            (cond 
             ((< g1 g2) t)                           
             ((> g1 g2) nil)
             (t (let ((p1 (gain-pp gain-1))                                                         ;;;  3rd criterion coverage of positive tuples
                      (p2 (gain-pp gain-2)))
                  (cond 
                   ((< p1 p2) t)
                   ((> p1 p2) nil)
                   (t (let ((op1 (operator-prefered? operator-1))                                   ;;;  4th criterion operator preference             
                            (op2 (operator-prefered? operator-2)))
                        (cond
                         ((and (not op1) op2) t)
                         ((and op1 (not op2)) nil)
                         (t (let ((oo1 (or (position operator-1 *ACTIVE-FRONTIER-OPERATORS*)        ;;;  5th criterion operator order (implicit preference)
                                           (position operator-1 *ACTIVE-FAST-FRONTIER-OPERATORS*)))
                                  (oo2 (or (position operator-2 *ACTIVE-FRONTIER-OPERATORS*)
                                           (position operator-2 *ACTIVE-FAST-FRONTIER-OPERATORS*))))
                              (cond 
                               ((not (numberp oo2)) t)    ;; this means operator does not appear in *ACTIVE-FRONTIER-OPERATORS* which implies frontier is the default
                               ((not (numberp oo1)) nil)  ;; only explicitly prefered operators can unseat a default with equal gain and coverage.
                               ((> oo1 oo2) t)
                               ((< oo1 oo2) nil)
                               (t nil)))))))))))))))))                                              ;;;  6th criterion is application order (earlier application wins)
                                                                                                    ;;; [XXXX] maybe this last one should be (= 0 (random 2))

;;;__________________________________
;;; NEXT-SELECTED-NODE

(defun next-selected-node (node &key (include-self nil) (upper-bound nil))
  (when (node-p node)
    (or (next-selected-node-below (if include-self node (node-antecedents node)))
        (next-selected-node-above node upper-bound))))

;;;__________________________________
;;; NEXT-SELECTED-NODE-BELOW

(defun next-selected-node-below (node)
  (cond ((node-p node)
         (unless (node-deleted? node)
           (if (node-selected? node)
             node
             (next-selected-node-below (node-antecedents node)))))
        ((consp node)
         (or (next-selected-node-below (first node))
             (next-selected-node-below (rest node))))))

;;;__________________________________
;;; NEXT-SELECTED-NODE-ABOVE

(defun next-selected-node-above (node upper-bound)
  (when (node-p node)
    (unless (eq node upper-bound)
      (let ((consequent (node-consequent node)))
        (when consequent
          (let* ((disjunction (node-antecedents consequent))
                 (conjunction (find-if #'(lambda (c) (member node c)) disjunction)))
            (or (next-selected-node-below (rest (member node conjunction)))
                (next-selected-node-below (rest (member conjunction disjunction)))
                (next-selected-node-above consequent upper-bound))))))))


;;;;==============================================================================================================
;;;;  FRONTIER INDUCTION CODE
;;;;==============================================================================================================

;;;__________________________________
;;;  OP-FRONTIER

(defun op-frontier (graph input-vars input-type input-pos input-neg source state-value)
  (setq *initially-bound-frontier-vars* input-vars)    ;; quick hack to pass bound-vars into frontier-ops 
  (multiple-value-bind (graph best-evaluation) (return-best-frontier graph input-vars input-type input-pos input-neg source state-value)
    (clear-clause-and-literal-deletions)
    (let ((frontier-graph (extract-frontier graph *FRONTIER-INDUCTION*)))
      (when frontier-graph
        (let ((base (graph-base frontier-graph)))
          (if *FRONTIER-INDUCTION*
            (inductively-patch-graph frontier-graph input-pos input-neg input-vars input-type :FRONTIER-EBL)
            (insert-node-tuples base input-pos input-neg input-vars input-type :FRONTIER-EBL nil t t))
          (let* ((literal (convert-nodes-to-literals base frontier-graph source))
                 (coverage (find :FRONTIER-EBL (node-coverage base) :key #'coverage-from))
                 (current-output-vars (coverage-output-vars coverage))
                 (output-vars (make-old-vars (length current-output-vars)))
                 (var-alist (mapcar #'cons current-output-vars output-vars))
                 (output-type (coverage-output-type coverage))
                 (output-pos (coverage-output-pos coverage))
                 (output-neg (coverage-output-neg coverage)))
            (do ((l literal (literal-next l)))
                ((null l))
              (literal-transfer-bindings l var-alist))
            (values literal output-vars output-type output-pos output-neg best-evaluation)))))))


;;;___________________________________________________________________________
;;;  CREATE-FRONTIER-GRAPH
;;;
;;;  Given a node within a graph this function creates a new graph containing
;;;  only the paths from the node that lead to selected nodes.

(defun create-frontier-graph (frontier-base expand-frontier-graph)
  (let* ((frontier-graph (create-graph))
         (frontier-root (graph-root frontier-graph)))
    (setf (node-antecedents frontier-root) (list (list (duplicate-node frontier-base frontier-root frontier-graph expand-frontier-graph t))))
    (if expand-frontier-graph
      (apply-function-to-tree 
       (graph-base frontier-graph)
       #'(lambda (node)
           (when (or (node-deleted? node)
                     (and (not (node-selected? node))
                          (null (next-selected-node-below node)))
                     (and (not (node-selected? node))
                          (node-consequent node)
                          (node-selected? (node-consequent node))))
             (disconnect-and-free-tree node frontier-graph))))
      (select-node (graph-base frontier-graph)))
    frontier-graph))

;;;___________________________________________________________________________
;;;  EXTRACT-FRONTIER
;;;
;;;  The selected nodes in the graph form a frontier.
;;;  This function finds the lowest common ancestor of all the nodes in the
;;;  frontier and defines any new rules needed to describe the frontier
;;;  then creates a single prolog literal that expresses the frontier.
;;;  Optionally a graph of the frontier can be created. This graph will be
;;;  useful for preforming frontier induction and for doing revision.

(defun extract-frontier (graph expand-frontier-graph)
  (let ((frontier-base (lowest-common-ancestor-of-frontier graph))
        (frontier-graph nil))
    (when frontier-base
      (dolist (node (graph-used-nodes graph))
        (setf (node-aux node) (if (node-selected? node) :selected nil)))
      (define-new-relations-for-frontier frontier-base)
      (deselect-node frontier-base t)
      (dolist (node (graph-used-nodes graph))   
        (when (eq (node-aux node) :selected) (select-node node)))
      (setq frontier-graph (create-frontier-graph frontier-base expand-frontier-graph))
      (dolist (node (graph-used-nodes graph))
        (when (r-p (node-aux node))
          (setf (node-r-struct node) (node-aux node)))
        (setf (node-aux node) nil))
      frontier-graph)))
;;;___________________________________________________________________________
;;;  CREATE-VAR-MAPPING-FROM-A-TO-B
;;;
;;;  Create a mapping that takes the variables in A to the variables or constants
;;;  in B.  If a consistant mapping can not be generated nil is returned

(defun create-var-mapping-from-a-to-b (a b &optional (mapping (list t)))
  (cond ((consp a) (when (consp b) (create-var-mapping-from-a-to-b (rest a) (rest b) (create-var-mapping-from-a-to-b (first a) (first b) mapping))))
        ((pcvar-p a)
         (let ((bucket (assoc a (rest mapping))))
           (cond (bucket
                  (when (equal (second bucket) b)
                    mapping))
                 (mapping
                  (setf (rest mapping) (cons (list a b) (rest mapping)))
                  mapping))))
        ((equal a b) mapping)))

;;;___________________________________________________________________________
;;;  CREATE-NEW-VAR-MAPPING-FROM-A-TO-B

(defun create-new-var-mapping-from-a-to-b (a b &optional (mapping (list t)))
  (cond ((consp a) (when (consp b) (create-new-var-mapping-from-a-to-b (rest a) (rest b) (create-new-var-mapping-from-a-to-b (first a) (first b) mapping))))
        ((and (new-var? a) (new-var? b))
         (let ((bucket (assoc a (rest mapping))))
           (cond (bucket
                  (when (equal (second bucket) b)
                    mapping))
                 (mapping
                  (setf (rest mapping) (cons (list a b) (rest mapping)))
                  mapping))))
        ((equal a b) mapping)))

;;;___________________________________________________________________________
;;;  LITERALS-MATCH

(defun literals-match (G-literal S-literal bindings)
  (when bindings
    (cond ((and (null G-literal) (null S-literal)) bindings)
          ((null G-literal) nil)
          ((null S-literal) nil)
          (t (let ((G-pred (first G-literal))
                   (G-args (rest G-literal))
                   (S-pred (first S-literal))
                   (S-args (rest S-literal)))
               (when (equal G-pred S-pred)
                 (setq bindings
                       (cond ((eql G-pred 'not) (conjunctions-match G-args S-args bindings))       ;;; There is a limitation to this function in that it is not
                             ((eql G-pred 'and) (conjunctions-match G-args S-args bindings))       ;;; able to backtrack into a second order relation to try
                             ((eql G-pred 'or) (conjunctions-match G-args S-args bindings))        ;;; alternate bindings within the second order relation.
                             ((eql G-pred 'bagof) (create-var-mapping-from-a-to-b G-args S-args bindings))
                             ((eql G-pred 'call) (create-var-mapping-from-a-to-b G-args S-args bindings))
                             ((eql G-pred 'find-proofs) (create-var-mapping-from-a-to-b G-args S-args bindings))
                             ((eql G-pred 'setof) (create-var-mapping-from-a-to-b G-args S-args bindings))
                             (t (create-var-mapping-from-a-to-b G-args S-args bindings))))))))))
             
;;;___________________________________________________________________________
;;;  CONJUNCTIONS-MATCH

(defun conjunctions-match (G-literals S-literals bindings)
  (when bindings
    (cond ((and (null G-literals) (null S-literals)) bindings)
          ((null G-literals) nil)
          ((null S-literals) nil)
          (t (let ((G-literal (first G-literals))
                   (new-bindings nil))
               (do* ((untried S-literals (rest untried))
                     (S-literal (first untried) (first untried)))
                    ((or (setq new-bindings (conjunctions-match (rest G-literals) (remove S-literal S-literals) (literals-match G-literal S-literal (copy-list bindings))))
                         (null untried))))
                 new-bindings)))))

;;;___________________________________________________________________________
;;;  CLAUSES-MATCH

(defun clauses-match (G-clauses S-bodies S-parameters)
  (cond ((and (null G-clauses) (null S-bodies)) t)
        ((null G-clauses) nil)
        ((null S-bodies) nil)
        (t (let* ((G-clause (first G-clauses))
                  (G-body (rest G-clause))
                  (G-args (rest (first G-clause)))
                  (match nil))
             (do* ((untried S-bodies (rest untried))
                   (S-body (first untried) (first untried)))
                  ((or (setq match (and (= (length G-body) (length S-body))
                                        (conjunctions-match G-body S-body (create-var-mapping-from-a-to-b G-args S-parameters))
                                        (clauses-match (rest G-clauses) (remove S-body S-bodies) S-parameters)))
                       (null untried))))
             match))))

;;;___________________________________________________________________________
;;;  FIND-OR-CREATE-DESIRED-RULE

(defun find-or-create-desired-rule (initial-r vars clause-bodies)
  (or (find-if #'(lambda (r) 
                   (and (rule-p r)
                        (= (length vars) (r-arity r))
                        (equal (r-type r) (r-type initial-r))
                        (let ((clauses (get-clauses (r-name r))))
                          (and (= (length clauses) (length clause-bodies))
                               (clauses-match clauses clause-bodies vars)))))
               *r-structs*)
      (let ((name (r-name initial-r)))
        (multiple-value-bind (new-vars new-clause-bodies) (variablize-rule name vars clause-bodies)
          (let* ((new-r-struct (copy-r initial-r))
                 (new-name (unique-r-name name))
                 (head (cons new-name new-vars)))
            (setf (r-name new-r-struct) new-name
                  (r-vars new-r-struct) new-vars
                  (r-clauses new-r-struct) (mapcar #'(lambda (body) (cons head body)) new-clause-bodies)
                  new-r-struct (def-from-r-struct new-r-struct))
            (add-r-struct new-r-struct)
            (setq *relations-defined-during-learning* (push new-r-struct *relations-defined-during-learning*))
            new-r-struct)))))

;;;___________________________________________________________________________
;;;  DEFINE-NEW-RELATIONS-FOR-FRONTIER
;;;
;;;  Given a node within a graph this function traverses the subtree rooted
;;;  at the node in post order redefining the rules needed to describe the
;;;  frontier (indicated by selected nodes)

(defun define-new-relations-for-frontier (frontier-base)
  (apply-function-to-tree
   frontier-base
   #'(lambda (node)
       (let ((r-struct (node-r-struct node))
             (antecedents (node-antecedents node)))
         (when (and antecedents
                    (some #'(lambda (c) (some #'node-selected? c)) antecedents))
           (select-node node)
           (when (rule-p r-struct)
             (let* ((clause-bodies (mapcan #'(lambda (conjunction)
                                               (let ((prolog (convert-tree-to-prolog conjunction)))
                                                 (cond ((null prolog) nil)
                                                       ((and (eql (first prolog) 'and)
                                                             (not (node-and? (first conjunction))))
                                                        (list (rest prolog)))
                                                       (t (list (list prolog))))))
                                           antecedents))
                    (new-r-struct (find-or-create-desired-rule r-struct (node-vars node) clause-bodies)))
               (setf (node-aux node) r-struct
                     (node-r-struct node) new-r-struct))
             (deselect-node antecedents t)))))
   t)
  )

;;;__________________________________
;;;  VARIABLIZE-RULE 
;;;
;;;  Change all constants in parameters to variables and apply the same mapping to each clause body.

(defun variablize-rule (name parameters clause-bodies)
  (if (every #'pcvar-p parameters)
    (values parameters clause-bodies)
    (let* ((r-struct (get-r-struct name))
           (types (when (r-p r-struct) (r-type r-struct)))
           (vars nil)
           (mapping nil))
      (dolist (p parameters)
        (if (pcvar-p p)
          (push p vars)
          (let ((var (make-unique-pcvar)))
            (push var vars)
            (push (list p var) mapping)))
        (setq types (rest types)))
      (values (nreverse vars) (direct-substitute-args clause-bodies mapping)))))

;;;_______________________________________
;;;  REDEFINE-INDUCTIVELY-PATCHED-RELATIONS

(defun redefine-inductively-patched-relations (graph)
  (let ((root (graph-root graph))
        (nodes-to-redefine (mapcan #'(lambda (node)
                                       (when (eq (node-state node) :induction)
                                         (list (node-real-consequent node))))
                                   (graph-used-nodes graph))))
    (deselect-node root t)
    (dolist (node (delete-duplicates nodes-to-redefine))
      (let ((r-struct (node-r-struct node))
            (antecedents (node-antecedents node)))
        (select-node antecedents)
        (let ((head (cons (r-name r-struct) (r-vars r-struct))))
          (setf (r-clauses r-struct) (mapcar #'(lambda (conjunction)
                                                 (let ((prolog (convert-tree-to-prolog conjunction)))
                                                   (if (and (eql (first prolog) 'and)
                                                            (not (node-and? (first conjunction))))
                                                     (rplaca prolog head)
                                                     (list head prolog))))
                                             antecedents)
                (node-r-struct node) (def-from-r-struct r-struct)))
        (deselect-node antecedents t)))))

;;;_______________________________________
;;; LITERAL-TRANSFER-BINDINGS

(defun literal-transfer-bindings (literal &optional (input-alist nil))
  (cond ((null literal) nil)
        ((literal-negated? literal)
         (do ((l (literal-negated-literals literal) (literal-next l)))
             ((null l))
           (literal-transfer-bindings l input-alist)))
        (t (setf (literal-variablization literal) (transfer-parameter-bindings (literal-variablization literal) input-alist)))))



;;;;==============================================================================================================
;;;;  FRONTIER INDUCTION CODE
;;;;==============================================================================================================

;;;_______________________________________
;;; INDUCTIVELY-PATCH-GRAPH

(defun inductively-patch-graph (graph input-pos input-neg input-vars input-type example-source)
  ;;(when (fboundp 'display-in-window) (display-in-window graph))   ;;; this is a mac hack
  (cond (*revise-theory*
         (break))
        (t
         (inductively-patch-and-or-graph graph input-pos input-neg input-vars input-type example-source)
         (deselect-node (graph-root graph) t)
         (dolist (node (graph-used-nodes graph))
           (when (null (node-antecedents node))
             (select-node node)))
         (define-new-relations-for-frontier (graph-base graph)))))

;;;_______________________________________
;;; INDUCTIVELY-PATCH-AND-OR-GRAPH
;;;
;;; This code is designed to run find-a-clause at all viable points in an and-or tree.
;;; Select the best modification make that modification interms of the evaluation metric.
;;; Redistribute the examples according to that mofication and repeat until no
;;; negative tuples are covered or no modification is an improvement according to the
;;; evaluation metric.
;;;
;;; Assumptions:
;;;  - graph base is an intensional relation (perhaps not yet defined).
;;;  - intermediate nodes are only second-order {and, or, not}.
;;;     (strictly speaking intermediate node can be intensional, however the xxx-and-or-graph function
;;;      will not modify the definitions of intermediate concepts) 
;;;  - after patching a new function for the graph base is created and evaluated.
;;;
;;; This function repairs the top level concept without affecting the rest of the
;;; knowledge base.  Later we will want something that does theory revision.

(defun inductively-patch-and-or-graph (graph input-pos input-neg input-vars input-type example-source)
  (let* ((root (graph-root graph))
         (evaluation 0)
         (done nil))
    (do () (done graph)
      (insert-node-tuples (first (node-antecedents root)) input-pos input-neg input-vars input-type example-source (first (graph-views graph)) t t nil)
      (deselect-node (graph-root graph) t)
      (let* ((coverage (find example-source (node-coverage (first (last (first (node-antecedents root))))) :key #'coverage-from))
             (p (length (coverage-output-pos coverage)))
             (n (length (coverage-output-neg coverage)))
             (state-value (current-metric p n))
             (best-evaluation (gain-metric state-value p p n n))
             (best-patch nil))
        (dolist (patch (nreverse (inductively-patch-every-conjunction (first (node-antecedents root)) example-source)))
          (when (> (gain-gain (setf evaluation (evaluate-inductive-patch-to-and-or-graph graph patch state-value example-source)))
                   (gain-gain best-evaluation))
            (setq best-patch patch
                  best-evaluation evaluation)))
        (if best-patch
          (apply-inductive-patch-to-and-or-graph graph best-patch)
          (setq done t))))))


;;;_______________________________________
;;; EVALUATE-INDUCTIVE-PATCH-TO-AND-OR-GRAPH

(defun evaluate-inductive-patch-to-and-or-graph (graph patch current-state-value example-source)
  (let ((root (graph-root graph))
        (coverage (find example-source (node-coverage (graph-base graph)) :key #'coverage-from))
        (added-nodes (apply-inductive-patch-to-and-or-graph graph patch)))
    (deselect-node root t)
    (dolist (node (graph-used-nodes graph))
      (unless (node-antecedents node) (setf (node-selected? node) t)))
    (let ((frontier (convert-tree-to-prolog root)))
      (dolist (node (graph-used-nodes graph))
        (setf (node-selected? node) nil))
      (select-node added-nodes)
      (let* ((input-vars (coverage-input-vars coverage))
             (length-vars (length input-vars))
             (index-vars (make-old-vars length-vars))
             (function (focl-compile-clause-function (list (cons 'dummy input-vars) frontier) length-vars))
             (gain (info-gain-prove frontier function current-state-value (coverage-input-pos coverage) (coverage-input-neg coverage) index-vars (coverage-input-type coverage))))
        (remove-inductive-patch-from-and-or-graph graph added-nodes)
        gain))))

;;;_______________________________________
;;; APPLY-INDUCTIVE-PATCH-TO-AND-OR-GRAPH

(defun apply-inductive-patch-to-and-or-graph (graph patch)
  (multiple-value-bind (node-vars-alist induction-vars induction-type literals) (values-list patch)
    (declare (ignore induction-type))
    (let* ((node (first (first node-vars-alist)))
           (node-induction-vars (second (first node-vars-alist)))
           (consequent (node-consequent node))
           (conjunction (conjunction-containing-node node))
           (new-induction-vars (compute-new-vars literals induction-vars))
           (mapping (if new-induction-vars
                      (direct-mapping (append induction-vars new-induction-vars)
                                      (append node-induction-vars (make-list-unique-vars (length new-induction-vars))))   ;;; perhaps we want to use something other than make-list-unique-vars here (nicer var-names, insure uniqueness)
                      (direct-mapping induction-vars node-induction-vars)))
           (prolog-patch (direct-substitute (convert-literals-to-prolog literals) mapping))
           (nodes (connect-clause graph consequent prolog-patch nil :INDUCTION :never 0)))
      (nconc conjunction nodes)   
      (when (fboundp 'fixup-views-of-graph) (fixup-views-of-graph graph))
      nodes)))

;;;_______________________________________
;;; REMOVE-INDUCTIVE-PATCH-FROM-AND-OR-GRAPH

(defun remove-inductive-patch-from-and-or-graph (graph nodes)
  (disconnect-and-free-tree nodes graph)
  (when (fboundp 'fixup-views-of-graph) (fixup-views-of-graph graph)))

;;;_______________________________________
;;; INDUCTIVELY-PATCH-EVERY-CONJUNCTION
;;;
;;;  fix to handle second order nodes correctly and should never have induction run at it
;;;  ors only if there are multiple disjuncts.

(defun inductively-patch-every-conjunction (nodes example-source &optional (patches nil))
  (cond ((null nodes) patches)
        ((node-p nodes)
         (select-node-only nodes)
         (let ((patch (inductively-patch nodes example-source)))
           (unless (eq patch :not-needed)
             (setf patches (inductively-patch-every-conjunction (node-antecedents nodes) example-source (push patch patches)))))
         (deselect-node nodes t))
        ((conjunction-p nodes)
         (select-node-only nodes)
         (let ((node (first (last nodes))))
           (if (and (null (rest nodes))
                    (member (node-kind node) '(:and :or))
                    (null (rest (node-antecedents node))))
             (setf patches (inductively-patch-every-conjunction (node-antecedents node) example-source patches))
             (let ((patch (inductively-patch node example-source)))
               (unless (eq patch :not-needed)
                 (setf patches (push patch patches))
                 (dolist (n nodes)
                   (setf patches (inductively-patch-every-conjunction (node-antecedents n) example-source patches)))))))
         (deselect-node nodes t))
        ((disjunction-p nodes)
         (dolist (conjunction nodes)
           (setf patches (inductively-patch-every-conjunction conjunction example-source patches)))))
  patches)

;;;_______________________________________
;;; INDUCTIVELY-PATCH
;;;
;;; INPUT
;;;   node            the last node of a conjunction in and and-or graph of a concept description which has an associated coverage structure, and
;;;   example-source  the source of tuples in the coverage structure.
;;;
;;;  OUTPUT
;;;   a list containing:
;;;     an alist associating a node with a list of variables in the clause containing node than correspond to the induction vars
;;;     a list of varaibles available for induction
;;;     a list of the type of each varaible available for induction
;;;     a linked list of literal structures that when added to the clause would prevent it from covering any negative tuples.

(defun inductively-patch (node example-source &optional (max-new-vars *max-new-variables*) (selection-function nil))
  (when (node-p node)
    (multiple-value-bind (induction-vars induction-types pos-tuples neg-tuples node-vars-alist) (collect-tuples-for-induction node example-source)
      (multiple-value-setq (pos-tuples neg-tuples) (swap-pos-and-neg-if-node-negated node pos-tuples neg-tuples))
      (if neg-tuples
        (let ((*pos-examples-num* (length pos-tuples))
              (*neg-examples-num* (length neg-tuples))
              (*display-learning?* nil)
              (*LEARNED-DESCRIPTION-WINDOW* nil)
              (*selection-function* (or selection-function *selection-function*)))
          (list node-vars-alist
                induction-vars
                induction-types
                (find-clause (get-pred *predicate-being-learned*) induction-vars induction-types nil pos-tuples neg-tuples nil max-new-vars nil)))
        :NOT-NEEDED))))

;;;_______________________________________
;;; NODE-NET-NEGATED?

(defun node-net-negated? (node)
  (do ((negated nil (if (node-not? consequent) (not negated) negated))
       (consequent (node-consequent node) (node-consequent consequent)))
      ((null consequent) negated)))

;;;_______________________________________
;;; SWAP-POS-AND-NEG-IF-NODE-NEGATED

(defun swap-pos-and-neg-if-node-negated (node pos neg)
  (if (node-net-negated? node)
    (values neg pos)
    (values pos neg)))

;;;_______________________________________
;;; COLLECT-TUPLES-FOR-INDUCTION
;;;
;;; INPUT
;;;   node            the last node of a conjunction in and and-or graph of a concept description which has an associated coverage structure, and
;;;   example-source  the source of tuples in the coverage structure.
;;;
;;;  OUTPUT
;;;   induction-vars   a list of vaiables whose id index into the tuple suitable for passing to find-clause
;;;   induction-types  a list of type associated with each induction variable
;;;   pos-tuples       a set of positive tuples 
;;;   neg-tuples       a set of negative tuples
;;;   node-arg-alist   an alist associating a node with a list of variables in the clause containing node that correspond to the induction vars

(defun collect-tuples-for-induction (node example-source)
  (when (node-p node)
    (let* ((graph (node-graph node))
           (consequent (node-real-consequent node))
           (r-struct (node-r-struct consequent))
           (consequents (and *frontier-induction-pool-tuples* (r-p r-struct) (remove-if-not #'(lambda (n) (member n (graph-used-nodes graph))) (r-nodes r-struct)))))
      (if (rest consequents)
        
        ;; Pool tuples - return tuples from every use of the clause
        
        (let ((induction-list nil)
              (node-arg-alist nil))
          (dolist (c consequents)
            (let* ((n (find-corresponding-node node consequent c))
                   (coverage (find example-source (node-coverage n) :key #'coverage-from))
                   (output-vars (coverage-output-vars coverage))
                   (output-pos (coverage-output-pos coverage))
                   (output-neg (coverage-output-neg coverage))
                   (consequent (node-real-consequent n))
                   (consequent-coverage (find example-source (node-coverage consequent) :key #'coverage-from))
                   (consequent-input-vars (coverage-input-vars consequent-coverage))
                   (consequent-input-type (coverage-input-type consequent-coverage)))
              (multiple-value-bind (induction-args induction-types) (induction-args-types node consequent consequent-input-vars consequent-input-type)
                (multiple-value-bind (pos-tuples neg-tuples induction-vars) (insert-constants output-pos output-neg output-vars induction-args)
                  (push (list induction-vars induction-types pos-tuples neg-tuples) induction-list)
                  (push (list n induction-args) node-arg-alist)))))
          (let ((induction-vars (first (first induction-list)))
                (induction-types (second (first induction-list)))
                (pos-tuples nil)
                (neg-tuples nil))
            (if (every #'(lambda (il) (equal (first il) induction-vars)) (rest induction-list))
              (setq pos-tuples (mapcan #'third induction-list)
                    neg-tuples (mapcan #'fourth induction-list))
              (setq pos-tuples (mapcan #'(lambda (iv) (reduce-tuples (first iv) (make-old-vars (length (first (third iv)))) (third iv))) induction-list)
                    neg-tuples (mapcan #'(lambda (iv) (reduce-tuples (first iv) (make-old-vars (length (first (fourth iv)))) (fourth iv))) induction-list)
                    induction-vars (make-old-vars (length induction-vars))))
            (values induction-vars induction-types pos-tuples neg-tuples node-arg-alist)))
        
        ;; Don't pool tuples - only return tuples from this use of the clause
        
        (let* ((coverage (find example-source (node-coverage node) :key #'coverage-from))
               (output-vars (coverage-output-vars coverage))
               (output-pos (coverage-output-pos coverage))
               (output-neg (coverage-output-neg coverage))
               (consequent (node-real-consequent node))
               (consequent-coverage (find example-source (node-coverage consequent) :key #'coverage-from))
               (consequent-input-vars (coverage-input-vars consequent-coverage))
               (consequent-input-type (coverage-input-type consequent-coverage)))
          (multiple-value-bind (induction-args induction-types) (induction-args-types node consequent consequent-input-vars consequent-input-type)
            (multiple-value-bind (pos-tuples neg-tuples induction-vars) (insert-constants output-pos output-neg output-vars induction-args)
              (values induction-vars induction-types pos-tuples neg-tuples (list (list node induction-args))))))))))

;;;_______________________________________
;;; CREATE-INDUCTION-VARS
;;;
;;; INPUT
;;;  tuples-vars   a list of the variables associated with each datum in a tuple   e.g. (?v1 ?v2 ?v3 ?v4) 
;;;  reduced-args  a subset of the tuple-vars possibly augmented with constants    e.g. (?v1 ?v4 5 ?v3)
;;;
;;; OUTPUT
;;;  a set of variables corresponding to reduced-args whose ids would correctly    e.g. (?0 ?3 ?4 ?2) 
;;;  index into the tuples            

(defun create-induction-vars (tuples-vars reduced-args)
  (let ((next-var-id (- (length tuples-vars) 1)))
    (mapcar #'(lambda (v) (if (variable-p v) (make-pcvar :id (position v tuples-vars)) (make-pcvar :id (incf next-var-id)))) reduced-args)))

;;;_______________________________________
;;; INSERT-CONSTANTS
;;;
;;; INPUT
;;;   pos-tuples    a list of tuples                                                  e.g. ((0 0 0 0) (0 0 0 1) (0 0 1 0) (0 0 1 1)) 
;;;   neg-tuples    a list of tuples                                                  e.g. ((1 0 0 0) (1 0 0 1) (1 0 1 0) (1 0 1 1)) 
;;;   tuples-vars   a list of the variables associated with each datum in the tuple   e.g. (?v1 ?v2 ?v3 ?v4) 
;;;   reduced-args  a subset of the tuple-vars possibly augmented with constants      e.g. (?v1 ?v4 5 ?v3)
;;;
;;; OUTPUT
;;;   a new set of pos-tuples augmented to contain any constants in reduced-args,     e.g. ((0 0 0 0 5) (0 0 0 1 5) (0 0 1 0 5) (0 0 1 1 5))
;;;   a new set of neg-tuples augmented to contain any constants in reduced-args,     e.g. ((1 0 0 0 5) (1 0 0 1 5) (1 0 1 0 5) (1 0 1 1 5))
;;;   a new set of variables whose ids correctly index into these tuples              e.g. (?0 ?3 ?4 ?2)     

(defun insert-constants (pos-tuples neg-tuples tuples-vars reduced-args)
  (if (some #'(lambda (v) (not (variable-p v))) reduced-args)
    (let ((extension (mapcan #'(lambda (v) (if (variable-p v) nil (list v))) reduced-args)))
      (values (mapcar #'(lambda (tuple) (append tuple extension)) pos-tuples)
              (mapcar #'(lambda (tuple) (append tuple extension)) neg-tuples)
              (create-induction-vars tuples-vars reduced-args)))
    (values pos-tuples neg-tuples (create-induction-vars tuples-vars reduced-args))))

;;;_______________________________________
;;; INDUCTION-ARGS-TYPES

(defun induction-args-types (node base bound-vars bound-types &optional (include-self t))
  (multiple-value-bind (args types) (filter-through-args-types base bound-vars bound-types t)
    (do* ((done nil (eq node a))
          (c base a)
          (a (antecedent-leading-to-node c node) (antecedent-leading-to-node c node)))
         (done)
      (do* ((ns (conjunction-containing-node a) (rest ns))
            (n (first ns) (first ns)))
           ((eq a n))
        (multiple-value-setq (args types) (add-to-args-types n args types nil)))
      (if (eq a node)
        (when include-self
          (multiple-value-setq (args types) (add-to-args-types a args types nil)))
        (multiple-value-setq (args types) (filter-through-args-types a args types t))))
    (values args types)))

;;;_______________________________________
;;; ADD-TO-ARGS-TYPES

(defun add-to-args-types (node args types include-constants)
  (when (node-p node)
    (case (node-kind node)
      (:and (dolist (n (first (node-antecedents node)))
              (multiple-value-setq (args types) (add-to-args-types n args types include-constants)))
            (values args types))
      ((:or :not) (values args types))
      (otherwise (let ((r-struct (node-r-struct node)))
                   (setq args (nreverse args)
                         types (nreverse types))
                   (do* ((n-vars (node-vars node) (rest n-vars))
                         (ts (when (r-p r-struct) (r-type r-struct)) (rest ts))
                         (type (first ts) (first ts)))
                        ((null n-vars))
                     (unless (or (eql type :goal)
                                 (eql type :goals)
                                 (eql type :expression))
                       (do* ((vs (first n-vars) (if (listp vs) (rest vs) nil))
                             (v (if (listp vs) (first vs) vs) (if (listp vs) (first vs) vs)))
                            ((null vs))
                         (when (and (not (member v args :test #'var-eq))
                                    (or (variable-p v)
                                        include-constants))
                           (push v args)
                           (push (or type :anything) types)))))
                   (values (nreverse args)(nreverse types)))))))

;;;_______________________________________
;;; FILTER-THROUGH-ARGS-TYPES

(defun filter-through-args-types (node args types include-constants)
  (when (node-p node)
    (case (node-kind node)
      ((:and :or :not) (values args types))
      (otherwise (let (new-args new-types position)
                   (do* ((as (node-vars node) (rest as))
                         (a (first as) (first as))
                         (ats (r-type (node-r-struct node)) (rest ats))
                         (at (or (first ats) :anything) (or (first ats) :anything)))
                        ((null as))
                     (cond ((variable-p a)
                            (when (setf position (position a args :test #'var-eq))
                              (push a new-args)
                              (push  (if (eq at :anything) (nth position types) at) new-types)))
                           (include-constants
                            (push a new-args) 
                            (push at new-types))))
                   (values (nreverse new-args) (nreverse new-types)))))))

