;;;; Copyright (c) 1990, 1991 by the University of California, Irvine. 
;;;; This program may be freely copied, used, or modified provided that this
;;;; copyright notice is included in each copy of this code.  This program
;;;; may not be sold or incorporated into another product to be sold without
;;;; written permission from the Regents of the University of California.
;;;; This program was written by Michael Pazzani, Cliff Brunk, Glenn Silverstein
;;;; and Kamal Ali.  

(in-package :user)

;;;_______________________________________________________________________________
;;;  FOCL
;;;
;;;  A relational learner which uses a separate and conquer approach guided by an 
;;;  information based heuristic to produce a concept description that covers all the 
;;;  positive examples and excludes all the negative examples.  Given a collection of 
;;;  classified examples of a concept, a set of extensionally defined predicates, and 
;;;  a possibly empty set of intensionally defined rules, one of may be distinguished 
;;;  as the goal concept, FOCL produces a Horn-clause description of the concept in 
;;;  terms of the extensional predicates.

(defun focl (pred-name &rest focl-parameters &key the-ignored-key &allow-other-keys)
  (declare (ignore the-ignored-key))
  (apply #'set-learning-parameters pred-name focl-parameters)
  (when (getf focl-parameters :trace) (setq *trace-learning?* t))
  (find-concept-description))

;;;_______________________________________________________________________________
;;;  SET-LEARNING-PARAMETERS

(defun set-learning-parameters
       (pred-name
        
        &key
        (clauses nil)
        (goal-concept-name (if clauses (intern (format nil "~a~a" (caaar clauses) '_rule)) nil))
        (use-goal-concept t)
        (max-new-variables 0)
        (overfit-ratio 0)
        (operationalize-intensional t)
        (extensional-induction t)
        (builtin-induction t)
        (intensional-induction nil)
        (trace nil)
        (display (if *user-interface-available*                 ;; defaults to display everything on mac when interface is available
                   '(:current-gain :best-gain :ebl :learned-description :work)
                   nil))
        (simplify-clauses t)
        (max-winners 1)
        (save-examples (if *user-interface-available* t nil))   ;; defaults to t on mac when interface is available
        (prefer-theory t)
        (prefer-deletions nil)
        (create-preds-from-cliches nil)
        (prefer-children nil)
        (proof-vars-available t)
        (noise-tolerance nil)               ;; :FOIL, :RECURSIVE, or NIL (cant handle noisy egs)
        (partial-dt-0-gain nil)             ;; Explore domain theory even when it has no gain
        (simplify-operationalizations nil)
        (try-all-conjunctions nil)          ;; Try all 2 literal conjunctions
        (batch-mode nil)                    ;; Suppress errors and just return them as strings
        (stop-when-all-pos-covered t)
        (builtin-threshold-only nil)
        (use-cliches t)
        (refinement :leaves)                ;; :leaves or :frontier
        (active-frontier-operators *active-frontier-operators*)
        (gain-function :information)        ;; :information, :ratio or :prob
        (always-try-cliches t)              ;; Try cliches at the same level as everything else
        (max-new-cliche-vars nil)
        (available-cliches *available-relational-cliches*)
        (enable-determinate-literals nil)
        (max-determinacy 1)
        (determinate-addition-bound 5)                      ;; These are the same maintained for compatability
        (max-determinate-depth determinate-addition-bound)  ;;
        (selection-function :maximum)       ;; :maximum or :probabilistic
        (cliches-can-have-negated-components? t)
        (reset-hash-tables t)
        (reset-statistics t)
        (define-rule-for-learned-description (if *user-interface-available* t nil))   ;; defaults to t on mac when interface is available
        (frontier-induction nil)
        (frontier-induction-pool-tuples nil)
        (frontier-simplification nil)
        (revise-theory nil)

        (literal-better-function #'greater-gain)
        (ebl-better-function #'greater-gain)
        (delete-better-function #'greater-gain-and-not-worse-accuracy)
        ;&allow-other-keys
        )
  
  (let* ((pred (get-pred pred-name))
         (arity (if pred (r-arity pred) 0))
         (type (if pred (r-type pred) nil)))
    
    (when clauses
      (eval `(def-rule ,goal-concept-name
               :type ,type
               :clauses ,(subst goal-concept-name (caaar clauses) clauses)
               :induction nil)))

    (setf *predicate-being-learned* pred-name
          *goal-concept* (when (get-rule goal-concept-name) (cons goal-concept-name (make-old-vars arity)))
          *use-goal-concept* (and *goal-concept* use-goal-concept)
          *max-new-variables* max-new-variables
          *overfit-ratio* overfit-ratio
          *operationalize-intensional* operationalize-intensional
          *extensional-induction* extensional-induction
          *builtin-induction* builtin-induction
          *intensional-induction* intensional-induction 
          *focl-trace-level* trace
          *focl-display-level* display
          *simplify-clauses* simplify-clauses
          *max-winners* max-winners 
          *save-examples* save-examples
          *theory-mode*  prefer-theory	
          *prefer-deletions*  prefer-deletions
          *create-preds-from-cliches* create-preds-from-cliches
          *prefer-children*  prefer-children
          *proof-vars-available* proof-vars-available
          *stopping-criteria-enabled* (case noise-tolerance
                                        ((:FOIL :RECURSIVE) noise-tolerance)
                                        (otherwise nil))
          *partial-dt-0-gain* partial-dt-0-gain
          *simplify-operationalizations* simplify-operationalizations
          *try-all-conjunctions* try-all-conjunctions
          *focl-batch-mode* batch-mode
          *stop-when-all-pos-covered* stop-when-all-pos-covered
          *builtin-threshold-only* builtin-threshold-only
          *use-relational-cliches* use-cliches
          *refinement* refinement
          *active-frontier-operators* active-frontier-operators
          *gain-function* gain-function
          *always-try-cliches* always-try-cliches
          *max-new-cliche-vars* (or max-new-cliche-vars max-new-variables)
          *available-relational-cliches* available-cliches
          *enable-determinate-literals* enable-determinate-literals
          *max-determinacy* max-determinacy
          *max-determinate-depth* max-determinate-depth
          *selection-function* selection-function
          *cliches-can-have-negated-components?* cliches-can-have-negated-components?
          *reset-hash-tables* reset-hash-tables
          *reset-statistics* reset-statistics
          *define-rule-for-learned-description* define-rule-for-learned-description
          *frontier-induction* frontier-induction
          *frontier-induction-pool-tuples* frontier-induction-pool-tuples
          *frontier-simplification* frontier-simplification
          *revise-theory* revise-theory

          *literal-better-function* literal-better-function
          *ebl-better-function* ebl-better-function    
          *delete-better-function* delete-better-function
 
          *remove-goal-concept-rule* (if clauses (first *goal-concept*))
          )))

;;;_______________________________________________________________________________
;;;  SET-METRIC-FUNCTIONS

(defun set-metric-functions ()
  (case *gain-function*
    (:information
     (setf *literal-better-function* #'greater-gain)
     (setf *ebl-better-function* #'greater-gain)
     (setf *delete-better-function* #'greater-gain-and-not-worse-accuracy))
    (:ratio
     (setf *literal-better-function* #'greater-ratio-then-coverage)
     (setf *ebl-better-function* #'greater-ratio-then-coverage)
     (setf *delete-better-function* #'greater-ratio-then-coverage))
    (:prob
     (setf *literal-better-function* #'greater-gain)
     (setf *ebl-better-function* #'greater-gain)
     (setf *delete-better-function* #'greater-gain-and-not-worse-accuracy))
    (otherwise nil))
  )

;;;_______________________________________________________________________________
;;;  FIND-CONCEPT-DESCRIPTION

(defun find-concept-description ()
  (let* ((pred (get-pred *predicate-being-learned*))
         (old-vars (make-old-vars (r-arity pred)))
         (old-mode (r-mode pred))
         (type (r-type pred))
         (pos-examples (r-pos pred))
         (neg-examples (r-neg pred))
         (uncovered-pos pos-examples)
         (covered-neg nil)
         (newly-covered-neg nil)
         (goal-rule (get-rule (first *goal-concept*)))
         (goal-concept-and-rule (if (rule-p goal-rule) (cons (list *goal-concept*) goal-rule)))
         (*batch* t)                 ;; don't ask questions about predicate values during learning
         (last-clause t))            ;; set to value of find-clause

    ;;; Pre-Learning Initialization
    
    (set-status :learning)
    (push-status :initiating)
    (set-metric-functions)
    (reset-last-new-var-id)
    (setf *relations-defined-during-learning* nil)
    (when *reset-hash-tables* (reset-hash-tables))
    (when *reset-statistics*
      (setf *variablizations-checked* 0
            *cliche-variabilizations-checked* 0)
      (clear-work *theory-work*))
    (setf *learned-description-head* (copy-r pred)
          (r-name *learned-description-head*) (unique-learned-description-name *learned-description-head*)
          (r-vars *learned-description-head*) old-vars
          *learned-description* nil
          *pos-examples-num* (length pos-examples)
          *neg-examples-num* (length neg-examples)
          *example-arity* (length (or (first pos-examples) (first neg-examples))))
    (when *stopping-criteria-enabled*
      (init-stopping-tables goal-concept-and-rule pos-examples neg-examples))
    
    (dispose-graph *ebl-graph*)
    (setf *ebl-graph* (create-graph :permanent? t))
    (when goal-concept-and-rule
      (set-graph-base *ebl-graph* (connect-literal *ebl-graph* (graph-root *ebl-graph*) *goal-concept* nil :ebl :never 0))
      )
    
    (clrhash *conjunction-function-hash*)
    (set-variable-depths-to-zero old-vars)
    
    ;;; This checks to see if there are multiple calls to the same rule in the goal concept
    ;;; if there are and *TREAT-ALL-CALLS-WITHIN-FRONTIER-INDEPENDENTLY* is t then we use the
    ;;; slow code - if either is not true then we recompile the rules to allow the deletion
    
    (when (eql *refinement* :FRONTIER)
      (setq *MULTIPLE-CALLS* (and *TREAT-ALL-CALLS-WITHIN-FRONTIER-INDEPENDENTLY* 
                                  (rule-makes-multiple-calls-to-same-rule? (first *goal-concept*)))))
    
    (when *display-learning?*
      (init-display-learning-windows)
      (display-description-coverage *pos-examples-num* *neg-examples-num* 0 0))
    
    ;;; Concept Formation Loop
    
    (do () 
        ((or (null uncovered-pos) (null last-clause)))
      (when (and *trace-learning?* (member :ct *focl-trace-level*))
        (format t "~%~%================================================================~%")
        (format t "Positive Tuples: ") (print uncovered-pos)
        (format t "~%Negative Tuples: ") (print neg-examples) (format t "~%"))
      (when *stopping-criteria-enabled*
        (set-clause-training-set-values-for-stopping-criteria uncovered-pos neg-examples))
      (setf last-clause (find-clause pred old-vars type
                                     goal-concept-and-rule
                                     uncovered-pos neg-examples covered-neg
                                     *max-new-variables* ':first-time))
      (when (and (listp last-clause) (eql (first last-clause) 'error)) ;; No gain error
        (return-from find-concept-description last-clause))
      (when last-clause
        (multiple-value-setq (last-clause uncovered-pos newly-covered-neg)
          (simplify-clause last-clause uncovered-pos neg-examples old-vars old-mode))
        (setf *learned-description* (nconc *learned-description* (list last-clause)))
        (dolist (neg newly-covered-neg) (setf covered-neg (pushnew neg covered-neg)))
        (when (and *trace-learning?* (member :c *focl-trace-level*))
          (format t "~%~%*********Simplified Clause: ~a" last-clause)
          (when (member :w *focl-trace-level*)
            (format t "~%                            ~A~%" (print-work nil *clause-work*))))
        (when *display-learning?*
          (display-description-coverage
           *pos-examples-num*  *neg-examples-num*
           (- *pos-examples-num* (length uncovered-pos)) (length covered-neg))
          (when (and *display-learning?* uncovered-pos (member :pause-after-clause *focl-display-level*))
            (pause))))
      )
    
    ;;; Post-Learning Cleanup
    
    (set-status :finished-learning)
    (when (and *trace-learning?* (member :w *focl-trace-level*))
      (format t "~%Theory Work [ ~A ]~%" (print-work nil *theory-work*)))
    (when *remove-goal-concept-rule* (delete-r-struct *remove-goal-concept-rule*))
    (when *define-rule-for-learned-description* (define-rule-for-learned-description))
    (when (and *print-relations-defined-during-learning*
               *relations-defined-during-learning*)
      (format t "~%~%Relations Defined During Learning~%~%")
      (dolist (r *relations-defined-during-learning*)
        (print-r-struct r t nil)
        (format t "~%")))
    *learned-description*))


;;;_______________________________________________________________________________
;;; FIND-CLAUSE

(defun find-clause (pred old-vars type
                    goal-concept-and-rule
                    pos-tuples neg-tuples previously-covered-neg
                    max-new-variables use-hash-tables)
  (let (winner
        (vars old-vars)            ;; vars grows, old-vars stays the same
        (continue t)               ;; nil if find-literal fails
        new-vars                   ;; additional variables,
        new-neg-tuples 
        new-pos-tuples
        new-type                   ;; additional type information (of new-vars)
        (first-literal nil)        ;; start of clause
        (new-literal nil)          ;; value of find-literal
        (last-literal nil)         ;; end of clause (place to add next literal)
        current-state-value
        (new-gain 'didnt-compute)
        (vars-added? t)
        (number-previously-covered-pos (- *pos-examples-num* (length pos-tuples))))

    (clear-work *clause-work*)
    
    (do ((*covered-all-pos-tuples*)) 
        ((or (null neg-tuples) (null continue))   ;; exit condition
         (and *enable-determinate-literals*
              (null vars-added?)
              (= new-gain 0)))   ;; can now exit even if neg-tuples arent 0 under certain cond.s
      (declare (special *covered-all-pos-tuples*))
      (setf *covered-all-pos-tuples* nil      
            continue nil
            current-state-value (CURRENT-METRIC (length pos-tuples) (length neg-tuples)))
      (when (setf winner (find-literal current-state-value
                                       first-literal pred vars type
                                       goal-concept-and-rule pos-tuples neg-tuples 
                                       max-new-variables old-vars use-hash-tables))
        (setf  continue t
               new-vars (winner-vars winner)
               new-type (winner-types winner)
               new-pos-tuples (winner-pos winner)
               new-literal (winner-literal winner)
               new-neg-tuples (winner-neg winner)
               new-gain (winner-gain winner))

        (when (null new-pos-tuples) (break "clause didn't cover any positive examples"))

        (cond
         (first-literal                                ;; make last literal point to new-literal
          (setf (literal-next last-literal) new-literal)
          (when *display-learning?*
            (let ((new-pos (winner-pp winner))
                  (new-neg (winner-nn winner)))
              (add-literal-to-last-clause-in-lcd-graph new-literal new-pos new-neg)
              (display-description-coverage
               *pos-examples-num* *neg-examples-num*
               (+ number-previously-covered-pos new-pos)
               (+ (length previously-covered-neg) new-neg)))
            ))
         (t                                            ;; make new literal first (if no first)
          (setf first-literal new-literal)
          (when *display-learning?*
            (let ((new-pos (winner-pp winner))
                  (new-neg (winner-nn winner)))
              (add-new-clause-to-lcd-graph new-literal new-pos new-neg)
              (display-description-coverage
               *pos-examples-num* *neg-examples-num*
               (+ number-previously-covered-pos new-pos)
               (+ (length previously-covered-neg) new-neg)))
            )))

        (setf (literal-prev new-literal) last-literal ;; last literal may be nil (if first clause) 
              last-literal (last-literal new-literal) ;; update last-literal
              pos-tuples new-pos-tuples
              neg-tuples new-neg-tuples  ;; after first literal, negative tuples may be reduced, so
              use-hash-tables nil        ;; can't use hash taples that count # of matching neg-tuples
              vars-added? (> (length new-vars) (length vars))
              vars (append vars new-vars)
              type (append type new-type))
        
        (when (eq :ebl (derivation-type (literal-derivation new-literal)))
          (setq goal-concept-and-rule nil))      ;; prevents operationalize of goal concept twice in a clause

        (when *trace-learning?*
          (when (member :l *focl-trace-level*)
            (format t "~%++++++New literal: ~a~%" new-literal)
            (when (member :w *focl-trace-level*)
              (format t "                   ~A~%" (print-work nil *literal-work*))))
          (when (member :ld *focl-trace-level*)
            (format t "pos-w-extensions: ~a neg-w-extensions: ~a new-gain: ~a bits-needed: ~5f~%"
                    (count-originals-extended (r-pos pred) pos-tuples)
                    (count-originals-extended (r-neg pred) neg-tuples)
                    new-gain
                    (if *stopping-criteria-enabled* 
                      (bits-to-encode-literal winner type)
                      nil)))
          )

        (discard-winner winner)
        
        (if *stopping-criteria-enabled*
          (update-clause-values-for-stopping-criteria new-literal pos-tuples type))
        
        (add-work *literal-work* *clause-work*)
        (add-work *literal-work* *theory-work*)
        
        (when *display-learning?*
          (display-work *literal-work* *clause-work* *theory-work*)
          (when (and (not *covered-all-pos-tuples*) (member :pause-after-literal *focl-display-level*))
            (pause)))
        ))

    (when *trace-learning?*
      (when (member :c *focl-trace-level*)
        (format t "~%~%*********New clause: ~a covers ~a pos, ~a neg" 
                first-literal (length pos-tuples) (length neg-tuples))))

    first-literal))

;;;_______________________________________________________________________________
;;; FIND-LITERAL

(defun find-literal
       (current-state-value    ;; I(Ti)
        current-clause         ;; linked list of literal-structs 
        pred                   ;; r-struct of predicate being learned
        vars                   ;; current set of bound variables
        type                   ;; type of bound vars
        goal-concept-and-rule           ;; goal concept (if any) to operationlize for EBL
        pos-tuples           
        neg-tuples 
        max-new-variables 
        old-vars               ;; original vars (for recursion detection)
        use-hash-tables)
  (clear-work *literal-work*)
  (when *display-learning?*
    (clear-gain-window *CURRENT-GAIN-WINDOW*)
    (clear-gain-window *BEST-GAIN-WINDOW*))
  (when (and *trace-learning?* (member :lt *focl-trace-level*) current-clause)
    (format t "~%~%________________________________________________________________~%")
    (format t "Positive Tuples: ")
    (print pos-tuples)
    (format t "~%Negative Tuples: ")
    (print neg-tuples))
  (let ((winners (make-winners :all-winners nil :new-winners nil)))
    (setq *determinate-rs-and-vars* nil
          *rs-and-vars-introducing-new-vars-covering-all-pos* nil
          *rs-and-vars-introducing-new-vars* nil)
    (find-discriminating-literal current-state-value    
                                 pred vars type
                                 goal-concept-and-rule pos-tuples neg-tuples
                                 max-new-variables old-vars use-hash-tables winners)
    (create-non-discriminating-literals-if-needed vars type pos-tuples neg-tuples winners)
    (let ((winner (choose-a-winner winners)))
      (set-variable-depths winner vars)
      (values winner winners))))

;;;_______________________________________________________________________________
;;; FIND-DISCRIMINATING-LITERAL

(defun find-discriminating-literal
       (current-state-value    ;; I(Ti)
        pred                   ;; predicate structure of pred being learned
        vars                   ;; current set of bound variables
        type                   ;; type of bound vars
        goal-concept-and-rule           ;; goal concept (if any) to operationlize for EBL
        pos-tuples           
        neg-tuples 
        max-new-variables 
        old-vars               ;; original vars (for recursion detection)
        use-hash-tables
        winners)

  (when *use-goal-concept*
    (change-status :ebl)
    (find-literal-from-goal-concept current-state-value
                                    goal-concept-and-rule vars type
                                    pos-tuples neg-tuples use-hash-tables winners))

  (unless (and *theory-mode* (winners-all-winners winners))
    
    ;;_____ template induction ____________________________
    (when (and *template-induction*
               *example-templates*
               (not *covered-all-pos-tuples*))
      (change-status :extensional)   ;;; should be :template
      (setq *determinate-rs-and-vars* nil)
      (find-literal-template current-state-value 
                             pred vars type max-new-variables
                             pos-tuples neg-tuples old-vars use-hash-tables winners))
    
    (unless (and *template-induction* (winners-all-winners winners))
      
      ;;_____ extensional induction _________________________
      (when (and *extensional-induction*
                 (not *covered-all-pos-tuples*))					
        (change-status :extensional)
        (find-literal-extensional current-state-value 
                                  pred vars type max-new-variables
                                  pos-tuples neg-tuples old-vars use-hash-tables winners))
      
      ;;_____ builtin induction _____________________________
      (when (and *builtin-induction*
                 (not *covered-all-pos-tuples*))
        (change-status :builtin)
        (find-literal-builtin current-state-value
                              pred vars type
                              pos-tuples neg-tuples old-vars use-hash-tables winners))
      
      ;;_____ intensional induction _________________________
      (when (and *intensional-induction*
                 (not *covered-all-pos-tuples*))
        (change-status :intensional)
        (find-literal-intensional current-state-value 
                                  pred vars type max-new-variables
                                  pos-tuples neg-tuples old-vars use-hash-tables winners))
      
      ;;_____ cliche instantiation __________________________
      (when (and *use-relational-cliches*
                 (or *always-try-cliches*
                     (null (winners-all-winners winners))))
        (change-status :cliche)
        (find-literal-cliches current-state-value 
                              pred vars type *max-new-cliche-vars*
                              pos-tuples neg-tuples old-vars use-hash-tables winners))
      ))
  winners)

;;;_______________________________________________________________________________
;;; CREATE-NON-DISCRIMINATING-LITERALS-IF-NEEDED

(defun create-non-discriminating-literals-if-needed (vars type pos-tuples neg-tuples winners)
  (when *enable-determinate-literals*
    (when (or (null (winners-all-winners winners))
              (let* ((lpos (length pos-tuples))
                     (lneg (length neg-tuples))
                     (max-possible-gain (gain-gain (gain-metric (current-metric lpos lneg) lpos lpos 0 0))))
                (< (winner-gain (best-winner winners)) (* *add-determinate-gain-threshold* max-possible-gain))))
      (change-status :determinate)
      (create-literals-from-rs-and-vars *determinate-rs-and-vars* vars type pos-tuples neg-tuples winners))
    (when (null (winners-all-winners winners))
      (change-status :determinate)
      (create-literals-from-rs-and-vars *rs-and-vars-introducing-new-vars-covering-all-pos* vars type pos-tuples neg-tuples winners))
    (when (null (winners-all-winners winners))
      (change-status :determinate)
      (create-literals-from-rs-and-vars (list (random-element *rs-and-vars-introducing-new-vars*)) vars type pos-tuples neg-tuples winners)))
  winners)