;;; -*- Mode: LISP; Syntax: Ansi-common-lisp; Package: CL-LIB; Base: 10 -*-

;;; ****************************************************************    
;;; Locatives ******************************************************
;;; ****************************************************************
;;;
;;; This is the locatives package written June 1993 by 
;;;   Bradford W. Miller
;;;   miller@cs.rochester.edu
;;;   University of Rochester, Department of Computer Science
;;;   610 CS Building, Comp Sci Dept., U. Rochester, Rochester NY 14627-0226
;;;   716-275-1118
;;; I will be glad to respond to bug reports or feature requests.
;;;
;;; This version was NOT obtained from the directory
;;; /afs/cs.cmu.edu/user/mkant/Public/Lisp-Utilities/locatives.lisp
;;; via anonymous ftp from a.gp.cs.cmu.edu. (you got it in cl-lib).
;;;
;;;
;;; Bug reports, improvements, and feature requests should be sent
;;; to miller@cs.rochester.edu. Ports to other lisps are also welcome.
;;; (It would be appreciated if you would also cc mkant@cs.cmu.edu.)
;;;
;;; Copyright (C) 1993 by Bradford W. Miller (miller@cs.rochester.edu)
;;;                       and the Trustees of the University of Rochester
;;; Right of use & redistribution is granted as per the terms of the 
;;; GNU LIBRARY GENERAL PUBLIC LICENCE version 2 which is incorporated here by
;;; reference. 
;;;
;;; Acknowledgements:
;;; This code has benefited from the comments of:
;;;  Steven Haflich, (smh@franz.com)
;;;
;;; Please note that while I beleive the above have contributed in
;;; a positive way to what you see before you, I remain soley
;;; responsible for the contents. In particular any errors, omissions, 
;;; or major blunders are purely my own and do not reflect on anyone 
;;; listed above!
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; Gnu Library General Public License for more details.
;;;
;;; You should have received a copy of the Gnu Library General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;

;;; ********************************
;;; Motivation *********************
;;; ********************************
;;;

;;; Locatives are useful for allowing destructive manipulation of objects
;;; based on reference, rather than keeping track of the objects themselves.
;;; This is similar to using pointers in C. Normally only a low-level hacker
;;; will need them.


;;; ********************************
;;; Limitations ********************
;;; ********************************
;;;
;;;
;;;Note that in most common lisps we don't have the full generality of
;;;really being able to refer to cells on the machine (though the lispms
;;;provided this). Therefore, this implementation is NOT compatible (in the 
;;;sense that it doesn't use pointers) with
;;;what is available on the lisp machines, though it does try to emulate it
;;;in spirit. In particular, the storage consumed by this implementation is much
;;;larger than it would be on a lisp machine, since (except for lists), it must
;;;store two closures. It is possible to optimize cases other than simple lists
;;;to not require closures as well, at some minor compile-time and run-time cost.
;;;In addition the structure object itself requires 6 words, instead of the 2-3
;;;words a lispm needed for a locative. 

;;;Large numbers of locatives are therefore possibly not a good idea.

;;;
;;;in particular, this simple implementation expects that the reference
;;;argument to locf contains a setf'able reference, and the code generated by
;;;setf is reversable (see reinvert-setter). To the extent the function isn't
;;;recognized by setf (get-setf-expansion) or the code generated by that isn't
;;;recognized by reinvert-setter, we will fail.
;;;
;;; All the common, likely cases under Allegro are handled, e.g. c*r, aref, elt, svref,
;;; nth, etc. The strategy is simple; we store away what it is the arguments to the 
;;; reference function are (e.g. for car, the list), and remember what function it
;;; is that we are trying to use. So in some sense we just snapshot the arguments to
;;; our function and use that to evaluate the function application later (when we
;;; do location-contents or setf it).



;;; ********************************
;;; Description ********************
;;; ********************************
;;;
;;; (locative-p x) true if x is a locative
;;; (location-contents locative) returns the contents of what the
;;;                              locative points to, use setf to update
;;; (locf reference)  convert reference to a locative. See above limitations.
;;;
;;; Examples of use:
;;; (setq bar '(a b c))
;;; (setq foo (locf (cadr bar)))
;;; bar -> (a b c)
;;; (location-contents foo) -> b
;;; (setf (location-contents foo) 'd)
;;; bar -> (a d c)
;;; (setf (location-contents (locf (symbol-value 'bar))) '(1 2 3))
;;; bar -> (1 2 3)
;;; (setf (location-contents foo) 'e)
;;; bar -> (1 2 3)

(in-package CL-LIB)

;; (export '(locf location-contents locative-p locative))

;;; in this implementation, a locative is a closure over a read and write function
(defstruct (locative (:print-function print-locative))
  (read #'identity :type function)
  (write #'identity :type function)
  (flags 0 :type fixnum))

(defconstant +car-loc+ 1)
(defconstant +cdr-loc+ 2)

(defun print-locative (loc output-stream depth)
  (if (and *print-level* (> depth *print-level*))
      (format output-stream "#")
    (print-unreadable-object (loc output-stream :type t :identity t))))

(defun location-contents (locative)
  (case (locative-flags locative)
   (0
    (funcall (locative-read locative)))
   (1                           ; +car-loc+
    (car (locative-read locative)))
   (2                           ; +cdr-loc+
    (cdr (locative-read locative)))))

(defsetf location-contents (locative) (new-value)
  `(case (locative-flags ,locative)
     (0
      (funcall (locative-write ,locative) ,new-value))
     (,+car-loc+
      (setf (car (locative-read ,locative)) ,new-value))
     (,+cdr-loc+
      (setf (cdr (locative-read ,locative)) ,new-value))))

;; we provide a translator for franz allegro. Others will need to write their own lookup table.
;; Some of these have been provided thanks to "trial and error"; others thanks to "source code", i.e. setf.cl
#-(or excl ccl) (eval-when (compile load eval)
         (error "table to invert setf function not supplied. Locative code will not work!"))

;;; I haven't exhastively gone thru all the ccl ones yet. But we signal an error if we try to
;;; use an undefined one, so it should be ok.
(defun reinvert-setter (setf-function)
  (case setf-function 
    (#+excl excl::.inv-car
     #+ccl ccl::set-car
     'car)
    (#+excl excl::.inv-cdr
     #+ccl ccl::set-cdr
     'cdr)
    #+ccl
    (#+ccl set-cadr
     'second)
    #+ccl
    (#+ccl set-cdar
     'cdar)
    #+ccl
    (#+ccl set-caar
     'caar)
    #+ccl
    (#+ccl set-cddr
     'cddr)
    (#+excl excl::.inv-elt
     #+ccl ccl::set-elt
     'elt)
    (#+excl excl::.inv-s-aref
     #+ccl ccl::aset
     'aref)
    (#+excl excl::.inv-svref
     #+ccl ccl::svset
     'svref)
    #+excl
    (#+excl excl::.inv-structure-ref
     'structure-ref)
    #+excl
    (#+excl excl::.inv-standard-instance-ref
     'standard-instance-ref)
    (#+excl excl::.inv-schar
     #+ccl ccl::set-schar
     'schar)
    (#+excl excl::.inv-sbit
     #+ccl ccl::%sbitset
     'sbit)
    (#+excl excl::.inv-symbol-function
     #+ccl fset
     'symbol-function)
    #+ccl 
    (#+ccl set-fill-pointer
     'fill-pointer)
    #+ccl
    (#+ccl set-cdddr
     'cdddr)
    #+ccl
    (#+ccl set-cddar
     'cddar)    
    #+ccl
    (#+ccl set-caddr
     'caddr)    
    #+ccl
    (#+ccl set-cadar
     'cadar)
    #+ccl
    (#+ccl set-caadr
     'caadr)
    #+ccl
    (#+ccl set-caaar
     'caaar)
    #+ccl
    (#+ccl  set-cdaar
     'cdaar)
    #+ccl
    (#+ccl set-cdadr
     'cdadr)
    (#+excl excl::.inv-symbol-plist
     #+ccl ccl::set-symbol-plist
     'symbol-plist)
    (#+excl excl::.inv-nth 
     #+ccl ccl::%setnth
     'nth)
    (#+excl excl::.inv-fill-pointer
     #+ccl ccl::set-fill-pointer
     'fill-pointer)
    #+:tr-strings
    (excl::.inv-translated-string 'translated-string)
    (#+excl excl::.inv-get
     #+ccl ccl::set-get
     'get)
    (#+excl excl::.inv-macro-function
     #+ccl ccl::set-macro-function
     'macro-function)
    (#+excl excl::.inv-compiler-macro-function
     #+ccl ccl::set-compiler-macro-function
     'compiler-macro-function)
    (#+excl excl::%puthash
     #+ccl ccl::puthash
     'gethash)
    (set 'symbol-value)
    (t (error "Can't figure out inversion"))
    ))

(defun fixup-args (setf-function) ; some of the args are reversed, etc.
  (case setf-function
    #+excl (excl::.inv-s-aref 'cdr)          ; these take the "new" arg first
    #+excl (excl::.inv-sbit 'cdr)
    (t 'butlast)                ; everything else takes new arg last
    ))

(defmacro locf (reference &environment env)
  "Much like seeing a setf"
  ;; we have to capture what our reference refers to, then use it to grab stuff at run-time.
  (mlet (temps formals setter-var setf-code access-code)
      (#-ccl-2 get-setf-expansion #+ccl-2 get-setf-method-multiple-value reference env) ; find out which is the important "setter", everything else is "reference".
    (declare (ignore setter-var access-code))
    (let ((ref-syms (mapcar #'(lambda (x) x (gensym)) (funcall (fixup-args (car setf-code)) (cdr setf-code))))
          (ref-code (funcall (fixup-args (car setf-code)) (cdr setf-code))))
      (do ((formal formals (cdr formal))
           (temp temps (cdr temp)))
          ((null formal))
        (labels ((walk-and-substitute (l)
                   (if (consp l)
                       (mapcar #'walk-and-substitute l)
                     (if (eq l (car temp)) (car formal) l))))
          (setq ref-code (mapcar #'walk-and-substitute ref-code))))
      (case (reinvert-setter (car setf-code))
        (car
         `(make-locative :read ,(car ref-code) :flags ,+car-loc+))
        (cdr
         `(make-locative :read ,(car ref-code) :flags ,+cdr-loc+))
        (t
         `(let (,@(mapcar #'(lambda (ref setter)
                              (list ref setter))
                          ref-syms
                          ref-code))
            (make-locative :read #'(lambda () (,(reinvert-setter (car setf-code)) ,@ref-syms))
                           ;; do it this way because we discarded the needed args.
                           :write #'(lambda (new) (setf (,(reinvert-setter (car setf-code)) ,@ref-syms) new))))))))) 

(cl-lib:macro-indent-rule locf (like setf))

